"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
function deployNewFunc(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    return new lib_1.IotToLambda(stack, 'test-iot-lambda-integration', props);
}
function useExistingFunc(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.PYTHON_3_6,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const props = {
        existingLambdaObj: new lambda.Function(stack, 'MyExistingFunction', lambdaFunctionProps),
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    return new lib_1.IotToLambda(stack, 'test-iot-lambda-integration', props);
}
test('check lambda function properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testiotlambdaintegrationLambdaFunctionServiceRole27C3EE41",
                "Arn"
            ]
        },
        Runtime: "nodejs14.x"
    });
});
test('check lambda function permission for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Permission', {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            "Fn::GetAtt": [
                "testiotlambdaintegrationLambdaFunctionC5329DBA",
                "Arn"
            ]
        },
        Principal: "iot.amazonaws.com",
        SourceArn: {
            "Fn::GetAtt": [
                "testiotlambdaintegrationIotTopic18B6A735",
                "Arn"
            ]
        }
    });
});
test('check iot lambda function role for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/lambda/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "LambdaFunctionServiceRolePolicy"
            }
        ]
    });
});
test('check iot topic rule properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Lambda: {
                        FunctionArn: {
                            "Fn::GetAtt": [
                                "testiotlambdaintegrationLambdaFunctionC5329DBA",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'connectedcar/dtc/#'"
        }
    });
});
test('check lambda function properties for deploy: false', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "MyExistingFunctionServiceRoleF9E14BFD",
                "Arn"
            ]
        },
        Runtime: "python3.6"
    });
});
test('check lambda function permissions for deploy: false', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Permission', {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            "Fn::GetAtt": [
                "MyExistingFunction4D772515",
                "Arn"
            ]
        },
        Principal: "iot.amazonaws.com",
        SourceArn: {
            "Fn::GetAtt": [
                "testiotlambdaintegrationIotTopic18B6A735",
                "Arn"
            ]
        }
    });
});
test('check iot lambda function role for deploy: false', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        ManagedPolicyArns: [
            {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                    ]
                ]
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.iotTopicRule !== null);
    expect(construct.lambdaFunction !== null);
});
test('check exception for Missing existingObj from props for deploy = false', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    try {
        new lib_1.IotToLambda(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check deploy = true and no prop', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    try {
        new lib_1.IotToLambda(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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