"""
Tests brewblox_devcon_spark.api
"""

import asyncio
import os

import pytest
from brewblox_service import scheduler

from brewblox_codec_spark import codec
from brewblox_devcon_spark import commander_sim, datastore, device
from brewblox_devcon_spark.api import (alias_api, conflict_api, debug_api,
                                       error_response, object_api, profile_api,
                                       system_api)


@pytest.fixture
def object_args():
    return dict(
        id='testobj',
        profiles=[1, 4, 7],
        type='OneWireTempSensor',
        data=dict(
            settings=dict(
                address='FF',
                offset=20
            ),
            state=dict(
                value=12345,
                connected=True
            )
        )
    )


@pytest.fixture
def database_test_file():
    def remove(f):
        try:
            os.remove(f)
        except FileNotFoundError:
            pass

    f = 'api_test_file.json'
    remove(f)
    yield f
    remove(f)


@pytest.fixture
async def app(app, database_test_file, loop):
    """App + controller routes"""
    app['config']['database'] = database_test_file

    scheduler.setup(app)
    commander_sim.setup(app)
    datastore.setup(app)
    codec.setup(app)
    device.setup(app)

    error_response.setup(app)
    debug_api.setup(app)
    alias_api.setup(app)
    conflict_api.setup(app)
    object_api.setup(app)
    profile_api.setup(app)
    system_api.setup(app)

    return app


async def response(request):
    retv = await request
    assert retv.status == 200
    return await retv.json()


async def test_do(app, client):
    command = dict(command='create_object', data={
        'object_type': 'OneWireTempSensor',
        'profiles': [1, 2, 3],
        'object_data': {
            'settings': {
                'address': 'FF',
                'offset': 20
            },
            'state': {
                'value': 1234,
                'connected': True
            }
        }
    })

    retv = await client.post('/_debug/do', json=command)
    assert retv.status == 200


async def test_create(app, client, object_args):
    # Create object
    retd = await response(client.post('/objects', json=object_args))
    assert retd['id'] == object_args['id']

    # Allowed to recreate, but we don't get provided ID
    retd = await response(client.post('/objects', json=object_args))
    assert retd['id'] != object_args['id']

    object_args['id'] = 'other_obj'
    retd = await response(client.post('/objects', json=object_args))
    assert retd['id'] == 'other_obj'


async def test_create_performance(app, client, object_args):
    num_items = 50
    coros = [client.post('/objects', json=object_args) for _ in range(num_items)]
    responses = await asyncio.gather(*coros)
    assert [retv.status for retv in responses] == [200]*num_items

    retd = await response(client.get('/saved_objects'))
    assert len(retd) == num_items


async def test_read(app, client, object_args):
    retv = await client.get('/objects/testobj')
    assert retv.status == 500  # Object does not exist

    await client.post('/objects', json=object_args)

    retd = await response(client.get('/objects/testobj'))
    assert retd['id'] == 'testobj'


async def test_read_performance(app, client, object_args):
    await client.post('/objects', json=object_args)

    coros = [client.get('/objects/testobj') for _ in range(100)]
    responses = await asyncio.gather(*coros)
    assert [retv.status for retv in responses] == [200]*100


async def test_update(app, client, object_args):
    await client.post('/objects', json=object_args)
    assert await response(client.put('/objects/testobj', json=object_args))


async def test_delete(app, client, object_args):
    await client.post('/objects', json=object_args)

    retd = await response(client.delete('/objects/testobj'))
    assert retd['id'] == 'testobj'

    retv = await client.get('/objects/testobj')
    assert retv.status == 500


async def test_all(app, client, object_args):
    assert await response(client.get('/saved_objects')) == []

    await client.post('/objects', json=object_args)
    retd = await response(client.get('/saved_objects'))
    assert len(retd) == 1


async def test_active(app, client, object_args):
    retd = await response(client.get('/objects'))
    assert retd == []

    await client.post('/objects', json=object_args)
    retd = await response(client.get('/objects'))
    assert retd == []

    await client.post('/profiles', json=[1])

    retd = await response(client.get('/objects'))
    assert len(retd) == 1
    assert retd[0]['id'] == object_args['id']


async def test_system_read(app, client, object_args):
    # No system objects found
    # TODO(Bob): add preset system objects to simulator
    await response(client.get('/system/onewirebus'))


async def test_system_update(app, client, object_args):
    # No system objects found
    # TODO(Bob): add preset system objects to simulator
    await response(client.put('/system/onewirebus', json=object_args))


async def test_profiles(app, client):
    retd = await response(client.get('/profiles'))
    assert retd == []

    active = [1, 6, 7]
    retd = await response(client.post('/profiles', json=active))
    assert retd == active

    retd = await response(client.get('/profiles'))
    assert retd == active


@pytest.mark.parametrize('input_id', [
    'flabber',
    'FLABBER',
    'f',
    'a;ljfoihoewr*&(%&^&*%*&^(*&^(',
    'l1214235234231',
    'yes!'*50,
])
async def test_validate_service_id(input_id):
    alias_api.validate_service_id(input_id)


@pytest.mark.parametrize('input_id', [
    '1',
    '1adsjlfdsf',
    'pancakes[delicious]',
    '[',
    'f]abbergasted',
    '',
    'yes!'*51,
])
async def test_validate_service_id_error(input_id):
    with pytest.raises(ValueError):
        alias_api.validate_service_id(input_id)


async def test_alias_create(app, client):
    new_alias = dict(
        service_id='name',
        controller_id=456
    )
    retv = await client.post('/aliases', json=new_alias)
    assert retv.status == 200

    retv = await client.post('/aliases', json=new_alias)
    assert retv.status == 409


async def test_alias_update(app, client, object_args):
    await client.post('/objects', json=object_args)

    retv = await client.get('/objects/newname')
    assert retv.status == 500

    retv = await client.put('/aliases/testobj', json={'id': 'newname'})
    assert retv.status == 200

    retv = await client.get('/objects/newname')
    assert retv.status == 200


async def test_conflict_all(app, client):
    objects = [
        {'service_id': 'sid', 'controller_id': 8},
        {'service_id': 'sid', 'controller_id': 9}
    ]

    store = datastore.get_object_store(app)
    await store.insert_multiple(objects)

    retd = await response(client.get('/conflicts'))
    assert retd == {}

    retv = await client.get('/objects/sid')
    assert retv.status == 428

    retd = await response(client.get('/conflicts'))
    assert retd == {
        'service_id': {
            'sid': objects
        }
    }


async def test_conflict_resolve(app, client, object_args):
    store = datastore.get_object_store(app)
    argid = object_args['id']

    await client.post('/objects', json=object_args)
    await store.insert({'service_id': argid, 'dummy': True})

    retv = await client.get('/objects/' + argid)
    assert retv.status == 428

    retv = await client.get('/conflicts')
    objects = (await retv.json())['service_id'][argid]
    assert len(objects) == 2

    # Pick the one that's not the dummy
    real = next(o for o in objects if 'dummy' not in o)

    retv = await client.post('/conflicts', json={'id_key': 'service_id', 'data': real})
    assert retv.status == 200

    retv = await client.get('/objects/' + argid)
    assert retv.status == 200
