"""LinearStiffnessProperties"""

from __future__ import annotations

from typing import ClassVar, TYPE_CHECKING

from mastapy._private._internal import utility
from mastapy._private._internal.cast_exception import CastException
from mastapy._private._internal.dataclasses import extended_dataclass
from mastapy._private._internal.python_net import (
    python_net_import,
    pythonnet_property_get,
    pythonnet_property_set,
)
from mastapy._private._internal.type_enforcement import enforce_parameter_types
from mastapy._private.nodal_analysis import _46

_LINEAR_STIFFNESS_PROPERTIES = python_net_import(
    "SMT.MastaAPI.NodalAnalysis", "LinearStiffnessProperties"
)

if TYPE_CHECKING:
    from typing import Any, Type, TypeVar

    Self = TypeVar("Self", bound="LinearStiffnessProperties")
    CastSelf = TypeVar(
        "CastSelf", bound="LinearStiffnessProperties._Cast_LinearStiffnessProperties"
    )


__docformat__ = "restructuredtext en"
__all__ = ("LinearStiffnessProperties",)


@extended_dataclass(frozen=True, slots=True, weakref_slot=True)
class _Cast_LinearStiffnessProperties:
    """Special nested class for casting LinearStiffnessProperties to subclasses."""

    __parent__: "LinearStiffnessProperties"

    @property
    def abstract_linear_connection_properties(
        self: "CastSelf",
    ) -> "_46.AbstractLinearConnectionProperties":
        return self.__parent__._cast(_46.AbstractLinearConnectionProperties)

    @property
    def linear_stiffness_properties(self: "CastSelf") -> "LinearStiffnessProperties":
        return self.__parent__

    def __getattr__(self: "CastSelf", name: str) -> "Any":
        try:
            return self.__getattribute__(name)
        except AttributeError:
            class_name = utility.camel(name)
            raise CastException(
                f'Detected an invalid cast. Cannot cast to type "{class_name}"'
            ) from None


@extended_dataclass(frozen=True, slots=True, weakref_slot=True)
class LinearStiffnessProperties(_46.AbstractLinearConnectionProperties):
    """LinearStiffnessProperties

    This is a mastapy class.
    """

    TYPE: ClassVar["Type"] = _LINEAR_STIFFNESS_PROPERTIES

    wrapped: "Any"

    def __post_init__(self: "Self") -> None:
        """Override of the post initialisation magic method."""
        if not hasattr(self.wrapped, "reference_count"):
            self.wrapped.reference_count = 0

        self.wrapped.reference_count += 1

    @property
    def axial_stiffness(self: "Self") -> "float":
        """float"""
        temp = pythonnet_property_get(self.wrapped, "AxialStiffness")

        if temp is None:
            return 0.0

        return temp

    @axial_stiffness.setter
    @enforce_parameter_types
    def axial_stiffness(self: "Self", value: "float") -> None:
        pythonnet_property_set(
            self.wrapped, "AxialStiffness", float(value) if value is not None else 0.0
        )

    @property
    def radial_stiffness(self: "Self") -> "float":
        """float"""
        temp = pythonnet_property_get(self.wrapped, "RadialStiffness")

        if temp is None:
            return 0.0

        return temp

    @radial_stiffness.setter
    @enforce_parameter_types
    def radial_stiffness(self: "Self", value: "float") -> None:
        pythonnet_property_set(
            self.wrapped, "RadialStiffness", float(value) if value is not None else 0.0
        )

    @property
    def tilt_stiffness(self: "Self") -> "float":
        """float"""
        temp = pythonnet_property_get(self.wrapped, "TiltStiffness")

        if temp is None:
            return 0.0

        return temp

    @tilt_stiffness.setter
    @enforce_parameter_types
    def tilt_stiffness(self: "Self", value: "float") -> None:
        pythonnet_property_set(
            self.wrapped, "TiltStiffness", float(value) if value is not None else 0.0
        )

    @property
    def torsional_stiffness(self: "Self") -> "float":
        """float"""
        temp = pythonnet_property_get(self.wrapped, "TorsionalStiffness")

        if temp is None:
            return 0.0

        return temp

    @torsional_stiffness.setter
    @enforce_parameter_types
    def torsional_stiffness(self: "Self", value: "float") -> None:
        pythonnet_property_set(
            self.wrapped,
            "TorsionalStiffness",
            float(value) if value is not None else 0.0,
        )

    @property
    def cast_to(self: "Self") -> "_Cast_LinearStiffnessProperties":
        """Cast to another type.

        Returns:
            _Cast_LinearStiffnessProperties
        """
        return _Cast_LinearStiffnessProperties(self)
