"""ComponentConnection"""

from __future__ import annotations

from typing import ClassVar, TYPE_CHECKING

from PIL.Image import Image

from mastapy._private._internal import constructor, conversion, utility
from mastapy._private._internal.cast_exception import CastException
from mastapy._private._internal.dataclasses import extended_dataclass
from mastapy._private._internal.python_net import (
    python_net_import,
    pythonnet_method_call,
    pythonnet_property_get,
)
from mastapy._private.system_model.connections_and_sockets import _2326

_COMPONENT_CONNECTION = python_net_import(
    "SMT.MastaAPI.SystemModel.ConnectionsAndSockets", "ComponentConnection"
)

if TYPE_CHECKING:
    from typing import Any, Type, TypeVar

    from mastapy._private.system_model.connections_and_sockets import _2330
    from mastapy._private.system_model.part_model import _2500

    Self = TypeVar("Self", bound="ComponentConnection")
    CastSelf = TypeVar(
        "CastSelf", bound="ComponentConnection._Cast_ComponentConnection"
    )


__docformat__ = "restructuredtext en"
__all__ = ("ComponentConnection",)


@extended_dataclass(frozen=True, slots=True, weakref_slot=True)
class _Cast_ComponentConnection:
    """Special nested class for casting ComponentConnection to subclasses."""

    __parent__: "ComponentConnection"

    @property
    def component_measurer(self: "CastSelf") -> "_2326.ComponentMeasurer":
        return self.__parent__._cast(_2326.ComponentMeasurer)

    @property
    def cylindrical_component_connection(
        self: "CastSelf",
    ) -> "_2330.CylindricalComponentConnection":
        from mastapy._private.system_model.connections_and_sockets import _2330

        return self.__parent__._cast(_2330.CylindricalComponentConnection)

    @property
    def component_connection(self: "CastSelf") -> "ComponentConnection":
        return self.__parent__

    def __getattr__(self: "CastSelf", name: str) -> "Any":
        try:
            return self.__getattribute__(name)
        except AttributeError:
            class_name = utility.camel(name)
            raise CastException(
                f'Detected an invalid cast. Cannot cast to type "{class_name}"'
            ) from None


@extended_dataclass(frozen=True, slots=True, weakref_slot=True)
class ComponentConnection(_2326.ComponentMeasurer):
    """ComponentConnection

    This is a mastapy class.
    """

    TYPE: ClassVar["Type"] = _COMPONENT_CONNECTION

    wrapped: "Any"

    def __post_init__(self: "Self") -> None:
        """Override of the post initialisation magic method."""
        if not hasattr(self.wrapped, "reference_count"):
            self.wrapped.reference_count = 0

        self.wrapped.reference_count += 1

    @property
    def assembly_view(self: "Self") -> "Image":
        """Image

        Note:
            This property is readonly.
        """
        temp = pythonnet_property_get(self.wrapped, "AssemblyView")

        if temp is None:
            return None

        value = conversion.pn_to_mp_smt_bitmap(temp)

        if value is None:
            return None

        return value

    @property
    def connected_components_socket(self: "Self") -> "str":
        """str

        Note:
            This property is readonly.
        """
        temp = pythonnet_property_get(self.wrapped, "ConnectedComponentsSocket")

        if temp is None:
            return ""

        return temp

    @property
    def detail_view(self: "Self") -> "Image":
        """Image

        Note:
            This property is readonly.
        """
        temp = pythonnet_property_get(self.wrapped, "DetailView")

        if temp is None:
            return None

        value = conversion.pn_to_mp_smt_bitmap(temp)

        if value is None:
            return None

        return value

    @property
    def socket(self: "Self") -> "str":
        """str

        Note:
            This property is readonly.
        """
        temp = pythonnet_property_get(self.wrapped, "Socket")

        if temp is None:
            return ""

        return temp

    @property
    def connected_component(self: "Self") -> "_2500.Component":
        """mastapy.system_model.part_model.Component

        Note:
            This property is readonly.
        """
        temp = pythonnet_property_get(self.wrapped, "ConnectedComponent")

        if temp is None:
            return None

        type_ = temp.GetType()
        return constructor.new(type_.Namespace, type_.Name)(temp)

    def delete(self: "Self") -> None:
        """Method does not return."""
        pythonnet_method_call(self.wrapped, "Delete")

    def swap(self: "Self") -> None:
        """Method does not return."""
        pythonnet_method_call(self.wrapped, "Swap")

    @property
    def cast_to(self: "Self") -> "_Cast_ComponentConnection":
        """Cast to another type.

        Returns:
            _Cast_ComponentConnection
        """
        return _Cast_ComponentConnection(self)
