from typing import List as List_, Optional as Optional_, Literal as Literal_, Annotated as Annotated_, NewType as NewType_

from pydantic import BaseModel as BaseModel
from pydantic import Field, Extra


class Element(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Base definition for all elements in a resource."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class BackboneElement(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Base definition for all elements that are defined inside a resource - but not those in a data type."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


base64Binary = NewType_('base64Binary', str)


'A stream of bytes'


boolean = NewType_('boolean', bool)


'Value of "true" or "false"'


canonical = NewType_('canonical', str)


'A URI that is a reference to a canonical URL on a FHIR resource'


code = NewType_('code', str)


'A string which has at least one character and no leading or trailing whitespace and where there is no whitespace other than single spaces in the contents'


date = NewType_('date', str)


'A date or partial date (e.g. just year or year + month). There is no UTC offset. The format is a union of the schema types gYear, gYearMonth and date.  Dates SHALL be valid dates.'


dateTime = NewType_('dateTime', str)


'A date, date-time or partial date (e.g. just year or year + month).  If hours and minutes are specified, a UTC offset SHALL be populated. The format is a union of the schema types gYear, gYearMonth, date and dateTime. Seconds must be provided due to schema type constraints but may be zero-filled and may be ignored.                 Dates SHALL be valid dates.'


decimal = NewType_('decimal', int)


'A rational number with implicit precision'


id = NewType_('id', str)


'Any combination of letters, numerals, "-" and ".", with a length limit of 64 characters.  (This might be an integer, an unprefixed OID, UUID or any other identifier pattern that meets these constraints.)  Ids are case-insensitive.'


instant = NewType_('instant', str)


'An instant in time - known at least to the second'


integer = NewType_('integer', int)


'A whole number'


integer64 = NewType_('integer64', int)


'A very large whole number'


markdown = NewType_('markdown', str)


'A string that may contain Github Flavored Markdown syntax for optional processing by a mark down presentation engine'


oid = NewType_('oid', str)


'An OID represented as a URI'


positiveInt = NewType_('positiveInt', str)


'An integer with a value that is positive (e.g. >0)'


string = NewType_('string', str)


'A sequence of Unicode characters'


time = NewType_('time', str)


'A time during the day, with no date specified'


unsignedInt = NewType_('unsignedInt', str)


'An integer with a value that is not negative (e.g. >= 0)'


uri = NewType_('uri', str)


'String of characters used to identify a name or a resource'


url = NewType_('url', str)


'A URI that is a literal reference'


uuid = NewType_('uuid', str)


'A UUID, represented as a URI'


xhtml = NewType_('xhtml', str)


'XHTML'


class Address(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An address expressed using postal conventions (as opposed to GPS or other location definition formats).  This data type may be used to convey addresses for use in delivering mail as well as for visiting locations which might not be valid for mail delivery.  There are a variety of postal address formats defined around the world.
The ISO21090-codedString may be used to provide a coded representation of the contents of strings in an Address."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    use: Optional_['code'] = Field(default=None)
    'The purpose of this address.'
    type: Optional_['code'] = Field(default=None)
    'Distinguishes between physical addresses (those you can visit) and mailing addresses (e.g. PO Boxes and care-of addresses). Most addresses are both.'
    text: Optional_['string'] = Field(default=None)
    'Specifies the entire address as it should be displayed e.g. on a postal label. This may be provided instead of or as well as the specific parts.'
    line: List_['string'] = Field(default_factory=list)
    'This component contains the house number, apartment number, street name, street direction,  P.O. Box number, delivery hints, and similar address information.'
    city: Optional_['string'] = Field(default=None)
    'The name of the city, town, suburb, village or other community or delivery center.'
    state: Optional_['string'] = Field(default=None)
    'Sub-unit of a country with limited sovereignty in a federally organized country. A code may be used if codes are in common use (e.g. US 2 letter state codes).'
    period: Optional_['Period'] = Field(default=None)
    'Time period when address was/is in use.'
    country: Optional_['string'] = Field(default=None)
    'Country - a nation as commonly understood or generally accepted.'
    district: Optional_['string'] = Field(default=None)
    'The name of the administrative area (county).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    postalCode: Optional_['string'] = Field(default=None)
    'A postal code designating a region defined by the postal service.'


class Age(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A duration of time during which an organism (or a process) has existed."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    unit: Optional_['string'] = Field(default=None)
    'A human-readable form of the unit.'
    value: Optional_['decimal'] = Field(default=None)
    'The value of the measured amount. The value includes an implicit precision in the presentation of the value.'
    system: Optional_['uri'] = Field(default=None)
    'The identification of the system that provides the coded form of the unit.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    comparator: Optional_['code'] = Field(default=None)
    'How the value should be understood and represented - whether the actual value is greater or less than the stated value due to measurement issues; e.g. if the comparator is "<" , then the real value is < stated value.'
    code: Optional_['code'] = Field(default=None)
    'A computer processable form of the unit in some unit representation system.'


class Annotation(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A  text note which also  contains information about who made the statement and when."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    time: Optional_['dateTime'] = Field(default=None)
    'Indicates when this particular annotation was made.'
    text: 'markdown' = Field(default=...)
    'The text of the annotation in markdown format.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    authorReference: Optional_['Reference'] = Field(default=None)
    'The individual responsible for making the annotation.'
    authorString: Optional_['string'] = Field(default=None)
    'The individual responsible for making the annotation.'


class Attachment(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """For referring to data content defined in other formats."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    data: Optional_['base64Binary'] = Field(default=None)
    'The actual data of the attachment - a sequence of bytes, base64 encoded.'
    size: Optional_['integer64'] = Field(default=None)
    'The number of bytes of data that make up this attachment (before base64 encoding, if that is done).'
    hash: Optional_['base64Binary'] = Field(default=None)
    'The calculated hash of the data using SHA-1. Represented using base64.'
    title: Optional_['string'] = Field(default=None)
    'A label or set of text to display in place of the data.'
    width: Optional_['positiveInt'] = Field(default=None)
    'Width of the image in pixels (photo/video).'
    pages: Optional_['positiveInt'] = Field(default=None)
    'The number of pages when printed.'
    height: Optional_['positiveInt'] = Field(default=None)
    'Height of the image in pixels (photo/video).'
    frames: Optional_['positiveInt'] = Field(default=None)
    'The number of frames in a photo. This is used with a multi-page fax, or an imaging acquisition context that takes multiple slices in a single image, or an animated gif. If there is more than one frame, this SHALL have a value in order to alert interface software that a multi-frame capable rendering widget is required.'
    language: Optional_['code'] = Field(default=None)
    'The human language of the content. The value can be any valid value according to BCP 47.'
    creation: Optional_['dateTime'] = Field(default=None)
    'The date that the attachment was first created.'
    duration: Optional_['decimal'] = Field(default=None)
    'The duration of the recording in seconds - for audio and video.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    contentType: Optional_['code'] = Field(default=None)
    'Identifies the type of the data in the attachment and allows a method to be chosen to interpret or render the data. Includes mime type parameters such as charset where appropriate.'
    url: Optional_['url'] = Field(default=None)
    'A location where the data can be accessed.'


class AvailabilityNotAvailableTime(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Not available during this time due to provided reason."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    during: Optional_['Period'] = Field(default=None)
    'Service not available during this period.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    description: Optional_['string'] = Field(default=None)
    'Reason presented to the user explaining why time not available.'


class AvailabilityAvailableTime(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Times the {item} is available."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    allDay: Optional_['boolean'] = Field(default=None)
    'Always available? i.e. 24 hour service.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    daysOfWeek: List_['code'] = Field(default_factory=list)
    'mon | tue | wed | thu | fri | sat | sun.'
    availableEndTime: Optional_['time'] = Field(default=None)
    'Closing time of day (ignored if allDay = true).'
    availableStartTime: Optional_['time'] = Field(default=None)
    'Opening time of day (ignored if allDay = true).'


class Availability(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Availability data for an {item}."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    availableTime: List_['AvailabilityAvailableTime'] = Field(default_factory=list)
    'Times the {item} is available.'
    notAvailableTime: List_['AvailabilityNotAvailableTime'] = Field(default_factory=list)
    'Not available during this time due to provided reason.'


class BackboneType(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Base definition for the few data types that are allowed to carry modifier extensions."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Base(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Base definition for all types defined in FHIR type system."""


class CodeableConcept(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A concept that may be defined by a formal reference to a terminology or ontology or may be provided by text."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    text: Optional_['string'] = Field(default=None)
    'A human language representation of the concept as seen/selected/uttered by the user who entered the data and/or which represents the intended meaning of the user.'
    coding: List_['Coding'] = Field(default_factory=list)
    'A reference to a code defined by a terminology system.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class CodeableReference(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A reference to a resource (by instance), or instead, a reference to a concept defined in a terminology or ontology (by class)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    concept: Optional_['CodeableConcept'] = Field(default=None)
    'A reference to a concept - e.g. the information is identified by its general class to the degree of precision found in the terminology.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: Optional_['Reference'] = Field(default=None)
    'A reference to a resource the provides exact details about the information being referenced.'


class Coding(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A reference to a code defined by a terminology system."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    system: Optional_['uri'] = Field(default=None)
    'The identification of the code system that defines the meaning of the symbol in the code.'
    version: Optional_['string'] = Field(default=None)
    'The version of the code system which was used when choosing this code. Note that a well-maintained code system does not need the version reported, because the meaning of codes is consistent across versions. However this cannot consistently be assured, and when the meaning is not guaranteed to be consistent, the version SHOULD be exchanged.'
    display: Optional_['string'] = Field(default=None)
    'A representation of the meaning of the code in the system, following the rules of the system.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    userSelected: Optional_['boolean'] = Field(default=None)
    'Indicates that this coding was chosen by a user directly - e.g. off a pick list of available items (codes or displays).'
    code: Optional_['code'] = Field(default=None)
    'A symbol in syntax defined by the system. The symbol may be a predefined code or an expression in a syntax defined by the coding system (e.g. post-coordination).'


class ContactDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specifies contact information for a person or organization."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    name: Optional_['string'] = Field(default=None)
    'The name of an individual to contact.'
    telecom: List_['ContactPoint'] = Field(default_factory=list)
    'The contact details for the individual (if a name was provided) or the organization.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class ContactPoint(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Details for all kinds of technology mediated contact points for a person or organization, including telephone, email, etc."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    use: Optional_['code'] = Field(default=None)
    'Identifies the purpose for the contact point.'
    rank: Optional_['positiveInt'] = Field(default=None)
    'Specifies a preferred order in which to use a set of contacts. ContactPoints with lower rank values are more preferred than those with higher rank values.'
    value: Optional_['string'] = Field(default=None)
    'The actual contact point details, in a form that is meaningful to the designated communication system (i.e. phone number or email address).'
    system: Optional_['code'] = Field(default=None)
    'Telecommunications form for contact point - what communications system is required to make use of the contact.'
    period: Optional_['Period'] = Field(default=None)
    'Time period when the contact point was/is in use.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class Contributor(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A contributor to the content of a knowledge asset, including authors, editors, reviewers, and endorsers."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'The type of contributor.'
    name: 'string' = Field(default=...)
    'The name of the individual or organization responsible for the contribution.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the contributor.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class Count(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A measured amount (or an amount that can potentially be measured). Note that measured amounts include amounts that are not precisely quantified, including amounts involving arbitrary units and floating currencies."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    unit: Optional_['string'] = Field(default=None)
    'A human-readable form of the unit.'
    value: Optional_['decimal'] = Field(default=None)
    'The value of the measured amount. The value includes an implicit precision in the presentation of the value.'
    system: Optional_['uri'] = Field(default=None)
    'The identification of the system that provides the coded form of the unit.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    comparator: Optional_['code'] = Field(default=None)
    'How the value should be understood and represented - whether the actual value is greater or less than the stated value due to measurement issues; e.g. if the comparator is "<" , then the real value is < stated value.'
    code: Optional_['code'] = Field(default=None)
    'A computer processable form of the unit in some unit representation system.'


class DataRequirementValueFilter(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Value filters specify additional constraints on the data for elements other than code-valued or date-valued. Each value filter specifies an additional constraint on the data (i.e. valueFilters are AND'ed, not OR'ed)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    path: Optional_['string'] = Field(default=None)
    'The attribute of the filter. The specified path SHALL be a FHIRPath resolvable on the specified type of the DataRequirement, and SHALL consist only of identifiers, constant indexers, and .resolve(). The path is allowed to contain qualifiers (.) to traverse sub-elements, as well as indexers ([x]) to traverse multiple-cardinality sub-elements (see the [Simple FHIRPath Profile](fhirpath.html#simple) for full details). Note that the index must be an integer constant. The path must resolve to an element of a type that is comparable to the valueFilter.value[x] element for the filter.'
    valueDateTime: Optional_['dateTime'] = Field(default=None)
    'The value of the filter.'
    valuePeriod: Optional_['Period'] = Field(default=None)
    'The value of the filter.'
    valueDuration: Optional_['Duration'] = Field(default=None)
    'The value of the filter.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    comparator: Optional_['code'] = Field(default=None)
    'The comparator to be used to determine whether the value is matching.'
    searchParam: Optional_['string'] = Field(default=None)
    'A search parameter defined on the specified type of the DataRequirement, and which searches on elements of a type compatible with the type of the valueFilter.value[x] for the filter.'


class DataRequirementDateFilter(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Date filters specify additional constraints on the data in terms of the applicable date range for specific elements. Each date filter specifies an additional constraint on the data, i.e. date filters are AND'ed, not OR'ed."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    path: Optional_['string'] = Field(default=None)
    'The date-valued attribute of the filter. The specified path SHALL be a FHIRPath resolvable on the specified type of the DataRequirement, and SHALL consist only of identifiers, constant indexers, and .resolve(). The path is allowed to contain qualifiers (.) to traverse sub-elements, as well as indexers ([x]) to traverse multiple-cardinality sub-elements (see the [Simple FHIRPath Profile](fhirpath.html#simple) for full details). Note that the index must be an integer constant. The path must resolve to an element of type date, dateTime, Period, Schedule, or Timing.'
    valueDateTime: Optional_['dateTime'] = Field(default=None)
    'The value of the filter. If period is specified, the filter will return only those data items that fall within the bounds determined by the Period, inclusive of the period boundaries. If dateTime is specified, the filter will return only those data items that are equal to the specified dateTime. If a Duration is specified, the filter will return only those data items that fall within Duration before now.'
    valuePeriod: Optional_['Period'] = Field(default=None)
    'The value of the filter. If period is specified, the filter will return only those data items that fall within the bounds determined by the Period, inclusive of the period boundaries. If dateTime is specified, the filter will return only those data items that are equal to the specified dateTime. If a Duration is specified, the filter will return only those data items that fall within Duration before now.'
    valueDuration: Optional_['Duration'] = Field(default=None)
    'The value of the filter. If period is specified, the filter will return only those data items that fall within the bounds determined by the Period, inclusive of the period boundaries. If dateTime is specified, the filter will return only those data items that are equal to the specified dateTime. If a Duration is specified, the filter will return only those data items that fall within Duration before now.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    searchParam: Optional_['string'] = Field(default=None)
    'A date parameter that refers to a search parameter defined on the specified type of the DataRequirement, and which searches on elements of type date, dateTime, Period, Schedule, or Timing.'


class DataRequirementCodeFilter(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Code filters specify additional constraints on the data, specifying the value set of interest for a particular element of the data. Each code filter defines an additional constraint on the data, i.e. code filters are AND'ed, not OR'ed."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    path: Optional_['string'] = Field(default=None)
    'The code-valued attribute of the filter. The specified path SHALL be a FHIRPath resolvable on the specified type of the DataRequirement, and SHALL consist only of identifiers, constant indexers, and .resolve(). The path is allowed to contain qualifiers (.) to traverse sub-elements, as well as indexers ([x]) to traverse multiple-cardinality sub-elements (see the [Simple FHIRPath Profile](fhirpath.html#simple) for full details). Note that the index must be an integer constant. The path must resolve to an element of type code, Coding, or CodeableConcept.'
    code: List_['Coding'] = Field(default_factory=list)
    'The codes for the code filter. If values are given, the filter will return only those data items for which the code-valued attribute specified by the path has a value that is one of the specified codes. If codes are specified in addition to a value set, the filter returns items matching a code in the value set or one of the specified codes.'
    valueSet: Optional_['canonical'] = Field(default=None)
    'The valueset for the code filter. The valueSet and code elements are additive. If valueSet is specified, the filter will return only those data items for which the value of the code-valued element specified in the path is a member of the specified valueset.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    searchParam: Optional_['string'] = Field(default=None)
    'A token parameter that refers to a search parameter defined on the specified type of the DataRequirement, and which searches on elements of type code, Coding, or CodeableConcept.'


class DataRequirementSort(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specifies the order of the results to be returned."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    path: 'string' = Field(default=...)
    'The attribute of the sort. The specified path must be resolvable from the type of the required data. The path is allowed to contain qualifiers (.) to traverse sub-elements, as well as indexers ([x]) to traverse multiple-cardinality sub-elements. Note that the index must be an integer constant.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    direction: 'code' = Field(default=...)
    'The direction of the sort, ascending or descending.'


class DataRequirement(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Describes a required data item for evaluation in terms of the type of data, and optional code or date-based filters of the data."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'The type of the required data, specified as the type name of a resource. For profiles, this value is set to the type of the base resource of the profile.'
    sort: List_['DataRequirementSort'] = Field(default_factory=list)
    'Specifies the order of the results to be returned.'
    limit: Optional_['positiveInt'] = Field(default=None)
    'Specifies a maximum number of results that are required (uses the _count search parameter).'
    profile: List_['canonical'] = Field(default_factory=list)
    'The profile of the required data, specified as the uri of the profile definition.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    subjectCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The intended subjects of the data requirement. If this element is not provided, a Patient subject is assumed.'
    subjectReference: Optional_['Reference'] = Field(default=None)
    'The intended subjects of the data requirement. If this element is not provided, a Patient subject is assumed.'
    codeFilter: List_['DataRequirementCodeFilter'] = Field(default_factory=list)
    "Code filters specify additional constraints on the data, specifying the value set of interest for a particular element of the data. Each code filter defines an additional constraint on the data, i.e. code filters are AND'ed, not OR'ed."
    dateFilter: List_['DataRequirementDateFilter'] = Field(default_factory=list)
    "Date filters specify additional constraints on the data in terms of the applicable date range for specific elements. Each date filter specifies an additional constraint on the data, i.e. date filters are AND'ed, not OR'ed."
    mustSupport: List_['string'] = Field(default_factory=list)
    'Indicates that specific elements of the type are referenced by the knowledge module and must be supported by the consumer in order to obtain an effective evaluation. This does not mean that a value is required for this element, only that the consuming system must understand the element and be able to provide values for it if they are available. \n\nThe value of mustSupport SHALL be a FHIRPath resolvable on the type of the DataRequirement. The path SHALL consist only of identifiers, constant indexers, and .resolve() (see the [Simple FHIRPath Profile](fhirpath.html#simple) for full details).'
    valueFilter: List_['DataRequirementValueFilter'] = Field(default_factory=list)
    "Value filters specify additional constraints on the data for elements other than code-valued or date-valued. Each value filter specifies an additional constraint on the data (i.e. valueFilters are AND'ed, not OR'ed)."


class DataType(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The base class for all re-useable types defined as part of the FHIR Specification."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class Distance(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A length - a value with a unit that is a physical distance."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    unit: Optional_['string'] = Field(default=None)
    'A human-readable form of the unit.'
    value: Optional_['decimal'] = Field(default=None)
    'The value of the measured amount. The value includes an implicit precision in the presentation of the value.'
    system: Optional_['uri'] = Field(default=None)
    'The identification of the system that provides the coded form of the unit.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    comparator: Optional_['code'] = Field(default=None)
    'How the value should be understood and represented - whether the actual value is greater or less than the stated value due to measurement issues; e.g. if the comparator is "<" , then the real value is < stated value.'
    code: Optional_['code'] = Field(default=None)
    'A computer processable form of the unit in some unit representation system.'


class DosageDoseAndRate(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Depending on the resource,this is the amount of medication administered, to  be administered or typical amount to be administered."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The kind of dose or rate specified, for example, ordered or calculated.'
    doseRange: Optional_['Range'] = Field(default=None)
    'Amount of medication per dose.'
    doseQuantity: Optional_['Quantity'] = Field(default=None)
    'Amount of medication per dose.'
    rateRatio: Optional_['Ratio'] = Field(default=None)
    'Amount of medication per unit of time.'
    rateRange: Optional_['Range'] = Field(default=None)
    'Amount of medication per unit of time.'
    rateQuantity: Optional_['Quantity'] = Field(default=None)
    'Amount of medication per unit of time.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class Dosage(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates how the medication is/was taken or should be taken by the patient."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    text: Optional_['string'] = Field(default=None)
    'Free text dosage instructions e.g. SIG.'
    site: Optional_['CodeableConcept'] = Field(default=None)
    'Body site to administer to.'
    route: Optional_['CodeableConcept'] = Field(default=None)
    'How drug should enter body.'
    timing: Optional_['Timing'] = Field(default=None)
    'When medication should be administered.'
    method: Optional_['CodeableConcept'] = Field(default=None)
    'Technique for administering medication.'
    sequence: Optional_['integer'] = Field(default=None)
    'Indicates the order in which the dosage instructions should be applied or interpreted.'
    asNeeded: Optional_['boolean'] = Field(default=None)
    'Indicates whether the Medication is only taken when needed within a specific dosing schedule (Boolean option).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    asNeededFor: List_['CodeableConcept'] = Field(default_factory=list)
    'Indicates whether the Medication is only taken based on a precondition for taking the Medication (CodeableConcept).'
    doseAndRate: List_['DosageDoseAndRate'] = Field(default_factory=list)
    'Depending on the resource,this is the amount of medication administered, to  be administered or typical amount to be administered.'
    maxDosePerPeriod: List_['Ratio'] = Field(default_factory=list)
    'Upper limit on medication per unit of time.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    patientInstruction: Optional_['string'] = Field(default=None)
    'Instructions in terms that are understood by the patient or consumer.'
    maxDosePerLifetime: Optional_['Quantity'] = Field(default=None)
    'Upper limit on medication per lifetime of the patient.'
    additionalInstruction: List_['CodeableConcept'] = Field(default_factory=list)
    'Supplemental instructions to the patient on how to take the medication  (e.g. "with meals" or"take half to one hour before food") or warnings for the patient about the medication (e.g. "may cause drowsiness" or "avoid exposure of skin to direct sunlight or sunlamps").'
    maxDosePerAdministration: Optional_['Quantity'] = Field(default=None)
    'Upper limit on medication per administration.'


class Duration(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A length of time."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    unit: Optional_['string'] = Field(default=None)
    'A human-readable form of the unit.'
    value: Optional_['decimal'] = Field(default=None)
    'The value of the measured amount. The value includes an implicit precision in the presentation of the value.'
    system: Optional_['uri'] = Field(default=None)
    'The identification of the system that provides the coded form of the unit.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    comparator: Optional_['code'] = Field(default=None)
    'How the value should be understood and represented - whether the actual value is greater or less than the stated value due to measurement issues; e.g. if the comparator is "<" , then the real value is < stated value.'
    code: Optional_['code'] = Field(default=None)
    'A computer processable form of the unit in some unit representation system.'


class ElementDefinitionConstraint(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Formal constraints such as co-occurrence and other constraints that can be computationally evaluated within the context of the instance."""
    key: 'id' = Field(default=...)
    'Allows identification of which elements have their cardinalities impacted by the constraint.  Will not be referenced for constraints that do not affect cardinality.'
    human: 'string' = Field(default=...)
    'Text that can be used to describe the constraint in messages identifying that the constraint has been violated.'
    source: Optional_['canonical'] = Field(default=None)
    'A reference to the original source of the constraint, for traceability purposes.'
    severity: 'code' = Field(default=...)
    'Identifies the impact constraint violation has on the conformance of the instance.'
    suppress: Optional_['boolean'] = Field(default=None)
    'If true, indicates that the warning or best practice guideline should be suppressed.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    expression: Optional_['string'] = Field(default=None)
    'A [FHIRPath](fhirpath.html) expression of constraint that can be executed to see if this constraint is met.'
    requirements: Optional_['markdown'] = Field(default=None)
    'Description of why this constraint is necessary or appropriate.'
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'


class ElementDefinitionMapping(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Identifies a concept from an external specification that roughly corresponds to this element."""
    map: 'string' = Field(default=...)
    'Expresses what part of the target specification corresponds to this element.'
    comment: Optional_['markdown'] = Field(default=None)
    'Comments that provide information about the mapping or its use.'
    identity: 'id' = Field(default=...)
    'An internal reference to the definition of a mapping.'
    language: Optional_['code'] = Field(default=None)
    'Identifies the computable language in which mapping.map is expressed.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'


class ElementDefinitionBinding(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Binds to a value set if this element is coded (code, Coding, CodeableConcept, Quantity), or the data types (string, uri)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    strength: 'code' = Field(default=...)
    'Indicates the degree of conformance expectations associated with this binding - that is, the degree to which the provided value set must be adhered to in the instances.'
    valueSet: Optional_['canonical'] = Field(default=None)
    'Refers to the value set that identifies the set of codes the binding refers to.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    additional: List_['ElementDefinitionBindingAdditional'] = Field(default_factory=list)
    'Additional bindings that help applications implementing this element. Additional bindings do not replace the main binding but provide more information and/or context.'
    description: Optional_['markdown'] = Field(default=None)
    'Describes the intended use of this particular set of codes.'


class ElementDefinitionBindingAdditional(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Additional bindings that help applications implementing this element. Additional bindings do not replace the main binding but provide more information and/or context."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    any: Optional_['boolean'] = Field(default=None)
    'Whether the binding applies to all repeats, or just to any one of them. This is only relevant for elements that can repeat.'
    usage: List_['UsageContext'] = Field(default_factory=list)
    'Qualifies the usage of the binding. Typically bindings are qualified by jurisdiction, but they may also be qualified by gender, workflow status, clinical domain etc. The information to decide whether a usege context applies is usually outside the resource, determined by context, and this might present challenges for validation tooling.'
    purpose: 'code' = Field(default=...)
    'The use of this additional binding.'
    valueSet: 'canonical' = Field(default=...)
    'The valueSet that is being bound for the purpose.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    shortDoco: Optional_['string'] = Field(default=None)
    'Concise documentation - for summary tables.'
    documentation: Optional_['markdown'] = Field(default=None)
    'Documentation of the purpose of use of the bindingproviding additional information about how it is intended to be used.'


class ElementDefinitionExample(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A sample value for this element demonstrating the type of information that would typically be found in the element."""
    label: 'string' = Field(default=...)
    'Describes the purpose of this example among the set of examples.'
    valueBase64Binary: Optional_['base64Binary'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueBoolean: Optional_['boolean'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueCanonical: Optional_['canonical'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueCode: Optional_['code'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueDate: Optional_['date'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueDateTime: Optional_['dateTime'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueDecimal: Optional_['decimal'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueId: Optional_['id'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueInstant: Optional_['instant'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueInteger: Optional_['integer'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueInteger64: Optional_['integer64'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueMarkdown: Optional_['markdown'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueOid: Optional_['oid'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valuePositiveInt: Optional_['positiveInt'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueString: Optional_['string'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueTime: Optional_['time'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueUnsignedInt: Optional_['unsignedInt'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueUri: Optional_['uri'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueUrl: Optional_['url'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueUuid: Optional_['uuid'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueAddress: Optional_['Address'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueAge: Optional_['Age'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueAnnotation: Optional_['Annotation'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueAttachment: Optional_['Attachment'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueCodeableReference: Optional_['CodeableReference'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueCoding: Optional_['Coding'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueContactPoint: Optional_['ContactPoint'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueCount: Optional_['Count'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueDistance: Optional_['Distance'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueDuration: Optional_['Duration'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueHumanName: Optional_['HumanName'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueIdentifier: Optional_['Identifier'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueMoney: Optional_['Money'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valuePeriod: Optional_['Period'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueRange: Optional_['Range'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueRatio: Optional_['Ratio'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueRatioRange: Optional_['RatioRange'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueReference: Optional_['Reference'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueSampledData: Optional_['SampledData'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueSignature: Optional_['Signature'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueTiming: Optional_['Timing'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueContactDetail: Optional_['ContactDetail'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueDataRequirement: Optional_['DataRequirement'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueExpression: Optional_['Expression'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueParameterDefinition: Optional_['ParameterDefinition'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueRelatedArtifact: Optional_['RelatedArtifact'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueTriggerDefinition: Optional_['TriggerDefinition'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueUsageContext: Optional_['UsageContext'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueAvailability: Optional_['Availability'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueExtendedContactDetail: Optional_['ExtendedContactDetail'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueDosage: Optional_['Dosage'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    valueMeta: Optional_['Meta'] = Field(default=None)
    'The actual value for the element, which must be one of the types allowed for this element.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'


class ElementDefinitionSlicing(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates that the element is sliced into a set of alternative definitions (i.e. in a structure definition, there are multiple different constraints on a single element in the base resource). Slicing can be used in any resource that has cardinality ..* on the base resource, or any resource with a choice of types. The set of slices is any elements that come after this in the element sequence that have the same path, until a shorter path occurs (the shorter path terminates the set)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    rules: 'code' = Field(default=...)
    'Whether additional slices are allowed or not. When the slices are ordered, profile authors can also say that additional slices are only allowed at the end.'
    ordered: Optional_['boolean'] = Field(default=None)
    'If the matching elements have to occur in the same order as defined in the profile.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    description: Optional_['string'] = Field(default=None)
    'A human-readable text description of how the slicing works. If there is no discriminator, this is required to be present to provide whatever information is possible about how the slices can be differentiated.'
    discriminator: List_['ElementDefinitionSlicingDiscriminator'] = Field(default_factory=list)
    'Designates which child elements are used to discriminate between the slices when processing an instance. If one or more discriminators are provided, the value of the child elements in the instance data SHALL completely distinguish which slice the element in the resource matches based on the allowed values for those elements in each of the slices.'


class ElementDefinitionSlicingDiscriminator(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Designates which child elements are used to discriminate between the slices when processing an instance. If one or more discriminators are provided, the value of the child elements in the instance data SHALL completely distinguish which slice the element in the resource matches based on the allowed values for those elements in each of the slices."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'How the element value is interpreted when discrimination is evaluated.'
    path: 'string' = Field(default=...)
    'A FHIRPath expression, using [the simple subset of FHIRPath](fhirpath.html#simple), that is used to identify the element on which discrimination is based.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class ElementDefinitionType(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The data type or resource that the value of this element is permitted to be."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    profile: List_['canonical'] = Field(default_factory=list)
    'Identifies a profile structure or implementation Guide that applies to the datatype this element refers to. If any profiles are specified, then the content must conform to at least one of them. The URL can be a local reference - to a contained StructureDefinition, or a reference to another StructureDefinition or Implementation Guide by a canonical URL. When an implementation guide is specified, the type SHALL conform to at least one profile defined in the implementation guide.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    versioning: Optional_['code'] = Field(default=None)
    'Whether this reference needs to be version specific or version independent, or whether either can be used.'
    aggregation: List_['code'] = Field(default_factory=list)
    'If the type is a reference to another resource, how the resource is or can be aggregated - is it a contained resource, or a reference, and if the context is a bundle, is it included in the bundle.'
    targetProfile: List_['canonical'] = Field(default_factory=list)
    'Used when the type is "Reference" or "canonical", and identifies a profile structure or implementation Guide that applies to the target of the reference this element refers to. If any profiles are specified, then the content must conform to at least one of them. The URL can be a local reference - to a contained StructureDefinition, or a reference to another StructureDefinition or Implementation Guide by a canonical URL. When an implementation guide is specified, the target resource SHALL conform to at least one profile defined in the implementation guide.'
    code: 'uri' = Field(default=...)
    'URL of Data type or Resource that is a(or the) type used for this element. References are URLs that are relative to http://hl7.org/fhir/StructureDefinition e.g. "string" is a reference to http://hl7.org/fhir/StructureDefinition/string. Absolute URLs are only allowed in logical models.'


class ElementDefinitionBase(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information about the base definition of the element, provided to make it unnecessary for tools to trace the deviation of the element through the derived and related profiles. When the element definition is not the original definition of an element - e.g. either in a constraint on another type, or for elements from a super type in a snap shot - then the information in provided in the element definition may be different to the base definition. On the original definition of the element, it will be same."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    min: 'unsignedInt' = Field(default=...)
    'Minimum cardinality of the base element identified by the path.'
    max: 'string' = Field(default=...)
    'Maximum cardinality of the base element identified by the path.'
    path: 'string' = Field(default=...)
    'The Path that identifies the base element - this matches the ElementDefinition.path for that element. Across FHIR, there is only one base definition of any element - that is, an element definition on a [StructureDefinition](structuredefinition.html#) without a StructureDefinition.base.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class ElementDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Captures constraints on each element within the resource, profile, or extension."""
    min: Optional_['unsignedInt'] = Field(default=None)
    'The minimum number of times this element SHALL appear in the instance.'
    max: Optional_['string'] = Field(default=None)
    'The maximum number of times this element is permitted to appear in the instance.'
    path: 'string' = Field(default=...)
    'The path identifies the element and is expressed as a "."-separated list of ancestor elements, beginning with the name of the resource or extension.'
    base: Optional_['ElementDefinitionBase'] = Field(default=None)
    'Information about the base definition of the element, provided to make it unnecessary for tools to trace the deviation of the element through the derived and related profiles. When the element definition is not the original definition of an element - e.g. either in a constraint on another type, or for elements from a super type in a snap shot - then the information in provided in the element definition may be different to the base definition. On the original definition of the element, it will be same.'
    type: List_['ElementDefinitionType'] = Field(default_factory=list)
    'The data type or resource that the value of this element is permitted to be.'
    label: Optional_['string'] = Field(default=None)
    'A single preferred label which is the text to display beside the element indicating its meaning or to use to prompt for the element in a user display or form.'
    short: Optional_['string'] = Field(default=None)
    'A concise description of what this element means (e.g. for use in autogenerated summaries).'
    alias: List_['string'] = Field(default_factory=list)
    'Identifies additional names by which this element might also be known.'
    slicing: Optional_['ElementDefinitionSlicing'] = Field(default=None)
    'Indicates that the element is sliced into a set of alternative definitions (i.e. in a structure definition, there are multiple different constraints on a single element in the base resource). Slicing can be used in any resource that has cardinality ..* on the base resource, or any resource with a choice of types. The set of slices is any elements that come after this in the element sequence that have the same path, until a shorter path occurs (the shorter path terminates the set).'
    comment: Optional_['markdown'] = Field(default=None)
    'Explanatory notes and implementation guidance about the data element, including notes about how to use the data properly, exceptions to proper use, etc. (Note: The text you are reading is specified in ElementDefinition.comment).'
    example: List_['ElementDefinitionExample'] = Field(default_factory=list)
    'A sample value for this element demonstrating the type of information that would typically be found in the element.'
    binding: Optional_['ElementDefinitionBinding'] = Field(default=None)
    'Binds to a value set if this element is coded (code, Coding, CodeableConcept, Quantity), or the data types (string, uri).'
    mapping: List_['ElementDefinitionMapping'] = Field(default_factory=list)
    'Identifies a concept from an external specification that roughly corresponds to this element.'
    fixedBase64Binary: Optional_['base64Binary'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedBoolean: Optional_['boolean'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedCanonical: Optional_['canonical'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedCode: Optional_['code'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedDate: Optional_['date'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedDateTime: Optional_['dateTime'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedDecimal: Optional_['decimal'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedId: Optional_['id'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedInstant: Optional_['instant'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedInteger: Optional_['integer'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedInteger64: Optional_['integer64'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedMarkdown: Optional_['markdown'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedOid: Optional_['oid'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedPositiveInt: Optional_['positiveInt'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedString: Optional_['string'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedTime: Optional_['time'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedUnsignedInt: Optional_['unsignedInt'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedUri: Optional_['uri'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedUrl: Optional_['url'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedUuid: Optional_['uuid'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedAddress: Optional_['Address'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedAge: Optional_['Age'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedAnnotation: Optional_['Annotation'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedAttachment: Optional_['Attachment'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedCodeableReference: Optional_['CodeableReference'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedCoding: Optional_['Coding'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedContactPoint: Optional_['ContactPoint'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedCount: Optional_['Count'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedDistance: Optional_['Distance'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedDuration: Optional_['Duration'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedHumanName: Optional_['HumanName'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedIdentifier: Optional_['Identifier'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedMoney: Optional_['Money'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedPeriod: Optional_['Period'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedQuantity: Optional_['Quantity'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedRange: Optional_['Range'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedRatio: Optional_['Ratio'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedRatioRange: Optional_['RatioRange'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedReference: Optional_['Reference'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedSampledData: Optional_['SampledData'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedSignature: Optional_['Signature'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedTiming: Optional_['Timing'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedContactDetail: Optional_['ContactDetail'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedDataRequirement: Optional_['DataRequirement'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedExpression: Optional_['Expression'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedParameterDefinition: Optional_['ParameterDefinition'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedRelatedArtifact: Optional_['RelatedArtifact'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedTriggerDefinition: Optional_['TriggerDefinition'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedUsageContext: Optional_['UsageContext'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedAvailability: Optional_['Availability'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedExtendedContactDetail: Optional_['ExtendedContactDetail'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedDosage: Optional_['Dosage'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    fixedMeta: Optional_['Meta'] = Field(default=None)
    'Specifies a value that SHALL be exactly the value  for this element in the instance, if present. For purposes of comparison, non-significant whitespace is ignored, and all values must be an exact match (case and accent sensitive). Missing elements/attributes must also be missing.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    sliceName: Optional_['string'] = Field(default=None)
    'The name of this element definition slice, when slicing is working. The name must be a token with no dots or spaces. This is a unique name referring to a specific set of constraints applied to this element, used to provide a name to different slices of the same element.'
    maxLength: Optional_['integer'] = Field(default=None)
    'Indicates the maximum length in characters that is permitted to be present in conformant instances and which is expected to be supported by conformant consumers that support the element. ```maxLength``` SHOULD only be used on primitive data types that have a string representation (see [http://hl7.org/fhir/StructureDefinition/structuredefinition-type-characteristics](http://hl7.org/fhir/extensions/StructureDefinition-structuredefinition-type-characteristics.html)).'
    condition: List_['id'] = Field(default_factory=list)
    'A reference to an invariant that may make additional statements about the cardinality or value in the instance.'
    isSummary: Optional_['boolean'] = Field(default=None)
    'Whether the element should be included if a client requests a search with the parameter _summary=true.'
    definition: Optional_['markdown'] = Field(default=None)
    'Provides a complete explanation of the meaning of the data element for human readability.  For the case of elements derived from existing elements (e.g. constraints), the definition SHALL be consistent with the base definition, but convey the meaning of the element in the particular context of use of the resource. (Note: The text you are reading is specified in ElementDefinition.definition).'
    patternBase64Binary: Optional_['base64Binary'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternBoolean: Optional_['boolean'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternCanonical: Optional_['canonical'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternCode: Optional_['code'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternDate: Optional_['date'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternDateTime: Optional_['dateTime'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternDecimal: Optional_['decimal'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternId: Optional_['id'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternInstant: Optional_['instant'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternInteger: Optional_['integer'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternInteger64: Optional_['integer64'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternMarkdown: Optional_['markdown'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternOid: Optional_['oid'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternPositiveInt: Optional_['positiveInt'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternString: Optional_['string'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternTime: Optional_['time'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternUnsignedInt: Optional_['unsignedInt'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternUri: Optional_['uri'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternUrl: Optional_['url'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternUuid: Optional_['uuid'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternAddress: Optional_['Address'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternAge: Optional_['Age'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternAnnotation: Optional_['Annotation'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternAttachment: Optional_['Attachment'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternCodeableReference: Optional_['CodeableReference'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternCoding: Optional_['Coding'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternContactPoint: Optional_['ContactPoint'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternCount: Optional_['Count'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternDistance: Optional_['Distance'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternDuration: Optional_['Duration'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternHumanName: Optional_['HumanName'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternIdentifier: Optional_['Identifier'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternMoney: Optional_['Money'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternPeriod: Optional_['Period'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternQuantity: Optional_['Quantity'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternRange: Optional_['Range'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternRatio: Optional_['Ratio'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternRatioRange: Optional_['RatioRange'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternReference: Optional_['Reference'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternSampledData: Optional_['SampledData'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternSignature: Optional_['Signature'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternTiming: Optional_['Timing'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternContactDetail: Optional_['ContactDetail'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternDataRequirement: Optional_['DataRequirement'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternExpression: Optional_['Expression'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternParameterDefinition: Optional_['ParameterDefinition'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternRelatedArtifact: Optional_['RelatedArtifact'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternTriggerDefinition: Optional_['TriggerDefinition'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternUsageContext: Optional_['UsageContext'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternAvailability: Optional_['Availability'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternExtendedContactDetail: Optional_['ExtendedContactDetail'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternDosage: Optional_['Dosage'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    patternMeta: Optional_['Meta'] = Field(default=None)
    'Specifies a value that each occurrence of the element in the instance SHALL follow - that is, any value in the pattern must be found in the instance, if the element has a value. Other additional values may be found too. This is effectively constraint by example.  \n\nWhen pattern[x] is used to constrain a primitive, it means that the value provided in the pattern[x] must match the instance value exactly.\n\nWhen an element within a pattern[x] is used to constrain an array, it means that each element provided in the pattern[x] must (recursively) match at least one element from the instance array.\n\nWhen pattern[x] is used to constrain a complex object, it means that each property in the pattern must be present in the complex object, and its value must recursively match -- i.e.,\n\n1. If primitive: it must match exactly the pattern value\n2. If a complex object: it must match (recursively) the pattern value\n3. If an array: it must match (recursively) the pattern value\n\nIf a pattern[x] is declared on a repeating element, the pattern applies to all repetitions.  If the desire is for a pattern to apply to only one element or a subset of elements, slicing must be used. See [Examples of Patterns](elementdefinition-examples.html#pattern-examples) for examples of pattern usage and the effect it will have.'
    constraint: List_['ElementDefinitionConstraint'] = Field(default_factory=list)
    'Formal constraints such as co-occurrence and other constraints that can be computationally evaluated within the context of the instance.'
    isModifier: Optional_['boolean'] = Field(default=None)
    'If true, the value of this element affects the interpretation of the element or resource that contains it, and the value of the element cannot be ignored. Typically, this is used for status, negation and qualification codes. The effect of this is that the element cannot be ignored by systems: they SHALL either recognize the element and process it, and/or a pre-determination has been made that it is not relevant to their particular system. When used on the root element in an extension definition, this indicates whether or not the extension is a modifier extension.'
    minValueDate: Optional_['date'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    minValueDateTime: Optional_['dateTime'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    minValueInstant: Optional_['instant'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    minValueTime: Optional_['time'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    minValueDecimal: Optional_['decimal'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    minValueInteger: Optional_['integer'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    minValueInteger64: Optional_['integer64'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    minValuePositiveInt: Optional_['positiveInt'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    minValueUnsignedInt: Optional_['unsignedInt'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    minValueQuantity: Optional_['Quantity'] = Field(default=None)
    'The minimum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValueDate: Optional_['date'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValueDateTime: Optional_['dateTime'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValueInstant: Optional_['instant'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValueTime: Optional_['time'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValueDecimal: Optional_['decimal'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValueInteger: Optional_['integer'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValueInteger64: Optional_['integer64'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValuePositiveInt: Optional_['positiveInt'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValueUnsignedInt: Optional_['unsignedInt'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    maxValueQuantity: Optional_['Quantity'] = Field(default=None)
    'The maximum allowed value for the element. The value is inclusive. This is allowed for the types date, dateTime, instant, time, decimal, integer, and Quantity.'
    mustSupport: Optional_['boolean'] = Field(default=None)
    'If true, implementations that produce or consume resources SHALL provide "support" for the element in some meaningful way. Note that this is being phased out and replaced by obligations (see below).  If false, the element may be ignored and not supported. If false, whether to populate or use the data element in any way is at the discretion of the implementation.'
    requirements: Optional_['markdown'] = Field(default=None)
    'This element is for traceability of why the element was created and why the constraints exist as they do. This may be used to point to source materials or specifications that drove the structure of this element.'
    orderMeaning: Optional_['string'] = Field(default=None)
    'If present, indicates that the order of the repeating element has meaning and describes what that meaning is.  If absent, it means that the order of the element has no meaning.'
    mustHaveValue: Optional_['boolean'] = Field(default=None)
    'Specifies for a primitive data type that the value of the data type cannot be replaced by an extension.'
    representation: List_['code'] = Field(default_factory=list)
    "Codes that define how this element is represented in instances, when the deviation varies from the normal case. No extensions are allowed on elements with a representation of 'xmlAttr', no matter what FHIR serialization format is used."
    defaultValueBase64Binary: Optional_['base64Binary'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueBoolean: Optional_['boolean'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueCanonical: Optional_['canonical'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueCode: Optional_['code'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueDate: Optional_['date'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueDateTime: Optional_['dateTime'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueDecimal: Optional_['decimal'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueId: Optional_['id'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueInstant: Optional_['instant'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueInteger: Optional_['integer'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueInteger64: Optional_['integer64'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueMarkdown: Optional_['markdown'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueOid: Optional_['oid'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValuePositiveInt: Optional_['positiveInt'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueString: Optional_['string'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueTime: Optional_['time'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueUnsignedInt: Optional_['unsignedInt'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueUri: Optional_['uri'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueUrl: Optional_['url'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueUuid: Optional_['uuid'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueAddress: Optional_['Address'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueAge: Optional_['Age'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueAnnotation: Optional_['Annotation'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueAttachment: Optional_['Attachment'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueCodeableReference: Optional_['CodeableReference'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueCoding: Optional_['Coding'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueContactPoint: Optional_['ContactPoint'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueCount: Optional_['Count'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueDistance: Optional_['Distance'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueDuration: Optional_['Duration'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueHumanName: Optional_['HumanName'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueIdentifier: Optional_['Identifier'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueMoney: Optional_['Money'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValuePeriod: Optional_['Period'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueQuantity: Optional_['Quantity'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueRange: Optional_['Range'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueRatio: Optional_['Ratio'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueRatioRange: Optional_['RatioRange'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueReference: Optional_['Reference'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueSampledData: Optional_['SampledData'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueSignature: Optional_['Signature'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueTiming: Optional_['Timing'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueContactDetail: Optional_['ContactDetail'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueDataRequirement: Optional_['DataRequirement'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueExpression: Optional_['Expression'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueParameterDefinition: Optional_['ParameterDefinition'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueRelatedArtifact: Optional_['RelatedArtifact'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueTriggerDefinition: Optional_['TriggerDefinition'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueUsageContext: Optional_['UsageContext'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueAvailability: Optional_['Availability'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueExtendedContactDetail: Optional_['ExtendedContactDetail'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueDosage: Optional_['Dosage'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    defaultValueMeta: Optional_['Meta'] = Field(default=None)
    "The value that should be used if there is no value stated in the instance (e.g. 'if not otherwise specified, the abstract is false')."
    contentReference: Optional_['uri'] = Field(default=None)
    'Identifies an element defined elsewhere in the definition whose content rules should be applied to the current element. ContentReferences bring across all the rules that are in the ElementDefinition for the element, including definitions, cardinality constraints, bindings, invariants etc.'
    isModifierReason: Optional_['string'] = Field(default=None)
    'Explains how that element affects the interpretation of the resource or element that contains it.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    valueAlternatives: List_['canonical'] = Field(default_factory=list)
    'Specifies a list of extensions that can appear in place of a primitive value.'
    meaningWhenMissing: Optional_['markdown'] = Field(default=None)
    "The Implicit meaning that is to be understood when this element is missing (e.g. 'when this element is missing, the period is ongoing')."
    sliceIsConstraining: Optional_['boolean'] = Field(default=None)
    'If true, indicates that this slice definition is constraining a slice definition with the same name in an inherited profile. If false, the slice is not overriding any slice in an inherited profile. If missing, the slice might or might not be overriding a slice in an inherited profile, depending on the sliceName.'
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: List_['Coding'] = Field(default_factory=list)
    'A code that has the same meaning as the element in a particular terminology.'


class Expression(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A expression that is evaluated in a specified context and returns a value. The context of use of the expression must specify the context in which the expression is evaluated, and how the result of the expression is used."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    name: Optional_['code'] = Field(default=None)
    'A short name assigned to the expression to allow for multiple reuse of the expression in the context where it is defined.'
    language: Optional_['code'] = Field(default=None)
    'The media type of the language for the expression.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: Optional_['uri'] = Field(default=None)
    'A URI that defines where the expression is found.'
    expression: Optional_['string'] = Field(default=None)
    'An expression in the specified language that returns a value.'
    description: Optional_['string'] = Field(default=None)
    'A brief, natural language description of the condition that effectively communicates the intended semantics.'


class ExtendedContactDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specifies contact information for a specific purpose over a period of time, might be handled/monitored by a specific named person or organization."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    name: List_['HumanName'] = Field(default_factory=list)
    'The name of an individual to contact, some types of contact detail are usually blank.'
    period: Optional_['Period'] = Field(default=None)
    'Period that this contact was valid for usage.'
    purpose: Optional_['CodeableConcept'] = Field(default=None)
    'The purpose/type of contact.'
    telecom: List_['ContactPoint'] = Field(default_factory=list)
    'The contact details application for the purpose defined.'
    address: Optional_['Address'] = Field(default=None)
    'Address for the contact.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    organization: Optional_['Reference'] = Field(default=None)
    'This contact detail is handled/monitored by a specific organization. If the name is provided in the contact, then it is referring to the named individual within this organization.'


class Extension(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Optional Extension Element - found in all resources."""
    valueBase64Binary: Optional_['base64Binary'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueBoolean: Optional_['boolean'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueCanonical: Optional_['canonical'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueCode: Optional_['code'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueDate: Optional_['date'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueDateTime: Optional_['dateTime'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueDecimal: Optional_['decimal'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueId: Optional_['id'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueInstant: Optional_['instant'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueInteger: Optional_['integer'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueInteger64: Optional_['integer64'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueMarkdown: Optional_['markdown'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueOid: Optional_['oid'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valuePositiveInt: Optional_['positiveInt'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueString: Optional_['string'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueTime: Optional_['time'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueUnsignedInt: Optional_['unsignedInt'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueUri: Optional_['uri'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueUrl: Optional_['url'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueUuid: Optional_['uuid'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueAddress: Optional_['Address'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueAge: Optional_['Age'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueAnnotation: Optional_['Annotation'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueAttachment: Optional_['Attachment'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueCodeableReference: Optional_['CodeableReference'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueCoding: Optional_['Coding'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueContactPoint: Optional_['ContactPoint'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueCount: Optional_['Count'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueDistance: Optional_['Distance'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueDuration: Optional_['Duration'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueHumanName: Optional_['HumanName'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueIdentifier: Optional_['Identifier'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueMoney: Optional_['Money'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valuePeriod: Optional_['Period'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueRange: Optional_['Range'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueRatio: Optional_['Ratio'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueRatioRange: Optional_['RatioRange'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueReference: Optional_['Reference'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueSampledData: Optional_['SampledData'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueSignature: Optional_['Signature'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueTiming: Optional_['Timing'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueContactDetail: Optional_['ContactDetail'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueDataRequirement: Optional_['DataRequirement'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueExpression: Optional_['Expression'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueParameterDefinition: Optional_['ParameterDefinition'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueRelatedArtifact: Optional_['RelatedArtifact'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueTriggerDefinition: Optional_['TriggerDefinition'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueUsageContext: Optional_['UsageContext'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueAvailability: Optional_['Availability'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueExtendedContactDetail: Optional_['ExtendedContactDetail'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueDosage: Optional_['Dosage'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    valueMeta: Optional_['Meta'] = Field(default=None)
    'Value of extension - must be one of a constrained set of the data types (see [Extensibility](extensibility.html) for a list).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    url: 'str' = Field(default=...)
    'Source of the definition for the extension code - a logical name or a URL.'


class HumanName(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A name, normally of a human, that can be used for other living entities (e.g. animals but not organizations) that have been assigned names by a human and may need the use of name parts or the need for usage information."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    use: Optional_['code'] = Field(default=None)
    'Identifies the purpose for this name.'
    text: Optional_['string'] = Field(default=None)
    'Specifies the entire name as it should be displayed e.g. on an application UI. This may be provided instead of or as well as the specific parts.'
    given: List_['string'] = Field(default_factory=list)
    'Given name.'
    family: Optional_['string'] = Field(default=None)
    'The part of a name that links to the genealogy. In some cultures (e.g. Eritrea) the family name of a son is the first name of his father.'
    prefix: List_['string'] = Field(default_factory=list)
    'Part of the name that is acquired as a title due to academic, legal, employment or nobility status, etc. and that appears at the start of the name.'
    suffix: List_['string'] = Field(default_factory=list)
    'Part of the name that is acquired as a title due to academic, legal, employment or nobility status, etc. and that appears at the end of the name.'
    period: Optional_['Period'] = Field(default=None)
    'Indicates the period of time when this name was valid for the named person.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class Identifier(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An identifier - identifies some entity uniquely and unambiguously. Typically this is used for business identifiers."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    use: Optional_['code'] = Field(default=None)
    'The purpose of this identifier.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'A coded type for the identifier that can be used to determine which identifier to use for a specific purpose.'
    value: Optional_['string'] = Field(default=None)
    'The portion of the identifier typically relevant to the user and which is unique within the context of the system.'
    system: Optional_['uri'] = Field(default=None)
    'Establishes the namespace for the value - that is, an absolute URL that describes a set values that are unique.'
    period: Optional_['Period'] = Field(default=None)
    'Time period during which identifier is/was valid for use.'
    assigner: Optional_['Reference'] = Field(default=None)
    'Organization that issued/manages the identifier.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class MarketingStatus(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The marketing status describes the date when a medicinal product is actually put on the market or the date as of which it is no longer available."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    status: 'CodeableConcept' = Field(default=...)
    'This attribute provides information on the status of the marketing of the medicinal product See ISO/TS 20443 for more information and examples.'
    country: Optional_['CodeableConcept'] = Field(default=None)
    'The country in which the marketing authorization has been granted shall be specified It should be specified using the ISO 3166 ‑ 1 alpha-2 code elements.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    dateRange: Optional_['Period'] = Field(default=None)
    'The date when the Medicinal Product is placed on the market by the Marketing Authorization Holder (or where applicable, the manufacturer/distributor) in a country and/or jurisdiction shall be provided A complete date consisting of day, month and year shall be specified using the ISO 8601 date format NOTE “Placed on the market” refers to the release of the Medicinal Product into the distribution chain.'
    restoreDate: Optional_['dateTime'] = Field(default=None)
    'The date when the Medicinal Product is placed on the market by the Marketing Authorization Holder (or where applicable, the manufacturer/distributor) in a country and/or jurisdiction shall be provided A complete date consisting of day, month and year shall be specified using the ISO 8601 date format NOTE “Placed on the market” refers to the release of the Medicinal Product into the distribution chain.'
    jurisdiction: Optional_['CodeableConcept'] = Field(default=None)
    'Where a Medicines Regulatory Agency has granted a marketing authorization for which specific provisions within a jurisdiction apply, the jurisdiction can be specified using an appropriate controlled terminology The controlled term and the controlled term identifier shall be specified.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Meta(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The metadata about a resource. This is content in the resource that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource."""
    tag: List_['Coding'] = Field(default_factory=list)
    'Tags applied to this resource. Tags are intended to be used to identify and relate resources to process and workflow, and applications are not required to consider the tags when interpreting the meaning of a resource.'
    source: Optional_['uri'] = Field(default=None)
    'A uri that identifies the source system of the resource. This provides a minimal amount of [Provenance](provenance.html#) information that can be used to track or differentiate the source of information in the resource. The source may identify another FHIR server, document, message, database, etc.'
    profile: List_['canonical'] = Field(default_factory=list)
    'A list of profiles (references to [StructureDefinition](structuredefinition.html#) resources) that this resource claims to conform to. The URL is a reference to [StructureDefinition.url](structuredefinition-definitions.html#StructureDefinition.url).'
    security: List_['Coding'] = Field(default_factory=list)
    'Security labels applied to this resource. These tags connect specific resources to the overall security policy and infrastructure.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    versionId: Optional_['id'] = Field(default=None)
    'The version specific identifier, as it appears in the version portion of the URL. This value changes when the resource is created, updated, or deleted.'
    lastUpdated: Optional_['instant'] = Field(default=None)
    'When the resource last changed - e.g. when the version changed.'
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'


class MonetaryComponent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Availability data for an {item}."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'base | surcharge | deduction | discount | tax | informational.'
    factor: Optional_['decimal'] = Field(default=None)
    'Factor used for calculating this component.'
    amount: Optional_['Money'] = Field(default=None)
    'Explicit value amount to be used.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'Codes may be used to differentiate between kinds of taxes, surcharges, discounts etc.'


class Money(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An amount of economic utility in some recognized currency."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    value: Optional_['decimal'] = Field(default=None)
    'Numerical value (with implicit precision).'
    currency: Optional_['code'] = Field(default=None)
    'ISO 4217 Currency Code.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class Narrative(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A human-readable summary of the resource conveying the essential clinical and business information for the resource."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    div: 'xhtml' = Field(default=...)
    'The actual narrative content, a stripped down version of XHTML.'
    status: 'code' = Field(default=...)
    "The status of the narrative - whether it's entirely generated (from just the defined data or the extensions too), or whether a human authored it and it may contain additional data."
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class ParameterDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The parameters to the module. This collection specifies both the input and output parameters. Input parameters are provided by the caller as part of the $evaluate operation. Output parameters are included in the GuidanceResponse."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    use: 'code' = Field(default=...)
    'Whether the parameter is input or output for the module.'
    min: Optional_['integer'] = Field(default=None)
    'The minimum number of times this parameter SHALL appear in the request or response.'
    max: Optional_['string'] = Field(default=None)
    'The maximum number of times this element is permitted to appear in the request or response.'
    name: Optional_['code'] = Field(default=None)
    'The name of the parameter used to allow access to the value of the parameter in evaluation contexts.'
    type: 'code' = Field(default=...)
    'The type of the parameter.'
    profile: Optional_['canonical'] = Field(default=None)
    'If specified, this indicates a profile that the input data must conform to, or that the output data will conform to.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    documentation: Optional_['string'] = Field(default=None)
    'A brief discussion of what the parameter is for and how it is used by the module.'


class Period(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A time period defined by a start and end date and optionally time."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    end: Optional_['dateTime'] = Field(default=None)
    'The end of the period. If the end of the period is missing, it means no end was known or planned at the time the instance was created. The start may be in the past, and the end date in the future, which means that period is expected/planned to end at that time.'
    start: Optional_['dateTime'] = Field(default=None)
    'The start of the period. The boundary is inclusive.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class PrimitiveType(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The base type for all re-useable types defined that have a simple property."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class ProductShelfLife(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The shelf-life and storage information for a medicinal product item or container can be described using this class."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'This describes the shelf life, taking into account various scenarios such as shelf life of the packaged Medicinal Product itself, shelf life after transformation where necessary and shelf life after the first opening of a bottle, etc. The shelf life type shall be specified using an appropriate controlled vocabulary The controlled term and the controlled term identifier shall be specified.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    periodDuration: Optional_['Duration'] = Field(default=None)
    'The shelf life time period can be specified using a numerical value for the period of time and its unit of time measurement The unit of measurement shall be specified in accordance with ISO 11240 and the resulting terminology The symbol and the symbol identifier shall be used.'
    periodString: Optional_['string'] = Field(default=None)
    'The shelf life time period can be specified using a numerical value for the period of time and its unit of time measurement The unit of measurement shall be specified in accordance with ISO 11240 and the resulting terminology The symbol and the symbol identifier shall be used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    specialPrecautionsForStorage: List_['CodeableConcept'] = Field(default_factory=list)
    'Special precautions for storage, if any, can be specified using an appropriate controlled vocabulary The controlled term and the controlled term identifier shall be specified.'


class Quantity(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A measured amount (or an amount that can potentially be measured). Note that measured amounts include amounts that are not precisely quantified, including amounts involving arbitrary units and floating currencies."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    unit: Optional_['string'] = Field(default=None)
    'A human-readable form of the unit.'
    value: Optional_['decimal'] = Field(default=None)
    'The value of the measured amount. The value includes an implicit precision in the presentation of the value.'
    system: Optional_['uri'] = Field(default=None)
    'The identification of the system that provides the coded form of the unit.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    comparator: Optional_['code'] = Field(default=None)
    'How the value should be understood and represented - whether the actual value is greater or less than the stated value due to measurement issues; e.g. if the comparator is "<" , then the real value is < stated value.'
    code: Optional_['code'] = Field(default=None)
    'A computer processable form of the unit in some unit representation system.'


class Range(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A set of ordered Quantities defined by a low and high limit."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    low: Optional_['Quantity'] = Field(default=None)
    'The low limit. The boundary is inclusive.'
    high: Optional_['Quantity'] = Field(default=None)
    'The high limit. The boundary is inclusive.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class Ratio(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A relationship of two Quantity values - expressed as a numerator and a denominator."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    numerator: Optional_['Quantity'] = Field(default=None)
    'The value of the numerator.'
    denominator: Optional_['Quantity'] = Field(default=None)
    'The value of the denominator.'


class RatioRange(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A range of ratios expressed as a low and high numerator and a denominator."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    denominator: Optional_['Quantity'] = Field(default=None)
    'The value of the denominator.'
    lowNumerator: Optional_['Quantity'] = Field(default=None)
    'The value of the low limit numerator.'
    highNumerator: Optional_['Quantity'] = Field(default=None)
    'The value of the high limit numerator.'


class Reference(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A reference from one resource to another."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['uri'] = Field(default=None)
    'The expected type of the target of the reference. If both Reference.type and Reference.reference are populated and Reference.reference is a FHIR URL, both SHALL be consistent.\n\nThe type is the Canonical URL of Resource Definition that is the type this reference refers to. References are URLs that are relative to http://hl7.org/fhir/StructureDefinition/ e.g. "Patient" is a reference to http://hl7.org/fhir/StructureDefinition/Patient. Absolute URLs are only allowed for logical models (and can only be used in references in logical models, not resources).'
    display: Optional_['string'] = Field(default=None)
    'Plain text narrative that identifies the resource in addition to the resource reference.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: Optional_['string'] = Field(default=None)
    "A reference to a location at which the other resource is found. The reference may be a relative reference, in which case it is relative to the service base URL, or an absolute URL that resolves to the location where the resource is found. The reference may be version specific or not. If the reference is not to a FHIR RESTful server, then it should be assumed to be version specific. Internal fragment references (start with '#') refer to contained resources."
    identifier: Optional_['Identifier'] = Field(default=None)
    'An identifier for the target resource. This is used when there is no way to reference the other resource directly, either because the entity it represents is not available through a FHIR server, or because there is no way for the author of the resource to convert a known identifier to an actual location. There is no requirement that a Reference.identifier point to something that is actually exposed as a FHIR instance, but it SHALL point to a business concept that would be expected to be exposed as a FHIR instance, and that instance would need to be of a FHIR resource type allowed by the reference.'


class RelatedArtifact(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Related artifacts such as additional documentation, justification, or bibliographic references."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'The type of relationship to the related artifact.'
    label: Optional_['string'] = Field(default=None)
    'A short label that can be used to reference the citation from elsewhere in the containing artifact, such as a footnote index.'
    display: Optional_['string'] = Field(default=None)
    'A brief description of the document or knowledge resource being referenced, suitable for display to a consumer.'
    citation: Optional_['markdown'] = Field(default=None)
    'A bibliographic citation for the related artifact. This text SHOULD be formatted according to an accepted citation format.'
    document: Optional_['Attachment'] = Field(default=None)
    'The document being referenced, represented as an attachment. This is exclusive with the resource element.'
    resource: Optional_['canonical'] = Field(default=None)
    'The related artifact, such as a library, value set, profile, or other knowledge resource.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    classifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Provides additional classifiers of the related artifact.'
    publicationDate: Optional_['date'] = Field(default=None)
    'The date of publication of the artifact being referred to.'
    resourceReference: Optional_['Reference'] = Field(default=None)
    'The related artifact, if the artifact is not a canonical resource, or a resource reference to a canonical resource.'
    publicationStatus: Optional_['code'] = Field(default=None)
    'The publication status of the artifact being referred to.'


class SampledData(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A series of measurements taken by a device, with upper and lower limits. There may be more than one dimension in the data."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    data: Optional_['string'] = Field(default=None)
    'A series of data points which are decimal values or codes separated by a single space (character u20). The special codes "E" (error), "L" (below detection limit) and "U" (above detection limit) are also defined for used in place of decimal values.'
    origin: 'Quantity' = Field(default=...)
    'The base quantity that a measured value of zero represents. In addition, this provides the units of the entire measurement series.'
    factor: Optional_['decimal'] = Field(default=None)
    'A correction factor that is applied to the sampled data points before they are added to the origin.'
    codeMap: Optional_['canonical'] = Field(default=None)
    'Reference to ConceptMap that defines the codes used in the data.'
    offsets: Optional_['string'] = Field(default=None)
    'A series of data points which are decimal values separated by a single space (character u20).  The units in which the offsets are expressed are found in intervalUnit.  The absolute point at which the measurements begin SHALL be conveyed outside the scope of this datatype, e.g. Observation.effectiveDateTime for a timing offset.'
    interval: Optional_['decimal'] = Field(default=None)
    'Amount of intervalUnits between samples, e.g. milliseconds for time-based sampling.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    lowerLimit: Optional_['decimal'] = Field(default=None)
    'The lower limit of detection of the measured points. This is needed if any of the data points have the value "L" (lower than detection limit).'
    upperLimit: Optional_['decimal'] = Field(default=None)
    'The upper limit of detection of the measured points. This is needed if any of the data points have the value "U" (higher than detection limit).'
    dimensions: 'positiveInt' = Field(default=...)
    'The number of sample points at each time point. If this value is greater than one, then the dimensions will be interlaced - all the sample points for a point in time will be recorded at once.'
    intervalUnit: 'code' = Field(default=...)
    'The measurement unit in which the sample interval is expressed.'


class Signature(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A signature along with supporting context. The signature may be a digital signature that is cryptographic in nature, or some other signature acceptable to the domain. This other signature may be as simple as a graphical image representing a hand-written signature, or a signature ceremony Different signature approaches have different utilities."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    who: Optional_['Reference'] = Field(default=None)
    'A reference to an application-usable description of the identity that signed  (e.g. the signature used their private key).'
    type: List_['Coding'] = Field(default_factory=list)
    'An indication of the reason that the entity signed this document. This may be explicitly included as part of the signature information and can be used when determining accountability for various actions concerning the document.'
    when: Optional_['instant'] = Field(default=None)
    'When the digital signature was signed.'
    data: Optional_['base64Binary'] = Field(default=None)
    'The base64 encoding of the Signature content. When signature is not recorded electronically this element would be empty.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    sigFormat: Optional_['code'] = Field(default=None)
    'A mime type that indicates the technical format of the signature. Important mime types are application/signature+xml for X ML DigSig, application/jose for JWS, and image/* for a graphical image of a signature, etc.'
    onBehalfOf: Optional_['Reference'] = Field(default=None)
    'A reference to an application-usable description of the identity that is represented by the signature.'
    targetFormat: Optional_['code'] = Field(default=None)
    'A mime type that indicates the technical format of the target resources signed by the signature.'


class TimingRepeat(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A set of rules that describe when the event is scheduled."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    when: List_['code'] = Field(default_factory=list)
    'An approximate time period during the day, potentially linked to an event of daily living that indicates when the action should occur.'
    count: Optional_['positiveInt'] = Field(default=None)
    'A total count of the desired number of repetitions across the duration of the entire timing specification. If countMax is present, this element indicates the lower bound of the allowed range of count values.'
    period: Optional_['decimal'] = Field(default=None)
    'Indicates the duration of time over which repetitions are to occur; e.g. to express "3 times per day", 3 would be the frequency and "1 day" would be the period. If periodMax is present, this element indicates the lower bound of the allowed range of the period length.'
    offset: Optional_['unsignedInt'] = Field(default=None)
    'The number of minutes from the event. If the event code does not indicate whether the minutes is before or after the event, then the offset is assumed to be after the event.'
    countMax: Optional_['positiveInt'] = Field(default=None)
    'If present, indicates that the count is a range - so to perform the action between [count] and [countMax] times.'
    duration: Optional_['decimal'] = Field(default=None)
    'How long this thing happens for when it happens. If durationMax is present, this element indicates the lower bound of the allowed range of the duration.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    boundsDuration: Optional_['Duration'] = Field(default=None)
    'Either a duration for the length of the timing schedule, a range of possible length, or outer bounds for start and/or end limits of the timing schedule.'
    boundsRange: Optional_['Range'] = Field(default=None)
    'Either a duration for the length of the timing schedule, a range of possible length, or outer bounds for start and/or end limits of the timing schedule.'
    boundsPeriod: Optional_['Period'] = Field(default=None)
    'Either a duration for the length of the timing schedule, a range of possible length, or outer bounds for start and/or end limits of the timing schedule.'
    frequency: Optional_['positiveInt'] = Field(default=None)
    'The number of times to repeat the action within the specified period. If frequencyMax is present, this element indicates the lower bound of the allowed range of the frequency.'
    periodMax: Optional_['decimal'] = Field(default=None)
    'If present, indicates that the period is a range from [period] to [periodMax], allowing expressing concepts such as "do this once every 3-5 days.'
    dayOfWeek: List_['code'] = Field(default_factory=list)
    'If one or more days of week is provided, then the action happens only on the specified day(s).'
    timeOfDay: List_['time'] = Field(default_factory=list)
    'Specified time of day for action to take place.'
    periodUnit: Optional_['code'] = Field(default=None)
    "The units of time for the period in UCUM units\nNormal practice is to use the 'mo' code as a calendar month when calculating the next occurrence."
    durationMax: Optional_['decimal'] = Field(default=None)
    'If present, indicates that the duration is a range - so to perform the action between [duration] and [durationMax] time length.'
    durationUnit: Optional_['code'] = Field(default=None)
    "The units of time for the duration, in UCUM units\nNormal practice is to use the 'mo' code as a calendar month when calculating the next occurrence."
    frequencyMax: Optional_['positiveInt'] = Field(default=None)
    'If present, indicates that the frequency is a range - so to repeat between [frequency] and [frequencyMax] times within the period or period range.'


class Timing(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specifies an event that may occur multiple times. Timing schedules are used to record when things are planned, expected or requested to occur. The most common usage is in dosage instructions for medications. They are also used when planning care of various kinds, and may be used for reporting the schedule to which past regular activities were carried out."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'A code for the timing schedule (or just text in code.text). Some codes such as BID are ubiquitous, but many institutions define their own additional codes. If a code is provided, the code is understood to be a complete statement of whatever is specified in the structured timing data, and either the code or the data may be used to interpret the Timing, with the exception that .repeat.bounds still applies over the code (and is not contained in the code).'
    event: List_['dateTime'] = Field(default_factory=list)
    'Identifies specific times when the event occurs.'
    repeat: Optional_['TimingRepeat'] = Field(default=None)
    'A set of rules that describe when the event is scheduled.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class TriggerDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A description of a triggering event. Triggering events can be named events, data events, or periodic, as determined by the type element."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'The type of triggering event.'
    name: Optional_['string'] = Field(default=None)
    'A formal name for the event. This may be an absolute URI that identifies the event formally (e.g. from a trigger registry), or a simple relative URI that identifies the event in a local context.'
    data: List_['DataRequirement'] = Field(default_factory=list)
    'The triggering data of the event (if this is a data trigger). If more than one data is requirement is specified, then all the data requirements must be true.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    timingTiming: Optional_['Timing'] = Field(default=None)
    'The timing of the event (if this is a periodic trigger).'
    timingReference: Optional_['Reference'] = Field(default=None)
    'The timing of the event (if this is a periodic trigger).'
    timingDate: Optional_['date'] = Field(default=None)
    'The timing of the event (if this is a periodic trigger).'
    timingDateTime: Optional_['dateTime'] = Field(default=None)
    'The timing of the event (if this is a periodic trigger).'
    condition: Optional_['Expression'] = Field(default=None)
    'A boolean-valued expression that is evaluated in the context of the container of the trigger definition and returns whether or not the trigger fires.'
    subscriptionTopic: Optional_['canonical'] = Field(default=None)
    'A reference to a SubscriptionTopic resource that defines the event. If this element is provided, no other information about the trigger definition may be supplied.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'A code that identifies the event.'


class UsageContext(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specifies clinical/business/etc. metadata that can be used to retrieve, index and/or categorize an artifact. This metadata can either be specific to the applicable population (e.g., age category, DRG) or the specific context of care (e.g., venue, care setting, provider of care)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: 'Coding' = Field(default=...)
    'A code that identifies the type of context being specified by this usage context.'
    valueCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'A value that defines the context specified in this context of use. The interpretation of the value is defined by the code.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'A value that defines the context specified in this context of use. The interpretation of the value is defined by the code.'
    valueRange: Optional_['Range'] = Field(default=None)
    'A value that defines the context specified in this context of use. The interpretation of the value is defined by the code.'
    valueReference: Optional_['Reference'] = Field(default=None)
    'A value that defines the context specified in this context of use. The interpretation of the value is defined by the code.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'


class VirtualServiceDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Virtual Service Contact Details."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    addressUrl: Optional_['url'] = Field(default=None)
    'What address or number needs to be used for a user to connect to the virtual service to join. The channelType informs as to which datatype is appropriate to use (requires knowledge of the specific type).'
    addressString: Optional_['string'] = Field(default=None)
    'What address or number needs to be used for a user to connect to the virtual service to join. The channelType informs as to which datatype is appropriate to use (requires knowledge of the specific type).'
    addressContactPoint: Optional_['ContactPoint'] = Field(default=None)
    'What address or number needs to be used for a user to connect to the virtual service to join. The channelType informs as to which datatype is appropriate to use (requires knowledge of the specific type).'
    addressExtendedContactDetail: Optional_['ExtendedContactDetail'] = Field(default=None)
    'What address or number needs to be used for a user to connect to the virtual service to join. The channelType informs as to which datatype is appropriate to use (requires knowledge of the specific type).'
    sessionKey: Optional_['string'] = Field(default=None)
    'Session Key required by the virtual service.'
    channelType: Optional_['Coding'] = Field(default=None)
    'The type of virtual service to connect to (i.e. Teams, Zoom, Specific VMR technology, WhatsApp).'
    additionalInfo: List_['url'] = Field(default_factory=list)
    'Address to see alternative connection details.'
    maxParticipants: Optional_['positiveInt'] = Field(default=None)
    'Maximum number of participants supported by the virtual service.'


class MoneyQuantity(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """There SHALL be a code if there is a value and it SHALL be an expression of currency.  If system is present, it SHALL be ISO 4217 (system = "urn:iso:std:iso:4217" - currency)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    unit: Optional_['string'] = Field(default=None)
    'A human-readable form of the unit.'
    value: Optional_['decimal'] = Field(default=None)
    'The value of the measured amount. The value includes an implicit precision in the presentation of the value.'
    system: Optional_['uri'] = Field(default=None)
    'The identification of the system that provides the coded form of the unit.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    comparator: Optional_['code'] = Field(default=None)
    'How the value should be understood and represented - whether the actual value is greater or less than the stated value due to measurement issues; e.g. if the comparator is "<" , then the real value is < stated value.'
    code: Optional_['code'] = Field(default=None)
    'A computer processable form of the unit in some unit representation system.'


class SimpleQuantity(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The comparator is not used on a SimpleQuantity"""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    unit: Optional_['string'] = Field(default=None)
    'A human-readable form of the unit.'
    value: Optional_['decimal'] = Field(default=None)
    'The value of the measured amount. The value includes an implicit precision in the presentation of the value.'
    system: Optional_['uri'] = Field(default=None)
    'The identification of the system that provides the coded form of the unit.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    comparator: List_['code'] = Field(default_factory=list)
    'Not allowed to be used in this context'
    code: Optional_['code'] = Field(default=None)
    'A computer processable form of the unit in some unit representation system.'


class Resource(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """This is the base resource type for everything."""
    resourceType: Literal_['Resource'] = Field(default=...)
    'Base Resource'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'


class AccountRelatedAccount(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Other associated accounts related to this account."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    account: 'Reference' = Field(default=...)
    'Reference to an associated Account.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    relationship: Optional_['CodeableConcept'] = Field(default=None)
    'Relationship of the associated Account.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AccountProcedure(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """When using an account for billing a specific Encounter the set of procedures that are relevant for billing are stored here on the account where they are able to be sequenced appropriately prior to processing to produce claim(s)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: 'CodeableReference' = Field(default=...)
    'The procedure relevant to the account.'
    type: List_['CodeableConcept'] = Field(default_factory=list)
    'How this procedure value should be used in charging the account.'
    device: List_['Reference'] = Field(default_factory=list)
    'Any devices that were associated with the procedure relevant to the account.'
    sequence: Optional_['positiveInt'] = Field(default=None)
    'Ranking of the procedure (for each type).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    packageCode: List_['CodeableConcept'] = Field(default_factory=list)
    'The package code can be used to group procedures that may be priced or delivered as a single product. Such as DRGs.'
    dateOfService: Optional_['dateTime'] = Field(default=None)
    'Date of the procedure when using a coded procedure. If using a reference to a procedure, then the date on the procedure should be used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AccountDiagnosis(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """When using an account for billing a specific Encounter the set of diagnoses that are relevant for billing are stored here on the account where they are able to be sequenced appropriately prior to processing to produce claim(s)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: List_['CodeableConcept'] = Field(default_factory=list)
    'Type that this diagnosis has relevant to the account (e.g. admission, billing, discharge …).'
    sequence: Optional_['positiveInt'] = Field(default=None)
    'Ranking of the diagnosis (for each type).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    condition: 'CodeableReference' = Field(default=...)
    'The diagnosis relevant to the account.'
    onAdmission: Optional_['boolean'] = Field(default=None)
    'Was the Diagnosis present on Admission in the related Encounter.'
    packageCode: List_['CodeableConcept'] = Field(default_factory=list)
    'The package code can be used to group diagnoses that may be priced or delivered as a single product. Such as DRGs.'
    dateOfDiagnosis: Optional_['dateTime'] = Field(default=None)
    'Ranking of the diagnosis (for each type).'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AccountGuarantor(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The parties responsible for balancing the account if other payment options fall short."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    party: 'Reference' = Field(default=...)
    'The entity who is responsible.'
    onHold: Optional_['boolean'] = Field(default=None)
    'A guarantor may be placed on credit hold or otherwise have their role temporarily suspended.'
    period: Optional_['Period'] = Field(default=None)
    'The timeframe during which the guarantor accepts responsibility for the account.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AccountCoverage(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The party(s) that are responsible for covering the payment of this account, and what order should they be applied to the account."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    coverage: 'Reference' = Field(default=...)
    'The party(s) that contribute to payment (or part of) of the charges applied to this account (including self-pay).\n\nA coverage may only be responsible for specific types of charges, and the sequence of the coverages in the account could be important when processing billing.'
    priority: Optional_['positiveInt'] = Field(default=None)
    'The priority of the coverage in the context of this account.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AccountBalance(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The calculated account balances - these are calculated and processed by the finance system.\r\rThe balances with a `term` that is not current are usually generated/updated by an invoicing or similar process."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    term: Optional_['CodeableConcept'] = Field(default=None)
    'The term of the account balances - The balance value is the amount that was outstanding for this age.'
    amount: 'Money' = Field(default=...)
    'The actual balance value calculated for the age defined in the term property.'
    estimate: Optional_['boolean'] = Field(default=None)
    'The amount is only an estimated value - this is likely common for `current` term balances, but not with known terms (that were generated by a backend process).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    aggregate: Optional_['CodeableConcept'] = Field(default=None)
    'Who is expected to pay this part of the balance.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Account(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A financial tool for tracking value accrued for a particular purpose.  In the healthcare field, used to track charges for a patient, cost centers, etc."""
    resourceType: Literal_['Account'] = Field(default=...)
    'Tracks balance, charges, for patient or cost center'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'Categorizes the account for reporting and searching purposes.'
    name: Optional_['string'] = Field(default=None)
    'Name used for the account when displaying it to humans in reports, etc.'
    owner: Optional_['Reference'] = Field(default=None)
    'Indicates the service area, hospital, department, etc. with responsibility for managing the Account.'
    status: 'code' = Field(default=...)
    'Indicates whether the account is presently used/usable or not.'
    subject: List_['Reference'] = Field(default_factory=list)
    'Identifies the entity which incurs the expenses. While the immediate recipients of services or goods might be entities related to the subject, the expenses were ultimately incurred by the subject of the Account.'
    balance: List_['AccountBalance'] = Field(default_factory=list)
    'The calculated account balances - these are calculated and processed by the finance system.\r\rThe balances with a `term` that is not current are usually generated/updated by an invoicing or similar process.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    coverage: List_['AccountCoverage'] = Field(default_factory=list)
    'The party(s) that are responsible for covering the payment of this account, and what order should they be applied to the account.'
    currency: Optional_['CodeableConcept'] = Field(default=None)
    'The default currency for the account.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    guarantor: List_['AccountGuarantor'] = Field(default_factory=list)
    'The parties responsible for balancing the account if other payment options fall short.'
    diagnosis: List_['AccountDiagnosis'] = Field(default_factory=list)
    'When using an account for billing a specific Encounter the set of diagnoses that are relevant for billing are stored here on the account where they are able to be sequenced appropriately prior to processing to produce claim(s).'
    procedure: List_['AccountProcedure'] = Field(default_factory=list)
    'When using an account for billing a specific Encounter the set of procedures that are relevant for billing are stored here on the account where they are able to be sequenced appropriately prior to processing to produce claim(s).'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Unique identifier used to reference the account.  Might or might not be intended for human use (e.g. credit card number).'
    description: Optional_['markdown'] = Field(default=None)
    'Provides additional information about what the account tracks and how it is used.'
    calculatedAt: Optional_['instant'] = Field(default=None)
    'Time the balance amount was calculated.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    billingStatus: Optional_['CodeableConcept'] = Field(default=None)
    'The BillingStatus tracks the lifecycle of the account through the billing process. It indicates how transactions are treated when they are allocated to the account.'
    servicePeriod: Optional_['Period'] = Field(default=None)
    'The date range of services associated with this account.'
    relatedAccount: List_['AccountRelatedAccount'] = Field(default_factory=list)
    'Other associated accounts related to this account.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ActivityDefinitionDynamicValue(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Dynamic values that will be evaluated to produce values for elements of the resulting resource. For example, if the dosage of a medication must be computed based on the patient's weight, a dynamic value would be used to specify an expression that calculated the weight, and the path on the request resource that would contain the result."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    path: 'string' = Field(default=...)
    'The path to the element to be customized. This is the path on the resource that will hold the result of the calculation defined by the expression. The specified path SHALL be a FHIRPath resolvable on the specified target type of the ActivityDefinition, and SHALL consist only of identifiers, constant indexers, and a restricted subset of functions. The path is allowed to contain qualifiers (.) to traverse sub-elements, as well as indexers ([x]) to traverse multiple-cardinality sub-elements (see the [Simple FHIRPath Profile](fhirpath.html#simple) for full details).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    expression: 'Expression' = Field(default=...)
    'An expression specifying the value of the customized element.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ActivityDefinitionParticipant(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates who should participate in performing the action described."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['code'] = Field(default=None)
    'The type of participant in the action.'
    role: Optional_['CodeableConcept'] = Field(default=None)
    'The role the participant should play in performing the described action.'
    function: Optional_['CodeableConcept'] = Field(default=None)
    'Indicates how the actor will be involved in the action - author, reviewer, witness, etc.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    typeCanonical: Optional_['canonical'] = Field(default=None)
    'The type of participant in the action.'
    typeReference: Optional_['Reference'] = Field(default=None)
    'The type of participant in the action.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ActivityDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """This resource allows for the definition of some activity to be performed, independent of a particular patient, practitioner, or other performance context."""
    resourceType: Literal_['ActivityDefinition'] = Field(default=...)
    'The definition of a specific activity to be taken, independent of any particular patient or context'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this activity definition when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this activity definition is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the activity definition is stored on different servers.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the activity definition. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    kind: Optional_['code'] = Field(default=None)
    'A description of the kind of resource the activity definition is representing. For example, a MedicationRequest, a ServiceRequest, or a CommunicationRequest.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the activity definition.'
    usage: Optional_['markdown'] = Field(default=None)
    'A detailed description of how the activity definition is used from a clinical perspective.'
    topic: List_['CodeableConcept'] = Field(default_factory=list)
    'Descriptive topics related to the content of the activity. Topics provide a high-level categorization of the activity that can be useful for filtering and searching.'
    status: 'code' = Field(default=...)
    'The status of this activity definition. Enables tracking the life-cycle of the content.'
    author: List_['ContactDetail'] = Field(default_factory=list)
    'An individiual or organization primarily involved in the creation and maintenance of the content.'
    editor: List_['ContactDetail'] = Field(default_factory=list)
    'An individual or organization primarily responsible for internal coherence of the content.'
    intent: Optional_['code'] = Field(default=None)
    'Indicates the level of authority/intentionality associated with the activity and where the request should fit into the workflow chain.'
    dosage: List_['Dosage'] = Field(default_factory=list)
    'Provides detailed dosage instructions in the same way that they are described for MedicationRequest resources.'
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the activity definition when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the activity definition author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence. To provide a version consistent with the Decision Support Service specification, use the format Major.Minor.Revision (e.g. 1.0.0). For more information on versioning knowledge assets, refer to the Decision Support Service specification. Note that a version is required for non-experimental active assets.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    purpose: Optional_['markdown'] = Field(default=None)
    'Explanation of why this activity definition is needed and why it has been designed as it has.'
    library: List_['canonical'] = Field(default_factory=list)
    'A reference to a Library resource containing any formal logic used by the activity definition.'
    profile: Optional_['canonical'] = Field(default=None)
    'A profile to which the target of the activity definition is expected to conform.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    subtitle: Optional_['string'] = Field(default=None)
    'An explanatory or alternate title for the activity definition giving additional information about its content.'
    reviewer: List_['ContactDetail'] = Field(default_factory=list)
    'An individual or organization asserted by the publisher to be primarily responsible for review of some aspect of the content.'
    endorser: List_['ContactDetail'] = Field(default_factory=list)
    'An individual or organization asserted by the publisher to be responsible for officially endorsing the content for use in some setting.'
    priority: Optional_['code'] = Field(default=None)
    'Indicates how quickly the activity  should be addressed with respect to other requests.'
    location: Optional_['CodeableReference'] = Field(default=None)
    'Identifies the facility where the activity will occur; e.g. home, hospital, specific clinic, etc.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'Identifies the quantity expected to be consumed at once (per dose, per meal, etc.).'
    bodySite: List_['CodeableConcept'] = Field(default_factory=list)
    "Indicates the sites on the subject's body where the procedure should be performed (I.e. the target sites)."
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual responsible for the release and ongoing maintenance of the activity definition.'
    copyright: Optional_['markdown'] = Field(default=None)
    'A copyright statement relating to the activity definition and/or its contents. Copyright statements are generally legal restrictions on the use and publishing of the activity definition.'
    timingTiming: Optional_['Timing'] = Field(default=None)
    'The timing or frequency upon which the described activity is to occur.'
    timingAge: Optional_['Age'] = Field(default=None)
    'The timing or frequency upon which the described activity is to occur.'
    timingRange: Optional_['Range'] = Field(default=None)
    'The timing or frequency upon which the described activity is to occur.'
    timingDuration: Optional_['Duration'] = Field(default=None)
    'The timing or frequency upon which the described activity is to occur.'
    transform: Optional_['canonical'] = Field(default=None)
    'A reference to a StructureMap resource that defines a transform that can be executed to produce the intent resource using the ActivityDefinition instance as the input.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this activity definition when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    subjectCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'A code, group definition, or canonical reference that describes  or identifies the intended subject of the activity being defined.  Canonical references are allowed to support the definition of protocols for drug and substance quality specifications, and is allowed to reference a MedicinalProductDefinition, SubstanceDefinition, AdministrableProductDefinition, ManufacturedItemDefinition, or PackagedProductDefinition resource.'
    subjectReference: Optional_['Reference'] = Field(default=None)
    'A code, group definition, or canonical reference that describes  or identifies the intended subject of the activity being defined.  Canonical references are allowed to support the definition of protocols for drug and substance quality specifications, and is allowed to reference a MedicinalProductDefinition, SubstanceDefinition, AdministrableProductDefinition, ManufacturedItemDefinition, or PackagedProductDefinition resource.'
    subjectCanonical: Optional_['canonical'] = Field(default=None)
    'A code, group definition, or canonical reference that describes  or identifies the intended subject of the activity being defined.  Canonical references are allowed to support the definition of protocols for drug and substance quality specifications, and is allowed to reference a MedicinalProductDefinition, SubstanceDefinition, AdministrableProductDefinition, ManufacturedItemDefinition, or PackagedProductDefinition resource.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate activity definition instances.'
    productReference: Optional_['Reference'] = Field(default=None)
    'Identifies the food, drug or other product being consumed or supplied in the activity.'
    productCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Identifies the food, drug or other product being consumed or supplied in the activity.'
    description: Optional_['markdown'] = Field(default=None)
    "A free text natural language description of the activity definition from a consumer's perspective."
    asNeededBoolean: Optional_['boolean'] = Field(default=None)
    'If a CodeableConcept is present, it indicates the pre-condition for performing the service.  For example "pain", "on flare-up", etc.'
    asNeededCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'If a CodeableConcept is present, it indicates the pre-condition for performing the service.  For example "pain", "on flare-up", etc.'
    participant: List_['ActivityDefinitionParticipant'] = Field(default_factory=list)
    'Indicates who should participate in performing the action described.'
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this activity definition is authored for testing purposes (or education/evaluation/marketing) and is not intended to be used for genuine usage.'
    jurisdiction: List_['CodeableConcept'] = Field(default_factory=list)
    'A legal or geographic region in which the activity definition is intended to be used.'
    approvalDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was approved by the publisher. Approval happens once when the content is officially approved for usage.'
    doNotPerform: Optional_['boolean'] = Field(default=None)
    'Set this to true if the definition is to indicate that a particular activity should NOT be performed. If true, this element should be interpreted to reinforce a negative coding. For example NPO as a code with a doNotPerform of true would still indicate to NOT perform the action.'
    dynamicValue: List_['ActivityDefinitionDynamicValue'] = Field(default_factory=list)
    "Dynamic values that will be evaluated to produce values for elements of the resulting resource. For example, if the dosage of a medication must be computed based on the patient's weight, a dynamic value would be used to specify an expression that calculated the weight, and the path on the request resource that would contain the result."
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    copyrightLabel: Optional_['string'] = Field(default=None)
    "A short string (<50 characters), suitable for inclusion in a page footer that identifies the copyright holder, effective period, and optionally whether rights are resctricted. (e.g. 'All rights reserved', 'Some rights reserved')."
    lastReviewDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was last reviewed. Review happens periodically after approval but does not change the original approval date.'
    effectivePeriod: Optional_['Period'] = Field(default=None)
    'The period during which the activity definition content was or is planned to be in active use.'
    relatedArtifact: List_['RelatedArtifact'] = Field(default_factory=list)
    'Related artifacts such as additional documentation, justification, or bibliographic references.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    specimenRequirement: List_['canonical'] = Field(default_factory=list)
    'Defines specimen requirements for the action to be performed, such as required specimens for a lab test.'
    observationRequirement: List_['canonical'] = Field(default_factory=list)
    'Defines observation requirements for the action to be performed, such as body weight or surface area.'
    observationResultRequirement: List_['canonical'] = Field(default_factory=list)
    'Defines the observations that are expected to be produced by the action.'
    date: Optional_['dateTime'] = Field(default=None)
    'The date  (and optionally time) when the activity definition was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the activity definition changes.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'Detailed description of the type of activity; e.g. What lab test, what procedure, what kind of encounter.'


class ActorDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Describes an actor - a human or an application that plays a role in data exchange, and that may have obligations associated with the role the actor plays."""
    resourceType: Literal_['ActorDefinition'] = Field(default=...)
    'An application that exchanges data'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the actor definition. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    date: Optional_['dateTime'] = Field(default=None)
    'The date  (and optionally time) when the actor definition was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the actor definition changes.'
    type: 'code' = Field(default=...)
    'Whether the actor represents a human or an appliction.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the actor definition.'
    status: 'code' = Field(default=...)
    'The status of this actor definition. Enables tracking the life-cycle of the content.'
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the actor definition when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the actor definition author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    purpose: Optional_['markdown'] = Field(default=None)
    'Explanation of why this actor definition is needed and why it has been designed as it has.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual responsible for the release and ongoing maintenance of the actor definition.'
    copyright: Optional_['markdown'] = Field(default=None)
    'A copyright statement relating to the actor definition and/or its contents. Copyright statements are generally legal restrictions on the use and publishing of the actor definition.'
    reference: List_['url'] = Field(default_factory=list)
    'A reference to additional documentation about the actor, but description and documentation.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this actor definition when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate actor definition instances.'
    description: Optional_['markdown'] = Field(default=None)
    'A free text natural language description of the actor.'
    derivedFrom: List_['canonical'] = Field(default_factory=list)
    'A url that identifies the definition of this actor in another IG (which IG must be listed in the dependencies). This actor inherits all the obligations etc. as defined in the other IG.'
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this actor definition is authored for testing purposes (or education/evaluation/marketing) and is not intended to be used for genuine usage.'
    jurisdiction: List_['CodeableConcept'] = Field(default_factory=list)
    'A legal or geographic region in which the actor definition is intended to be used.'
    capabilities: Optional_['canonical'] = Field(default=None)
    'The capability statement for the actor (if the concept is applicable).'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    documentation: Optional_['markdown'] = Field(default=None)
    'Documentation about the functionality of the actor.'
    copyrightLabel: Optional_['string'] = Field(default=None)
    "A short string (<50 characters), suitable for inclusion in a page footer that identifies the copyright holder, effective period, and optionally whether rights are resctricted. (e.g. 'All rights reserved', 'Some rights reserved')."
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this actor definition when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this actor definition is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the actor definition is stored on different servers.'


class AdministrableProductDefinitionRouteOfAdministration(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The path by which the product is taken into or makes contact with the body. In some regions this is referred to as the licenced or approved route. RouteOfAdministration cannot be used when the 'formOf' product already uses MedicinalProductDefinition.route (and vice versa)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: 'CodeableConcept' = Field(default=...)
    'Coded expression for the route.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    firstDose: Optional_['Quantity'] = Field(default=None)
    'The first dose (dose quantity) administered can be specified for the product, using a numerical value and its unit of measurement.'
    maxSingleDose: Optional_['Quantity'] = Field(default=None)
    'The maximum single dose that can be administered, specified using a numerical value and its unit of measurement.'
    maxDosePerDay: Optional_['Quantity'] = Field(default=None)
    'The maximum dose per day (maximum dose quantity to be administered in any one 24-h period) that can be administered.'
    targetSpecies: List_['AdministrableProductDefinitionRouteOfAdministrationTargetSpecies'] = Field(default_factory=list)
    'A species for which this route applies.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    maxTreatmentPeriod: Optional_['Duration'] = Field(default=None)
    'The maximum treatment period during which the product can be administered.'
    maxDosePerTreatmentPeriod: Optional_['Ratio'] = Field(default=None)
    'The maximum dose per treatment period that can be administered.'


class AdministrableProductDefinitionRouteOfAdministrationTargetSpecies(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A species for which this route applies."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: 'CodeableConcept' = Field(default=...)
    'Coded expression for the species.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    withdrawalPeriod: List_['AdministrableProductDefinitionRouteOfAdministrationTargetSpeciesWithdrawalPeriod'] = Field(default_factory=list)
    'A species specific time during which consumption of animal product is not appropriate.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AdministrableProductDefinitionRouteOfAdministrationTargetSpeciesWithdrawalPeriod(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A species specific time during which consumption of animal product is not appropriate."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    value: 'Quantity' = Field(default=...)
    'A value for the time.'
    tissue: 'CodeableConcept' = Field(default=...)
    'Coded expression for the type of tissue for which the withdrawal period applies, e.g. meat, milk.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    supportingInformation: Optional_['string'] = Field(default=None)
    'Extra information about the withdrawal period.'


class AdministrableProductDefinitionProperty(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Characteristics e.g. a product's onset of action."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'A code expressing the type of characteristic.'
    status: Optional_['CodeableConcept'] = Field(default=None)
    'The status of characteristic e.g. assigned or pending.'
    valueCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'A value for the characteristic.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'A value for the characteristic.'
    valueDate: Optional_['date'] = Field(default=None)
    'A value for the characteristic.'
    valueBoolean: Optional_['boolean'] = Field(default=None)
    'A value for the characteristic.'
    valueMarkdown: Optional_['markdown'] = Field(default=None)
    'A value for the characteristic.'
    valueAttachment: Optional_['Attachment'] = Field(default=None)
    'A value for the characteristic.'
    valueReference: Optional_['Reference'] = Field(default=None)
    'A value for the characteristic.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AdministrableProductDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A medicinal product in the final form which is suitable for administering to a patient (after any mixing of multiple components, dissolution etc. has been performed)."""
    resourceType: Literal_['AdministrableProductDefinition'] = Field(default=...)
    'A medicinal product in the final form, suitable for administration - after any mixing of multiple components'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    status: 'code' = Field(default=...)
    'The status of this administrable product. Enables tracking the life-cycle of the content.'
    formOf: List_['Reference'] = Field(default_factory=list)
    "References a product from which one or more of the constituent parts of that product can be prepared and used as described by this administrable product.  If this administrable product describes the administration of a crushed tablet, the 'formOf' would be the product representing a distribution containing tablets and possibly also a cream.  This is distinct from the 'producedFrom' which refers to the specific components of the product that are used in this preparation, rather than the product as a whole."
    device: Optional_['Reference'] = Field(default=None)
    'A device that is integral to the medicinal product, in effect being considered as an "ingredient" of the medicinal product. This is not intended for devices that are just co-packaged.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    property: List_['AdministrableProductDefinitionProperty'] = Field(default_factory=list)
    "Characteristics e.g. a product's onset of action."
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'An identifier for the administrable product.'
    ingredient: List_['CodeableConcept'] = Field(default_factory=list)
    'The ingredients of this administrable medicinal product. This is only needed if the ingredients are not specified either using ManufacturedItemDefiniton (via AdministrableProductDefinition.producedFrom) to state which component items are used to make this, or using by incoming references from the Ingredient resource, to state in detail which substances exist within this. This element allows a basic coded ingredient to be used.'
    description: Optional_['markdown'] = Field(default=None)
    'A general description of the product, when in its final form, suitable for administration e.g. effervescent blue liquid, to be swallowed. Intended to be used when the other structured properties of this resource are insufficient or cannot be supported. It is not intended to duplicate information already carried elswehere.'
    producedFrom: List_['Reference'] = Field(default_factory=list)
    "Indicates the specific manufactured items that are part of the 'formOf' product that are used in the preparation of this specific administrable form.  In some cases, an administrable form might use all of the items from the overall product (or there might only be one item), while in other cases, an administrable form might use only a subset of the items available in the overall product.  For example, an administrable form might involve combining a liquid and a powder available as part of an overall product, but not involve applying the also supplied cream."
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    unitOfPresentation: Optional_['CodeableConcept'] = Field(default=None)
    "The presentation type in which this item is given to a patient. e.g. for a spray - 'puff' (as in 'contains 100 mcg per puff'), or for a liquid - 'vial' (as in 'contains 5 ml per vial')."
    administrableDoseForm: Optional_['CodeableConcept'] = Field(default=None)
    "The dose form of the final product after necessary reconstitution or processing. Contrasts to the manufactured dose form (see ManufacturedItemDefinition). If the manufactured form was 'powder for solution for injection', the administrable dose form could be 'solution for injection' (once mixed with another item having manufactured form 'solvent for solution for injection')."
    routeOfAdministration: List_['AdministrableProductDefinitionRouteOfAdministration'] = Field(default=...)
    "The path by which the product is taken into or makes contact with the body. In some regions this is referred to as the licenced or approved route. RouteOfAdministration cannot be used when the 'formOf' product already uses MedicinalProductDefinition.route (and vice versa)."


class AdverseEventContributingFactor(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The contributing factors suspected to have increased the probability or severity of the adverse event."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    itemReference: Optional_['Reference'] = Field(default=None)
    'The item that is suspected to have increased the probability or severity of the adverse event.'
    itemCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The item that is suspected to have increased the probability or severity of the adverse event.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AdverseEventMitigatingAction(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The ameliorating action taken after the adverse event occured in order to reduce the extent of harm."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    itemReference: Optional_['Reference'] = Field(default=None)
    'The ameliorating action taken after the adverse event occured in order to reduce the extent of harm.'
    itemCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The ameliorating action taken after the adverse event occured in order to reduce the extent of harm.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AdverseEventPreventiveAction(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Preventive actions that contributed to avoiding the adverse event."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    itemReference: Optional_['Reference'] = Field(default=None)
    'The action that contributed to avoiding the adverse event.'
    itemCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The action that contributed to avoiding the adverse event.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AdverseEventSupportingInfo(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Supporting information relevant to the event."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    itemReference: Optional_['Reference'] = Field(default=None)
    "Relevant past history for the subject. In a clinical care context, an example being a patient had an adverse event following a pencillin administration and the patient had a previously documented penicillin allergy. In a clinical trials context, an example is a bunion or rash that was present prior to the study. Additionally, the supporting item can be a document that is relevant to this instance of the adverse event that is not part of the subject's medical history. For example, a clinical note, staff list, or material safety data sheet (MSDS).  Supporting information is not a contributing factor, preventive action, or mitigating action."
    itemCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    "Relevant past history for the subject. In a clinical care context, an example being a patient had an adverse event following a pencillin administration and the patient had a previously documented penicillin allergy. In a clinical trials context, an example is a bunion or rash that was present prior to the study. Additionally, the supporting item can be a document that is relevant to this instance of the adverse event that is not part of the subject's medical history. For example, a clinical note, staff list, or material safety data sheet (MSDS).  Supporting information is not a contributing factor, preventive action, or mitigating action."
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AdverseEventSuspectEntity(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Describes the entity that is suspected to have caused the adverse event."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    causality: Optional_['AdverseEventSuspectEntityCausality'] = Field(default=None)
    'Information on the possible cause of the event.'
    instanceCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Identifies the actual instance of what caused the adverse event.  May be a substance, medication, medication administration, medication statement or a device.'
    instanceReference: Optional_['Reference'] = Field(default=None)
    'Identifies the actual instance of what caused the adverse event.  May be a substance, medication, medication administration, medication statement or a device.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AdverseEventSuspectEntityCausality(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information on the possible cause of the event."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    author: Optional_['Reference'] = Field(default=None)
    'The author of the information on the possible cause of the event.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    assessmentMethod: Optional_['CodeableConcept'] = Field(default=None)
    'The method of evaluating the relatedness of the suspected entity to the event.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    entityRelatedness: Optional_['CodeableConcept'] = Field(default=None)
    'The result of the assessment regarding the relatedness of the suspected entity to the event.'


class AdverseEventParticipant(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates who or what participated in the adverse event and how they were involved."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    actor: 'Reference' = Field(default=...)
    'Indicates who or what participated in the event.'
    function: Optional_['CodeableConcept'] = Field(default=None)
    'Distinguishes the type of involvement of the actor in the adverse event, such as contributor or informant.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AdverseEvent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An event (i.e. any change to current patient status) that may be related to unintended effects on a patient or research participant. The unintended effects may require additional monitoring, treatment, hospitalization, or may result in death. The AdverseEvent resource also extends to potential or avoided events that could have had such effects. There are two major domains where the AdverseEvent resource is expected to be used. One is in clinical care reported adverse events and the other is in reporting adverse events in clinical  research trial management. Adverse events can be reported by healthcare providers, patients, caregivers or by medical products manufacturers. Given the differences between these two concepts, we recommend consulting the domain specific implementation guides when implementing the AdverseEvent Resource. The implementation guides include specific extensions, value sets and constraints."""
    resourceType: Literal_['AdverseEvent'] = Field(default=...)
    'An event that may be related to unintended effects on a patient or research participant'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Comments made about the adverse event by the performer, subject or other participants.'
    study: List_['Reference'] = Field(default_factory=list)
    'The research study that the subject is enrolled in.'
    status: 'code' = Field(default=...)
    'The current state of the adverse event or potential adverse event.'
    subject: 'Reference' = Field(default=...)
    'This subject or group impacted by the event.'
    outcome: List_['CodeableConcept'] = Field(default_factory=list)
    'Describes the type of outcome from the adverse event, such as resolved, recovering, ongoing, resolved-with-sequelae, or fatal.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'The overall type of event, intended for search and filtering purposes.'
    detected: Optional_['dateTime'] = Field(default=None)
    'Estimated or actual date the AdverseEvent began, in the opinion of the reporter.'
    location: Optional_['Reference'] = Field(default=None)
    'The information about where the adverse event occurred.'
    recorder: Optional_['Reference'] = Field(default=None)
    'Information on who recorded the adverse event.  May be the patient or a practitioner.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    actuality: 'code' = Field(default=...)
    'Whether the event actually happened or was a near miss. Note that this is independent of whether anyone was affected or harmed or how severely.'
    encounter: Optional_['Reference'] = Field(default=None)
    'The Encounter associated with the start of the AdverseEvent.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Business identifiers assigned to this adverse event by the performer or other systems which remain constant as the resource is updated and propagates from server to server.'
    seriousness: Optional_['CodeableConcept'] = Field(default=None)
    'Assessment whether this event, or averted event, was of clinical importance.'
    participant: List_['AdverseEventParticipant'] = Field(default_factory=list)
    'Indicates who or what participated in the adverse event and how they were involved.'
    recordedDate: Optional_['dateTime'] = Field(default=None)
    'The date on which the existence of the AdverseEvent was first recorded.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    occurrenceDateTime: Optional_['dateTime'] = Field(default=None)
    'The date (and perhaps time) when the adverse event occurred.'
    occurrencePeriod: Optional_['Period'] = Field(default=None)
    'The date (and perhaps time) when the adverse event occurred.'
    occurrenceTiming: Optional_['Timing'] = Field(default=None)
    'The date (and perhaps time) when the adverse event occurred.'
    suspectEntity: List_['AdverseEventSuspectEntity'] = Field(default_factory=list)
    'Describes the entity that is suspected to have caused the adverse event.'
    supportingInfo: List_['AdverseEventSupportingInfo'] = Field(default_factory=list)
    'Supporting information relevant to the event.'
    resultingEffect: List_['Reference'] = Field(default_factory=list)
    'Information about the condition that occurred as a result of the adverse event, such as hives due to the exposure to a substance (for example, a drug or a chemical) or a broken leg as a result of the fall.'
    preventiveAction: List_['AdverseEventPreventiveAction'] = Field(default_factory=list)
    'Preventive actions that contributed to avoiding the adverse event.'
    mitigatingAction: List_['AdverseEventMitigatingAction'] = Field(default_factory=list)
    'The ameliorating action taken after the adverse event occured in order to reduce the extent of harm.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    contributingFactor: List_['AdverseEventContributingFactor'] = Field(default_factory=list)
    'The contributing factors suspected to have increased the probability or severity of the adverse event.'
    expectedInResearchStudy: Optional_['boolean'] = Field(default=None)
    'Considered likely or probable or anticipated in the research study.  Whether the reported event matches any of the outcomes for the patient that are considered by the study as known or likely.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'Specific event that occurred or that was averted, such as patient fall, wrong organ removed, or wrong blood transfused.'


class AllergyIntoleranceParticipant(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates who or what participated in the activities related to the allergy or intolerance and how they were involved."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    actor: 'Reference' = Field(default=...)
    'Indicates who or what participated in the activities related to the allergy or intolerance.'
    function: Optional_['CodeableConcept'] = Field(default=None)
    'Distinguishes the type of involvement of the actor in the activities related to the allergy or intolerance.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AllergyIntoleranceReaction(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Details about each adverse reaction event linked to exposure to the identified substance."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Additional text about the adverse reaction event not captured in other fields.'
    onset: Optional_['dateTime'] = Field(default=None)
    'Record of the date and/or time of the onset of the Reaction.'
    severity: Optional_['code'] = Field(default=None)
    'Clinical assessment of the severity of the reaction event as a whole, potentially considering multiple different manifestations.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    substance: Optional_['CodeableConcept'] = Field(default=None)
    "Identification of the specific substance (or pharmaceutical product) considered to be responsible for the Adverse Reaction event. Note: the substance for a specific reaction may be different from the substance identified as the cause of the risk, but it must be consistent with it. For instance, it may be a more specific substance (e.g. a brand medication) or a composite product that includes the identified substance. It must be clinically safe to only process the 'code' and ignore the 'reaction.substance'.  If a receiving system is unable to confirm that AllergyIntolerance.reaction.substance falls within the semantic scope of AllergyIntolerance.code, then the receiving system should ignore AllergyIntolerance.reaction.substance."
    description: Optional_['string'] = Field(default=None)
    'Text description about the reaction as a whole, including details of the manifestation if required.'
    manifestation: List_['CodeableReference'] = Field(default=...)
    'Clinical symptoms and/or signs that are observed or associated with the adverse reaction event.'
    exposureRoute: Optional_['CodeableConcept'] = Field(default=None)
    'Identification of the route by which the subject was exposed to the substance.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AllergyIntolerance(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Risk of harmful or undesirable physiological response which is specific to an individual and associated with exposure to a substance."""
    resourceType: Literal_['AllergyIntolerance'] = Field(default=...)
    'Allergy or Intolerance (generally: Risk of adverse reaction to a substance)'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'Identification of the underlying physiological mechanism for the reaction risk.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Additional narrative about the propensity for the Adverse Reaction, not captured in other fields.'
    patient: 'Reference' = Field(default=...)
    'The patient who has the allergy or intolerance.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    category: List_['code'] = Field(default_factory=list)
    'Category of the identified substance.'
    onsetDateTime: Optional_['dateTime'] = Field(default=None)
    'Estimated or actual date,  date-time, or age when allergy or intolerance was identified.'
    onsetAge: Optional_['Age'] = Field(default=None)
    'Estimated or actual date,  date-time, or age when allergy or intolerance was identified.'
    onsetPeriod: Optional_['Period'] = Field(default=None)
    'Estimated or actual date,  date-time, or age when allergy or intolerance was identified.'
    onsetRange: Optional_['Range'] = Field(default=None)
    'Estimated or actual date,  date-time, or age when allergy or intolerance was identified.'
    onsetString: Optional_['string'] = Field(default=None)
    'Estimated or actual date,  date-time, or age when allergy or intolerance was identified.'
    reaction: List_['AllergyIntoleranceReaction'] = Field(default_factory=list)
    'Details about each adverse reaction event linked to exposure to the identified substance.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: Optional_['Reference'] = Field(default=None)
    'The encounter when the allergy or intolerance was asserted.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Business identifiers assigned to this AllergyIntolerance by the performer or other systems which remain constant as the resource is updated and propagates from server to server.'
    criticality: Optional_['code'] = Field(default=None)
    'Estimate of the potential clinical harm, or seriousness, of the reaction to the identified substance.'
    participant: List_['AllergyIntoleranceParticipant'] = Field(default_factory=list)
    'Indicates who or what participated in the activities related to the allergy or intolerance and how they were involved.'
    recordedDate: Optional_['dateTime'] = Field(default=None)
    'The recordedDate represents when this particular AllergyIntolerance record was created in the system, which is often a system-generated date.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    clinicalStatus: Optional_['CodeableConcept'] = Field(default=None)
    'The clinical status of the allergy or intolerance.'
    lastOccurrence: Optional_['dateTime'] = Field(default=None)
    'Represents the date and/or time of the last known occurrence of a reaction event.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    verificationStatus: Optional_['CodeableConcept'] = Field(default=None)
    'Assertion about certainty associated with the propensity, or potential risk, of a reaction to the identified substance (including pharmaceutical product).  The verification status pertains to the allergy or intolerance, itself, not to any specific AllergyIntolerance attribute.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'Code for an allergy or intolerance statement (either a positive or a negated/excluded statement).  This may be a code for a substance or pharmaceutical product that is considered to be responsible for the adverse reaction risk (e.g., "Latex"), an allergy or intolerance condition (e.g., "Latex allergy"), or a negated/excluded code for a specific substance or class (e.g., "No latex allergy") or a general or categorical negated statement (e.g.,  "No known allergy", "No known drug allergies").  Note: the substance for a specific reaction may be different from the substance identified as the cause of the risk, but it must be consistent with it. For instance, it may be a more specific substance (e.g. a brand medication) or a composite product that includes the identified substance. It must be clinically safe to only process the \'code\' and ignore the \'reaction.substance\'.  If a receiving system is unable to confirm that AllergyIntolerance.reaction.substance falls within the semantic scope of AllergyIntolerance.code, then the receiving system should ignore AllergyIntolerance.reaction.substance.'


class AppointmentRecurrenceTemplate(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The details of the recurrence pattern or template that is used to generate recurring appointments."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    timezone: Optional_['CodeableConcept'] = Field(default=None)
    'The timezone of the recurring appointment occurrences.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    excludingDate: List_['date'] = Field(default_factory=list)
    'Any dates, such as holidays, that should be excluded from the recurrence.'
    recurrenceType: 'CodeableConcept' = Field(default=...)
    'How often the appointment series should recur.'
    occurrenceDate: List_['date'] = Field(default_factory=list)
    'The list of specific dates that will have appointments generated.'
    weeklyTemplate: Optional_['AppointmentRecurrenceTemplateWeeklyTemplate'] = Field(default=None)
    'Information about weekly recurring appointments.'
    yearlyTemplate: Optional_['AppointmentRecurrenceTemplateYearlyTemplate'] = Field(default=None)
    'Information about yearly recurring appointments.'
    occurrenceCount: Optional_['positiveInt'] = Field(default=None)
    'How many appointments are planned in the recurrence.'
    monthlyTemplate: Optional_['AppointmentRecurrenceTemplateMonthlyTemplate'] = Field(default=None)
    'Information about monthly recurring appointments.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    lastOccurrenceDate: Optional_['date'] = Field(default=None)
    'Recurring appointments will not occur after this date.'
    excludingRecurrenceId: List_['positiveInt'] = Field(default_factory=list)
    'Any dates, such as holidays, that should be excluded from the recurrence.'


class AppointmentRecurrenceTemplateMonthlyTemplate(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information about monthly recurring appointments."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    dayOfWeek: Optional_['Coding'] = Field(default=None)
    'Indicates which day of the week the recurring appointments should occur each nth week.'
    dayOfMonth: Optional_['positiveInt'] = Field(default=None)
    'Indicates that appointments in the series of recurring appointments should occur on a specific day of the month.'
    monthInterval: 'positiveInt' = Field(default=...)
    'Indicates that recurring appointments should occur every nth month.'
    nthWeekOfMonth: Optional_['Coding'] = Field(default=None)
    'Indicates which week within a month the appointments in the series of recurring appointments should occur on.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AppointmentRecurrenceTemplateYearlyTemplate(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information about yearly recurring appointments."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    yearInterval: 'positiveInt' = Field(default=...)
    'Appointment recurs every nth year.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AppointmentRecurrenceTemplateWeeklyTemplate(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information about weekly recurring appointments."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    monday: Optional_['boolean'] = Field(default=None)
    'Indicates that recurring appointments should occur on Mondays.'
    friday: Optional_['boolean'] = Field(default=None)
    'Indicates that recurring appointments should occur on Fridays.'
    sunday: Optional_['boolean'] = Field(default=None)
    'Indicates that recurring appointments should occur on Sundays.'
    tuesday: Optional_['boolean'] = Field(default=None)
    'Indicates that recurring appointments should occur on Tuesdays.'
    thursday: Optional_['boolean'] = Field(default=None)
    'Indicates that recurring appointments should occur on Thursdays.'
    saturday: Optional_['boolean'] = Field(default=None)
    'Indicates that recurring appointments should occur on Saturdays.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    wednesday: Optional_['boolean'] = Field(default=None)
    'Indicates that recurring appointments should occur on Wednesdays.'
    weekInterval: Optional_['positiveInt'] = Field(default=None)
    'The interval defines if the recurrence is every nth week. The default is every week, so it is expected that this value will be 2 or more.\r\re.g. For recurring every second week this interval would be 2, or every third week the interval would be 3.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AppointmentParticipant(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """List of participants involved in the appointment."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: List_['CodeableConcept'] = Field(default_factory=list)
    'Role of participant in the appointment.'
    actor: Optional_['Reference'] = Field(default=None)
    'The individual, device, location, or service participating in the appointment.'
    period: Optional_['Period'] = Field(default=None)
    'Participation period of the actor.'
    status: 'code' = Field(default=...)
    'Participation status of the actor.'
    required: Optional_['boolean'] = Field(default=None)
    'Whether this participant is required to be present at the meeting. If false, the participant is optional.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Appointment(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A booking of a healthcare event among patient(s), practitioner(s), related person(s) and/or device(s) for a specific date/time. This may result in one or more Encounter(s)."""
    resourceType: Literal_['Appointment'] = Field(default=...)
    'A booking of a healthcare event among patient(s), practitioner(s), related person(s) and/or device(s) for a specific date/time. This may result in one or more Encounter(s)'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    end: Optional_['instant'] = Field(default=None)
    'Date/Time that the appointment is to conclude.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    slot: List_['Reference'] = Field(default_factory=list)
    "The slots from the participants' schedules that will be filled by the appointment."
    note: List_['Annotation'] = Field(default_factory=list)
    'Additional notes/comments about the appointment.'
    class_: List_['CodeableConcept'] = Field(default_factory=list, alias='class')
    'Concepts representing classification of patient encounter such as ambulatory (outpatient), inpatient, emergency, home health or others due to local variations.'
    start: Optional_['instant'] = Field(default=None)
    'Date/Time that the appointment is to take place.'
    status: 'code' = Field(default=...)
    'The overall status of the Appointment. Each of the participants has their own participation status which indicates their involvement in the process, however this status indicates the shared status.'
    reason: List_['CodeableReference'] = Field(default_factory=list)
    'The reason that this appointment is being scheduled. This is more clinical than administrative. This can be coded, or as specified using information from another resource. When the patient arrives and the encounter begins it may be used as the admission diagnosis. The indication will typically be a Condition (with other resources referenced in the evidence.detail), or a Procedure.'
    account: List_['Reference'] = Field(default_factory=list)
    'The set of accounts that is expected to be used for billing the activities that result from this Appointment.'
    created: Optional_['dateTime'] = Field(default=None)
    'The date that this appointment was initially created. This could be different to the meta.lastModified value on the initial entry, as this could have been before the resource was created on the FHIR server, and should remain unchanged over the lifespan of the appointment.'
    basedOn: List_['Reference'] = Field(default_factory=list)
    'The request this appointment is allocated to assess (e.g. incoming referral or procedure request).'
    subject: Optional_['Reference'] = Field(default=None)
    'The patient or group associated with the appointment, if they are to be present (usually) then they should also be included in the participant backbone element.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    priority: Optional_['CodeableConcept'] = Field(default=None)
    'The priority of the appointment. Can be used to make informed decisions if needing to re-prioritize appointments. (The iCal Standard specifies 0 as undefined, 1 as highest, 9 as lowest priority).'
    replaces: List_['Reference'] = Field(default_factory=list)
    'Appointment replaced by this Appointment in cases where there is a cancellation, the details of the cancellation can be found in the cancellationReason property (on the referenced resource).'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    specialty: List_['CodeableConcept'] = Field(default_factory=list)
    'The specialty of a practitioner that would be required to perform the service requested in this appointment.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'This records identifiers associated with this appointment concern that are defined by business processes and/or used to refer to it when a direct URL reference to the resource itself is not appropriate (e.g. in CDA documents, or in written / printed documentation).'
    serviceType: List_['CodeableReference'] = Field(default_factory=list)
    'The specific service that is to be performed during this appointment.'
    description: Optional_['string'] = Field(default=None)
    'The brief description of the appointment as would be shown on a subject line in a meeting request, or appointment list. Detailed or expanded information should be put in the note field.'
    participant: List_['AppointmentParticipant'] = Field(default=...)
    'List of participants involved in the appointment.'
    recurrenceId: Optional_['positiveInt'] = Field(default=None)
    'The sequence number that identifies a specific appointment in a recurring pattern.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    virtualService: List_['VirtualServiceDetail'] = Field(default_factory=list)
    'Connection details of a virtual service (e.g. conference call).'
    serviceCategory: List_['CodeableConcept'] = Field(default_factory=list)
    'A broad categorization of the service that is to be performed during this appointment.'
    appointmentType: Optional_['CodeableConcept'] = Field(default=None)
    'The style of appointment or patient that has been booked in the slot (not service type).'
    minutesDuration: Optional_['positiveInt'] = Field(default=None)
    'Number of minutes that the appointment is to take. This can be less than the duration between the start and end times.  For example, where the actual time of appointment is only an estimate or if a 30 minute appointment is being requested, but any time would work.  Also, if there is, for example, a planned 15 minute break in the middle of a long appointment, the duration may be 15 minutes less than the difference between the start and end.'
    requestedPeriod: List_['Period'] = Field(default_factory=list)
    'A set of date ranges (potentially including times) that the appointment is preferred to be scheduled within.\n\nThe duration (usually in minutes) could also be provided to indicate the length of the appointment to fill and populate the start/end times for the actual allocated time. However, in other situations the duration may be calculated by the scheduling system.'
    cancellationDate: Optional_['dateTime'] = Field(default=None)
    'The date/time describing when the appointment was cancelled.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    occurrenceChanged: Optional_['boolean'] = Field(default=None)
    'This appointment varies from the recurring pattern.'
    cancellationReason: Optional_['CodeableConcept'] = Field(default=None)
    'The coded reason for the appointment being cancelled. This is often used in reporting/billing/futher processing to determine if further actions are required, or specific fees apply.'
    patientInstruction: List_['CodeableReference'] = Field(default_factory=list)
    'While Appointment.note contains information for internal use, Appointment.patientInstructions is used to capture patient facing information about the Appointment (e.g. please bring your referral or fast from 8pm night before).'
    recurrenceTemplate: List_['AppointmentRecurrenceTemplate'] = Field(default_factory=list)
    'The details of the recurrence pattern or template that is used to generate recurring appointments.'
    previousAppointment: Optional_['Reference'] = Field(default=None)
    'The previous appointment in a series of related appointments.'
    supportingInformation: List_['Reference'] = Field(default_factory=list)
    'Additional information to support the appointment provided when making the appointment.'
    originatingAppointment: Optional_['Reference'] = Field(default=None)
    'The originating appointment in a recurring set of related appointments.'


class AppointmentResponse(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A reply to an appointment request for a patient and/or practitioner(s), such as a confirmation or rejection."""
    resourceType: Literal_['AppointmentResponse'] = Field(default=...)
    'A reply to an appointment request for a patient and/or practitioner(s), such as a confirmation or rejection'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    end: Optional_['instant'] = Field(default=None)
    'This may be either the same as the appointment request to confirm the details of the appointment, or alternately a new time to request a re-negotiation of the end time.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    start: Optional_['instant'] = Field(default=None)
    'Date/Time that the appointment is to take place, or requested new start time.'
    actor: Optional_['Reference'] = Field(default=None)
    'A Person, Location, HealthcareService, or Device that is participating in the appointment.'
    comment: Optional_['markdown'] = Field(default=None)
    'Additional comments about the appointment.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    recurring: Optional_['boolean'] = Field(default=None)
    'Indicates that this AppointmentResponse applies to all occurrences in a recurring request.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'This records identifiers associated with this appointment response concern that are defined by business processes and/ or used to refer to it when a direct URL reference to the resource itself is not appropriate.'
    appointment: 'Reference' = Field(default=...)
    'Appointment that this response is replying to.'
    recurrenceId: Optional_['positiveInt'] = Field(default=None)
    'The recurrence ID (sequence number) of the specific appointment when responding to a recurring request.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    occurrenceDate: Optional_['date'] = Field(default=None)
    'The original date within a recurring request. This could be used in place of the recurrenceId to be more direct (or where the template is provided through the simple list of dates in `Appointment.occurrenceDate`).'
    proposedNewTime: Optional_['boolean'] = Field(default=None)
    'Indicates that the response is proposing a different time that was initially requested.  The new proposed time will be indicated in the start and end properties.'
    participantType: List_['CodeableConcept'] = Field(default_factory=list)
    'Role of participant in the appointment.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    participantStatus: 'code' = Field(default=...)
    'Participation status of the participant. When the status is declined or tentative if the start/end times are different to the appointment, then these times should be interpreted as a requested time change. When the status is accepted, the times can either be the time of the appointment (as a confirmation of the time) or can be empty.'


class ArtifactAssessmentContent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A component comment, classifier, or rating of the artifact."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'Indicates what type of content this component represents.'
    path: List_['uri'] = Field(default_factory=list)
    'A URI that points to what the comment is about, such as a line of text in the CQL, or a specific element in a resource.'
    author: Optional_['Reference'] = Field(default=None)
    'Indicates who or what authored the content.'
    summary: Optional_['markdown'] = Field(default=None)
    'A brief summary of the content of this component.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'A quantitative rating of the artifact.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    component: List_['ArtifactAssessmentContent'] = Field(default_factory=list)
    'If the informationType is container, the components of the content.'
    classifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Represents a rating, classifier, or assessment of the artifact.'
    freeToShare: Optional_['boolean'] = Field(default=None)
    'Acceptable to publicly share the comment, classifier or rating.'
    informationType: Optional_['code'] = Field(default=None)
    'The type of information this component of the content represents.'
    relatedArtifact: List_['RelatedArtifact'] = Field(default_factory=list)
    'Additional related artifacts that provide supporting documentation, additional evidence, or further information related to the content.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ArtifactAssessment(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """This Resource provides one or more comments, classifiers or ratings about a Resource and supports attribution and rights management metadata for the added content."""
    resourceType: Literal_['ArtifactAssessment'] = Field(default=...)
    'Adds metadata-supported comments, classifiers or ratings related to a Resource'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    title: Optional_['string'] = Field(default=None)
    'A short title for the assessment for use in displaying and selecting.'
    content: List_['ArtifactAssessmentContent'] = Field(default_factory=list)
    'A component comment, classifier, or rating of the artifact.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    citeAsReference: Optional_['Reference'] = Field(default=None)
    'Display of or reference to the bibliographic citation of the comment, classifier, or rating.'
    citeAsMarkdown: Optional_['markdown'] = Field(default=None)
    'Display of or reference to the bibliographic citation of the comment, classifier, or rating.'
    copyright: Optional_['markdown'] = Field(default=None)
    'A copyright statement relating to the artifact assessment and/or its contents. Copyright statements are generally legal restrictions on the use and publishing of the artifact assessment.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this artifact assessment when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    artifactReference: Optional_['Reference'] = Field(default=None)
    'A reference to a resource, canonical resource, or non-FHIR resource which the comment or assessment is about.'
    artifactCanonical: Optional_['canonical'] = Field(default=None)
    'A reference to a resource, canonical resource, or non-FHIR resource which the comment or assessment is about.'
    artifactUri: Optional_['uri'] = Field(default=None)
    'A reference to a resource, canonical resource, or non-FHIR resource which the comment or assessment is about.'
    disposition: Optional_['code'] = Field(default=None)
    'Indicates the disposition of the responsible party to the comment or change request.'
    approvalDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was approved by the publisher. Approval happens once when the content is officially approved for usage.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    lastReviewDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was last reviewed. Review happens periodically after approval but does not change the original approval date.'
    workflowStatus: Optional_['code'] = Field(default=None)
    'Indicates the workflow status of the comment or change request.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    date: Optional_['dateTime'] = Field(default=None)
    'The date  (and optionally time) when the artifact assessment was published. The date must change when the disposition changes and it must change if the workflow status code changes. In addition, it should change when the substantive content of the artifact assessment changes.'


class AuditEventOutcome(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates whether the event succeeded or failed. A free text descripiton can be given in outcome.text."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: 'Coding' = Field(default=...)
    'Indicates whether the event succeeded or failed.'
    detail: List_['CodeableConcept'] = Field(default_factory=list)
    'Additional details about the error. This may be a text description of the error or a system code that identifies the error.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AuditEventEntity(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specific instances of data or objects that have been accessed."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    what: Optional_['Reference'] = Field(default=None)
    'Identifies a specific instance of the entity. The reference should be version specific. This is allowed to be a Parameters resource.'
    role: Optional_['CodeableConcept'] = Field(default=None)
    'Code representing the role the entity played in the event being audited.'
    query: Optional_['base64Binary'] = Field(default=None)
    'The query parameters for a query-type entities.'
    agent: List_['AuditEventAgent'] = Field(default_factory=list)
    "The entity is attributed to an agent to express the agent's responsibility for that entity in the activity. This is most used to indicate when persistence media (the entity) are used by an agent. For example when importing data from a device, the device would be described in an entity, and the user importing data from that media would be indicated as the entity.agent."
    detail: List_['AuditEventEntityDetail'] = Field(default_factory=list)
    'Tagged value pairs for conveying additional information about the entity.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    securityLabel: List_['CodeableConcept'] = Field(default_factory=list)
    'Security labels for the identified entity.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AuditEventEntityDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Tagged value pairs for conveying additional information about the entity."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'The type of extra detail provided in the value.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'The  value of the extra detail.'
    valueCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The  value of the extra detail.'
    valueString: Optional_['string'] = Field(default=None)
    'The  value of the extra detail.'
    valueBoolean: Optional_['boolean'] = Field(default=None)
    'The  value of the extra detail.'
    valueInteger: Optional_['integer'] = Field(default=None)
    'The  value of the extra detail.'
    valueRange: Optional_['Range'] = Field(default=None)
    'The  value of the extra detail.'
    valueRatio: Optional_['Ratio'] = Field(default=None)
    'The  value of the extra detail.'
    valueTime: Optional_['time'] = Field(default=None)
    'The  value of the extra detail.'
    valueDateTime: Optional_['dateTime'] = Field(default=None)
    'The  value of the extra detail.'
    valuePeriod: Optional_['Period'] = Field(default=None)
    'The  value of the extra detail.'
    valueBase64Binary: Optional_['base64Binary'] = Field(default=None)
    'The  value of the extra detail.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AuditEventSource(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The actor that is reporting the event."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    site: Optional_['Reference'] = Field(default=None)
    'Logical source location within the healthcare enterprise network.  For example, a hospital or other provider location within a multi-entity provider group.'
    type: List_['CodeableConcept'] = Field(default_factory=list)
    'Code specifying the type of source where event originated.'
    observer: 'Reference' = Field(default=...)
    'Identifier of the source where the event was detected.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AuditEventAgent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An actor taking an active role in the event or activity that is logged."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    who: 'Reference' = Field(default=...)
    'Reference to who this agent is that was involved in the event.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The Functional Role of the user when performing the event.'
    role: List_['CodeableConcept'] = Field(default_factory=list)
    "The structural roles of the agent indicating the agent's competency. The security role enabling the agent with respect to the activity."
    policy: List_['uri'] = Field(default_factory=list)
    'Where the policy(ies) are known that authorized the agent participation in the event. Typically, a single activity may have multiple applicable policies, such as patient consent, guarantor funding, etc. The policy would also indicate the security token used.'
    location: Optional_['Reference'] = Field(default=None)
    'Where the agent location is known, the agent location when the event occurred.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    requestor: Optional_['boolean'] = Field(default=None)
    'Indicator that the user is or is not the requestor, or initiator, for the event being audited.'
    networkReference: Optional_['Reference'] = Field(default=None)
    'When the event utilizes a network there should be an agent describing the local system, and an agent describing remote system, with the network interface details.'
    networkUri: Optional_['uri'] = Field(default=None)
    'When the event utilizes a network there should be an agent describing the local system, and an agent describing remote system, with the network interface details.'
    networkString: Optional_['string'] = Field(default=None)
    'When the event utilizes a network there should be an agent describing the local system, and an agent describing remote system, with the network interface details.'
    authorization: List_['CodeableConcept'] = Field(default_factory=list)
    'The authorization (e.g., PurposeOfUse) that was used during the event being recorded.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class AuditEvent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A record of an event relevant for purposes such as operations, privacy, security, maintenance, and performance analysis."""
    resourceType: Literal_['AuditEvent'] = Field(default=...)
    'Record of an event'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    agent: List_['AuditEventAgent'] = Field(default=...)
    'An actor taking an active role in the event or activity that is logged.'
    action: Optional_['code'] = Field(default=None)
    'Indicator for type of action performed during the event that generated the audit.'
    source: 'AuditEventSource' = Field(default=...)
    'The actor that is reporting the event.'
    entity: List_['AuditEventEntity'] = Field(default_factory=list)
    'Specific instances of data or objects that have been accessed.'
    outcome: Optional_['AuditEventOutcome'] = Field(default=None)
    'Indicates whether the event succeeded or failed. A free text descripiton can be given in outcome.text.'
    basedOn: List_['Reference'] = Field(default_factory=list)
    'Allows tracing of authorizatino for the events and tracking whether proposals/recommendations were acted upon.'
    patient: Optional_['Reference'] = Field(default=None)
    'The patient element is available to enable deterministic tracking of activities that involve the patient as the subject of the data used in an activity.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'Classification of the type of event.'
    severity: Optional_['code'] = Field(default=None)
    'Indicates and enables segmentation of various severity including debugging from critical.'
    recorded: 'instant' = Field(default=...)
    'The time when the event was recorded.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: Optional_['Reference'] = Field(default=None)
    'This will typically be the encounter the event occurred, but some events may be initiated prior to or after the official completion of an encounter but still be tied to the context of the encounter (e.g. pre-admission lab tests).'
    occurredPeriod: Optional_['Period'] = Field(default=None)
    'The time or period during which the activity occurred.'
    occurredDateTime: Optional_['dateTime'] = Field(default=None)
    'The time or period during which the activity occurred.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    authorization: List_['CodeableConcept'] = Field(default_factory=list)
    'The authorization (e.g., PurposeOfUse) that was used during the event being recorded.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'CodeableConcept' = Field(default=...)
    'Describes what happened. The most specific code for the event.'


class Basic(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Basic is used for handling concepts not yet defined in FHIR, narrative-only resources that don't map to an existing resource, and custom resources not appropriate for inclusion in the FHIR specification."""
    resourceType: Literal_['Basic'] = Field(default=...)
    'Resource for non-supported content'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    author: Optional_['Reference'] = Field(default=None)
    'Indicates who was responsible for creating the resource instance.'
    subject: Optional_['Reference'] = Field(default=None)
    'Identifies the patient, practitioner, device or any other resource that is the "focus" of this resource.'
    created: Optional_['dateTime'] = Field(default=None)
    'Identifies when the resource was first created.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Identifier assigned to the resource for business purposes, outside the context of FHIR.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'CodeableConcept' = Field(default=...)
    "Identifies the 'type' of resource - equivalent to the resource name for other resources."


class Binary(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A resource that represents the data of a single raw artifact as digital content accessible in its native format.  A Binary resource can contain any content, whether text, image, pdf, zip archive, etc."""
    resourceType: Literal_['Binary'] = Field(default=...)
    'Pure binary content defined by a format other than FHIR'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    data: Optional_['base64Binary'] = Field(default=None)
    'The actual content, base64 encoded.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    contentType: 'code' = Field(default=...)
    'MimeType of the binary content represented as a standard MimeType (BCP 13).'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    securityContext: Optional_['Reference'] = Field(default=None)
    'This element identifies another resource that can be used as a proxy of the security sensitivity to use when deciding and enforcing access control rules for the Binary resource. Given that the Binary resource contains very few elements that can be used to determine the sensitivity of the data and relationships to individuals, the referenced resource stands in as a proxy equivalent for this purpose. This referenced resource may be related to the Binary (e.g. DocumentReference), or may be some non-related Resource purely as a security proxy. E.g. to identify that the binary resource relates to a patient, and access should only be granted to applications that have access to the patient.'


class BiologicallyDerivedProductCollection(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """How this product was collected."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    source: Optional_['Reference'] = Field(default=None)
    'The patient or entity, such as a hospital or vendor in the case of a processed/manipulated/manufactured product, providing the product.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    collector: Optional_['Reference'] = Field(default=None)
    'Healthcare professional who is performing the collection.'
    collectedDateTime: Optional_['dateTime'] = Field(default=None)
    'Time of product collection.'
    collectedPeriod: Optional_['Period'] = Field(default=None)
    'Time of product collection.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class BiologicallyDerivedProductProperty(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A property that is specific to this BiologicallyDerviedProduct instance."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'Code that specifies the property. It should reference an established coding system.'
    valueBoolean: Optional_['boolean'] = Field(default=None)
    'Property values.'
    valueInteger: Optional_['integer'] = Field(default=None)
    'Property values.'
    valueCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Property values.'
    valuePeriod: Optional_['Period'] = Field(default=None)
    'Property values.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'Property values.'
    valueRange: Optional_['Range'] = Field(default=None)
    'Property values.'
    valueRatio: Optional_['Ratio'] = Field(default=None)
    'Property values.'
    valueString: Optional_['string'] = Field(default=None)
    'Property values.'
    valueAttachment: Optional_['Attachment'] = Field(default=None)
    'Property values.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class BiologicallyDerivedProduct(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """This resource reflects an instance of a biologically derived product. A material substance originating from a biological entity intended to be transplanted or infused
into another (possibly the same) biological entity."""
    resourceType: Literal_['BiologicallyDerivedProduct'] = Field(default=...)
    'This resource reflects an instance of a biologically derived product'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    parent: List_['Reference'] = Field(default_factory=list)
    'Parent product (if any) for this biologically-derived product.'
    request: List_['Reference'] = Field(default_factory=list)
    'Request to obtain and/or infuse this biologically derived product.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    division: Optional_['string'] = Field(default=None)
    'A unique identifier for an aliquot of a product.  Used to distinguish individual aliquots of a product carrying the same biologicalSource and productCode identifiers.'
    property: List_['BiologicallyDerivedProductProperty'] = Field(default_factory=list)
    'A property that is specific to this BiologicallyDerviedProduct instance.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Unique instance identifiers assigned to a biologically derived product. Note: This is a business identifier, not a resource identifier.'
    collection: Optional_['BiologicallyDerivedProductCollection'] = Field(default=None)
    'How this product was collected.'
    productCode: Optional_['CodeableConcept'] = Field(default=None)
    'A codified value that systematically supports characterization and classification of medical products of human origin inclusive of processing conditions such as additives, volumes and handling conditions.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    productStatus: Optional_['Coding'] = Field(default=None)
    'Whether the product is currently available.'
    expirationDate: Optional_['dateTime'] = Field(default=None)
    'Date, and where relevant time, of expiration.'
    productCategory: Optional_['Coding'] = Field(default=None)
    'Broad category of this product.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    processingFacility: List_['Reference'] = Field(default_factory=list)
    'Processing facilities responsible for the labeling and distribution of this biologically derived product.'
    biologicalSourceEvent: Optional_['Identifier'] = Field(default=None)
    'An identifier that supports traceability to the event during which material in this product from one or more biological entities was obtained or pooled.'
    storageTempRequirements: Optional_['Range'] = Field(default=None)
    'The temperature requirements for storage of the biologically-derived product.'


class BiologicallyDerivedProductDispensePerformer(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates who or what performed an action."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    actor: 'Reference' = Field(default=...)
    'Identifies the person responsible for the action.'
    function: Optional_['CodeableConcept'] = Field(default=None)
    'Identifies the function of the performer during the dispense.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class BiologicallyDerivedProductDispense(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """This resource reflects an instance of a biologically derived product dispense. The supply or dispense of a biologically derived product from the supply organization or department (e.g. hospital transfusion laboratory) to the clinical team responsible for clinical application."""
    resourceType: Literal_['BiologicallyDerivedProductDispense'] = Field(default=...)
    'A record of dispensation of a biologically derived product'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Additional notes.'
    partOf: List_['Reference'] = Field(default_factory=list)
    'A larger event of which this particular event is a component.'
    status: 'code' = Field(default=...)
    'A code specifying the state of the dispense event.'
    basedOn: List_['Reference'] = Field(default_factory=list)
    'The order or request that the dispense is fulfilling. This is a reference to a ServiceRequest resource.'
    product: 'Reference' = Field(default=...)
    'A link to a resource identifying the biologically derived product that is being dispensed.'
    patient: 'Reference' = Field(default=...)
    'A link to a resource representing the patient that the product is dispensed for.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    location: Optional_['Reference'] = Field(default=None)
    'The physical location where the dispense was performed.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'The amount of product in the dispense. Quantity will depend on the product being dispensed. Examples are: volume; cell count; concentration.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    performer: List_['BiologicallyDerivedProductDispensePerformer'] = Field(default_factory=list)
    'Indicates who or what performed an action.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Unique instance identifiers assigned to a biologically derived product dispense. Note: This is a business identifier, not a resource identifier.'
    matchStatus: Optional_['CodeableConcept'] = Field(default=None)
    'Indicates the type of matching associated with the dispense.'
    destination: Optional_['Reference'] = Field(default=None)
    'Link to a resource identifying the physical location that the product was dispatched to.'
    preparedDate: Optional_['dateTime'] = Field(default=None)
    'When the product was selected/ matched.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    whenHandedOver: Optional_['dateTime'] = Field(default=None)
    'When the product was dispatched for clinical use.'
    usageInstruction: Optional_['string'] = Field(default=None)
    'Specific instructions for use.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    originRelationshipType: Optional_['CodeableConcept'] = Field(default=None)
    'Indicates the relationship between the donor of the biologically derived product and the intended recipient.'


class BodyStructureIncludedStructure(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The anatomical location(s) or region(s) of the specimen, lesion, or body structure."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    structure: 'CodeableConcept' = Field(default=...)
    'Code that represents the included structure.'
    qualifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Code that represents the included structure qualifier.'
    laterality: Optional_['CodeableConcept'] = Field(default=None)
    'Code that represents the included structure laterality.'
    spatialReference: List_['Reference'] = Field(default_factory=list)
    'XY or XYZ-coordinate orientation for structure.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    bodyLandmarkOrientation: List_['BodyStructureIncludedStructureBodyLandmarkOrientation'] = Field(default_factory=list)
    'Body locations in relation to a specific body landmark (tatoo, scar, other body structure).'


class BodyStructureIncludedStructureBodyLandmarkOrientation(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Body locations in relation to a specific body landmark (tatoo, scar, other body structure)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    clockFacePosition: List_['CodeableConcept'] = Field(default_factory=list)
    'An description of the direction away from a landmark something is located based on a radial clock dial.'
    surfaceOrientation: List_['CodeableConcept'] = Field(default_factory=list)
    'The surface area a body location is in relation to a landmark.'
    landmarkDescription: List_['CodeableConcept'] = Field(default_factory=list)
    'A description of a landmark on the body used as a reference to locate something else.'
    distanceFromLandmark: List_['BodyStructureIncludedStructureBodyLandmarkOrientationDistanceFromLandmark'] = Field(default_factory=list)
    'The distance in centimeters a certain observation is made from a body landmark.'


class BodyStructureIncludedStructureBodyLandmarkOrientationDistanceFromLandmark(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The distance in centimeters a certain observation is made from a body landmark."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    value: List_['Quantity'] = Field(default_factory=list)
    'The measured distance (e.g., in cm) from a body landmark.'
    device: List_['CodeableReference'] = Field(default_factory=list)
    'An instrument, tool, analyzer, etc. used in the measurement.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class BodyStructure(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Record details about an anatomical structure.  This resource may be used when a coded concept does not provide the necessary detail needed for the use case."""
    resourceType: Literal_['BodyStructure'] = Field(default=...)
    'Specific and identified anatomical structure'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    image: List_['Attachment'] = Field(default_factory=list)
    'Image or images used to identify a location.'
    active: Optional_['boolean'] = Field(default=None)
    'Whether this body site is in active use.'
    patient: 'Reference' = Field(default=...)
    'The person to which the body site belongs.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Identifier for this instance of the anatomical structure.'
    morphology: Optional_['CodeableConcept'] = Field(default=None)
    'The kind of structure being represented by the body structure at `BodyStructure.location`.  This can define both normal and abnormal morphologies.'
    description: Optional_['markdown'] = Field(default=None)
    'A summary, characterization or explanation of the body structure.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    includedStructure: List_['BodyStructureIncludedStructure'] = Field(default=...)
    'The anatomical location(s) or region(s) of the specimen, lesion, or body structure.'
    excludedStructure: List_['BodyStructureIncludedStructure'] = Field(default_factory=list)
    'The anatomical location(s) or region(s) not occupied or represented by the specimen, lesion, or body structure.'


class BundleEntry(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An entry in a bundle resource - will either contain a resource or information about a resource (transactions and history only)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    link: List_['BundleLink'] = Field(default_factory=list)
    'A series of links that provide context to this entry.'
    search: Optional_['BundleEntrySearch'] = Field(default=None)
    'Information about the search process that lead to the creation of this entry.'
    fullUrl: Optional_['uri'] = Field(default=None)
    'The Absolute URL for the resource. Except for transactions and batches, each entry in a Bundle must have a fullUrl. The fullUrl SHALL NOT disagree with the id in the resource - i.e. if the fullUrl is not a urn:uuid, the URL shall be version-independent URL consistent with the Resource.id. The fullUrl is a version independent reference to the resource. Even when not required, fullUrl MAY be set to a urn:uuid to allow referencing entries in a transaction. The fullUrl can be an arbitrary URI and is not limited to urn:uuid, urn:oid, http, and https. The fullUrl element SHALL have a value except when: \n* invoking a create\n* invoking or responding to an operation where the body is not a single identified resource\n* invoking or returning the results of a search or history operation.'
    request: Optional_['BundleEntryRequest'] = Field(default=None)
    'Additional information about how this entry should be processed as part of a transaction or batch.  For history, it shows how the entry was processed to create the version contained in the entry.'
    resource: Optional_['AnyResource'] = Field(default=None)
    'The Resource for the entry. The purpose/meaning of the resource is determined by the Bundle.type. This is allowed to be a Parameters resource if and only if it is referenced by something else within the Bundle that provides context/meaning.'
    response: Optional_['BundleEntryResponse'] = Field(default=None)
    "Indicates the results of processing the corresponding 'request' entry in the batch or transaction being responded to or what the results of an operation where when returning history."
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class BundleEntryResponse(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates the results of processing the corresponding 'request' entry in the batch or transaction being responded to or what the results of an operation where when returning history."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    etag: Optional_['string'] = Field(default=None)
    'The Etag for the resource, if the operation for the entry produced a versioned resource (see [Resource Metadata and Versioning](http.html#versioning) and [Managing Resource Contention](http.html#concurrency)).'
    status: 'string' = Field(default=...)
    'The status code returned by processing this entry. The status SHALL start with a 3 digit HTTP code (e.g. 404) and may contain the standard HTTP description associated with the status code.'
    outcome: Optional_['AnyResource'] = Field(default=None)
    'An OperationOutcome containing hints and warnings produced as part of processing this entry in a batch or transaction.'
    location: Optional_['uri'] = Field(default=None)
    'The location header created by processing this operation, populated if the operation returns a location.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    lastModified: Optional_['instant'] = Field(default=None)
    'The date/time that the resource was modified on the server.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class BundleEntryRequest(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Additional information about how this entry should be processed as part of a transaction or batch.  For history, it shows how the entry was processed to create the version contained in the entry."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    url: 'uri' = Field(default=...)
    'The URL for this entry, relative to the root (the address to which the request is posted).'
    method: 'code' = Field(default=...)
    'In a transaction or batch, this is the HTTP action to be executed for this entry. In a history bundle, this indicates the HTTP action that occurred.'
    ifMatch: Optional_['string'] = Field(default=None)
    'Only perform the operation if the Etag value matches. For more information, see the API section ["Managing Resource Contention"](http.html#concurrency).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    ifNoneMatch: Optional_['string'] = Field(default=None)
    'If the ETag values match, return a 304 Not Modified status. See the API documentation for ["Conditional Read"](http.html#cread).'
    ifNoneExist: Optional_['string'] = Field(default=None)
    'Instruct the server not to perform the create if a specified resource already exists. For further information, see the API documentation for ["Conditional Create"](http.html#ccreate). This is just the query portion of the URL - what follows the "?" (not including the "?").'
    ifModifiedSince: Optional_['instant'] = Field(default=None)
    'Only perform the operation if the last updated date matches. See the API documentation for ["Conditional Read"](http.html#cread).'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class BundleEntrySearch(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information about the search process that lead to the creation of this entry."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    mode: Optional_['code'] = Field(default=None)
    "Why this entry is in the result set - whether it's included as a match or because of an _include requirement, or to convey information or warning information about the search process."
    score: Optional_['decimal'] = Field(default=None)
    "When searching, the server's search ranking score for the entry."
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class BundleLink(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A series of links that provide context to this bundle."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    url: 'uri' = Field(default=...)
    'The reference details for the link.'
    relation: 'code' = Field(default=...)
    'A name which details the functional use for this link - see [http://www.iana.org/assignments/link-relations/link-relations.xhtml#link-relations-1](http://www.iana.org/assignments/link-relations/link-relations.xhtml#link-relations-1).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Bundle(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A container for a collection of resources."""
    resourceType: Literal_['Bundle'] = Field(default=...)
    'Contains a collection of resources'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    type: 'code' = Field(default=...)
    'Indicates the purpose of this bundle - how it is intended to be used.'
    link: List_['BundleLink'] = Field(default_factory=list)
    'A series of links that provide context to this bundle.'
    total: Optional_['unsignedInt'] = Field(default=None)
    "If a set of search matches, this is the (potentially estimated) total number of entries of type 'match' across all pages in the search.  It does not include search.mode = 'include' or 'outcome' entries and it does not provide a count of the number of entries in the Bundle."
    entry: List_['BundleEntry'] = Field(default_factory=list)
    'An entry in a bundle resource - will either contain a resource or information about a resource (transactions and history only).'
    issues: Optional_['AnyResource'] = Field(default=None)
    'Captures issues and warnings that relate to the construction of the Bundle and the content within it.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    timestamp: Optional_['instant'] = Field(default=None)
    'The date/time that the bundle was assembled - i.e. when the resources were placed in the bundle.'
    signature: Optional_['Signature'] = Field(default=None)
    'Digital Signature - base64 encoded. XML-DSig or a JWS.'
    identifier: Optional_['Identifier'] = Field(default=None)
    "A persistent identifier for the bundle that won't change as a bundle is copied from server to server."
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'


class CanonicalResource(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Common Interface declaration for conformance and knowledge artifact resources."""
    resourceType: Literal_['CanonicalResource'] = Field(default=...)
    'Common Interface declaration for definitional resources'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this {{title}} when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this {{title}} is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the {{title}} is stored on different servers.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the {{title}}. This name should be usable as an identifier for the resource by machine processing applications such as code generation.'
    date: Optional_['dateTime'] = Field(default=None)
    'The date (and optionally time) when the {{title}} was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the {{title}} changes.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the {{title}}.'
    status: 'code' = Field(default=...)
    'The current state of this {{title}}. '
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the {{title}} when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the {{title}} author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence without additional knowledge.  (See the versionAlgorithm element.)'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    purpose: Optional_['markdown'] = Field(default=None)
    'Explanation of why this {{title}} is needed and why it has been designed as it has.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual responsible for the release and ongoing maintenance of the {{title}}.'
    copyright: Optional_['markdown'] = Field(default=None)
    'A copyright statement relating to the {{title}} and/or its contents. Copyright statements are generally legal restrictions on the use and publishing of the {{title}}.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this {{title}} when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate {{title}}s.'
    description: Optional_['markdown'] = Field(default=None)
    "A free text natural language description of the {{title}} from a consumer's perspective."
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this {{title}} is authored for testing purposes (or education/evaluation/marketing) and is not intended for genuine usage.'
    jurisdiction: List_['CodeableConcept'] = Field(default_factory=list)
    'A legal or geographic region in which the {{title}} is intended to be used.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    copyrightLabel: Optional_['string'] = Field(default=None)
    "A short string (<50 characters), suitable for inclusion in a page footer that identifies the copyright holder, effective period, and optionally whether rights are resctricted. (e.g. 'All rights reserved', 'Some rights reserved')."
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'


class CapabilityStatementImplementation(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Identifies a specific implementation instance that is described by the capability statement - i.e. a particular installation, rather than the capabilities of a software program."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    custodian: Optional_['Reference'] = Field(default=None)
    'The organization responsible for the management of the instance and oversight of the data on the server at the specified URL.'
    description: 'markdown' = Field(default=...)
    'Information about the specific installation that this capability statement relates to.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    url: Optional_['url'] = Field(default=None)
    'An absolute base URL for the implementation.  This forms the base for REST interfaces as well as the mailbox and document interfaces.'


class CapabilityStatementMessaging(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A description of the messaging capabilities of the solution."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    endpoint: List_['CapabilityStatementMessagingEndpoint'] = Field(default_factory=list)
    'An endpoint (network accessible address) to which messages and/or replies are to be sent.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reliableCache: Optional_['unsignedInt'] = Field(default=None)
    "Length if the receiver's reliable messaging cache in minutes (if a receiver) or how long the cache length on the receiver should be (if a sender)."
    documentation: Optional_['markdown'] = Field(default=None)
    "Documentation about the system's messaging capabilities for this endpoint not otherwise documented by the capability statement.  For example, the process for becoming an authorized messaging exchange partner."
    supportedMessage: List_['CapabilityStatementMessagingSupportedMessage'] = Field(default_factory=list)
    'References to message definitions for messages this system can send or receive.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CapabilityStatementMessagingSupportedMessage(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """References to message definitions for messages this system can send or receive."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    mode: 'code' = Field(default=...)
    'The mode of this event declaration - whether application is sender or receiver.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    definition: 'canonical' = Field(default=...)
    'Points to a message definition that identifies the messaging event, message structure, allowed responses, etc.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CapabilityStatementMessagingEndpoint(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An endpoint (network accessible address) to which messages and/or replies are to be sent."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    address: 'url' = Field(default=...)
    'The network address of the endpoint. For solutions that do not use network addresses for routing, it can be just an identifier.'
    protocol: 'Coding' = Field(default=...)
    'A list of the messaging transport protocol(s) identifiers, supported by this endpoint.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CapabilityStatementDocument(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A document definition."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    mode: 'code' = Field(default=...)
    'Mode of this document declaration - whether an application is a producer or consumer.'
    profile: 'canonical' = Field(default=...)
    'A profile on the document Bundle that constrains which resources are present, and their contents.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    documentation: Optional_['markdown'] = Field(default=None)
    'A description of how the application supports or uses the specified document profile.  For example, when documents are created, what action is taken with consumed documents, etc.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CapabilityStatementSoftware(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Software that is covered by this capability statement.  It is used when the capability statement describes the capabilities of a particular software version, independent of an installation."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    name: 'string' = Field(default=...)
    'Name the software is known by.'
    version: Optional_['string'] = Field(default=None)
    'The version identifier for the software covered by this statement.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    releaseDate: Optional_['dateTime'] = Field(default=None)
    'Date this version of the software was released.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CapabilityStatementRest(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A definition of the restful capabilities of the solution, if any."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    mode: 'code' = Field(default=...)
    'Identifies whether this portion of the statement is describing the ability to initiate or receive restful operations.'
    security: Optional_['CapabilityStatementRestSecurity'] = Field(default=None)
    'Information about security implementation from an interface perspective - what a client needs to know.'
    resource: List_['CapabilityStatementRestResource'] = Field(default_factory=list)
    'A specification of the restful capabilities of the solution for a specific resource type.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    operation: List_['CapabilityStatementRestResourceOperation'] = Field(default_factory=list)
    'Definition of an operation or a named query together with its parameters and their meaning and type.'
    interaction: List_['CapabilityStatementRestInteraction'] = Field(default_factory=list)
    'A specification of restful operations supported by the system.'
    searchParam: List_['CapabilityStatementRestResourceSearchParam'] = Field(default_factory=list)
    'Search parameters that are supported for searching all resources for implementations to support and/or make use of - either references to ones defined in the specification, or additional ones defined for/by the implementation. This is only for searches executed against the system-level endpoint.'
    compartment: List_['canonical'] = Field(default_factory=list)
    'An absolute URI which is a reference to the definition of a compartment that the system supports. The reference is to a CompartmentDefinition resource by its canonical URL .'
    documentation: Optional_['markdown'] = Field(default=None)
    "Information about the system's restful capabilities that apply across all applications, such as security."
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CapabilityStatementRestInteraction(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A specification of restful operations supported by the system."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    documentation: Optional_['markdown'] = Field(default=None)
    'Guidance specific to the implementation of this operation, such as limitations on the kind of transactions allowed, or information about system wide search is implemented.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'code' = Field(default=...)
    'A coded identifier of the operation, supported by the system.'


class CapabilityStatementRestResource(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A specification of the restful capabilities of the solution for a specific resource type."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'A type of resource exposed via the restful interface.'
    profile: Optional_['canonical'] = Field(default=None)
    'A system-wide profile that is applied across *all* instances of the resource supported by the system. For example, if declared on Observation, this profile is the "superset" of capabilities for laboratory *and* vitals *and* other domains. See further discussion in [Using Profiles](profiling.html#profile-uses).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    operation: List_['CapabilityStatementRestResourceOperation'] = Field(default_factory=list)
    'Definition of an operation or a named query together with its parameters and their meaning and type. Consult the definition of the operation for details about how to invoke the operation, and the parameters.'
    versioning: Optional_['code'] = Field(default=None)
    "This field is set to no-version to specify that the system does not support (server) or use (client) versioning for this resource type. If this has some other value, the server must at least correctly track and populate the versionId meta-property on resources. If the value is 'versioned-update', then the server supports all the versioning features, including using e-tags for version integrity in the API."
    interaction: List_['CapabilityStatementRestResourceInteraction'] = Field(default_factory=list)
    'Identifies a restful operation supported by the solution.'
    readHistory: Optional_['boolean'] = Field(default=None)
    'A flag for whether the server is able to return past versions as part of the vRead operation.'
    searchParam: List_['CapabilityStatementRestResourceSearchParam'] = Field(default_factory=list)
    'Search parameters for implementations to support and/or make use of - either references to ones defined in the specification, or additional ones defined for/by the implementation.'
    updateCreate: Optional_['boolean'] = Field(default=None)
    'A flag to indicate that the server allows or needs to allow the client to create new identities on the server (that is, the client PUTs to a location where there is no existing resource). Allowing this operation means that the server allows the client to create new identities on the server.'
    documentation: Optional_['markdown'] = Field(default=None)
    'Additional information about the resource type used by the system.'
    searchInclude: List_['string'] = Field(default_factory=list)
    'A list of _include values supported by the server.'
    conditionalRead: Optional_['code'] = Field(default=None)
    'A code that indicates how the server supports conditional read.'
    referencePolicy: List_['code'] = Field(default_factory=list)
    'A set of flags that defines how references are supported.'
    supportedProfile: List_['canonical'] = Field(default_factory=list)
    'A list of profiles representing different use cases the system hosts/produces. A supported profile is a statement about the functionality of the data and services provided by the server (or the client) for supported use cases. For example, a system can define and declare multiple Observation profiles for laboratory observations, vital sign observations, etc. By declaring supported profiles, systems provide a way to determine whether individual resources are conformant. See further discussion in [Using Profiles](profiling.html#profile-uses).'
    conditionalPatch: Optional_['boolean'] = Field(default=None)
    'A flag that indicates that the server supports conditional patch.'
    searchRevInclude: List_['string'] = Field(default_factory=list)
    'A list of _revinclude (reverse include) values supported by the server.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    conditionalCreate: Optional_['boolean'] = Field(default=None)
    'A flag that indicates that the server supports conditional create.'
    conditionalUpdate: Optional_['boolean'] = Field(default=None)
    'A flag that indicates that the server supports conditional update.'
    conditionalDelete: Optional_['code'] = Field(default=None)
    'A code that indicates how the server supports conditional delete.'


class CapabilityStatementRestResourceSearchParam(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Search parameters for implementations to support and/or make use of - either references to ones defined in the specification, or additional ones defined for/by the implementation."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    name: 'string' = Field(default=...)
    "The label used for the search parameter in this particular system's API - i.e. the 'name' portion of the name-value pair that will appear as part of the search URL.  This SHOULD be the same as the SearchParameter.code of the defining SearchParameter.  However, it can sometimes differ if necessary to disambiguate when a server supports multiple SearchParameters that happen to share the same code."
    type: 'code' = Field(default=...)
    'The type of value a search parameter refers to, and how the content is interpreted.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    definition: Optional_['canonical'] = Field(default=None)
    'An absolute URI that is a formal reference to where this parameter was first defined, so that a client can be confident of the meaning of the search parameter (a reference to [SearchParameter.url](searchparameter-definitions.html#SearchParameter.url)). This element SHALL be populated if the search parameter refers to a SearchParameter defined by the FHIR core specification or externally defined IGs.'
    documentation: Optional_['markdown'] = Field(default=None)
    'This allows documentation of any distinct behaviors about how the search parameter is used.  For example, text matching algorithms.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CapabilityStatementRestResourceInteraction(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Identifies a restful operation supported by the solution."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    documentation: Optional_['markdown'] = Field(default=None)
    "Guidance specific to the implementation of this operation, such as 'delete is a logical delete' or 'updates are only allowed with version id' or 'creates permitted from pre-authorized certificates only'."
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'code' = Field(default=...)
    'Coded identifier of the operation, supported by the system resource.'


class CapabilityStatementRestResourceOperation(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Definition of an operation or a named query together with its parameters and their meaning and type. Consult the definition of the operation for details about how to invoke the operation, and the parameters."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    name: 'string' = Field(default=...)
    'The name of the operation or query. For an operation, this name is prefixed with $ and used in the URL. For a query, this is the name used in the _query parameter when the query is called. This SHOULD be the same as the OperationDefinition.code of the defining OperationDefinition.  However, it can sometimes differ if necessary to disambiguate when a server supports multiple OperationDefinition that happen to share the same code.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    definition: 'canonical' = Field(default=...)
    "Where the formal definition can be found. If a server references the base definition of an Operation (i.e. from the specification itself such as ```http://hl7.org/fhir/OperationDefinition/ValueSet-expand```), that means it supports the full capabilities of the operation - e.g. both GET and POST invocation.  If it only supports a subset, it must define its own custom [OperationDefinition](operationdefinition.html#) with a 'base' of the original OperationDefinition.  The custom definition would describe the specific subset of functionality supported."
    documentation: Optional_['markdown'] = Field(default=None)
    'Documentation that describes anything special about the operation behavior, possibly detailing different behavior for system, type and instance-level invocation of the operation.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CapabilityStatementRestSecurity(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information about security implementation from an interface perspective - what a client needs to know."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    cors: Optional_['boolean'] = Field(default=None)
    'Server adds CORS headers when responding to requests - this enables Javascript applications to use the server.'
    service: List_['CodeableConcept'] = Field(default_factory=list)
    'Types of security services that are supported/required by the system.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    description: Optional_['markdown'] = Field(default=None)
    'General description of how security works.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CapabilityStatement(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A Capability Statement documents a set of capabilities (behaviors) of a FHIR Server or Client for a particular version of FHIR that may be used as a statement of actual server functionality or a statement of required or desired server implementation."""
    resourceType: Literal_['CapabilityStatement'] = Field(default=...)
    'A statement of system capabilities'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this capability statement when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this capability statement is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the capability statement is stored on different servers.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the capability statement. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    date: 'dateTime' = Field(default=...)
    'The date  (and optionally time) when the capability statement was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the capability statement changes.'
    kind: 'code' = Field(default=...)
    'The way that this statement is intended to be used, to describe an actual running instance of software, a particular product (kind, not instance of software) or a class of implementation (e.g. a desired purchase).'
    rest: List_['CapabilityStatementRest'] = Field(default_factory=list)
    'A definition of the restful capabilities of the solution, if any.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the capability statement.'
    status: 'code' = Field(default=...)
    'The status of this capability statement. Enables tracking the life-cycle of the content.'
    format: List_['code'] = Field(default=...)
    'A list of the formats supported by this implementation using their content types.'
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the capability statement when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the capability statement author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    purpose: Optional_['markdown'] = Field(default=None)
    'Explanation of why this capability statement is needed and why it has been designed as it has.'
    imports: List_['canonical'] = Field(default_factory=list)
    'Reference to a canonical URL of another CapabilityStatement that this software adds to. The capability statement automatically includes everything in the other statement, and it is not duplicated, though the server may repeat the same resources, interactions and operations to add additional details to them.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    software: Optional_['CapabilityStatementSoftware'] = Field(default=None)
    'Software that is covered by this capability statement.  It is used when the capability statement describes the capabilities of a particular software version, independent of an installation.'
    document: List_['CapabilityStatementDocument'] = Field(default_factory=list)
    'A document definition.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual responsible for the release and ongoing maintenance of the capability statement.'
    copyright: Optional_['markdown'] = Field(default=None)
    'A copyright statement relating to the capability statement and/or its contents. Copyright statements are generally legal restrictions on the use and publishing of the capability statement.'
    messaging: List_['CapabilityStatementMessaging'] = Field(default_factory=list)
    'A description of the messaging capabilities of the solution.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this CapabilityStatement when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate capability statement instances.'
    description: Optional_['markdown'] = Field(default=None)
    "A free text natural language description of the capability statement from a consumer's perspective. Typically, this is used when the capability statement describes a desired rather than an actual solution, for example as a formal expression of requirements as part of an RFP."
    fhirVersion: 'code' = Field(default=...)
    'The version of the FHIR specification that this CapabilityStatement describes (which SHALL be the same as the FHIR version of the CapabilityStatement itself). There is no default value.'
    patchFormat: List_['code'] = Field(default_factory=list)
    'A list of the patch formats supported by this implementation using their content types.'
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this capability statement is authored for testing purposes (or education/evaluation/marketing) and is not intended to be used for genuine usage.'
    jurisdiction: List_['CodeableConcept'] = Field(default_factory=list)
    'A legal or geographic region in which the capability statement is intended to be used.'
    instantiates: List_['canonical'] = Field(default_factory=list)
    'Reference to a canonical URL of another CapabilityStatement that this software implements. This capability statement is a published API description that corresponds to a business service. The server may actually implement a subset of the capability statement it claims to implement, so the capability statement must specify the full capability details.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    copyrightLabel: Optional_['string'] = Field(default=None)
    "A short string (<50 characters), suitable for inclusion in a page footer that identifies the copyright holder, effective period, and optionally whether rights are resctricted. (e.g. 'All rights reserved', 'Some rights reserved')."
    implementation: Optional_['CapabilityStatementImplementation'] = Field(default=None)
    'Identifies a specific implementation instance that is described by the capability statement - i.e. a particular installation, rather than the capabilities of a software program.'
    acceptLanguage: List_['code'] = Field(default_factory=list)
    'A list of the languages supported by this implementation that are usefully supported in the ```Accept-Language``` header.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    implementationGuide: List_['canonical'] = Field(default_factory=list)
    'A list of implementation guides that the server does (or should) support in their entirety.'


class CarePlanActivity(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Identifies an action that has occurred or is a planned action to occur as part of the plan. For example, a medication to be used, lab tests to perform, self-monitoring that has occurred, education etc."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    progress: List_['Annotation'] = Field(default_factory=list)
    'Notes about the adherence/status/progress of the activity.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    performedActivity: List_['CodeableReference'] = Field(default_factory=list)
    'Identifies the activity that was performed. For example, an activity could be patient education, exercise, or a medication administration. The reference to an "event" resource, such as Procedure or Encounter or Observation, represents the activity that was performed. The requested activity can be conveyed using the CarePlan.activity.plannedActivityReference (a reference to a “request” resource).'
    plannedActivityReference: Optional_['Reference'] = Field(default=None)
    'The details of the proposed activity represented in a specific resource.'


class CarePlan(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Describes the intention of how one or more practitioners intend to deliver care for a particular patient, group or community for a period of time, possibly limited to care for a specific condition or set of conditions."""
    resourceType: Literal_['CarePlan'] = Field(default=...)
    'Healthcare plan for patient or group'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    goal: List_['Reference'] = Field(default_factory=list)
    'Describes the intended objective(s) of carrying out the care plan.'
    note: List_['Annotation'] = Field(default_factory=list)
    'General notes about the care plan not covered elsewhere.'
    title: Optional_['string'] = Field(default=None)
    'Human-friendly name for the care plan.'
    partOf: List_['Reference'] = Field(default_factory=list)
    'A larger care plan of which this particular care plan is a component or step.'
    status: 'code' = Field(default=...)
    'Indicates whether the plan is currently being acted upon, represents future intentions or is now a historical record.'
    intent: 'code' = Field(default=...)
    'Indicates the level of authority/intentionality associated with the care plan and where the care plan fits into the workflow chain.'
    period: Optional_['Period'] = Field(default=None)
    'Indicates when the plan did (or is intended to) come into effect and end.'
    basedOn: List_['Reference'] = Field(default_factory=list)
    'A higher-level request resource (i.e. a plan, proposal or order) that is fulfilled in whole or in part by this care plan.'
    subject: 'Reference' = Field(default=...)
    'Identifies the patient or group whose intended care is described by the plan.'
    created: Optional_['dateTime'] = Field(default=None)
    'Represents when this particular CarePlan record was created in the system, which is often a system-generated date.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    replaces: List_['Reference'] = Field(default_factory=list)
    'Completed or terminated care plan whose function is taken by this new care plan.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'Identifies what "kind" of plan this is to support differentiation between multiple co-existing plans; e.g. "Home health", "psychiatric", "asthma", "disease management", "wellness plan", etc.'
    careTeam: List_['Reference'] = Field(default_factory=list)
    'Identifies all people and organizations who are expected to be involved in the care envisioned by this plan.'
    activity: List_['CarePlanActivity'] = Field(default_factory=list)
    'Identifies an action that has occurred or is a planned action to occur as part of the plan. For example, a medication to be used, lab tests to perform, self-monitoring that has occurred, education etc.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: Optional_['Reference'] = Field(default=None)
    'The Encounter during which this CarePlan was created or to which the creation of this record is tightly associated.'
    custodian: Optional_['Reference'] = Field(default=None)
    'When populated, the custodian is responsible for the care plan. The care plan is attributed to the custodian.'
    addresses: List_['CodeableReference'] = Field(default_factory=list)
    'Identifies the conditions/problems/concerns/diagnoses/etc. whose management and/or mitigation are handled by this plan.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Business identifiers assigned to this care plan by the performer or other systems which remain constant as the resource is updated and propagates from server to server.'
    description: Optional_['string'] = Field(default=None)
    'A description of the scope and nature of the plan.'
    contributor: List_['Reference'] = Field(default_factory=list)
    'Identifies the individual(s), organization or device who provided the contents of the care plan.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    supportingInfo: List_['Reference'] = Field(default_factory=list)
    "Identifies portions of the patient's record that specifically influenced the formation of the plan.  These might include comorbidities, recent procedures, limitations, recent assessments, etc."
    instantiatesUri: List_['uri'] = Field(default_factory=list)
    'The URL pointing to an externally maintained protocol, guideline, questionnaire or other definition that is adhered to in whole or in part by this CarePlan.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    instantiatesCanonical: List_['canonical'] = Field(default_factory=list)
    'The URL pointing to a FHIR-defined protocol, guideline, questionnaire or other definition that is adhered to in whole or in part by this CarePlan.'


class CareTeamParticipant(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Identifies all people and organizations who are expected to be involved in the care team."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    role: Optional_['CodeableConcept'] = Field(default=None)
    'Indicates specific responsibility of an individual within the care team, such as "Primary care physician", "Trained social worker counselor", "Caregiver", etc.'
    member: Optional_['Reference'] = Field(default=None)
    'The specific person or organization who is participating/expected to participate in the care team.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    onBehalfOf: Optional_['Reference'] = Field(default=None)
    'The organization of the practitioner.'
    coveragePeriod: Optional_['Period'] = Field(default=None)
    'When the member is generally available within this care team.'
    coverageTiming: Optional_['Timing'] = Field(default=None)
    'When the member is generally available within this care team.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CareTeam(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The Care Team includes all the people and organizations who plan to participate in the coordination and delivery of care."""
    resourceType: Literal_['CareTeam'] = Field(default=...)
    'Planned participants in the coordination and delivery of care'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A label for human use intended to distinguish like teams.  E.g. the "red" vs. "green" trauma teams.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Comments made about the CareTeam.'
    status: Optional_['code'] = Field(default=None)
    'Indicates the current state of the care team.'
    period: Optional_['Period'] = Field(default=None)
    'Indicates when the team did (or is intended to) come into effect and end.'
    reason: List_['CodeableReference'] = Field(default_factory=list)
    'Describes why the care team exists.'
    subject: Optional_['Reference'] = Field(default=None)
    'Identifies the patient or group whose intended care is handled by the team.'
    telecom: List_['ContactPoint'] = Field(default_factory=list)
    'A central contact detail for the care team (that applies to all members).'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'Identifies what kind of team.  This is to support differentiation between multiple co-existing teams, such as care plan team, episode of care team, longitudinal care team.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Business identifiers assigned to this care team by the performer or other systems which remain constant as the resource is updated and propagates from server to server.'
    participant: List_['CareTeamParticipant'] = Field(default_factory=list)
    'Identifies all people and organizations who are expected to be involved in the care team.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    managingOrganization: List_['Reference'] = Field(default_factory=list)
    'The organization responsible for the care team.'


class ChargeItemPerformer(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates who or what performed or participated in the charged service."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    actor: 'Reference' = Field(default=...)
    'The device, practitioner, etc. who performed or participated in the service.'
    function: Optional_['CodeableConcept'] = Field(default=None)
    'Describes the type of performance or participation(e.g. primary surgeon, anesthesiologiest, etc.).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ChargeItem(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The resource ChargeItem describes the provision of healthcare provider products for a certain patient, therefore referring not only to the product, but containing in addition details of the provision, like date, time, amounts and participating organizations and persons. Main Usage of the ChargeItem is to enable the billing process and internal cost allocation."""
    resourceType: Literal_['ChargeItem'] = Field(default=...)
    'Item containing charge code(s) associated with the provision of healthcare provider products'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Comments made about the event by the performer, subject or other participants.'
    status: 'code' = Field(default=...)
    'The current state of the ChargeItem.'
    partOf: List_['Reference'] = Field(default_factory=list)
    'ChargeItems can be grouped to larger ChargeItems covering the whole set.'
    reason: List_['CodeableConcept'] = Field(default_factory=list)
    'Describes why the event occurred in coded or textual form.'
    subject: 'Reference' = Field(default=...)
    'The individual or set of individuals the action is being or was performed on.'
    enterer: Optional_['Reference'] = Field(default=None)
    'The device, practitioner, etc. who entered the charge item.'
    service: List_['CodeableReference'] = Field(default_factory=list)
    'Indicated the rendered service that caused this charge.'
    product: List_['CodeableReference'] = Field(default_factory=list)
    'Identifies the device, food, drug or other product being charged either by type code or reference to an instance.'
    account: List_['Reference'] = Field(default_factory=list)
    'Account into which this ChargeItems belongs.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'Quantity of which the charge item has been serviced.'
    bodysite: List_['CodeableConcept'] = Field(default_factory=list)
    'The anatomical location where the related service has been applied.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: Optional_['Reference'] = Field(default=None)
    'This ChargeItem has the details of how the associated Encounter should be billed or otherwise be handled by finance systems.'
    performer: List_['ChargeItemPerformer'] = Field(default_factory=list)
    'Indicates who or what performed or participated in the charged service.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Identifiers assigned to this event performer or other systems.'
    costCenter: Optional_['Reference'] = Field(default=None)
    'The financial cost center permits the tracking of charge attribution.'
    enteredDate: Optional_['dateTime'] = Field(default=None)
    'Date the charge item was entered.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    definitionUri: List_['uri'] = Field(default_factory=list)
    'References the (external) source of pricing information, rules of application for the code this ChargeItem uses.'
    occurrenceDateTime: Optional_['dateTime'] = Field(default=None)
    'Date/time(s) or duration when the charged service was applied.'
    occurrencePeriod: Optional_['Period'] = Field(default=None)
    'Date/time(s) or duration when the charged service was applied.'
    occurrenceTiming: Optional_['Timing'] = Field(default=None)
    'Date/time(s) or duration when the charged service was applied.'
    overrideReason: Optional_['CodeableConcept'] = Field(default=None)
    'If the list price or the rule-based factor associated with the code is overridden, this attribute can capture a text to indicate the  reason for this action.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    unitPriceComponent: Optional_['MonetaryComponent'] = Field(default=None)
    'The unit price of the chargable item.'
    definitionCanonical: List_['canonical'] = Field(default_factory=list)
    'References the source of pricing information, rules of application for the code this ChargeItem uses.'
    totalPriceComponent: Optional_['MonetaryComponent'] = Field(default=None)
    'The total price for the chargable item, accounting for the quantity.'
    supportingInformation: List_['Reference'] = Field(default_factory=list)
    'Further information supporting this charge.'
    performingOrganization: Optional_['Reference'] = Field(default=None)
    'The organization performing the service.'
    requestingOrganization: Optional_['Reference'] = Field(default=None)
    'The organization requesting the service.'
    code: 'CodeableConcept' = Field(default=...)
    'A code that identifies the charge, like a billing code.'


class ChargeItemDefinitionPropertyGroup(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Group of properties which are applicable under the same conditions. If no applicability rules are established for the group, then all properties always apply."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    applicability: List_['ChargeItemDefinitionApplicability'] = Field(default_factory=list)
    'Expressions that describe applicability criteria for the priceComponent.'
    priceComponent: List_['MonetaryComponent'] = Field(default_factory=list)
    'The price for a ChargeItem may be calculated as a base price with surcharges/deductions that apply in certain conditions. A ChargeItemDefinition resource that defines the prices, factors and conditions that apply to a billing code is currently under development. The priceComponent element can be used to offer transparency to the recipient of the Invoice of how the prices have been calculated.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ChargeItemDefinitionApplicability(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Expressions that describe applicability criteria for the billing code."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    condition: Optional_['Expression'] = Field(default=None)
    'An expression that returns true or false, indicating whether the condition is satisfied. When using FHIRPath expressions, the %context environment variable must be replaced at runtime with the ChargeItem resource to which this definition is applied.'
    effectivePeriod: Optional_['Period'] = Field(default=None)
    'The period during which the charge item definition content was or is planned to be in active use.'
    relatedArtifact: Optional_['RelatedArtifact'] = Field(default=None)
    'Reference to / quotation of the external source of the group of properties.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ChargeItemDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The ChargeItemDefinition resource provides the properties that apply to the (billing) codes necessary to calculate costs and prices. The properties may differ largely depending on type and realm, therefore this resource gives only a rough structure and requires profiling for each type of billing code system."""
    resourceType: Literal_['ChargeItemDefinition'] = Field(default=...)
    'Definition of properties and rules about how the price and the applicability of a ChargeItem can be determined'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this charge item definition when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this charge item definition is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the charge item definition is stored on different servers.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the ChargeItemDefinition. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the charge item definition.'
    partOf: List_['canonical'] = Field(default_factory=list)
    'A larger definition of which this particular definition is a component or step.'
    status: 'code' = Field(default=...)
    'The current state of the ChargeItemDefinition.'
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the charge item definition when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the charge item definition author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence. To provide a version consistent with the Decision Support Service specification, use the format Major.Minor.Revision (e.g. 1.0.0). For more information on versioning knowledge assets, refer to the Decision Support Service specification. Note that a version is required for non-experimental active assets.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    purpose: Optional_['markdown'] = Field(default=None)
    'Explanation of why this charge item definition is needed and why it has been designed as it has.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    replaces: List_['canonical'] = Field(default_factory=list)
    'As new versions of a protocol or guideline are defined, allows identification of what versions are replaced by a new instance.'
    instance: List_['Reference'] = Field(default_factory=list)
    'The defined billing details in this resource pertain to the given product instance(s).'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual responsible for the release and ongoing maintenance of the charge item definition.'
    copyright: Optional_['markdown'] = Field(default=None)
    'A copyright statement relating to the charge item definition and/or its contents. Copyright statements are generally legal restrictions on the use and publishing of the charge item definition.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this charge item definition when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate charge item definition instances.'
    description: Optional_['markdown'] = Field(default=None)
    "A free text natural language description of the charge item definition from a consumer's perspective."
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this charge item definition is authored for testing purposes (or education/evaluation/marketing) and is not intended to be used for genuine usage.'
    jurisdiction: List_['CodeableConcept'] = Field(default_factory=list)
    'A legal or geographic region in which the charge item definition is intended to be used.'
    approvalDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was approved by the publisher. Approval happens once when the content is officially approved for usage.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    applicability: List_['ChargeItemDefinitionApplicability'] = Field(default_factory=list)
    'Expressions that describe applicability criteria for the billing code.'
    propertyGroup: List_['ChargeItemDefinitionPropertyGroup'] = Field(default_factory=list)
    'Group of properties which are applicable under the same conditions. If no applicability rules are established for the group, then all properties always apply.'
    derivedFromUri: List_['uri'] = Field(default_factory=list)
    'The URL pointing to an externally-defined charge item definition that is adhered to in whole or in part by this definition.'
    copyrightLabel: Optional_['string'] = Field(default=None)
    "A short string (<50 characters), suitable for inclusion in a page footer that identifies the copyright holder, effective period, and optionally whether rights are resctricted. (e.g. 'All rights reserved', 'Some rights reserved')."
    lastReviewDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was last reviewed. Review happens periodically after approval but does not change the original approval date.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    date: Optional_['dateTime'] = Field(default=None)
    'The date  (and optionally time) when the charge item definition was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the charge item definition changes.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'The defined billing details in this resource pertain to the given billing code.'


class CitationClassification(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The assignment to an organizing scheme."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The kind of classifier (e.g. publication type, keyword).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    classifier: List_['CodeableConcept'] = Field(default_factory=list)
    'The specific classification value.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationCitedArtifact(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The article or artifact being described."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    part: Optional_['CitationCitedArtifactPart'] = Field(default=None)
    'The component of the article or artifact.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Any additional information or content for the article or artifact.'
    title: List_['CitationCitedArtifactTitle'] = Field(default_factory=list)
    'The title details of the article or artifact.'
    version: Optional_['CitationCitedArtifactVersion'] = Field(default=None)
    'The defined version of the cited artifact.'
    abstract: List_['CitationCitedArtifactAbstract'] = Field(default_factory=list)
    'The abstract may be used to convey article-contained abstracts, externally-created abstracts, or other descriptive summaries.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    relatesTo: List_['CitationCitedArtifactRelatesTo'] = Field(default_factory=list)
    'The artifact related to the cited artifact.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify the cited artifact when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    statusDate: List_['CitationCitedArtifactStatusDate'] = Field(default_factory=list)
    'An effective date or period, historical or future, actual or expected, for a status of the cited artifact.'
    webLocation: List_['CitationCitedArtifactWebLocation'] = Field(default_factory=list)
    'Used for any URL for the article or artifact cited.'
    dateAccessed: Optional_['dateTime'] = Field(default=None)
    'When the cited artifact was accessed.'
    currentState: List_['CodeableConcept'] = Field(default_factory=list)
    'The status of the cited artifact.'
    classification: List_['CitationCitedArtifactClassification'] = Field(default_factory=list)
    'The assignment to an organizing scheme.'
    publicationForm: List_['CitationCitedArtifactPublicationForm'] = Field(default_factory=list)
    'If multiple, used to represent alternative forms of the article that are not separate citations.'
    contributorship: Optional_['CitationCitedArtifactContributorship'] = Field(default=None)
    'This element is used to list authors and other contributors, their contact information, specific contributions, and summary statements.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    relatedIdentifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify things closely related to the cited artifact.'


class CitationCitedArtifactContributorship(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """This element is used to list authors and other contributors, their contact information, specific contributions, and summary statements."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    entry: List_['CitationCitedArtifactContributorshipEntry'] = Field(default_factory=list)
    'An individual entity named as a contributor, for example in the author list or contributor list.'
    summary: List_['CitationCitedArtifactContributorshipSummary'] = Field(default_factory=list)
    'Used to record a display of the author/contributor list without separate data element for each list member.'
    complete: Optional_['boolean'] = Field(default=None)
    'Indicates if the list includes all authors and/or contributors.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationCitedArtifactContributorshipSummary(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Used to record a display of the author/contributor list without separate data element for each list member."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'Used most commonly to express an author list or a contributorship statement.'
    style: Optional_['CodeableConcept'] = Field(default=None)
    'The format for the display string, such as author last name with first letter capitalized followed by forename initials.'
    value: 'markdown' = Field(default=...)
    'The display string for the author list, contributor list, or contributorship statement.'
    source: Optional_['CodeableConcept'] = Field(default=None)
    'Used to code the producer or rule for creating the display string.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationCitedArtifactContributorshipEntry(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An individual entity named as a contributor, for example in the author list or contributor list."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    role: Optional_['CodeableConcept'] = Field(default=None)
    'The role of the contributor (e.g. author, editor, reviewer, funder).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    contributor: 'Reference' = Field(default=...)
    'The identity of the individual contributor.'
    affiliation: List_['Reference'] = Field(default_factory=list)
    'Organization affiliated with the contributor.'
    rankingOrder: Optional_['positiveInt'] = Field(default=None)
    'Provides a numerical ranking to represent the degree of contributorship relative to other contributors, such as 1 for first author and 2 for second author.'
    forenameInitials: Optional_['string'] = Field(default=None)
    'For citation styles that use initials.'
    contributionType: List_['CodeableConcept'] = Field(default_factory=list)
    'This element identifies the specific nature of an individual’s contribution with respect to the cited work.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    contributionInstance: List_['CitationCitedArtifactContributorshipEntryContributionInstance'] = Field(default_factory=list)
    'Contributions with accounting for time or number.'
    correspondingContact: Optional_['boolean'] = Field(default=None)
    'Whether the contributor is the corresponding contributor for the role.'


class CitationCitedArtifactContributorshipEntryContributionInstance(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Contributions with accounting for time or number."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'The specific contribution.'
    time: Optional_['dateTime'] = Field(default=None)
    'The time that the contribution was made.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationCitedArtifactPublicationForm(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """If multiple, used to represent alternative forms of the article that are not separate citations."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    issue: Optional_['string'] = Field(default=None)
    'Issue, part or supplement of journal or other collection in which the article is published.'
    volume: Optional_['string'] = Field(default=None)
    'Volume number of journal or other collection in which the article is published.'
    language: List_['CodeableConcept'] = Field(default_factory=list)
    'The language or languages in which this form of the article is published.'
    lastPage: Optional_['string'] = Field(default=None)
    'Used for isolated representation of last page.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    firstPage: Optional_['string'] = Field(default=None)
    'Used for isolated representation of first page.'
    pageCount: Optional_['string'] = Field(default=None)
    'Actual or approximate number of pages or screens. Distinct from reporting the page numbers.'
    copyright: Optional_['markdown'] = Field(default=None)
    'Copyright notice for the full article or artifact.'
    pageString: Optional_['string'] = Field(default=None)
    'Used for full display of pagination.'
    publishedIn: Optional_['CitationCitedArtifactPublicationFormPublishedIn'] = Field(default=None)
    'The collection the cited article or artifact is published in.'
    citedMedium: Optional_['CodeableConcept'] = Field(default=None)
    'Describes the form of the medium cited. Common codes are "Internet" or "Print". The CitedMedium value set has 6 codes. The codes internet, print, and offline-digital-storage are the common codes for a typical publication form, though internet and print are more common for study citations. Three additional codes (each appending one of the primary codes with "-without-issue" are used for situations when a study is published both within an issue (of a periodical release as commonly done for journals) AND is published separately from the issue (as commonly done with early online publication), to represent specific identification of the publication form not associated with the issue.'
    articleDate: Optional_['dateTime'] = Field(default=None)
    'The date the article was added to the database, or the date the article was released.'
    accessionNumber: Optional_['string'] = Field(default=None)
    'Entry number or identifier for inclusion in a database.'
    lastRevisionDate: Optional_['dateTime'] = Field(default=None)
    'The date the article was last revised or updated in the database.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    publicationDateText: Optional_['string'] = Field(default=None)
    'Text representation of the date on which the issue of the cited artifact was published.'
    publicationDateSeason: Optional_['string'] = Field(default=None)
    'Spring, Summer, Fall/Autumn, Winter.'


class CitationCitedArtifactPublicationFormPublishedIn(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The collection the cited article or artifact is published in."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'Kind of container (e.g. Periodical, database, or book).'
    title: Optional_['string'] = Field(default=None)
    'Name of the database or title of the book or journal.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['Reference'] = Field(default=None)
    'Name of or resource describing the publisher.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Journal identifiers include ISSN, ISO Abbreviation and NLMuniqueID; Book identifiers include ISBN.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    publisherLocation: Optional_['string'] = Field(default=None)
    'Geographic location of the publisher.'


class CitationCitedArtifactClassification(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The assignment to an organizing scheme."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The kind of classifier (e.g. publication type, keyword).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    classifier: List_['CodeableConcept'] = Field(default_factory=list)
    'The specific classification value.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    artifactAssessment: List_['Reference'] = Field(default_factory=list)
    'Complex or externally created classification.'


class CitationCitedArtifactWebLocation(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Used for any URL for the article or artifact cited."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    url: Optional_['uri'] = Field(default=None)
    'The specific URL.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    classifier: List_['CodeableConcept'] = Field(default_factory=list)
    'A characterization of the object expected at the web location.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationCitedArtifactStatusDate(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An effective date or period, historical or future, actual or expected, for a status of the cited artifact."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    actual: Optional_['boolean'] = Field(default=None)
    'Either occurred or expected.'
    period: 'Period' = Field(default=...)
    'When the status started and/or ended.'
    activity: 'CodeableConcept' = Field(default=...)
    'A definition of the status associated with a date or period.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationCitedArtifactRelatesTo(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The artifact related to the cited artifact."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'The type of relationship to the related artifact.'
    label: Optional_['string'] = Field(default=None)
    'A short label that can be used to reference the related artifact from elsewhere in the containing artifact, such as a footnote index.'
    display: Optional_['string'] = Field(default=None)
    'A brief description of the document or knowledge resource being referenced, suitable for display to a consumer.'
    citation: Optional_['markdown'] = Field(default=None)
    'A bibliographic citation for the related artifact. This text SHOULD be formatted according to an accepted citation format.'
    document: Optional_['Attachment'] = Field(default=None)
    'The document being referenced, represented as an attachment. Do not use this element if using the resource element to provide the canonical to the related artifact.'
    resource: Optional_['canonical'] = Field(default=None)
    'The related artifact, such as a library, value set, profile, or other knowledge resource.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    classifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Provides additional classifiers of the related artifact.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    resourceReference: Optional_['Reference'] = Field(default=None)
    'The related artifact, if the artifact is not a canonical resource, or a resource reference to a canonical resource.'


class CitationCitedArtifactAbstract(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The abstract may be used to convey article-contained abstracts, externally-created abstracts, or other descriptive summaries."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'Used to express the reason for or classification of the abstract.'
    text: 'markdown' = Field(default=...)
    'Abstract content.'
    language: Optional_['CodeableConcept'] = Field(default=None)
    'Used to express the specific language of the abstract.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    copyright: Optional_['markdown'] = Field(default=None)
    'Copyright notice for the abstract.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationCitedArtifactVersion(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The defined version of the cited artifact."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    value: 'string' = Field(default=...)
    'The version number or other version identifier.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    baseCitation: Optional_['Reference'] = Field(default=None)
    'Citation for the main version of the cited artifact.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationCitedArtifactTitle(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The title details of the article or artifact."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: List_['CodeableConcept'] = Field(default_factory=list)
    'Used to express the reason for or classification of the title.'
    text: 'markdown' = Field(default=...)
    'The title of the article or artifact.'
    language: Optional_['CodeableConcept'] = Field(default=None)
    'Used to express the specific language of the title.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationCitedArtifactPart(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The component of the article or artifact."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The kind of component.'
    value: Optional_['string'] = Field(default=None)
    'The specification of the component.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    baseCitation: Optional_['Reference'] = Field(default=None)
    'The citation for the full article or artifact.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationStatusDate(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The state or status of the citation record paired with an effective date or period for that state."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    actual: Optional_['boolean'] = Field(default=None)
    'Whether the status date is actual (has occurred) or expected (estimated or anticipated).'
    period: 'Period' = Field(default=...)
    'When the status started and/or ended.'
    activity: 'CodeableConcept' = Field(default=...)
    'The state or status of the citation record (that will be paired with the period).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CitationSummary(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A human-readable display of key concepts to represent the citation."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    text: 'markdown' = Field(default=...)
    'The human-readable display of the citation summary.'
    style: Optional_['CodeableConcept'] = Field(default=None)
    'Format for display of the citation summary.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Citation(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The Citation Resource enables reference to any knowledge artifact for purposes of identification and attribution. The Citation Resource supports existing reference structures and developing publication practices such as versioning, expressing complex contributorship roles, and referencing computable resources."""
    resourceType: Literal_['Citation'] = Field(default=...)
    'A description of identification, location, or contributorship of a publication (article or artifact)'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this citation record when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this summary is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the summary is stored on different servers.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the citation record. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Used for general notes and annotations not coded elsewhere.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the citation record.'
    status: 'code' = Field(default=...)
    'The status of this summary. Enables tracking the life-cycle of the content.'
    author: List_['ContactDetail'] = Field(default_factory=list)
    'Who authored or created the citation record.'
    editor: List_['ContactDetail'] = Field(default_factory=list)
    'Who edited or revised the citation record.'
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the citation record when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the citation record author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    purpose: Optional_['markdown'] = Field(default=None)
    'Explanation of why this citation is needed and why it has been designed as it has.'
    summary: List_['CitationSummary'] = Field(default_factory=list)
    'A human-readable display of key concepts to represent the citation.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    reviewer: List_['ContactDetail'] = Field(default_factory=list)
    'Who reviewed the citation record.'
    endorser: List_['ContactDetail'] = Field(default_factory=list)
    'Who endorsed the citation record.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual that published the citation record.'
    copyright: Optional_['markdown'] = Field(default=None)
    'Use and/or publishing restrictions for the citation record, not for the cited artifact.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this citation record when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate citation record instances.'
    statusDate: List_['CitationStatusDate'] = Field(default_factory=list)
    'The state or status of the citation record paired with an effective date or period for that state.'
    description: Optional_['markdown'] = Field(default=None)
    "A free text natural language description of the citation from a consumer's perspective."
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this citation record is authored for testing purposes (or education/evaluation/marketing) and is not intended to be used for genuine usage.'
    jurisdiction: List_['CodeableConcept'] = Field(default_factory=list)
    'A legal or geographic region in which the citation record is intended to be used.'
    approvalDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was approved by the publisher. Approval happens once when the content is officially approved for usage.'
    currentState: List_['CodeableConcept'] = Field(default_factory=list)
    'The status of the citation record.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    citedArtifact: Optional_['CitationCitedArtifact'] = Field(default=None)
    'The article or artifact being described.'
    copyrightLabel: Optional_['string'] = Field(default=None)
    "A short string (<50 characters), suitable for inclusion in a page footer that identifies the copyright holder, effective period, and optionally whether rights are resctricted. (e.g. 'All rights reserved', 'Some rights reserved')."
    lastReviewDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was last reviewed. Review happens periodically after approval but does not change the original approval date.'
    classification: List_['CitationClassification'] = Field(default_factory=list)
    'The assignment to an organizing scheme.'
    effectivePeriod: Optional_['Period'] = Field(default=None)
    'The period during which the citation record content was or is planned to be in active use.'
    relatedArtifact: List_['RelatedArtifact'] = Field(default_factory=list)
    'Artifact related to the citation record.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    date: Optional_['dateTime'] = Field(default=None)
    'The date (and optionally time) when the citation record was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the citation record changes.'


class ClaimSupportingInfo(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Additional information codes regarding exceptions, special considerations, the condition, situation, prior or concurrent issues."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'System and code pertaining to the specific information regarding special conditions relating to the setting, treatment or patient  for which care is sought.'
    reason: Optional_['CodeableConcept'] = Field(default=None)
    'Provides the reason in the situation where a reason code is required in addition to the content.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify supporting information entries.'
    category: 'CodeableConcept' = Field(default=...)
    'The general class of the information supplied: information; exception; accident, employment; onset, etc.'
    valueBoolean: Optional_['boolean'] = Field(default=None)
    'Additional data or information such as resources, documents, images etc. including references to the data or the actual inclusion of the data.'
    valueString: Optional_['string'] = Field(default=None)
    'Additional data or information such as resources, documents, images etc. including references to the data or the actual inclusion of the data.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'Additional data or information such as resources, documents, images etc. including references to the data or the actual inclusion of the data.'
    valueAttachment: Optional_['Attachment'] = Field(default=None)
    'Additional data or information such as resources, documents, images etc. including references to the data or the actual inclusion of the data.'
    valueReference: Optional_['Reference'] = Field(default=None)
    'Additional data or information such as resources, documents, images etc. including references to the data or the actual inclusion of the data.'
    valueIdentifier: Optional_['Identifier'] = Field(default=None)
    'Additional data or information such as resources, documents, images etc. including references to the data or the actual inclusion of the data.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    timingDate: Optional_['date'] = Field(default=None)
    'The date when or period to which this information refers.'
    timingPeriod: Optional_['Period'] = Field(default=None)
    'The date when or period to which this information refers.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimInsurance(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Financial instruments for reimbursement for the health care products and services specified on the claim."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    focal: 'boolean' = Field(default=...)
    'A flag to indicate that this Coverage is to be used for adjudication of this claim when set to true.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify insurance entries and provide a sequence of coverages to convey coordination of benefit order.'
    coverage: 'Reference' = Field(default=...)
    "Reference to the insurance card level information contained in the Coverage resource. The coverage issuing insurer will use these details to locate the patient's actual coverage within the insurer's information system."
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    identifier: Optional_['Identifier'] = Field(default=None)
    'The business identifier to be used when the claim is sent for adjudication against this insurance policy.'
    preAuthRef: List_['string'] = Field(default_factory=list)
    'Reference numbers previously provided by the insurer to the provider to be quoted on subsequent claims containing services or products related to the prior authorization.'
    claimResponse: Optional_['Reference'] = Field(default=None)
    'The result of the adjudication of the line items for the Coverage specified in this insurance.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    businessArrangement: Optional_['string'] = Field(default=None)
    'A business agreement number established between the provider and the insurer for special business processing purposes.'


class ClaimProcedure(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Procedures performed on the patient relevant to the billing items with the claim."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    udi: List_['Reference'] = Field(default_factory=list)
    'Unique Device Identifiers associated with this line item.'
    type: List_['CodeableConcept'] = Field(default_factory=list)
    'When the condition was observed or the relative ranking.'
    date: Optional_['dateTime'] = Field(default=None)
    'Date and optionally time the procedure was performed.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify procedure entries.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    procedureCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The code or reference to a Procedure resource which identifies the clinical intervention performed.'
    procedureReference: Optional_['Reference'] = Field(default=None)
    'The code or reference to a Procedure resource which identifies the clinical intervention performed.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimDiagnosis(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information about diagnoses relevant to the claim items."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: List_['CodeableConcept'] = Field(default_factory=list)
    'When the condition was observed or the relative ranking.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify diagnosis entries.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    onAdmission: Optional_['CodeableConcept'] = Field(default=None)
    'Indication of whether the diagnosis was present on admission to a facility.'
    diagnosisCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The nature of illness or problem in a coded form or as a reference to an external defined Condition.'
    diagnosisReference: Optional_['Reference'] = Field(default=None)
    'The nature of illness or problem in a coded form or as a reference to an external defined Condition.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimAccident(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Details of an accident which resulted in injuries which required the products and services listed in the claim."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The type or context of the accident event for the purposes of selection of potential insurance coverages and determination of coordination between insurers.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    locationAddress: Optional_['Address'] = Field(default=None)
    'The physical location of the accident event.'
    locationReference: Optional_['Reference'] = Field(default=None)
    'The physical location of the accident event.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    date: 'date' = Field(default=...)
    'Date of an accident event  related to the products and services contained in the claim.'


class ClaimCareTeam(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The members of the team who provided the products and services."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    role: Optional_['CodeableConcept'] = Field(default=None)
    'The lead, assisting or supervising practitioner and their discipline if a multidisciplinary team.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify care team entries.'
    provider: 'Reference' = Field(default=...)
    'Member of the team who provided the product or service.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    specialty: Optional_['CodeableConcept'] = Field(default=None)
    'The specialization of the practitioner or provider which is applicable for this service.'
    responsible: Optional_['boolean'] = Field(default=None)
    'The party who is billing and/or responsible for the claimed products or services.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimRelated(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Other claims which are related to this claim such as prior submissions or claims for related services or for the same event."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    claim: Optional_['Reference'] = Field(default=None)
    'Reference to a related claim.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: Optional_['Identifier'] = Field(default=None)
    'An alternate organizational reference to the case or file to which this particular claim pertains.'
    relationship: Optional_['CodeableConcept'] = Field(default=None)
    'A code to convey how the claims are related.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimEvent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information code for an event with a corresponding date or period."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'A coded event such as when a service is expected or a card printed.'
    whenDateTime: Optional_['dateTime'] = Field(default=None)
    'A date or period in the past or future indicating when the event occurred or is expectd to occur.'
    whenPeriod: Optional_['Period'] = Field(default=None)
    'A date or period in the past or future indicating when the event occurred or is expectd to occur.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimPayee(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The party to be reimbursed for cost of the products and services according to the terms of the policy."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'Type of Party to be reimbursed: subscriber, provider, other.'
    party: Optional_['Reference'] = Field(default=None)
    'Reference to the individual or organization to whom any payment will be made.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimItem(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A claim line. Either a simple  product or service or a 'group' of details which can each be a simple items or groups of sub-details."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    tax: Optional_['Money'] = Field(default=None)
    'The total of taxes applicable for this product or service.'
    net: Optional_['Money'] = Field(default=None)
    'The total amount claimed for the group (if a grouper) or the line item. Net = unit price * quantity * factor.'
    udi: List_['Reference'] = Field(default_factory=list)
    'Unique Device Identifiers associated with this line item.'
    factor: Optional_['decimal'] = Field(default=None)
    'A real number that represents a multiplier used in determining the overall value of services delivered and/or goods received. The concept of a Factor allows for a discount or surcharge multiplier to be applied to a monetary amount.'
    detail: List_['ClaimItemDetail'] = Field(default_factory=list)
    "A claim detail line. Either a simple (a product or service) or a 'group' of sub-details which are simple items."
    revenue: Optional_['CodeableConcept'] = Field(default=None)
    'The type of revenue or cost center providing the product and/or service.'
    request: List_['Reference'] = Field(default_factory=list)
    'Request or Referral for Goods or Service to be rendered.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify item entries.'
    category: Optional_['CodeableConcept'] = Field(default=None)
    'Code to identify the general type of benefits under which products and services are provided.'
    modifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Item typification or modifiers codes to convey additional context for the product or service.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'The number of repetitions of a service or product.'
    bodySite: List_['ClaimItemBodySite'] = Field(default_factory=list)
    'Physical location where the service is performed or applies.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    unitPrice: Optional_['Money'] = Field(default=None)
    'If the item is not a group then this is the fee for the product or service, otherwise this is the total of the fees for the details of the group.'
    encounter: List_['Reference'] = Field(default_factory=list)
    'Healthcare encounters related to this claim.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    programCode: List_['CodeableConcept'] = Field(default_factory=list)
    'Identifies the program under which this may be recovered.'
    servicedDate: Optional_['date'] = Field(default=None)
    'The date or dates when the service or product was supplied, performed or completed.'
    servicedPeriod: Optional_['Period'] = Field(default=None)
    'The date or dates when the service or product was supplied, performed or completed.'
    locationCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Where the product or service was provided.'
    locationAddress: Optional_['Address'] = Field(default=None)
    'Where the product or service was provided.'
    locationReference: Optional_['Reference'] = Field(default=None)
    'Where the product or service was provided.'
    patientPaid: Optional_['Money'] = Field(default=None)
    'The amount paid by the patient, in total at the claim claim level or specifically for the item and detail level, to the provider for goods and services.'
    careTeamSequence: List_['positiveInt'] = Field(default_factory=list)
    'CareTeam members related to this service or product.'
    productOrService: Optional_['CodeableConcept'] = Field(default=None)
    'When the value is a group code then this item collects a set of related item details, otherwise this contains the product, service, drug or other billing code for the item. This element may be the start of a range of .productOrService codes used in conjunction with .productOrServiceEnd or it may be a solo element where .productOrServiceEnd is not used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    diagnosisSequence: List_['positiveInt'] = Field(default_factory=list)
    'Diagnosis applicable for this service or product.'
    procedureSequence: List_['positiveInt'] = Field(default_factory=list)
    'Procedures applicable for this service or product.'
    informationSequence: List_['positiveInt'] = Field(default_factory=list)
    'Exceptions, special conditions and supporting information applicable for this service or product.'
    productOrServiceEnd: Optional_['CodeableConcept'] = Field(default=None)
    'This contains the end of a range of product, service, drug or other billing codes for the item. This element is not used when the .productOrService is a group code. This value may only be present when a .productOfService code has been provided to convey the start of the range. Typically this value may be used only with preauthorizations and not with claims.'


class ClaimItemBodySite(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Physical location where the service is performed or applies."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    site: List_['CodeableReference'] = Field(default=...)
    'Physical service site on the patient (limb, tooth, etc.).'
    subSite: List_['CodeableConcept'] = Field(default_factory=list)
    'A region or surface of the bodySite, e.g. limb region or tooth surface(s).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimItemDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A claim detail line. Either a simple (a product or service) or a 'group' of sub-details which are simple items."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    tax: Optional_['Money'] = Field(default=None)
    'The total of taxes applicable for this product or service.'
    net: Optional_['Money'] = Field(default=None)
    'The total amount claimed for the group (if a grouper) or the line item.detail. Net = unit price * quantity * factor.'
    udi: List_['Reference'] = Field(default_factory=list)
    'Unique Device Identifiers associated with this line item.'
    factor: Optional_['decimal'] = Field(default=None)
    'A real number that represents a multiplier used in determining the overall value of services delivered and/or goods received. The concept of a Factor allows for a discount or surcharge multiplier to be applied to a monetary amount.'
    revenue: Optional_['CodeableConcept'] = Field(default=None)
    'The type of revenue or cost center providing the product and/or service.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify item entries.'
    category: Optional_['CodeableConcept'] = Field(default=None)
    'Code to identify the general type of benefits under which products and services are provided.'
    modifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Item typification or modifiers codes to convey additional context for the product or service.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'The number of repetitions of a service or product.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    unitPrice: Optional_['Money'] = Field(default=None)
    'If the item is not a group then this is the fee for the product or service, otherwise this is the total of the fees for the details of the group.'
    subDetail: List_['ClaimItemDetailSubDetail'] = Field(default_factory=list)
    "A claim detail line. Either a simple (a product or service) or a 'group' of sub-details which are simple items."
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    programCode: List_['CodeableConcept'] = Field(default_factory=list)
    'Identifies the program under which this may be recovered.'
    patientPaid: Optional_['Money'] = Field(default=None)
    'The amount paid by the patient, in total at the claim claim level or specifically for the item and detail level, to the provider for goods and services.'
    productOrService: Optional_['CodeableConcept'] = Field(default=None)
    'When the value is a group code then this item collects a set of related item details, otherwise this contains the product, service, drug or other billing code for the item. This element may be the start of a range of .productOrService codes used in conjunction with .productOrServiceEnd or it may be a solo element where .productOrServiceEnd is not used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    productOrServiceEnd: Optional_['CodeableConcept'] = Field(default=None)
    'This contains the end of a range of product, service, drug or other billing codes for the item. This element is not used when the .productOrService is a group code. This value may only be present when a .productOfService code has been provided to convey the start of the range. Typically this value may be used only with preauthorizations and not with claims.'


class ClaimItemDetailSubDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A claim detail line. Either a simple (a product or service) or a 'group' of sub-details which are simple items."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    tax: Optional_['Money'] = Field(default=None)
    'The total of taxes applicable for this product or service.'
    net: Optional_['Money'] = Field(default=None)
    'The total amount claimed for line item.detail.subDetail. Net = unit price * quantity * factor.'
    udi: List_['Reference'] = Field(default_factory=list)
    'Unique Device Identifiers associated with this line item.'
    factor: Optional_['decimal'] = Field(default=None)
    'A real number that represents a multiplier used in determining the overall value of services delivered and/or goods received. The concept of a Factor allows for a discount or surcharge multiplier to be applied to a monetary amount.'
    revenue: Optional_['CodeableConcept'] = Field(default=None)
    'The type of revenue or cost center providing the product and/or service.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify item entries.'
    category: Optional_['CodeableConcept'] = Field(default=None)
    'Code to identify the general type of benefits under which products and services are provided.'
    modifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Item typification or modifiers codes to convey additional context for the product or service.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'The number of repetitions of a service or product.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    unitPrice: Optional_['Money'] = Field(default=None)
    'If the item is not a group then this is the fee for the product or service, otherwise this is the total of the fees for the details of the group.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    programCode: List_['CodeableConcept'] = Field(default_factory=list)
    'Identifies the program under which this may be recovered.'
    patientPaid: Optional_['Money'] = Field(default=None)
    'The amount paid by the patient, in total at the claim claim level or specifically for the item and detail level, to the provider for goods and services.'
    productOrService: Optional_['CodeableConcept'] = Field(default=None)
    'When the value is a group code then this item collects a set of related item details, otherwise this contains the product, service, drug or other billing code for the item. This element may be the start of a range of .productOrService codes used in conjunction with .productOrServiceEnd or it may be a solo element where .productOrServiceEnd is not used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    productOrServiceEnd: Optional_['CodeableConcept'] = Field(default=None)
    'This contains the end of a range of product, service, drug or other billing codes for the item. This element is not used when the .productOrService is a group code. This value may only be present when a .productOfService code has been provided to convey the start of the range. Typically this value may be used only with preauthorizations and not with claims.'


class Claim(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A provider issued list of professional services and products which have been provided, or are to be provided, to a patient which is sent to an insurer for reimbursement."""
    resourceType: Literal_['Claim'] = Field(default=...)
    'Claim, Pre-determination or Pre-authorization'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    use: 'code' = Field(default=...)
    "A code to indicate whether the nature of the request is: Claim - A request to an Insurer to adjudicate the supplied charges for health care goods and services under the identified policy and to pay the determined Benefit amount, if any; Preauthorization - A request to an Insurer to adjudicate the supplied proposed future charges for health care goods and services under the identified policy and to approve the services and provide the expected benefit amounts and potentially to reserve funds to pay the benefits when Claims for the indicated services are later submitted; or, Pre-determination - A request to an Insurer to adjudicate the supplied 'what if' charges for health care goods and services under the identified policy and report back what the Benefit payable would be had the services actually been provided."
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    type: 'CodeableConcept' = Field(default=...)
    'The category of claim, e.g. oral, pharmacy, vision, institutional, professional.'
    item: List_['ClaimItem'] = Field(default_factory=list)
    "A claim line. Either a simple  product or service or a 'group' of details which can each be a simple items or groups of sub-details."
    payee: Optional_['ClaimPayee'] = Field(default=None)
    'The party to be reimbursed for cost of the products and services according to the terms of the policy.'
    event: List_['ClaimEvent'] = Field(default_factory=list)
    'Information code for an event with a corresponding date or period.'
    total: Optional_['Money'] = Field(default=None)
    'The total value of the all the items in the claim.'
    status: 'code' = Field(default=...)
    'The status of the resource instance.'
    subType: Optional_['CodeableConcept'] = Field(default=None)
    'A finer grained suite of claim type codes which may convey additional information such as Inpatient vs Outpatient and/or a specialty service.'
    patient: 'Reference' = Field(default=...)
    'The party to whom the professional services and/or products have been supplied or are being considered and for whom actual or forecast reimbursement is sought.'
    created: 'dateTime' = Field(default=...)
    'The date this resource was created.'
    enterer: Optional_['Reference'] = Field(default=None)
    'Individual who created the claim, predetermination or preauthorization.'
    insurer: Optional_['Reference'] = Field(default=None)
    'The Insurer who is target of the request.'
    related: List_['ClaimRelated'] = Field(default_factory=list)
    'Other claims which are related to this claim such as prior submissions or claims for related services or for the same event.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    provider: Optional_['Reference'] = Field(default=None)
    'The provider which is responsible for the claim, predetermination or preauthorization.'
    priority: Optional_['CodeableConcept'] = Field(default=None)
    'The provider-required urgency of processing the request. Typical values include: stat, normal, deferred.'
    referral: Optional_['Reference'] = Field(default=None)
    'The referral information received by the claim author, it is not to be used when the author generates a referral for a patient. A copy of that referral may be provided as supporting information. Some insurers require proof of referral to pay for services or to pay specialist rates for services.'
    facility: Optional_['Reference'] = Field(default=None)
    'Facility where the services were provided.'
    careTeam: List_['ClaimCareTeam'] = Field(default_factory=list)
    'The members of the team who provided the products and services.'
    accident: Optional_['ClaimAccident'] = Field(default=None)
    'Details of an accident which resulted in injuries which required the products and services listed in the claim.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: List_['Reference'] = Field(default_factory=list)
    'Healthcare encounters related to this claim.'
    diagnosis: List_['ClaimDiagnosis'] = Field(default_factory=list)
    'Information about diagnoses relevant to the claim items.'
    procedure: List_['ClaimProcedure'] = Field(default_factory=list)
    'Procedures performed on the patient relevant to the billing items with the claim.'
    insurance: List_['ClaimInsurance'] = Field(default_factory=list)
    'Financial instruments for reimbursement for the health care products and services specified on the claim.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A unique identifier assigned to this claim.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    patientPaid: Optional_['Money'] = Field(default=None)
    'The amount paid by the patient, in total at the claim claim level or specifically for the item and detail level, to the provider for goods and services.'
    fundsReserve: Optional_['CodeableConcept'] = Field(default=None)
    'A code to indicate whether and for whom funds are to be reserved for future claims.'
    prescription: Optional_['Reference'] = Field(default=None)
    "Prescription is the document/authorization given to the claim author for them to provide products and services for which consideration (reimbursement) is sought. Could be a RX for medications, an 'order' for oxygen or wheelchair or physiotherapy treatments."
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    billablePeriod: Optional_['Period'] = Field(default=None)
    'The period for which charges are being submitted.'
    supportingInfo: List_['ClaimSupportingInfo'] = Field(default_factory=list)
    'Additional information codes regarding exceptions, special considerations, the condition, situation, prior or concurrent issues.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    originalPrescription: Optional_['Reference'] = Field(default=None)
    'Original prescription which has been superseded by this prescription to support the dispensing of pharmacy services, medications or products.'
    diagnosisRelatedGroup: Optional_['CodeableConcept'] = Field(default=None)
    'A package billing code or bundle code used to group products and services to a particular health condition (such as heart attack) which is based on a predetermined grouping code system.'


class ClaimResponseProcessNote(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A note that describes or explains adjudication results in a human readable form."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The business purpose of the note text.'
    text: 'string' = Field(default=...)
    'The explanation or description associated with the processing.'
    number: Optional_['positiveInt'] = Field(default=None)
    'A number to uniquely identify a note entry.'
    language: Optional_['CodeableConcept'] = Field(default=None)
    'A code to define the language used in the text of the note.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimResponseInsurance(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Financial instruments for reimbursement for the health care products and services specified on the claim."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    focal: 'boolean' = Field(default=...)
    'A flag to indicate that this Coverage is to be used for adjudication of this claim when set to true.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify insurance entries and provide a sequence of coverages to convey coordination of benefit order.'
    coverage: 'Reference' = Field(default=...)
    "Reference to the insurance card level information contained in the Coverage resource. The coverage issuing insurer will use these details to locate the patient's actual coverage within the insurer's information system."
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    claimResponse: Optional_['Reference'] = Field(default=None)
    'The result of the adjudication of the line items for the Coverage specified in this insurance.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    businessArrangement: Optional_['string'] = Field(default=None)
    'A business agreement number established between the provider and the insurer for special business processing purposes.'


class ClaimResponsePayment(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Payment details for the adjudication of the claim."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'Whether this represents partial or complete payment of the benefits payable.'
    amount: 'Money' = Field(default=...)
    'Benefits payable less any payment adjustment.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    adjustment: Optional_['Money'] = Field(default=None)
    "Total amount of all adjustments to this payment included in this transaction which are not related to this claim's adjudication."
    identifier: Optional_['Identifier'] = Field(default=None)
    "Issuer's unique identifier for the payment instrument."
    adjustmentReason: Optional_['CodeableConcept'] = Field(default=None)
    'Reason for the payment adjustment.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    date: Optional_['date'] = Field(default=None)
    'Estimated date the payment will be issued or the actual issue date of payment.'


class ClaimResponseAddItem(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The first-tier service adjudications for payor added product or service lines."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    tax: Optional_['Money'] = Field(default=None)
    'The total of taxes applicable for this product or service.'
    net: Optional_['Money'] = Field(default=None)
    'The total amount claimed for the group (if a grouper) or the addItem. Net = unit price * quantity * factor.'
    factor: Optional_['decimal'] = Field(default=None)
    'A real number that represents a multiplier used in determining the overall value of services delivered and/or goods received. The concept of a Factor allows for a discount or surcharge multiplier to be applied to a monetary amount.'
    detail: List_['ClaimResponseAddItemDetail'] = Field(default_factory=list)
    'The second-tier service adjudications for payor added services.'
    revenue: Optional_['CodeableConcept'] = Field(default=None)
    'The type of revenue or cost center providing the product and/or service.'
    request: List_['Reference'] = Field(default_factory=list)
    'Request or Referral for Goods or Service to be rendered.'
    provider: List_['Reference'] = Field(default_factory=list)
    'The providers who are authorized for the services rendered to the patient.'
    modifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Item typification or modifiers codes to convey additional context for the product or service.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'The number of repetitions of a service or product.'
    bodySite: List_['ClaimResponseAddItemBodySite'] = Field(default_factory=list)
    'Physical location where the service is performed or applies.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    unitPrice: Optional_['Money'] = Field(default=None)
    'If the item is not a group then this is the fee for the product or service, otherwise this is the total of the fees for the details of the group.'
    noteNumber: List_['positiveInt'] = Field(default_factory=list)
    'The numbers associated with notes below which apply to the adjudication of this item.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    programCode: List_['CodeableConcept'] = Field(default_factory=list)
    'Identifies the program under which this may be recovered.'
    servicedDate: Optional_['date'] = Field(default=None)
    'The date or dates when the service or product was supplied, performed or completed.'
    servicedPeriod: Optional_['Period'] = Field(default=None)
    'The date or dates when the service or product was supplied, performed or completed.'
    locationCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Where the product or service was provided.'
    locationAddress: Optional_['Address'] = Field(default=None)
    'Where the product or service was provided.'
    locationReference: Optional_['Reference'] = Field(default=None)
    'Where the product or service was provided.'
    itemSequence: List_['positiveInt'] = Field(default_factory=list)
    'Claim items which this service line is intended to replace.'
    adjudication: List_['ClaimResponseItemAdjudication'] = Field(default_factory=list)
    'The adjudication results.'
    reviewOutcome: Optional_['ClaimResponseItemReviewOutcome'] = Field(default=None)
    'The high-level results of the adjudication if adjudication has been performed.'
    detailSequence: List_['positiveInt'] = Field(default_factory=list)
    'The sequence number of the details within the claim item which this line is intended to replace.'
    productOrService: Optional_['CodeableConcept'] = Field(default=None)
    'When the value is a group code then this item collects a set of related item details, otherwise this contains the product, service, drug or other billing code for the item. This element may be the start of a range of .productOrService codes used in conjunction with .productOrServiceEnd or it may be a solo element where .productOrServiceEnd is not used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    subdetailSequence: List_['positiveInt'] = Field(default_factory=list)
    'The sequence number of the sub-details within the details within the claim item which this line is intended to replace.'
    productOrServiceEnd: Optional_['CodeableConcept'] = Field(default=None)
    'This contains the end of a range of product, service, drug or other billing codes for the item. This element is not used when the .productOrService is a group code. This value may only be present when a .productOfService code has been provided to convey the start of the range. Typically this value may be used only with preauthorizations and not with claims.'


class ClaimResponseAddItemBodySite(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Physical location where the service is performed or applies."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    site: List_['CodeableReference'] = Field(default=...)
    'Physical service site on the patient (limb, tooth, etc.).'
    subSite: List_['CodeableConcept'] = Field(default_factory=list)
    'A region or surface of the bodySite, e.g. limb region or tooth surface(s).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimResponseAddItemDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The second-tier service adjudications for payor added services."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    tax: Optional_['Money'] = Field(default=None)
    'The total of taxes applicable for this product or service.'
    net: Optional_['Money'] = Field(default=None)
    'The total amount claimed for the group (if a grouper) or the addItem.detail. Net = unit price * quantity * factor.'
    factor: Optional_['decimal'] = Field(default=None)
    'A real number that represents a multiplier used in determining the overall value of services delivered and/or goods received. The concept of a Factor allows for a discount or surcharge multiplier to be applied to a monetary amount.'
    revenue: Optional_['CodeableConcept'] = Field(default=None)
    'The type of revenue or cost center providing the product and/or service.'
    modifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Item typification or modifiers codes to convey additional context for the product or service.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'The number of repetitions of a service or product.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    unitPrice: Optional_['Money'] = Field(default=None)
    'If the item is not a group then this is the fee for the product or service, otherwise this is the total of the fees for the details of the group.'
    subDetail: List_['ClaimResponseAddItemDetailSubDetail'] = Field(default_factory=list)
    'The third-tier service adjudications for payor added services.'
    noteNumber: List_['positiveInt'] = Field(default_factory=list)
    'The numbers associated with notes below which apply to the adjudication of this item.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    adjudication: List_['ClaimResponseItemAdjudication'] = Field(default_factory=list)
    'The adjudication results.'
    reviewOutcome: Optional_['ClaimResponseItemReviewOutcome'] = Field(default=None)
    'The high-level results of the adjudication if adjudication has been performed.'
    productOrService: Optional_['CodeableConcept'] = Field(default=None)
    'When the value is a group code then this item collects a set of related item details, otherwise this contains the product, service, drug or other billing code for the item. This element may be the start of a range of .productOrService codes used in conjunction with .productOrServiceEnd or it may be a solo element where .productOrServiceEnd is not used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    productOrServiceEnd: Optional_['CodeableConcept'] = Field(default=None)
    'This contains the end of a range of product, service, drug or other billing codes for the item. This element is not used when the .productOrService is a group code. This value may only be present when a .productOfService code has been provided to convey the start of the range. Typically this value may be used only with preauthorizations and not with claims.'


class ClaimResponseAddItemDetailSubDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The third-tier service adjudications for payor added services."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    tax: Optional_['Money'] = Field(default=None)
    'The total of taxes applicable for this product or service.'
    net: Optional_['Money'] = Field(default=None)
    'The total amount claimed for the addItem.detail.subDetail. Net = unit price * quantity * factor.'
    factor: Optional_['decimal'] = Field(default=None)
    'A real number that represents a multiplier used in determining the overall value of services delivered and/or goods received. The concept of a Factor allows for a discount or surcharge multiplier to be applied to a monetary amount.'
    revenue: Optional_['CodeableConcept'] = Field(default=None)
    'The type of revenue or cost center providing the product and/or service.'
    modifier: List_['CodeableConcept'] = Field(default_factory=list)
    'Item typification or modifiers codes to convey additional context for the product or service.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'The number of repetitions of a service or product.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    unitPrice: Optional_['Money'] = Field(default=None)
    'If the item is not a group then this is the fee for the product or service, otherwise this is the total of the fees for the details of the group.'
    noteNumber: List_['positiveInt'] = Field(default_factory=list)
    'The numbers associated with notes below which apply to the adjudication of this item.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    adjudication: List_['ClaimResponseItemAdjudication'] = Field(default_factory=list)
    'The adjudication results.'
    reviewOutcome: Optional_['ClaimResponseItemReviewOutcome'] = Field(default=None)
    'The high-level results of the adjudication if adjudication has been performed.'
    productOrService: Optional_['CodeableConcept'] = Field(default=None)
    'When the value is a group code then this item collects a set of related item details, otherwise this contains the product, service, drug or other billing code for the item. This element may be the start of a range of .productOrService codes used in conjunction with .productOrServiceEnd or it may be a solo element where .productOrServiceEnd is not used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    productOrServiceEnd: Optional_['CodeableConcept'] = Field(default=None)
    'This contains the end of a range of product, service, drug or other billing codes for the item. This element is not used when the .productOrService is a group code. This value may only be present when a .productOfService code has been provided to convey the start of the range. Typically this value may be used only with preauthorizations and not with claims.'


class ClaimResponseError(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Errors encountered during the processing of the adjudication."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: 'CodeableConcept' = Field(default=...)
    'An error code, from a specified code system, which details why the claim could not be adjudicated.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    expression: List_['string'] = Field(default_factory=list)
    'A [simple subset of FHIRPath](fhirpath.html#simple) limited to element names, repetition indicators and the default child accessor that identifies one of the elements in the resource that caused this issue to be raised.'
    itemSequence: Optional_['positiveInt'] = Field(default=None)
    'The sequence number of the line item submitted which contains the error. This value is omitted when the error occurs outside of the item structure.'
    detailSequence: Optional_['positiveInt'] = Field(default=None)
    'The sequence number of the detail within the line item submitted which contains the error. This value is omitted when the error occurs outside of the item structure.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    subDetailSequence: Optional_['positiveInt'] = Field(default=None)
    'The sequence number of the sub-detail within the detail within the line item submitted which contains the error. This value is omitted when the error occurs outside of the item structure.'


class ClaimResponseTotal(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Categorized monetary totals for the adjudication."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    amount: 'Money' = Field(default=...)
    'Monetary total amount associated with the category.'
    category: 'CodeableConcept' = Field(default=...)
    'A code to indicate the information type of this adjudication record. Information types may include: the value submitted, maximum values or percentages allowed or payable under the plan, amounts that the patient is responsible for in aggregate or pertaining to this item, amounts paid by other coverages, and the benefit payable for this item.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimResponseEvent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information code for an event with a corresponding date or period."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'A coded event such as when a service is expected or a card printed.'
    whenDateTime: Optional_['dateTime'] = Field(default=None)
    'A date or period in the past or future indicating when the event occurred or is expectd to occur.'
    whenPeriod: Optional_['Period'] = Field(default=None)
    'A date or period in the past or future indicating when the event occurred or is expectd to occur.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimResponseItem(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A claim line. Either a simple (a product or service) or a 'group' of details which can also be a simple items or groups of sub-details."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    detail: List_['ClaimResponseItemDetail'] = Field(default_factory=list)
    "A claim detail. Either a simple (a product or service) or a 'group' of sub-details which are simple items."
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    noteNumber: List_['positiveInt'] = Field(default_factory=list)
    'The numbers associated with notes below which apply to the adjudication of this item.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    itemSequence: 'positiveInt' = Field(default=...)
    'A number to uniquely reference the claim item entries.'
    adjudication: List_['ClaimResponseItemAdjudication'] = Field(default_factory=list)
    'If this item is a group then the values here are a summary of the adjudication of the detail items. If this item is a simple product or service then this is the result of the adjudication of this item.'
    reviewOutcome: Optional_['ClaimResponseItemReviewOutcome'] = Field(default=None)
    'The high-level results of the adjudication if adjudication has been performed.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimResponseItemReviewOutcome(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The high-level results of the adjudication if adjudication has been performed."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    reason: List_['CodeableConcept'] = Field(default_factory=list)
    'The reasons for the result of the claim, predetermination, or preauthorization adjudication.'
    decision: Optional_['CodeableConcept'] = Field(default=None)
    'The result of the claim, predetermination, or preauthorization adjudication.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    preAuthRef: Optional_['string'] = Field(default=None)
    'Reference from the Insurer which is used in later communications which refers to this adjudication.'
    preAuthPeriod: Optional_['Period'] = Field(default=None)
    'The time frame during which this authorization is effective.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimResponseItemAdjudication(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """If this item is a group then the values here are a summary of the adjudication of the detail items. If this item is a simple product or service then this is the result of the adjudication of this item."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    reason: Optional_['CodeableConcept'] = Field(default=None)
    'A code supporting the understanding of the adjudication result and explaining variance from expected amount.'
    amount: Optional_['Money'] = Field(default=None)
    'Monetary amount associated with the category.'
    category: 'CodeableConcept' = Field(default=...)
    'A code to indicate the information type of this adjudication record. Information types may include the value submitted, maximum values or percentages allowed or payable under the plan, amounts that: the patient is responsible for in aggregate or pertaining to this item; amounts paid by other coverages; and, the benefit payable for this item.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'A non-monetary value associated with the category. Mutually exclusive to the amount element above.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimResponseItemDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A claim detail. Either a simple (a product or service) or a 'group' of sub-details which are simple items."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    subDetail: List_['ClaimResponseItemDetailSubDetail'] = Field(default_factory=list)
    'A sub-detail adjudication of a simple product or service.'
    noteNumber: List_['positiveInt'] = Field(default_factory=list)
    'The numbers associated with notes below which apply to the adjudication of this item.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    adjudication: List_['ClaimResponseItemAdjudication'] = Field(default_factory=list)
    'The adjudication results.'
    reviewOutcome: Optional_['ClaimResponseItemReviewOutcome'] = Field(default=None)
    'The high-level results of the adjudication if adjudication has been performed.'
    detailSequence: 'positiveInt' = Field(default=...)
    'A number to uniquely reference the claim detail entry.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClaimResponseItemDetailSubDetail(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A sub-detail adjudication of a simple product or service."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    noteNumber: List_['positiveInt'] = Field(default_factory=list)
    'The numbers associated with notes below which apply to the adjudication of this item.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    adjudication: List_['ClaimResponseItemAdjudication'] = Field(default_factory=list)
    'The adjudication results.'
    reviewOutcome: Optional_['ClaimResponseItemReviewOutcome'] = Field(default=None)
    'The high-level results of the adjudication if adjudication has been performed.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    subDetailSequence: 'positiveInt' = Field(default=...)
    'A number to uniquely reference the claim sub-detail entry.'


class ClaimResponse(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """This resource provides the adjudication details from the processing of a Claim resource."""
    resourceType: Literal_['ClaimResponse'] = Field(default=...)
    'Response to a claim predetermination or preauthorization'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    use: 'code' = Field(default=...)
    "A code to indicate whether the nature of the request is: Claim - A request to an Insurer to adjudicate the supplied charges for health care goods and services under the identified policy and to pay the determined Benefit amount, if any; Preauthorization - A request to an Insurer to adjudicate the supplied proposed future charges for health care goods and services under the identified policy and to approve the services and provide the expected benefit amounts and potentially to reserve funds to pay the benefits when Claims for the indicated services are later submitted; or, Pre-determination - A request to an Insurer to adjudicate the supplied 'what if' charges for health care goods and services under the identified policy and report back what the Benefit payable would be had the services actually been provided."
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    type: 'CodeableConcept' = Field(default=...)
    'A finer grained suite of claim type codes which may convey additional information such as Inpatient vs Outpatient and/or a specialty service.'
    item: List_['ClaimResponseItem'] = Field(default_factory=list)
    "A claim line. Either a simple (a product or service) or a 'group' of details which can also be a simple items or groups of sub-details."
    form: Optional_['Attachment'] = Field(default=None)
    'The actual form, by reference or inclusion, for printing the content or an EOB.'
    event: List_['ClaimResponseEvent'] = Field(default_factory=list)
    'Information code for an event with a corresponding date or period.'
    total: List_['ClaimResponseTotal'] = Field(default_factory=list)
    'Categorized monetary totals for the adjudication.'
    error: List_['ClaimResponseError'] = Field(default_factory=list)
    'Errors encountered during the processing of the adjudication.'
    status: 'code' = Field(default=...)
    'The status of the resource instance.'
    subType: Optional_['CodeableConcept'] = Field(default=None)
    'A finer grained suite of claim type codes which may convey additional information such as Inpatient vs Outpatient and/or a specialty service.'
    patient: 'Reference' = Field(default=...)
    'The party to whom the professional services and/or products have been supplied or are being considered and for whom actual for facast reimbursement is sought.'
    created: 'dateTime' = Field(default=...)
    'The date this resource was created.'
    insurer: Optional_['Reference'] = Field(default=None)
    'The party responsible for authorization, adjudication and reimbursement.'
    request: Optional_['Reference'] = Field(default=None)
    'Original request resource reference.'
    outcome: 'code' = Field(default=...)
    'The outcome of the claim, predetermination, or preauthorization processing.'
    addItem: List_['ClaimResponseAddItem'] = Field(default_factory=list)
    'The first-tier service adjudications for payor added product or service lines.'
    payment: Optional_['ClaimResponsePayment'] = Field(default=None)
    'Payment details for the adjudication of the claim.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    decision: Optional_['CodeableConcept'] = Field(default=None)
    'The result of the claim, predetermination, or preauthorization adjudication.'
    formCode: Optional_['CodeableConcept'] = Field(default=None)
    'A code for the form to be used for printing the content.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    requestor: Optional_['Reference'] = Field(default=None)
    'The provider which is responsible for the claim, predetermination or preauthorization.'
    payeeType: Optional_['CodeableConcept'] = Field(default=None)
    'Type of Party to be reimbursed: subscriber, provider, other.'
    encounter: List_['Reference'] = Field(default_factory=list)
    'Healthcare encounters related to this claim.'
    insurance: List_['ClaimResponseInsurance'] = Field(default_factory=list)
    'Financial instruments for reimbursement for the health care products and services specified on the claim.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A unique identifier assigned to this claim response.'
    preAuthRef: Optional_['string'] = Field(default=None)
    'Reference from the Insurer which is used in later communications which refers to this adjudication.'
    traceNumber: List_['Identifier'] = Field(default_factory=list)
    'Trace number for tracking purposes. May be defined at the jurisdiction level or between trading partners.'
    disposition: Optional_['string'] = Field(default=None)
    'A human readable description of the status of the adjudication.'
    processNote: List_['ClaimResponseProcessNote'] = Field(default_factory=list)
    'A note that describes or explains adjudication results in a human readable form.'
    adjudication: List_['ClaimResponseItemAdjudication'] = Field(default_factory=list)
    'The adjudication results which are presented at the header level rather than at the line-item or add-item levels.'
    fundsReserve: Optional_['CodeableConcept'] = Field(default=None)
    'A code, used only on a response to a preauthorization, to indicate whether the benefits payable have been reserved and for whom.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    preAuthPeriod: Optional_['Period'] = Field(default=None)
    'The time frame during which this authorization is effective.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    communicationRequest: List_['Reference'] = Field(default_factory=list)
    'Request for additional supporting or authorizing information.'
    diagnosisRelatedGroup: Optional_['CodeableConcept'] = Field(default=None)
    'A package billing code or bundle code used to group products and services to a particular health condition (such as heart attack) which is based on a predetermined grouping code system.'


class ClinicalImpressionFinding(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specific findings or diagnoses that were considered likely or relevant to ongoing treatment."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    item: Optional_['CodeableReference'] = Field(default=None)
    'Specific text, code or reference for finding or diagnosis, which may include ruled-out or resolved conditions.'
    basis: Optional_['string'] = Field(default=None)
    'Which investigations support finding or diagnosis.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClinicalImpression(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A record of a clinical assessment performed to determine what problem(s) may affect the patient and before planning the treatments or management strategies that are best to manage a patient's condition. Assessments are often 1:1 with a clinical consultation / encounter,  but this varies greatly depending on the clinical workflow. This resource is called "ClinicalImpression" rather than "ClinicalAssessment" to avoid confusion with the recording of assessment tools such as Apgar score."""
    resourceType: Literal_['ClinicalImpression'] = Field(default=...)
    'A clinical assessment performed when planning treatments and management strategies for a patient'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    date: Optional_['dateTime'] = Field(default=None)
    'Indicates when the documentation of the assessment was complete.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Commentary about the impression, typically recorded after the impression itself was made, though supplemental notes by the original author could also appear.'
    status: 'code' = Field(default=...)
    'Identifies the workflow status of the assessment.'
    subject: 'Reference' = Field(default=...)
    'The patient or group of individuals assessed as part of this record.'
    problem: List_['Reference'] = Field(default_factory=list)
    'A list of the relevant problems/conditions for a patient.'
    summary: Optional_['string'] = Field(default=None)
    'A text summary of the investigations and the diagnosis.'
    finding: List_['ClinicalImpressionFinding'] = Field(default_factory=list)
    'Specific findings or diagnoses that were considered likely or relevant to ongoing treatment.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    previous: Optional_['Reference'] = Field(default=None)
    "A reference to the last assessment that was conducted on this patient. Assessments are often/usually ongoing in nature; a care provider (practitioner or team) will make new assessments on an ongoing basis as new data arises or the patient's conditions changes."
    protocol: List_['uri'] = Field(default_factory=list)
    'Reference to a specific published clinical protocol that was followed during this assessment, and/or that provides evidence in support of the diagnosis.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: Optional_['Reference'] = Field(default=None)
    'The Encounter during which this ClinicalImpression was created or to which the creation of this record is tightly associated.'
    performer: Optional_['Reference'] = Field(default=None)
    'The clinician performing the assessment.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Business identifiers assigned to this clinical impression by the performer or other systems which remain constant as the resource is updated and propagates from server to server.'
    description: Optional_['string'] = Field(default=None)
    'A summary of the context and/or cause of the assessment - why / where it was performed, and what patient events/status prompted it.'
    statusReason: Optional_['CodeableConcept'] = Field(default=None)
    'Captures the reason for the current state of the ClinicalImpression.'
    effectiveDateTime: Optional_['dateTime'] = Field(default=None)
    'The point in time or period over which the subject was assessed.'
    effectivePeriod: Optional_['Period'] = Field(default=None)
    'The point in time or period over which the subject was assessed.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    changePattern: Optional_['CodeableConcept'] = Field(default=None)
    "Change in the status/pattern of a subject's condition since previously assessed, such as worsening, improving, or no change.  It is a subjective assessment of the direction of the change."
    supportingInfo: List_['Reference'] = Field(default_factory=list)
    'Information supporting the clinical impression, which can contain investigation results.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    prognosisReference: List_['Reference'] = Field(default_factory=list)
    'RiskAssessment expressing likely outcome.'
    prognosisCodeableConcept: List_['CodeableConcept'] = Field(default_factory=list)
    'Estimate of likely outcome.'


class ClinicalUseDefinitionUndesirableEffect(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Describe the possible undesirable effects (negative outcomes) from the use of the medicinal product as treatment."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    classification: Optional_['CodeableConcept'] = Field(default=None)
    'High level classification of the effect.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    frequencyOfOccurrence: Optional_['CodeableConcept'] = Field(default=None)
    'How often the effect is seen.'
    symptomConditionEffect: Optional_['CodeableReference'] = Field(default=None)
    'The situation in which the undesirable effect may manifest.'


class ClinicalUseDefinitionContraindication(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specifics for when this is a contraindication."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    indication: List_['Reference'] = Field(default_factory=list)
    'The indication which this is a contraidication for.'
    comorbidity: List_['CodeableReference'] = Field(default_factory=list)
    'A comorbidity (concurrent condition) or coinfection.'
    otherTherapy: List_['ClinicalUseDefinitionContraindicationOtherTherapy'] = Field(default_factory=list)
    'Information about the use of the medicinal product in relation to other therapies described as part of the contraindication.'
    diseaseStatus: Optional_['CodeableReference'] = Field(default=None)
    'The status of the disease or symptom for the contraindication, for example "chronic" or "metastatic".'
    applicability: Optional_['Expression'] = Field(default=None)
    'An expression that returns true or false, indicating whether the indication is applicable or not, after having applied its other elements.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    diseaseSymptomProcedure: Optional_['CodeableReference'] = Field(default=None)
    'The situation that is being documented as contraindicating against this item.'


class ClinicalUseDefinitionContraindicationOtherTherapy(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information about the use of the medicinal product in relation to other therapies described as part of the contraindication."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    treatment: 'CodeableReference' = Field(default=...)
    'Reference to a specific medication (active substance, medicinal product or class of products, biological, food etc.) as part of an indication or contraindication.'
    relationshipType: 'CodeableConcept' = Field(default=...)
    'The type of relationship between the medicinal product indication or contraindication and another therapy.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClinicalUseDefinitionInteraction(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specifics for when this is an interaction."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The type of the interaction e.g. drug-drug interaction, drug-food interaction, drug-lab test interaction.'
    effect: Optional_['CodeableReference'] = Field(default=None)
    'The effect of the interaction, for example "reduced gastric absorption of primary medication".'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    incidence: Optional_['CodeableConcept'] = Field(default=None)
    'The incidence of the interaction, e.g. theoretical, observed.'
    management: List_['CodeableConcept'] = Field(default_factory=list)
    'Actions for managing the interaction.'
    interactant: List_['ClinicalUseDefinitionInteractionInteractant'] = Field(default_factory=list)
    'The specific medication, product, food, substance etc. or laboratory test that interacts.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClinicalUseDefinitionInteractionInteractant(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The specific medication, product, food, substance etc. or laboratory test that interacts."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    itemReference: Optional_['Reference'] = Field(default=None)
    'The specific medication, product, food, substance etc. or laboratory test that interacts.'
    itemCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The specific medication, product, food, substance etc. or laboratory test that interacts.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClinicalUseDefinitionIndication(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Specifics for when this is an indication."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    comorbidity: List_['CodeableReference'] = Field(default_factory=list)
    'A comorbidity (concurrent condition) or coinfection as part of the indication.'
    durationRange: Optional_['Range'] = Field(default=None)
    'Timing or duration information, that may be associated with use with the indicated condition e.g. Adult patients suffering from myocardial infarction (from a few days until less than 35 days), ischaemic stroke (from 7 days until less than 6 months).'
    durationString: Optional_['string'] = Field(default=None)
    'Timing or duration information, that may be associated with use with the indicated condition e.g. Adult patients suffering from myocardial infarction (from a few days until less than 35 days), ischaemic stroke (from 7 days until less than 6 months).'
    otherTherapy: List_['ClinicalUseDefinitionContraindicationOtherTherapy'] = Field(default_factory=list)
    'Information about the use of the medicinal product in relation to other therapies described as part of the indication.'
    diseaseStatus: Optional_['CodeableReference'] = Field(default=None)
    'The status of the disease or symptom for the indication, for example "chronic" or "metastatic".'
    applicability: Optional_['Expression'] = Field(default=None)
    'An expression that returns true or false, indicating whether the indication is applicable or not, after having applied its other elements.'
    intendedEffect: Optional_['CodeableReference'] = Field(default=None)
    'The intended effect, aim or strategy to be achieved.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    undesirableEffect: List_['Reference'] = Field(default_factory=list)
    'An unwanted side effect or negative outcome that may happen if you use the drug (or other subject of this resource) for this indication.'
    diseaseSymptomProcedure: Optional_['CodeableReference'] = Field(default=None)
    'The situation that is being documented as an indicaton for this item.'


class ClinicalUseDefinitionWarning(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A critical piece of information about environmental, health or physical risks or hazards that serve as caution to the user. For example 'Do not operate heavy machinery', 'May cause drowsiness', or 'Get medical advice/attention if you feel unwell'."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'A coded or unformatted textual definition of this warning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    description: Optional_['markdown'] = Field(default=None)
    'A textual definition of this warning, with formatting.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ClinicalUseDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A single issue - either an indication, contraindication, interaction or an undesirable effect for a medicinal product, medication, device or procedure."""
    resourceType: Literal_['ClinicalUseDefinition'] = Field(default=...)
    'A single issue - either an indication, contraindication, interaction or an undesirable effect for a medicinal product, medication, device or procedure'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    type: 'code' = Field(default=...)
    'indication | contraindication | interaction | undesirable-effect | warning.'
    status: Optional_['CodeableConcept'] = Field(default=None)
    'Whether this is a current issue or one that has been retired etc.'
    subject: List_['Reference'] = Field(default_factory=list)
    'The medication, product, substance, device, procedure etc. for which this is an indication.'
    library: List_['canonical'] = Field(default_factory=list)
    'Logic used by the clinical use definition.'
    warning: Optional_['ClinicalUseDefinitionWarning'] = Field(default=None)
    "A critical piece of information about environmental, health or physical risks or hazards that serve as caution to the user. For example 'Do not operate heavy machinery', 'May cause drowsiness', or 'Get medical advice/attention if you feel unwell'."
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'A categorisation of the issue, primarily for dividing warnings into subject heading areas such as "Pregnancy and Lactation", "Overdose", "Effects on Ability to Drive and Use Machines".'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Business identifier for this issue.'
    indication: Optional_['ClinicalUseDefinitionIndication'] = Field(default=None)
    'Specifics for when this is an indication.'
    population: List_['Reference'] = Field(default_factory=list)
    'The population group to which this applies.'
    interaction: Optional_['ClinicalUseDefinitionInteraction'] = Field(default=None)
    'Specifics for when this is an interaction.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    contraindication: Optional_['ClinicalUseDefinitionContraindication'] = Field(default=None)
    'Specifics for when this is a contraindication.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    undesirableEffect: Optional_['ClinicalUseDefinitionUndesirableEffect'] = Field(default=None)
    'Describe the possible undesirable effects (negative outcomes) from the use of the medicinal product as treatment.'


class CodeSystemProperty(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A property defines an additional slot through which additional information can be provided about a concept."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'The type of the property value. Properties of type "code" contain a code defined by the code system (e.g. a reference to another defined concept).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    description: Optional_['string'] = Field(default=None)
    'A description of the property- why it is defined, and how its value might be used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    uri: Optional_['uri'] = Field(default=None)
    'Reference to the formal meaning of the property. One possible source of meaning is the [Concept Properties](codesystem-concept-properties.html) code system.'
    code: 'code' = Field(default=...)
    'A code that is used to identify the property. The code is used internally (in CodeSystem.concept.property.code) and also externally, such as in property filters.'


class CodeSystemConcept(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Concepts that are in the code system. The concept definitions are inherently hierarchical, but the definitions must be consulted to determine what the meanings of the hierarchical relationships are."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    display: Optional_['string'] = Field(default=None)
    'A human readable string that is the recommended default way to present this concept to a user.'
    concept: List_['CodeSystemConcept'] = Field(default_factory=list)
    'Defines children of a concept to produce a hierarchy of concepts. The nature of the relationships is variable (is-a/contains/categorizes) - see hierarchyMeaning.'
    property: List_['CodeSystemConceptProperty'] = Field(default_factory=list)
    'A property value for this concept.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    definition: Optional_['string'] = Field(default=None)
    'The formal definition of the concept. The code system resource does not make formal definitions required, because of the prevalence of legacy systems. However, they are highly recommended, as without them there is no formal meaning associated with the concept.'
    designation: List_['CodeSystemConceptDesignation'] = Field(default_factory=list)
    'Additional representations for the concept - other languages, aliases, specialized purposes, used for particular purposes, etc.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'code' = Field(default=...)
    'A code - a text symbol - that uniquely identifies the concept within the code system.'


class CodeSystemConceptDesignation(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Additional representations for the concept - other languages, aliases, specialized purposes, used for particular purposes, etc."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    use: Optional_['Coding'] = Field(default=None)
    'A code that details how this designation would be used.'
    value: 'string' = Field(default=...)
    'The text value for this designation.'
    language: Optional_['code'] = Field(default=None)
    'The language this designation is defined for.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    additionalUse: List_['Coding'] = Field(default_factory=list)
    'Additional codes that detail how this designation would be used, if there is more than one use.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CodeSystemConceptProperty(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A property value for this concept."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    valueCode: Optional_['code'] = Field(default=None)
    'The value of this property.'
    valueCoding: Optional_['Coding'] = Field(default=None)
    'The value of this property.'
    valueString: Optional_['string'] = Field(default=None)
    'The value of this property.'
    valueInteger: Optional_['integer'] = Field(default=None)
    'The value of this property.'
    valueBoolean: Optional_['boolean'] = Field(default=None)
    'The value of this property.'
    valueDateTime: Optional_['dateTime'] = Field(default=None)
    'The value of this property.'
    valueDecimal: Optional_['decimal'] = Field(default=None)
    'The value of this property.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'code' = Field(default=...)
    'A code that is a reference to CodeSystem.property.code.'


class CodeSystemFilter(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A filter that can be used in a value set compose statement when selecting concepts using a filter."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    value: 'string' = Field(default=...)
    'A description of what the value for the filter should be.'
    operator: List_['code'] = Field(default=...)
    'A list of operators that can be used with the filter.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    description: Optional_['string'] = Field(default=None)
    'A description of how or why the filter is used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'code' = Field(default=...)
    'The code that identifies this filter when it is used as a filter in [ValueSet](valueset.html#).compose.include.filter.'


class CodeSystem(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The CodeSystem resource is used to declare the existence of and describe a code system or code system supplement and its key properties, and optionally define a part or all of its content."""
    resourceType: Literal_['CodeSystem'] = Field(default=...)
    'Declares the existence of and describes a code system or code system supplement'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this code system when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this code system is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the code system is stored on different servers. This is used in [Coding](datatypes.html#Coding).system.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the code system. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the code system.'
    topic: List_['CodeableConcept'] = Field(default_factory=list)
    'Descriptions related to the content of the CodeSystem. Topics provide a high-level categorization as well as keywords for the CodeSystem that can be useful for filtering and searching.'
    count: Optional_['unsignedInt'] = Field(default=None)
    'The total number of concepts defined by the code system. Where the code system has a compositional grammar, the basis of this count is defined by the system steward.'
    status: 'code' = Field(default=...)
    'The status of this code system. Enables tracking the life-cycle of the content.'
    author: List_['ContactDetail'] = Field(default_factory=list)
    'An individiual or organization primarily involved in the creation and maintenance of the CodeSystem.'
    editor: List_['ContactDetail'] = Field(default_factory=list)
    'An individual or organization primarily responsible for internal coherence of the CodeSystem.'
    filter: List_['CodeSystemFilter'] = Field(default_factory=list)
    'A filter that can be used in a value set compose statement when selecting concepts using a filter.'
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the code system when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the code system author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence. This is used in [Coding](datatypes.html#Coding).version.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    purpose: Optional_['markdown'] = Field(default=None)
    'Explanation of why this code system is needed and why it has been designed as it has.'
    content: 'code' = Field(default=...)
    'The extent of the content of the code system (the concepts and codes it defines) are represented in this resource instance.'
    concept: List_['CodeSystemConcept'] = Field(default_factory=list)
    'Concepts that are in the code system. The concept definitions are inherently hierarchical, but the definitions must be consulted to determine what the meanings of the hierarchical relationships are.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    reviewer: List_['ContactDetail'] = Field(default_factory=list)
    'An individual or organization asserted by the publisher to be primarily responsible for review of some aspect of the CodeSystem.'
    endorser: List_['ContactDetail'] = Field(default_factory=list)
    'An individual or organization asserted by the publisher to be responsible for officially endorsing the CodeSystem for use in some setting.'
    valueSet: Optional_['canonical'] = Field(default=None)
    'Canonical reference to the value set that contains all codes in the code system independent of code status.'
    property: List_['CodeSystemProperty'] = Field(default_factory=list)
    'A property defines an additional slot through which additional information can be provided about a concept.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual responsible for the release and ongoing maintenance of the code system.'
    copyright: Optional_['markdown'] = Field(default=None)
    'A copyright statement relating to the code system and/or its contents. Copyright statements are generally legal restrictions on the use and publishing of the code system.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this code system when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate code system instances.'
    description: Optional_['markdown'] = Field(default=None)
    "A free text natural language description of the code system from a consumer's perspective."
    supplements: Optional_['canonical'] = Field(default=None)
    'The canonical URL of the code system that this code system supplement is adding designations and properties to.'
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this code system is authored for testing purposes (or education/evaluation/marketing) and is not intended to be used for genuine usage.'
    jurisdiction: List_['CodeableConcept'] = Field(default_factory=list)
    'A legal or geographic region in which the code system is intended to be used.'
    approvalDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was approved by the publisher. Approval happens once when the content is officially approved for usage.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    caseSensitive: Optional_['boolean'] = Field(default=None)
    'If code comparison is case sensitive when codes within this system are compared to each other.'
    compositional: Optional_['boolean'] = Field(default=None)
    'The code system defines a compositional (post-coordination) grammar.'
    versionNeeded: Optional_['boolean'] = Field(default=None)
    'This flag is used to signify that the code system does not commit to concept permanence across versions. If true, a version must be specified when referencing this code system.'
    copyrightLabel: Optional_['string'] = Field(default=None)
    "A short string (<50 characters), suitable for inclusion in a page footer that identifies the copyright holder, effective period, and optionally whether rights are resctricted. (e.g. 'All rights reserved', 'Some rights reserved')."
    lastReviewDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was last reviewed. Review happens periodically after approval but does not change the original approval date.'
    effectivePeriod: Optional_['Period'] = Field(default=None)
    'The period during which the CodeSystem content was or is planned to be in active use.'
    relatedArtifact: List_['RelatedArtifact'] = Field(default_factory=list)
    'Related artifacts such as additional documentation, justification, dependencies, bibliographic references, and predecessor and successor artifacts.'
    hierarchyMeaning: Optional_['code'] = Field(default=None)
    'The meaning of the hierarchy of concepts as represented in this resource.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which CodeSystem is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which CodeSystem is more current.'
    date: Optional_['dateTime'] = Field(default=None)
    'The date  (and optionally time) when the code system was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the code system changes.'


class CommunicationPayload(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Text, attachment(s), or resource(s) that was communicated to the recipient."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    contentAttachment: Optional_['Attachment'] = Field(default=None)
    'A communicated content (or for multi-part communications, one portion of the communication).'
    contentReference: Optional_['Reference'] = Field(default=None)
    'A communicated content (or for multi-part communications, one portion of the communication).'
    contentCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'A communicated content (or for multi-part communications, one portion of the communication).'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Communication(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A clinical or business level record of information being transmitted or shared; e.g. an alert that was sent to a responsible provider, a public health agency communication to a provider/reporter in response to a case report for a reportable condition."""
    resourceType: Literal_['Communication'] = Field(default=...)
    'A clinical or business level record of information being transmitted or shared'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    sent: Optional_['dateTime'] = Field(default=None)
    'The time when this communication was sent.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Additional notes or commentary about the communication by the sender, receiver or other interested parties.'
    topic: Optional_['CodeableConcept'] = Field(default=None)
    'Description of the purpose/content, similar to a subject line in an email.'
    about: List_['Reference'] = Field(default_factory=list)
    'Other resources that pertain to this communication and to which this communication should be associated.'
    partOf: List_['Reference'] = Field(default_factory=list)
    'A larger event (e.g. Communication, Procedure) of which this particular communication is a component or step.'
    status: 'code' = Field(default=...)
    'The status of the transmission.'
    medium: List_['CodeableConcept'] = Field(default_factory=list)
    'A channel that was used for this communication (e.g. email, fax).'
    sender: Optional_['Reference'] = Field(default=None)
    'The entity (e.g. person, organization, clinical information system, or device) which is the source of the communication.'
    reason: List_['CodeableReference'] = Field(default_factory=list)
    'The reason or justification for the communication.'
    basedOn: List_['Reference'] = Field(default_factory=list)
    'An order, proposal or plan fulfilled in whole or in part by this Communication.'
    subject: Optional_['Reference'] = Field(default=None)
    'The patient or group that was the focus of this communication.'
    payload: List_['CommunicationPayload'] = Field(default_factory=list)
    'Text, attachment(s), or resource(s) that was communicated to the recipient.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'The type of message conveyed such as alert, notification, reminder, instruction, etc.'
    priority: Optional_['code'] = Field(default=None)
    'Characterizes how quickly the planned or in progress communication must be addressed. Includes concepts such as stat, urgent, routine.'
    received: Optional_['dateTime'] = Field(default=None)
    'The time when this communication arrived at the destination.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: Optional_['Reference'] = Field(default=None)
    'The Encounter during which this Communication was created or to which the creation of this record is tightly associated.'
    recipient: List_['Reference'] = Field(default_factory=list)
    'The entity (e.g. person, organization, clinical information system, care team or device) which is the target of the communication.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Business identifiers assigned to this communication by the performer or other systems which remain constant as the resource is updated and propagates from server to server.'
    inResponseTo: List_['Reference'] = Field(default_factory=list)
    'Prior communication that this communication is in response to.'
    statusReason: Optional_['CodeableConcept'] = Field(default=None)
    'Captures the reason for the current state of the Communication.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    instantiatesUri: List_['uri'] = Field(default_factory=list)
    'The URL pointing to an externally maintained protocol, guideline, orderset or other definition that is adhered to in whole or in part by this Communication.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    instantiatesCanonical: List_['canonical'] = Field(default_factory=list)
    'The URL pointing to a FHIR-defined protocol, guideline, orderset or other definition that is adhered to in whole or in part by this Communication.'


class CommunicationRequestPayload(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Text, attachment(s), or resource(s) to be communicated to the recipient."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    contentAttachment: Optional_['Attachment'] = Field(default=None)
    'The communicated content (or for multi-part communications, one portion of the communication).'
    contentReference: Optional_['Reference'] = Field(default=None)
    'The communicated content (or for multi-part communications, one portion of the communication).'
    contentCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The communicated content (or for multi-part communications, one portion of the communication).'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CommunicationRequest(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A request to convey information; e.g. the CDS system proposes that an alert be sent to a responsible provider, the CDS system proposes that the public health agency be notified about a reportable condition."""
    resourceType: Literal_['CommunicationRequest'] = Field(default=...)
    'A request for information to be sent to a receiver'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Comments made about the request by the requester, sender, recipient, subject or other participants.'
    about: List_['Reference'] = Field(default_factory=list)
    'Other resources that pertain to this communication request and to which this communication request should be associated.'
    status: 'code' = Field(default=...)
    'The status of the proposal or order.'
    intent: 'code' = Field(default=...)
    'Indicates the level of authority/intentionality associated with the CommunicationRequest and where the request fits into the workflow chain.'
    medium: List_['CodeableConcept'] = Field(default_factory=list)
    'A channel that was used for this communication (e.g. email, fax).'
    reason: List_['CodeableReference'] = Field(default_factory=list)
    'Describes why the request is being made in coded or textual form.'
    basedOn: List_['Reference'] = Field(default_factory=list)
    'A plan or proposal that is fulfilled in whole or in part by this request.'
    subject: Optional_['Reference'] = Field(default=None)
    'The patient or group that is the focus of this communication request.'
    payload: List_['CommunicationRequestPayload'] = Field(default_factory=list)
    'Text, attachment(s), or resource(s) to be communicated to the recipient.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    replaces: List_['Reference'] = Field(default_factory=list)
    'Completed or terminated request(s) whose function is taken by this new request.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'The type of message to be sent such as alert, notification, reminder, instruction, etc.'
    priority: Optional_['code'] = Field(default=None)
    'Characterizes how quickly the proposed act must be initiated. Includes concepts such as stat, urgent, routine.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: Optional_['Reference'] = Field(default=None)
    'The Encounter during which this CommunicationRequest was created or to which the creation of this record is tightly associated.'
    requester: Optional_['Reference'] = Field(default=None)
    'The device, individual, or organization who asks for the information to be shared.'
    recipient: List_['Reference'] = Field(default_factory=list)
    'The entity (e.g. person, organization, clinical information system, device, group, or care team) which is the intended target of the communication.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Business identifiers assigned to this communication request by the performer or other systems which remain constant as the resource is updated and propagates from server to server.'
    authoredOn: Optional_['dateTime'] = Field(default=None)
    'For draft requests, indicates the date of initial creation.  For requests with other statuses, indicates the date of activation.'
    statusReason: Optional_['CodeableConcept'] = Field(default=None)
    'Captures the reason for the current state of the CommunicationRequest.'
    doNotPerform: Optional_['boolean'] = Field(default=None)
    'If true indicates that the CommunicationRequest is asking for the specified action to *not* occur.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    occurrenceDateTime: Optional_['dateTime'] = Field(default=None)
    'The time when this communication is to occur.'
    occurrencePeriod: Optional_['Period'] = Field(default=None)
    'The time when this communication is to occur.'
    groupIdentifier: Optional_['Identifier'] = Field(default=None)
    'A shared identifier common to multiple independent Request instances that were activated/authorized more or less simultaneously by a single author.  The presence of the same identifier on each request ties those requests together and may have business ramifications in terms of reporting of results, billing, etc.  E.g. a requisition number shared by a set of lab tests ordered together, or a prescription number shared by all meds ordered at one time.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    informationProvider: List_['Reference'] = Field(default_factory=list)
    'The entity (e.g. person, organization, clinical information system, or device) which is to be the source of the communication.'


class CompartmentDefinitionResource(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information about how a resource is related to the compartment."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    param: List_['string'] = Field(default_factory=list)
    'The name of a search parameter that represents the link to the compartment. More than one may be listed because a resource may be linked to a compartment in more than one way,.'
    endParam: Optional_['uri'] = Field(default=None)
    'Search Parameter for mapping requests made with $everything.end (e.g. on [Patient.$everything](patient-operation-everything.html)).'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    startParam: Optional_['uri'] = Field(default=None)
    'Search Parameter for mapping requests made with $everything.start (e.g. on [Patient.$everything](patient-operation-everything.html)).'
    documentation: Optional_['string'] = Field(default=None)
    'Additional documentation about the resource and compartment.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'code' = Field(default=...)
    'The name of a resource supported by the server.'


class CompartmentDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A compartment definition that defines how resources are accessed on a server."""
    resourceType: Literal_['CompartmentDefinition'] = Field(default=...)
    'Compartment Definition for a resource'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: 'uri' = Field(default=...)
    'An absolute URI that is used to identify this compartment definition when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this compartment definition is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the compartment definition is stored on different servers.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: 'string' = Field(default=...)
    'A natural language name identifying the compartment definition. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    date: Optional_['dateTime'] = Field(default=None)
    'The date  (and optionally time) when the compartment definition was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the compartment definition changes.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the capability statement.'
    status: 'code' = Field(default=...)
    'The status of this compartment definition. Enables tracking the life-cycle of the content.'
    search: 'boolean' = Field(default=...)
    'Whether the search syntax is supported,.'
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the compartment definition when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the compartment definition author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    purpose: Optional_['markdown'] = Field(default=None)
    'Explanation of why this compartment definition is needed and why it has been designed as it has.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    resource: List_['CompartmentDefinitionResource'] = Field(default_factory=list)
    'Information about how a resource is related to the compartment.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual responsible for the release and ongoing maintenance of the compartment definition.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate compartment definition instances.'
    description: Optional_['markdown'] = Field(default=None)
    "A free text natural language description of the compartment definition from a consumer's perspective."
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this compartment definition is authored for testing purposes (or education/evaluation/marketing) and is not intended to be used for genuine usage.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    code: 'code' = Field(default=...)
    'Which compartment this definition describes.'


class CompositionAttester(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A participant who has attested to the accuracy of the composition/document."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    mode: 'CodeableConcept' = Field(default=...)
    'The type of attestation the authenticator offers.'
    time: Optional_['dateTime'] = Field(default=None)
    'When the composition was attested by the party.'
    party: Optional_['Reference'] = Field(default=None)
    'Who attested the composition in the specified way.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CompositionSection(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The root of the sections that make up the composition."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'A code identifying the kind of content contained within the section. This must be consistent with the section title.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains the attested content of the section, used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative.'
    title: Optional_['string'] = Field(default=None)
    'The label for this particular section.  This will be part of the rendered content for the document, and is often used to build a table of contents.'
    focus: Optional_['Reference'] = Field(default=None)
    'The actual focus of the section when it is not the subject of the composition, but instead represents something or someone associated with the subject such as (for a patient subject) a spouse, parent, fetus, or donor. If not focus is specified, the focus is assumed to be focus of the parent section, or, for a section in the Composition itself, the subject of the composition. Sections with a focus SHALL only include resources where the logical subject (patient, subject, focus, etc.) matches the section focus, or the resources have no logical subject (few resources).'
    entry: List_['Reference'] = Field(default_factory=list)
    'A reference to the actual resource from which the narrative in the section is derived.'
    author: List_['Reference'] = Field(default_factory=list)
    'Identifies who is responsible for the information in this section, not necessarily who typed it in.'
    section: List_['CompositionSection'] = Field(default_factory=list)
    'A nested sub-section within this section.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    orderedBy: Optional_['CodeableConcept'] = Field(default=None)
    'Specifies the order applied to the items in the section entries.'
    emptyReason: Optional_['CodeableConcept'] = Field(default=None)
    'If the section is empty, why the list is empty. An empty section typically has some text explaining the empty reason.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CompositionEvent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The clinical service, such as a colonoscopy or an appendectomy, being documented."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    period: Optional_['Period'] = Field(default=None)
    'The period of time covered by the documentation. There is no assertion that the documentation is a complete representation for this period, only that it documents events during this time.'
    detail: List_['CodeableReference'] = Field(default_factory=list)
    'Represents the main clinical acts, such as a colonoscopy or an appendectomy, being documented. In some cases, the event is inherent in the typeCode, such as a "History and Physical Report" in which case the procedure being documented is necessarily a "History and Physical" act. The events may be included as a code or as a reference to an other resource.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Composition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A set of healthcare-related information that is assembled together into a single logical package that provides a single coherent statement of meaning, establishes its own context and that has clinical attestation with regard to who is making the statement. A Composition defines the structure and narrative content necessary for a document. However, a Composition alone does not constitute a document. Rather, the Composition must be the first entry in a Bundle where Bundle.type=document, and any other resources referenced from Composition must be included as subsequent entries in the Bundle (for example Patient, Practitioner, Encounter, etc.)."""
    resourceType: Literal_['Composition'] = Field(default=...)
    'A set of resources composed into a single coherent clinical statement with clinical attestation'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this Composition when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this Composition is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the Composition is stored on different servers.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    type: 'CodeableConcept' = Field(default=...)
    'Specifies the particular kind of composition (e.g. History and Physical, Discharge Summary, Progress Note). This usually equates to the purpose of making the composition.'
    date: 'dateTime' = Field(default=...)
    'The composition editing time, when the composition was last logically changed by the author.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the {{title}}. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    note: List_['Annotation'] = Field(default_factory=list)
    'For any additional notes.'
    title: 'string' = Field(default=...)
    'Official human-readable label for the composition.'
    event: List_['CompositionEvent'] = Field(default_factory=list)
    'The clinical service, such as a colonoscopy or an appendectomy, being documented.'
    status: 'code' = Field(default=...)
    'The workflow/clinical status of this composition. The status is a marker for the clinical standing of the document.'
    author: List_['Reference'] = Field(default=...)
    'Identifies who is responsible for the information in the composition, not necessarily who typed it in.'
    version: Optional_['string'] = Field(default=None)
    'An explicitly assigned identifer of a variation of the content in the Composition.'
    subject: List_['Reference'] = Field(default_factory=list)
    'Who or what the composition is about. The composition can be about a person, (patient or healthcare practitioner), a device (e.g. a machine) or even a group of subjects (such as a document about a herd of livestock, or a set of patients that share a common exposure).'
    section: List_['CompositionSection'] = Field(default_factory=list)
    'The root of the sections that make up the composition.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'A categorization for the type of the composition - helps for indexing and searching. This may be implied by or derived from the code specified in the Composition Type.'
    attester: List_['CompositionAttester'] = Field(default_factory=list)
    'A participant who has attested to the accuracy of the composition/document.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: Optional_['Reference'] = Field(default=None)
    'Describes the clinical encounter or type of care this documentation is associated with.'
    custodian: Optional_['Reference'] = Field(default=None)
    'Identifies the organization or group who is responsible for ongoing maintenance of and access to the composition/document information.'
    relatesTo: List_['RelatedArtifact'] = Field(default_factory=list)
    'Relationships that this composition has with other compositions or documents that already exist.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A version-independent identifier for the Composition. This identifier stays constant as the composition is changed over time.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate Composition instances.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConceptMapAdditionalAttribute(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An additionalAttribute defines an additional data element found in the source or target data model where the data will come from or be mapped to. Some mappings are based on data in addition to the source data element, where codes in multiple fields are combined to a single field (or vice versa)."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'The type of the source data contained in this concept map for this data element.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    description: Optional_['string'] = Field(default=None)
    'A description of the additional attribute and/or the data element it refers to - why it is defined, and how the value might be used in mappings, and a discussion of issues associated with the use of the data element.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    uri: Optional_['uri'] = Field(default=None)
    'Reference to the formal definition of the source/target data element. For elements defined by the FHIR specification, or using a FHIR logical model, the correct format is {canonical-url}#{element-id}.'
    code: 'code' = Field(default=...)
    'A code that is used to identify this additional data attribute. The code is used internally in ConceptMap.group.element.target.dependsOn.attribute and ConceptMap.group.element.target.product.attribute.'


class ConceptMapProperty(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A property defines a slot through which additional information can be provided about a map from source -> target."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'The type of the property value.'
    system: Optional_['canonical'] = Field(default=None)
    'The CodeSystem that defines the codes from which values of type ```code``` in property values.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    description: Optional_['string'] = Field(default=None)
    'A description of the property - why it is defined, and how its value might be used.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    uri: Optional_['uri'] = Field(default=None)
    'Reference to the formal meaning of the property.'
    code: 'code' = Field(default=...)
    'A code that is used to identify the property. The code is used internally (in ConceptMap.group.element.target.property.code) and also in the $translate operation.'


class ConceptMapGroup(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A group of mappings that all have the same source and target system."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    source: Optional_['canonical'] = Field(default=None)
    'An absolute URI that identifies the source system where the concepts to be mapped are defined.'
    target: Optional_['canonical'] = Field(default=None)
    'An absolute URI that identifies the target system that the concepts will be mapped to.'
    element: List_['ConceptMapGroupElement'] = Field(default=...)
    'Mappings for an individual concept in the source to one or more concepts in the target.'
    unmapped: Optional_['ConceptMapGroupUnmapped'] = Field(default=None)
    'What to do when there is no mapping to a target concept from the source concept and ConceptMap.group.element.noMap is not true. This provides the "default" to be applied when there is no target concept mapping specified or the expansion of ConceptMap.group.element.target.valueSet is empty.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConceptMapGroupUnmapped(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """What to do when there is no mapping to a target concept from the source concept and ConceptMap.group.element.noMap is not true. This provides the "default" to be applied when there is no target concept mapping specified or the expansion of ConceptMap.group.element.target.valueSet is empty."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    mode: 'code' = Field(default=...)
    'Defines which action to take if there is no match for the source concept in the target system designated for the group. One of 3 actions are possible: use the unmapped source code (this is useful when doing a mapping between versions, and only a few codes have changed), use a fixed code (a default code), or alternatively, a reference to a different concept map can be provided (by canonical URL).'
    display: Optional_['string'] = Field(default=None)
    'The display for the code. The display is only provided to help editors when editing the concept map.'
    valueSet: Optional_['canonical'] = Field(default=None)
    "The set of fixed codes to use when the mode = 'fixed'  - all unmapped codes are mapped to each of the fixed codes."
    otherMap: Optional_['canonical'] = Field(default=None)
    'The canonical reference to an additional ConceptMap resource instance to use for mapping if this ConceptMap resource contains no matching mapping for the source concept.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    relationship: Optional_['code'] = Field(default=None)
    "The default relationship value to apply between the source and target concepts when the source code is unmapped and the mode is 'fixed' or 'use-source-code'."
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: Optional_['code'] = Field(default=None)
    "The fixed code to use when the mode = 'fixed'  - all unmapped codes are mapped to a single fixed code."


class ConceptMapGroupElement(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Mappings for an individual concept in the source to one or more concepts in the target."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    noMap: Optional_['boolean'] = Field(default=None)
    'If noMap = true this indicates that no mapping to a target concept exists for this source concept.'
    target: List_['ConceptMapGroupElementTarget'] = Field(default_factory=list)
    'A concept from the target value set that this concept maps to.'
    display: Optional_['string'] = Field(default=None)
    'The display for the code. The display is only provided to help editors when editing the concept map.'
    valueSet: Optional_['canonical'] = Field(default=None)
    'The set of concepts from the ConceptMap.group.source code system which are all being mapped to the target as part of this mapping rule.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: Optional_['code'] = Field(default=None)
    'Identity (code or path) or the element/item being mapped.'


class ConceptMapGroupElementTarget(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A concept from the target value set that this concept maps to."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    display: Optional_['string'] = Field(default=None)
    'The display for the code. The display is only provided to help editors when editing the concept map.'
    comment: Optional_['string'] = Field(default=None)
    'A description of status/issues in mapping that conveys additional information not represented in  the structured data.'
    product: List_['ConceptMapGroupElementTargetDependsOn'] = Field(default_factory=list)
    'Product is the output of a ConceptMap that provides additional values that go in other attributes / data elemnts of the target data.'
    valueSet: Optional_['canonical'] = Field(default=None)
    'The set of concepts from the ConceptMap.group.target code system which are all being mapped to as part of this mapping rule. The effect of using this data element is the same as having multiple ConceptMap.group.element.target elements with one for each concept in the ConceptMap.group.element.target.valueSet value set.'
    property: List_['ConceptMapGroupElementTargetProperty'] = Field(default_factory=list)
    'A property value for this source -> target mapping.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    dependsOn: List_['ConceptMapGroupElementTargetDependsOn'] = Field(default_factory=list)
    'A set of additional dependencies for this mapping to hold. This mapping is only applicable if the specified data attribute can be resolved, and it has the specified value.'
    relationship: 'code' = Field(default=...)
    'The relationship between the source and target concepts. The relationship is read from source to target (e.g. source-is-narrower-than-target).'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: Optional_['code'] = Field(default=None)
    'Identity (code or path) or the element/item that the map refers to.'


class ConceptMapGroupElementTargetDependsOn(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A set of additional dependencies for this mapping to hold. This mapping is only applicable if the specified data attribute can be resolved, and it has the specified value."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    valueCode: Optional_['code'] = Field(default=None)
    'Data element value that the map depends on / produces.'
    valueCoding: Optional_['Coding'] = Field(default=None)
    'Data element value that the map depends on / produces.'
    valueString: Optional_['string'] = Field(default=None)
    'Data element value that the map depends on / produces.'
    valueBoolean: Optional_['boolean'] = Field(default=None)
    'Data element value that the map depends on / produces.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'Data element value that the map depends on / produces.'
    valueSet: Optional_['canonical'] = Field(default=None)
    'This mapping applies if the data element value is a code from this value set.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    attribute: 'code' = Field(default=...)
    'A reference to the additional attribute that holds a value the map depends on.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConceptMapGroupElementTargetProperty(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A property value for this source -> target mapping."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    valueCoding: Optional_['Coding'] = Field(default=None)
    "The value of this property. If the type chosen for this element is 'code', then the property SHALL be defined in a ConceptMap.property element."
    valueString: Optional_['string'] = Field(default=None)
    "The value of this property. If the type chosen for this element is 'code', then the property SHALL be defined in a ConceptMap.property element."
    valueInteger: Optional_['integer'] = Field(default=None)
    "The value of this property. If the type chosen for this element is 'code', then the property SHALL be defined in a ConceptMap.property element."
    valueBoolean: Optional_['boolean'] = Field(default=None)
    "The value of this property. If the type chosen for this element is 'code', then the property SHALL be defined in a ConceptMap.property element."
    valueDateTime: Optional_['dateTime'] = Field(default=None)
    "The value of this property. If the type chosen for this element is 'code', then the property SHALL be defined in a ConceptMap.property element."
    valueDecimal: Optional_['decimal'] = Field(default=None)
    "The value of this property. If the type chosen for this element is 'code', then the property SHALL be defined in a ConceptMap.property element."
    valueCode: Optional_['code'] = Field(default=None)
    "The value of this property. If the type chosen for this element is 'code', then the property SHALL be defined in a ConceptMap.property element."
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'code' = Field(default=...)
    'A reference to a mapping property defined in ConceptMap.property.'


class ConceptMap(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A statement of relationships from one set of concepts to one or more other concepts - either concepts in code systems, or data element/data element concepts, or classes in class models."""
    resourceType: Literal_['ConceptMap'] = Field(default=...)
    'A map from one set of concepts to one or more other concepts'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this concept map when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this concept map is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the concept map is stored on different servers.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the concept map. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the concept map.'
    topic: List_['CodeableConcept'] = Field(default_factory=list)
    'Descriptions related to the content of the ConceptMap. Topics provide a high-level categorization as well as keywords for the ConceptMap that can be useful for filtering and searching.'
    group: List_['ConceptMapGroup'] = Field(default_factory=list)
    'A group of mappings that all have the same source and target system.'
    status: 'code' = Field(default=...)
    'The status of this concept map. Enables tracking the life-cycle of the content.'
    author: List_['ContactDetail'] = Field(default_factory=list)
    'An individiual or organization primarily involved in the creation and maintenance of the ConceptMap.'
    editor: List_['ContactDetail'] = Field(default_factory=list)
    'An individual or organization primarily responsible for internal coherence of the ConceptMap.'
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the concept map when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the concept map author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    purpose: Optional_['markdown'] = Field(default=None)
    'Explanation of why this concept map is needed and why it has been designed as it has.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    reviewer: List_['ContactDetail'] = Field(default_factory=list)
    'An individual or organization asserted by the publisher to be primarily responsible for review of some aspect of the ConceptMap.'
    endorser: List_['ContactDetail'] = Field(default_factory=list)
    'An individual or organization asserted by the publisher to be responsible for officially endorsing the ConceptMap for use in some setting.'
    property: List_['ConceptMapProperty'] = Field(default_factory=list)
    'A property defines a slot through which additional information can be provided about a map from source -> target.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual responsible for the release and ongoing maintenance of the concept map.'
    copyright: Optional_['markdown'] = Field(default=None)
    'A copyright statement relating to the concept map and/or its contents. Copyright statements are generally legal restrictions on the use and publishing of the concept map.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this concept map when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate concept map instances.'
    description: Optional_['markdown'] = Field(default=None)
    "A free text natural language description of the concept map from a consumer's perspective."
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this concept map is authored for testing purposes (or education/evaluation/marketing) and is not intended to be used for genuine usage.'
    jurisdiction: List_['CodeableConcept'] = Field(default_factory=list)
    'A legal or geographic region in which the concept map is intended to be used.'
    approvalDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was approved by the publisher. Approval happens once when the content is officially approved for usage.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    copyrightLabel: Optional_['string'] = Field(default=None)
    "A short string (<50 characters), suitable for inclusion in a page footer that identifies the copyright holder, effective period, and optionally whether rights are resctricted. (e.g. 'All rights reserved', 'Some rights reserved')."
    lastReviewDate: Optional_['date'] = Field(default=None)
    'The date on which the resource content was last reviewed. Review happens periodically after approval but does not change the original approval date.'
    sourceScopeUri: Optional_['uri'] = Field(default=None)
    'Identifier for the source value set that contains the concepts that are being mapped and provides context for the mappings.  Limits the scope of the map to source codes (ConceptMap.group.element code or valueSet) that are members of this value set.'
    sourceScopeCanonical: Optional_['canonical'] = Field(default=None)
    'Identifier for the source value set that contains the concepts that are being mapped and provides context for the mappings.  Limits the scope of the map to source codes (ConceptMap.group.element code or valueSet) that are members of this value set.'
    targetScopeUri: Optional_['uri'] = Field(default=None)
    'Identifier for the target value set that provides important context about how the mapping choices are made.  Limits the scope of the map to target codes (ConceptMap.group.element.target code or valueSet) that are members of this value set.'
    targetScopeCanonical: Optional_['canonical'] = Field(default=None)
    'Identifier for the target value set that provides important context about how the mapping choices are made.  Limits the scope of the map to target codes (ConceptMap.group.element.target code or valueSet) that are members of this value set.'
    effectivePeriod: Optional_['Period'] = Field(default=None)
    'The period during which the ConceptMap content was or is planned to be in active use.'
    relatedArtifact: List_['RelatedArtifact'] = Field(default_factory=list)
    'Related artifacts such as additional documentation, justification, dependencies, bibliographic references, and predecessor and successor artifacts.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which ConceptMap is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which ConceptMap is more current.'
    additionalAttribute: List_['ConceptMapAdditionalAttribute'] = Field(default_factory=list)
    'An additionalAttribute defines an additional data element found in the source or target data model where the data will come from or be mapped to. Some mappings are based on data in addition to the source data element, where codes in multiple fields are combined to a single field (or vice versa).'
    date: Optional_['dateTime'] = Field(default=None)
    'The date  (and optionally time) when the concept map was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the concept map changes.'


class ConditionParticipant(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Indicates who or what participated in the activities related to the condition and how they were involved."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    actor: 'Reference' = Field(default=...)
    'Indicates who or what participated in the activities related to the condition.'
    function: Optional_['CodeableConcept'] = Field(default=None)
    'Distinguishes the type of involvement of the actor in the activities related to the condition.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConditionStage(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A simple summary of the stage such as "Stage 3" or "Early Onset". The determination of the stage is disease-specific, such as cancer, retinopathy of prematurity, kidney diseases, Alzheimer's, or Parkinson disease."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The kind of staging, such as pathological or clinical staging.'
    summary: Optional_['CodeableConcept'] = Field(default=None)
    'A simple summary of the stage such as "Stage 3" or "Early Onset". The determination of the stage is disease-specific, such as cancer, retinopathy of prematurity, kidney diseases, Alzheimer\'s, or Parkinson disease.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    assessment: List_['Reference'] = Field(default_factory=list)
    'Reference to a formal record of the evidence on which the staging assessment is based.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Condition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A clinical condition, problem, diagnosis, or other event, situation, issue, or clinical concept that has risen to a level of concern."""
    resourceType: Literal_['Condition'] = Field(default=...)
    'Detailed information about conditions, problems or diagnoses'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Additional information about the Condition. This is a general notes/comments entry  for description of the Condition, its diagnosis and prognosis.'
    stage: List_['ConditionStage'] = Field(default_factory=list)
    'A simple summary of the stage such as "Stage 3" or "Early Onset". The determination of the stage is disease-specific, such as cancer, retinopathy of prematurity, kidney diseases, Alzheimer\'s, or Parkinson disease.'
    subject: 'Reference' = Field(default=...)
    'Indicates the patient or group who the condition record is associated with.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'A category assigned to the condition.'
    severity: Optional_['CodeableConcept'] = Field(default=None)
    'A subjective assessment of the severity of the condition as evaluated by the clinician.'
    bodySite: List_['CodeableConcept'] = Field(default_factory=list)
    'The anatomical location where this condition manifests itself.'
    onsetDateTime: Optional_['dateTime'] = Field(default=None)
    'Estimated or actual date or date-time  the condition began, in the opinion of the clinician.'
    onsetAge: Optional_['Age'] = Field(default=None)
    'Estimated or actual date or date-time  the condition began, in the opinion of the clinician.'
    onsetPeriod: Optional_['Period'] = Field(default=None)
    'Estimated or actual date or date-time  the condition began, in the opinion of the clinician.'
    onsetRange: Optional_['Range'] = Field(default=None)
    'Estimated or actual date or date-time  the condition began, in the opinion of the clinician.'
    onsetString: Optional_['string'] = Field(default=None)
    'Estimated or actual date or date-time  the condition began, in the opinion of the clinician.'
    evidence: List_['CodeableReference'] = Field(default_factory=list)
    "Supporting evidence / manifestations that are the basis of the Condition's verification status, such as evidence that confirmed or refuted the condition."
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    encounter: Optional_['Reference'] = Field(default=None)
    'The Encounter during which this Condition was created or to which the creation of this record is tightly associated.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Business identifiers assigned to this condition by the performer or other systems which remain constant as the resource is updated and propagates from server to server.'
    participant: List_['ConditionParticipant'] = Field(default_factory=list)
    'Indicates who or what participated in the activities related to the condition and how they were involved.'
    abatementDateTime: Optional_['dateTime'] = Field(default=None)
    'The date or estimated date that the condition resolved or went into remission. This is called "abatement" because of the many overloaded connotations associated with "remission" or "resolution" - Some conditions, such as chronic conditions, are never really resolved, but they can abate.'
    abatementAge: Optional_['Age'] = Field(default=None)
    'The date or estimated date that the condition resolved or went into remission. This is called "abatement" because of the many overloaded connotations associated with "remission" or "resolution" - Some conditions, such as chronic conditions, are never really resolved, but they can abate.'
    abatementPeriod: Optional_['Period'] = Field(default=None)
    'The date or estimated date that the condition resolved or went into remission. This is called "abatement" because of the many overloaded connotations associated with "remission" or "resolution" - Some conditions, such as chronic conditions, are never really resolved, but they can abate.'
    abatementRange: Optional_['Range'] = Field(default=None)
    'The date or estimated date that the condition resolved or went into remission. This is called "abatement" because of the many overloaded connotations associated with "remission" or "resolution" - Some conditions, such as chronic conditions, are never really resolved, but they can abate.'
    abatementString: Optional_['string'] = Field(default=None)
    'The date or estimated date that the condition resolved or went into remission. This is called "abatement" because of the many overloaded connotations associated with "remission" or "resolution" - Some conditions, such as chronic conditions, are never really resolved, but they can abate.'
    recordedDate: Optional_['dateTime'] = Field(default=None)
    'The recordedDate represents when this particular Condition record was created in the system, which is often a system-generated date.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    clinicalStatus: 'CodeableConcept' = Field(default=...)
    'The clinical status of the condition.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    verificationStatus: Optional_['CodeableConcept'] = Field(default=None)
    'The verification status to support the clinical status of the condition.  The verification status pertains to the condition, itself, not to any specific condition attribute.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'Identification of the condition, problem or diagnosis.'


class ConditionDefinitionQuestionnaire(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Questionnaire for this condition."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    purpose: 'code' = Field(default=...)
    'Use of the questionnaire.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: 'Reference' = Field(default=...)
    'Specific Questionnaire.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConditionDefinitionPrecondition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An observation that suggests that this condition applies."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'code' = Field(default=...)
    'Kind of pre-condition.'
    valueCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Value of Observation.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'Value of Observation.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    code: 'CodeableConcept' = Field(default=...)
    'Code for relevant Observation.'


class ConditionDefinitionObservation(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Observations particularly relevant to this condition."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'Code for relevant Observation.'
    category: Optional_['CodeableConcept'] = Field(default=None)
    'Category that is relevant.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConditionDefinitionMedication(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Medications particularly relevant for this condition."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: Optional_['CodeableConcept'] = Field(default=None)
    'Code for relevant Medication.'
    category: Optional_['CodeableConcept'] = Field(default=None)
    'Category that is relevant.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConditionDefinitionPlan(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Plan that is appropriate."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    role: Optional_['CodeableConcept'] = Field(default=None)
    'Use for the plan.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: 'Reference' = Field(default=...)
    'The actual plan.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConditionDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A definition of a condition and information relevant to managing it."""
    resourceType: Literal_['ConditionDefinition'] = Field(default=...)
    'A definition of a condition'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'An absolute URI that is used to identify this condition definition when it is referenced in a specification, model, design or an instance; also called its canonical identifier. This SHOULD be globally unique and SHOULD be a literal address at which an authoritative instance of this condition definition is (or will be) published. This URL can be the target of a canonical reference. It SHALL remain the same when the condition definition is stored on different servers.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying the condition definition. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    date: Optional_['dateTime'] = Field(default=None)
    'The date  (and optionally time) when the condition definition was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the condition definition changes.'
    team: List_['Reference'] = Field(default_factory=list)
    'Appropriate team for this condition.'
    plan: List_['ConditionDefinitionPlan'] = Field(default_factory=list)
    'Plan that is appropriate.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for the condition definition.'
    stage: Optional_['CodeableConcept'] = Field(default=None)
    'Clinical stage or grade of a condition. May include formal severity assessments.'
    status: 'code' = Field(default=...)
    'The status of this condition definition. Enables tracking the life-cycle of the content.'
    version: Optional_['string'] = Field(default=None)
    'The identifier that is used to identify this version of the condition definition when it is referenced in a specification, model, design or instance. This is an arbitrary value managed by the condition definition author and is not expected to be globally unique. For example, it might be a timestamp (e.g. yyyymmdd) if a managed version is not available. There is also no expectation that versions can be placed in a lexicographical sequence.'
    contact: List_['ContactDetail'] = Field(default_factory=list)
    'Contact details to assist a user in finding and communicating with the publisher.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    subtitle: Optional_['string'] = Field(default=None)
    'An explanatory or alternate title for the event definition giving additional information about its content.'
    severity: Optional_['CodeableConcept'] = Field(default=None)
    'A subjective assessment of the severity of the condition as evaluated by the clinician.'
    bodySite: Optional_['CodeableConcept'] = Field(default=None)
    'The anatomical location where this condition manifests itself.'
    hasStage: Optional_['boolean'] = Field(default=None)
    'Whether stage is appropriate to collect for this condition.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['string'] = Field(default=None)
    'The name of the organization or individual responsible for the release and ongoing maintenance of the condition definition.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'A formal identifier that is used to identify this condition definition when it is represented in other formats, or referenced in a specification, model, design or an instance.'
    useContext: List_['UsageContext'] = Field(default_factory=list)
    'The content was developed with a focus and intent of supporting the contexts that are listed. These contexts may be general categories (gender, age, ...) or may be references to specific programs (insurance plans, studies, ...) and may be used to assist with indexing and searching for appropriate condition definition instances.'
    definition: List_['uri'] = Field(default_factory=list)
    'Formal definitions of the condition. These may be references to ontologies, published clinical protocols or research papers.'
    medication: List_['ConditionDefinitionMedication'] = Field(default_factory=list)
    'Medications particularly relevant for this condition.'
    description: Optional_['markdown'] = Field(default=None)
    "A free text natural language description of the condition definition from a consumer's perspective."
    hasSeverity: Optional_['boolean'] = Field(default=None)
    'Whether Severity is appropriate to collect for this condition.'
    hasBodySite: Optional_['boolean'] = Field(default=None)
    'Whether bodySite is appropriate to collect for this condition.'
    observation: List_['ConditionDefinitionObservation'] = Field(default_factory=list)
    'Observations particularly relevant to this condition.'
    experimental: Optional_['boolean'] = Field(default=None)
    'A Boolean value to indicate that this condition definition is authored for testing purposes (or education/evaluation/marketing) and is not intended to be used for genuine usage.'
    jurisdiction: List_['CodeableConcept'] = Field(default_factory=list)
    'A legal or geographic region in which the condition definition is intended to be used.'
    precondition: List_['ConditionDefinitionPrecondition'] = Field(default_factory=list)
    'An observation that suggests that this condition applies.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    questionnaire: List_['ConditionDefinitionQuestionnaire'] = Field(default_factory=list)
    'Questionnaire for this condition.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    versionAlgorithmString: Optional_['string'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    versionAlgorithmCoding: Optional_['Coding'] = Field(default=None)
    'Indicates the mechanism used to compare versions to determine which is more current.'
    code: 'CodeableConcept' = Field(default=...)
    'Identification of the condition, problem or diagnosis.'


class ConsentVerification(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Whether a treatment instruction (e.g. artificial respiration: yes or no) was verified with the patient, his/her family or another authorized person."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    verified: 'boolean' = Field(default=...)
    'Has the instruction been verified.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    verifiedBy: Optional_['Reference'] = Field(default=None)
    'The person who conducted the verification/validation of the Grantor decision.'
    verifiedWith: Optional_['Reference'] = Field(default=None)
    'Who verified the instruction (Patient, Relative or other Authorized Person).'
    verificationType: Optional_['CodeableConcept'] = Field(default=None)
    'Extensible list of verification type starting with verification and re-validation.'
    verificationDate: List_['dateTime'] = Field(default_factory=list)
    'Date(s) verification was collected.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConsentPolicyBasis(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A Reference or URL used to uniquely identify the policy the organization will enforce for this Consent. This Reference or URL should be specific to the version of the policy and should be dereferencable to a computable policy of some form."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: Optional_['Reference'] = Field(default=None)
    'A Reference that identifies the policy the organization will enforce for this Consent.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    url: Optional_['url'] = Field(default=None)
    'A URL that links to a computable version of the policy the organization will enforce for this Consent.'


class ConsentProvision(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An exception to the base policy of this consent. An exception can be an addition or removal of access permissions."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: List_['CodeableConcept'] = Field(default_factory=list)
    'If this code is found in an instance, then the provision applies.'
    data: List_['ConsentProvisionData'] = Field(default_factory=list)
    'The resources controlled by this provision if specific resources are referenced.'
    actor: List_['ConsentProvisionActor'] = Field(default_factory=list)
    "Who or what is controlled by this provision. Use group to identify a set of actors by some property they share (e.g. 'admitting officers')."
    period: Optional_['Period'] = Field(default=None)
    'Timeframe for this provision.'
    action: List_['CodeableConcept'] = Field(default_factory=list)
    'Actions controlled by this provision.'
    purpose: List_['Coding'] = Field(default_factory=list)
    'The context of the activities a user is taking - why the user is accessing the data - that are controlled by this provision.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    provision: List_['ConsentProvision'] = Field(default_factory=list)
    'Provisions which provide exceptions to the base provision or subprovisions.'
    dataPeriod: Optional_['Period'] = Field(default=None)
    'Clinical or Operational Relevant period of time that bounds the data controlled by this provision.'
    expression: Optional_['Expression'] = Field(default=None)
    'A computable (FHIRPath or other) definition of what is controlled by this consent.'
    documentType: List_['Coding'] = Field(default_factory=list)
    'The documentType(s) covered by this provision. The type can be a CDA document, or some other type that indicates what sort of information the consent relates to.'
    resourceType: List_['Coding'] = Field(default_factory=list)
    'The resourceType(s) covered by this provision. The type can be a FHIR resource type or a profile on a type that indicates what information the consent relates to.'
    securityLabel: List_['Coding'] = Field(default_factory=list)
    'A security label, comprised of 0..* security label fields (Privacy tags), which define which resources are controlled by this exception.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConsentProvisionActor(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Who or what is controlled by this provision. Use group to identify a set of actors by some property they share (e.g. 'admitting officers')."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    role: Optional_['CodeableConcept'] = Field(default=None)
    'How the individual is involved in the resources content that is described in the exception.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: Optional_['Reference'] = Field(default=None)
    "The resource that identifies the actor. To identify actors by type, use group to identify a set of actors by some property they share (e.g. 'admitting officers')."
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ConsentProvisionData(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The resources controlled by this provision if specific resources are referenced."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    meaning: 'code' = Field(default=...)
    'How the resource reference is interpreted when testing consent restrictions.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: 'Reference' = Field(default=...)
    'A reference to a specific resource that defines which resources are covered by this consent.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Consent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A record of a healthcare consumer’s  choices  or choices made on their behalf by a third party, which permits or denies identified recipient(s) or recipient role(s) to perform one or more actions within a given policy context, for specific purposes and periods of time."""
    resourceType: Literal_['Consent'] = Field(default=...)
    "A healthcare consumer's  or third party's choices to permit or deny recipients or roles to perform actions for specific purposes and periods of time"
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    status: 'code' = Field(default=...)
    'Indicates the current state of this Consent resource.'
    period: Optional_['Period'] = Field(default=None)
    'Effective period for this Consent Resource and all provisions unless specified in that provision.'
    subject: Optional_['Reference'] = Field(default=None)
    'The patient/healthcare practitioner or group of persons to whom this consent applies.'
    grantor: List_['Reference'] = Field(default_factory=list)
    'The entity responsible for granting the rights listed in a Consent Directive.'
    grantee: List_['Reference'] = Field(default_factory=list)
    'The entity responsible for complying with the Consent Directive, including any obligations or limitations on authorizations and enforcement of prohibitions.'
    manager: List_['Reference'] = Field(default_factory=list)
    'The actor that manages the consent through its lifecycle.'
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    category: List_['CodeableConcept'] = Field(default_factory=list)
    'A classification of the type of consents found in the statement. This element supports indexing and retrieval of consent statements.'
    decision: Optional_['code'] = Field(default=None)
    'Action to take - permit or deny - as default.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    provision: List_['ConsentProvision'] = Field(default_factory=list)
    'An exception to the base policy of this consent. An exception can be an addition or removal of access permissions.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Unique identifier for this copy of the Consent Statement.'
    controller: List_['Reference'] = Field(default_factory=list)
    'The actor that controls/enforces the access according to the consent.'
    policyText: List_['Reference'] = Field(default_factory=list)
    'A Reference to the human readable policy explaining the basis for the Consent.'
    policyBasis: Optional_['ConsentPolicyBasis'] = Field(default=None)
    'A Reference or URL used to uniquely identify the policy the organization will enforce for this Consent. This Reference or URL should be specific to the version of the policy and should be dereferencable to a computable policy of some form.'
    verification: List_['ConsentVerification'] = Field(default_factory=list)
    'Whether a treatment instruction (e.g. artificial respiration: yes or no) was verified with the patient, his/her family or another authorized person.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    sourceReference: List_['Reference'] = Field(default_factory=list)
    'A reference to a consent that links back to such a source, a reference to a document repository (e.g. XDS) that stores the original consent document.'
    regulatoryBasis: List_['CodeableConcept'] = Field(default_factory=list)
    'A set of codes that indicate the regulatory basis (if any) that this consent supports.'
    sourceAttachment: List_['Attachment'] = Field(default_factory=list)
    'The source on which this consent statement is based. The source might be a scanned original paper form.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    date: Optional_['date'] = Field(default=None)
    'Date the consent instance was agreed to.'


class ContractContentDefinition(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Precusory content developed with a focus and intent of supporting the formation a Contract instance, which may be associated with and transformable into a Contract."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'Precusory content structure and use, i.e., a boilerplate, template, application for a contract such as an insurance policy or benefits under a program, e.g., workers compensation.'
    subType: Optional_['CodeableConcept'] = Field(default=None)
    'Detailed Precusory content type.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    publisher: Optional_['Reference'] = Field(default=None)
    'The  individual or organization that published the Contract precursor content.'
    copyright: Optional_['markdown'] = Field(default=None)
    'A copyright statement relating to Contract precursor content. Copyright statements are generally legal restrictions on the use and publishing of the Contract precursor content.'
    publicationDate: Optional_['dateTime'] = Field(default=None)
    'The date (and optionally time) when the contract was last significantly changed. The date must change when the business version changes and it must change if the status code changes. In addition, it should change when the substantive content of the contract changes.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    publicationStatus: 'code' = Field(default=...)
    'amended | appended | cancelled | disputed | entered-in-error | executable +.'


class ContractFriendly(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The "patient friendly language" versionof the Contract in whole or in parts. "Patient friendly language" means the representation of the Contract and Contract Provisions in a manner that is readily accessible and understandable by a layperson in accordance with best practices for communication styles that ensure that those agreeing to or signing the Contract understand the roles, actions, obligations, responsibilities, and implication of the agreement."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    contentAttachment: Optional_['Attachment'] = Field(default=None)
    'Human readable rendering of this Contract in a format and representation intended to enhance comprehension and ensure understandability.'
    contentReference: Optional_['Reference'] = Field(default=None)
    'Human readable rendering of this Contract in a format and representation intended to enhance comprehension and ensure understandability.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ContractSigner(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Parties with legal standing in the Contract, including the principal parties, the grantor(s) and grantee(s), which are any person or organization bound by the contract, and any ancillary parties, which facilitate the execution of the contract such as a notary or witness."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'Coding' = Field(default=...)
    'Role of this Contract signer, e.g. notary, grantee.'
    party: 'Reference' = Field(default=...)
    'Party which is a signator to this Contract.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    signature: List_['Signature'] = Field(default=...)
    'Legally binding Contract DSIG signature contents in Base64.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ContractLegal(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """List of Legal expressions or representations of this Contract."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    contentAttachment: Optional_['Attachment'] = Field(default=None)
    'Contract legal text in human renderable form.'
    contentReference: Optional_['Reference'] = Field(default=None)
    'Contract legal text in human renderable form.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ContractRule(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """List of Computable Policy Rule Language Representations of this Contract."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    contentAttachment: Optional_['Attachment'] = Field(default=None)
    'Computable Contract conveyed using a policy rule language (e.g. XACML, DKAL, SecPal).'
    contentReference: Optional_['Reference'] = Field(default=None)
    'Computable Contract conveyed using a policy rule language (e.g. XACML, DKAL, SecPal).'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ContractTerm(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """One or more Contract Provisions, which may be related and conveyed as a group, and may contain nested groups."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'A legal clause or condition contained within a contract that requires one or both parties to perform a particular requirement by some specified time or prevents one or both parties from performing a particular requirement by some specified time.'
    text: Optional_['string'] = Field(default=None)
    'Statement of a provision in a policy or a contract.'
    offer: 'ContractTermOffer' = Field(default=...)
    'The matter of concern in the context of this provision of the agrement.'
    asset: List_['ContractTermAsset'] = Field(default_factory=list)
    'Contract Term Asset List.'
    group: List_['ContractTerm'] = Field(default_factory=list)
    'Nested group of Contract Provisions.'
    issued: Optional_['dateTime'] = Field(default=None)
    'When this Contract Provision was issued.'
    action: List_['ContractTermAction'] = Field(default_factory=list)
    'An actor taking a role in an activity for which it can be assigned some degree of responsibility for the activity taking place.'
    applies: Optional_['Period'] = Field(default=None)
    'Relevant time or time-period when this Contract Provision is applicable.'
    subType: Optional_['CodeableConcept'] = Field(default=None)
    'A specialized legal clause or condition based on overarching contract type.'
    topicCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'The entity that the term applies to.'
    topicReference: Optional_['Reference'] = Field(default=None)
    'The entity that the term applies to.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    identifier: Optional_['Identifier'] = Field(default=None)
    'Unique identifier for this particular Contract Provision.'
    securityLabel: List_['ContractTermSecurityLabel'] = Field(default_factory=list)
    'Security labels that protect the handling of information about the term and its elements, which may be specifically identified.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ContractTermSecurityLabel(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Security labels that protect the handling of information about the term and its elements, which may be specifically identified."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    number: List_['unsignedInt'] = Field(default_factory=list)
    'Number used to link this term or term element to the applicable Security Label.'
    control: List_['Coding'] = Field(default_factory=list)
    'Security label privacy tag that specifies the manner in which term and/or term elements are to be protected.'
    category: List_['Coding'] = Field(default_factory=list)
    'Security label privacy tag that specifies the applicable privacy and security policies governing this term and/or term elements.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    classification: 'Coding' = Field(default=...)
    'Security label privacy tag that specifies the level of confidentiality protection required for this term and/or term elements.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ContractTermAction(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """An actor taking a role in an activity for which it can be assigned some degree of responsibility for the activity taking place."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'Activity or service obligation to be done or not done, performed or not performed, effectuated or not by this Contract term.'
    note: List_['Annotation'] = Field(default_factory=list)
    'Comments made about the term action made by the requester, performer, subject or other participants.'
    intent: 'CodeableConcept' = Field(default=...)
    'Reason or purpose for the action stipulated by this Contract Provision.'
    linkId: List_['string'] = Field(default_factory=list)
    'Id [identifier??] of the clause or question text related to this action in the referenced form or QuestionnaireResponse.'
    status: 'CodeableConcept' = Field(default=...)
    'Current state of the term action.'
    reason: List_['CodeableReference'] = Field(default_factory=list)
    'Rationale for the action to be performed or not performed. Describes why the action is permitted or prohibited. Either a coded concept, or another resource whose existence justifies permitting or not permitting this action.'
    subject: List_['ContractTermActionSubject'] = Field(default_factory=list)
    'Entity of the action.'
    context: Optional_['Reference'] = Field(default=None)
    'Encounter or Episode with primary association to the specified term activity.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    requester: List_['Reference'] = Field(default_factory=list)
    'Who or what initiated the action and has responsibility for its activation.'
    performer: Optional_['Reference'] = Field(default=None)
    'Indicates who or what is being asked to perform (or not perform) the ction.'
    doNotPerform: Optional_['boolean'] = Field(default=None)
    'True if the term prohibits the  action.'
    reasonLinkId: List_['string'] = Field(default_factory=list)
    'Id [identifier??] of the clause or question text related to the reason type or reference of this  action in the referenced form or QuestionnaireResponse.'
    contextLinkId: List_['string'] = Field(default_factory=list)
    'Id [identifier??] of the clause or question text related to the requester of this action in the referenced form or QuestionnaireResponse.'
    occurrenceDateTime: Optional_['dateTime'] = Field(default=None)
    'When action happens.'
    occurrencePeriod: Optional_['Period'] = Field(default=None)
    'When action happens.'
    occurrenceTiming: Optional_['Timing'] = Field(default=None)
    'When action happens.'
    performerType: List_['CodeableConcept'] = Field(default_factory=list)
    'The type of individual that is desired or required to perform or not perform the action.'
    performerRole: Optional_['CodeableConcept'] = Field(default=None)
    'The type of role or competency of an individual desired or required to perform or not perform the action.'
    requesterLinkId: List_['string'] = Field(default_factory=list)
    'Id [identifier??] of the clause or question text related to the requester of this action in the referenced form or QuestionnaireResponse.'
    performerLinkId: List_['string'] = Field(default_factory=list)
    'Id [identifier??] of the clause or question text related to the reason type or reference of this  action in the referenced form or QuestionnaireResponse.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    securityLabelNumber: List_['unsignedInt'] = Field(default_factory=list)
    'Security labels that protects the action.'


class ContractTermActionSubject(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Entity of the action."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    role: Optional_['CodeableConcept'] = Field(default=None)
    'Role type of agent assigned roles in this Contract.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: List_['Reference'] = Field(default=...)
    'The entity the action is performed or not performed on or for.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ContractTermAsset(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Contract Term Asset List."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: List_['CodeableConcept'] = Field(default_factory=list)
    'Target entity type about which the term may be concerned.'
    text: Optional_['string'] = Field(default=None)
    'Clause or question text (Prose Object) concerning the asset in a linked form, such as a QuestionnaireResponse used in the formation of the contract.'
    scope: Optional_['CodeableConcept'] = Field(default=None)
    'Differentiates the kind of the asset .'
    period: List_['Period'] = Field(default_factory=list)
    'Asset relevant contractual time period.'
    linkId: List_['string'] = Field(default_factory=list)
    'Id [identifier??] of the clause or question text about the asset in the referenced form or QuestionnaireResponse.'
    answer: List_['ContractTermOfferAnswer'] = Field(default_factory=list)
    'Response to assets.'
    subtype: List_['CodeableConcept'] = Field(default_factory=list)
    'May be a subtype or part of an offered asset.'
    context: List_['ContractTermAssetContext'] = Field(default_factory=list)
    'Circumstance of the asset.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    condition: Optional_['string'] = Field(default=None)
    'Description of the quality and completeness of the asset that may be a factor in its valuation.'
    usePeriod: List_['Period'] = Field(default_factory=list)
    'Time period of asset use.'
    periodType: List_['CodeableConcept'] = Field(default_factory=list)
    'Type of Asset availability for use or ownership.'
    valuedItem: List_['ContractTermAssetValuedItem'] = Field(default_factory=list)
    'Contract Valued Item List.'
    relationship: Optional_['Coding'] = Field(default=None)
    'Specifies the applicability of the term to an asset resource instance, and instances it refers to or instances that refer to it, and/or are owned by the offeree.'
    typeReference: List_['Reference'] = Field(default_factory=list)
    'Associated entities.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    securityLabelNumber: List_['unsignedInt'] = Field(default_factory=list)
    'Security labels that protects the asset.'


class ContractTermAssetValuedItem(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Contract Valued Item List."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    net: Optional_['Money'] = Field(default=None)
    'Expresses the product of the Contract Valued Item unitQuantity and the unitPriceAmt. For example, the formula: unit Quantity * unit Price (Cost per Point) * factor Number  * points = net Amount. Quantity, factor and points are assumed to be 1 if not supplied.'
    factor: Optional_['decimal'] = Field(default=None)
    'A real number that represents a multiplier used in determining the overall value of the Contract Valued Item delivered. The concept of a Factor allows for a discount or surcharge multiplier to be applied to a monetary amount.'
    points: Optional_['decimal'] = Field(default=None)
    'An amount that expresses the weighting (based on difficulty, cost and/or resource intensiveness) associated with the Contract Valued Item delivered. The concept of Points allows for assignment of point values for a Contract Valued Item, such that a monetary amount can be assigned to each point.'
    linkId: List_['string'] = Field(default_factory=list)
    'Id  of the clause or question text related to the context of this valuedItem in the referenced form or QuestionnaireResponse.'
    payment: Optional_['string'] = Field(default=None)
    'Terms of valuation.'
    quantity: Optional_['Quantity'] = Field(default=None)
    'Specifies the units by which the Contract Valued Item is measured or counted, and quantifies the countable or measurable Contract Valued Item instances.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    entityCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Specific type of Contract Valued Item that may be priced.'
    entityReference: Optional_['Reference'] = Field(default=None)
    'Specific type of Contract Valued Item that may be priced.'
    unitPrice: Optional_['Money'] = Field(default=None)
    'A Contract Valued Item unit valuation measure.'
    recipient: Optional_['Reference'] = Field(default=None)
    'Who will receive payment.'
    identifier: Optional_['Identifier'] = Field(default=None)
    'Identifies a Contract Valued Item instance.'
    paymentDate: Optional_['dateTime'] = Field(default=None)
    'When payment is due.'
    responsible: Optional_['Reference'] = Field(default=None)
    'Who will make payment.'
    effectiveTime: Optional_['dateTime'] = Field(default=None)
    'Indicates the time during which this Contract ValuedItem information is effective.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    securityLabelNumber: List_['unsignedInt'] = Field(default_factory=list)
    'A set of security labels that define which terms are controlled by this condition.'


class ContractTermAssetContext(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Circumstance of the asset."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    code: List_['CodeableConcept'] = Field(default_factory=list)
    'Coded representation of the context generally or of the Referenced entity, such as the asset holder type or location.'
    text: Optional_['string'] = Field(default=None)
    'Context description.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: Optional_['Reference'] = Field(default=None)
    'Asset context reference may include the creator, custodian, or owning Person or Organization (e.g., bank, repository),  location held, e.g., building,  jurisdiction.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ContractTermOffer(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """The matter of concern in the context of this provision of the agrement."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'Type of Contract Provision such as specific requirements, purposes for actions, obligations, prohibitions, e.g. life time maximum benefit.'
    text: Optional_['string'] = Field(default=None)
    'Human readable form of this Contract Offer.'
    party: List_['ContractTermOfferParty'] = Field(default_factory=list)
    'Offer Recipient.'
    topic: Optional_['Reference'] = Field(default=None)
    'The owner of an asset has the residual control rights over the asset: the right to decide all usages of the asset in any way not inconsistent with a prior contract, custom, or law (Hart, 1995, p. 30).'
    answer: List_['ContractTermOfferAnswer'] = Field(default_factory=list)
    'Response to offer text.'
    linkId: List_['string'] = Field(default_factory=list)
    'The id of the clause or question text of the offer in the referenced questionnaire/response.'
    decision: Optional_['CodeableConcept'] = Field(default=None)
    'Type of choice made by accepting party with respect to an offer made by an offeror/ grantee.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Unique identifier for this particular Contract Provision.'
    decisionMode: List_['CodeableConcept'] = Field(default_factory=list)
    'How the decision about a Contract was conveyed.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    securityLabelNumber: List_['unsignedInt'] = Field(default_factory=list)
    'Security labels that protects the offer.'


class ContractTermOfferAnswer(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Response to offer text."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    valueBoolean: Optional_['boolean'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueDecimal: Optional_['decimal'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueInteger: Optional_['integer'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueDate: Optional_['date'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueDateTime: Optional_['dateTime'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueTime: Optional_['time'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueString: Optional_['string'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueUri: Optional_['uri'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueAttachment: Optional_['Attachment'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueCoding: Optional_['Coding'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    valueReference: Optional_['Reference'] = Field(default=None)
    'Response to an offer clause or question text,  which enables selection of values to be agreed to, e.g., the period of participation, the date of occupancy of a rental, warranty duration, or whether biospecimen may be used for further research.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class ContractTermOfferParty(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Offer Recipient."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    role: 'CodeableConcept' = Field(default=...)
    'How the party participates in the offer.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    reference: List_['Reference'] = Field(default=...)
    'Participant in the offer.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Contract(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Legally enforceable, formally recorded unilateral or bilateral directive i.e., a policy or agreement."""
    resourceType: Literal_['Contract'] = Field(default=...)
    'Legal Agreement'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    url: Optional_['uri'] = Field(default=None)
    'Canonical identifier for this contract, represented as a URI (globally unique).'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    site: List_['Reference'] = Field(default_factory=list)
    'Sites in which the contract is complied with,  exercised, or in force.'
    name: Optional_['string'] = Field(default=None)
    'A natural language name identifying this Contract definition, derivative, or instance in any legal state. Provides additional information about its content. This name should be usable as an identifier for the module by machine processing applications such as code generation.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    "A high-level category for the legal instrument, whether constructed as a Contract definition, derivative, or instance in any legal state.  Provides additional information about its content within the context of the Contract's scope to distinguish the kinds of systems that would be interested in the contract."
    term: List_['ContractTerm'] = Field(default_factory=list)
    'One or more Contract Provisions, which may be related and conveyed as a group, and may contain nested groups.'
    rule: List_['ContractRule'] = Field(default_factory=list)
    'List of Computable Policy Rule Language Representations of this Contract.'
    title: Optional_['string'] = Field(default=None)
    'A short, descriptive, user-friendly title for this Contract definition, derivative, or instance in any legal state.'
    alias: List_['string'] = Field(default_factory=list)
    'Alternative representation of the title for this Contract definition, derivative, or instance in any legal state., e.g., a domain specific contract number related to legislation.'
    scope: Optional_['CodeableConcept'] = Field(default=None)
    'A selector of legal concerns for this Contract definition, derivative, or instance in any legal state.'
    legal: List_['ContractLegal'] = Field(default_factory=list)
    'List of Legal expressions or representations of this Contract.'
    status: Optional_['code'] = Field(default=None)
    'The status of the resource instance.'
    issued: Optional_['dateTime'] = Field(default=None)
    'When this  Contract was issued.'
    domain: List_['Reference'] = Field(default_factory=list)
    'Recognized governance framework or system operating with a circumscribed scope in accordance with specified principles, policies, processes or procedures for managing rights, actions, or behaviors of parties or principals relative to resources.'
    author: Optional_['Reference'] = Field(default=None)
    'The individual or organization that authored the Contract definition, derivative, or instance in any legal state.'
    signer: List_['ContractSigner'] = Field(default_factory=list)
    'Parties with legal standing in the Contract, including the principal parties, the grantor(s) and grantee(s), which are any person or organization bound by the contract, and any ancillary parties, which facilitate the execution of the contract such as a notary or witness.'
    version: Optional_['string'] = Field(default=None)
    'An edition identifier used for business purposes to label business significant variants.'
    applies: Optional_['Period'] = Field(default=None)
    'Relevant time or time-period when this Contract is applicable.'
    subject: List_['Reference'] = Field(default_factory=list)
    'The target entity impacted by or of interest to parties to the agreement.'
    subType: List_['CodeableConcept'] = Field(default_factory=list)
    "Sub-category for the Contract that distinguishes the kinds of systems that would be interested in the Contract within the context of the Contract's scope."
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    subtitle: Optional_['string'] = Field(default=None)
    'A more detailed or qualifying explanatory or alternate user-friendly title for this Contract definition, derivative, or instance in any legal state.'
    topicCodeableConcept: Optional_['CodeableConcept'] = Field(default=None)
    'Narrows the range of legal concerns to focus on the achievement of specific contractual objectives.'
    topicReference: Optional_['Reference'] = Field(default=None)
    'Narrows the range of legal concerns to focus on the achievement of specific contractual objectives.'
    friendly: List_['ContractFriendly'] = Field(default_factory=list)
    'The "patient friendly language" versionof the Contract in whole or in parts. "Patient friendly language" means the representation of the Contract and Contract Provisions in a manner that is readily accessible and understandable by a layperson in accordance with best practices for communication styles that ensure that those agreeing to or signing the Contract understand the roles, actions, obligations, responsibilities, and implication of the agreement.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    authority: List_['Reference'] = Field(default_factory=list)
    'A formally or informally recognized grouping of people, principals, organizations, or jurisdictions formed for the purpose of achieving some form of collective action such as the promulgation, administration and enforcement of contracts and policies.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'Unique identifier for this Contract or a derivative that references a Source Contract.'
    legalState: Optional_['CodeableConcept'] = Field(default=None)
    'Legal states of the formation of a legal instrument, which is a formally executed written document that can be formally attributed to its author, records and formally expresses a legally enforceable act, process, or contractual duty, obligation, or right, and therefore evidences that act, process, or agreement.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    expirationType: Optional_['CodeableConcept'] = Field(default=None)
    'Event resulting in discontinuation or termination of this Contract instance by one or more parties to the contract.'
    supportingInfo: List_['Reference'] = Field(default_factory=list)
    'Information that may be needed by/relevant to the performer in their execution of this term action.'
    instantiatesUri: Optional_['uri'] = Field(default=None)
    'The URL pointing to an externally maintained definition that is adhered to in whole or in part by this Contract.'
    relevantHistory: List_['Reference'] = Field(default_factory=list)
    'Links to Provenance records for past versions of this Contract definition, derivative, or instance, which identify key state transitions or updates that are likely to be relevant to a user looking at the current version of the Contract.  The Provenance.entity indicates the target that was changed in the update (see [Provenance.entity](provenance-definitions.html#Provenance.entity)).'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    contentDerivative: Optional_['CodeableConcept'] = Field(default=None)
    'The minimal content derived from the basal information source at a specific stage in its lifecycle.'
    contentDefinition: Optional_['ContractContentDefinition'] = Field(default=None)
    'Precusory content developed with a focus and intent of supporting the formation a Contract instance, which may be associated with and transformable into a Contract.'
    legallyBindingAttachment: Optional_['Attachment'] = Field(default=None)
    'Legally binding Contract: This is the signed and legally recognized representation of the Contract, which is considered the "source of truth" and which would be the basis for legal action related to enforcement of this Contract.'
    legallyBindingReference: Optional_['Reference'] = Field(default=None)
    'Legally binding Contract: This is the signed and legally recognized representation of the Contract, which is considered the "source of truth" and which would be the basis for legal action related to enforcement of this Contract.'
    instantiatesCanonical: Optional_['Reference'] = Field(default=None)
    'The URL pointing to a FHIR-defined Contract Definition that is adhered to in whole or part by this Contract.'


class CoverageCostToBeneficiary(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A suite of codes indicating the cost category and associated amount which have been detailed in the policy and may have been  included on the health card."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The category of patient centric costs associated with treatment.'
    unit: Optional_['CodeableConcept'] = Field(default=None)
    'Indicates if the benefits apply to an individual or to the family.'
    term: Optional_['CodeableConcept'] = Field(default=None)
    "The term or period of the values such as 'maximum lifetime benefit' or 'maximum annual visits'."
    network: Optional_['CodeableConcept'] = Field(default=None)
    'Is a flag to indicate whether the benefits refer to in-network providers or out-of-network providers.'
    category: Optional_['CodeableConcept'] = Field(default=None)
    'Code to identify the general type of benefits under which products and services are provided.'
    valueQuantity: Optional_['Quantity'] = Field(default=None)
    'The amount due from the patient for the cost category.'
    valueMoney: Optional_['Money'] = Field(default=None)
    'The amount due from the patient for the cost category.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    exception: List_['CoverageCostToBeneficiaryException'] = Field(default_factory=list)
    'A suite of codes indicating exceptions or reductions to patient costs and their effective periods.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CoverageCostToBeneficiaryException(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A suite of codes indicating exceptions or reductions to patient costs and their effective periods."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'The code for the specific exception.'
    period: Optional_['Period'] = Field(default=None)
    'The timeframe the exception is in force.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CoveragePaymentBy(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Link to the paying party and optionally what specifically they will be responsible to pay."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    party: 'Reference' = Field(default=...)
    'The list of parties providing non-insurance payment for the treatment costs.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    responsibility: Optional_['string'] = Field(default=None)
    ' Description of the financial responsibility.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CoverageClass(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """A suite of underwriter specific classifiers."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'The type of classification for which an insurer-specific class label or number and optional name is provided.  For example, type may be used to identify a class of coverage or employer group, policy, or plan.'
    name: Optional_['string'] = Field(default=None)
    'A short description for the class.'
    value: 'Identifier' = Field(default=...)
    'The alphanumeric identifier associated with the insurer issued label.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class Coverage(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Financial instrument which may be used to reimburse or pay for health care products and services. Includes both insurance and self-payment."""
    resourceType: Literal_['Coverage'] = Field(default=...)
    'Insurance or medical plan or a payment agreement'
    id: Optional_['str'] = Field(default=None)
    'The logical id of the resource, as used in the URL for the resource. Once assigned, this value never changes.'
    meta: Optional_['Meta'] = Field(default=None)
    'The metadata about the resource. This is content that is maintained by the infrastructure. Changes to the content might not always be associated with version changes to the resource.'
    text: Optional_['Narrative'] = Field(default=None)
    'A human-readable narrative that contains a summary of the resource and can be used to represent the content of the resource to a human. The narrative need not encode all the structured data, but is required to contain sufficient detail to make it "clinically safe" for a human to just read the narrative. Resource definitions may define what content should be represented in the narrative to ensure clinical safety.'
    kind: 'code' = Field(default=...)
    'The nature of the coverage be it insurance, or cash payment such as self-pay.'
    type: Optional_['CodeableConcept'] = Field(default=None)
    'The type of coverage: social program, medical plan, accident coverage (workers compensation, auto), group health or payment by an individual or organization.'
    class_: List_['CoverageClass'] = Field(default_factory=list, alias='class')
    'A suite of underwriter specific classifiers.'
    order: Optional_['positiveInt'] = Field(default=None)
    "The order of applicability of this coverage relative to other coverages which are currently in force. Note, there may be gaps in the numbering and this does not imply primary, secondary etc. as the specific positioning of coverages depends upon the episode of care. For example; a patient might have (0) auto insurance (1) their own health insurance and (2) spouse's health insurance. When claiming for treatments which were not the result of an auto accident then only coverages (1) and (2) above would be applicatble and would apply in the order specified in parenthesis."
    status: 'code' = Field(default=...)
    'The status of the resource instance.'
    period: Optional_['Period'] = Field(default=None)
    "Time period during which the coverage is in force. A missing start date indicates the start date isn't known, a missing end date means the coverage is continuing to be in force."
    insurer: Optional_['Reference'] = Field(default=None)
    'The program or plan underwriter, payor, insurance company.'
    network: Optional_['string'] = Field(default=None)
    "The insurer-specific identifier for the insurer-defined network of providers to which the beneficiary may seek treatment which will be covered at the 'in-network' rate, otherwise 'out of network' terms and conditions apply."
    language: Optional_['code'] = Field(default=None)
    'The base language in which the resource is written.'
    contract: List_['Reference'] = Field(default_factory=list)
    'The policy(s) which constitute this insurance coverage.'
    contained: List_['AnyResource'] = Field(default_factory=list)
    'These resources do not have an independent existence apart from the resource that contains them - they cannot be identified independently, nor can they have their own independent transaction scope. This is allowed to be a Parameters resource if and only if it is referenced by a resource that provides context/meaning.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the resource. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    paymentBy: List_['CoveragePaymentBy'] = Field(default_factory=list)
    'Link to the paying party and optionally what specifically they will be responsible to pay.'
    dependent: Optional_['string'] = Field(default=None)
    'A designator for a dependent under the coverage.'
    identifier: List_['Identifier'] = Field(default_factory=list)
    'The identifier of the coverage as issued by the insurer.'
    subscriber: Optional_['Reference'] = Field(default=None)
    "The party who has signed-up for or 'owns' the contractual relationship to the policy or to whom the benefit of the policy for services rendered to them or their family is due."
    beneficiary: 'Reference' = Field(default=...)
    'The party who benefits from the insurance coverage; the patient when products and/or services are provided.'
    subrogation: Optional_['boolean'] = Field(default=None)
    "When 'subrogation=true' this insurance instance has been included not for adjudication but to provide insurers with the details to recover costs."
    policyHolder: Optional_['Reference'] = Field(default=None)
    "The party who 'owns' the insurance policy."
    subscriberId: List_['Identifier'] = Field(default_factory=list)
    'The insurer assigned ID for the Subscriber.'
    relationship: Optional_['CodeableConcept'] = Field(default=None)
    'The relationship of beneficiary (patient) to the subscriber.'
    implicitRules: Optional_['uri'] = Field(default=None)
    'A reference to a set of rules that were followed when the resource was constructed, and which must be understood when processing the content. Often, this is a reference to an implementation guide that defines the special rules along with other profiles etc.'
    insurancePlan: Optional_['Reference'] = Field(default=None)
    'The insurance plan details, benefits and costs, which constitute this insurance coverage.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the resource and that modifies the understanding of the element that contains it and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer is allowed to define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    costToBeneficiary: List_['CoverageCostToBeneficiary'] = Field(default_factory=list)
    'A suite of codes indicating the cost category and associated amount which have been detailed in the policy and may have been  included on the health card.'


class CoverageEligibilityRequestSupportingInfo(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Additional information codes regarding exceptions, special considerations, the condition, situation, prior or concurrent issues."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    sequence: 'positiveInt' = Field(default=...)
    'A number to uniquely identify supporting information entries.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    information: 'Reference' = Field(default=...)
    'Additional data or information such as resources, documents, images etc. including references to the data or the actual inclusion of the data.'
    appliesToAll: Optional_['boolean'] = Field(default=None)
    'The supporting materials are applicable for all detail items, product/servce categories and specific billing codes.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."


class CoverageEligibilityRequestInsurance(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Financial instruments for reimbursement for the health care products and services."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    focal: Optional_['boolean'] = Field(default=None)
    'A flag to indicate that this Coverage is to be used for evaluation of this request when set to true.'
    coverage: 'Reference' = Field(default=...)
    "Reference to the insurance card level information contained in the Coverage resource. The coverage issuing insurer will use these details to locate the patient's actual coverage within the insurer's information system."
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension.'
    modifierExtension: List_['Extension'] = Field(default_factory=list)
    "May be used to represent additional information that is not part of the basic definition of the element and that modifies the understanding of the element in which it is contained and/or the understanding of the containing element's descendants. Usually modifier elements provide negation or qualification. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the definition of the extension. Applications processing a resource are required to check for modifier extensions.\n\nModifier extensions SHALL NOT change the meaning of any elements on Resource or DomainResource (including cannot change the meaning of modifierExtension itself)."
    businessArrangement: Optional_['string'] = Field(default=None)
    'A business agreement number established between the provider and the insurer for special business processing purposes.'


class CoverageEligibilityRequestEvent(BaseModel, extra=Extra.ignore, validate_assignment=True):
    """Information code for an event with a corresponding date or period."""
    id: Optional_['str'] = Field(default=None)
    'Unique id for the element within a resource (for internal references). This may be any string value that does not contain spaces.'
    type: 'CodeableConcept' = Field(default=...)
    'A coded event such as when a service is expected or a card printed.'
    whenDateTime: Optional_['dateTime'] = Field(default=None)
    'A date or period in the past or future indicating when the event occurred or is expectd to occur.'
    whenPeriod: Optional_['Period'] = Field(default=None)
    'A date or period in the past or future indicating when the event occurred or is expectd to occur.'
    extension: List_['Extension'] = Field(default_factory=list)
    'May be used to represent additional information that is not part of the basic definition of the element. To make the use of extensions safe and managable, there is a strict set of governance applied to the definition and use of extensions. Though any implementer can define an extension, there is a set of requirements that SHALL be met as part of the defi