import { aws_ec2 as ec2, aws_efs as efs, aws_events as events, aws_lambda as lambda, aws_s3 as s3, aws_route53 as r53, aws_sns as sns, Duration, RemovalPolicy } from 'aws-cdk-lib';
import { Construct } from 'constructs';
export declare enum CertificateStorageType {
    /**
     * Store the certificate in AWS Secrets Manager
     */
    SECRETS_MANAGER = "secretsmanager",
    /**
     * Store the certificates in S3
     */
    S3 = "s3",
    /**
     * Store the certificates as a parameter in AWS Systems Manager Parameter Store  with encryption
     */
    SSM_SECURE = "ssm_secure",
    /**
     * Store the certificates in EFS, mounted to the Lambda function
     */
    EFS = "efs"
}
export interface CertbotProps {
    /**
     * The comma delimited list of domains for which the Let's Encrypt certificate will be valid. Primary domain should be first.
     */
    readonly letsencryptDomains: string;
    /**
     * The email to associate with the Let's Encrypt certificate request.
     */
    readonly letsencryptEmail: string;
    /**
     * Any additional Lambda layers to use with the created function. For example Lambda Extensions
     */
    readonly layers?: lambda.ILayerVersion[];
    /**
     * Hosted zone names that will be required for DNS verification with certbot
     */
    readonly hostedZoneNames?: string[];
    /**
     * The hosted zones that will be required for DNS verification with certbot
     */
    readonly hostedZones?: r53.IHostedZone[];
    /**
     * The S3 bucket to place the resulting certificates in. If no bucket is given one will be created automatically.
     */
    readonly bucket?: s3.Bucket;
    /**
     * The prefix to apply to the final S3 key name for the certificates. Default is no prefix.
     * Also used for EFS.
     */
    readonly objectPrefix?: string;
    /**
     * The numbers of days left until the prior cert expires before issuing a new one.
     *
     * @default 30
     */
    readonly reIssueDays?: number;
    /**
     * Set the preferred certificate chain.
     *
     * @default 'None'
     */
    readonly preferredChain?: string;
    /**
     * The SNS topic to notify when a new cert is issued. If no topic is given one will be created automatically.
     */
    readonly snsTopic?: sns.Topic;
    /**
     * Whether or not to enable Lambda Insights
     *
     * @default false
     */
    readonly enableInsights?: boolean;
    /**
     * Insights layer ARN for your region. Defaults to layer for US-EAST-1
     */
    readonly insightsARN?: string;
    /**
     * The timeout duration for Lambda function
     *
     * @default Duraction.seconds(180)
     */
    readonly timeout?: Duration;
    /**
     * The schedule for the certificate check trigger.
     *
     * @default events.Schedule.cron({ minute: '0', hour: '0', weekDay: '1' })
     */
    readonly schedule?: events.Schedule;
    /**
     * Whether or not to schedule a trigger to run the function after each deployment
     *
     * @default true
     */
    readonly runOnDeploy?: boolean;
    /**
     * How many minutes to wait before running the post deployment Lambda trigger
     *
     * @default 10
     */
    readonly runOnDeployWaitMinutes?: number;
    /**
     * The description for the resulting Lambda function.
     */
    readonly functionDescription?: string;
    /**
     * The name of the resulting Lambda function.
     */
    readonly functionName?: string;
    /**
     * The removal policy for the S3 bucket that is automatically created.
     *
     * Has no effect if a bucket is given as a property
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Whether or not to enable automatic object deletion if the provided bucket is deleted.
     *
     * Has no effect if a bucket is given as a property
     *
     * @default false
     */
    readonly enableObjectDeletion?: boolean;
    /**
     * The method of storage for the resulting certificates.
     *
     * @default CertificateStorageType.S3
     */
    readonly certificateStorage?: CertificateStorageType;
    /**
     * The path to store the certificates in AWS Secrets Manager
     *
     * @default `/certbot/certificates/${letsencryptDomains.split(',')[0]}/`
     */
    readonly secretsManagerPath?: string;
    /**
     * The path to store the certificates in AWS Systems Manager Parameter Store
     *
     * @default `/certbot/certificates/${letsencryptDomains.split(',')[0]}/`
     */
    readonly ssmSecurePath?: string;
    /**
     * The KMS key to use for encryption of the certificates in Secrets Manager
     * or Systems Manager Parameter Store
     *
     * @default AWS managed key
     */
    readonly kmsKeyAlias?: string;
    /**
     * The EFS access point to store the certificates
     */
    readonly efsAccessPoint?: efs.AccessPoint;
    /**
     * The VPC to run the Lambda function in.
     * This is needed if you are using EFS.
     * It should be the same VPC as the EFS filesystem
     *
     * @default none
     */
    readonly vpc?: ec2.IVpc;
}
export declare class Certbot extends Construct {
    readonly handler: lambda.Function;
    constructor(scope: Construct, id: string, props: CertbotProps);
}
