"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Certbot = exports.CertificateStorageType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const oneTimeEvents = require("@renovosolutions/cdk-library-one-time-event");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const required_policies_1 = require("./required-policies");
const storage_helpers_1 = require("./storage-helpers");
var CertificateStorageType;
(function (CertificateStorageType) {
    /**
     * Store the certificate in AWS Secrets Manager
     */
    CertificateStorageType["SECRETS_MANAGER"] = "secretsmanager";
    /**
     * Store the certificates in S3
     */
    CertificateStorageType["S3"] = "s3";
    /**
     * Store the certificates as a parameter in AWS Systems Manager Parameter Store  with encryption
     */
    CertificateStorageType["SSM_SECURE"] = "ssm_secure";
    /**
     * Store the certificates in EFS, mounted to the Lambda function
     */
    CertificateStorageType["EFS"] = "efs";
})(CertificateStorageType = exports.CertificateStorageType || (exports.CertificateStorageType = {}));
class Certbot extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Set property defaults
        let layers = props.layers ?? [];
        let runOnDeploy = props.runOnDeploy ?? true;
        let functionDescription = props.functionDescription ?? 'Certbot Renewal Lambda for domain ' + props.letsencryptDomains.split(',')[0];
        let enableInsights = props.enableInsights ?? false;
        let insightsARN = props.insightsARN ?? 'arn:aws:lambda:' + aws_cdk_lib_1.Stack.of(this).region + ':580247275435:layer:LambdaInsightsExtension:14';
        if (props.hostedZoneNames === undefined && props.hostedZones === undefined) {
            throw new Error('You must provide either hostedZoneNames or hostedZones');
        }
        // Create an SNS topic if one is not provided and add the defined email to it
        let snsTopic = props.snsTopic ?? new aws_cdk_lib_1.aws_sns.Topic(this, 'topic');
        if (props.snsTopic === undefined) {
            snsTopic.addSubscription(new aws_cdk_lib_1.aws_sns_subscriptions.EmailSubscription(props.letsencryptEmail));
        }
        let hostedZones = [];
        if (props.hostedZoneNames != undefined) {
            props.hostedZoneNames.forEach((domainName) => {
                hostedZones.push(aws_cdk_lib_1.aws_route53.HostedZone.fromLookup(this, 'zone' + domainName, {
                    domainName,
                    privateZone: false,
                }).hostedZoneArn);
            });
        }
        if (props.hostedZones != undefined) {
            props.hostedZones.forEach((hostedZone) => {
                hostedZones.push(hostedZone.hostedZoneArn);
            });
        }
        const role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        required_policies_1.assignRequiredPoliciesToRole(this, {
            role,
            snsTopic,
            hostedZones,
        });
        const functionDir = path.join(__dirname, '../function/src');
        const bundlingCmds = [
            'mkdir -p /asset-output',
            'pip install -r /asset-input/requirements.txt -t /asset-output',
            'cp index.py /asset-output/index.py',
        ];
        // Create the Lambda function
        this.handler = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_10,
            role,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(functionDir, {
                bundling: {
                    image: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_10.bundlingImage,
                    command: [
                        'bash', '-c', bundlingCmds.join(' && '),
                    ],
                },
            }),
            handler: 'index.handler',
            functionName: props.functionName,
            description: functionDescription,
            environment: {
                LETSENCRYPT_DOMAINS: props.letsencryptDomains,
                LETSENCRYPT_EMAIL: props.letsencryptEmail,
                OBJECT_PREFIX: props.objectPrefix || '',
                REISSUE_DAYS: (props.reIssueDays === undefined) ? '30' : String(props.reIssueDays),
                PREFERRED_CHAIN: props.preferredChain || 'None',
                NOTIFICATION_SNS_ARN: snsTopic.topicArn,
                DRY_RUN: 'False',
            },
            layers,
            timeout: props.timeout || aws_cdk_lib_1.Duration.seconds(180),
            filesystem: props.efsAccessPoint ? aws_cdk_lib_1.aws_lambda.FileSystem.fromEfsAccessPoint(props.efsAccessPoint, '/mnt/efs') : undefined,
            vpc: props.vpc,
        });
        let bucket;
        if (props.bucket === undefined && (props.certificateStorage == CertificateStorageType.S3 || props.certificateStorage == undefined)) {
            bucket = new aws_cdk_lib_1.aws_s3.Bucket(this, 'bucket', {
                objectOwnership: aws_cdk_lib_1.aws_s3.ObjectOwnership.BUCKET_OWNER_PREFERRED,
                removalPolicy: props.removalPolicy || aws_cdk_lib_1.RemovalPolicy.RETAIN,
                autoDeleteObjects: props.enableObjectDeletion ?? false,
                versioned: true,
                lifecycleRules: [{
                        enabled: true,
                        abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1),
                    }],
                encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
                enforceSSL: true,
                blockPublicAccess: aws_cdk_lib_1.aws_s3.BlockPublicAccess.BLOCK_ALL,
            });
            bucket.grantReadWrite(this.handler);
            this.handler.addEnvironment('CERTIFICATE_BUCKET', bucket.bucketName);
            this.handler.addEnvironment('CERTIFICATE_STORAGE', 's3');
        }
        if (props.bucket && (props.certificateStorage == CertificateStorageType.S3 || props.certificateStorage == undefined)) {
            bucket = props.bucket;
            bucket.grantReadWrite(this.handler);
            this.handler.addEnvironment('CERTIFICATE_BUCKET', bucket.bucketName);
            this.handler.addEnvironment('CERTIFICATE_STORAGE', 's3');
        }
        if (props.certificateStorage == CertificateStorageType.SECRETS_MANAGER) {
            this.handler.addEnvironment('CERTIFICATE_STORAGE', 'secretsmanager');
            this.handler.addEnvironment('CERTIFICATE_SECRET_PATH', props.secretsManagerPath || `/certbot/certificates/${props.letsencryptDomains.split(',')[0]}/`);
            if (props.kmsKeyAlias) {
                this.handler.addEnvironment('CUSTOM_KMS_KEY_ID', props.kmsKeyAlias);
            }
            storage_helpers_1.configureSecretsManagerStorage(this, {
                role,
                secretsManagerPath: props.secretsManagerPath || `/certbot/certificates/${props.letsencryptDomains.split(',')[0]}/`,
                kmsKeyAlias: props.kmsKeyAlias,
            });
        }
        ;
        if (props.certificateStorage == CertificateStorageType.SSM_SECURE) {
            this.handler.addEnvironment('CERTIFICATE_STORAGE', 'ssm_secure');
            this.handler.addEnvironment('CERTIFICATE_PARAMETER_PATH', props.ssmSecurePath || `/certbot/certificates/${props.letsencryptDomains.split(',')[0]}/`);
            if (props.kmsKeyAlias) {
                this.handler.addEnvironment('CUSTOM_KMS_KEY_ID', props.kmsKeyAlias);
            }
            storage_helpers_1.configureSSMStorage(this, {
                role,
                parameterStorePath: props.ssmSecurePath || `/certbot/certificates/${props.letsencryptDomains.split(',')[0]}/`,
                kmsKeyAlias: props.kmsKeyAlias,
            });
        }
        if (props.certificateStorage == CertificateStorageType.EFS) {
            if (!props.efsAccessPoint) {
                throw new Error('You must provide an EFS Access Point to use EFS storage');
            }
            else {
                this.handler.addEnvironment('CERTIFICATE_STORAGE', 'efs');
                this.handler.addEnvironment('EFS_PATH', '/mnt/efs');
            }
        }
        if (props.vpc) {
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        if (enableInsights) {
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchLambdaInsightsExecutionRolePolicy'));
            this.handler.addLayers(aws_cdk_lib_1.aws_lambda.LayerVersion.fromLayerVersionArn(this, 'insightsLayer', insightsARN));
        }
        // Add function triggers
        new aws_cdk_lib_1.aws_events.Rule(this, 'trigger', {
            schedule: props.schedule || aws_cdk_lib_1.aws_events.Schedule.cron({ minute: '0', hour: '0', weekDay: '1' }),
            targets: [new aws_cdk_lib_1.aws_events_targets.LambdaFunction(this.handler)],
        });
        if (runOnDeploy) {
            new aws_cdk_lib_1.aws_events.Rule(this, 'triggerImmediate', {
                schedule: new oneTimeEvents.OnDeploy(this, 'schedule', {
                    offsetMinutes: props.runOnDeployWaitMinutes || 10,
                }).schedule,
                targets: [new aws_cdk_lib_1.aws_events_targets.LambdaFunction(this.handler)],
            });
        }
    }
}
exports.Certbot = Certbot;
_a = JSII_RTTI_SYMBOL_1;
Certbot[_a] = { fqn: "@renovosolutions/cdk-library-certbot.Certbot", version: "2.6.0" };
//# sourceMappingURL=data:application/json;base64,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