"""Construction of Conan data"""

import shutil
from pathlib import Path

from pydantic import DirectoryPath

from cppython.plugins.conan.schema import ConanDependency, ConanfileGenerationData


class Builder:
    """Aids in building the information needed for the Conan plugin"""

    def __init__(self) -> None:
        """Initialize the builder"""
        self._filename = 'conanfile.py'

    @staticmethod
    def _create_base_conanfile(
        base_file: Path,
        dependencies: list[ConanDependency],
        dependency_groups: dict[str, list[ConanDependency]],
        cmake_binary: Path | None = None,
    ) -> None:
        """Creates a conanfile_base.py with CPPython managed dependencies.

        Args:
            base_file: Path to write the base conanfile
            dependencies: List of main dependencies
            dependency_groups: Dictionary of dependency groups (e.g., 'test')
            cmake_binary: Optional path to CMake binary to use
        """
        test_dependencies = dependency_groups.get('test', [])

        # Generate requirements method content
        requires_lines = []
        for dep in dependencies:
            requires_lines.append(f'        self.requires("{dep.requires()}")')
        requires_content = '\n'.join(requires_lines) if requires_lines else '        pass  # No requirements'

        # Generate build_requirements method content
        test_requires_lines = []
        for dep in test_dependencies:
            test_requires_lines.append(f'        self.test_requires("{dep.requires()}")')
        test_requires_content = (
            '\n'.join(test_requires_lines) if test_requires_lines else '        pass  # No test requirements'
        )

        # Generate configure method content for cmake_program if specified
        if cmake_binary:
            # Use forward slashes for cross-platform compatibility in Conan
            cmake_path_str = str(cmake_binary.resolve()).replace('\\', '/')
            configure_content = f'''    def configure(self):
        """CPPython managed configuration."""
        self.conf.define("tools.cmake:cmake_program", "{cmake_path_str}")'''
        else:
            configure_content = ''

        content = f'''"""CPPython managed base ConanFile.

This file is auto-generated by CPPython. Do not edit manually.
Dependencies are managed through pyproject.toml.
"""

from conan import ConanFile


class CPPythonBase(ConanFile):
    """Base ConanFile with CPPython managed dependencies."""
{configure_content}

    def requirements(self):
        """CPPython managed requirements."""
{requires_content}

    def build_requirements(self):
        """CPPython managed build and test requirements."""
{test_requires_content}
'''
        base_file.write_text(content, encoding='utf-8')

    @staticmethod
    def _create_conanfile(
        conan_file: Path,
        name: str,
        version: str,
    ) -> None:
        """Creates a conanfile.py file that inherits from CPPython base."""
        class_name = name.replace('-', '_').title().replace('_', '')
        content = f'''import os
from conan.tools.cmake import CMake, CMakeDeps, CMakeToolchain, cmake_layout
from conan.tools.files import copy

from conanfile_base import CPPythonBase


class {class_name}Package(CPPythonBase):
    """Conan recipe for {name}."""
    
    name = "{name}"
    version = "{version}"
    settings = "os", "compiler", "build_type", "arch"
    exports = "conanfile_base.py"

    def requirements(self):
        """Declare package dependencies.
        
        CPPython managed dependencies are inherited from CPPythonBase.
        Add your custom requirements here.
        """
        super().requirements()  # Get CPPython managed dependencies
        # Add your custom requirements here

    def build_requirements(self):
        """Declare build and test dependencies.
        
        CPPython managed test dependencies are inherited from CPPythonBase.
        Add your custom build requirements here.
        """
        super().build_requirements()  # Get CPPython managed test dependencies
        # Add your custom build requirements here

    def layout(self):
        cmake_layout(self)

    def generate(self):
        deps = CMakeDeps(self)
        deps.generate()
        tc = CMakeToolchain(self)
        tc.user_presets_path = None
        tc.generate()

    def build(self):
        cmake = CMake(self)
        cmake.configure()
        cmake.build()

    def package(self):
        cmake = CMake(self)
        cmake.install()

    def package_info(self):
        # Use native CMake config files to preserve FILE_SET information for C++ modules
        # This tells CMakeDeps to skip generating files and use the package's native config
        self.cpp_info.set_property("cmake_find_mode", "none")
        self.cpp_info.builddirs = ["."]

    def export_sources(self):
        copy(self, "CMakeLists.txt", src=self.recipe_folder, dst=self.export_sources_folder)
        copy(self, "src/*", src=self.recipe_folder, dst=self.export_sources_folder)
        copy(self, "cmake/*", src=self.recipe_folder, dst=self.export_sources_folder)
'''
        conan_file.write_text(content, encoding='utf-8')

    def generate_conanfile(
        self,
        directory: DirectoryPath,
        data: ConanfileGenerationData,
    ) -> None:
        """Generate conanfile.py and conanfile_base.py for the project.

        Always generates the base conanfile with managed dependencies.
        Only creates conanfile.py if it doesn't exist (never modifies existing files).

        Args:
            directory: The project directory
            data: Generation data containing dependencies, project info, and cmake binary path.
                If cmake_binary is not provided, attempts to find cmake in the current
                Python environment.
        """
        directory.mkdir(parents=True, exist_ok=True)

        # Resolve cmake binary path
        resolved_cmake: Path | None = None
        if data.cmake_binary and data.cmake_binary != 'cmake':
            resolved_cmake = Path(data.cmake_binary).resolve()
        else:
            # Try to find cmake in the current Python environment (venv)
            cmake_path = shutil.which('cmake')
            if cmake_path:
                resolved_cmake = Path(cmake_path).resolve()

        # Always regenerate the base conanfile with managed dependencies
        base_file = directory / 'conanfile_base.py'
        self._create_base_conanfile(base_file, data.dependencies, data.dependency_groups, resolved_cmake)

        # Only create conanfile.py if it doesn't exist
        conan_file = directory / self._filename
        if not conan_file.exists():
            self._create_conanfile(conan_file, data.name, data.version)
