import os
from dataclasses import dataclass

from utils import File, Log, TSVFile

from utils_future import Alt, LatLng

log = Log('PlacesInfo')


@dataclass
class PlaceInfo:
    name: str
    latlng: LatLng
    alt: Alt

    def __str__(self) -> str:
        return f'{self.name} ({self.latlng}, {self.alt})'

    @staticmethod
    def from_dict(d: dict) -> 'PlaceInfo':
        return PlaceInfo(
            name=d['name'],
            latlng=LatLng.from_str(d['latlng']),
            alt=Alt.from_str(d['alt']),
        )

    @staticmethod
    def list_from_file(file_path: str) -> list['PlaceInfo']:
        place_info_list = []
        data_list = TSVFile(file_path).read()
        for d in data_list:
            place_info = PlaceInfo.from_dict(d)
            place_info_list.append(place_info)
        log.debug(f'Loaded {len(place_info_list)} places from {file_path}.')
        return place_info_list

    @staticmethod
    def list_from_dir(dir_path: str) -> list['PlaceInfo']:
        place_info_list = []
        for file_name in os.listdir(dir_path):
            if not file_name.endswith('tsv'):
                continue

            place_info_list_for_file = PlaceInfo.list_from_file(
                os.path.join(dir_path, file_name)
            )
            place_info_list.extend(place_info_list_for_file)
        log.debug(f'Loaded {len(place_info_list)} places from {dir_path}.')
        return place_info_list

    @staticmethod
    def validate_for_duplicates(place_info_list: list['PlaceInfo']) -> bool:
        name_set = set()
        duplicate_name_list = []
        for place_info in place_info_list:
            name = place_info.name
            if name in name_set:
                log.error(f'Duplicate place: {place_info}')
                duplicate_name_list.append(name)
            else:
                name_set.add(name)
        if duplicate_name_list:
            log.error(f'Found {len(duplicate_name_list)} duplicate names.')
            return False
        log.info('No duplicate places found.')
        return True

    @staticmethod
    def _build_list():
        DIR_DATA = 'data'
        d_list = sorted(
            PlaceInfo.list_from_dir(DIR_DATA), key=lambda d: d.name
        )
        PlaceInfo.validate_for_duplicates(d_list)

        n = len(d_list)
        lines = [
            '# ⚠️ DO NOT EDIT' '# Auto-Generated by PlaceInfo.py',
            '',
            'from places_lk.core.PlaceInfo import PlaceInfo',
            'from utils_future import Alt, LatLng',
            '',
            'PLACE_INFO_LIST = [',
        ]
        for i, d in enumerate(d_list):
            lat, lng = d.latlng.tuple_normalized
            lines.extend(
                [
                    f'    # {i + 1:,} / {n:,}',
                    '    PlaceInfo(',
                    f'        name="{d.name}",',
                    f'        latlng=LatLng({lat}, {lng}),',
                    f'        alt=Alt({d.alt.alt_m_normalized}),',
                    '    ),',
                ]
            )
        lines.extend([']', ''])

        PLACE_INFO_LIST_PATH = os.path.join(
            'src', 'places_lk', 'core', 'PLACE_INFO_LIST.py'
        )
        File(PLACE_INFO_LIST_PATH).write_lines(lines)
        log.info(f'Wrote {n} places to {PLACE_INFO_LIST_PATH}.')
