import abc
import json
import logging
import pandas
from IPython.core.display_functions import clear_output
from IPython.core.getipython import get_ipython
from IPython.core.display import Javascript

from sagemaker_studio_dataengineering_sessions.sagemaker_base_session_manager.base_session_manager import BaseSessionManager
from sagemaker_studio_dataengineering_sessions.sagemaker_base_session_manager.common.constants import (Language, DATAZONE_STAGE)
from sagemaker_studio_dataengineering_sessions.sagemaker_base_session_manager.common.glue_gateway import GlueGateway
from sagemaker_studio_dataengineering_sessions.sagemaker_base_session_manager.common.sagemaker_toolkit_utils import SageMakerToolkitUtils
from sagemaker_studio_dataengineering_sessions.sagemaker_spark_session_manager.spark_commands.send_to_spark_command import send_dict_to_spark_command, \
    send_str_to_spark_command, send_pandas_df_to_spark_command, send_datazone_metadata_command
from sagemaker_studio_dataengineering_sessions.sagemaker_spark_session_manager.utils.common_utils import get_glue_endpoint, get_redshift_endpoint
from sagemaker_studio_dataengineering_sessions.sagemaker_spark_session_manager.utils.lib_utils import LibProvider
from sagemaker_studio_dataengineering_sessions.sagemaker_base_session_manager.common.sagemaker_connection_display import SageMakerConnectionDisplay
from sagemaker_studio_dataengineering_sessions.sagemaker_spark_session_manager.spark_session_manager.spark_connection import \
    SparkConnection
from IPython.display import display
import ipywidgets as widgets
from botocore.exceptions import ClientError

from sagemaker_studio_dataengineering_sessions.sagemaker_base_session_manager.common.constants import \
    CONNECTION_TYPE_SPARK_GLUE, CONNECTION_TYPE_SPARK_EMR_SERVERLESS

class SparkSession(BaseSessionManager, metaclass=abc.ABCMeta):
    lib_provider = LibProvider()
    logger = logging.getLogger(__name__)
    auto_add_catalogs = True

    def __init__(self, connection_name):
        super().__init__()
        self.connection_name = connection_name
        aws_location = self._get_connection_aws_location()
        self.region = aws_location["awsRegion"]
        self.account_id = aws_location["awsAccountId"]
        self.glue_endpoint = get_glue_endpoint(self.region, DATAZONE_STAGE)
        self.redshift_endpoint = get_redshift_endpoint(self.region, DATAZONE_STAGE)
        self.glue_client = GlueGateway()
        if DATAZONE_STAGE == "gamma":
            self.glue_client.initialize_clients(region=self.region, endpoint_url=self.glue_endpoint)
        else:
            # Use boto3 default endpoint if stage is not gamma
            self.glue_client.initialize_clients(region=self.region)
        self._gen_default_spark_configuration()

    def send_to_remote(self, local_var: str, remote_var: str, language=Language.python):
        try:
            local = get_ipython().ev(f"{local_var}")
            if type(local) is dict:
                command = send_dict_to_spark_command(local, remote_var, language)
            elif type(local) is str:
                command = send_str_to_spark_command(local, remote_var, language)
            elif type(local) is pandas.DataFrame:
                command = send_pandas_df_to_spark_command(local, remote_var, language)
            else:
                raise NotImplementedError(f"Local variable {type(local)} is not supported.")
            if not self.is_session_connectable():
                self.create_session()
            self.run_statement(command, language)
        except NameError:
            self.get_logger().error(f"local variable  does not exist.")
            raise RuntimeError(f"local variable {local_var} does not exist.")

    def send_datazone_metadata_to_remote(self, language=Language.python):
        if language == Language.python:
            # Only send metadata if language is python
            command = send_datazone_metadata_command(language)
            self.run_statement(command, language)

    def _configure_core(self, cell):
        raise NotImplementedError('Must define _configure_core to use this configure function.')

    def _get_connection_aws_location(self):
        connection_details = SageMakerToolkitUtils.get_connection_detail(self.connection_name, True)
        return connection_details["physicalEndpoints"][0]["awsLocation"]

    def _gen_default_spark_configuration(self):
        self.default_spark_configuration = {
            "conf": {
                "spark.sql.catalog.spark_catalog": "org.apache.iceberg.spark.SparkSessionCatalog",
                "spark.sql.catalog.spark_catalog.catalog-impl": "org.apache.iceberg.aws.glue.GlueCatalog",
                "spark.sql.catalog.spark_catalog.glue.id": self.account_id,
                "spark.sql.catalog.spark_catalog.glue.account-id": self.account_id,
                "spark.sql.catalog.spark_catalog.client.region": self.region,
                "spark.sql.catalog.spark_catalog.glue.endpoint": get_glue_endpoint(self.region),

                "spark.sql.extensions": "org.apache.iceberg.spark.extensions.IcebergSparkSessionExtensions",
                "spark.datasource.redshift.community.glue_endpoint": self.glue_endpoint,
                "spark.datasource.redshift.community.data_api_endpoint": self.redshift_endpoint,
                "spark.hadoop.fs.s3.impl": "com.amazon.ws.emr.hadoop.fs.EmrFileSystem"
            }
        }
        if self.auto_add_catalogs:
            self._gen_catalog_config()
        self.logger.info(f"update default configuration: {self.default_spark_configuration}")

    def _gen_catalog_config(self):
        try:
            catalogs = self.glue_client.get_catalogs()
            conf = self.default_spark_configuration['conf']
            for catalog in catalogs:
                if (catalog['CatalogType'] == "FEDERATED"
                        and catalog['FederatedCatalog']['ConnectionName'] != "aws:s3tables"):
                    pass
                else:
                    # Confirmed with glue team. If a catalog hierarchy looks like level_1 -> level_2 -> level_3 -> dev
                    # The ParentCatalogNames list of catalog dev would be
                    # index 0: level_1
                    # index 1: level_2
                    # index 2: level_3
                    catalog_name = "_".join(catalog['ParentCatalogNames'])
                    catalog_name = f"{catalog_name}_{catalog['Name']}" if catalog_name else catalog['Name']
                    conf[f"spark.sql.catalog.{catalog_name}"] = "org.apache.iceberg.spark.SparkCatalog"
                    conf[f"spark.sql.catalog.{catalog_name}.catalog-impl"]\
                        = "org.apache.iceberg.aws.glue.GlueCatalog"
                    conf[f"spark.sql.catalog.{catalog_name}.glue.id"] = f"{catalog['CatalogId']}"
                    conf[f"spark.sql.catalog.{catalog_name}.glue.catalog-arn"] = f"{catalog['ResourceArn']}"
                    conf[f"spark.sql.catalog.{catalog_name}.glue.endpoint"] = self.glue_endpoint
                    conf[f"spark.sql.catalog.{catalog_name}.client.region"] = self.region
        except ClientError as e:
            if e.response['Error']['Code'] == 'AccessDeniedException':
                SageMakerConnectionDisplay.send_error(
                    "Lakehouse catalog configurations could not be automatically added because your role does not have "
                    "the necessary permissions to call glue:getCatalogs. Please verify your permissions.")
            else:
                raise e

    def _set_auto_add_catalogs(self, val):
        self.auto_add_catalogs = False if val.casefold() == "false" else True
        # Regenerate spark_configurations if auto_add_catalogs is defined by customer
        self._gen_default_spark_configuration()

    def _update_spark_configuration_to_connection_default(self, connection: SparkConnection):
        if hasattr(connection, 'spark_configs') and isinstance(connection.spark_configs, dict):
            self.default_spark_configuration['conf'].update(connection.spark_configs)
        else:
            self.logger.warning(f"spark_configs not found in connection {connection}")

    def _if_lake_formation_error(self, error_message: str) -> bool:
        if "org.apache.spark.fgac.error" in error_message:
            return True
        else:
            return False

    def _lakeformation_session_level_setting_supported(self) -> bool:
        return False

    def get_lakeformation_config_suggestion(self):
        if not self._lakeformation_session_level_setting_supported():
            return None
        if self.connection_type == CONNECTION_TYPE_SPARK_GLUE:
            configure = "\"--enable-lakeformation-fine-grained-access\" : \"false\""
        elif self.connection_type == CONNECTION_TYPE_SPARK_EMR_SERVERLESS:
            configure = """"conf": {
        "spark.emr-serverless.lakeformation.enabled": "false"
    }"""
        else:
            return None

        content = f"""%%configure --name {self.connection_name} --f
{{
    {configure}
}}"""
        return content


    def display_spark_error(self, error_message: str) -> None:
        if self._if_lake_formation_error(error_message):
            # define output widgets for use with the button
            output = widgets.Output()
            output.layout.height = "0dp"
            output.add_class("lake-formation-error-helper")

            # define html widget to show hyperlink to public doc
            learn_more_message_widget = widgets.HTML(
                value="""<p>Learn more about <a target="_blank" rel="noopener noreferrer" href="https://docs.aws.amazon.com/sagemaker-unified-studio/latest/userguide/jupyterlab.html" style="color: #64B5F6">Jupyter compute and permission modes</a><svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M6.83333 4.5V5.66667H3.91667V12.0833H10.3333V9.16667H11.5V12.6667C11.5 12.9888 11.2388 13.25 10.9167 13.25H3.33333C3.01117 13.25 2.75 12.9888 2.75 12.6667V5.08333C2.75 4.76117 3.01117 4.5 3.33333 4.5H6.83333ZM13.25 2.75V7.41667H12.0833L12.0833 4.74092L7.53747 9.28747L6.71252 8.46252L11.2579 3.91667H8.58333V2.75H13.25Z" fill="#64B5F6"/>
</svg>.
</p>""")
            learn_more_message_widget.add_class("lake-formation-error-helper")

            # define Javascript that will copy to clipboard
            copy_js = Javascript(f"navigator.clipboard.writeText({json.dumps(self.get_lakeformation_config_suggestion())})")

            # define Javascript that will change the style
            # running this javascript will:
            # add "jp-RenderedText" class and update mimeType for the custom type "lake-formation-error-helper"
            # by adding that all the lake-formation-error-helper class item to show in error stype in jupyter
            show_in_error_js = Javascript(f"""var elements = document.getElementsByClassName("lake-formation-error-helper");
            for (var i = 0; i < elements.length; i++) {{
                elements[i].parentNode.dataset.mimeType = 'application/vnd.jupyter.stderr'
                elements[i].parentNode.classList.add("jp-RenderedText")
            }}""")

            # define copy command button
            def on_clicked(_: widgets.Button) -> None:
                output.layout.visibility = "hidden"
                output.layout.display = "none"
                with output:
                    display(copy_js)

            button = widgets.Button(description="Copy command")
            button.on_click(on_clicked)
            button.add_class("lake-formation-error-helper")

            # display helper
            SageMakerConnectionDisplay.send_error("The cell failed to run, verify your compute permissionMode is set to compatibility.\n")
            SageMakerConnectionDisplay.send_error("To change your compute permissionMode, follow these steps:\n")

            if self.get_lakeformation_config_suggestion():
                SageMakerConnectionDisplay.send_error("1. For the current cell, select a compute that is configured with permissionMode = compatibility")
                SageMakerConnectionDisplay.send_error("2. Re-run the cell\n")
                SageMakerConnectionDisplay.send_error("alternatively, you can:\n")
                SageMakerConnectionDisplay.send_error("1. Insert a new cell above")
                SageMakerConnectionDisplay.send_error("2. Copy and paste the %%configure magic command below")
                SageMakerConnectionDisplay.send_error("3. Insert command into the cell")
                SageMakerConnectionDisplay.send_error("4. Select the compute type of the cell to be \"Local Python\"")
                SageMakerConnectionDisplay.send_error("5. Run the new cell and any impacted cells\n")
                SageMakerConnectionDisplay.send_error(f"Command:\n\n{self.get_lakeformation_config_suggestion()}")
                display(button, output)
            else:
                SageMakerConnectionDisplay.send_error("1. For the current cell, select a compute that is configured with permissionMode = compatibility")
                SageMakerConnectionDisplay.send_error("2. Re-run the cell")

            display(learn_more_message_widget)
            display(show_in_error_js)

        SageMakerConnectionDisplay.send_error(error_message)



