"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineService = exports.COMPONENTS_NAMESPACE = exports.PIPELINE_COMPONENTS_NAMESPACE = exports.RUNTIME_IMAGES_NAMESPACE = exports.RUNTIMES_NAMESPACE = exports.KFP_SCHEMA = void 0;
const jsx_runtime_1 = require("react/jsx-runtime");
/*
 * Copyright 2018-2021 Elyra Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const services_1 = require("@elyra/services");
const ui_components_1 = require("@elyra/ui-components");
const apputils_1 = require("@jupyterlab/apputils");
const coreutils_1 = require("@jupyterlab/coreutils");
exports.KFP_SCHEMA = 'kfp';
exports.RUNTIMES_NAMESPACE = 'runtimes';
exports.RUNTIME_IMAGES_NAMESPACE = 'runtime-images';
exports.PIPELINE_COMPONENTS_NAMESPACE = 'component-registries';
exports.COMPONENTS_NAMESPACE = 'components';
var ContentType;
(function (ContentType) {
    ContentType["notebook"] = "execute-notebook-node";
    ContentType["python"] = "execute-python-node";
    ContentType["r"] = "execute-r-node";
    ContentType["other"] = "other";
})(ContentType || (ContentType = {}));
const CONTENT_TYPE_MAPPER = new Map([
    ['.py', ContentType.python],
    ['.ipynb', ContentType.notebook],
    ['.r', ContentType.r]
]);
class PipelineService {
    /**
     * Returns a list of external runtime configurations available as
     * `runtimes metadata`. This is used to submit the pipeline to be
     * executed on these runtimes.
     */
    static getRuntimes(showError = true, action) {
        return __awaiter(this, void 0, void 0, function* () {
            return services_1.MetadataService.getMetadata(exports.RUNTIMES_NAMESPACE).then(runtimes => {
                if (showError && Object.keys(runtimes).length === 0) {
                    return ui_components_1.RequestErrors.noMetadataError('runtime', action);
                }
                return runtimes;
            });
        });
    }
    /**
     * Submit the pipeline to be executed on an external runtime (e.g. Kbeflow Pipelines)
     *
     * @param pipeline
     * @param runtimeName
     */
    static getRuntimeComponents(runtimeName) {
        return __awaiter(this, void 0, void 0, function* () {
            return services_1.RequestHandler.makeGetRequest(`elyra/pipeline/components/${runtimeName}`);
        });
    }
    /**
     * Submit the pipeline to be executed on an external runtime (e.g. Kbeflow Pipelines)
     *
     * @param pipeline
     * @param runtimeName
     */
    static getComponentProperties(runtimeName, componentId) {
        return __awaiter(this, void 0, void 0, function* () {
            return services_1.RequestHandler.makeGetRequest(`elyra/pipeline/components/${runtimeName}/${componentId}/properties`);
        });
    }
    /**
     * Returns a list of runtime schema
     */
    static getRuntimesSchema(showError = true) {
        return __awaiter(this, void 0, void 0, function* () {
            return services_1.MetadataService.getSchema(exports.RUNTIMES_NAMESPACE).then(schema => {
                if (showError && Object.keys(schema).length === 0) {
                    return ui_components_1.RequestErrors.noMetadataError('schema');
                }
                return schema;
            });
        });
    }
    /**
     * Return a list of configured docker images that are used as runtimes environments
     * to run the pipeline nodes.
     */
    static getRuntimeImages() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                let runtimeImages = yield services_1.MetadataService.getMetadata('runtime-images');
                runtimeImages = runtimeImages.sort((a, b) => 0 - (a.name > b.name ? -1 : 1));
                if (Object.keys(runtimeImages).length === 0) {
                    return ui_components_1.RequestErrors.noMetadataError('runtime image');
                }
                const images = {};
                for (const image in runtimeImages) {
                    const imageName = runtimeImages[image]['metadata']['image_name'];
                    images[imageName] = runtimeImages[image]['display_name'];
                }
                return images;
            }
            catch (error) {
                Promise.reject(error);
            }
        });
    }
    static getDisplayName(name, metadataArr) {
        var _a;
        return (_a = metadataArr.find(r => r['name'] === name)) === null || _a === void 0 ? void 0 : _a['display_name'];
    }
    /**
     * The runtime name is currently based on the schema name (one schema per runtime)
     * @param name
     * @param metadataArr
     */
    static getRuntimeName(name, metadataArr) {
        var _a;
        return (_a = metadataArr.find(r => r['name'] === name)) === null || _a === void 0 ? void 0 : _a['schema_name'];
    }
    /**
     * Creates a Dialog for passing to makeServerRequest
     */
    static getWaitDialog(title = 'Making server request...', body = 'This may take some time') {
        return new apputils_1.Dialog({
            title: title,
            body: body,
            buttons: [apputils_1.Dialog.okButton()]
        });
    }
    /**
     * Submit the pipeline to be executed on an external runtime (e.g. Kbeflow Pipelines)
     *
     * @param pipeline
     * @param runtimeName
     */
    static submitPipeline(pipeline, runtimeName) {
        return __awaiter(this, void 0, void 0, function* () {
            return services_1.RequestHandler.makePostRequest('elyra/pipeline/schedule', JSON.stringify(pipeline), this.getWaitDialog('Packaging and submitting pipeline ...')).then(response => {
                let dialogTitle;
                let dialogBody;
                if (response['run_url']) {
                    // pipeline executed remotely in a runtime of choice
                    dialogTitle = 'Job submission to ' + runtimeName + ' succeeded';
                    dialogBody = (jsx_runtime_1.jsxs("p", { children: [response['platform'] == 'airflow' ? (jsx_runtime_1.jsxs("p", { children: ["Apache Airflow DAG has been pushed to the", ' ', jsx_runtime_1.jsx("a", Object.assign({ href: response['git_url'], target: "_blank", rel: "noopener noreferrer" }, { children: "GitHub Repository." }), void 0),
                                    jsx_runtime_1.jsx("br", {}, void 0)] }, void 0)) : null, "Check the status of your job at", ' ', jsx_runtime_1.jsx("a", Object.assign({ href: response['run_url'], target: "_blank", rel: "noopener noreferrer" }, { children: "Run Details." }), void 0),
                            jsx_runtime_1.jsx("br", {}, void 0), "The results and outputs are in the ", response['object_storage_path'], ' ', "working directory in", ' ', jsx_runtime_1.jsx("a", Object.assign({ href: response['object_storage_url'], target: "_blank", rel: "noopener noreferrer" }, { children: "object storage" }), void 0), "."] }, void 0));
                }
                else {
                    // pipeline executed in-place locally
                    dialogTitle = 'Job execution succeeded';
                    dialogBody = (jsx_runtime_1.jsx("p", { children: "Your job has been executed in-place in your local environment." }, void 0));
                }
                return apputils_1.showDialog({
                    title: dialogTitle,
                    body: dialogBody,
                    buttons: [apputils_1.Dialog.okButton()]
                });
            });
        });
    }
    /**
     * Export a pipeline to different formats (e.g. DSL, YAML, etc). These formats
     * are understood by a given runtime.
     *
     * @param pipeline
     * @param pipeline_export_format
     * @param pipeline_export_path
     * @param overwrite
     */
    static exportPipeline(pipeline, pipeline_export_format, pipeline_export_path, overwrite) {
        return __awaiter(this, void 0, void 0, function* () {
            console.log('Exporting pipeline to [' + pipeline_export_format + '] format');
            console.log('Overwriting existing file: ' + overwrite);
            const body = {
                pipeline: pipeline,
                export_format: pipeline_export_format,
                export_path: pipeline_export_path,
                overwrite: overwrite
            };
            return services_1.RequestHandler.makePostRequest('elyra/pipeline/export', JSON.stringify(body), this.getWaitDialog('Generating pipeline artifacts ...')).then(response => {
                return apputils_1.showDialog({
                    title: 'Pipeline export succeeded',
                    body: jsx_runtime_1.jsxs("p", { children: ["Exported file: ", response['export_path'], " "] }, void 0),
                    buttons: [apputils_1.Dialog.okButton()]
                });
            });
        });
    }
    static getNodeType(filepath) {
        const extension = coreutils_1.PathExt.extname(filepath);
        const type = CONTENT_TYPE_MAPPER.get(extension);
        // TODO: throw error when file extension is not supported?
        return type;
    }
    /**
     * Check if a given file is allowed to be added to the pipeline
     * @param item
     */
    static isSupportedNode(file) {
        if (PipelineService.getNodeType(file.path)) {
            return true;
        }
        else {
            return false;
        }
    }
    static getPipelineRelativeNodePath(pipelinePath, nodePath) {
        const relativePath = coreutils_1.PathExt.relative(coreutils_1.PathExt.dirname(pipelinePath), nodePath);
        return relativePath;
    }
    static getWorkspaceRelativeNodePath(pipelinePath, nodePath) {
        // since resolve returns an "absolute" path we need to strip off the leading '/'
        const workspacePath = coreutils_1.PathExt.resolve(coreutils_1.PathExt.dirname(pipelinePath), nodePath);
        return workspacePath;
    }
    static setNodePathsRelativeToPipeline(pipeline, pipelinePath) {
        for (const node of pipeline.nodes) {
            if (node.op === 'execute-notebook-node' ||
                node.op === 'execute-python-node' ||
                node.op === 'execute-r-node') {
                node.app_data.component_parameters.filename = this.getPipelineRelativeNodePath(pipelinePath, node.app_data.component_parameters.filename);
            }
        }
        return pipeline;
    }
    static setNodePathsRelativeToWorkspace(pipeline, pipelinePath) {
        for (const node of pipeline.nodes) {
            if (node.op === 'execute-notebook-node' ||
                node.op === 'execute-python-node' ||
                node.op === 'execute-r-node') {
                node.app_data.component_parameters.filename = this.getWorkspaceRelativeNodePath(pipelinePath, node.app_data.component_parameters.filename);
            }
        }
        return pipeline;
    }
}
exports.PipelineService = PipelineService;
/**
 * Returns a list of external runtime configurations
 * based on the runtimePlatform (Airflow or Kubeflow)
 */
PipelineService.filterRuntimes = (runtimes, runtimePlatform) => runtimes.filter(runtime => runtime.schema_name === runtimePlatform);
/**
 * Returns a list of external schema configurations
 * based a list of runtimes instances
 */
PipelineService.filterValidSchema = (runtimes, schema) => schema.filter(s => runtimes.some(runtime => runtime.schema_name === s.name));
/**
 * Sorts given list of runtimes by the display_name property
 */
PipelineService.sortRuntimesByDisplayName = (runtimes) => {
    runtimes.sort((r1, r2) => r1.display_name.localeCompare(r2.display_name));
};
//# sourceMappingURL=PipelineService.js.map