#cython: boundscheck=False
#cython: wraparound=False
#cython: cdivision=True
#cython: nonecheck=False
#cython: overflowcheck=False
#cython: embedsignature=True
#cython: infer_types=False
r"""
Quad4 - Quadrilateral element with mixed integration (:mod:`pyfe3d.quad4`)
==============================================================================

.. currentmodule:: pyfe3d.quad4

The :class:`.Quad4` is the recommended quadrilateral plane stress finite
element.

Another option is the :mod:`pyfe3d.quad4r` with full reduced
integration, more efficient, but with an hourglass control to compensate the
reduced integration that is not robust and creates significant artificial
stiffness.

The :class:`.Quad4` element has 6 degrees-of-freedom (DOF): `u`, `v`, `w`,
`r_x`, `r_y`, `r_z`. All DOF are interpolated bi-linearly between the nodes,
such that any of the DOF gradients can be constant over the element when the
element is rectangular.

The stiffness for the degrees of freedom `w`, `r_x` and `r_y` is based on the
paper below, where `r_x = \theta_1` and `r_y = \theta_2`:

    Hughes T.J.R., Taylor R.L., Kanoknukulchai W. "A simple and efficient
    finite element for plate bending". International Journal of  Numerical
    Methods in Engineering, Volume 11, 1977.
    https://doi.org/10.1002/nme.1620111005

Hughes et al. (1977) proposed the following integration scheme:

- For thin plates, when `h/\ell < 1`, where `\ell` is the element
  characteristic length, here calculated as the square root of the element area
  `\ell = \sqrt{\text{area}}`

-- two-by-two quadrature for the bending energy terms

-- one-point quadrature for the transverse shear energy terms

- For thick plates, when `h/\ell >= 1`

-- two-by-two quadrature for the bending energy terms

-- two-by-two quadrature for the transverse shear terms with gradients

-- one-point quadrature for the transverse shear terms without gradients

The in-plane stiffness terms are integrated with 2 quadrature points, and the
drilling stiffness is integrated with 1 quadrature point. These are
not specified in the paper of Hughes et al. (1977).


"""
#TODO bending stiffness vanishes when thickness -> zero, so a correction is applied:
#     maximum allowable aspect ratio for plate: 10^5/8, beyond which the shear
#     stiffness is multiplied by (thickness/h)^2 * (max aspect ratio allowed)^2,
#     where the max aspect ratio allowed for plates is 10^5/8.
from libc.math cimport fabs

import numpy as np

from .shellprop cimport ShellProp

cdef int DOF = 6
cdef int NUM_NODES = 4


#cdef int init_double(double* a, int size, double value) noexcept nogil:
    #cdef int i
    #for i in range(size):
        #a[i] = value


cdef class Quad4Data:
    r"""
    Used to allocate memory for the sparse matrices.

    Attributes
    ----------
    KC0_SPARSE_SIZE, : int
        ``KC0_SPARSE_SIZE = 576``

    KG_SPARSE_SIZE, : int
        ``KG_SPARSE_SIZE = 144``

    M_SPARSE_SIZE, : int
        ``M_SPARSE_SIZE = 480``

    KA_BETA_SPARSE_SIZE, : int
        ``KA_BETA_SPARSE_SIZE = 144``

    KA_GAMMA_SPARSE_SIZE, : int
        ``KA_GAMMA_SPARSE_SIZE = 144``

    CA_SPARSE_SIZE, : int
        ``CA_SPARSE_SIZE = 144``

    """
    cdef public int KC0_SPARSE_SIZE
    cdef public int KG_SPARSE_SIZE
    cdef public int M_SPARSE_SIZE
    cdef public int KA_BETA_SPARSE_SIZE
    cdef public int KA_GAMMA_SPARSE_SIZE
    cdef public int CA_SPARSE_SIZE
    def __cinit__(Quad4Data self):
        self.KC0_SPARSE_SIZE = 576
        self.KG_SPARSE_SIZE = 144
        self.M_SPARSE_SIZE = 480
        self.KA_BETA_SPARSE_SIZE = 144
        self.KA_GAMMA_SPARSE_SIZE = 144
        self.CA_SPARSE_SIZE = 144


cdef class Quad4Probe:
    r"""
    Probe used for local coordinates, local displacements, local stiffness,
    local stresses etc...

    The idea behind using a probe is to avoid allocating larger memory buffers
    per finite element. The memory buffers are allocated per probe, and one
    probe can be shared amongst many finite elements, with the information
    being updated and retrieved on demand.

    Attributes
    ----------
    xe, : array-like
        Array of size ``NUM_NODES*DOF//2=12`` containing the nodal coordinates
        in the element coordinate system, in the following order `{x_e}_1,
        {y_e}_1, {z_e}_1, `{x_e}_2, {y_e}_2, {z_e}_2`, `{x_e}_3, {y_e}_3,
        {z_e}_3`, `{x_e}_4, {y_e}_4, {z_e}_4`.
    ue, : array-like
        Array of size ``NUM_NODES*DOF=24`` containing the element displacements
        in the following order `{u_e}_1, {v_e}_1, {w_e}_1, {{r_x}_e}_1,
        {{r_y}_e}_1, {{r_z}_e}_1`, `{u_e}_2, {v_e}_2, {w_e}_2, {{r_x}_e}_2,
        {{r_y}_e}_2, {{r_z}_e}_2`, `{u_e}_3, {v_e}_3, {w_e}_3, {{r_x}_e}_3,
        {{r_y}_e}_3, {{r_z}_e}_3`, `{u_e}_4, {v_e}_4, {w_e}_4, {{r_x}_e}_4,
        {{r_y}_e}_4, {{r_z}_e}_4`.
    KC0ve, : array-like
        Local stiffness matrix stored as a 1D array of size
        ``(NUM_NODES*DOF)**2``.
    BLexx, BLeyy, BLgxy : array-like
        Arrays of size ``NUM_NODES*DOF=24`` containing the in-plane strain
        interpolation functions evaluated at a given natural coordinate point
        `\xi`, `\eta`.
    BLkxx, BLkyy, BLkxy : array-like
        Arrays of size ``NUM_NODES*DOF=24`` containing the bending strain
        interpolation functions evaluated at a given natural coordinate point
        `\xi`, `\eta`.
    BLgyz_grad, BLgyz_rot, BLgxz_grad, BLgxz_rot : array-like
        Arrays of size ``NUM_NODES*DOF=24`` containing the transverse shear
        strain interpolation functions evaluated at a given natural coordinate
        point `\xi`, `\eta`.

    """
    cdef public double [::1] xe
    cdef public double [::1] ue
    cdef public double [::1] KC0ve
    cdef public double [::1] BLexx
    cdef public double [::1] BLeyy
    cdef public double [::1] BLgxy
    cdef public double [::1] BLkxx
    cdef public double [::1] BLkyy
    cdef public double [::1] BLkxy
    cdef public double [::1] BLgyz_grad
    cdef public double [::1] BLgyz_rot
    cdef public double [::1] BLgxz_grad
    cdef public double [::1] BLgxz_rot

    def __cinit__(Quad4Probe self):
        self.xe = np.zeros(NUM_NODES*DOF//2, dtype=np.float64)
        self.ue = np.zeros(NUM_NODES*DOF, dtype=np.float64)
        self.KC0ve = np.zeros((NUM_NODES*DOF)**2, dtype=np.float64)

        self.BLexx = np.zeros(NUM_NODES*DOF, dtype=np.float64)
        self.BLeyy = np.zeros(NUM_NODES*DOF, dtype=np.float64)
        self.BLgxy = np.zeros(NUM_NODES*DOF, dtype=np.float64)

        self.BLkxx = np.zeros(NUM_NODES*DOF, dtype=np.float64)
        self.BLkyy = np.zeros(NUM_NODES*DOF, dtype=np.float64)
        self.BLkxy = np.zeros(NUM_NODES*DOF, dtype=np.float64)

        self.BLgyz_grad = np.zeros(NUM_NODES*DOF, dtype=np.float64)
        self.BLgyz_rot = np.zeros(NUM_NODES*DOF, dtype=np.float64)
        self.BLgxz_grad = np.zeros(NUM_NODES*DOF, dtype=np.float64)
        self.BLgxz_rot = np.zeros(NUM_NODES*DOF, dtype=np.float64)

    cpdef void update_BL(Quad4Probe self, double xi, double eta):
        r"""
        Update all components of the interpolation matrix `\pmb{B_L}` at a
        given natural coordinate point `\xi`, `\eta`.

        """
        cdef double x1, x2, x3, x4, y1, y2, y3, y4
        cdef double J11, J12, J21, J22
        cdef double j11, j12, j21, j22
        cdef double N1, N2, N3, N4
        cdef double N1x, N2x, N3x, N4x
        cdef double N1y, N2y, N3y, N4y

        x1 = self.xe[0]
        y1 = self.xe[1]
        # z1 = self.xe[2]
        x2 = self.xe[3]
        y2 = self.xe[4]
        # z2 = self.xe[5]
        x3 = self.xe[6]
        y3 = self.xe[7]
        # z3 = self.xe[8]
        x4 = self.xe[9]
        y4 = self.xe[10]
        # z4 = self.xe[11]

        J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
        J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
        J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
        J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

        j11 = J22/(J11*J22 - J12*J21)
        j12 = -J12/(J11*J22 - J12*J21)
        j21 = -J21/(J11*J22 - J12*J21)
        j22 = J11/(J11*J22 - J12*J21)

        N1 = eta*xi/4. - eta/4. - xi/4. + 1/4.
        N2 = -eta*xi/4. - eta/4. + xi/4. + 1/4.
        N3 = eta*xi/4. + eta/4. + xi/4. + 1/4.
        N4 = -eta*xi/4. + eta/4. - xi/4. + 1/4.

        N1x = 0.25*j11*(eta - 1) + 0.25*j12*(xi - 1)
        N2x = -0.25*eta*j11 + 0.25*j11 - 0.25*j12*xi - 0.25*j12
        N3x = 0.25*j11*(eta + 1) + 0.25*j12*(xi + 1)
        N4x = -0.25*eta*j11 - 0.25*j11 - 0.25*j12*xi + 0.25*j12

        N1y = 0.25*j21*(eta - 1) + 0.25*j22*(xi - 1)
        N2y = -0.25*eta*j21 + 0.25*j21 - 0.25*j22*xi - 0.25*j22
        N3y = 0.25*j21*(eta + 1) + 0.25*j22*(xi + 1)
        N4y = -0.25*eta*j21 - 0.25*j21 - 0.25*j22*xi + 0.25*j22

        self.BLexx[0] = N1x
        self.BLexx[6] = N2x
        self.BLexx[12] = N3x
        self.BLexx[18] = N4x

        self.BLeyy[1] = N1y
        self.BLeyy[7] = N2y
        self.BLeyy[13] = N3y
        self.BLeyy[19] = N4y

        self.BLgxy[0] = N1y
        self.BLgxy[6] = N2y
        self.BLgxy[12] = N3y
        self.BLgxy[18] = N4y
        self.BLgxy[1] = N1x
        self.BLgxy[7] = N2x
        self.BLgxy[13] = N3x
        self.BLgxy[19] = N4x

        self.BLkxx[4] = N1x
        self.BLkxx[10] = N2x
        self.BLkxx[16] = N3x
        self.BLkxx[22] = N4x

        self.BLkyy[3] = -N1y
        self.BLkyy[9] = -N2y
        self.BLkyy[15] = -N3y
        self.BLkyy[21] = -N4y

        self.BLkxy[3] = -N1x
        self.BLkxy[9] = -N2x
        self.BLkxy[15] = -N3x
        self.BLkxy[21] = -N4x
        self.BLkxy[4] = N1y
        self.BLkxy[10] = N2y
        self.BLkxy[16] = N3y
        self.BLkxy[22] = N4y

        self.BLgyz_grad[2] = N1y
        self.BLgyz_grad[8] = N2y
        self.BLgyz_grad[14] = N3y
        self.BLgyz_grad[20] = N4y

        self.BLgyz_rot[3] = -N1
        self.BLgyz_rot[9] = -N2
        self.BLgyz_rot[15] = -N3
        self.BLgyz_rot[21] = -N4

        self.BLgxz_grad[2] = N1x
        self.BLgxz_grad[8] = N2x
        self.BLgxz_grad[14] = N3x
        self.BLgxz_grad[20] = N4x

        self.BLgxz_rot[4] = N1
        self.BLgxz_rot[10] = N2
        self.BLgxz_rot[16] = N3
        self.BLgxz_rot[22] = N4


cdef class Quad4:
    r"""
    Nodal connectivity for the plate element similar to Nastran's CQUAD4::

         ^ y
         |
        4 ________ 3
         |       |
         |       |   --> x
         |       |
         |_______|
        1         2

    The element coordinate system is determined identically what is explained
    in Nastran's quick reference guide for the CQUAD4 element, as illustrated
    below.

    .. image:: ../figures/nastran_cquad4.svg

    Attributes
    ----------
    eid, : int
        Element identification number.
    area, : double
        Element area.
    K6ROT, : double
        Element drilling stiffness, added only to the diagonal of the local
        stiffness matrix. The default value is according to AUTODESK NASTRAN's
        quick reference guide is ``K6ROT = 100.`` for static analysis.  For
        modal solutions, this value should be ``K6ROT=1.e4``.  MSC NASTRAN's
        quick reference guide states that ``K6ROT > 100.`` should not be used,
        but this is controversial, already being controversial to what AUTODESK
        NASTRAN's manual says.
    r11, r12, r13, r21, r22, r23, r31, r32, r33 : double
        Rotation matrix from local to global coordinates.
    m11, m12, m21, m22 : double
        Rotation matrix only for the constitutive relations. Used when a
        material direction is used instead of the element local coordinates.
    c1, c2, c3, c4 : int
        Position of each node in the global stiffness matrix.
    n1, n2, n3, n4 : int
        Node identification number.
    init_k_KC0, init_k_KG, init_k_M : int
        Position in the arrays storing the sparse data for the structural
        matrices.
    init_k_KA_beta, init_k_KA_gamma, init_k_CA : int
        Position in the arrays storing the sparse data for the aerodynamic
        matrices based on the Piston theory.
    probe, : :class:`.Quad4Probe` object
        Pointer to the probe.

    """
    cdef public int eid
    cdef public int n1, n2, n3, n4
    cdef public int c1, c2, c3, c4
    cdef public int init_k_KC0, init_k_KG, init_k_M
    cdef public int init_k_KA_beta, init_k_KA_gamma, init_k_CA
    cdef public double area
    cdef public double K6ROT # drilling stiffness
    cdef public double r11, r12, r13, r21, r22, r23, r31, r32, r33
    cdef public double m11, m12, m21, m22
    cdef public Quad4Probe probe


    def __cinit__(Quad4 self, Quad4Probe p):
        self.probe = p
        self.eid = -1
        self.n1 = -1
        self.n2 = -1
        self.n3 = -1
        self.n4 = -1
        self.c1 = -1
        self.c2 = -1
        self.c3 = -1
        self.c4 = -1
        self.init_k_KC0 = 0
        # self.init_k_KCNL = 0
        self.init_k_KG = 0
        self.init_k_M = 0
        self.init_k_KA_beta = 0
        self.init_k_KA_gamma = 0
        self.init_k_CA = 0
        self.area = 0
        self.K6ROT = 10.
        self.r11 = self.r12 = self.r13 = 0.
        self.r21 = self.r22 = self.r23 = 0.
        self.r31 = self.r32 = self.r33 = 0.
        self.m11 = 1.
        self.m12 = 0.
        self.m21 = 0.
        self.m22 = 1.


    cpdef void update_rotation_matrix(Quad4 self, double [::1] x,
            double xmati=0., double xmatj=0., double xmatk=0.):
        r"""Update the rotation matrix of the element

        Attributes ``r11,r12,r13,r21,r22,r23,r31,r32,r33`` are updated,
        corresponding to the rotation matrix from local to global coordinates.

        The element coordinate system is determined, identifying the `ijk`
        components of each axis: `{x_e}_i, {x_e}_j, {x_e}_k`; `{y_e}_i,
        {y_e}_j, {y_e}_k`; `{z_e}_i, {z_e}_j, {z_e}_k`.

        The rotation matrix terms are calculated after solving 9 equations.

        Parameters
        ----------
        x : array-like
            Array with global nodal coordinates, for a total of `M` nodes in
            the model, this array will be arranged as: `x_1, y_1, z_1, x_2,
            y_2, z_2, ..., x_M, y_M, z_M`.

        xmati, xmatj, xmatk: array-like
            Vector in global coordinates representing the material direction.
            This vector is projected onto the plate element, thus becoming the
            material direction. The `ABD` matrix defining the constitutive
            behavior of the element is rotated from the material direction to
            the element `x` axis while calculating the stiffness matrices.

        """
        cdef double xi, xj, xk, yi, yj, yk, zi, zj, zk
        cdef double x1i, x1j, x1k, x2i, x2j, x2k, x3i, x3j, x3k, x4i, x4j, x4k
        cdef double v13i, v13j, v13k, v42i, v42j, v42k
        cdef double tmp, xmatnorm, ymati, ymatj, ymatk
        cdef double tol

        with nogil:
            x1i = x[self.c1//2 + 0]
            x1j = x[self.c1//2 + 1]
            x1k = x[self.c1//2 + 2]
            x2i = x[self.c2//2 + 0]
            x2j = x[self.c2//2 + 1]
            x2k = x[self.c2//2 + 2]
            x3i = x[self.c3//2 + 0]
            x3j = x[self.c3//2 + 1]
            x3k = x[self.c3//2 + 2]
            x4i = x[self.c4//2 + 0]
            x4j = x[self.c4//2 + 1]
            x4k = x[self.c4//2 + 2]

            v13i = x3i - x1i
            v13j = x3j - x1j
            v13k = x3k - x1k
            v42i = x2i - x4i
            v42j = x2j - x4j
            v42k = x2k - x4k

            zi = v42j*v13k - v42k*v13j
            zj = -v42i*v13k + v42k*v13i
            zk = v42i*v13j - v42j*v13i
            tmp = (zi**2 + zj**2 + zk**2)**0.5
            zi /= tmp
            zj /= tmp
            zk /= tmp
            # NOTE defining tolerance to be 1/1e10 of normal vector norm
            tol = tmp/1e10

            xi = (v13i + v42i)/2.
            xj = (v13j + v42j)/2.
            xk = (v13k + v42k)/2.
            tmp = (xi**2 + xj**2 + xk**2)**0.5
            xi /= tmp
            xj /= tmp
            xk /= tmp

            # y = z X x
            yi = zj*xk - zk*xj
            yj = zk*xi - zi*xk
            yk = zi*xj - zj*xi
            tmp = (yi**2 + yj**2 + yk**2)**0.5
            yi /= tmp
            yj /= tmp
            yk /= tmp

            self.r11 = xi
            self.r21 = xj
            self.r31 = xk
            self.r12 = yi
            self.r22 = yj
            self.r32 = yk
            self.r13 = zi
            self.r23 = zj
            self.r33 = zk

            xmatnorm = (xmati**2 + xmatj**2 + xmatk**2)**0.5
            xmati /= xmatnorm
            xmatj /= xmatnorm
            xmatk /= xmatnorm

            if xmatnorm > tol:
                # Project X Material Vector into Element CSYS
                # ymat = z X xmat
                ymati = zj*xmatk - zk*xmatj
                ymatj = zk*xmati - zi*xmatk
                ymatk = zi*xmatj - zj*xmati
                tmp = (ymati**2 + ymatj**2 + ymatk**2)**0.5
                ymati /= tmp
                ymatj /= tmp
                ymatk /= tmp
                if tmp > tol:
                    # NOTE ovewriting xmati,xmatj,xmatk, they now represent the projected xmat axis
                    # xmat_projected = ymat X z
                    xmati = ymatj*zk - ymatk*zj
                    xmatj = ymatk*zi - ymati*zk
                    xmatk = ymati*zj - ymatj*zi
                    tmp = (xmati**2 + xmatj**2 + xmatk**2)**0.5
                    xmati /= tmp
                    xmatj /= tmp
                    xmatk /= tmp

                    # NOTE angle between xmat_projected and xelem
                    # NOTE assuming they are already normalized (no need to normalize)
                    self.m11 = xmati*xi + xmatj*xj + xmatk*xk # costheta
                    self.m22 = self.m11
                    # NOTE sign of costheta
                    #     - the sign is positive when rotating from the material
                    #       to the element coordinate
                    #     - the sign only affects sintheta
                    # xmat dot_product y
                    if (xmati*yi + xmatj*yj + xmatk*yk) > 0:
                        self.m12 = -(1 - self.m11**2)**0.5 # sintheta
                        self.m21 = -self.m12
                    else:
                        # NOTE theta is negative
                        self.m12 = (1 - self.m11**2)**0.5 # sintheta
                        self.m21 = -self.m12


    cpdef void update_probe_ue(Quad4 self, double [::1] u):
        r"""Update the local displacement vector of the probe of the element

        .. note:: The ``probe`` attribute object :class:`.Quad4Probe` is
                  updated, not the element object.

        Parameters
        ----------
        u : array-like
            Array with global displacements, for a total of `M` nodes in
            the model, this array will be arranged as: `u_1, v_1, w_1, {r_x}_1,
            {r_y}_1, {r_z}_1, u_2, v_2, w_2, {r_x}_2, {r_y}_2, {r_z}_2, ...,
            u_M, v_M, w_M, {r_x}_M, {r_y}_M, {r_z}_M`.

        """
        cdef int i, j
        cdef int c[4]
        cdef double s1[3]
        cdef double s2[3]
        cdef double s3[3]

        with nogil:
            # positions in the global stiffness matrix
            c[0] = self.c1
            c[1] = self.c2
            c[2] = self.c3
            c[3] = self.c4

            # global to local transformation of displacements (R.T)
            s1[0] = self.r11
            s1[1] = self.r21
            s1[2] = self.r31
            s2[0] = self.r12
            s2[1] = self.r22
            s2[2] = self.r32
            s3[0] = self.r13
            s3[1] = self.r23
            s3[2] = self.r33

            for j in range(NUM_NODES):
                for i in range(DOF):
                    self.probe.ue[j*DOF + i] = 0

            for j in range(NUM_NODES):
                for i in range(DOF//2):
                    # transforming translations
                    self.probe.ue[j*DOF + 0] += s1[i]*u[c[j] + 0 + i]
                    self.probe.ue[j*DOF + 1] += s2[i]*u[c[j] + 0 + i]
                    self.probe.ue[j*DOF + 2] += s3[i]*u[c[j] + 0 + i]
                    # transforming rotations
                    self.probe.ue[j*DOF + 3] += s1[i]*u[c[j] + 3 + i]
                    self.probe.ue[j*DOF + 4] += s2[i]*u[c[j] + 3 + i]
                    self.probe.ue[j*DOF + 5] += s3[i]*u[c[j] + 3 + i]


    cpdef void update_probe_xe(Quad4 self, double [::1] x):
        r"""Update the 3D coordinates of the probe of the element

        .. note:: The ``probe`` attribute object :class:`.Quad4Probe` is
                  updated, not the element object.

        Parameters
        ----------
        x : array-like
            Array with global nodal coordinates, for a total of `M` nodes in
            the model, this array will be arranged as: `x_1, y_1, z_1, x_2,
            y_2, z_2, ..., x_M, y_M, z_M`.

        """
        cdef int i, j
        cdef int c[4]
        cdef double s1[3]
        cdef double s2[3]
        cdef double s3[3]

        with nogil:
            # positions in the global stiffness matrix
            c[0] = self.c1
            c[1] = self.c2
            c[2] = self.c3
            c[3] = self.c4

            # global to local transformation of displacements (R.T)
            s1[0] = self.r11
            s1[1] = self.r21
            s1[2] = self.r31
            s2[0] = self.r12
            s2[1] = self.r22
            s2[2] = self.r32
            s3[0] = self.r13
            s3[1] = self.r23
            s3[2] = self.r33

            for j in range(NUM_NODES):
                for i in range(DOF//2):
                    self.probe.xe[j*DOF//2 + i] = 0

            for j in range(NUM_NODES):
                for i in range(DOF//2):
                    self.probe.xe[j*DOF//2 + 0] += s1[i]*x[c[j]//2 + i]
                    self.probe.xe[j*DOF//2 + 1] += s2[i]*x[c[j]//2 + i]
                    self.probe.xe[j*DOF//2 + 2] += s3[i]*x[c[j]//2 + i]

        self.update_area()


    cpdef void update_area(Quad4 self):
        r"""Update element area

        """
        cdef double x1, x2, x3, x4, y1, y2, y3, y4
        with nogil:
            # NOTE ignoring z in local coordinates
            x1 = self.probe.xe[0]
            y1 = self.probe.xe[1]
            # z1 = self.probe.xe[2]
            x2 = self.probe.xe[3]
            y2 = self.probe.xe[4]
            # z2 = self.probe.xe[5]
            x3 = self.probe.xe[6]
            y3 = self.probe.xe[7]
            # z3 = self.probe.xe[8]
            x4 = self.probe.xe[9]
            y4 = self.probe.xe[10]
            # z4 = self.probe.xe[11]
            self.area = 1/2.*fabs((x1*y2 + x2*y3 + x3*y4 + x4*y1) - (x2*y1 + x3*y2 + x4*y3 + x1*y4))


    cpdef void update_KC0(Quad4 self,
                          long [::1] KC0r,
                          long [::1] KC0c,
                          double [::1] KC0v,
                          ShellProp prop,
                          int update_KC0v_only=0,
                          ):
        r"""Update sparse vectors for linear constitutive stiffness matrix KC0


        Drilling stiffness is used according to Adam et al. 2013:

            Adam, F. M., Mohamed, A. E., and Hassaballa, A. E., 2013,
            “Degenerated Four Nodes Shell Element with Drilling Degree of
            Freedom,” IOSR J. Eng., 3(8), pp. 10–20.


        Parameters
        ----------
        KC0r : np.array
            Array to store row positions of sparse values
        KC0c : np.array
            Array to store column positions of sparse values
        KC0v : np.array
            Array to store sparse values
        prop : :class:`.ShellProp` object
            Shell property object from where the stiffness and mass attributes
            are read from.
        update_KC0v_only : int
            The default ``0`` means that the row and column indices ``KC0r``
            and ``KC0c`` should also be updated. Any other value will only
            update the stiffness matrix values ``KC0v``.

        """
        cdef int i, j, k, ke
        cdef int node_i, node_j, m, n
        cdef int c[4]
        cdef double x1, x2, x3, x4, y1, y2, y3, y4
        # NOTE ABD in the material direction
        cdef double A11mat, A12mat, A16mat, A22mat, A26mat, A66mat
        cdef double B11mat, B12mat, B16mat, B22mat, B26mat, B66mat
        cdef double D11mat, D12mat, D16mat, D22mat, D26mat, D66mat
        cdef double E44, E45, E55
        # NOTE ABD in the element direction
        cdef double A11, A12, A16, A22, A26, A66
        cdef double B11, B12, B16, B22, B26, B66
        cdef double D11, D12, D16, D22, D26, D66
        cdef double K6ROT
        cdef double h, length
        cdef double r[6][6]
        cdef double m11, m12, m21, m22
        cdef double N1, N2, N3, N4
        cdef double N1x, N2x, N3x, N4x
        cdef double N1y, N2y, N3y, N4y
        cdef double xi, eta, wij, J11, J12, J21, J22, detJ
        cdef double j11, j12, j21, j22
        cdef double points[2]
        cdef int pti, ptj
        cdef double* BLexx
        cdef double* BLeyy
        cdef double* BLgxy
        cdef double* BLkxx
        cdef double* BLkyy
        cdef double* BLkxy
        cdef double* BLgyz_rot
        cdef double* BLgyz_grad
        cdef double* BLgxz_rot
        cdef double* BLgxz_grad
        cdef double exx, eyy, gxy, kxx, kyy, kxy
        cdef double gyz_rot, gxz_rot, gyz_grad, gxz_grad

        with nogil:
            BLexx = &self.probe.BLexx[0]
            BLeyy = &self.probe.BLeyy[0]
            BLgxy = &self.probe.BLgxy[0]
            BLkxx = &self.probe.BLkxx[0]
            BLkyy = &self.probe.BLkyy[0]
            BLkxy = &self.probe.BLkxy[0]
            BLgyz_rot = &self.probe.BLgyz_rot[0]
            BLgyz_grad = &self.probe.BLgyz_grad[0]
            BLgxz_rot = &self.probe.BLgxz_rot[0]
            BLgxz_grad = &self.probe.BLgxz_grad[0]

            A11mat = prop.A11
            A12mat = prop.A12
            A16mat = prop.A16
            A22mat = prop.A22
            A26mat = prop.A26
            A66mat = prop.A66
            B11mat = prop.B11
            B12mat = prop.B12
            B16mat = prop.B16
            B22mat = prop.B22
            B26mat = prop.B26
            B66mat = prop.B66
            D11mat = prop.D11
            D12mat = prop.D12
            D16mat = prop.D16
            D22mat = prop.D22
            D26mat = prop.D26
            D66mat = prop.D66

            # NOTE using self.m12 as a criterion to check if material
            #     coordinates were defined
            if self.m12 == 0:
                A11 = A11mat
                A12 = A12mat
                A16 = A16mat
                A22 = A22mat
                A26 = A26mat
                A66 = A66mat
                B11 = B11mat
                B12 = B12mat
                B16 = B16mat
                B22 = B22mat
                B26 = B26mat
                B66 = B66mat
                D11 = D11mat
                D12 = D12mat
                D16 = D16mat
                D22 = D22mat
                D26 = D26mat
                D66 = D66mat
            else:
                m11 = self.m11
                m12 = self.m12
                m21 = self.m21
                m22 = self.m22
                A11 = m11**2*(A11mat*m11**2 + A12mat*m12**2 + 2*A16mat*m11*m12) + 2*m11*m12*(A16mat*m11**2 + A26mat*m12**2 + 2*A66mat*m11*m12) + m12**2*(A12mat*m11**2 + A22mat*m12**2 + 2*A26mat*m11*m12)
                A12 = m21**2*(A11mat*m11**2 + A12mat*m12**2 + 2*A16mat*m11*m12) + 2*m21*m22*(A16mat*m11**2 + A26mat*m12**2 + 2*A66mat*m11*m12) + m22**2*(A12mat*m11**2 + A22mat*m12**2 + 2*A26mat*m11*m12)
                A16 = m11*m21*(A11mat*m11**2 + A12mat*m12**2 + 2*A16mat*m11*m12) + m12*m22*(A12mat*m11**2 + A22mat*m12**2 + 2*A26mat*m11*m12) + (m11*m22 + m12*m21)*(A16mat*m11**2 + A26mat*m12**2 + 2*A66mat*m11*m12)
                # A21 = m11**2*(A11mat*m21**2 + A12mat*m22**2 + 2*A16mat*m21*m22) + 2*m11*m12*(A16mat*m21**2 + A26mat*m22**2 + 2*A66mat*m21*m22) + m12**2*(A12mat*m21**2 + A22mat*m22**2 + 2*A26mat*m21*m22)
                A22 = m21**2*(A11mat*m21**2 + A12mat*m22**2 + 2*A16mat*m21*m22) + 2*m21*m22*(A16mat*m21**2 + A26mat*m22**2 + 2*A66mat*m21*m22) + m22**2*(A12mat*m21**2 + A22mat*m22**2 + 2*A26mat*m21*m22)
                A26 = m11*m21*(A11mat*m21**2 + A12mat*m22**2 + 2*A16mat*m21*m22) + m12*m22*(A12mat*m21**2 + A22mat*m22**2 + 2*A26mat*m21*m22) + (m11*m22 + m12*m21)*(A16mat*m21**2 + A26mat*m22**2 + 2*A66mat*m21*m22)
                # A61 = m11**2*(A11mat*m11*m21 + A12mat*m12*m22 + A16mat*(m11*m22 + m12*m21)) + 2*m11*m12*(A16mat*m11*m21 + A26mat*m12*m22 + A66mat*(m11*m22 + m12*m21)) + m12**2*(A12mat*m11*m21 + A22mat*m12*m22 + A26mat*(m11*m22 + m12*m21))
                # A62 = m21**2*(A11mat*m11*m21 + A12mat*m12*m22 + A16mat*(m11*m22 + m12*m21)) + 2*m21*m22*(A16mat*m11*m21 + A26mat*m12*m22 + A66mat*(m11*m22 + m12*m21)) + m22**2*(A12mat*m11*m21 + A22mat*m12*m22 + A26mat*(m11*m22 + m12*m21))
                A66 = m11*m21*(A11mat*m11*m21 + A12mat*m12*m22 + A16mat*(m11*m22 + m12*m21)) + m12*m22*(A12mat*m11*m21 + A22mat*m12*m22 + A26mat*(m11*m22 + m12*m21)) + (m11*m22 + m12*m21)*(A16mat*m11*m21 + A26mat*m12*m22 + A66mat*(m11*m22 + m12*m21))

                B11 = m11**2*(B11mat*m11**2 + B12mat*m12**2 + 2*B16mat*m11*m12) + 2*m11*m12*(B16mat*m11**2 + B26mat*m12**2 + 2*B66mat*m11*m12) + m12**2*(B12mat*m11**2 + B22mat*m12**2 + 2*B26mat*m11*m12)
                B12 = m21**2*(B11mat*m11**2 + B12mat*m12**2 + 2*B16mat*m11*m12) + 2*m21*m22*(B16mat*m11**2 + B26mat*m12**2 + 2*B66mat*m11*m12) + m22**2*(B12mat*m11**2 + B22mat*m12**2 + 2*B26mat*m11*m12)
                B16 = m11*m21*(B11mat*m11**2 + B12mat*m12**2 + 2*B16mat*m11*m12) + m12*m22*(B12mat*m11**2 + B22mat*m12**2 + 2*B26mat*m11*m12) + (m11*m22 + m12*m21)*(B16mat*m11**2 + B26mat*m12**2 + 2*B66mat*m11*m12)
                # B21 = m11**2*(B11mat*m21**2 + B12mat*m22**2 + 2*B16mat*m21*m22) + 2*m11*m12*(B16mat*m21**2 + B26mat*m22**2 + 2*B66mat*m21*m22) + m12**2*(B12mat*m21**2 + B22mat*m22**2 + 2*B26mat*m21*m22)
                B22 = m21**2*(B11mat*m21**2 + B12mat*m22**2 + 2*B16mat*m21*m22) + 2*m21*m22*(B16mat*m21**2 + B26mat*m22**2 + 2*B66mat*m21*m22) + m22**2*(B12mat*m21**2 + B22mat*m22**2 + 2*B26mat*m21*m22)
                B26 = m11*m21*(B11mat*m21**2 + B12mat*m22**2 + 2*B16mat*m21*m22) + m12*m22*(B12mat*m21**2 + B22mat*m22**2 + 2*B26mat*m21*m22) + (m11*m22 + m12*m21)*(B16mat*m21**2 + B26mat*m22**2 + 2*B66mat*m21*m22)
                # B61 = m11**2*(B11mat*m11*m21 + B12mat*m12*m22 + B16mat*(m11*m22 + m12*m21)) + 2*m11*m12*(B16mat*m11*m21 + B26mat*m12*m22 + B66mat*(m11*m22 + m12*m21)) + m12**2*(B12mat*m11*m21 + B22mat*m12*m22 + B26mat*(m11*m22 + m12*m21))
                # B62 = m21**2*(B11mat*m11*m21 + B12mat*m12*m22 + B16mat*(m11*m22 + m12*m21)) + 2*m21*m22*(B16mat*m11*m21 + B26mat*m12*m22 + B66mat*(m11*m22 + m12*m21)) + m22**2*(B12mat*m11*m21 + B22mat*m12*m22 + B26mat*(m11*m22 + m12*m21))
                B66 = m11*m21*(B11mat*m11*m21 + B12mat*m12*m22 + B16mat*(m11*m22 + m12*m21)) + m12*m22*(B12mat*m11*m21 + B22mat*m12*m22 + B26mat*(m11*m22 + m12*m21)) + (m11*m22 + m12*m21)*(B16mat*m11*m21 + B26mat*m12*m22 + B66mat*(m11*m22 + m12*m21))

                D11 = m11**2*(D11mat*m11**2 + D12mat*m12**2 + 2*D16mat*m11*m12) + 2*m11*m12*(D16mat*m11**2 + D26mat*m12**2 + 2*D66mat*m11*m12) + m12**2*(D12mat*m11**2 + D22mat*m12**2 + 2*D26mat*m11*m12)
                D12 = m21**2*(D11mat*m11**2 + D12mat*m12**2 + 2*D16mat*m11*m12) + 2*m21*m22*(D16mat*m11**2 + D26mat*m12**2 + 2*D66mat*m11*m12) + m22**2*(D12mat*m11**2 + D22mat*m12**2 + 2*D26mat*m11*m12)
                D16 = m11*m21*(D11mat*m11**2 + D12mat*m12**2 + 2*D16mat*m11*m12) + m12*m22*(D12mat*m11**2 + D22mat*m12**2 + 2*D26mat*m11*m12) + (m11*m22 + m12*m21)*(D16mat*m11**2 + D26mat*m12**2 + 2*D66mat*m11*m12)
                # D21 = m11**2*(D11mat*m21**2 + D12mat*m22**2 + 2*D16mat*m21*m22) + 2*m11*m12*(D16mat*m21**2 + D26mat*m22**2 + 2*D66mat*m21*m22) + m12**2*(D12mat*m21**2 + D22mat*m22**2 + 2*D26mat*m21*m22)
                D22 = m21**2*(D11mat*m21**2 + D12mat*m22**2 + 2*D16mat*m21*m22) + 2*m21*m22*(D16mat*m21**2 + D26mat*m22**2 + 2*D66mat*m21*m22) + m22**2*(D12mat*m21**2 + D22mat*m22**2 + 2*D26mat*m21*m22)
                D26 = m11*m21*(D11mat*m21**2 + D12mat*m22**2 + 2*D16mat*m21*m22) + m12*m22*(D12mat*m21**2 + D22mat*m22**2 + 2*D26mat*m21*m22) + (m11*m22 + m12*m21)*(D16mat*m21**2 + D26mat*m22**2 + 2*D66mat*m21*m22)
                # D61 = m11**2*(D11mat*m11*m21 + D12mat*m12*m22 + D16mat*(m11*m22 + m12*m21)) + 2*m11*m12*(D16mat*m11*m21 + D26mat*m12*m22 + D66mat*(m11*m22 + m12*m21)) + m12**2*(D12mat*m11*m21 + D22mat*m12*m22 + D26mat*(m11*m22 + m12*m21))
                # D62 = m21**2*(D11mat*m11*m21 + D12mat*m12*m22 + D16mat*(m11*m22 + m12*m21)) + 2*m21*m22*(D16mat*m11*m21 + D26mat*m12*m22 + D66mat*(m11*m22 + m12*m21)) + m22**2*(D12mat*m11*m21 + D22mat*m12*m22 + D26mat*(m11*m22 + m12*m21))
                D66 = m11*m21*(D11mat*m11*m21 + D12mat*m12*m22 + D16mat*(m11*m22 + m12*m21)) + m12*m22*(D12mat*m11*m21 + D22mat*m12*m22 + D26mat*(m11*m22 + m12*m21)) + (m11*m22 + m12*m21)*(D16mat*m11*m21 + D26mat*m12*m22 + D66mat*(m11*m22 + m12*m21))

            h = prop.h
            length = self.area**0.5

            E44 = prop.E44*prop.scf_k23
            E45 = prop.E45*0.5*(prop.scf_k13 + prop.scf_k23)
            E55 = prop.E55*prop.scf_k13

            # NOTE ignoring z in local coordinates
            x1 = self.probe.xe[0]
            y1 = self.probe.xe[1]
            # z1 = self.probe.xe[2]
            x2 = self.probe.xe[3]
            y2 = self.probe.xe[4]
            # z2 = self.probe.xe[5]
            x3 = self.probe.xe[6]
            y3 = self.probe.xe[7]
            # z3 = self.probe.xe[8]
            x4 = self.probe.xe[9]
            y4 = self.probe.xe[10]
            # z4 = self.probe.xe[11]

            # local to global transformation
            # translation DOFs
            r[0][0] = self.r11
            r[0][1] = self.r12
            r[0][2] = self.r13
            r[1][0] = self.r21
            r[1][1] = self.r22
            r[1][2] = self.r23
            r[2][0] = self.r31
            r[2][1] = self.r32
            r[2][2] = self.r33
            # rotation DOFs
            r[0+3][0+3] = self.r11
            r[0+3][1+3] = self.r12
            r[0+3][2+3] = self.r13
            r[1+3][0+3] = self.r21
            r[1+3][1+3] = self.r22
            r[1+3][2+3] = self.r23
            r[2+3][0+3] = self.r31
            r[2+3][1+3] = self.r32
            r[2+3][2+3] = self.r33
            # coupled translation-rotation DOFs
            r[0][0+3] = 0.
            r[0][1+3] = 0.
            r[0][2+3] = 0.
            r[1][0+3] = 0.
            r[1][1+3] = 0.
            r[1][2+3] = 0.
            r[2][0+3] = 0.
            r[2][1+3] = 0.
            r[2][2+3] = 0.
            # coupled translation-rotation DOFs
            r[0+3][0] = 0.
            r[0+3][1] = 0.
            r[0+3][2] = 0.
            r[1+3][0] = 0.
            r[1+3][1] = 0.
            r[1+3][2] = 0.
            r[2+3][0] = 0.
            r[2+3][1] = 0.
            r[2+3][2] = 0.

            K6ROT = self.K6ROT

            if update_KC0v_only == 0:
                # positions in the global stiffness matrix
                c[0] = self.c1
                c[1] = self.c2
                c[2] = self.c3
                c[3] = self.c4

                # initializing row and column indices
                #
                # TODO use r[3][3] instead
                for node_i in range(NUM_NODES):
                    for m in range(DOF):
                        for node_j in range(NUM_NODES):
                            for n in range(DOF):
                                k = self.init_k_KC0 + 24*(node_i*DOF + m) + node_j*DOF + n
                                KC0r[k] = c[node_i] + m
                                KC0c[k] = c[node_j] + n

            # zeroing probe KC0ve attribute
            for i in range(24):
                for j in range(24):
                    ke = 24*i + j
                    self.probe.KC0ve[ke] = 0.

            # NOTE full integration with two-point Gauss-Legendre quadrature
            wij = 1.
            points[0] = -0.5773502691896257645092
            points[1] = +0.5773502691896257645092

            for pti in range(2):
                xi = points[pti]
                for ptj in range(2):
                    eta = points[ptj]

                    J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
                    J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
                    J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
                    J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

                    detJ = J11*J22 - J12*J21

                    j11 = J22/(J11*J22 - J12*J21)
                    j12 = -J12/(J11*J22 - J12*J21)
                    j21 = -J21/(J11*J22 - J12*J21)
                    j22 = J11/(J11*J22 - J12*J21)

                    N1 = eta*xi/4. - eta/4. - xi/4. + 1/4.
                    N2 = -eta*xi/4. - eta/4. + xi/4. + 1/4.
                    N3 = eta*xi/4. + eta/4. + xi/4. + 1/4.
                    N4 = -eta*xi/4. + eta/4. - xi/4. + 1/4.

                    N1x = 0.25*j11*(eta - 1) + 0.25*j12*(xi - 1)
                    N2x = -0.25*eta*j11 + 0.25*j11 - 0.25*j12*xi - 0.25*j12
                    N3x = 0.25*j11*(eta + 1) + 0.25*j12*(xi + 1)
                    N4x = -0.25*eta*j11 - 0.25*j11 - 0.25*j12*xi + 0.25*j12

                    N1y = 0.25*j21*(eta - 1) + 0.25*j22*(xi - 1)
                    N2y = -0.25*eta*j21 + 0.25*j21 - 0.25*j22*xi - 0.25*j22
                    N3y = 0.25*j21*(eta + 1) + 0.25*j22*(xi + 1)
                    N4y = -0.25*eta*j21 - 0.25*j21 - 0.25*j22*xi + 0.25*j22

                    BLexx[0] = N1x
                    BLexx[6] = N2x
                    BLexx[12] = N3x
                    BLexx[18] = N4x

                    BLeyy[1] = N1y
                    BLeyy[7] = N2y
                    BLeyy[13] = N3y
                    BLeyy[19] = N4y

                    BLgxy[0] = N1y
                    BLgxy[6] = N2y
                    BLgxy[12] = N3y
                    BLgxy[18] = N4y
                    BLgxy[1] = N1x
                    BLgxy[7] = N2x
                    BLgxy[13] = N3x
                    BLgxy[19] = N4x

                    BLkxx[4] = N1x
                    BLkxx[10] = N2x
                    BLkxx[16] = N3x
                    BLkxx[22] = N4x

                    BLkyy[3] = -N1y
                    BLkyy[9] = -N2y
                    BLkyy[15] = -N3y
                    BLkyy[21] = -N4y

                    BLkxy[3] = -N1x
                    BLkxy[9] = -N2x
                    BLkxy[15] = -N3x
                    BLkxy[21] = -N4x
                    BLkxy[4] = N1y
                    BLkxy[10] = N2y
                    BLkxy[16] = N3y
                    BLkxy[22] = N4y

                    for i in range(24):
                        exx = BLexx[i]
                        eyy = BLeyy[i]
                        gxy = BLgxy[i]
                        kxx = BLkxx[i]
                        kyy = BLkyy[i]
                        kxy = BLkxy[i]
                        for j in range(24):
                            ke = 24*i + j
                            self.probe.KC0ve[ke] += wij*detJ*(
                            # membrane
                                exx*A11*BLexx[j] + exx*A12*BLeyy[j] + exx*A16*BLgxy[j]
                              + eyy*A12*BLexx[j] + eyy*A22*BLeyy[j] + eyy*A26*BLgxy[j]
                              + gxy*A16*BLexx[j] + gxy*A26*BLeyy[j] + gxy*A66*BLgxy[j]

                            # coupled membrane-bending
                              + exx*B11*BLkxx[j] + exx*B12*BLkyy[j] + exx*B16*BLkxy[j]
                              + eyy*B12*BLkxx[j] + eyy*B22*BLkyy[j] + eyy*B26*BLkxy[j]
                              + gxy*B16*BLkxx[j] + gxy*B26*BLkyy[j] + gxy*B66*BLkxy[j]

                              + kxx*B11*BLexx[j] + kxx*B12*BLeyy[j] + kxx*B16*BLgxy[j]
                              + kyy*B12*BLexx[j] + kyy*B22*BLeyy[j] + kyy*B26*BLgxy[j]
                              + kxy*B16*BLexx[j] + kxy*B26*BLeyy[j] + kxy*B66*BLgxy[j]

                            # bending
                              + kxx*D11*BLkxx[j] + kxx*D12*BLkyy[j] + kxx*D16*BLkxy[j]
                              + kyy*D12*BLkxx[j] + kyy*D22*BLkyy[j] + kyy*D26*BLkxy[j]
                              + kxy*D16*BLkxx[j] + kxy*D26*BLkyy[j] + kxy*D66*BLkxy[j]
                            )

                    if h/length >= 1.: # thick elements
                        BLgyz_grad[2] = N1y
                        BLgyz_grad[8] = N2y
                        BLgyz_grad[14] = N3y
                        BLgyz_grad[20] = N4y

                        BLgxz_grad[2] = N1x
                        BLgxz_grad[8] = N2x
                        BLgxz_grad[14] = N3x
                        BLgxz_grad[20] = N4x
                        for i in range(24):
                            gyz_grad = BLgyz_grad[i]
                            gxz_grad = BLgxz_grad[i]
                            for j in range(24):
                                ke = 24*i + j
                                self.probe.KC0ve[ke] += wij*detJ*(
                                # transverse shear (gradient term)
                                    gyz_grad*E44*BLgyz_grad[j] + gyz_grad*E45*BLgxz_grad[j]
                                  + gxz_grad*E45*BLgyz_grad[j] + gxz_grad*E55*BLgxz_grad[j]
                                )

            # NOTE reduced integration with one point at the center
            wij = 4.
            xi = 0.
            eta = 0.

            J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
            J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
            J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
            J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

            detJ = J11*J22 - J12*J21

            j11 = J22/(J11*J22 - J12*J21)
            j12 = -J12/(J11*J22 - J12*J21)
            j21 = -J21/(J11*J22 - J12*J21)
            j22 = J11/(J11*J22 - J12*J21)

            N1 = eta*xi/4. - eta/4. - xi/4. + 1/4.
            N2 = -eta*xi/4. - eta/4. + xi/4. + 1/4.
            N3 = eta*xi/4. + eta/4. + xi/4. + 1/4.
            N4 = -eta*xi/4. + eta/4. - xi/4. + 1/4.

            N1x = 0.25*j11*(eta - 1) + 0.25*j12*(xi - 1)
            N2x = -0.25*eta*j11 + 0.25*j11 - 0.25*j12*xi - 0.25*j12
            N3x = 0.25*j11*(eta + 1) + 0.25*j12*(xi + 1)
            N4x = -0.25*eta*j11 - 0.25*j11 - 0.25*j12*xi + 0.25*j12

            N1y = 0.25*j21*(eta - 1) + 0.25*j22*(xi - 1)
            N2y = -0.25*eta*j21 + 0.25*j21 - 0.25*j22*xi - 0.25*j22
            N3y = 0.25*j21*(eta + 1) + 0.25*j22*(xi + 1)
            N4y = -0.25*eta*j21 - 0.25*j21 - 0.25*j22*xi + 0.25*j22

            BLgyz_grad[2] = N1y
            BLgyz_grad[8] = N2y
            BLgyz_grad[14] = N3y
            BLgyz_grad[20] = N4y

            BLgyz_rot[3] = -N1
            BLgyz_rot[9] = -N2
            BLgyz_rot[15] = -N3
            BLgyz_rot[21] = -N4

            BLgxz_grad[2] = N1x
            BLgxz_grad[8] = N2x
            BLgxz_grad[14] = N3x
            BLgxz_grad[20] = N4x

            BLgxz_rot[4] = N1
            BLgxz_rot[10] = N2
            BLgxz_rot[16] = N3
            BLgxz_rot[22] = N4

            if h/length < 1.: # thin elements
                for i in range(24):
                    gyz_rot = BLgyz_rot[i]
                    gxz_rot = BLgxz_rot[i]
                    gyz_grad = BLgyz_grad[i]
                    gxz_grad = BLgxz_grad[i]
                    for j in range(24):
                        ke = 24*i + j
                        self.probe.KC0ve[ke] += wij*detJ*(
                        # transverse shear (gradient term)
                            gyz_grad*E44*BLgyz_grad[j] + gyz_grad*E45*BLgxz_grad[j]
                          + gxz_grad*E45*BLgyz_grad[j] + gxz_grad*E55*BLgxz_grad[j]

                        # transverse shear (coupled terms)
                          + gyz_grad*E44*BLgyz_rot[j] + gyz_grad*E45*BLgxz_rot[j]
                          + gxz_grad*E45*BLgyz_rot[j] + gxz_grad*E55*BLgxz_rot[j]

                          + gyz_rot*E44*BLgyz_grad[j] + gyz_rot*E45*BLgxz_grad[j]
                          + gxz_rot*E45*BLgyz_grad[j] + gxz_rot*E55*BLgxz_grad[j]

                        # transverse shear (rotation term)
                          + gyz_rot*E44*BLgyz_rot[j] + gyz_rot*E45*BLgxz_rot[j]
                          + gxz_rot*E45*BLgyz_rot[j] + gxz_rot*E55*BLgxz_rot[j]
                        )

            else: # thick elements
                for i in range(24):
                    gyz_rot = BLgyz_rot[i]
                    gxz_rot = BLgxz_rot[i]
                    gyz_grad = BLgyz_grad[i]
                    gxz_grad = BLgxz_grad[i]
                    for j in range(24):
                        ke = 24*i + j
                        self.probe.KC0ve[ke] += wij*detJ*(
                        # transverse shear (coupled terms)
                            gyz_grad*E44*BLgyz_rot[j] + gyz_grad*E45*BLgxz_rot[j]
                          + gxz_grad*E45*BLgyz_rot[j] + gxz_grad*E55*BLgxz_rot[j]

                          + gyz_rot*E44*BLgyz_grad[j] + gyz_rot*E45*BLgxz_grad[j]
                          + gxz_rot*E45*BLgyz_grad[j] + gxz_rot*E55*BLgxz_grad[j]

                        # transverse shear (rotation term)
                          + gyz_rot*E44*BLgyz_rot[j] + gyz_rot*E45*BLgxz_rot[j]
                          + gxz_rot*E45*BLgyz_rot[j] + gxz_rot*E55*BLgxz_rot[j]
                        )

            # drilling
            for node_i in range(NUM_NODES):
                node_j = node_i # NOTE only diagonal terms are affected
                ke = 24*(node_i*DOF + 5) + node_j*DOF + 5
                self.probe.KC0ve[ke] += K6ROT

            # NOTE from element to global coordinates:
            #
            # Kg = R @ Ke @ R.T
            #
            # in tensor notation:
            #
            # Kg_{mn} = r_{mi} * Ke_{ij} * r_{nj}
            #
            # TODO use r[3][3] instead
            for node_i in range(NUM_NODES):
                for m in range(DOF):
                    for node_j in range(NUM_NODES):
                        for n in range(DOF):
                            k = self.init_k_KC0 + 24*(node_i*DOF + m) + node_j*DOF + n
                            for i in range(DOF):
                                for j in range(DOF):
                                    ke = 24*(node_i*DOF + i) + node_j*DOF + j
                                    KC0v[k] += r[m][i]*self.probe.KC0ve[ke]*r[n][j]


    cpdef void update_KG(Quad4 self,
                         long [::1] KGr,
                         long [::1] KGc,
                         double [::1] KGv,
                         ShellProp prop
                         ):
        r"""Update sparse vectors for geometric stiffness matrix KG

        Two-point Gauss-Legendre quadrature is used, which showed more accuracy
        for linear buckling load predictions.

        Before this function is called, the probe :class:`.Quad4Probe`
        attribute of the :class:`.Quad4` object must be updated using
        :func:`.update_probe_ue` with the correct pre-buckling (fundamental
        state) displacements; and :func:`.update_probe_xe` with the node
        coordinates.

        Parameters
        ----------
        KGr : np.array
           Array to store row positions of sparse values
        KGc : np.array
           Array to store column positions of sparse values
        KGv : np.array
            Array to store sparse values
        prop : :class:`.ShellProp` object
            Shell property object from where the stiffness and mass attributes
            are read from.

        """
        cdef double *ue
        cdef int c1, c2, c3, c4, i, j, k
        cdef double x1, x2, x3, x4
        cdef double y1, y2, y3, y4
        cdef double xi, eta, wij, J11, J12, J21, J22, detJ
        cdef double points[2]
        # NOTE ABD in the material direction
        cdef double A11mat, A12mat, A16mat, A22mat, A26mat, A66mat
        cdef double B11mat, B12mat, B16mat, B22mat, B26mat, B66mat
        # NOTE ABD in the element direction
        cdef double A11, A12, A16, A22, A26, A66
        cdef double B11, B12, B16, B22, B26, B66
        cdef double r11, r12, r13, r21, r22, r23, r31, r32, r33
        cdef double m11, m12, m21, m22
        cdef double j11, j12, j21, j22
        cdef double N1x, N2x, N3x, N4x, N1y, N2y, N3y, N4y
        cdef double Nxx, Nyy, Nxy

        with nogil:
            A11mat = prop.A11
            A12mat = prop.A12
            A16mat = prop.A16
            A22mat = prop.A22
            A26mat = prop.A26
            A66mat = prop.A66
            B11mat = prop.B11
            B12mat = prop.B12
            B16mat = prop.B16
            B22mat = prop.B22
            B26mat = prop.B26
            B66mat = prop.B66

            # NOTE using self.m12 as a criterion to check if material
            #     coordinates were defined
            if self.m12 == 0:
                A11 = A11mat
                A12 = A12mat
                A16 = A16mat
                A22 = A22mat
                A26 = A26mat
                A66 = A66mat
                B11 = B11mat
                B12 = B12mat
                B16 = B16mat
                B22 = B22mat
                B26 = B26mat
                B66 = B66mat
            else:
                m11 = self.m11
                m12 = self.m12
                m21 = self.m21
                m22 = self.m22
                A11 = m11**2*(A11mat*m11**2 + A12mat*m12**2 + 2*A16mat*m11*m12) + 2*m11*m12*(A16mat*m11**2 + A26mat*m12**2 + 2*A66mat*m11*m12) + m12**2*(A12mat*m11**2 + A22mat*m12**2 + 2*A26mat*m11*m12)
                A12 = m21**2*(A11mat*m11**2 + A12mat*m12**2 + 2*A16mat*m11*m12) + 2*m21*m22*(A16mat*m11**2 + A26mat*m12**2 + 2*A66mat*m11*m12) + m22**2*(A12mat*m11**2 + A22mat*m12**2 + 2*A26mat*m11*m12)
                A16 = m11*m21*(A11mat*m11**2 + A12mat*m12**2 + 2*A16mat*m11*m12) + m12*m22*(A12mat*m11**2 + A22mat*m12**2 + 2*A26mat*m11*m12) + (m11*m22 + m12*m21)*(A16mat*m11**2 + A26mat*m12**2 + 2*A66mat*m11*m12)
                # A21 = m11**2*(A11mat*m21**2 + A12mat*m22**2 + 2*A16mat*m21*m22) + 2*m11*m12*(A16mat*m21**2 + A26mat*m22**2 + 2*A66mat*m21*m22) + m12**2*(A12mat*m21**2 + A22mat*m22**2 + 2*A26mat*m21*m22)
                A22 = m21**2*(A11mat*m21**2 + A12mat*m22**2 + 2*A16mat*m21*m22) + 2*m21*m22*(A16mat*m21**2 + A26mat*m22**2 + 2*A66mat*m21*m22) + m22**2*(A12mat*m21**2 + A22mat*m22**2 + 2*A26mat*m21*m22)
                A26 = m11*m21*(A11mat*m21**2 + A12mat*m22**2 + 2*A16mat*m21*m22) + m12*m22*(A12mat*m21**2 + A22mat*m22**2 + 2*A26mat*m21*m22) + (m11*m22 + m12*m21)*(A16mat*m21**2 + A26mat*m22**2 + 2*A66mat*m21*m22)
                # A61 = m11**2*(A11mat*m11*m21 + A12mat*m12*m22 + A16mat*(m11*m22 + m12*m21)) + 2*m11*m12*(A16mat*m11*m21 + A26mat*m12*m22 + A66mat*(m11*m22 + m12*m21)) + m12**2*(A12mat*m11*m21 + A22mat*m12*m22 + A26mat*(m11*m22 + m12*m21))
                # A62 = m21**2*(A11mat*m11*m21 + A12mat*m12*m22 + A16mat*(m11*m22 + m12*m21)) + 2*m21*m22*(A16mat*m11*m21 + A26mat*m12*m22 + A66mat*(m11*m22 + m12*m21)) + m22**2*(A12mat*m11*m21 + A22mat*m12*m22 + A26mat*(m11*m22 + m12*m21))
                A66 = m11*m21*(A11mat*m11*m21 + A12mat*m12*m22 + A16mat*(m11*m22 + m12*m21)) + m12*m22*(A12mat*m11*m21 + A22mat*m12*m22 + A26mat*(m11*m22 + m12*m21)) + (m11*m22 + m12*m21)*(A16mat*m11*m21 + A26mat*m12*m22 + A66mat*(m11*m22 + m12*m21))

                B11 = m11**2*(B11mat*m11**2 + B12mat*m12**2 + 2*B16mat*m11*m12) + 2*m11*m12*(B16mat*m11**2 + B26mat*m12**2 + 2*B66mat*m11*m12) + m12**2*(B12mat*m11**2 + B22mat*m12**2 + 2*B26mat*m11*m12)
                B12 = m21**2*(B11mat*m11**2 + B12mat*m12**2 + 2*B16mat*m11*m12) + 2*m21*m22*(B16mat*m11**2 + B26mat*m12**2 + 2*B66mat*m11*m12) + m22**2*(B12mat*m11**2 + B22mat*m12**2 + 2*B26mat*m11*m12)
                B16 = m11*m21*(B11mat*m11**2 + B12mat*m12**2 + 2*B16mat*m11*m12) + m12*m22*(B12mat*m11**2 + B22mat*m12**2 + 2*B26mat*m11*m12) + (m11*m22 + m12*m21)*(B16mat*m11**2 + B26mat*m12**2 + 2*B66mat*m11*m12)
                # B21 = m11**2*(B11mat*m21**2 + B12mat*m22**2 + 2*B16mat*m21*m22) + 2*m11*m12*(B16mat*m21**2 + B26mat*m22**2 + 2*B66mat*m21*m22) + m12**2*(B12mat*m21**2 + B22mat*m22**2 + 2*B26mat*m21*m22)
                B22 = m21**2*(B11mat*m21**2 + B12mat*m22**2 + 2*B16mat*m21*m22) + 2*m21*m22*(B16mat*m21**2 + B26mat*m22**2 + 2*B66mat*m21*m22) + m22**2*(B12mat*m21**2 + B22mat*m22**2 + 2*B26mat*m21*m22)
                B26 = m11*m21*(B11mat*m21**2 + B12mat*m22**2 + 2*B16mat*m21*m22) + m12*m22*(B12mat*m21**2 + B22mat*m22**2 + 2*B26mat*m21*m22) + (m11*m22 + m12*m21)*(B16mat*m21**2 + B26mat*m22**2 + 2*B66mat*m21*m22)
                # B61 = m11**2*(B11mat*m11*m21 + B12mat*m12*m22 + B16mat*(m11*m22 + m12*m21)) + 2*m11*m12*(B16mat*m11*m21 + B26mat*m12*m22 + B66mat*(m11*m22 + m12*m21)) + m12**2*(B12mat*m11*m21 + B22mat*m12*m22 + B26mat*(m11*m22 + m12*m21))
                # B62 = m21**2*(B11mat*m11*m21 + B12mat*m12*m22 + B16mat*(m11*m22 + m12*m21)) + 2*m21*m22*(B16mat*m11*m21 + B26mat*m12*m22 + B66mat*(m11*m22 + m12*m21)) + m22**2*(B12mat*m11*m21 + B22mat*m12*m22 + B26mat*(m11*m22 + m12*m21))
                B66 = m11*m21*(B11mat*m11*m21 + B12mat*m12*m22 + B16mat*(m11*m22 + m12*m21)) + m12*m22*(B12mat*m11*m21 + B22mat*m12*m22 + B26mat*(m11*m22 + m12*m21)) + (m11*m22 + m12*m21)*(B16mat*m11*m21 + B26mat*m12*m22 + B66mat*(m11*m22 + m12*m21))

            # local to global transformation
            r11 = self.r11
            r12 = self.r12
            r13 = self.r13
            r21 = self.r21
            r22 = self.r22
            r23 = self.r23
            r31 = self.r31
            r32 = self.r32
            r33 = self.r33

            # NOTE ignoring z in local coordinates
            x1 = self.probe.xe[0]
            y1 = self.probe.xe[1]
            # z1 = self.probe.xe[2]
            x2 = self.probe.xe[3]
            y2 = self.probe.xe[4]
            # z2 = self.probe.xe[5]
            x3 = self.probe.xe[6]
            y3 = self.probe.xe[7]
            # z3 = self.probe.xe[8]
            x4 = self.probe.xe[9]
            y4 = self.probe.xe[10]
            # z4 = self.probe.xe[11]

            ue = &self.probe.ue[0]

            # positions of nodes 1,2,3,4 in the global matrix
            c1 = self.c1
            c2 = self.c2
            c3 = self.c3
            c4 = self.c4

            k = self.init_k_KG
            KGr[k] = 0+c1
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 2+c4

            # NOTE full integration for KG with two-point Gauss-Legendre quadrature
            wij = 1.
            points[0] = -0.5773502691896257645092
            points[1] = +0.5773502691896257645092

            for i in range(2):
                xi = points[i]
                for j in range(2):
                    eta = points[j]

                    J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
                    J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
                    J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
                    J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

                    detJ = J11*J22 - J12*J21

                    j11 = J22/(J11*J22 - J12*J21)
                    j12 = -J12/(J11*J22 - J12*J21)
                    j21 = -J21/(J11*J22 - J12*J21)
                    j22 = J11/(J11*J22 - J12*J21)

                    N1x = 0.25*j11*(eta - 1) + 0.25*j12*(xi - 1)
                    N2x = -0.25*eta*j11 + 0.25*j11 - 0.25*j12*xi - 0.25*j12
                    N3x = 0.25*j11*(eta + 1) + 0.25*j12*(xi + 1)
                    N4x = -0.25*eta*j11 - 0.25*j11 - 0.25*j12*xi + 0.25*j12

                    N1y = 0.25*j21*(eta - 1) + 0.25*j22*(xi - 1)
                    N2y = -0.25*eta*j21 + 0.25*j21 - 0.25*j22*xi - 0.25*j22
                    N3y = 0.25*j21*(eta + 1) + 0.25*j22*(xi + 1)
                    N4y = -0.25*eta*j21 - 0.25*j21 - 0.25*j22*xi + 0.25*j22

                    Nxx = ue[0]*(A11*N1x + A16*N1y) + ue[10]*(B11*N2x + B16*N2y) + ue[12]*(A11*N3x + A16*N3y) + ue[13]*(A12*N3y + A16*N3x) - ue[15]*(B12*N3y + B16*N3x) + ue[16]*(B11*N3x + B16*N3y) + ue[18]*(A11*N4x + A16*N4y) + ue[19]*(A12*N4y + A16*N4x) + ue[1]*(A12*N1y + A16*N1x) - ue[21]*(B12*N4y + B16*N4x) + ue[22]*(B11*N4x + B16*N4y) - ue[3]*(B12*N1y + B16*N1x) + ue[4]*(B11*N1x + B16*N1y) + ue[6]*(A11*N2x + A16*N2y) + ue[7]*(A12*N2y + A16*N2x) - ue[9]*(B12*N2y + B16*N2x)
                    Nyy = ue[0]*(A12*N1x + A26*N1y) + ue[10]*(B12*N2x + B26*N2y) + ue[12]*(A12*N3x + A26*N3y) + ue[13]*(A22*N3y + A26*N3x) - ue[15]*(B22*N3y + B26*N3x) + ue[16]*(B12*N3x + B26*N3y) + ue[18]*(A12*N4x + A26*N4y) + ue[19]*(A22*N4y + A26*N4x) + ue[1]*(A22*N1y + A26*N1x) - ue[21]*(B22*N4y + B26*N4x) + ue[22]*(B12*N4x + B26*N4y) - ue[3]*(B22*N1y + B26*N1x) + ue[4]*(B12*N1x + B26*N1y) + ue[6]*(A12*N2x + A26*N2y) + ue[7]*(A22*N2y + A26*N2x) - ue[9]*(B22*N2y + B26*N2x)
                    Nxy = ue[0]*(A16*N1x + A66*N1y) + ue[10]*(B16*N2x + B66*N2y) + ue[12]*(A16*N3x + A66*N3y) + ue[13]*(A26*N3y + A66*N3x) - ue[15]*(B26*N3y + B66*N3x) + ue[16]*(B16*N3x + B66*N3y) + ue[18]*(A16*N4x + A66*N4y) + ue[19]*(A26*N4y + A66*N4x) + ue[1]*(A26*N1y + A66*N1x) - ue[21]*(B26*N4y + B66*N4x) + ue[22]*(B16*N4x + B66*N4y) - ue[3]*(B26*N1y + B66*N1x) + ue[4]*(B16*N1x + B66*N1y) + ue[6]*(A16*N2x + A66*N2y) + ue[7]*(A26*N2y + A66*N2x) - ue[9]*(B26*N2y + B66*N2x)

                    k = self.init_k_KG
                    KGv[k] += r13**2*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))


    cpdef void update_KG_given_stress(Quad4 self,
                                      double Nxx, double Nyy, double Nxy,
                                      long [::1] KGr,
                                      long [::1] KGc,
                                      double [::1] KGv
                                      ):
        r"""Update sparse vectors for geometric stiffness matrix KG

        .. note:: A constant stress state is assumed within the element,
                  according to the given values of `N_{xx}, N_{yy}, N_{xy}`.

        Two-point Gauss-Legendre quadrature is used, which showed more accuracy
        for linear buckling load predictions.

        Before this function is called, the probe :class:`.Quad4Probe`
        attribute of the :class:`.Quad4` object must be updated using
        :func:`.update_probe_xe` with the node coordinates.

        Parameters
        ----------
        KGr : np.array
           Array to store row positions of sparse values
        KGc : np.array
           Array to store column positions of sparse values
        KGv : np.array
            Array to store sparse values

        """
        cdef int c1, c2, c3, c4, i, j, k
        cdef double x1, x2, x3, x4
        cdef double y1, y2, y3, y4
        cdef double xi, eta, wij, J11, J12, J21, J22, detJ
        cdef double points[2]
        cdef double r11, r12, r13, r21, r22, r23, r31, r32, r33
        cdef double j11, j12, j21, j22
        cdef double N1x, N2x, N3x, N4x, N1y, N2y, N3y, N4y

        with nogil:
            # local to global transformation
            r11 = self.r11
            r12 = self.r12
            r13 = self.r13
            r21 = self.r21
            r22 = self.r22
            r23 = self.r23
            r31 = self.r31
            r32 = self.r32
            r33 = self.r33

            # NOTE ignoring z in local coordinates
            x1 = self.probe.xe[0]
            y1 = self.probe.xe[1]
            # z1 = self.probe.xe[2]
            x2 = self.probe.xe[3]
            y2 = self.probe.xe[4]
            # z2 = self.probe.xe[5]
            x3 = self.probe.xe[6]
            y3 = self.probe.xe[7]
            # z3 = self.probe.xe[8]
            x4 = self.probe.xe[9]
            y4 = self.probe.xe[10]
            # z4 = self.probe.xe[11]

            # positions of nodes 1,2,3,4 in the global matrix
            c1 = self.c1
            c2 = self.c2
            c3 = self.c3
            c4 = self.c4

            k = self.init_k_KG
            KGr[k] = 0+c1
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 0+c1
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 1+c1
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 2+c1
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 0+c2
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 1+c2
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 2+c2
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 0+c3
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 1+c3
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 2+c3
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 0+c4
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 1+c4
            KGc[k] = 2+c4
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 0+c1
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 1+c1
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 2+c1
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 0+c2
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 1+c2
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 2+c2
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 0+c3
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 1+c3
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 2+c3
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 0+c4
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 1+c4
            k += 1
            KGr[k] = 2+c4
            KGc[k] = 2+c4

            # NOTE full integration for KG with two-point Gauss-Legendre quadrature
            wij = 1.
            points[0] = -0.5773502691896257645092
            points[1] = +0.5773502691896257645092

            for i in range(2):
                xi = points[i]
                for j in range(2):
                    eta = points[j]

                    J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
                    J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
                    J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
                    J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

                    detJ = J11*J22 - J12*J21

                    j11 = J22/(J11*J22 - J12*J21)
                    j12 = -J12/(J11*J22 - J12*J21)
                    j21 = -J21/(J11*J22 - J12*J21)
                    j22 = J11/(J11*J22 - J12*J21)

                    N1x = 0.25*j11*(eta - 1) + 0.25*j12*(xi - 1)
                    N2x = -0.25*eta*j11 + 0.25*j11 - 0.25*j12*xi - 0.25*j12
                    N3x = 0.25*j11*(eta + 1) + 0.25*j12*(xi + 1)
                    N4x = -0.25*eta*j11 - 0.25*j11 - 0.25*j12*xi + 0.25*j12

                    N1y = 0.25*j21*(eta - 1) + 0.25*j22*(xi - 1)
                    N2y = -0.25*eta*j21 + 0.25*j21 - 0.25*j22*xi - 0.25*j22
                    N3y = 0.25*j21*(eta + 1) + 0.25*j22*(xi + 1)
                    N4y = -0.25*eta*j21 - 0.25*j21 - 0.25*j22*xi + 0.25*j22

                    k = self.init_k_KG
                    KGv[k] += r13**2*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N1x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N1y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N2x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N2y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N3x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N3y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N4x*(N1x*Nxx*detJ*wij + N1y*Nxy*detJ*wij) + N4y*(N1x*Nxy*detJ*wij + N1y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N1x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N1y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N2x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N2y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N3x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N3y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N4x*(N2x*Nxx*detJ*wij + N2y*Nxy*detJ*wij) + N4y*(N2x*Nxy*detJ*wij + N2y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N1x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N1y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N2x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N2y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N3x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N3y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N4x*(N3x*Nxx*detJ*wij + N3y*Nxy*detJ*wij) + N4y*(N3x*Nxy*detJ*wij + N3y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13**2*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r23*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23**2*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N1x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N1y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N2x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N2y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N3x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N3y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r13*r33*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r23*r33*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))
                    k += 1
                    KGv[k] += r33**2*(N4x*(N4x*Nxx*detJ*wij + N4y*Nxy*detJ*wij) + N4y*(N4x*Nxy*detJ*wij + N4y*Nyy*detJ*wij))


    cpdef void update_M(Quad4 self,
                        long [::1] Mr,
                        long [::1] Mc,
                        double [::1] Mv,
                        ShellProp prop,
                        int mtype=0,
                        ):
        r"""Update sparse vectors for mass matrix M

        Different integration schemes are available by means of the ``mtype``
        parameter.

        Parameters
        ----------
        Mr : np.array
            Array to store row positions of sparse values
        Mc : np.array
            Array to store column positions of sparse values
        Mv : np.array
            Array to store sparse values
        mtype : int, optional
            0 for consistent mass matrix using method from Brockman 1987
            1 for reduced integration mass matrix using method from Brockman 1987
            2 for lumped mass matrix using method from Brockman 1987

        """
        cdef int c1, c2, c3, c4, i, j, k
        cdef double intrho, intrhoz, intrhoz2
        cdef double r11, r12, r13, r21, r22, r23, r31, r32, r33
        cdef double x1, x2, x3, x4
        cdef double y1, y2, y3, y4
        cdef double h11, h12, h13, h14, h22, h23, h24, h33, h34, h44, valH1
        cdef double xi, eta, wij, J11, J12, J21, J22, detJ
        cdef double N1, N2, N3, N4
        cdef double points[2]

        with nogil:
            intrho = prop.intrho
            intrhoz = prop.intrhoz
            intrhoz2 = prop.intrhoz2

            valH1 = 0.0625*self.area

            # NOTE ignoring z in local coordinates
            x1 = self.probe.xe[0]
            y1 = self.probe.xe[1]
            # z1 = self.probe.xe[2]
            x2 = self.probe.xe[3]
            y2 = self.probe.xe[4]
            # z2 = self.probe.xe[5]
            x3 = self.probe.xe[6]
            y3 = self.probe.xe[7]
            # z3 = self.probe.xe[8]
            x4 = self.probe.xe[9]
            y4 = self.probe.xe[10]
            # z4 = self.probe.xe[11]

            # local to global transformation
            r11 = self.r11
            r12 = self.r12
            r13 = self.r13
            r21 = self.r21
            r22 = self.r22
            r23 = self.r23
            r31 = self.r31
            r32 = self.r32
            r33 = self.r33

            # positions the global matrices
            c1 = self.c1
            c2 = self.c2
            c3 = self.c3
            c4 = self.c4

            if mtype == 0: # M_cons consistent mass matrix, using two-point Gauss-Legendre quadrature
                k = self.init_k_M
                Mr[k] = 0+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c4

                # NOTE two-point Gauss-Legendre quadrature
                wij = 1.
                points[0] = -0.5773502691896257645092
                points[1] = +0.5773502691896257645092
                h11 = 0.
                h12 = 0.
                h13 = 0.
                h14 = 0.
                h22 = 0.
                h23 = 0.
                h24 = 0.
                h33 = 0.
                h34 = 0.
                h44 = 0.
                for i in range(2):
                    xi = points[i]
                    for j in range(2):
                        eta = points[j]

                        J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
                        J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
                        J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
                        J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

                        detJ = J11*J22 - J12*J21

                        N1 = eta*xi/4. - eta/4. - xi/4. + 1/4.
                        N2 = -eta*xi/4. - eta/4. + xi/4. + 1/4.
                        N3 = eta*xi/4. + eta/4. + xi/4. + 1/4.
                        N4 = -eta*xi/4. + eta/4. - xi/4. + 1/4.

                        h11 += N1**2*detJ*wij
                        h12 += N1*N2*detJ*wij
                        h13 += N1*N3*detJ*wij
                        h14 += N1*N4*detJ*wij
                        h22 += N2**2*detJ*wij
                        h23 += N2*N3*detJ*wij
                        h24 += N2*N4*detJ*wij
                        h33 += N3**2*detJ*wij
                        h34 += N3*N4*detJ*wij
                        h44 += N4**2*detJ*wij

                k = self.init_k_M
                Mv[k] += h11*intrho*r11**2 + h11*intrho*r12**2 + h11*intrho*r13**2
                k += 1
                Mv[k] += h11*intrho*r11*r21 + h11*intrho*r12*r22 + h11*intrho*r13*r23
                k += 1
                Mv[k] += h11*intrho*r11*r31 + h11*intrho*r12*r32 + h11*intrho*r13*r33
                k += 1
                Mv[k] += h11*intrhoz*r11*r22 - h11*intrhoz*r12*r21
                k += 1
                Mv[k] += h11*intrhoz*r11*r32 - h11*intrhoz*r12*r31
                k += 1
                Mv[k] += h12*intrho*r11**2 + h12*intrho*r12**2 + h12*intrho*r13**2
                k += 1
                Mv[k] += h12*intrho*r11*r21 + h12*intrho*r12*r22 + h12*intrho*r13*r23
                k += 1
                Mv[k] += h12*intrho*r11*r31 + h12*intrho*r12*r32 + h12*intrho*r13*r33
                k += 1
                Mv[k] += h12*intrhoz*r11*r22 - h12*intrhoz*r12*r21
                k += 1
                Mv[k] += h12*intrhoz*r11*r32 - h12*intrhoz*r12*r31
                k += 1
                Mv[k] += h13*intrho*r11**2 + h13*intrho*r12**2 + h13*intrho*r13**2
                k += 1
                Mv[k] += h13*intrho*r11*r21 + h13*intrho*r12*r22 + h13*intrho*r13*r23
                k += 1
                Mv[k] += h13*intrho*r11*r31 + h13*intrho*r12*r32 + h13*intrho*r13*r33
                k += 1
                Mv[k] += h13*intrhoz*r11*r22 - h13*intrhoz*r12*r21
                k += 1
                Mv[k] += h13*intrhoz*r11*r32 - h13*intrhoz*r12*r31
                k += 1
                Mv[k] += h14*intrho*r11**2 + h14*intrho*r12**2 + h14*intrho*r13**2
                k += 1
                Mv[k] += h14*intrho*r11*r21 + h14*intrho*r12*r22 + h14*intrho*r13*r23
                k += 1
                Mv[k] += h14*intrho*r11*r31 + h14*intrho*r12*r32 + h14*intrho*r13*r33
                k += 1
                Mv[k] += h14*intrhoz*r11*r22 - h14*intrhoz*r12*r21
                k += 1
                Mv[k] += h14*intrhoz*r11*r32 - h14*intrhoz*r12*r31
                k += 1
                Mv[k] += h11*intrho*r11*r21 + h11*intrho*r12*r22 + h11*intrho*r13*r23
                k += 1
                Mv[k] += h11*intrho*r21**2 + h11*intrho*r22**2 + h11*intrho*r23**2
                k += 1
                Mv[k] += h11*intrho*r21*r31 + h11*intrho*r22*r32 + h11*intrho*r23*r33
                k += 1
                Mv[k] += -h11*intrhoz*r11*r22 + h11*intrhoz*r12*r21
                k += 1
                Mv[k] += h11*intrhoz*r21*r32 - h11*intrhoz*r22*r31
                k += 1
                Mv[k] += h12*intrho*r11*r21 + h12*intrho*r12*r22 + h12*intrho*r13*r23
                k += 1
                Mv[k] += h12*intrho*r21**2 + h12*intrho*r22**2 + h12*intrho*r23**2
                k += 1
                Mv[k] += h12*intrho*r21*r31 + h12*intrho*r22*r32 + h12*intrho*r23*r33
                k += 1
                Mv[k] += -h12*intrhoz*r11*r22 + h12*intrhoz*r12*r21
                k += 1
                Mv[k] += h12*intrhoz*r21*r32 - h12*intrhoz*r22*r31
                k += 1
                Mv[k] += h13*intrho*r11*r21 + h13*intrho*r12*r22 + h13*intrho*r13*r23
                k += 1
                Mv[k] += h13*intrho*r21**2 + h13*intrho*r22**2 + h13*intrho*r23**2
                k += 1
                Mv[k] += h13*intrho*r21*r31 + h13*intrho*r22*r32 + h13*intrho*r23*r33
                k += 1
                Mv[k] += -h13*intrhoz*r11*r22 + h13*intrhoz*r12*r21
                k += 1
                Mv[k] += h13*intrhoz*r21*r32 - h13*intrhoz*r22*r31
                k += 1
                Mv[k] += h14*intrho*r11*r21 + h14*intrho*r12*r22 + h14*intrho*r13*r23
                k += 1
                Mv[k] += h14*intrho*r21**2 + h14*intrho*r22**2 + h14*intrho*r23**2
                k += 1
                Mv[k] += h14*intrho*r21*r31 + h14*intrho*r22*r32 + h14*intrho*r23*r33
                k += 1
                Mv[k] += -h14*intrhoz*r11*r22 + h14*intrhoz*r12*r21
                k += 1
                Mv[k] += h14*intrhoz*r21*r32 - h14*intrhoz*r22*r31
                k += 1
                Mv[k] += h11*intrho*r11*r31 + h11*intrho*r12*r32 + h11*intrho*r13*r33
                k += 1
                Mv[k] += h11*intrho*r21*r31 + h11*intrho*r22*r32 + h11*intrho*r23*r33
                k += 1
                Mv[k] += h11*intrho*r31**2 + h11*intrho*r32**2 + h11*intrho*r33**2
                k += 1
                Mv[k] += -h11*intrhoz*r11*r32 + h11*intrhoz*r12*r31
                k += 1
                Mv[k] += -h11*intrhoz*r21*r32 + h11*intrhoz*r22*r31
                k += 1
                Mv[k] += h12*intrho*r11*r31 + h12*intrho*r12*r32 + h12*intrho*r13*r33
                k += 1
                Mv[k] += h12*intrho*r21*r31 + h12*intrho*r22*r32 + h12*intrho*r23*r33
                k += 1
                Mv[k] += h12*intrho*r31**2 + h12*intrho*r32**2 + h12*intrho*r33**2
                k += 1
                Mv[k] += -h12*intrhoz*r11*r32 + h12*intrhoz*r12*r31
                k += 1
                Mv[k] += -h12*intrhoz*r21*r32 + h12*intrhoz*r22*r31
                k += 1
                Mv[k] += h13*intrho*r11*r31 + h13*intrho*r12*r32 + h13*intrho*r13*r33
                k += 1
                Mv[k] += h13*intrho*r21*r31 + h13*intrho*r22*r32 + h13*intrho*r23*r33
                k += 1
                Mv[k] += h13*intrho*r31**2 + h13*intrho*r32**2 + h13*intrho*r33**2
                k += 1
                Mv[k] += -h13*intrhoz*r11*r32 + h13*intrhoz*r12*r31
                k += 1
                Mv[k] += -h13*intrhoz*r21*r32 + h13*intrhoz*r22*r31
                k += 1
                Mv[k] += h14*intrho*r11*r31 + h14*intrho*r12*r32 + h14*intrho*r13*r33
                k += 1
                Mv[k] += h14*intrho*r21*r31 + h14*intrho*r22*r32 + h14*intrho*r23*r33
                k += 1
                Mv[k] += h14*intrho*r31**2 + h14*intrho*r32**2 + h14*intrho*r33**2
                k += 1
                Mv[k] += -h14*intrhoz*r11*r32 + h14*intrhoz*r12*r31
                k += 1
                Mv[k] += -h14*intrhoz*r21*r32 + h14*intrhoz*r22*r31
                k += 1
                Mv[k] += -h11*intrhoz*r11*r22 + h11*intrhoz*r12*r21
                k += 1
                Mv[k] += -h11*intrhoz*r11*r32 + h11*intrhoz*r12*r31
                k += 1
                Mv[k] += h11*intrhoz2*r11**2 + h11*intrhoz2*r12**2
                k += 1
                Mv[k] += h11*intrhoz2*r11*r21 + h11*intrhoz2*r12*r22
                k += 1
                Mv[k] += h11*intrhoz2*r11*r31 + h11*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h12*intrhoz*r11*r22 + h12*intrhoz*r12*r21
                k += 1
                Mv[k] += -h12*intrhoz*r11*r32 + h12*intrhoz*r12*r31
                k += 1
                Mv[k] += h12*intrhoz2*r11**2 + h12*intrhoz2*r12**2
                k += 1
                Mv[k] += h12*intrhoz2*r11*r21 + h12*intrhoz2*r12*r22
                k += 1
                Mv[k] += h12*intrhoz2*r11*r31 + h12*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h13*intrhoz*r11*r22 + h13*intrhoz*r12*r21
                k += 1
                Mv[k] += -h13*intrhoz*r11*r32 + h13*intrhoz*r12*r31
                k += 1
                Mv[k] += h13*intrhoz2*r11**2 + h13*intrhoz2*r12**2
                k += 1
                Mv[k] += h13*intrhoz2*r11*r21 + h13*intrhoz2*r12*r22
                k += 1
                Mv[k] += h13*intrhoz2*r11*r31 + h13*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h14*intrhoz*r11*r22 + h14*intrhoz*r12*r21
                k += 1
                Mv[k] += -h14*intrhoz*r11*r32 + h14*intrhoz*r12*r31
                k += 1
                Mv[k] += h14*intrhoz2*r11**2 + h14*intrhoz2*r12**2
                k += 1
                Mv[k] += h14*intrhoz2*r11*r21 + h14*intrhoz2*r12*r22
                k += 1
                Mv[k] += h14*intrhoz2*r11*r31 + h14*intrhoz2*r12*r32
                k += 1
                Mv[k] += h11*intrhoz*r11*r22 - h11*intrhoz*r12*r21
                k += 1
                Mv[k] += -h11*intrhoz*r21*r32 + h11*intrhoz*r22*r31
                k += 1
                Mv[k] += h11*intrhoz2*r11*r21 + h11*intrhoz2*r12*r22
                k += 1
                Mv[k] += h11*intrhoz2*r21**2 + h11*intrhoz2*r22**2
                k += 1
                Mv[k] += h11*intrhoz2*r21*r31 + h11*intrhoz2*r22*r32
                k += 1
                Mv[k] += h12*intrhoz*r11*r22 - h12*intrhoz*r12*r21
                k += 1
                Mv[k] += -h12*intrhoz*r21*r32 + h12*intrhoz*r22*r31
                k += 1
                Mv[k] += h12*intrhoz2*r11*r21 + h12*intrhoz2*r12*r22
                k += 1
                Mv[k] += h12*intrhoz2*r21**2 + h12*intrhoz2*r22**2
                k += 1
                Mv[k] += h12*intrhoz2*r21*r31 + h12*intrhoz2*r22*r32
                k += 1
                Mv[k] += h13*intrhoz*r11*r22 - h13*intrhoz*r12*r21
                k += 1
                Mv[k] += -h13*intrhoz*r21*r32 + h13*intrhoz*r22*r31
                k += 1
                Mv[k] += h13*intrhoz2*r11*r21 + h13*intrhoz2*r12*r22
                k += 1
                Mv[k] += h13*intrhoz2*r21**2 + h13*intrhoz2*r22**2
                k += 1
                Mv[k] += h13*intrhoz2*r21*r31 + h13*intrhoz2*r22*r32
                k += 1
                Mv[k] += h14*intrhoz*r11*r22 - h14*intrhoz*r12*r21
                k += 1
                Mv[k] += -h14*intrhoz*r21*r32 + h14*intrhoz*r22*r31
                k += 1
                Mv[k] += h14*intrhoz2*r11*r21 + h14*intrhoz2*r12*r22
                k += 1
                Mv[k] += h14*intrhoz2*r21**2 + h14*intrhoz2*r22**2
                k += 1
                Mv[k] += h14*intrhoz2*r21*r31 + h14*intrhoz2*r22*r32
                k += 1
                Mv[k] += h11*intrhoz*r11*r32 - h11*intrhoz*r12*r31
                k += 1
                Mv[k] += h11*intrhoz*r21*r32 - h11*intrhoz*r22*r31
                k += 1
                Mv[k] += h11*intrhoz2*r11*r31 + h11*intrhoz2*r12*r32
                k += 1
                Mv[k] += h11*intrhoz2*r21*r31 + h11*intrhoz2*r22*r32
                k += 1
                Mv[k] += h11*intrhoz2*r31**2 + h11*intrhoz2*r32**2
                k += 1
                Mv[k] += h12*intrhoz*r11*r32 - h12*intrhoz*r12*r31
                k += 1
                Mv[k] += h12*intrhoz*r21*r32 - h12*intrhoz*r22*r31
                k += 1
                Mv[k] += h12*intrhoz2*r11*r31 + h12*intrhoz2*r12*r32
                k += 1
                Mv[k] += h12*intrhoz2*r21*r31 + h12*intrhoz2*r22*r32
                k += 1
                Mv[k] += h12*intrhoz2*r31**2 + h12*intrhoz2*r32**2
                k += 1
                Mv[k] += h13*intrhoz*r11*r32 - h13*intrhoz*r12*r31
                k += 1
                Mv[k] += h13*intrhoz*r21*r32 - h13*intrhoz*r22*r31
                k += 1
                Mv[k] += h13*intrhoz2*r11*r31 + h13*intrhoz2*r12*r32
                k += 1
                Mv[k] += h13*intrhoz2*r21*r31 + h13*intrhoz2*r22*r32
                k += 1
                Mv[k] += h13*intrhoz2*r31**2 + h13*intrhoz2*r32**2
                k += 1
                Mv[k] += h14*intrhoz*r11*r32 - h14*intrhoz*r12*r31
                k += 1
                Mv[k] += h14*intrhoz*r21*r32 - h14*intrhoz*r22*r31
                k += 1
                Mv[k] += h14*intrhoz2*r11*r31 + h14*intrhoz2*r12*r32
                k += 1
                Mv[k] += h14*intrhoz2*r21*r31 + h14*intrhoz2*r22*r32
                k += 1
                Mv[k] += h14*intrhoz2*r31**2 + h14*intrhoz2*r32**2
                k += 1
                Mv[k] += h12*intrho*r11**2 + h12*intrho*r12**2 + h12*intrho*r13**2
                k += 1
                Mv[k] += h12*intrho*r11*r21 + h12*intrho*r12*r22 + h12*intrho*r13*r23
                k += 1
                Mv[k] += h12*intrho*r11*r31 + h12*intrho*r12*r32 + h12*intrho*r13*r33
                k += 1
                Mv[k] += h12*intrhoz*r11*r22 - h12*intrhoz*r12*r21
                k += 1
                Mv[k] += h12*intrhoz*r11*r32 - h12*intrhoz*r12*r31
                k += 1
                Mv[k] += h22*intrho*r11**2 + h22*intrho*r12**2 + h22*intrho*r13**2
                k += 1
                Mv[k] += h22*intrho*r11*r21 + h22*intrho*r12*r22 + h22*intrho*r13*r23
                k += 1
                Mv[k] += h22*intrho*r11*r31 + h22*intrho*r12*r32 + h22*intrho*r13*r33
                k += 1
                Mv[k] += h22*intrhoz*r11*r22 - h22*intrhoz*r12*r21
                k += 1
                Mv[k] += h22*intrhoz*r11*r32 - h22*intrhoz*r12*r31
                k += 1
                Mv[k] += h23*intrho*r11**2 + h23*intrho*r12**2 + h23*intrho*r13**2
                k += 1
                Mv[k] += h23*intrho*r11*r21 + h23*intrho*r12*r22 + h23*intrho*r13*r23
                k += 1
                Mv[k] += h23*intrho*r11*r31 + h23*intrho*r12*r32 + h23*intrho*r13*r33
                k += 1
                Mv[k] += h23*intrhoz*r11*r22 - h23*intrhoz*r12*r21
                k += 1
                Mv[k] += h23*intrhoz*r11*r32 - h23*intrhoz*r12*r31
                k += 1
                Mv[k] += h24*intrho*r11**2 + h24*intrho*r12**2 + h24*intrho*r13**2
                k += 1
                Mv[k] += h24*intrho*r11*r21 + h24*intrho*r12*r22 + h24*intrho*r13*r23
                k += 1
                Mv[k] += h24*intrho*r11*r31 + h24*intrho*r12*r32 + h24*intrho*r13*r33
                k += 1
                Mv[k] += h24*intrhoz*r11*r22 - h24*intrhoz*r12*r21
                k += 1
                Mv[k] += h24*intrhoz*r11*r32 - h24*intrhoz*r12*r31
                k += 1
                Mv[k] += h12*intrho*r11*r21 + h12*intrho*r12*r22 + h12*intrho*r13*r23
                k += 1
                Mv[k] += h12*intrho*r21**2 + h12*intrho*r22**2 + h12*intrho*r23**2
                k += 1
                Mv[k] += h12*intrho*r21*r31 + h12*intrho*r22*r32 + h12*intrho*r23*r33
                k += 1
                Mv[k] += -h12*intrhoz*r11*r22 + h12*intrhoz*r12*r21
                k += 1
                Mv[k] += h12*intrhoz*r21*r32 - h12*intrhoz*r22*r31
                k += 1
                Mv[k] += h22*intrho*r11*r21 + h22*intrho*r12*r22 + h22*intrho*r13*r23
                k += 1
                Mv[k] += h22*intrho*r21**2 + h22*intrho*r22**2 + h22*intrho*r23**2
                k += 1
                Mv[k] += h22*intrho*r21*r31 + h22*intrho*r22*r32 + h22*intrho*r23*r33
                k += 1
                Mv[k] += -h22*intrhoz*r11*r22 + h22*intrhoz*r12*r21
                k += 1
                Mv[k] += h22*intrhoz*r21*r32 - h22*intrhoz*r22*r31
                k += 1
                Mv[k] += h23*intrho*r11*r21 + h23*intrho*r12*r22 + h23*intrho*r13*r23
                k += 1
                Mv[k] += h23*intrho*r21**2 + h23*intrho*r22**2 + h23*intrho*r23**2
                k += 1
                Mv[k] += h23*intrho*r21*r31 + h23*intrho*r22*r32 + h23*intrho*r23*r33
                k += 1
                Mv[k] += -h23*intrhoz*r11*r22 + h23*intrhoz*r12*r21
                k += 1
                Mv[k] += h23*intrhoz*r21*r32 - h23*intrhoz*r22*r31
                k += 1
                Mv[k] += h24*intrho*r11*r21 + h24*intrho*r12*r22 + h24*intrho*r13*r23
                k += 1
                Mv[k] += h24*intrho*r21**2 + h24*intrho*r22**2 + h24*intrho*r23**2
                k += 1
                Mv[k] += h24*intrho*r21*r31 + h24*intrho*r22*r32 + h24*intrho*r23*r33
                k += 1
                Mv[k] += -h24*intrhoz*r11*r22 + h24*intrhoz*r12*r21
                k += 1
                Mv[k] += h24*intrhoz*r21*r32 - h24*intrhoz*r22*r31
                k += 1
                Mv[k] += h12*intrho*r11*r31 + h12*intrho*r12*r32 + h12*intrho*r13*r33
                k += 1
                Mv[k] += h12*intrho*r21*r31 + h12*intrho*r22*r32 + h12*intrho*r23*r33
                k += 1
                Mv[k] += h12*intrho*r31**2 + h12*intrho*r32**2 + h12*intrho*r33**2
                k += 1
                Mv[k] += -h12*intrhoz*r11*r32 + h12*intrhoz*r12*r31
                k += 1
                Mv[k] += -h12*intrhoz*r21*r32 + h12*intrhoz*r22*r31
                k += 1
                Mv[k] += h22*intrho*r11*r31 + h22*intrho*r12*r32 + h22*intrho*r13*r33
                k += 1
                Mv[k] += h22*intrho*r21*r31 + h22*intrho*r22*r32 + h22*intrho*r23*r33
                k += 1
                Mv[k] += h22*intrho*r31**2 + h22*intrho*r32**2 + h22*intrho*r33**2
                k += 1
                Mv[k] += -h22*intrhoz*r11*r32 + h22*intrhoz*r12*r31
                k += 1
                Mv[k] += -h22*intrhoz*r21*r32 + h22*intrhoz*r22*r31
                k += 1
                Mv[k] += h23*intrho*r11*r31 + h23*intrho*r12*r32 + h23*intrho*r13*r33
                k += 1
                Mv[k] += h23*intrho*r21*r31 + h23*intrho*r22*r32 + h23*intrho*r23*r33
                k += 1
                Mv[k] += h23*intrho*r31**2 + h23*intrho*r32**2 + h23*intrho*r33**2
                k += 1
                Mv[k] += -h23*intrhoz*r11*r32 + h23*intrhoz*r12*r31
                k += 1
                Mv[k] += -h23*intrhoz*r21*r32 + h23*intrhoz*r22*r31
                k += 1
                Mv[k] += h24*intrho*r11*r31 + h24*intrho*r12*r32 + h24*intrho*r13*r33
                k += 1
                Mv[k] += h24*intrho*r21*r31 + h24*intrho*r22*r32 + h24*intrho*r23*r33
                k += 1
                Mv[k] += h24*intrho*r31**2 + h24*intrho*r32**2 + h24*intrho*r33**2
                k += 1
                Mv[k] += -h24*intrhoz*r11*r32 + h24*intrhoz*r12*r31
                k += 1
                Mv[k] += -h24*intrhoz*r21*r32 + h24*intrhoz*r22*r31
                k += 1
                Mv[k] += -h12*intrhoz*r11*r22 + h12*intrhoz*r12*r21
                k += 1
                Mv[k] += -h12*intrhoz*r11*r32 + h12*intrhoz*r12*r31
                k += 1
                Mv[k] += h12*intrhoz2*r11**2 + h12*intrhoz2*r12**2
                k += 1
                Mv[k] += h12*intrhoz2*r11*r21 + h12*intrhoz2*r12*r22
                k += 1
                Mv[k] += h12*intrhoz2*r11*r31 + h12*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h22*intrhoz*r11*r22 + h22*intrhoz*r12*r21
                k += 1
                Mv[k] += -h22*intrhoz*r11*r32 + h22*intrhoz*r12*r31
                k += 1
                Mv[k] += h22*intrhoz2*r11**2 + h22*intrhoz2*r12**2
                k += 1
                Mv[k] += h22*intrhoz2*r11*r21 + h22*intrhoz2*r12*r22
                k += 1
                Mv[k] += h22*intrhoz2*r11*r31 + h22*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h23*intrhoz*r11*r22 + h23*intrhoz*r12*r21
                k += 1
                Mv[k] += -h23*intrhoz*r11*r32 + h23*intrhoz*r12*r31
                k += 1
                Mv[k] += h23*intrhoz2*r11**2 + h23*intrhoz2*r12**2
                k += 1
                Mv[k] += h23*intrhoz2*r11*r21 + h23*intrhoz2*r12*r22
                k += 1
                Mv[k] += h23*intrhoz2*r11*r31 + h23*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h24*intrhoz*r11*r22 + h24*intrhoz*r12*r21
                k += 1
                Mv[k] += -h24*intrhoz*r11*r32 + h24*intrhoz*r12*r31
                k += 1
                Mv[k] += h24*intrhoz2*r11**2 + h24*intrhoz2*r12**2
                k += 1
                Mv[k] += h24*intrhoz2*r11*r21 + h24*intrhoz2*r12*r22
                k += 1
                Mv[k] += h24*intrhoz2*r11*r31 + h24*intrhoz2*r12*r32
                k += 1
                Mv[k] += h12*intrhoz*r11*r22 - h12*intrhoz*r12*r21
                k += 1
                Mv[k] += -h12*intrhoz*r21*r32 + h12*intrhoz*r22*r31
                k += 1
                Mv[k] += h12*intrhoz2*r11*r21 + h12*intrhoz2*r12*r22
                k += 1
                Mv[k] += h12*intrhoz2*r21**2 + h12*intrhoz2*r22**2
                k += 1
                Mv[k] += h12*intrhoz2*r21*r31 + h12*intrhoz2*r22*r32
                k += 1
                Mv[k] += h22*intrhoz*r11*r22 - h22*intrhoz*r12*r21
                k += 1
                Mv[k] += -h22*intrhoz*r21*r32 + h22*intrhoz*r22*r31
                k += 1
                Mv[k] += h22*intrhoz2*r11*r21 + h22*intrhoz2*r12*r22
                k += 1
                Mv[k] += h22*intrhoz2*r21**2 + h22*intrhoz2*r22**2
                k += 1
                Mv[k] += h22*intrhoz2*r21*r31 + h22*intrhoz2*r22*r32
                k += 1
                Mv[k] += h23*intrhoz*r11*r22 - h23*intrhoz*r12*r21
                k += 1
                Mv[k] += -h23*intrhoz*r21*r32 + h23*intrhoz*r22*r31
                k += 1
                Mv[k] += h23*intrhoz2*r11*r21 + h23*intrhoz2*r12*r22
                k += 1
                Mv[k] += h23*intrhoz2*r21**2 + h23*intrhoz2*r22**2
                k += 1
                Mv[k] += h23*intrhoz2*r21*r31 + h23*intrhoz2*r22*r32
                k += 1
                Mv[k] += h24*intrhoz*r11*r22 - h24*intrhoz*r12*r21
                k += 1
                Mv[k] += -h24*intrhoz*r21*r32 + h24*intrhoz*r22*r31
                k += 1
                Mv[k] += h24*intrhoz2*r11*r21 + h24*intrhoz2*r12*r22
                k += 1
                Mv[k] += h24*intrhoz2*r21**2 + h24*intrhoz2*r22**2
                k += 1
                Mv[k] += h24*intrhoz2*r21*r31 + h24*intrhoz2*r22*r32
                k += 1
                Mv[k] += h12*intrhoz*r11*r32 - h12*intrhoz*r12*r31
                k += 1
                Mv[k] += h12*intrhoz*r21*r32 - h12*intrhoz*r22*r31
                k += 1
                Mv[k] += h12*intrhoz2*r11*r31 + h12*intrhoz2*r12*r32
                k += 1
                Mv[k] += h12*intrhoz2*r21*r31 + h12*intrhoz2*r22*r32
                k += 1
                Mv[k] += h12*intrhoz2*r31**2 + h12*intrhoz2*r32**2
                k += 1
                Mv[k] += h22*intrhoz*r11*r32 - h22*intrhoz*r12*r31
                k += 1
                Mv[k] += h22*intrhoz*r21*r32 - h22*intrhoz*r22*r31
                k += 1
                Mv[k] += h22*intrhoz2*r11*r31 + h22*intrhoz2*r12*r32
                k += 1
                Mv[k] += h22*intrhoz2*r21*r31 + h22*intrhoz2*r22*r32
                k += 1
                Mv[k] += h22*intrhoz2*r31**2 + h22*intrhoz2*r32**2
                k += 1
                Mv[k] += h23*intrhoz*r11*r32 - h23*intrhoz*r12*r31
                k += 1
                Mv[k] += h23*intrhoz*r21*r32 - h23*intrhoz*r22*r31
                k += 1
                Mv[k] += h23*intrhoz2*r11*r31 + h23*intrhoz2*r12*r32
                k += 1
                Mv[k] += h23*intrhoz2*r21*r31 + h23*intrhoz2*r22*r32
                k += 1
                Mv[k] += h23*intrhoz2*r31**2 + h23*intrhoz2*r32**2
                k += 1
                Mv[k] += h24*intrhoz*r11*r32 - h24*intrhoz*r12*r31
                k += 1
                Mv[k] += h24*intrhoz*r21*r32 - h24*intrhoz*r22*r31
                k += 1
                Mv[k] += h24*intrhoz2*r11*r31 + h24*intrhoz2*r12*r32
                k += 1
                Mv[k] += h24*intrhoz2*r21*r31 + h24*intrhoz2*r22*r32
                k += 1
                Mv[k] += h24*intrhoz2*r31**2 + h24*intrhoz2*r32**2
                k += 1
                Mv[k] += h13*intrho*r11**2 + h13*intrho*r12**2 + h13*intrho*r13**2
                k += 1
                Mv[k] += h13*intrho*r11*r21 + h13*intrho*r12*r22 + h13*intrho*r13*r23
                k += 1
                Mv[k] += h13*intrho*r11*r31 + h13*intrho*r12*r32 + h13*intrho*r13*r33
                k += 1
                Mv[k] += h13*intrhoz*r11*r22 - h13*intrhoz*r12*r21
                k += 1
                Mv[k] += h13*intrhoz*r11*r32 - h13*intrhoz*r12*r31
                k += 1
                Mv[k] += h23*intrho*r11**2 + h23*intrho*r12**2 + h23*intrho*r13**2
                k += 1
                Mv[k] += h23*intrho*r11*r21 + h23*intrho*r12*r22 + h23*intrho*r13*r23
                k += 1
                Mv[k] += h23*intrho*r11*r31 + h23*intrho*r12*r32 + h23*intrho*r13*r33
                k += 1
                Mv[k] += h23*intrhoz*r11*r22 - h23*intrhoz*r12*r21
                k += 1
                Mv[k] += h23*intrhoz*r11*r32 - h23*intrhoz*r12*r31
                k += 1
                Mv[k] += h33*intrho*r11**2 + h33*intrho*r12**2 + h33*intrho*r13**2
                k += 1
                Mv[k] += h33*intrho*r11*r21 + h33*intrho*r12*r22 + h33*intrho*r13*r23
                k += 1
                Mv[k] += h33*intrho*r11*r31 + h33*intrho*r12*r32 + h33*intrho*r13*r33
                k += 1
                Mv[k] += h33*intrhoz*r11*r22 - h33*intrhoz*r12*r21
                k += 1
                Mv[k] += h33*intrhoz*r11*r32 - h33*intrhoz*r12*r31
                k += 1
                Mv[k] += h34*intrho*r11**2 + h34*intrho*r12**2 + h34*intrho*r13**2
                k += 1
                Mv[k] += h34*intrho*r11*r21 + h34*intrho*r12*r22 + h34*intrho*r13*r23
                k += 1
                Mv[k] += h34*intrho*r11*r31 + h34*intrho*r12*r32 + h34*intrho*r13*r33
                k += 1
                Mv[k] += h34*intrhoz*r11*r22 - h34*intrhoz*r12*r21
                k += 1
                Mv[k] += h34*intrhoz*r11*r32 - h34*intrhoz*r12*r31
                k += 1
                Mv[k] += h13*intrho*r11*r21 + h13*intrho*r12*r22 + h13*intrho*r13*r23
                k += 1
                Mv[k] += h13*intrho*r21**2 + h13*intrho*r22**2 + h13*intrho*r23**2
                k += 1
                Mv[k] += h13*intrho*r21*r31 + h13*intrho*r22*r32 + h13*intrho*r23*r33
                k += 1
                Mv[k] += -h13*intrhoz*r11*r22 + h13*intrhoz*r12*r21
                k += 1
                Mv[k] += h13*intrhoz*r21*r32 - h13*intrhoz*r22*r31
                k += 1
                Mv[k] += h23*intrho*r11*r21 + h23*intrho*r12*r22 + h23*intrho*r13*r23
                k += 1
                Mv[k] += h23*intrho*r21**2 + h23*intrho*r22**2 + h23*intrho*r23**2
                k += 1
                Mv[k] += h23*intrho*r21*r31 + h23*intrho*r22*r32 + h23*intrho*r23*r33
                k += 1
                Mv[k] += -h23*intrhoz*r11*r22 + h23*intrhoz*r12*r21
                k += 1
                Mv[k] += h23*intrhoz*r21*r32 - h23*intrhoz*r22*r31
                k += 1
                Mv[k] += h33*intrho*r11*r21 + h33*intrho*r12*r22 + h33*intrho*r13*r23
                k += 1
                Mv[k] += h33*intrho*r21**2 + h33*intrho*r22**2 + h33*intrho*r23**2
                k += 1
                Mv[k] += h33*intrho*r21*r31 + h33*intrho*r22*r32 + h33*intrho*r23*r33
                k += 1
                Mv[k] += -h33*intrhoz*r11*r22 + h33*intrhoz*r12*r21
                k += 1
                Mv[k] += h33*intrhoz*r21*r32 - h33*intrhoz*r22*r31
                k += 1
                Mv[k] += h34*intrho*r11*r21 + h34*intrho*r12*r22 + h34*intrho*r13*r23
                k += 1
                Mv[k] += h34*intrho*r21**2 + h34*intrho*r22**2 + h34*intrho*r23**2
                k += 1
                Mv[k] += h34*intrho*r21*r31 + h34*intrho*r22*r32 + h34*intrho*r23*r33
                k += 1
                Mv[k] += -h34*intrhoz*r11*r22 + h34*intrhoz*r12*r21
                k += 1
                Mv[k] += h34*intrhoz*r21*r32 - h34*intrhoz*r22*r31
                k += 1
                Mv[k] += h13*intrho*r11*r31 + h13*intrho*r12*r32 + h13*intrho*r13*r33
                k += 1
                Mv[k] += h13*intrho*r21*r31 + h13*intrho*r22*r32 + h13*intrho*r23*r33
                k += 1
                Mv[k] += h13*intrho*r31**2 + h13*intrho*r32**2 + h13*intrho*r33**2
                k += 1
                Mv[k] += -h13*intrhoz*r11*r32 + h13*intrhoz*r12*r31
                k += 1
                Mv[k] += -h13*intrhoz*r21*r32 + h13*intrhoz*r22*r31
                k += 1
                Mv[k] += h23*intrho*r11*r31 + h23*intrho*r12*r32 + h23*intrho*r13*r33
                k += 1
                Mv[k] += h23*intrho*r21*r31 + h23*intrho*r22*r32 + h23*intrho*r23*r33
                k += 1
                Mv[k] += h23*intrho*r31**2 + h23*intrho*r32**2 + h23*intrho*r33**2
                k += 1
                Mv[k] += -h23*intrhoz*r11*r32 + h23*intrhoz*r12*r31
                k += 1
                Mv[k] += -h23*intrhoz*r21*r32 + h23*intrhoz*r22*r31
                k += 1
                Mv[k] += h33*intrho*r11*r31 + h33*intrho*r12*r32 + h33*intrho*r13*r33
                k += 1
                Mv[k] += h33*intrho*r21*r31 + h33*intrho*r22*r32 + h33*intrho*r23*r33
                k += 1
                Mv[k] += h33*intrho*r31**2 + h33*intrho*r32**2 + h33*intrho*r33**2
                k += 1
                Mv[k] += -h33*intrhoz*r11*r32 + h33*intrhoz*r12*r31
                k += 1
                Mv[k] += -h33*intrhoz*r21*r32 + h33*intrhoz*r22*r31
                k += 1
                Mv[k] += h34*intrho*r11*r31 + h34*intrho*r12*r32 + h34*intrho*r13*r33
                k += 1
                Mv[k] += h34*intrho*r21*r31 + h34*intrho*r22*r32 + h34*intrho*r23*r33
                k += 1
                Mv[k] += h34*intrho*r31**2 + h34*intrho*r32**2 + h34*intrho*r33**2
                k += 1
                Mv[k] += -h34*intrhoz*r11*r32 + h34*intrhoz*r12*r31
                k += 1
                Mv[k] += -h34*intrhoz*r21*r32 + h34*intrhoz*r22*r31
                k += 1
                Mv[k] += -h13*intrhoz*r11*r22 + h13*intrhoz*r12*r21
                k += 1
                Mv[k] += -h13*intrhoz*r11*r32 + h13*intrhoz*r12*r31
                k += 1
                Mv[k] += h13*intrhoz2*r11**2 + h13*intrhoz2*r12**2
                k += 1
                Mv[k] += h13*intrhoz2*r11*r21 + h13*intrhoz2*r12*r22
                k += 1
                Mv[k] += h13*intrhoz2*r11*r31 + h13*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h23*intrhoz*r11*r22 + h23*intrhoz*r12*r21
                k += 1
                Mv[k] += -h23*intrhoz*r11*r32 + h23*intrhoz*r12*r31
                k += 1
                Mv[k] += h23*intrhoz2*r11**2 + h23*intrhoz2*r12**2
                k += 1
                Mv[k] += h23*intrhoz2*r11*r21 + h23*intrhoz2*r12*r22
                k += 1
                Mv[k] += h23*intrhoz2*r11*r31 + h23*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h33*intrhoz*r11*r22 + h33*intrhoz*r12*r21
                k += 1
                Mv[k] += -h33*intrhoz*r11*r32 + h33*intrhoz*r12*r31
                k += 1
                Mv[k] += h33*intrhoz2*r11**2 + h33*intrhoz2*r12**2
                k += 1
                Mv[k] += h33*intrhoz2*r11*r21 + h33*intrhoz2*r12*r22
                k += 1
                Mv[k] += h33*intrhoz2*r11*r31 + h33*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h34*intrhoz*r11*r22 + h34*intrhoz*r12*r21
                k += 1
                Mv[k] += -h34*intrhoz*r11*r32 + h34*intrhoz*r12*r31
                k += 1
                Mv[k] += h34*intrhoz2*r11**2 + h34*intrhoz2*r12**2
                k += 1
                Mv[k] += h34*intrhoz2*r11*r21 + h34*intrhoz2*r12*r22
                k += 1
                Mv[k] += h34*intrhoz2*r11*r31 + h34*intrhoz2*r12*r32
                k += 1
                Mv[k] += h13*intrhoz*r11*r22 - h13*intrhoz*r12*r21
                k += 1
                Mv[k] += -h13*intrhoz*r21*r32 + h13*intrhoz*r22*r31
                k += 1
                Mv[k] += h13*intrhoz2*r11*r21 + h13*intrhoz2*r12*r22
                k += 1
                Mv[k] += h13*intrhoz2*r21**2 + h13*intrhoz2*r22**2
                k += 1
                Mv[k] += h13*intrhoz2*r21*r31 + h13*intrhoz2*r22*r32
                k += 1
                Mv[k] += h23*intrhoz*r11*r22 - h23*intrhoz*r12*r21
                k += 1
                Mv[k] += -h23*intrhoz*r21*r32 + h23*intrhoz*r22*r31
                k += 1
                Mv[k] += h23*intrhoz2*r11*r21 + h23*intrhoz2*r12*r22
                k += 1
                Mv[k] += h23*intrhoz2*r21**2 + h23*intrhoz2*r22**2
                k += 1
                Mv[k] += h23*intrhoz2*r21*r31 + h23*intrhoz2*r22*r32
                k += 1
                Mv[k] += h33*intrhoz*r11*r22 - h33*intrhoz*r12*r21
                k += 1
                Mv[k] += -h33*intrhoz*r21*r32 + h33*intrhoz*r22*r31
                k += 1
                Mv[k] += h33*intrhoz2*r11*r21 + h33*intrhoz2*r12*r22
                k += 1
                Mv[k] += h33*intrhoz2*r21**2 + h33*intrhoz2*r22**2
                k += 1
                Mv[k] += h33*intrhoz2*r21*r31 + h33*intrhoz2*r22*r32
                k += 1
                Mv[k] += h34*intrhoz*r11*r22 - h34*intrhoz*r12*r21
                k += 1
                Mv[k] += -h34*intrhoz*r21*r32 + h34*intrhoz*r22*r31
                k += 1
                Mv[k] += h34*intrhoz2*r11*r21 + h34*intrhoz2*r12*r22
                k += 1
                Mv[k] += h34*intrhoz2*r21**2 + h34*intrhoz2*r22**2
                k += 1
                Mv[k] += h34*intrhoz2*r21*r31 + h34*intrhoz2*r22*r32
                k += 1
                Mv[k] += h13*intrhoz*r11*r32 - h13*intrhoz*r12*r31
                k += 1
                Mv[k] += h13*intrhoz*r21*r32 - h13*intrhoz*r22*r31
                k += 1
                Mv[k] += h13*intrhoz2*r11*r31 + h13*intrhoz2*r12*r32
                k += 1
                Mv[k] += h13*intrhoz2*r21*r31 + h13*intrhoz2*r22*r32
                k += 1
                Mv[k] += h13*intrhoz2*r31**2 + h13*intrhoz2*r32**2
                k += 1
                Mv[k] += h23*intrhoz*r11*r32 - h23*intrhoz*r12*r31
                k += 1
                Mv[k] += h23*intrhoz*r21*r32 - h23*intrhoz*r22*r31
                k += 1
                Mv[k] += h23*intrhoz2*r11*r31 + h23*intrhoz2*r12*r32
                k += 1
                Mv[k] += h23*intrhoz2*r21*r31 + h23*intrhoz2*r22*r32
                k += 1
                Mv[k] += h23*intrhoz2*r31**2 + h23*intrhoz2*r32**2
                k += 1
                Mv[k] += h33*intrhoz*r11*r32 - h33*intrhoz*r12*r31
                k += 1
                Mv[k] += h33*intrhoz*r21*r32 - h33*intrhoz*r22*r31
                k += 1
                Mv[k] += h33*intrhoz2*r11*r31 + h33*intrhoz2*r12*r32
                k += 1
                Mv[k] += h33*intrhoz2*r21*r31 + h33*intrhoz2*r22*r32
                k += 1
                Mv[k] += h33*intrhoz2*r31**2 + h33*intrhoz2*r32**2
                k += 1
                Mv[k] += h34*intrhoz*r11*r32 - h34*intrhoz*r12*r31
                k += 1
                Mv[k] += h34*intrhoz*r21*r32 - h34*intrhoz*r22*r31
                k += 1
                Mv[k] += h34*intrhoz2*r11*r31 + h34*intrhoz2*r12*r32
                k += 1
                Mv[k] += h34*intrhoz2*r21*r31 + h34*intrhoz2*r22*r32
                k += 1
                Mv[k] += h34*intrhoz2*r31**2 + h34*intrhoz2*r32**2
                k += 1
                Mv[k] += h14*intrho*r11**2 + h14*intrho*r12**2 + h14*intrho*r13**2
                k += 1
                Mv[k] += h14*intrho*r11*r21 + h14*intrho*r12*r22 + h14*intrho*r13*r23
                k += 1
                Mv[k] += h14*intrho*r11*r31 + h14*intrho*r12*r32 + h14*intrho*r13*r33
                k += 1
                Mv[k] += h14*intrhoz*r11*r22 - h14*intrhoz*r12*r21
                k += 1
                Mv[k] += h14*intrhoz*r11*r32 - h14*intrhoz*r12*r31
                k += 1
                Mv[k] += h24*intrho*r11**2 + h24*intrho*r12**2 + h24*intrho*r13**2
                k += 1
                Mv[k] += h24*intrho*r11*r21 + h24*intrho*r12*r22 + h24*intrho*r13*r23
                k += 1
                Mv[k] += h24*intrho*r11*r31 + h24*intrho*r12*r32 + h24*intrho*r13*r33
                k += 1
                Mv[k] += h24*intrhoz*r11*r22 - h24*intrhoz*r12*r21
                k += 1
                Mv[k] += h24*intrhoz*r11*r32 - h24*intrhoz*r12*r31
                k += 1
                Mv[k] += h34*intrho*r11**2 + h34*intrho*r12**2 + h34*intrho*r13**2
                k += 1
                Mv[k] += h34*intrho*r11*r21 + h34*intrho*r12*r22 + h34*intrho*r13*r23
                k += 1
                Mv[k] += h34*intrho*r11*r31 + h34*intrho*r12*r32 + h34*intrho*r13*r33
                k += 1
                Mv[k] += h34*intrhoz*r11*r22 - h34*intrhoz*r12*r21
                k += 1
                Mv[k] += h34*intrhoz*r11*r32 - h34*intrhoz*r12*r31
                k += 1
                Mv[k] += h44*intrho*r11**2 + h44*intrho*r12**2 + h44*intrho*r13**2
                k += 1
                Mv[k] += h44*intrho*r11*r21 + h44*intrho*r12*r22 + h44*intrho*r13*r23
                k += 1
                Mv[k] += h44*intrho*r11*r31 + h44*intrho*r12*r32 + h44*intrho*r13*r33
                k += 1
                Mv[k] += h44*intrhoz*r11*r22 - h44*intrhoz*r12*r21
                k += 1
                Mv[k] += h44*intrhoz*r11*r32 - h44*intrhoz*r12*r31
                k += 1
                Mv[k] += h14*intrho*r11*r21 + h14*intrho*r12*r22 + h14*intrho*r13*r23
                k += 1
                Mv[k] += h14*intrho*r21**2 + h14*intrho*r22**2 + h14*intrho*r23**2
                k += 1
                Mv[k] += h14*intrho*r21*r31 + h14*intrho*r22*r32 + h14*intrho*r23*r33
                k += 1
                Mv[k] += -h14*intrhoz*r11*r22 + h14*intrhoz*r12*r21
                k += 1
                Mv[k] += h14*intrhoz*r21*r32 - h14*intrhoz*r22*r31
                k += 1
                Mv[k] += h24*intrho*r11*r21 + h24*intrho*r12*r22 + h24*intrho*r13*r23
                k += 1
                Mv[k] += h24*intrho*r21**2 + h24*intrho*r22**2 + h24*intrho*r23**2
                k += 1
                Mv[k] += h24*intrho*r21*r31 + h24*intrho*r22*r32 + h24*intrho*r23*r33
                k += 1
                Mv[k] += -h24*intrhoz*r11*r22 + h24*intrhoz*r12*r21
                k += 1
                Mv[k] += h24*intrhoz*r21*r32 - h24*intrhoz*r22*r31
                k += 1
                Mv[k] += h34*intrho*r11*r21 + h34*intrho*r12*r22 + h34*intrho*r13*r23
                k += 1
                Mv[k] += h34*intrho*r21**2 + h34*intrho*r22**2 + h34*intrho*r23**2
                k += 1
                Mv[k] += h34*intrho*r21*r31 + h34*intrho*r22*r32 + h34*intrho*r23*r33
                k += 1
                Mv[k] += -h34*intrhoz*r11*r22 + h34*intrhoz*r12*r21
                k += 1
                Mv[k] += h34*intrhoz*r21*r32 - h34*intrhoz*r22*r31
                k += 1
                Mv[k] += h44*intrho*r11*r21 + h44*intrho*r12*r22 + h44*intrho*r13*r23
                k += 1
                Mv[k] += h44*intrho*r21**2 + h44*intrho*r22**2 + h44*intrho*r23**2
                k += 1
                Mv[k] += h44*intrho*r21*r31 + h44*intrho*r22*r32 + h44*intrho*r23*r33
                k += 1
                Mv[k] += -h44*intrhoz*r11*r22 + h44*intrhoz*r12*r21
                k += 1
                Mv[k] += h44*intrhoz*r21*r32 - h44*intrhoz*r22*r31
                k += 1
                Mv[k] += h14*intrho*r11*r31 + h14*intrho*r12*r32 + h14*intrho*r13*r33
                k += 1
                Mv[k] += h14*intrho*r21*r31 + h14*intrho*r22*r32 + h14*intrho*r23*r33
                k += 1
                Mv[k] += h14*intrho*r31**2 + h14*intrho*r32**2 + h14*intrho*r33**2
                k += 1
                Mv[k] += -h14*intrhoz*r11*r32 + h14*intrhoz*r12*r31
                k += 1
                Mv[k] += -h14*intrhoz*r21*r32 + h14*intrhoz*r22*r31
                k += 1
                Mv[k] += h24*intrho*r11*r31 + h24*intrho*r12*r32 + h24*intrho*r13*r33
                k += 1
                Mv[k] += h24*intrho*r21*r31 + h24*intrho*r22*r32 + h24*intrho*r23*r33
                k += 1
                Mv[k] += h24*intrho*r31**2 + h24*intrho*r32**2 + h24*intrho*r33**2
                k += 1
                Mv[k] += -h24*intrhoz*r11*r32 + h24*intrhoz*r12*r31
                k += 1
                Mv[k] += -h24*intrhoz*r21*r32 + h24*intrhoz*r22*r31
                k += 1
                Mv[k] += h34*intrho*r11*r31 + h34*intrho*r12*r32 + h34*intrho*r13*r33
                k += 1
                Mv[k] += h34*intrho*r21*r31 + h34*intrho*r22*r32 + h34*intrho*r23*r33
                k += 1
                Mv[k] += h34*intrho*r31**2 + h34*intrho*r32**2 + h34*intrho*r33**2
                k += 1
                Mv[k] += -h34*intrhoz*r11*r32 + h34*intrhoz*r12*r31
                k += 1
                Mv[k] += -h34*intrhoz*r21*r32 + h34*intrhoz*r22*r31
                k += 1
                Mv[k] += h44*intrho*r11*r31 + h44*intrho*r12*r32 + h44*intrho*r13*r33
                k += 1
                Mv[k] += h44*intrho*r21*r31 + h44*intrho*r22*r32 + h44*intrho*r23*r33
                k += 1
                Mv[k] += h44*intrho*r31**2 + h44*intrho*r32**2 + h44*intrho*r33**2
                k += 1
                Mv[k] += -h44*intrhoz*r11*r32 + h44*intrhoz*r12*r31
                k += 1
                Mv[k] += -h44*intrhoz*r21*r32 + h44*intrhoz*r22*r31
                k += 1
                Mv[k] += -h14*intrhoz*r11*r22 + h14*intrhoz*r12*r21
                k += 1
                Mv[k] += -h14*intrhoz*r11*r32 + h14*intrhoz*r12*r31
                k += 1
                Mv[k] += h14*intrhoz2*r11**2 + h14*intrhoz2*r12**2
                k += 1
                Mv[k] += h14*intrhoz2*r11*r21 + h14*intrhoz2*r12*r22
                k += 1
                Mv[k] += h14*intrhoz2*r11*r31 + h14*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h24*intrhoz*r11*r22 + h24*intrhoz*r12*r21
                k += 1
                Mv[k] += -h24*intrhoz*r11*r32 + h24*intrhoz*r12*r31
                k += 1
                Mv[k] += h24*intrhoz2*r11**2 + h24*intrhoz2*r12**2
                k += 1
                Mv[k] += h24*intrhoz2*r11*r21 + h24*intrhoz2*r12*r22
                k += 1
                Mv[k] += h24*intrhoz2*r11*r31 + h24*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h34*intrhoz*r11*r22 + h34*intrhoz*r12*r21
                k += 1
                Mv[k] += -h34*intrhoz*r11*r32 + h34*intrhoz*r12*r31
                k += 1
                Mv[k] += h34*intrhoz2*r11**2 + h34*intrhoz2*r12**2
                k += 1
                Mv[k] += h34*intrhoz2*r11*r21 + h34*intrhoz2*r12*r22
                k += 1
                Mv[k] += h34*intrhoz2*r11*r31 + h34*intrhoz2*r12*r32
                k += 1
                Mv[k] += -h44*intrhoz*r11*r22 + h44*intrhoz*r12*r21
                k += 1
                Mv[k] += -h44*intrhoz*r11*r32 + h44*intrhoz*r12*r31
                k += 1
                Mv[k] += h44*intrhoz2*r11**2 + h44*intrhoz2*r12**2
                k += 1
                Mv[k] += h44*intrhoz2*r11*r21 + h44*intrhoz2*r12*r22
                k += 1
                Mv[k] += h44*intrhoz2*r11*r31 + h44*intrhoz2*r12*r32
                k += 1
                Mv[k] += h14*intrhoz*r11*r22 - h14*intrhoz*r12*r21
                k += 1
                Mv[k] += -h14*intrhoz*r21*r32 + h14*intrhoz*r22*r31
                k += 1
                Mv[k] += h14*intrhoz2*r11*r21 + h14*intrhoz2*r12*r22
                k += 1
                Mv[k] += h14*intrhoz2*r21**2 + h14*intrhoz2*r22**2
                k += 1
                Mv[k] += h14*intrhoz2*r21*r31 + h14*intrhoz2*r22*r32
                k += 1
                Mv[k] += h24*intrhoz*r11*r22 - h24*intrhoz*r12*r21
                k += 1
                Mv[k] += -h24*intrhoz*r21*r32 + h24*intrhoz*r22*r31
                k += 1
                Mv[k] += h24*intrhoz2*r11*r21 + h24*intrhoz2*r12*r22
                k += 1
                Mv[k] += h24*intrhoz2*r21**2 + h24*intrhoz2*r22**2
                k += 1
                Mv[k] += h24*intrhoz2*r21*r31 + h24*intrhoz2*r22*r32
                k += 1
                Mv[k] += h34*intrhoz*r11*r22 - h34*intrhoz*r12*r21
                k += 1
                Mv[k] += -h34*intrhoz*r21*r32 + h34*intrhoz*r22*r31
                k += 1
                Mv[k] += h34*intrhoz2*r11*r21 + h34*intrhoz2*r12*r22
                k += 1
                Mv[k] += h34*intrhoz2*r21**2 + h34*intrhoz2*r22**2
                k += 1
                Mv[k] += h34*intrhoz2*r21*r31 + h34*intrhoz2*r22*r32
                k += 1
                Mv[k] += h44*intrhoz*r11*r22 - h44*intrhoz*r12*r21
                k += 1
                Mv[k] += -h44*intrhoz*r21*r32 + h44*intrhoz*r22*r31
                k += 1
                Mv[k] += h44*intrhoz2*r11*r21 + h44*intrhoz2*r12*r22
                k += 1
                Mv[k] += h44*intrhoz2*r21**2 + h44*intrhoz2*r22**2
                k += 1
                Mv[k] += h44*intrhoz2*r21*r31 + h44*intrhoz2*r22*r32
                k += 1
                Mv[k] += h14*intrhoz*r11*r32 - h14*intrhoz*r12*r31
                k += 1
                Mv[k] += h14*intrhoz*r21*r32 - h14*intrhoz*r22*r31
                k += 1
                Mv[k] += h14*intrhoz2*r11*r31 + h14*intrhoz2*r12*r32
                k += 1
                Mv[k] += h14*intrhoz2*r21*r31 + h14*intrhoz2*r22*r32
                k += 1
                Mv[k] += h14*intrhoz2*r31**2 + h14*intrhoz2*r32**2
                k += 1
                Mv[k] += h24*intrhoz*r11*r32 - h24*intrhoz*r12*r31
                k += 1
                Mv[k] += h24*intrhoz*r21*r32 - h24*intrhoz*r22*r31
                k += 1
                Mv[k] += h24*intrhoz2*r11*r31 + h24*intrhoz2*r12*r32
                k += 1
                Mv[k] += h24*intrhoz2*r21*r31 + h24*intrhoz2*r22*r32
                k += 1
                Mv[k] += h24*intrhoz2*r31**2 + h24*intrhoz2*r32**2
                k += 1
                Mv[k] += h34*intrhoz*r11*r32 - h34*intrhoz*r12*r31
                k += 1
                Mv[k] += h34*intrhoz*r21*r32 - h34*intrhoz*r22*r31
                k += 1
                Mv[k] += h34*intrhoz2*r11*r31 + h34*intrhoz2*r12*r32
                k += 1
                Mv[k] += h34*intrhoz2*r21*r31 + h34*intrhoz2*r22*r32
                k += 1
                Mv[k] += h34*intrhoz2*r31**2 + h34*intrhoz2*r32**2
                k += 1
                Mv[k] += h44*intrhoz*r11*r32 - h44*intrhoz*r12*r31
                k += 1
                Mv[k] += h44*intrhoz*r21*r32 - h44*intrhoz*r22*r31
                k += 1
                Mv[k] += h44*intrhoz2*r11*r31 + h44*intrhoz2*r12*r32
                k += 1
                Mv[k] += h44*intrhoz2*r21*r31 + h44*intrhoz2*r22*r32
                k += 1
                Mv[k] += h44*intrhoz2*r31**2 + h44*intrhoz2*r32**2

            elif mtype == 1: # M_red mass matrix purely by reduced integration
                k = self.init_k_M
                Mr[k] = 0+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c4

                k = self.init_k_M
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11**2*valH1 + intrho*r12**2*valH1 + intrho*r13**2*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r21*valH1 + intrho*r12*r22*valH1 + intrho*r13*r23*valH1
                k += 1
                Mv[k] += intrho*r21**2*valH1 + intrho*r22**2*valH1 + intrho*r23**2*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrho*r11*r31*valH1 + intrho*r12*r32*valH1 + intrho*r13*r33*valH1
                k += 1
                Mv[k] += intrho*r21*r31*valH1 + intrho*r22*r32*valH1 + intrho*r23*r33*valH1
                k += 1
                Mv[k] += intrho*r31**2*valH1 + intrho*r32**2*valH1 + intrho*r33**2*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r22*valH1 + intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r11*r32*valH1 + intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11**2*valH1 + intrhoz2*r12**2*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r22*valH1 - intrhoz*r12*r21*valH1
                k += 1
                Mv[k] += -intrhoz*r21*r32*valH1 + intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r21*valH1 + intrhoz2*r12*r22*valH1
                k += 1
                Mv[k] += intrhoz2*r21**2*valH1 + intrhoz2*r22**2*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1
                k += 1
                Mv[k] += intrhoz*r11*r32*valH1 - intrhoz*r12*r31*valH1
                k += 1
                Mv[k] += intrhoz*r21*r32*valH1 - intrhoz*r22*r31*valH1
                k += 1
                Mv[k] += intrhoz2*r11*r31*valH1 + intrhoz2*r12*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r21*r31*valH1 + intrhoz2*r22*r32*valH1
                k += 1
                Mv[k] += intrhoz2*r31**2*valH1 + intrhoz2*r32**2*valH1

            elif mtype == 2: # M_lump lumped mass matrix using two-point Gauss-Lobatto quadrature
                k = self.init_k_M
                Mr[k] = 0+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c1
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c1
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c2
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c2
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c3
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c3
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 0+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 1+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c1
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c2
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c3
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 0+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 1+c4
                k += 1
                Mr[k] = 2+c4
                Mc[k] = 2+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 3+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 4+c4
                Mc[k] = 5+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c1
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c2
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c3
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 3+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 4+c4
                k += 1
                Mr[k] = 5+c4
                Mc[k] = 5+c4

                wij = 1.
                # NOTE two-point Gauss-Lobatto quadrature
                points[0] = -1.
                points[1] = +1.
                h11 = 0.
                h12 = 0.
                h13 = 0.
                h14 = 0.
                h22 = 0.
                h23 = 0.
                h24 = 0.
                h33 = 0.
                h34 = 0.
                h44 = 0.
                for i in range(2):
                    xi = points[i]
                    for j in range(2):
                        eta = points[j]

                        J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
                        J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
                        J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
                        J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

                        detJ = J11*J22 - J12*J21

                        N1 = eta*xi/4. - eta/4. - xi/4. + 1/4.
                        N2 = -eta*xi/4. - eta/4. + xi/4. + 1/4.
                        N3 = eta*xi/4. + eta/4. + xi/4. + 1/4.
                        N4 = -eta*xi/4. + eta/4. - xi/4. + 1/4.

                        h11 += N1**2*detJ*wij
                        h12 += N1*N2*detJ*wij
                        h13 += N1*N3*detJ*wij
                        h14 += N1*N4*detJ*wij
                        h22 += N2**2*detJ*wij
                        h23 += N2*N3*detJ*wij
                        h24 += N2*N4*detJ*wij
                        h33 += N3**2*detJ*wij
                        h34 += N3*N4*detJ*wij
                        h44 += N4**2*detJ*wij

                k = self.init_k_M
                Mv[k] += h11*intrho*r11**2 + h11*intrho*r12**2 + h11*intrho*r13**2
                k += 1
                Mv[k] += h11*intrho*r11*r21 + h11*intrho*r12*r22 + h11*intrho*r13*r23
                k += 1
                Mv[k] += h11*intrho*r11*r31 + h11*intrho*r12*r32 + h11*intrho*r13*r33
                k += 1
                Mv[k] += h12*intrho*r11**2 + h12*intrho*r12**2 + h12*intrho*r13**2
                k += 1
                Mv[k] += h12*intrho*r11*r21 + h12*intrho*r12*r22 + h12*intrho*r13*r23
                k += 1
                Mv[k] += h12*intrho*r11*r31 + h12*intrho*r12*r32 + h12*intrho*r13*r33
                k += 1
                Mv[k] += h13*intrho*r11**2 + h13*intrho*r12**2 + h13*intrho*r13**2
                k += 1
                Mv[k] += h13*intrho*r11*r21 + h13*intrho*r12*r22 + h13*intrho*r13*r23
                k += 1
                Mv[k] += h13*intrho*r11*r31 + h13*intrho*r12*r32 + h13*intrho*r13*r33
                k += 1
                Mv[k] += h14*intrho*r11**2 + h14*intrho*r12**2 + h14*intrho*r13**2
                k += 1
                Mv[k] += h14*intrho*r11*r21 + h14*intrho*r12*r22 + h14*intrho*r13*r23
                k += 1
                Mv[k] += h14*intrho*r11*r31 + h14*intrho*r12*r32 + h14*intrho*r13*r33
                k += 1
                Mv[k] += h11*intrho*r11*r21 + h11*intrho*r12*r22 + h11*intrho*r13*r23
                k += 1
                Mv[k] += h11*intrho*r21**2 + h11*intrho*r22**2 + h11*intrho*r23**2
                k += 1
                Mv[k] += h11*intrho*r21*r31 + h11*intrho*r22*r32 + h11*intrho*r23*r33
                k += 1
                Mv[k] += h12*intrho*r11*r21 + h12*intrho*r12*r22 + h12*intrho*r13*r23
                k += 1
                Mv[k] += h12*intrho*r21**2 + h12*intrho*r22**2 + h12*intrho*r23**2
                k += 1
                Mv[k] += h12*intrho*r21*r31 + h12*intrho*r22*r32 + h12*intrho*r23*r33
                k += 1
                Mv[k] += h13*intrho*r11*r21 + h13*intrho*r12*r22 + h13*intrho*r13*r23
                k += 1
                Mv[k] += h13*intrho*r21**2 + h13*intrho*r22**2 + h13*intrho*r23**2
                k += 1
                Mv[k] += h13*intrho*r21*r31 + h13*intrho*r22*r32 + h13*intrho*r23*r33
                k += 1
                Mv[k] += h14*intrho*r11*r21 + h14*intrho*r12*r22 + h14*intrho*r13*r23
                k += 1
                Mv[k] += h14*intrho*r21**2 + h14*intrho*r22**2 + h14*intrho*r23**2
                k += 1
                Mv[k] += h14*intrho*r21*r31 + h14*intrho*r22*r32 + h14*intrho*r23*r33
                k += 1
                Mv[k] += h11*intrho*r11*r31 + h11*intrho*r12*r32 + h11*intrho*r13*r33
                k += 1
                Mv[k] += h11*intrho*r21*r31 + h11*intrho*r22*r32 + h11*intrho*r23*r33
                k += 1
                Mv[k] += h11*intrho*r31**2 + h11*intrho*r32**2 + h11*intrho*r33**2
                k += 1
                Mv[k] += h12*intrho*r11*r31 + h12*intrho*r12*r32 + h12*intrho*r13*r33
                k += 1
                Mv[k] += h12*intrho*r21*r31 + h12*intrho*r22*r32 + h12*intrho*r23*r33
                k += 1
                Mv[k] += h12*intrho*r31**2 + h12*intrho*r32**2 + h12*intrho*r33**2
                k += 1
                Mv[k] += h13*intrho*r11*r31 + h13*intrho*r12*r32 + h13*intrho*r13*r33
                k += 1
                Mv[k] += h13*intrho*r21*r31 + h13*intrho*r22*r32 + h13*intrho*r23*r33
                k += 1
                Mv[k] += h13*intrho*r31**2 + h13*intrho*r32**2 + h13*intrho*r33**2
                k += 1
                Mv[k] += h14*intrho*r11*r31 + h14*intrho*r12*r32 + h14*intrho*r13*r33
                k += 1
                Mv[k] += h14*intrho*r21*r31 + h14*intrho*r22*r32 + h14*intrho*r23*r33
                k += 1
                Mv[k] += h14*intrho*r31**2 + h14*intrho*r32**2 + h14*intrho*r33**2
                k += 1
                Mv[k] += h11*intrhoz2*r11**2 + h11*intrhoz2*r12**2
                k += 1
                Mv[k] += h11*intrhoz2*r11*r21 + h11*intrhoz2*r12*r22
                k += 1
                Mv[k] += h11*intrhoz2*r11*r31 + h11*intrhoz2*r12*r32
                k += 1
                Mv[k] += h12*intrhoz2*r11**2 + h12*intrhoz2*r12**2
                k += 1
                Mv[k] += h12*intrhoz2*r11*r21 + h12*intrhoz2*r12*r22
                k += 1
                Mv[k] += h12*intrhoz2*r11*r31 + h12*intrhoz2*r12*r32
                k += 1
                Mv[k] += h13*intrhoz2*r11**2 + h13*intrhoz2*r12**2
                k += 1
                Mv[k] += h13*intrhoz2*r11*r21 + h13*intrhoz2*r12*r22
                k += 1
                Mv[k] += h13*intrhoz2*r11*r31 + h13*intrhoz2*r12*r32
                k += 1
                Mv[k] += h14*intrhoz2*r11**2 + h14*intrhoz2*r12**2
                k += 1
                Mv[k] += h14*intrhoz2*r11*r21 + h14*intrhoz2*r12*r22
                k += 1
                Mv[k] += h14*intrhoz2*r11*r31 + h14*intrhoz2*r12*r32
                k += 1
                Mv[k] += h11*intrhoz2*r11*r21 + h11*intrhoz2*r12*r22
                k += 1
                Mv[k] += h11*intrhoz2*r21**2 + h11*intrhoz2*r22**2
                k += 1
                Mv[k] += h11*intrhoz2*r21*r31 + h11*intrhoz2*r22*r32
                k += 1
                Mv[k] += h12*intrhoz2*r11*r21 + h12*intrhoz2*r12*r22
                k += 1
                Mv[k] += h12*intrhoz2*r21**2 + h12*intrhoz2*r22**2
                k += 1
                Mv[k] += h12*intrhoz2*r21*r31 + h12*intrhoz2*r22*r32
                k += 1
                Mv[k] += h13*intrhoz2*r11*r21 + h13*intrhoz2*r12*r22
                k += 1
                Mv[k] += h13*intrhoz2*r21**2 + h13*intrhoz2*r22**2
                k += 1
                Mv[k] += h13*intrhoz2*r21*r31 + h13*intrhoz2*r22*r32
                k += 1
                Mv[k] += h14*intrhoz2*r11*r21 + h14*intrhoz2*r12*r22
                k += 1
                Mv[k] += h14*intrhoz2*r21**2 + h14*intrhoz2*r22**2
                k += 1
                Mv[k] += h14*intrhoz2*r21*r31 + h14*intrhoz2*r22*r32
                k += 1
                Mv[k] += h11*intrhoz2*r11*r31 + h11*intrhoz2*r12*r32
                k += 1
                Mv[k] += h11*intrhoz2*r21*r31 + h11*intrhoz2*r22*r32
                k += 1
                Mv[k] += h11*intrhoz2*r31**2 + h11*intrhoz2*r32**2
                k += 1
                Mv[k] += h12*intrhoz2*r11*r31 + h12*intrhoz2*r12*r32
                k += 1
                Mv[k] += h12*intrhoz2*r21*r31 + h12*intrhoz2*r22*r32
                k += 1
                Mv[k] += h12*intrhoz2*r31**2 + h12*intrhoz2*r32**2
                k += 1
                Mv[k] += h13*intrhoz2*r11*r31 + h13*intrhoz2*r12*r32
                k += 1
                Mv[k] += h13*intrhoz2*r21*r31 + h13*intrhoz2*r22*r32
                k += 1
                Mv[k] += h13*intrhoz2*r31**2 + h13*intrhoz2*r32**2
                k += 1
                Mv[k] += h14*intrhoz2*r11*r31 + h14*intrhoz2*r12*r32
                k += 1
                Mv[k] += h14*intrhoz2*r21*r31 + h14*intrhoz2*r22*r32
                k += 1
                Mv[k] += h14*intrhoz2*r31**2 + h14*intrhoz2*r32**2
                k += 1
                Mv[k] += h12*intrho*r11**2 + h12*intrho*r12**2 + h12*intrho*r13**2
                k += 1
                Mv[k] += h12*intrho*r11*r21 + h12*intrho*r12*r22 + h12*intrho*r13*r23
                k += 1
                Mv[k] += h12*intrho*r11*r31 + h12*intrho*r12*r32 + h12*intrho*r13*r33
                k += 1
                Mv[k] += h22*intrho*r11**2 + h22*intrho*r12**2 + h22*intrho*r13**2
                k += 1
                Mv[k] += h22*intrho*r11*r21 + h22*intrho*r12*r22 + h22*intrho*r13*r23
                k += 1
                Mv[k] += h22*intrho*r11*r31 + h22*intrho*r12*r32 + h22*intrho*r13*r33
                k += 1
                Mv[k] += h23*intrho*r11**2 + h23*intrho*r12**2 + h23*intrho*r13**2
                k += 1
                Mv[k] += h23*intrho*r11*r21 + h23*intrho*r12*r22 + h23*intrho*r13*r23
                k += 1
                Mv[k] += h23*intrho*r11*r31 + h23*intrho*r12*r32 + h23*intrho*r13*r33
                k += 1
                Mv[k] += h24*intrho*r11**2 + h24*intrho*r12**2 + h24*intrho*r13**2
                k += 1
                Mv[k] += h24*intrho*r11*r21 + h24*intrho*r12*r22 + h24*intrho*r13*r23
                k += 1
                Mv[k] += h24*intrho*r11*r31 + h24*intrho*r12*r32 + h24*intrho*r13*r33
                k += 1
                Mv[k] += h12*intrho*r11*r21 + h12*intrho*r12*r22 + h12*intrho*r13*r23
                k += 1
                Mv[k] += h12*intrho*r21**2 + h12*intrho*r22**2 + h12*intrho*r23**2
                k += 1
                Mv[k] += h12*intrho*r21*r31 + h12*intrho*r22*r32 + h12*intrho*r23*r33
                k += 1
                Mv[k] += h22*intrho*r11*r21 + h22*intrho*r12*r22 + h22*intrho*r13*r23
                k += 1
                Mv[k] += h22*intrho*r21**2 + h22*intrho*r22**2 + h22*intrho*r23**2
                k += 1
                Mv[k] += h22*intrho*r21*r31 + h22*intrho*r22*r32 + h22*intrho*r23*r33
                k += 1
                Mv[k] += h23*intrho*r11*r21 + h23*intrho*r12*r22 + h23*intrho*r13*r23
                k += 1
                Mv[k] += h23*intrho*r21**2 + h23*intrho*r22**2 + h23*intrho*r23**2
                k += 1
                Mv[k] += h23*intrho*r21*r31 + h23*intrho*r22*r32 + h23*intrho*r23*r33
                k += 1
                Mv[k] += h24*intrho*r11*r21 + h24*intrho*r12*r22 + h24*intrho*r13*r23
                k += 1
                Mv[k] += h24*intrho*r21**2 + h24*intrho*r22**2 + h24*intrho*r23**2
                k += 1
                Mv[k] += h24*intrho*r21*r31 + h24*intrho*r22*r32 + h24*intrho*r23*r33
                k += 1
                Mv[k] += h12*intrho*r11*r31 + h12*intrho*r12*r32 + h12*intrho*r13*r33
                k += 1
                Mv[k] += h12*intrho*r21*r31 + h12*intrho*r22*r32 + h12*intrho*r23*r33
                k += 1
                Mv[k] += h12*intrho*r31**2 + h12*intrho*r32**2 + h12*intrho*r33**2
                k += 1
                Mv[k] += h22*intrho*r11*r31 + h22*intrho*r12*r32 + h22*intrho*r13*r33
                k += 1
                Mv[k] += h22*intrho*r21*r31 + h22*intrho*r22*r32 + h22*intrho*r23*r33
                k += 1
                Mv[k] += h22*intrho*r31**2 + h22*intrho*r32**2 + h22*intrho*r33**2
                k += 1
                Mv[k] += h23*intrho*r11*r31 + h23*intrho*r12*r32 + h23*intrho*r13*r33
                k += 1
                Mv[k] += h23*intrho*r21*r31 + h23*intrho*r22*r32 + h23*intrho*r23*r33
                k += 1
                Mv[k] += h23*intrho*r31**2 + h23*intrho*r32**2 + h23*intrho*r33**2
                k += 1
                Mv[k] += h24*intrho*r11*r31 + h24*intrho*r12*r32 + h24*intrho*r13*r33
                k += 1
                Mv[k] += h24*intrho*r21*r31 + h24*intrho*r22*r32 + h24*intrho*r23*r33
                k += 1
                Mv[k] += h24*intrho*r31**2 + h24*intrho*r32**2 + h24*intrho*r33**2
                k += 1
                Mv[k] += h12*intrhoz2*r11**2 + h12*intrhoz2*r12**2
                k += 1
                Mv[k] += h12*intrhoz2*r11*r21 + h12*intrhoz2*r12*r22
                k += 1
                Mv[k] += h12*intrhoz2*r11*r31 + h12*intrhoz2*r12*r32
                k += 1
                Mv[k] += h22*intrhoz2*r11**2 + h22*intrhoz2*r12**2
                k += 1
                Mv[k] += h22*intrhoz2*r11*r21 + h22*intrhoz2*r12*r22
                k += 1
                Mv[k] += h22*intrhoz2*r11*r31 + h22*intrhoz2*r12*r32
                k += 1
                Mv[k] += h23*intrhoz2*r11**2 + h23*intrhoz2*r12**2
                k += 1
                Mv[k] += h23*intrhoz2*r11*r21 + h23*intrhoz2*r12*r22
                k += 1
                Mv[k] += h23*intrhoz2*r11*r31 + h23*intrhoz2*r12*r32
                k += 1
                Mv[k] += h24*intrhoz2*r11**2 + h24*intrhoz2*r12**2
                k += 1
                Mv[k] += h24*intrhoz2*r11*r21 + h24*intrhoz2*r12*r22
                k += 1
                Mv[k] += h24*intrhoz2*r11*r31 + h24*intrhoz2*r12*r32
                k += 1
                Mv[k] += h12*intrhoz2*r11*r21 + h12*intrhoz2*r12*r22
                k += 1
                Mv[k] += h12*intrhoz2*r21**2 + h12*intrhoz2*r22**2
                k += 1
                Mv[k] += h12*intrhoz2*r21*r31 + h12*intrhoz2*r22*r32
                k += 1
                Mv[k] += h22*intrhoz2*r11*r21 + h22*intrhoz2*r12*r22
                k += 1
                Mv[k] += h22*intrhoz2*r21**2 + h22*intrhoz2*r22**2
                k += 1
                Mv[k] += h22*intrhoz2*r21*r31 + h22*intrhoz2*r22*r32
                k += 1
                Mv[k] += h23*intrhoz2*r11*r21 + h23*intrhoz2*r12*r22
                k += 1
                Mv[k] += h23*intrhoz2*r21**2 + h23*intrhoz2*r22**2
                k += 1
                Mv[k] += h23*intrhoz2*r21*r31 + h23*intrhoz2*r22*r32
                k += 1
                Mv[k] += h24*intrhoz2*r11*r21 + h24*intrhoz2*r12*r22
                k += 1
                Mv[k] += h24*intrhoz2*r21**2 + h24*intrhoz2*r22**2
                k += 1
                Mv[k] += h24*intrhoz2*r21*r31 + h24*intrhoz2*r22*r32
                k += 1
                Mv[k] += h12*intrhoz2*r11*r31 + h12*intrhoz2*r12*r32
                k += 1
                Mv[k] += h12*intrhoz2*r21*r31 + h12*intrhoz2*r22*r32
                k += 1
                Mv[k] += h12*intrhoz2*r31**2 + h12*intrhoz2*r32**2
                k += 1
                Mv[k] += h22*intrhoz2*r11*r31 + h22*intrhoz2*r12*r32
                k += 1
                Mv[k] += h22*intrhoz2*r21*r31 + h22*intrhoz2*r22*r32
                k += 1
                Mv[k] += h22*intrhoz2*r31**2 + h22*intrhoz2*r32**2
                k += 1
                Mv[k] += h23*intrhoz2*r11*r31 + h23*intrhoz2*r12*r32
                k += 1
                Mv[k] += h23*intrhoz2*r21*r31 + h23*intrhoz2*r22*r32
                k += 1
                Mv[k] += h23*intrhoz2*r31**2 + h23*intrhoz2*r32**2
                k += 1
                Mv[k] += h24*intrhoz2*r11*r31 + h24*intrhoz2*r12*r32
                k += 1
                Mv[k] += h24*intrhoz2*r21*r31 + h24*intrhoz2*r22*r32
                k += 1
                Mv[k] += h24*intrhoz2*r31**2 + h24*intrhoz2*r32**2
                k += 1
                Mv[k] += h13*intrho*r11**2 + h13*intrho*r12**2 + h13*intrho*r13**2
                k += 1
                Mv[k] += h13*intrho*r11*r21 + h13*intrho*r12*r22 + h13*intrho*r13*r23
                k += 1
                Mv[k] += h13*intrho*r11*r31 + h13*intrho*r12*r32 + h13*intrho*r13*r33
                k += 1
                Mv[k] += h23*intrho*r11**2 + h23*intrho*r12**2 + h23*intrho*r13**2
                k += 1
                Mv[k] += h23*intrho*r11*r21 + h23*intrho*r12*r22 + h23*intrho*r13*r23
                k += 1
                Mv[k] += h23*intrho*r11*r31 + h23*intrho*r12*r32 + h23*intrho*r13*r33
                k += 1
                Mv[k] += h33*intrho*r11**2 + h33*intrho*r12**2 + h33*intrho*r13**2
                k += 1
                Mv[k] += h33*intrho*r11*r21 + h33*intrho*r12*r22 + h33*intrho*r13*r23
                k += 1
                Mv[k] += h33*intrho*r11*r31 + h33*intrho*r12*r32 + h33*intrho*r13*r33
                k += 1
                Mv[k] += h34*intrho*r11**2 + h34*intrho*r12**2 + h34*intrho*r13**2
                k += 1
                Mv[k] += h34*intrho*r11*r21 + h34*intrho*r12*r22 + h34*intrho*r13*r23
                k += 1
                Mv[k] += h34*intrho*r11*r31 + h34*intrho*r12*r32 + h34*intrho*r13*r33
                k += 1
                Mv[k] += h13*intrho*r11*r21 + h13*intrho*r12*r22 + h13*intrho*r13*r23
                k += 1
                Mv[k] += h13*intrho*r21**2 + h13*intrho*r22**2 + h13*intrho*r23**2
                k += 1
                Mv[k] += h13*intrho*r21*r31 + h13*intrho*r22*r32 + h13*intrho*r23*r33
                k += 1
                Mv[k] += h23*intrho*r11*r21 + h23*intrho*r12*r22 + h23*intrho*r13*r23
                k += 1
                Mv[k] += h23*intrho*r21**2 + h23*intrho*r22**2 + h23*intrho*r23**2
                k += 1
                Mv[k] += h23*intrho*r21*r31 + h23*intrho*r22*r32 + h23*intrho*r23*r33
                k += 1
                Mv[k] += h33*intrho*r11*r21 + h33*intrho*r12*r22 + h33*intrho*r13*r23
                k += 1
                Mv[k] += h33*intrho*r21**2 + h33*intrho*r22**2 + h33*intrho*r23**2
                k += 1
                Mv[k] += h33*intrho*r21*r31 + h33*intrho*r22*r32 + h33*intrho*r23*r33
                k += 1
                Mv[k] += h34*intrho*r11*r21 + h34*intrho*r12*r22 + h34*intrho*r13*r23
                k += 1
                Mv[k] += h34*intrho*r21**2 + h34*intrho*r22**2 + h34*intrho*r23**2
                k += 1
                Mv[k] += h34*intrho*r21*r31 + h34*intrho*r22*r32 + h34*intrho*r23*r33
                k += 1
                Mv[k] += h13*intrho*r11*r31 + h13*intrho*r12*r32 + h13*intrho*r13*r33
                k += 1
                Mv[k] += h13*intrho*r21*r31 + h13*intrho*r22*r32 + h13*intrho*r23*r33
                k += 1
                Mv[k] += h13*intrho*r31**2 + h13*intrho*r32**2 + h13*intrho*r33**2
                k += 1
                Mv[k] += h23*intrho*r11*r31 + h23*intrho*r12*r32 + h23*intrho*r13*r33
                k += 1
                Mv[k] += h23*intrho*r21*r31 + h23*intrho*r22*r32 + h23*intrho*r23*r33
                k += 1
                Mv[k] += h23*intrho*r31**2 + h23*intrho*r32**2 + h23*intrho*r33**2
                k += 1
                Mv[k] += h33*intrho*r11*r31 + h33*intrho*r12*r32 + h33*intrho*r13*r33
                k += 1
                Mv[k] += h33*intrho*r21*r31 + h33*intrho*r22*r32 + h33*intrho*r23*r33
                k += 1
                Mv[k] += h33*intrho*r31**2 + h33*intrho*r32**2 + h33*intrho*r33**2
                k += 1
                Mv[k] += h34*intrho*r11*r31 + h34*intrho*r12*r32 + h34*intrho*r13*r33
                k += 1
                Mv[k] += h34*intrho*r21*r31 + h34*intrho*r22*r32 + h34*intrho*r23*r33
                k += 1
                Mv[k] += h34*intrho*r31**2 + h34*intrho*r32**2 + h34*intrho*r33**2
                k += 1
                Mv[k] += h13*intrhoz2*r11**2 + h13*intrhoz2*r12**2
                k += 1
                Mv[k] += h13*intrhoz2*r11*r21 + h13*intrhoz2*r12*r22
                k += 1
                Mv[k] += h13*intrhoz2*r11*r31 + h13*intrhoz2*r12*r32
                k += 1
                Mv[k] += h23*intrhoz2*r11**2 + h23*intrhoz2*r12**2
                k += 1
                Mv[k] += h23*intrhoz2*r11*r21 + h23*intrhoz2*r12*r22
                k += 1
                Mv[k] += h23*intrhoz2*r11*r31 + h23*intrhoz2*r12*r32
                k += 1
                Mv[k] += h33*intrhoz2*r11**2 + h33*intrhoz2*r12**2
                k += 1
                Mv[k] += h33*intrhoz2*r11*r21 + h33*intrhoz2*r12*r22
                k += 1
                Mv[k] += h33*intrhoz2*r11*r31 + h33*intrhoz2*r12*r32
                k += 1
                Mv[k] += h34*intrhoz2*r11**2 + h34*intrhoz2*r12**2
                k += 1
                Mv[k] += h34*intrhoz2*r11*r21 + h34*intrhoz2*r12*r22
                k += 1
                Mv[k] += h34*intrhoz2*r11*r31 + h34*intrhoz2*r12*r32
                k += 1
                Mv[k] += h13*intrhoz2*r11*r21 + h13*intrhoz2*r12*r22
                k += 1
                Mv[k] += h13*intrhoz2*r21**2 + h13*intrhoz2*r22**2
                k += 1
                Mv[k] += h13*intrhoz2*r21*r31 + h13*intrhoz2*r22*r32
                k += 1
                Mv[k] += h23*intrhoz2*r11*r21 + h23*intrhoz2*r12*r22
                k += 1
                Mv[k] += h23*intrhoz2*r21**2 + h23*intrhoz2*r22**2
                k += 1
                Mv[k] += h23*intrhoz2*r21*r31 + h23*intrhoz2*r22*r32
                k += 1
                Mv[k] += h33*intrhoz2*r11*r21 + h33*intrhoz2*r12*r22
                k += 1
                Mv[k] += h33*intrhoz2*r21**2 + h33*intrhoz2*r22**2
                k += 1
                Mv[k] += h33*intrhoz2*r21*r31 + h33*intrhoz2*r22*r32
                k += 1
                Mv[k] += h34*intrhoz2*r11*r21 + h34*intrhoz2*r12*r22
                k += 1
                Mv[k] += h34*intrhoz2*r21**2 + h34*intrhoz2*r22**2
                k += 1
                Mv[k] += h34*intrhoz2*r21*r31 + h34*intrhoz2*r22*r32
                k += 1
                Mv[k] += h13*intrhoz2*r11*r31 + h13*intrhoz2*r12*r32
                k += 1
                Mv[k] += h13*intrhoz2*r21*r31 + h13*intrhoz2*r22*r32
                k += 1
                Mv[k] += h13*intrhoz2*r31**2 + h13*intrhoz2*r32**2
                k += 1
                Mv[k] += h23*intrhoz2*r11*r31 + h23*intrhoz2*r12*r32
                k += 1
                Mv[k] += h23*intrhoz2*r21*r31 + h23*intrhoz2*r22*r32
                k += 1
                Mv[k] += h23*intrhoz2*r31**2 + h23*intrhoz2*r32**2
                k += 1
                Mv[k] += h33*intrhoz2*r11*r31 + h33*intrhoz2*r12*r32
                k += 1
                Mv[k] += h33*intrhoz2*r21*r31 + h33*intrhoz2*r22*r32
                k += 1
                Mv[k] += h33*intrhoz2*r31**2 + h33*intrhoz2*r32**2
                k += 1
                Mv[k] += h34*intrhoz2*r11*r31 + h34*intrhoz2*r12*r32
                k += 1
                Mv[k] += h34*intrhoz2*r21*r31 + h34*intrhoz2*r22*r32
                k += 1
                Mv[k] += h34*intrhoz2*r31**2 + h34*intrhoz2*r32**2
                k += 1
                Mv[k] += h14*intrho*r11**2 + h14*intrho*r12**2 + h14*intrho*r13**2
                k += 1
                Mv[k] += h14*intrho*r11*r21 + h14*intrho*r12*r22 + h14*intrho*r13*r23
                k += 1
                Mv[k] += h14*intrho*r11*r31 + h14*intrho*r12*r32 + h14*intrho*r13*r33
                k += 1
                Mv[k] += h24*intrho*r11**2 + h24*intrho*r12**2 + h24*intrho*r13**2
                k += 1
                Mv[k] += h24*intrho*r11*r21 + h24*intrho*r12*r22 + h24*intrho*r13*r23
                k += 1
                Mv[k] += h24*intrho*r11*r31 + h24*intrho*r12*r32 + h24*intrho*r13*r33
                k += 1
                Mv[k] += h34*intrho*r11**2 + h34*intrho*r12**2 + h34*intrho*r13**2
                k += 1
                Mv[k] += h34*intrho*r11*r21 + h34*intrho*r12*r22 + h34*intrho*r13*r23
                k += 1
                Mv[k] += h34*intrho*r11*r31 + h34*intrho*r12*r32 + h34*intrho*r13*r33
                k += 1
                Mv[k] += h44*intrho*r11**2 + h44*intrho*r12**2 + h44*intrho*r13**2
                k += 1
                Mv[k] += h44*intrho*r11*r21 + h44*intrho*r12*r22 + h44*intrho*r13*r23
                k += 1
                Mv[k] += h44*intrho*r11*r31 + h44*intrho*r12*r32 + h44*intrho*r13*r33
                k += 1
                Mv[k] += h14*intrho*r11*r21 + h14*intrho*r12*r22 + h14*intrho*r13*r23
                k += 1
                Mv[k] += h14*intrho*r21**2 + h14*intrho*r22**2 + h14*intrho*r23**2
                k += 1
                Mv[k] += h14*intrho*r21*r31 + h14*intrho*r22*r32 + h14*intrho*r23*r33
                k += 1
                Mv[k] += h24*intrho*r11*r21 + h24*intrho*r12*r22 + h24*intrho*r13*r23
                k += 1
                Mv[k] += h24*intrho*r21**2 + h24*intrho*r22**2 + h24*intrho*r23**2
                k += 1
                Mv[k] += h24*intrho*r21*r31 + h24*intrho*r22*r32 + h24*intrho*r23*r33
                k += 1
                Mv[k] += h34*intrho*r11*r21 + h34*intrho*r12*r22 + h34*intrho*r13*r23
                k += 1
                Mv[k] += h34*intrho*r21**2 + h34*intrho*r22**2 + h34*intrho*r23**2
                k += 1
                Mv[k] += h34*intrho*r21*r31 + h34*intrho*r22*r32 + h34*intrho*r23*r33
                k += 1
                Mv[k] += h44*intrho*r11*r21 + h44*intrho*r12*r22 + h44*intrho*r13*r23
                k += 1
                Mv[k] += h44*intrho*r21**2 + h44*intrho*r22**2 + h44*intrho*r23**2
                k += 1
                Mv[k] += h44*intrho*r21*r31 + h44*intrho*r22*r32 + h44*intrho*r23*r33
                k += 1
                Mv[k] += h14*intrho*r11*r31 + h14*intrho*r12*r32 + h14*intrho*r13*r33
                k += 1
                Mv[k] += h14*intrho*r21*r31 + h14*intrho*r22*r32 + h14*intrho*r23*r33
                k += 1
                Mv[k] += h14*intrho*r31**2 + h14*intrho*r32**2 + h14*intrho*r33**2
                k += 1
                Mv[k] += h24*intrho*r11*r31 + h24*intrho*r12*r32 + h24*intrho*r13*r33
                k += 1
                Mv[k] += h24*intrho*r21*r31 + h24*intrho*r22*r32 + h24*intrho*r23*r33
                k += 1
                Mv[k] += h24*intrho*r31**2 + h24*intrho*r32**2 + h24*intrho*r33**2
                k += 1
                Mv[k] += h34*intrho*r11*r31 + h34*intrho*r12*r32 + h34*intrho*r13*r33
                k += 1
                Mv[k] += h34*intrho*r21*r31 + h34*intrho*r22*r32 + h34*intrho*r23*r33
                k += 1
                Mv[k] += h34*intrho*r31**2 + h34*intrho*r32**2 + h34*intrho*r33**2
                k += 1
                Mv[k] += h44*intrho*r11*r31 + h44*intrho*r12*r32 + h44*intrho*r13*r33
                k += 1
                Mv[k] += h44*intrho*r21*r31 + h44*intrho*r22*r32 + h44*intrho*r23*r33
                k += 1
                Mv[k] += h44*intrho*r31**2 + h44*intrho*r32**2 + h44*intrho*r33**2
                k += 1
                Mv[k] += h14*intrhoz2*r11**2 + h14*intrhoz2*r12**2
                k += 1
                Mv[k] += h14*intrhoz2*r11*r21 + h14*intrhoz2*r12*r22
                k += 1
                Mv[k] += h14*intrhoz2*r11*r31 + h14*intrhoz2*r12*r32
                k += 1
                Mv[k] += h24*intrhoz2*r11**2 + h24*intrhoz2*r12**2
                k += 1
                Mv[k] += h24*intrhoz2*r11*r21 + h24*intrhoz2*r12*r22
                k += 1
                Mv[k] += h24*intrhoz2*r11*r31 + h24*intrhoz2*r12*r32
                k += 1
                Mv[k] += h34*intrhoz2*r11**2 + h34*intrhoz2*r12**2
                k += 1
                Mv[k] += h34*intrhoz2*r11*r21 + h34*intrhoz2*r12*r22
                k += 1
                Mv[k] += h34*intrhoz2*r11*r31 + h34*intrhoz2*r12*r32
                k += 1
                Mv[k] += h44*intrhoz2*r11**2 + h44*intrhoz2*r12**2
                k += 1
                Mv[k] += h44*intrhoz2*r11*r21 + h44*intrhoz2*r12*r22
                k += 1
                Mv[k] += h44*intrhoz2*r11*r31 + h44*intrhoz2*r12*r32
                k += 1
                Mv[k] += h14*intrhoz2*r11*r21 + h14*intrhoz2*r12*r22
                k += 1
                Mv[k] += h14*intrhoz2*r21**2 + h14*intrhoz2*r22**2
                k += 1
                Mv[k] += h14*intrhoz2*r21*r31 + h14*intrhoz2*r22*r32
                k += 1
                Mv[k] += h24*intrhoz2*r11*r21 + h24*intrhoz2*r12*r22
                k += 1
                Mv[k] += h24*intrhoz2*r21**2 + h24*intrhoz2*r22**2
                k += 1
                Mv[k] += h24*intrhoz2*r21*r31 + h24*intrhoz2*r22*r32
                k += 1
                Mv[k] += h34*intrhoz2*r11*r21 + h34*intrhoz2*r12*r22
                k += 1
                Mv[k] += h34*intrhoz2*r21**2 + h34*intrhoz2*r22**2
                k += 1
                Mv[k] += h34*intrhoz2*r21*r31 + h34*intrhoz2*r22*r32
                k += 1
                Mv[k] += h44*intrhoz2*r11*r21 + h44*intrhoz2*r12*r22
                k += 1
                Mv[k] += h44*intrhoz2*r21**2 + h44*intrhoz2*r22**2
                k += 1
                Mv[k] += h44*intrhoz2*r21*r31 + h44*intrhoz2*r22*r32
                k += 1
                Mv[k] += h14*intrhoz2*r11*r31 + h14*intrhoz2*r12*r32
                k += 1
                Mv[k] += h14*intrhoz2*r21*r31 + h14*intrhoz2*r22*r32
                k += 1
                Mv[k] += h14*intrhoz2*r31**2 + h14*intrhoz2*r32**2
                k += 1
                Mv[k] += h24*intrhoz2*r11*r31 + h24*intrhoz2*r12*r32
                k += 1
                Mv[k] += h24*intrhoz2*r21*r31 + h24*intrhoz2*r22*r32
                k += 1
                Mv[k] += h24*intrhoz2*r31**2 + h24*intrhoz2*r32**2
                k += 1
                Mv[k] += h34*intrhoz2*r11*r31 + h34*intrhoz2*r12*r32
                k += 1
                Mv[k] += h34*intrhoz2*r21*r31 + h34*intrhoz2*r22*r32
                k += 1
                Mv[k] += h34*intrhoz2*r31**2 + h34*intrhoz2*r32**2
                k += 1
                Mv[k] += h44*intrhoz2*r11*r31 + h44*intrhoz2*r12*r32
                k += 1
                Mv[k] += h44*intrhoz2*r21*r31 + h44*intrhoz2*r22*r32
                k += 1
                Mv[k] += h44*intrhoz2*r31**2 + h44*intrhoz2*r32**2


    cpdef void update_KA_beta(Quad4 self,
                        long [::1] KA_betar,
                        long [::1] KA_betac,
                        double [::1] KA_betav,
                        ):
        r"""Update sparse vectors for piston-theory aerodynamic matrix `KA_{\beta}`

        Parameters
        ----------
        KA_betar : np.array
            Array to store row positions of sparse values
        KA_betac : np.array
            Array to store column positions of sparse values
        KA_betav : np.array
            Array to store sparse values

        """
        cdef int c1, c2, c3, c4, i, j, k
        cdef double r11, r12, r13, r21, r22, r23, r31, r32, r33
        cdef double x1, x2, x3, x4
        cdef double y1, y2, y3, y4
        cdef double j11, j12, j21, j22
        cdef double N1, N2, N3, N4
        cdef double N1x, N2x, N3x, N4x
        cdef double N1y, N2y, N3y, N4y
        cdef double xi, eta, wij, J11, J12, J21, J22, detJ
        cdef double points[2]

        with nogil:

            # NOTE ignoring z in local coordinates
            x1 = self.probe.xe[0]
            y1 = self.probe.xe[1]
            # z1 = self.probe.xe[2]
            x2 = self.probe.xe[3]
            y2 = self.probe.xe[4]
            # z2 = self.probe.xe[5]
            x3 = self.probe.xe[6]
            y3 = self.probe.xe[7]
            # z3 = self.probe.xe[8]
            x4 = self.probe.xe[9]
            y4 = self.probe.xe[10]
            # z4 = self.probe.xe[11]

            # local to global transformation
            r11 = self.r11
            r12 = self.r12
            r13 = self.r13
            r21 = self.r21
            r22 = self.r22
            r23 = self.r23
            r31 = self.r31
            r32 = self.r32
            r33 = self.r33

            # positions the global matrices
            c1 = self.c1
            c2 = self.c2
            c3 = self.c3
            c4 = self.c4

            k = self.init_k_KA_beta
            KA_betar[k] = 0+c1
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 0+c1
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 1+c1
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 2+c1
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 0+c2
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 1+c2
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 2+c2
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 0+c3
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 1+c3
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 2+c3
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 0+c4
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 1+c4
            KA_betac[k] = 2+c4
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 0+c1
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 1+c1
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 2+c1
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 0+c2
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 1+c2
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 2+c2
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 0+c3
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 1+c3
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 2+c3
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 0+c4
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 1+c4
            k += 1
            KA_betar[k] = 2+c4
            KA_betac[k] = 2+c4

            # NOTE full integration for KG with two-point Gauss-Legendre quadrature
            wij = 1.
            points[0] = -0.5773502691896257645092
            points[1] = +0.5773502691896257645092

            for i in range(2):
                xi = points[i]
                for j in range(2):
                    eta = points[j]

                    J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
                    J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
                    J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
                    J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

                    detJ = J11*J22 - J12*J21

                    j11 = J22/(J11*J22 - J12*J21)
                    j12 = -J12/(J11*J22 - J12*J21)
                    j21 = -J21/(J11*J22 - J12*J21)
                    j22 = J11/(J11*J22 - J12*J21)

                    N1 = 0.25*eta*xi - 0.25*eta - 0.25*xi + 0.25
                    N2 = -0.25*eta*xi - 0.25*eta + 0.25*xi + 0.25
                    N3 = 0.25*eta*xi + 0.25*eta + 0.25*xi + 0.25
                    N4 = -0.25*eta*xi + 0.25*eta - 0.25*xi + 0.25

                    N1x = 0.25*j11*(eta - 1) + 0.25*j12*(xi - 1)
                    N2x = -0.25*eta*j11 + 0.25*j11 - 0.25*j12*xi - 0.25*j12
                    N3x = 0.25*j11*(eta + 1) + 0.25*j12*(xi + 1)
                    N4x = -0.25*eta*j11 - 0.25*j11 - 0.25*j12*xi + 0.25*j12

                    N1y = 0.25*j21*(eta - 1) + 0.25*j22*(xi - 1)
                    N2y = -0.25*eta*j21 + 0.25*j21 - 0.25*j22*xi - 0.25*j22
                    N3y = 0.25*j21*(eta + 1) + 0.25*j22*(xi + 1)
                    N4y = -0.25*eta*j21 - 0.25*j21 - 0.25*j22*xi + 0.25*j22

                    k = self.init_k_KA_beta
                    KA_betav[k] += -N1*detJ*r13**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r23*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r23*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r23*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r23*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r23*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r23*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r23*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r23*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r33**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r33**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r33**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r13*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r23*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N1*detJ*r33**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r23*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r23*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r23*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r23*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r23*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r23*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r23*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r23*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r33**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r33**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r33**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r13*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r23*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N2*detJ*r33**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r23*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r23*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r23*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r23*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r23*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r23*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r23*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r23*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r33**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r33**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r33**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r13*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r23*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N3*detJ*r33**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r23*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r23*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r23*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r23*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r23*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r23*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r23*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r23*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23**2*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23*r33*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r33**2*wij*(N1x*r11 + N1y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23*r33*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r33**2*wij*(N2x*r11 + N2y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23*r33*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r33**2*wij*(N3x*r11 + N3y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r13*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r23*r33*wij*(N4x*r11 + N4y*r21)
                    k += 1
                    KA_betav[k] += -N4*detJ*r33**2*wij*(N4x*r11 + N4y*r21)


    cpdef void update_KA_gamma(Quad4 self,
                        long [::1] KA_gammar,
                        long [::1] KA_gammac,
                        double [::1] KA_gammav,
                        ):
        r"""Update sparse vectors for piston-theory aerodynamic matrix `KA_{\gamma}`

        Parameters
        ----------
        KA_gammar : np.array
            Array to store row positions of sparse values
        KA_gammac : np.array
            Array to store column positions of sparse values
        KA_gammav : np.array
            Array to store sparse values

        """
        cdef int c1, c2, c3, c4, i, j, k
        cdef double r11, r12, r13, r21, r22, r23, r31, r32, r33
        cdef double x1, x2, x3, x4
        cdef double y1, y2, y3, y4
        cdef double N1, N2, N3, N4
        cdef double xi, eta, wij, J11, J12, J21, J22, detJ
        cdef double points[2]

        with nogil:

            # NOTE ignoring z in local coordinates
            x1 = self.probe.xe[0]
            y1 = self.probe.xe[1]
            # z1 = self.probe.xe[2]
            x2 = self.probe.xe[3]
            y2 = self.probe.xe[4]
            # z2 = self.probe.xe[5]
            x3 = self.probe.xe[6]
            y3 = self.probe.xe[7]
            # z3 = self.probe.xe[8]
            x4 = self.probe.xe[9]
            y4 = self.probe.xe[10]
            # z4 = self.probe.xe[11]

            # local to global transformation
            r11 = self.r11
            r12 = self.r12
            r13 = self.r13
            r21 = self.r21
            r22 = self.r22
            r23 = self.r23
            r31 = self.r31
            r32 = self.r32
            r33 = self.r33

            # positions the global matrices
            c1 = self.c1
            c2 = self.c2
            c3 = self.c3
            c4 = self.c4

            k = self.init_k_KA_gamma
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 0+c1
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 1+c1
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 2+c1
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 0+c2
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 1+c2
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 2+c2
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 0+c3
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 1+c3
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 2+c3
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 0+c4
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 1+c4
            KA_gammac[k] = 2+c4
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 0+c1
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 1+c1
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 2+c1
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 0+c2
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 1+c2
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 2+c2
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 0+c3
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 1+c3
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 2+c3
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 0+c4
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 1+c4
            k += 1
            KA_gammar[k] = 2+c4
            KA_gammac[k] = 2+c4

            # NOTE full integration for KG with two-point Gauss-Legendre quadrature
            wij = 1.
            points[0] = -0.5773502691896257645092
            points[1] = +0.5773502691896257645092

            for i in range(2):
                xi = points[i]
                for j in range(2):
                    eta = points[j]

                    J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
                    J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
                    J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
                    J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

                    detJ = J11*J22 - J12*J21

                    N1 = 0.25*eta*xi - 0.25*eta - 0.25*xi + 0.25
                    N2 = -0.25*eta*xi - 0.25*eta + 0.25*xi + 0.25
                    N3 = 0.25*eta*xi + 0.25*eta + 0.25*xi + 0.25
                    N4 = -0.25*eta*xi + 0.25*eta - 0.25*xi + 0.25

                    k = self.init_k_KA_gamma
                    KA_gammav[k] += N1**2*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N1**2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1**2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1**2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1**2*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N1**2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1**2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1**2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1**2*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2**2*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N2**2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2**2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2**2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2**2*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N2**2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N2*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N2**2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2**2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2**2*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N3**2*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N3**2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N3**2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N3**2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N3**2*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N3**2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N3*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N3*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N3**2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N3**2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N3**2*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N4**2*detJ*r13**2*wij
                    k += 1
                    KA_gammav[k] += N4**2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N4**2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N4**2*detJ*r13*r23*wij
                    k += 1
                    KA_gammav[k] += N4**2*detJ*r23**2*wij
                    k += 1
                    KA_gammav[k] += N4**2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N1*N4*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N2*N4*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N3*N4*detJ*r33**2*wij
                    k += 1
                    KA_gammav[k] += N4**2*detJ*r13*r33*wij
                    k += 1
                    KA_gammav[k] += N4**2*detJ*r23*r33*wij
                    k += 1
                    KA_gammav[k] += N4**2*detJ*r33**2*wij


    cpdef void update_CA(Quad4 self,
                        long [::1] CAr,
                        long [::1] CAc,
                        double [::1] CAv,
                        ):
        r"""Update sparse vectors for piston-theory aerodynamic damping matrix `CA`

        Parameters
        ----------
        CAr : np.array
            Array to store row positions of sparse values
        CAc : np.array
            Array to store column positions of sparse values
        CAv : np.array
            Array to store sparse values

        """
        cdef int c1, c2, c3, c4, i, j, k
        cdef double r11, r12, r13, r21, r22, r23, r31, r32, r33
        cdef double x1, x2, x3, x4
        cdef double y1, y2, y3, y4
        cdef double N1, N2, N3, N4
        cdef double xi, eta, wij, J11, J12, J21, J22, detJ
        cdef double points[2]

        with nogil:

            # NOTE ignoring z in local coordinates
            x1 = self.probe.xe[0]
            y1 = self.probe.xe[1]
            # z1 = self.probe.xe[2]
            x2 = self.probe.xe[3]
            y2 = self.probe.xe[4]
            # z2 = self.probe.xe[5]
            x3 = self.probe.xe[6]
            y3 = self.probe.xe[7]
            # z3 = self.probe.xe[8]
            x4 = self.probe.xe[9]
            y4 = self.probe.xe[10]
            # z4 = self.probe.xe[11]

            # local to global transformation
            r11 = self.r11
            r12 = self.r12
            r13 = self.r13
            r21 = self.r21
            r22 = self.r22
            r23 = self.r23
            r31 = self.r31
            r32 = self.r32
            r33 = self.r33

            # positions the global matrices
            c1 = self.c1
            c2 = self.c2
            c3 = self.c3
            c4 = self.c4

            k = self.init_k_CA
            CAr[k] = 0+c1
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 0+c1
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 1+c1
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 2+c1
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 0+c2
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 1+c2
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 2+c2
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 0+c3
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 1+c3
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 2+c3
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 0+c4
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 1+c4
            CAc[k] = 2+c4
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 0+c1
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 1+c1
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 2+c1
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 0+c2
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 1+c2
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 2+c2
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 0+c3
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 1+c3
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 2+c3
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 0+c4
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 1+c4
            k += 1
            CAr[k] = 2+c4
            CAc[k] = 2+c4

            # NOTE full integration for KG with two-point Gauss-Legendre quadrature
            wij = 1.
            points[0] = -0.5773502691896257645092
            points[1] = +0.5773502691896257645092

            for i in range(2):
                xi = points[i]
                for j in range(2):
                    eta = points[j]

                    J11 = -0.5*x1 + 0.5*x2 + 0.5*(eta + 1)*(0.5*x1 - 0.5*x2 + 0.5*x3 - 0.5*x4)
                    J12 = -0.5*y1 + 0.5*y2 + 0.5*(eta + 1)*(0.5*y1 - 0.5*y2 + 0.5*y3 - 0.5*y4)
                    J21 = -0.5*x1 + 0.5*x4 - 0.25*(-x1 + x2)*(xi + 1) + 0.25*(x3 - x4)*(xi + 1)
                    J22 = -0.5*y1 + 0.5*y4 - 0.25*(xi + 1)*(-y1 + y2) + 0.25*(xi + 1)*(y3 - y4)

                    detJ = J11*J22 - J12*J21

                    N1 = 0.25*eta*xi - 0.25*eta - 0.25*xi + 0.25
                    N2 = -0.25*eta*xi - 0.25*eta + 0.25*xi + 0.25
                    N3 = 0.25*eta*xi + 0.25*eta + 0.25*xi + 0.25
                    N4 = -0.25*eta*xi + 0.25*eta - 0.25*xi + 0.25

                    k = self.init_k_CA
                    CAv[k] += -N1**2*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N1**2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1**2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1**2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1**2*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N1**2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1**2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1**2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1**2*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2**2*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N2**2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2**2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2**2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2**2*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N2**2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N2*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N2**2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2**2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2**2*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N3**2*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N3**2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N3**2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N3**2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N3**2*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N3**2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N3*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2*N3*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N3**2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N3**2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N3**2*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N4**2*detJ*r13**2*wij
                    k += 1
                    CAv[k] += -N4**2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N4**2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N4**2*detJ*r13*r23*wij
                    k += 1
                    CAv[k] += -N4**2*detJ*r23**2*wij
                    k += 1
                    CAv[k] += -N4**2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N1*N4*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N2*N4*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N3*N4*detJ*r33**2*wij
                    k += 1
                    CAv[k] += -N4**2*detJ*r13*r33*wij
                    k += 1
                    CAv[k] += -N4**2*detJ*r23*r33*wij
                    k += 1
                    CAv[k] += -N4**2*detJ*r33**2*wij


