#
# Copyright (c) 2023 Airbyte, Inc., all rights reserved.
#

import sgqlc.types
import sgqlc.types.datetime
import sgqlc.types.relay


github_schema = sgqlc.types.Schema()


# Unexport Node/PageInfo, let schema re-declare them
github_schema -= sgqlc.types.relay.Node
github_schema -= sgqlc.types.relay.PageInfo


__docformat__ = "markdown"


########################################################################
# Scalars and Enumerations
########################################################################
class ActorType(sgqlc.types.Enum):
    """The actor's type.

    Enumeration Choices:

    * `TEAM`: Indicates a team actor.
    * `USER`: Indicates a user actor.
    """

    __schema__ = github_schema
    __choices__ = ("TEAM", "USER")


class AuditLogOrderField(sgqlc.types.Enum):
    """Properties by which Audit Log connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order audit log entries by timestamp
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class Base64String(sgqlc.types.Scalar):
    """A (potentially binary) string encoded using base64."""

    __schema__ = github_schema


class BigInt(sgqlc.types.Scalar):
    """Represents non-fractional signed whole numeric values. Since the
    value may exceed the size of a 32-bit integer, it's encoded as a
    string.
    """

    __schema__ = github_schema


Boolean = sgqlc.types.Boolean


class CheckAnnotationLevel(sgqlc.types.Enum):
    """Represents an annotation's information level.

    Enumeration Choices:

    * `FAILURE`: An annotation indicating an inescapable error.
    * `NOTICE`: An annotation indicating some information.
    * `WARNING`: An annotation indicating an ignorable error.
    """

    __schema__ = github_schema
    __choices__ = ("FAILURE", "NOTICE", "WARNING")


class CheckConclusionState(sgqlc.types.Enum):
    """The possible states for a check suite or run conclusion.

    Enumeration Choices:

    * `ACTION_REQUIRED`: The check suite or run requires action.
    * `CANCELLED`: The check suite or run has been cancelled.
    * `FAILURE`: The check suite or run has failed.
    * `NEUTRAL`: The check suite or run was neutral.
    * `SKIPPED`: The check suite or run was skipped.
    * `STALE`: The check suite or run was marked stale by GitHub. Only
      GitHub can use this conclusion.
    * `STARTUP_FAILURE`: The check suite or run has failed at startup.
    * `SUCCESS`: The check suite or run has succeeded.
    * `TIMED_OUT`: The check suite or run has timed out.
    """

    __schema__ = github_schema
    __choices__ = ("ACTION_REQUIRED", "CANCELLED", "FAILURE", "NEUTRAL", "SKIPPED", "STALE", "STARTUP_FAILURE", "SUCCESS", "TIMED_OUT")


class CheckRunState(sgqlc.types.Enum):
    """The possible states of a check run in a status rollup.

    Enumeration Choices:

    * `ACTION_REQUIRED`: The check run requires action.
    * `CANCELLED`: The check run has been cancelled.
    * `COMPLETED`: The check run has been completed.
    * `FAILURE`: The check run has failed.
    * `IN_PROGRESS`: The check run is in progress.
    * `NEUTRAL`: The check run was neutral.
    * `PENDING`: The check run is in pending state.
    * `QUEUED`: The check run has been queued.
    * `SKIPPED`: The check run was skipped.
    * `STALE`: The check run was marked stale by GitHub. Only GitHub
      can use this conclusion.
    * `STARTUP_FAILURE`: The check run has failed at startup.
    * `SUCCESS`: The check run has succeeded.
    * `TIMED_OUT`: The check run has timed out.
    * `WAITING`: The check run is in waiting state.
    """

    __schema__ = github_schema
    __choices__ = (
        "ACTION_REQUIRED",
        "CANCELLED",
        "COMPLETED",
        "FAILURE",
        "IN_PROGRESS",
        "NEUTRAL",
        "PENDING",
        "QUEUED",
        "SKIPPED",
        "STALE",
        "STARTUP_FAILURE",
        "SUCCESS",
        "TIMED_OUT",
        "WAITING",
    )


class CheckRunType(sgqlc.types.Enum):
    """The possible types of check runs.

    Enumeration Choices:

    * `ALL`: Every check run available.
    * `LATEST`: The latest check run.
    """

    __schema__ = github_schema
    __choices__ = ("ALL", "LATEST")


class CheckStatusState(sgqlc.types.Enum):
    """The possible states for a check suite or run status.

    Enumeration Choices:

    * `COMPLETED`: The check suite or run has been completed.
    * `IN_PROGRESS`: The check suite or run is in progress.
    * `PENDING`: The check suite or run is in pending state.
    * `QUEUED`: The check suite or run has been queued.
    * `REQUESTED`: The check suite or run has been requested.
    * `WAITING`: The check suite or run is in waiting state.
    """

    __schema__ = github_schema
    __choices__ = ("COMPLETED", "IN_PROGRESS", "PENDING", "QUEUED", "REQUESTED", "WAITING")


class CollaboratorAffiliation(sgqlc.types.Enum):
    """Collaborators affiliation level with a subject.

    Enumeration Choices:

    * `ALL`: All collaborators the authenticated user can see.
    * `DIRECT`: All collaborators with permissions to an organization-
      owned subject, regardless of organization membership status.
    * `OUTSIDE`: All outside collaborators of an organization-owned
      subject.
    """

    __schema__ = github_schema
    __choices__ = ("ALL", "DIRECT", "OUTSIDE")


class CommentAuthorAssociation(sgqlc.types.Enum):
    """A comment author association with repository.

    Enumeration Choices:

    * `COLLABORATOR`: Author has been invited to collaborate on the
      repository.
    * `CONTRIBUTOR`: Author has previously committed to the
      repository.
    * `FIRST_TIMER`: Author has not previously committed to GitHub.
    * `FIRST_TIME_CONTRIBUTOR`: Author has not previously committed to
      the repository.
    * `MANNEQUIN`: Author is a placeholder for an unclaimed user.
    * `MEMBER`: Author is a member of the organization that owns the
      repository.
    * `NONE`: Author has no association with the repository.
    * `OWNER`: Author is the owner of the repository.
    """

    __schema__ = github_schema
    __choices__ = ("COLLABORATOR", "CONTRIBUTOR", "FIRST_TIMER", "FIRST_TIME_CONTRIBUTOR", "MANNEQUIN", "MEMBER", "NONE", "OWNER")


class CommentCannotUpdateReason(sgqlc.types.Enum):
    """The possible errors that will prevent a user from updating a
    comment.

    Enumeration Choices:

    * `ARCHIVED`: Unable to create comment because repository is
      archived.
    * `DENIED`: You cannot update this comment
    * `INSUFFICIENT_ACCESS`: You must be the author or have write
      access to this repository to update this comment.
    * `LOCKED`: Unable to create comment because issue is locked.
    * `LOGIN_REQUIRED`: You must be logged in to update this comment.
    * `MAINTENANCE`: Repository is under maintenance.
    * `VERIFIED_EMAIL_REQUIRED`: At least one email address must be
      verified to update this comment.
    """

    __schema__ = github_schema
    __choices__ = ("ARCHIVED", "DENIED", "INSUFFICIENT_ACCESS", "LOCKED", "LOGIN_REQUIRED", "MAINTENANCE", "VERIFIED_EMAIL_REQUIRED")


class CommitContributionOrderField(sgqlc.types.Enum):
    """Properties by which commit contribution connections can be
    ordered.

    Enumeration Choices:

    * `COMMIT_COUNT`: Order commit contributions by how many commits
      they represent.
    * `OCCURRED_AT`: Order commit contributions by when they were
      made.
    """

    __schema__ = github_schema
    __choices__ = ("COMMIT_COUNT", "OCCURRED_AT")


class ComparisonStatus(sgqlc.types.Enum):
    """The status of a git comparison between two refs.

    Enumeration Choices:

    * `AHEAD`: The head ref is ahead of the base ref.
    * `BEHIND`: The head ref is behind the base ref.
    * `DIVERGED`: The head ref is both ahead and behind of the base
      ref, indicating git history has diverged.
    * `IDENTICAL`: The head ref and base ref are identical.
    """

    __schema__ = github_schema
    __choices__ = ("AHEAD", "BEHIND", "DIVERGED", "IDENTICAL")


class ContributionLevel(sgqlc.types.Enum):
    """Varying levels of contributions from none to many.

    Enumeration Choices:

    * `FIRST_QUARTILE`: Lowest 25% of days of contributions.
    * `FOURTH_QUARTILE`: Highest 25% of days of contributions. More
      contributions than the third quartile.
    * `NONE`: No contributions occurred.
    * `SECOND_QUARTILE`: Second lowest 25% of days of contributions.
      More contributions than the first quartile.
    * `THIRD_QUARTILE`: Second highest 25% of days of contributions.
      More contributions than second quartile, less than the fourth
      quartile.
    """

    __schema__ = github_schema
    __choices__ = ("FIRST_QUARTILE", "FOURTH_QUARTILE", "NONE", "SECOND_QUARTILE", "THIRD_QUARTILE")


Date = sgqlc.types.datetime.Date

DateTime = sgqlc.types.datetime.DateTime


class DefaultRepositoryPermissionField(sgqlc.types.Enum):
    """The possible base permissions for repositories.

    Enumeration Choices:

    * `ADMIN`: Can read, write, and administrate repos by default
    * `NONE`: No access
    * `READ`: Can read repos by default
    * `WRITE`: Can read and write repos by default
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "NONE", "READ", "WRITE")


class DependencyGraphEcosystem(sgqlc.types.Enum):
    """The possible ecosystems of a dependency graph package.

    Enumeration Choices:

    * `ACTIONS`: GitHub Actions
    * `COMPOSER`: PHP packages hosted at packagist.org
    * `GO`: Go modules
    * `MAVEN`: Java artifacts hosted at the Maven central repository
    * `NPM`: JavaScript packages hosted at npmjs.com
    * `NUGET`: .NET packages hosted at the NuGet Gallery
    * `PIP`: Python packages hosted at PyPI.org
    * `PUB`: Dart packages hosted at pub.dev
    * `RUBYGEMS`: Ruby gems hosted at RubyGems.org
    * `RUST`: Rust crates
    """

    __schema__ = github_schema
    __choices__ = ("ACTIONS", "COMPOSER", "GO", "MAVEN", "NPM", "NUGET", "PIP", "PUB", "RUBYGEMS", "RUST")


class DeploymentOrderField(sgqlc.types.Enum):
    """Properties by which deployment connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order collection by creation time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class DeploymentProtectionRuleType(sgqlc.types.Enum):
    """The possible protection rule types.

    Enumeration Choices:

    * `REQUIRED_REVIEWERS`: Required reviewers
    * `WAIT_TIMER`: Wait timer
    """

    __schema__ = github_schema
    __choices__ = ("REQUIRED_REVIEWERS", "WAIT_TIMER")


class DeploymentReviewState(sgqlc.types.Enum):
    """The possible states for a deployment review.

    Enumeration Choices:

    * `APPROVED`: The deployment was approved.
    * `REJECTED`: The deployment was rejected.
    """

    __schema__ = github_schema
    __choices__ = ("APPROVED", "REJECTED")


class DeploymentState(sgqlc.types.Enum):
    """The possible states in which a deployment can be.

    Enumeration Choices:

    * `ABANDONED`: The pending deployment was not updated after 30
      minutes.
    * `ACTIVE`: The deployment is currently active.
    * `DESTROYED`: An inactive transient deployment.
    * `ERROR`: The deployment experienced an error.
    * `FAILURE`: The deployment has failed.
    * `INACTIVE`: The deployment is inactive.
    * `IN_PROGRESS`: The deployment is in progress.
    * `PENDING`: The deployment is pending.
    * `QUEUED`: The deployment has queued
    * `SUCCESS`: The deployment was successful.
    * `WAITING`: The deployment is waiting.
    """

    __schema__ = github_schema
    __choices__ = (
        "ABANDONED",
        "ACTIVE",
        "DESTROYED",
        "ERROR",
        "FAILURE",
        "INACTIVE",
        "IN_PROGRESS",
        "PENDING",
        "QUEUED",
        "SUCCESS",
        "WAITING",
    )


class DeploymentStatusState(sgqlc.types.Enum):
    """The possible states for a deployment status.

    Enumeration Choices:

    * `ERROR`: The deployment experienced an error.
    * `FAILURE`: The deployment has failed.
    * `INACTIVE`: The deployment is inactive.
    * `IN_PROGRESS`: The deployment is in progress.
    * `PENDING`: The deployment is pending.
    * `QUEUED`: The deployment is queued
    * `SUCCESS`: The deployment was successful.
    * `WAITING`: The deployment is waiting.
    """

    __schema__ = github_schema
    __choices__ = ("ERROR", "FAILURE", "INACTIVE", "IN_PROGRESS", "PENDING", "QUEUED", "SUCCESS", "WAITING")


class DiffSide(sgqlc.types.Enum):
    """The possible sides of a diff.

    Enumeration Choices:

    * `LEFT`: The left side of the diff.
    * `RIGHT`: The right side of the diff.
    """

    __schema__ = github_schema
    __choices__ = ("LEFT", "RIGHT")


class DiscussionCloseReason(sgqlc.types.Enum):
    """The possible reasons for closing a discussion.

    Enumeration Choices:

    * `DUPLICATE`: The discussion is a duplicate of another
    * `OUTDATED`: The discussion is no longer relevant
    * `RESOLVED`: The discussion has been resolved
    """

    __schema__ = github_schema
    __choices__ = ("DUPLICATE", "OUTDATED", "RESOLVED")


class DiscussionOrderField(sgqlc.types.Enum):
    """Properties by which discussion connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order discussions by creation time.
    * `UPDATED_AT`: Order discussions by most recent modification
      time.
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "UPDATED_AT")


class DiscussionPollOptionOrderField(sgqlc.types.Enum):
    """Properties by which discussion poll option connections can be
    ordered.

    Enumeration Choices:

    * `AUTHORED_ORDER`: Order poll options by the order that the poll
      author specified when creating the poll.
    * `VOTE_COUNT`: Order poll options by the number of votes it has.
    """

    __schema__ = github_schema
    __choices__ = ("AUTHORED_ORDER", "VOTE_COUNT")


class DiscussionState(sgqlc.types.Enum):
    """The possible states of a discussion.

    Enumeration Choices:

    * `CLOSED`: A discussion that has been closed
    * `OPEN`: A discussion that is open
    """

    __schema__ = github_schema
    __choices__ = ("CLOSED", "OPEN")


class DiscussionStateReason(sgqlc.types.Enum):
    """The possible state reasons of a discussion.

    Enumeration Choices:

    * `DUPLICATE`: The discussion is a duplicate of another
    * `OUTDATED`: The discussion is no longer relevant
    * `REOPENED`: The discussion was reopened
    * `RESOLVED`: The discussion has been resolved
    """

    __schema__ = github_schema
    __choices__ = ("DUPLICATE", "OUTDATED", "REOPENED", "RESOLVED")


class DismissReason(sgqlc.types.Enum):
    """The possible reasons that a Dependabot alert was dismissed.

    Enumeration Choices:

    * `FIX_STARTED`: A fix has already been started
    * `INACCURATE`: This alert is inaccurate or incorrect
    * `NOT_USED`: Vulnerable code is not actually used
    * `NO_BANDWIDTH`: No bandwidth to fix this
    * `TOLERABLE_RISK`: Risk is tolerable to this project
    """

    __schema__ = github_schema
    __choices__ = ("FIX_STARTED", "INACCURATE", "NOT_USED", "NO_BANDWIDTH", "TOLERABLE_RISK")


class EnterpriseAdministratorInvitationOrderField(sgqlc.types.Enum):
    """Properties by which enterprise administrator invitation
    connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order enterprise administrator member invitations
      by creation time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class EnterpriseAdministratorRole(sgqlc.types.Enum):
    """The possible administrator roles in an enterprise account.

    Enumeration Choices:

    * `BILLING_MANAGER`: Represents a billing manager of the
      enterprise account.
    * `OWNER`: Represents an owner of the enterprise account.
    """

    __schema__ = github_schema
    __choices__ = ("BILLING_MANAGER", "OWNER")


class EnterpriseAllowPrivateRepositoryForkingPolicyValue(sgqlc.types.Enum):
    """The possible values for the enterprise allow private repository
    forking policy value.

    Enumeration Choices:

    * `ENTERPRISE_ORGANIZATIONS`: Members can fork a repository to an
      organization within this enterprise.
    * `ENTERPRISE_ORGANIZATIONS_USER_ACCOUNTS`: Members can fork a
      repository to their enterprise-managed user account or an
      organization inside this enterprise.
    * `EVERYWHERE`: Members can fork a repository to their user
      account or an organization, either inside or outside of this
      enterprise.
    * `SAME_ORGANIZATION`: Members can fork a repository only within
      the same organization (intra-org).
    * `SAME_ORGANIZATION_USER_ACCOUNTS`: Members can fork a repository
      to their user account or within the same organization.
    * `USER_ACCOUNTS`: Members can fork a repository to their user
      account.
    """

    __schema__ = github_schema
    __choices__ = (
        "ENTERPRISE_ORGANIZATIONS",
        "ENTERPRISE_ORGANIZATIONS_USER_ACCOUNTS",
        "EVERYWHERE",
        "SAME_ORGANIZATION",
        "SAME_ORGANIZATION_USER_ACCOUNTS",
        "USER_ACCOUNTS",
    )


class EnterpriseDefaultRepositoryPermissionSettingValue(sgqlc.types.Enum):
    """The possible values for the enterprise base repository permission
    setting.

    Enumeration Choices:

    * `ADMIN`: Organization members will be able to clone, pull, push,
      and add new collaborators to all organization repositories.
    * `NONE`: Organization members will only be able to clone and pull
      public repositories.
    * `NO_POLICY`: Organizations in the enterprise choose base
      repository permissions for their members.
    * `READ`: Organization members will be able to clone and pull all
      organization repositories.
    * `WRITE`: Organization members will be able to clone, pull, and
      push all organization repositories.
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "NONE", "NO_POLICY", "READ", "WRITE")


class EnterpriseEnabledDisabledSettingValue(sgqlc.types.Enum):
    """The possible values for an enabled/disabled enterprise setting.

    Enumeration Choices:

    * `DISABLED`: The setting is disabled for organizations in the
      enterprise.
    * `ENABLED`: The setting is enabled for organizations in the
      enterprise.
    * `NO_POLICY`: There is no policy set for organizations in the
      enterprise.
    """

    __schema__ = github_schema
    __choices__ = ("DISABLED", "ENABLED", "NO_POLICY")


class EnterpriseEnabledSettingValue(sgqlc.types.Enum):
    """The possible values for an enabled/no policy enterprise setting.

    Enumeration Choices:

    * `ENABLED`: The setting is enabled for organizations in the
      enterprise.
    * `NO_POLICY`: There is no policy set for organizations in the
      enterprise.
    """

    __schema__ = github_schema
    __choices__ = ("ENABLED", "NO_POLICY")


class EnterpriseMemberOrderField(sgqlc.types.Enum):
    """Properties by which enterprise member connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order enterprise members by creation time
    * `LOGIN`: Order enterprise members by login
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "LOGIN")


class EnterpriseMembersCanCreateRepositoriesSettingValue(sgqlc.types.Enum):
    """The possible values for the enterprise members can create
    repositories setting.

    Enumeration Choices:

    * `ALL`: Members will be able to create public and private
      repositories.
    * `DISABLED`: Members will not be able to create public or private
      repositories.
    * `NO_POLICY`: Organization administrators choose whether to allow
      members to create repositories.
    * `PRIVATE`: Members will be able to create only private
      repositories.
    * `PUBLIC`: Members will be able to create only public
      repositories.
    """

    __schema__ = github_schema
    __choices__ = ("ALL", "DISABLED", "NO_POLICY", "PRIVATE", "PUBLIC")


class EnterpriseMembersCanMakePurchasesSettingValue(sgqlc.types.Enum):
    """The possible values for the members can make purchases setting.

    Enumeration Choices:

    * `DISABLED`: The setting is disabled for organizations in the
      enterprise.
    * `ENABLED`: The setting is enabled for organizations in the
      enterprise.
    """

    __schema__ = github_schema
    __choices__ = ("DISABLED", "ENABLED")


class EnterpriseServerInstallationOrderField(sgqlc.types.Enum):
    """Properties by which Enterprise Server installation connections can
    be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order Enterprise Server installations by creation
      time
    * `CUSTOMER_NAME`: Order Enterprise Server installations by
      customer name
    * `HOST_NAME`: Order Enterprise Server installations by host name
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "CUSTOMER_NAME", "HOST_NAME")


class EnterpriseServerUserAccountEmailOrderField(sgqlc.types.Enum):
    """Properties by which Enterprise Server user account email
    connections can be ordered.

    Enumeration Choices:

    * `EMAIL`: Order emails by email
    """

    __schema__ = github_schema
    __choices__ = ("EMAIL",)


class EnterpriseServerUserAccountOrderField(sgqlc.types.Enum):
    """Properties by which Enterprise Server user account connections can
    be ordered.

    Enumeration Choices:

    * `LOGIN`: Order user accounts by login
    * `REMOTE_CREATED_AT`: Order user accounts by creation time on the
      Enterprise Server installation
    """

    __schema__ = github_schema
    __choices__ = ("LOGIN", "REMOTE_CREATED_AT")


class EnterpriseServerUserAccountsUploadOrderField(sgqlc.types.Enum):
    """Properties by which Enterprise Server user accounts upload
    connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order user accounts uploads by creation time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class EnterpriseServerUserAccountsUploadSyncState(sgqlc.types.Enum):
    """Synchronization state of the Enterprise Server user accounts
    upload

    Enumeration Choices:

    * `FAILURE`: The synchronization of the upload failed.
    * `PENDING`: The synchronization of the upload is pending.
    * `SUCCESS`: The synchronization of the upload succeeded.
    """

    __schema__ = github_schema
    __choices__ = ("FAILURE", "PENDING", "SUCCESS")


class EnterpriseUserAccountMembershipRole(sgqlc.types.Enum):
    """The possible roles for enterprise membership.

    Enumeration Choices:

    * `MEMBER`: The user is a member of an organization in the
      enterprise.
    * `OWNER`: The user is an owner of an organization in the
      enterprise.
    * `UNAFFILIATED`: The user is not an owner of the enterprise, and
      not a member or owner of any organizations in the enterprise;
      only for EMU-enabled enterprises.
    """

    __schema__ = github_schema
    __choices__ = ("MEMBER", "OWNER", "UNAFFILIATED")


class EnterpriseUserDeployment(sgqlc.types.Enum):
    """The possible GitHub Enterprise deployments where this user can
    exist.

    Enumeration Choices:

    * `CLOUD`: The user is part of a GitHub Enterprise Cloud
      deployment.
    * `SERVER`: The user is part of a GitHub Enterprise Server
      deployment.
    """

    __schema__ = github_schema
    __choices__ = ("CLOUD", "SERVER")


class FileViewedState(sgqlc.types.Enum):
    """The possible viewed states of a file .

    Enumeration Choices:

    * `DISMISSED`: The file has new changes since last viewed.
    * `UNVIEWED`: The file has not been marked as viewed.
    * `VIEWED`: The file has been marked as viewed.
    """

    __schema__ = github_schema
    __choices__ = ("DISMISSED", "UNVIEWED", "VIEWED")


Float = sgqlc.types.Float


class FundingPlatform(sgqlc.types.Enum):
    """The possible funding platforms for repository funding links.

    Enumeration Choices:

    * `COMMUNITY_BRIDGE`: Community Bridge funding platform.
    * `CUSTOM`: Custom funding platform.
    * `GITHUB`: GitHub funding platform.
    * `ISSUEHUNT`: IssueHunt funding platform.
    * `KO_FI`: Ko-fi funding platform.
    * `LFX_CROWDFUNDING`: LFX Crowdfunding funding platform.
    * `LIBERAPAY`: Liberapay funding platform.
    * `OPEN_COLLECTIVE`: Open Collective funding platform.
    * `OTECHIE`: Otechie funding platform.
    * `PATREON`: Patreon funding platform.
    * `TIDELIFT`: Tidelift funding platform.
    """

    __schema__ = github_schema
    __choices__ = (
        "COMMUNITY_BRIDGE",
        "CUSTOM",
        "GITHUB",
        "ISSUEHUNT",
        "KO_FI",
        "LFX_CROWDFUNDING",
        "LIBERAPAY",
        "OPEN_COLLECTIVE",
        "OTECHIE",
        "PATREON",
        "TIDELIFT",
    )


class GistOrderField(sgqlc.types.Enum):
    """Properties by which gist connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order gists by creation time
    * `PUSHED_AT`: Order gists by push time
    * `UPDATED_AT`: Order gists by update time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "PUSHED_AT", "UPDATED_AT")


class GistPrivacy(sgqlc.types.Enum):
    """The privacy of a Gist

    Enumeration Choices:

    * `ALL`: Gists that are public and secret
    * `PUBLIC`: Public
    * `SECRET`: Secret
    """

    __schema__ = github_schema
    __choices__ = ("ALL", "PUBLIC", "SECRET")


class GitObjectID(sgqlc.types.Scalar):
    """A Git object ID."""

    __schema__ = github_schema


class GitSSHRemote(sgqlc.types.Scalar):
    """Git SSH string"""

    __schema__ = github_schema


class GitSignatureState(sgqlc.types.Enum):
    """The state of a Git signature.

    Enumeration Choices:

    * `BAD_CERT`: The signing certificate or its chain could not be
      verified
    * `BAD_EMAIL`: Invalid email used for signing
    * `EXPIRED_KEY`: Signing key expired
    * `GPGVERIFY_ERROR`: Internal error - the GPG verification service
      misbehaved
    * `GPGVERIFY_UNAVAILABLE`: Internal error - the GPG verification
      service is unavailable at the moment
    * `INVALID`: Invalid signature
    * `MALFORMED_SIG`: Malformed signature
    * `NOT_SIGNING_KEY`: The usage flags for the key that signed this
      don't allow signing
    * `NO_USER`: Email used for signing not known to GitHub
    * `OCSP_ERROR`: Valid signature, though certificate revocation
      check failed
    * `OCSP_PENDING`: Valid signature, pending certificate revocation
      checking
    * `OCSP_REVOKED`: One or more certificates in chain has been
      revoked
    * `UNKNOWN_KEY`: Key used for signing not known to GitHub
    * `UNKNOWN_SIG_TYPE`: Unknown signature type
    * `UNSIGNED`: Unsigned
    * `UNVERIFIED_EMAIL`: Email used for signing unverified on GitHub
    * `VALID`: Valid signature and verified by GitHub
    """

    __schema__ = github_schema
    __choices__ = (
        "BAD_CERT",
        "BAD_EMAIL",
        "EXPIRED_KEY",
        "GPGVERIFY_ERROR",
        "GPGVERIFY_UNAVAILABLE",
        "INVALID",
        "MALFORMED_SIG",
        "NOT_SIGNING_KEY",
        "NO_USER",
        "OCSP_ERROR",
        "OCSP_PENDING",
        "OCSP_REVOKED",
        "UNKNOWN_KEY",
        "UNKNOWN_SIG_TYPE",
        "UNSIGNED",
        "UNVERIFIED_EMAIL",
        "VALID",
    )


class GitTimestamp(sgqlc.types.Scalar):
    """An ISO-8601 encoded date string. Unlike the DateTime type,
    GitTimestamp is not converted in UTC.
    """

    __schema__ = github_schema


class HTML(sgqlc.types.Scalar):
    """A string containing HTML code."""

    __schema__ = github_schema


ID = sgqlc.types.ID


class IdentityProviderConfigurationState(sgqlc.types.Enum):
    """The possible states in which authentication can be configured with
    an identity provider.

    Enumeration Choices:

    * `CONFIGURED`: Authentication with an identity provider is
      configured but not enforced.
    * `ENFORCED`: Authentication with an identity provider is
      configured and enforced.
    * `UNCONFIGURED`: Authentication with an identity provider is not
      configured.
    """

    __schema__ = github_schema
    __choices__ = ("CONFIGURED", "ENFORCED", "UNCONFIGURED")


Int = sgqlc.types.Int


class IpAllowListEnabledSettingValue(sgqlc.types.Enum):
    """The possible values for the IP allow list enabled setting.

    Enumeration Choices:

    * `DISABLED`: The setting is disabled for the owner.
    * `ENABLED`: The setting is enabled for the owner.
    """

    __schema__ = github_schema
    __choices__ = ("DISABLED", "ENABLED")


class IpAllowListEntryOrderField(sgqlc.types.Enum):
    """Properties by which IP allow list entry connections can be
    ordered.

    Enumeration Choices:

    * `ALLOW_LIST_VALUE`: Order IP allow list entries by the allow
      list value.
    * `CREATED_AT`: Order IP allow list entries by creation time.
    """

    __schema__ = github_schema
    __choices__ = ("ALLOW_LIST_VALUE", "CREATED_AT")


class IpAllowListForInstalledAppsEnabledSettingValue(sgqlc.types.Enum):
    """The possible values for the IP allow list configuration for
    installed GitHub Apps setting.

    Enumeration Choices:

    * `DISABLED`: The setting is disabled for the owner.
    * `ENABLED`: The setting is enabled for the owner.
    """

    __schema__ = github_schema
    __choices__ = ("DISABLED", "ENABLED")


class IssueClosedStateReason(sgqlc.types.Enum):
    """The possible state reasons of a closed issue.

    Enumeration Choices:

    * `COMPLETED`: An issue that has been closed as completed
    * `NOT_PLANNED`: An issue that has been closed as not planned
    """

    __schema__ = github_schema
    __choices__ = ("COMPLETED", "NOT_PLANNED")


class IssueCommentOrderField(sgqlc.types.Enum):
    """Properties by which issue comment connections can be ordered.

    Enumeration Choices:

    * `UPDATED_AT`: Order issue comments by update time
    """

    __schema__ = github_schema
    __choices__ = ("UPDATED_AT",)


class IssueOrderField(sgqlc.types.Enum):
    """Properties by which issue connections can be ordered.

    Enumeration Choices:

    * `COMMENTS`: Order issues by comment count
    * `CREATED_AT`: Order issues by creation time
    * `UPDATED_AT`: Order issues by update time
    """

    __schema__ = github_schema
    __choices__ = ("COMMENTS", "CREATED_AT", "UPDATED_AT")


class IssueState(sgqlc.types.Enum):
    """The possible states of an issue.

    Enumeration Choices:

    * `CLOSED`: An issue that has been closed
    * `OPEN`: An issue that is still open
    """

    __schema__ = github_schema
    __choices__ = ("CLOSED", "OPEN")


class IssueStateReason(sgqlc.types.Enum):
    """The possible state reasons of an issue.

    Enumeration Choices:

    * `COMPLETED`: An issue that has been closed as completed
    * `NOT_PLANNED`: An issue that has been closed as not planned
    * `REOPENED`: An issue that has been reopened
    """

    __schema__ = github_schema
    __choices__ = ("COMPLETED", "NOT_PLANNED", "REOPENED")


class IssueTimelineItemsItemType(sgqlc.types.Enum):
    """The possible item types found in a timeline.

    Enumeration Choices:

    * `ADDED_TO_PROJECT_EVENT`: Represents a 'added_to_project' event
      on a given issue or pull request.
    * `ASSIGNED_EVENT`: Represents an 'assigned' event on any
      assignable object.
    * `CLOSED_EVENT`: Represents a 'closed' event on any `Closable`.
    * `COMMENT_DELETED_EVENT`: Represents a 'comment_deleted' event on
      a given issue or pull request.
    * `CONNECTED_EVENT`: Represents a 'connected' event on a given
      issue or pull request.
    * `CONVERTED_NOTE_TO_ISSUE_EVENT`: Represents a
      'converted_note_to_issue' event on a given issue or pull
      request.
    * `CONVERTED_TO_DISCUSSION_EVENT`: Represents a
      'converted_to_discussion' event on a given issue.
    * `CROSS_REFERENCED_EVENT`: Represents a mention made by one issue
      or pull request to another.
    * `DEMILESTONED_EVENT`: Represents a 'demilestoned' event on a
      given issue or pull request.
    * `DISCONNECTED_EVENT`: Represents a 'disconnected' event on a
      given issue or pull request.
    * `ISSUE_COMMENT`: Represents a comment on an Issue.
    * `LABELED_EVENT`: Represents a 'labeled' event on a given issue
      or pull request.
    * `LOCKED_EVENT`: Represents a 'locked' event on a given issue or
      pull request.
    * `MARKED_AS_DUPLICATE_EVENT`: Represents a 'marked_as_duplicate'
      event on a given issue or pull request.
    * `MENTIONED_EVENT`: Represents a 'mentioned' event on a given
      issue or pull request.
    * `MILESTONED_EVENT`: Represents a 'milestoned' event on a given
      issue or pull request.
    * `MOVED_COLUMNS_IN_PROJECT_EVENT`: Represents a
      'moved_columns_in_project' event on a given issue or pull
      request.
    * `PINNED_EVENT`: Represents a 'pinned' event on a given issue or
      pull request.
    * `REFERENCED_EVENT`: Represents a 'referenced' event on a given
      `ReferencedSubject`.
    * `REMOVED_FROM_PROJECT_EVENT`: Represents a
      'removed_from_project' event on a given issue or pull request.
    * `RENAMED_TITLE_EVENT`: Represents a 'renamed' event on a given
      issue or pull request
    * `REOPENED_EVENT`: Represents a 'reopened' event on any
      `Closable`.
    * `SUBSCRIBED_EVENT`: Represents a 'subscribed' event on a given
      `Subscribable`.
    * `TRANSFERRED_EVENT`: Represents a 'transferred' event on a given
      issue or pull request.
    * `UNASSIGNED_EVENT`: Represents an 'unassigned' event on any
      assignable object.
    * `UNLABELED_EVENT`: Represents an 'unlabeled' event on a given
      issue or pull request.
    * `UNLOCKED_EVENT`: Represents an 'unlocked' event on a given
      issue or pull request.
    * `UNMARKED_AS_DUPLICATE_EVENT`: Represents an
      'unmarked_as_duplicate' event on a given issue or pull request.
    * `UNPINNED_EVENT`: Represents an 'unpinned' event on a given
      issue or pull request.
    * `UNSUBSCRIBED_EVENT`: Represents an 'unsubscribed' event on a
      given `Subscribable`.
    * `USER_BLOCKED_EVENT`: Represents a 'user_blocked' event on a
      given user.
    """

    __schema__ = github_schema
    __choices__ = (
        "ADDED_TO_PROJECT_EVENT",
        "ASSIGNED_EVENT",
        "CLOSED_EVENT",
        "COMMENT_DELETED_EVENT",
        "CONNECTED_EVENT",
        "CONVERTED_NOTE_TO_ISSUE_EVENT",
        "CONVERTED_TO_DISCUSSION_EVENT",
        "CROSS_REFERENCED_EVENT",
        "DEMILESTONED_EVENT",
        "DISCONNECTED_EVENT",
        "ISSUE_COMMENT",
        "LABELED_EVENT",
        "LOCKED_EVENT",
        "MARKED_AS_DUPLICATE_EVENT",
        "MENTIONED_EVENT",
        "MILESTONED_EVENT",
        "MOVED_COLUMNS_IN_PROJECT_EVENT",
        "PINNED_EVENT",
        "REFERENCED_EVENT",
        "REMOVED_FROM_PROJECT_EVENT",
        "RENAMED_TITLE_EVENT",
        "REOPENED_EVENT",
        "SUBSCRIBED_EVENT",
        "TRANSFERRED_EVENT",
        "UNASSIGNED_EVENT",
        "UNLABELED_EVENT",
        "UNLOCKED_EVENT",
        "UNMARKED_AS_DUPLICATE_EVENT",
        "UNPINNED_EVENT",
        "UNSUBSCRIBED_EVENT",
        "USER_BLOCKED_EVENT",
    )


class LabelOrderField(sgqlc.types.Enum):
    """Properties by which label connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order labels by creation time
    * `NAME`: Order labels by name
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "NAME")


class LanguageOrderField(sgqlc.types.Enum):
    """Properties by which language connections can be ordered.

    Enumeration Choices:

    * `SIZE`: Order languages by the size of all files containing the
      language
    """

    __schema__ = github_schema
    __choices__ = ("SIZE",)


class LockReason(sgqlc.types.Enum):
    """The possible reasons that an issue or pull request was locked.

    Enumeration Choices:

    * `OFF_TOPIC`: The issue or pull request was locked because the
      conversation was off-topic.
    * `RESOLVED`: The issue or pull request was locked because the
      conversation was resolved.
    * `SPAM`: The issue or pull request was locked because the
      conversation was spam.
    * `TOO_HEATED`: The issue or pull request was locked because the
      conversation was too heated.
    """

    __schema__ = github_schema
    __choices__ = ("OFF_TOPIC", "RESOLVED", "SPAM", "TOO_HEATED")


class MergeStateStatus(sgqlc.types.Enum):
    """Detailed status information about a pull request merge.

    Enumeration Choices:

    * `BEHIND`: The head ref is out of date.
    * `BLOCKED`: The merge is blocked.
    * `CLEAN`: Mergeable and passing commit status.
    * `DIRTY`: The merge commit cannot be cleanly created.
    * `HAS_HOOKS`: Mergeable with passing commit status and prereceive hooks.
    * `UNKNOWN`: The state cannot currently be determined.
    * `UNSTABLE`: Mergeable with non-passing commit status.
    """

    __schema__ = github_schema
    __choices__ = ("BEHIND", "BLOCKED", "CLEAN", "DIRTY", "HAS_HOOKS", "UNKNOWN", "UNSTABLE")


class MannequinOrderField(sgqlc.types.Enum):
    """Properties by which mannequins can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order mannequins why when they were created.
    * `LOGIN`: Order mannequins alphabetically by their source login.
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "LOGIN")


class MergeCommitMessage(sgqlc.types.Enum):
    """The possible default commit messages for merges.

    Enumeration Choices:

    * `BLANK`: Default to a blank commit message.
    * `PR_BODY`: Default to the pull request's body.
    * `PR_TITLE`: Default to the pull request's title.
    """

    __schema__ = github_schema
    __choices__ = ("BLANK", "PR_BODY", "PR_TITLE")


class MergeCommitTitle(sgqlc.types.Enum):
    """The possible default commit titles for merges.

    Enumeration Choices:

    * `MERGE_MESSAGE`: Default to the classic title for a merge
      message (e.g., Merge pull request #123 from branch-name).
    * `PR_TITLE`: Default to the pull request's title.
    """

    __schema__ = github_schema
    __choices__ = ("MERGE_MESSAGE", "PR_TITLE")


class MergeQueueEntryState(sgqlc.types.Enum):
    """The possible states for a merge queue entry.

    Enumeration Choices:

    * `AWAITING_CHECKS`: The entry is currently waiting for checks to
      pass.
    * `LOCKED`: The entry is currently locked.
    * `MERGEABLE`: The entry is currently mergeable.
    * `QUEUED`: The entry is currently queued.
    * `UNMERGEABLE`: The entry is currently unmergeable.
    """

    __schema__ = github_schema
    __choices__ = ("AWAITING_CHECKS", "LOCKED", "MERGEABLE", "QUEUED", "UNMERGEABLE")


class MergeQueueMergingStrategy(sgqlc.types.Enum):
    """The possible merging strategies for a merge queue.

    Enumeration Choices:

    * `ALLGREEN`: Entries only allowed to merge if they are passing.
    * `HEADGREEN`: Failing Entires are allowed to merge if they are
      with a passing entry.
    """

    __schema__ = github_schema
    __choices__ = ("ALLGREEN", "HEADGREEN")


class MergeableState(sgqlc.types.Enum):
    """Whether or not a PullRequest can be merged.

    Enumeration Choices:

    * `CONFLICTING`: The pull request cannot be merged due to merge
      conflicts.
    * `MERGEABLE`: The pull request can be merged.
    * `UNKNOWN`: The mergeability of the pull request is still being
      calculated.
    """

    __schema__ = github_schema
    __choices__ = ("CONFLICTING", "MERGEABLE", "UNKNOWN")


class MigrationSourceType(sgqlc.types.Enum):
    """Represents the different GitHub Enterprise Importer (GEI)
    migration sources.

    Enumeration Choices:

    * `AZURE_DEVOPS`: An Azure DevOps migration source.
    * `BITBUCKET_SERVER`: A Bitbucket Server migration source.
    * `GITHUB_ARCHIVE`: A GitHub Migration API source.
    """

    __schema__ = github_schema
    __choices__ = ("AZURE_DEVOPS", "BITBUCKET_SERVER", "GITHUB_ARCHIVE")


class MigrationState(sgqlc.types.Enum):
    """The GitHub Enterprise Importer (GEI) migration state.

    Enumeration Choices:

    * `FAILED`: The migration has failed.
    * `FAILED_VALIDATION`: The migration has invalid credentials.
    * `IN_PROGRESS`: The migration is in progress.
    * `NOT_STARTED`: The migration has not started.
    * `PENDING_VALIDATION`: The migration needs to have its
      credentials validated.
    * `QUEUED`: The migration has been queued.
    * `SUCCEEDED`: The migration has succeeded.
    """

    __schema__ = github_schema
    __choices__ = ("FAILED", "FAILED_VALIDATION", "IN_PROGRESS", "NOT_STARTED", "PENDING_VALIDATION", "QUEUED", "SUCCEEDED")


class MilestoneOrderField(sgqlc.types.Enum):
    """Properties by which milestone connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order milestones by when they were created.
    * `DUE_DATE`: Order milestones by when they are due.
    * `NUMBER`: Order milestones by their number.
    * `UPDATED_AT`: Order milestones by when they were last updated.
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "DUE_DATE", "NUMBER", "UPDATED_AT")


class MilestoneState(sgqlc.types.Enum):
    """The possible states of a milestone.

    Enumeration Choices:

    * `CLOSED`: A milestone that has been closed.
    * `OPEN`: A milestone that is still open.
    """

    __schema__ = github_schema
    __choices__ = ("CLOSED", "OPEN")


class NotificationRestrictionSettingValue(sgqlc.types.Enum):
    """The possible values for the notification restriction setting.

    Enumeration Choices:

    * `DISABLED`: The setting is disabled for the owner.
    * `ENABLED`: The setting is enabled for the owner.
    """

    __schema__ = github_schema
    __choices__ = ("DISABLED", "ENABLED")


class OIDCProviderType(sgqlc.types.Enum):
    """The OIDC identity provider type

    Enumeration Choices:

    * `AAD`: Azure Active Directory
    """

    __schema__ = github_schema
    __choices__ = ("AAD",)


class OauthApplicationCreateAuditEntryState(sgqlc.types.Enum):
    """The state of an OAuth application when it was created.

    Enumeration Choices:

    * `ACTIVE`: The OAuth application was active and allowed to have
      OAuth Accesses.
    * `PENDING_DELETION`: The OAuth application was in the process of
      being deleted.
    * `SUSPENDED`: The OAuth application was suspended from generating
      OAuth Accesses due to abuse or security concerns.
    """

    __schema__ = github_schema
    __choices__ = ("ACTIVE", "PENDING_DELETION", "SUSPENDED")


class OperationType(sgqlc.types.Enum):
    """The corresponding operation type for the action

    Enumeration Choices:

    * `ACCESS`: An existing resource was accessed
    * `AUTHENTICATION`: A resource performed an authentication event
    * `CREATE`: A new resource was created
    * `MODIFY`: An existing resource was modified
    * `REMOVE`: An existing resource was removed
    * `RESTORE`: An existing resource was restored
    * `TRANSFER`: An existing resource was transferred between
      multiple resources
    """

    __schema__ = github_schema
    __choices__ = ("ACCESS", "AUTHENTICATION", "CREATE", "MODIFY", "REMOVE", "RESTORE", "TRANSFER")


class OrderDirection(sgqlc.types.Enum):
    """Possible directions in which to order a list of items when
    provided an `orderBy` argument.

    Enumeration Choices:

    * `ASC`: Specifies an ascending order for a given `orderBy`
      argument.
    * `DESC`: Specifies a descending order for a given `orderBy`
      argument.
    """

    __schema__ = github_schema
    __choices__ = ("ASC", "DESC")


class OrgAddMemberAuditEntryPermission(sgqlc.types.Enum):
    """The permissions available to members on an Organization.

    Enumeration Choices:

    * `ADMIN`: Can read, clone, push, and add collaborators to
      repositories.
    * `READ`: Can read and clone repositories.
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "READ")


class OrgCreateAuditEntryBillingPlan(sgqlc.types.Enum):
    """The billing plans available for organizations.

    Enumeration Choices:

    * `BUSINESS`: Team Plan
    * `BUSINESS_PLUS`: Enterprise Cloud Plan
    * `FREE`: Free Plan
    * `TIERED_PER_SEAT`: Tiered Per Seat Plan
    * `UNLIMITED`: Legacy Unlimited Plan
    """

    __schema__ = github_schema
    __choices__ = ("BUSINESS", "BUSINESS_PLUS", "FREE", "TIERED_PER_SEAT", "UNLIMITED")


class OrgEnterpriseOwnerOrderField(sgqlc.types.Enum):
    """Properties by which enterprise owners can be ordered.

    Enumeration Choices:

    * `LOGIN`: Order enterprise owners by login.
    """

    __schema__ = github_schema
    __choices__ = ("LOGIN",)


class OrgRemoveBillingManagerAuditEntryReason(sgqlc.types.Enum):
    """The reason a billing manager was removed from an Organization.

    Enumeration Choices:

    * `SAML_EXTERNAL_IDENTITY_MISSING`: SAML external identity missing
    * `SAML_SSO_ENFORCEMENT_REQUIRES_EXTERNAL_IDENTITY`: SAML SSO
      enforcement requires an external identity
    * `TWO_FACTOR_REQUIREMENT_NON_COMPLIANCE`: The organization
      required 2FA of its billing managers and this user did not have
      2FA enabled.
    """

    __schema__ = github_schema
    __choices__ = (
        "SAML_EXTERNAL_IDENTITY_MISSING",
        "SAML_SSO_ENFORCEMENT_REQUIRES_EXTERNAL_IDENTITY",
        "TWO_FACTOR_REQUIREMENT_NON_COMPLIANCE",
    )


class OrgRemoveMemberAuditEntryMembershipType(sgqlc.types.Enum):
    """The type of membership a user has with an Organization.

    Enumeration Choices:

    * `ADMIN`: Organization administrators have full access and can
      change several settings, including the names of repositories
      that belong to the Organization and Owners team membership. In
      addition, organization admins can delete the organization and
      all of its repositories.
    * `BILLING_MANAGER`: A billing manager is a user who manages the
      billing settings for the Organization, such as updating payment
      information.
    * `DIRECT_MEMBER`: A direct member is a user that is a member of
      the Organization.
    * `OUTSIDE_COLLABORATOR`: An outside collaborator is a person who
      isn't explicitly a member of the Organization, but who has Read,
      Write, or Admin permissions to one or more repositories in the
      organization.
    * `SUSPENDED`: A suspended member.
    * `UNAFFILIATED`: An unaffiliated collaborator is a person who is
      not a member of the Organization and does not have access to any
      repositories in the Organization.
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "BILLING_MANAGER", "DIRECT_MEMBER", "OUTSIDE_COLLABORATOR", "SUSPENDED", "UNAFFILIATED")


class OrgRemoveMemberAuditEntryReason(sgqlc.types.Enum):
    """The reason a member was removed from an Organization.

    Enumeration Choices:

    * `SAML_EXTERNAL_IDENTITY_MISSING`: SAML external identity missing
    * `SAML_SSO_ENFORCEMENT_REQUIRES_EXTERNAL_IDENTITY`: SAML SSO
      enforcement requires an external identity
    * `TWO_FACTOR_ACCOUNT_RECOVERY`: User was removed from
      organization during account recovery
    * `TWO_FACTOR_REQUIREMENT_NON_COMPLIANCE`: The organization
      required 2FA of its billing managers and this user did not have
      2FA enabled.
    * `USER_ACCOUNT_DELETED`: User account has been deleted
    """

    __schema__ = github_schema
    __choices__ = (
        "SAML_EXTERNAL_IDENTITY_MISSING",
        "SAML_SSO_ENFORCEMENT_REQUIRES_EXTERNAL_IDENTITY",
        "TWO_FACTOR_ACCOUNT_RECOVERY",
        "TWO_FACTOR_REQUIREMENT_NON_COMPLIANCE",
        "USER_ACCOUNT_DELETED",
    )


class OrgRemoveOutsideCollaboratorAuditEntryMembershipType(sgqlc.types.Enum):
    """The type of membership a user has with an Organization.

    Enumeration Choices:

    * `BILLING_MANAGER`: A billing manager is a user who manages the
      billing settings for the Organization, such as updating payment
      information.
    * `OUTSIDE_COLLABORATOR`: An outside collaborator is a person who
      isn't explicitly a member of the Organization, but who has Read,
      Write, or Admin permissions to one or more repositories in the
      organization.
    * `UNAFFILIATED`: An unaffiliated collaborator is a person who is
      not a member of the Organization and does not have access to any
      repositories in the organization.
    """

    __schema__ = github_schema
    __choices__ = ("BILLING_MANAGER", "OUTSIDE_COLLABORATOR", "UNAFFILIATED")


class OrgRemoveOutsideCollaboratorAuditEntryReason(sgqlc.types.Enum):
    """The reason an outside collaborator was removed from an
    Organization.

    Enumeration Choices:

    * `SAML_EXTERNAL_IDENTITY_MISSING`: SAML external identity missing
    * `TWO_FACTOR_REQUIREMENT_NON_COMPLIANCE`: The organization
      required 2FA of its billing managers and this user did not have
      2FA enabled.
    """

    __schema__ = github_schema
    __choices__ = ("SAML_EXTERNAL_IDENTITY_MISSING", "TWO_FACTOR_REQUIREMENT_NON_COMPLIANCE")


class OrgUpdateDefaultRepositoryPermissionAuditEntryPermission(sgqlc.types.Enum):
    """The default permission a repository can have in an Organization.

    Enumeration Choices:

    * `ADMIN`: Can read, clone, push, and add collaborators to
      repositories.
    * `NONE`: No default permission value.
    * `READ`: Can read and clone repositories.
    * `WRITE`: Can read, clone and push to repositories.
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "NONE", "READ", "WRITE")


class OrgUpdateMemberAuditEntryPermission(sgqlc.types.Enum):
    """The permissions available to members on an Organization.

    Enumeration Choices:

    * `ADMIN`: Can read, clone, push, and add collaborators to
      repositories.
    * `READ`: Can read and clone repositories.
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "READ")


class OrgUpdateMemberRepositoryCreationPermissionAuditEntryVisibility(sgqlc.types.Enum):
    """The permissions available for repository creation on an
    Organization.

    Enumeration Choices:

    * `ALL`: All organization members are restricted from creating any
      repositories.
    * `INTERNAL`: All organization members are restricted from
      creating internal repositories.
    * `NONE`: All organization members are allowed to create any
      repositories.
    * `PRIVATE`: All organization members are restricted from creating
      private repositories.
    * `PRIVATE_INTERNAL`: All organization members are restricted from
      creating private or internal repositories.
    * `PUBLIC`: All organization members are restricted from creating
      public repositories.
    * `PUBLIC_INTERNAL`: All organization members are restricted from
      creating public or internal repositories.
    * `PUBLIC_PRIVATE`: All organization members are restricted from
      creating public or private repositories.
    """

    __schema__ = github_schema
    __choices__ = ("ALL", "INTERNAL", "NONE", "PRIVATE", "PRIVATE_INTERNAL", "PUBLIC", "PUBLIC_INTERNAL", "PUBLIC_PRIVATE")


class OrganizationInvitationRole(sgqlc.types.Enum):
    """The possible organization invitation roles.

    Enumeration Choices:

    * `ADMIN`: The user is invited to be an admin of the organization.
    * `BILLING_MANAGER`: The user is invited to be a billing manager
      of the organization.
    * `DIRECT_MEMBER`: The user is invited to be a direct member of
      the organization.
    * `REINSTATE`: The user's previous role will be reinstated.
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "BILLING_MANAGER", "DIRECT_MEMBER", "REINSTATE")


class OrganizationInvitationSource(sgqlc.types.Enum):
    """The possible organization invitation sources.

    Enumeration Choices:

    * `MEMBER`: The invitation was created from the web interface or
      from API
    * `SCIM`: The invitation was created from SCIM
    * `UNKNOWN`: The invitation was sent before this feature was added
    """

    __schema__ = github_schema
    __choices__ = ("MEMBER", "SCIM", "UNKNOWN")


class OrganizationInvitationType(sgqlc.types.Enum):
    """The possible organization invitation types.

    Enumeration Choices:

    * `EMAIL`: The invitation was to an email address.
    * `USER`: The invitation was to an existing user.
    """

    __schema__ = github_schema
    __choices__ = ("EMAIL", "USER")


class OrganizationMemberRole(sgqlc.types.Enum):
    """The possible roles within an organization for its members.

    Enumeration Choices:

    * `ADMIN`: The user is an administrator of the organization.
    * `MEMBER`: The user is a member of the organization.
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "MEMBER")


class OrganizationMembersCanCreateRepositoriesSettingValue(sgqlc.types.Enum):
    """The possible values for the members can create repositories
    setting on an organization.

    Enumeration Choices:

    * `ALL`: Members will be able to create public and private
      repositories.
    * `DISABLED`: Members will not be able to create public or private
      repositories.
    * `INTERNAL`: Members will be able to create only internal
      repositories.
    * `PRIVATE`: Members will be able to create only private
      repositories.
    """

    __schema__ = github_schema
    __choices__ = ("ALL", "DISABLED", "INTERNAL", "PRIVATE")


class OrganizationMigrationState(sgqlc.types.Enum):
    """The Octoshift Organization migration state.

    Enumeration Choices:

    * `FAILED`: The Octoshift migration has failed.
    * `FAILED_VALIDATION`: The Octoshift migration has invalid
      credentials.
    * `IN_PROGRESS`: The Octoshift migration is in progress.
    * `NOT_STARTED`: The Octoshift migration has not started.
    * `PENDING_VALIDATION`: The Octoshift migration needs to have its
      credentials validated.
    * `POST_REPO_MIGRATION`: The Octoshift migration is performing
      post repository migrations.
    * `PRE_REPO_MIGRATION`: The Octoshift migration is performing pre
      repository migrations.
    * `QUEUED`: The Octoshift migration has been queued.
    * `REPO_MIGRATION`: The Octoshift org migration is performing
      repository migrations.
    * `SUCCEEDED`: The Octoshift migration has succeeded.
    """

    __schema__ = github_schema
    __choices__ = (
        "FAILED",
        "FAILED_VALIDATION",
        "IN_PROGRESS",
        "NOT_STARTED",
        "PENDING_VALIDATION",
        "POST_REPO_MIGRATION",
        "PRE_REPO_MIGRATION",
        "QUEUED",
        "REPO_MIGRATION",
        "SUCCEEDED",
    )


class OrganizationOrderField(sgqlc.types.Enum):
    """Properties by which organization connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order organizations by creation time
    * `LOGIN`: Order organizations by login
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "LOGIN")


class PackageFileOrderField(sgqlc.types.Enum):
    """Properties by which package file connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order package files by creation time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class PackageOrderField(sgqlc.types.Enum):
    """Properties by which package connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order packages by creation time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class PackageType(sgqlc.types.Enum):
    """The possible types of a package.

    Enumeration Choices:

    * `DEBIAN`: A debian package.
    * `PYPI`: A python package.
    """

    __schema__ = github_schema
    __choices__ = ("DEBIAN", "PYPI")


class PackageVersionOrderField(sgqlc.types.Enum):
    """Properties by which package version connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order package versions by creation time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class PatchStatus(sgqlc.types.Enum):
    """The possible types of patch statuses.

    Enumeration Choices:

    * `ADDED`: The file was added. Git status 'A'.
    * `CHANGED`: The file's type was changed. Git status 'T'.
    * `COPIED`: The file was copied. Git status 'C'.
    * `DELETED`: The file was deleted. Git status 'D'.
    * `MODIFIED`: The file's contents were changed. Git status 'M'.
    * `RENAMED`: The file was renamed. Git status 'R'.
    """

    __schema__ = github_schema
    __choices__ = ("ADDED", "CHANGED", "COPIED", "DELETED", "MODIFIED", "RENAMED")


class PinnableItemType(sgqlc.types.Enum):
    """Represents items that can be pinned to a profile page or
    dashboard.

    Enumeration Choices:

    * `GIST`: A gist.
    * `ISSUE`: An issue.
    * `ORGANIZATION`: An organization.
    * `PROJECT`: A project.
    * `PULL_REQUEST`: A pull request.
    * `REPOSITORY`: A repository.
    * `TEAM`: A team.
    * `USER`: A user.
    """

    __schema__ = github_schema
    __choices__ = ("GIST", "ISSUE", "ORGANIZATION", "PROJECT", "PULL_REQUEST", "REPOSITORY", "TEAM", "USER")


class PinnedDiscussionGradient(sgqlc.types.Enum):
    """Preconfigured gradients that may be used to style discussions
    pinned within a repository.

    Enumeration Choices:

    * `BLUE_MINT`: A gradient of blue to mint
    * `BLUE_PURPLE`: A gradient of blue to purple
    * `PINK_BLUE`: A gradient of pink to blue
    * `PURPLE_CORAL`: A gradient of purple to coral
    * `RED_ORANGE`: A gradient of red to orange
    """

    __schema__ = github_schema
    __choices__ = ("BLUE_MINT", "BLUE_PURPLE", "PINK_BLUE", "PURPLE_CORAL", "RED_ORANGE")


class PinnedDiscussionPattern(sgqlc.types.Enum):
    """Preconfigured background patterns that may be used to style
    discussions pinned within a repository.

    Enumeration Choices:

    * `CHEVRON_UP`: An upward-facing chevron pattern
    * `DOT`: A hollow dot pattern
    * `DOT_FILL`: A solid dot pattern
    * `HEART_FILL`: A heart pattern
    * `PLUS`: A plus sign pattern
    * `ZAP`: A lightning bolt pattern
    """

    __schema__ = github_schema
    __choices__ = ("CHEVRON_UP", "DOT", "DOT_FILL", "HEART_FILL", "PLUS", "ZAP")


class PreciseDateTime(sgqlc.types.Scalar):
    """An ISO-8601 encoded UTC date string with millisecond precision."""

    __schema__ = github_schema


class ProjectCardArchivedState(sgqlc.types.Enum):
    """The possible archived states of a project card.

    Enumeration Choices:

    * `ARCHIVED`: A project card that is archived
    * `NOT_ARCHIVED`: A project card that is not archived
    """

    __schema__ = github_schema
    __choices__ = ("ARCHIVED", "NOT_ARCHIVED")


class ProjectCardState(sgqlc.types.Enum):
    """Various content states of a ProjectCard

    Enumeration Choices:

    * `CONTENT_ONLY`: The card has content only.
    * `NOTE_ONLY`: The card has a note only.
    * `REDACTED`: The card is redacted.
    """

    __schema__ = github_schema
    __choices__ = ("CONTENT_ONLY", "NOTE_ONLY", "REDACTED")


class ProjectColumnPurpose(sgqlc.types.Enum):
    """The semantic purpose of the column - todo, in progress, or done.

    Enumeration Choices:

    * `DONE`: The column contains cards which are complete
    * `IN_PROGRESS`: The column contains cards which are currently
      being worked on
    * `TODO`: The column contains cards still to be worked on
    """

    __schema__ = github_schema
    __choices__ = ("DONE", "IN_PROGRESS", "TODO")


class ProjectOrderField(sgqlc.types.Enum):
    """Properties by which project connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order projects by creation time
    * `NAME`: Order projects by name
    * `UPDATED_AT`: Order projects by update time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "NAME", "UPDATED_AT")


class ProjectState(sgqlc.types.Enum):
    """State of the project; either 'open' or 'closed'

    Enumeration Choices:

    * `CLOSED`: The project is closed.
    * `OPEN`: The project is open.
    """

    __schema__ = github_schema
    __choices__ = ("CLOSED", "OPEN")


class ProjectTemplate(sgqlc.types.Enum):
    """GitHub-provided templates for Projects

    Enumeration Choices:

    * `AUTOMATED_KANBAN_V2`: Create a board with v2 triggers to
      automatically move cards across To do, In progress and Done
      columns.
    * `AUTOMATED_REVIEWS_KANBAN`: Create a board with triggers to
      automatically move cards across columns with review automation.
    * `BASIC_KANBAN`: Create a board with columns for To do, In
      progress and Done.
    * `BUG_TRIAGE`: Create a board to triage and prioritize bugs with
      To do, priority, and Done columns.
    """

    __schema__ = github_schema
    __choices__ = ("AUTOMATED_KANBAN_V2", "AUTOMATED_REVIEWS_KANBAN", "BASIC_KANBAN", "BUG_TRIAGE")


class ProjectV2CustomFieldType(sgqlc.types.Enum):
    """The type of a project field.

    Enumeration Choices:

    * `DATE`: Date
    * `NUMBER`: Number
    * `SINGLE_SELECT`: Single Select
    * `TEXT`: Text
    """

    __schema__ = github_schema
    __choices__ = ("DATE", "NUMBER", "SINGLE_SELECT", "TEXT")


class ProjectV2FieldOrderField(sgqlc.types.Enum):
    """Properties by which project v2 field connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order project v2 fields by creation time
    * `NAME`: Order project v2 fields by name
    * `POSITION`: Order project v2 fields by position
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "NAME", "POSITION")


class ProjectV2FieldType(sgqlc.types.Enum):
    """The type of a project field.

    Enumeration Choices:

    * `ASSIGNEES`: Assignees
    * `DATE`: Date
    * `ITERATION`: Iteration
    * `LABELS`: Labels
    * `LINKED_PULL_REQUESTS`: Linked Pull Requests
    * `MILESTONE`: Milestone
    * `NUMBER`: Number
    * `REPOSITORY`: Repository
    * `REVIEWERS`: Reviewers
    * `SINGLE_SELECT`: Single Select
    * `TEXT`: Text
    * `TITLE`: Title
    * `TRACKED_BY`: Tracked by
    * `TRACKS`: Tracks
    """

    __schema__ = github_schema
    __choices__ = (
        "ASSIGNEES",
        "DATE",
        "ITERATION",
        "LABELS",
        "LINKED_PULL_REQUESTS",
        "MILESTONE",
        "NUMBER",
        "REPOSITORY",
        "REVIEWERS",
        "SINGLE_SELECT",
        "TEXT",
        "TITLE",
        "TRACKED_BY",
        "TRACKS",
    )


class ProjectV2ItemFieldValueOrderField(sgqlc.types.Enum):
    """Properties by which project v2 item field value connections can be
    ordered.

    Enumeration Choices:

    * `POSITION`: Order project v2 item field values by the their
      position in the project
    """

    __schema__ = github_schema
    __choices__ = ("POSITION",)


class ProjectV2ItemOrderField(sgqlc.types.Enum):
    """Properties by which project v2 item connections can be ordered.

    Enumeration Choices:

    * `POSITION`: Order project v2 items by the their position in the
      project
    """

    __schema__ = github_schema
    __choices__ = ("POSITION",)


class ProjectV2ItemType(sgqlc.types.Enum):
    """The type of a project item.

    Enumeration Choices:

    * `DRAFT_ISSUE`: Draft Issue
    * `ISSUE`: Issue
    * `PULL_REQUEST`: Pull Request
    * `REDACTED`: Redacted Item
    """

    __schema__ = github_schema
    __choices__ = ("DRAFT_ISSUE", "ISSUE", "PULL_REQUEST", "REDACTED")


class ProjectV2OrderField(sgqlc.types.Enum):
    """Properties by which projects can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: The project's date and time of creation
    * `NUMBER`: The project's number
    * `TITLE`: The project's title
    * `UPDATED_AT`: The project's date and time of update
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "NUMBER", "TITLE", "UPDATED_AT")


class ProjectV2Roles(sgqlc.types.Enum):
    """The possible roles of a collaborator on a project.

    Enumeration Choices:

    * `ADMIN`: The collaborator can view, edit, and maange the
      settings of the project
    * `NONE`: The collaborator has no direct access to the project
    * `READER`: The collaborator can view the project
    * `WRITER`: The collaborator can view and edit the project
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "NONE", "READER", "WRITER")


class ProjectV2SingleSelectFieldOptionColor(sgqlc.types.Enum):
    """The display color of a single-select field option.

    Enumeration Choices:

    * `BLUE`: BLUE
    * `GRAY`: GRAY
    * `GREEN`: GREEN
    * `ORANGE`: ORANGE
    * `PINK`: PINK
    * `PURPLE`: PURPLE
    * `RED`: RED
    * `YELLOW`: YELLOW
    """

    __schema__ = github_schema
    __choices__ = ("BLUE", "GRAY", "GREEN", "ORANGE", "PINK", "PURPLE", "RED", "YELLOW")


class ProjectV2State(sgqlc.types.Enum):
    """The possible states of a project v2.

    Enumeration Choices:

    * `CLOSED`: A project v2 that has been closed
    * `OPEN`: A project v2 that is still open
    """

    __schema__ = github_schema
    __choices__ = ("CLOSED", "OPEN")


class ProjectV2ViewLayout(sgqlc.types.Enum):
    """The layout of a project v2 view.

    Enumeration Choices:

    * `BOARD_LAYOUT`: Board layout
    * `ROADMAP_LAYOUT`: Roadmap layout
    * `TABLE_LAYOUT`: Table layout
    """

    __schema__ = github_schema
    __choices__ = ("BOARD_LAYOUT", "ROADMAP_LAYOUT", "TABLE_LAYOUT")


class ProjectV2ViewOrderField(sgqlc.types.Enum):
    """Properties by which project v2 view connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order project v2 views by creation time
    * `NAME`: Order project v2 views by name
    * `POSITION`: Order project v2 views by position
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "NAME", "POSITION")


class ProjectV2WorkflowsOrderField(sgqlc.types.Enum):
    """Properties by which project workflows can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: The workflows' date and time of creation
    * `NAME`: The workflows' name
    * `NUMBER`: The workflows' number
    * `UPDATED_AT`: The workflows' date and time of update
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "NAME", "NUMBER", "UPDATED_AT")


class PullRequestMergeMethod(sgqlc.types.Enum):
    """Represents available types of methods to use when merging a pull
    request.

    Enumeration Choices:

    * `MERGE`: Add all commits from the head branch to the base branch
      with a merge commit.
    * `REBASE`: Add all commits from the head branch onto the base
      branch individually.
    * `SQUASH`: Combine all commits from the head branch into a single
      commit in the base branch.
    """

    __schema__ = github_schema
    __choices__ = ("MERGE", "REBASE", "SQUASH")


class PullRequestOrderField(sgqlc.types.Enum):
    """Properties by which pull_requests connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order pull_requests by creation time
    * `UPDATED_AT`: Order pull_requests by update time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "UPDATED_AT")


class PullRequestReviewCommentState(sgqlc.types.Enum):
    """The possible states of a pull request review comment.

    Enumeration Choices:

    * `PENDING`: A comment that is part of a pending review
    * `SUBMITTED`: A comment that is part of a submitted review
    """

    __schema__ = github_schema
    __choices__ = ("PENDING", "SUBMITTED")


class PullRequestReviewDecision(sgqlc.types.Enum):
    """The review status of a pull request.

    Enumeration Choices:

    * `APPROVED`: The pull request has received an approving review.
    * `CHANGES_REQUESTED`: Changes have been requested on the pull
      request.
    * `REVIEW_REQUIRED`: A review is required before the pull request
      can be merged.
    """

    __schema__ = github_schema
    __choices__ = ("APPROVED", "CHANGES_REQUESTED", "REVIEW_REQUIRED")


class PullRequestReviewEvent(sgqlc.types.Enum):
    """The possible events to perform on a pull request review.

    Enumeration Choices:

    * `APPROVE`: Submit feedback and approve merging these changes.
    * `COMMENT`: Submit general feedback without explicit approval.
    * `DISMISS`: Dismiss review so it now longer effects merging.
    * `REQUEST_CHANGES`: Submit feedback that must be addressed before
      merging.
    """

    __schema__ = github_schema
    __choices__ = ("APPROVE", "COMMENT", "DISMISS", "REQUEST_CHANGES")


class PullRequestReviewState(sgqlc.types.Enum):
    """The possible states of a pull request review.

    Enumeration Choices:

    * `APPROVED`: A review allowing the pull request to merge.
    * `CHANGES_REQUESTED`: A review blocking the pull request from
      merging.
    * `COMMENTED`: An informational review.
    * `DISMISSED`: A review that has been dismissed.
    * `PENDING`: A review that has not yet been submitted.
    """

    __schema__ = github_schema
    __choices__ = ("APPROVED", "CHANGES_REQUESTED", "COMMENTED", "DISMISSED", "PENDING")


class PullRequestReviewThreadSubjectType(sgqlc.types.Enum):
    """The possible subject types of a pull request review comment.

    Enumeration Choices:

    * `FILE`: A comment that has been made against the file of a pull
      request
    * `LINE`: A comment that has been made against the line of a pull
      request
    """

    __schema__ = github_schema
    __choices__ = ("FILE", "LINE")


class PullRequestState(sgqlc.types.Enum):
    """The possible states of a pull request.

    Enumeration Choices:

    * `CLOSED`: A pull request that has been closed without being
      merged.
    * `MERGED`: A pull request that has been closed by being merged.
    * `OPEN`: A pull request that is still open.
    """

    __schema__ = github_schema
    __choices__ = ("CLOSED", "MERGED", "OPEN")


class PullRequestTimelineItemsItemType(sgqlc.types.Enum):
    """The possible item types found in a timeline.

    Enumeration Choices:

    * `ADDED_TO_MERGE_QUEUE_EVENT`: Represents an
      'added_to_merge_queue' event on a given pull request.
    * `ADDED_TO_PROJECT_EVENT`: Represents a 'added_to_project' event
      on a given issue or pull request.
    * `ASSIGNED_EVENT`: Represents an 'assigned' event on any
      assignable object.
    * `AUTOMATIC_BASE_CHANGE_FAILED_EVENT`: Represents a
      'automatic_base_change_failed' event on a given pull request.
    * `AUTOMATIC_BASE_CHANGE_SUCCEEDED_EVENT`: Represents a
      'automatic_base_change_succeeded' event on a given pull request.
    * `AUTO_MERGE_DISABLED_EVENT`: Represents a 'auto_merge_disabled'
      event on a given pull request.
    * `AUTO_MERGE_ENABLED_EVENT`: Represents a 'auto_merge_enabled'
      event on a given pull request.
    * `AUTO_REBASE_ENABLED_EVENT`: Represents a 'auto_rebase_enabled'
      event on a given pull request.
    * `AUTO_SQUASH_ENABLED_EVENT`: Represents a 'auto_squash_enabled'
      event on a given pull request.
    * `BASE_REF_CHANGED_EVENT`: Represents a 'base_ref_changed' event
      on a given issue or pull request.
    * `BASE_REF_DELETED_EVENT`: Represents a 'base_ref_deleted' event
      on a given pull request.
    * `BASE_REF_FORCE_PUSHED_EVENT`: Represents a
      'base_ref_force_pushed' event on a given pull request.
    * `CLOSED_EVENT`: Represents a 'closed' event on any `Closable`.
    * `COMMENT_DELETED_EVENT`: Represents a 'comment_deleted' event on
      a given issue or pull request.
    * `CONNECTED_EVENT`: Represents a 'connected' event on a given
      issue or pull request.
    * `CONVERTED_NOTE_TO_ISSUE_EVENT`: Represents a
      'converted_note_to_issue' event on a given issue or pull
      request.
    * `CONVERTED_TO_DISCUSSION_EVENT`: Represents a
      'converted_to_discussion' event on a given issue.
    * `CONVERT_TO_DRAFT_EVENT`: Represents a 'convert_to_draft' event
      on a given pull request.
    * `CROSS_REFERENCED_EVENT`: Represents a mention made by one issue
      or pull request to another.
    * `DEMILESTONED_EVENT`: Represents a 'demilestoned' event on a
      given issue or pull request.
    * `DEPLOYED_EVENT`: Represents a 'deployed' event on a given pull
      request.
    * `DEPLOYMENT_ENVIRONMENT_CHANGED_EVENT`: Represents a
      'deployment_environment_changed' event on a given pull request.
    * `DISCONNECTED_EVENT`: Represents a 'disconnected' event on a
      given issue or pull request.
    * `HEAD_REF_DELETED_EVENT`: Represents a 'head_ref_deleted' event
      on a given pull request.
    * `HEAD_REF_FORCE_PUSHED_EVENT`: Represents a
      'head_ref_force_pushed' event on a given pull request.
    * `HEAD_REF_RESTORED_EVENT`: Represents a 'head_ref_restored'
      event on a given pull request.
    * `ISSUE_COMMENT`: Represents a comment on an Issue.
    * `LABELED_EVENT`: Represents a 'labeled' event on a given issue
      or pull request.
    * `LOCKED_EVENT`: Represents a 'locked' event on a given issue or
      pull request.
    * `MARKED_AS_DUPLICATE_EVENT`: Represents a 'marked_as_duplicate'
      event on a given issue or pull request.
    * `MENTIONED_EVENT`: Represents a 'mentioned' event on a given
      issue or pull request.
    * `MERGED_EVENT`: Represents a 'merged' event on a given pull
      request.
    * `MILESTONED_EVENT`: Represents a 'milestoned' event on a given
      issue or pull request.
    * `MOVED_COLUMNS_IN_PROJECT_EVENT`: Represents a
      'moved_columns_in_project' event on a given issue or pull
      request.
    * `PINNED_EVENT`: Represents a 'pinned' event on a given issue or
      pull request.
    * `PULL_REQUEST_COMMIT`: Represents a Git commit part of a pull
      request.
    * `PULL_REQUEST_COMMIT_COMMENT_THREAD`: Represents a commit
      comment thread part of a pull request.
    * `PULL_REQUEST_REVIEW`: A review object for a given pull request.
    * `PULL_REQUEST_REVIEW_THREAD`: A threaded list of comments for a
      given pull request.
    * `PULL_REQUEST_REVISION_MARKER`: Represents the latest point in
      the pull request timeline for which the viewer has seen the pull
      request's commits.
    * `READY_FOR_REVIEW_EVENT`: Represents a 'ready_for_review' event
      on a given pull request.
    * `REFERENCED_EVENT`: Represents a 'referenced' event on a given
      `ReferencedSubject`.
    * `REMOVED_FROM_MERGE_QUEUE_EVENT`: Represents a
      'removed_from_merge_queue' event on a given pull request.
    * `REMOVED_FROM_PROJECT_EVENT`: Represents a
      'removed_from_project' event on a given issue or pull request.
    * `RENAMED_TITLE_EVENT`: Represents a 'renamed' event on a given
      issue or pull request
    * `REOPENED_EVENT`: Represents a 'reopened' event on any
      `Closable`.
    * `REVIEW_DISMISSED_EVENT`: Represents a 'review_dismissed' event
      on a given issue or pull request.
    * `REVIEW_REQUESTED_EVENT`: Represents an 'review_requested' event
      on a given pull request.
    * `REVIEW_REQUEST_REMOVED_EVENT`: Represents an
      'review_request_removed' event on a given pull request.
    * `SUBSCRIBED_EVENT`: Represents a 'subscribed' event on a given
      `Subscribable`.
    * `TRANSFERRED_EVENT`: Represents a 'transferred' event on a given
      issue or pull request.
    * `UNASSIGNED_EVENT`: Represents an 'unassigned' event on any
      assignable object.
    * `UNLABELED_EVENT`: Represents an 'unlabeled' event on a given
      issue or pull request.
    * `UNLOCKED_EVENT`: Represents an 'unlocked' event on a given
      issue or pull request.
    * `UNMARKED_AS_DUPLICATE_EVENT`: Represents an
      'unmarked_as_duplicate' event on a given issue or pull request.
    * `UNPINNED_EVENT`: Represents an 'unpinned' event on a given
      issue or pull request.
    * `UNSUBSCRIBED_EVENT`: Represents an 'unsubscribed' event on a
      given `Subscribable`.
    * `USER_BLOCKED_EVENT`: Represents a 'user_blocked' event on a
      given user.
    """

    __schema__ = github_schema
    __choices__ = (
        "ADDED_TO_MERGE_QUEUE_EVENT",
        "ADDED_TO_PROJECT_EVENT",
        "ASSIGNED_EVENT",
        "AUTOMATIC_BASE_CHANGE_FAILED_EVENT",
        "AUTOMATIC_BASE_CHANGE_SUCCEEDED_EVENT",
        "AUTO_MERGE_DISABLED_EVENT",
        "AUTO_MERGE_ENABLED_EVENT",
        "AUTO_REBASE_ENABLED_EVENT",
        "AUTO_SQUASH_ENABLED_EVENT",
        "BASE_REF_CHANGED_EVENT",
        "BASE_REF_DELETED_EVENT",
        "BASE_REF_FORCE_PUSHED_EVENT",
        "CLOSED_EVENT",
        "COMMENT_DELETED_EVENT",
        "CONNECTED_EVENT",
        "CONVERTED_NOTE_TO_ISSUE_EVENT",
        "CONVERTED_TO_DISCUSSION_EVENT",
        "CONVERT_TO_DRAFT_EVENT",
        "CROSS_REFERENCED_EVENT",
        "DEMILESTONED_EVENT",
        "DEPLOYED_EVENT",
        "DEPLOYMENT_ENVIRONMENT_CHANGED_EVENT",
        "DISCONNECTED_EVENT",
        "HEAD_REF_DELETED_EVENT",
        "HEAD_REF_FORCE_PUSHED_EVENT",
        "HEAD_REF_RESTORED_EVENT",
        "ISSUE_COMMENT",
        "LABELED_EVENT",
        "LOCKED_EVENT",
        "MARKED_AS_DUPLICATE_EVENT",
        "MENTIONED_EVENT",
        "MERGED_EVENT",
        "MILESTONED_EVENT",
        "MOVED_COLUMNS_IN_PROJECT_EVENT",
        "PINNED_EVENT",
        "PULL_REQUEST_COMMIT",
        "PULL_REQUEST_COMMIT_COMMENT_THREAD",
        "PULL_REQUEST_REVIEW",
        "PULL_REQUEST_REVIEW_THREAD",
        "PULL_REQUEST_REVISION_MARKER",
        "READY_FOR_REVIEW_EVENT",
        "REFERENCED_EVENT",
        "REMOVED_FROM_MERGE_QUEUE_EVENT",
        "REMOVED_FROM_PROJECT_EVENT",
        "RENAMED_TITLE_EVENT",
        "REOPENED_EVENT",
        "REVIEW_DISMISSED_EVENT",
        "REVIEW_REQUESTED_EVENT",
        "REVIEW_REQUEST_REMOVED_EVENT",
        "SUBSCRIBED_EVENT",
        "TRANSFERRED_EVENT",
        "UNASSIGNED_EVENT",
        "UNLABELED_EVENT",
        "UNLOCKED_EVENT",
        "UNMARKED_AS_DUPLICATE_EVENT",
        "UNPINNED_EVENT",
        "UNSUBSCRIBED_EVENT",
        "USER_BLOCKED_EVENT",
    )


class PullRequestUpdateState(sgqlc.types.Enum):
    """The possible target states when updating a pull request.

    Enumeration Choices:

    * `CLOSED`: A pull request that has been closed without being
      merged.
    * `OPEN`: A pull request that is still open.
    """

    __schema__ = github_schema
    __choices__ = ("CLOSED", "OPEN")


class ReactionContent(sgqlc.types.Enum):
    """Emojis that can be attached to Issues, Pull Requests and Comments.

    Enumeration Choices:

    * `CONFUSED`: Represents the `:confused:` emoji.
    * `EYES`: Represents the `:eyes:` emoji.
    * `HEART`: Represents the `:heart:` emoji.
    * `HOORAY`: Represents the `:hooray:` emoji.
    * `LAUGH`: Represents the `:laugh:` emoji.
    * `ROCKET`: Represents the `:rocket:` emoji.
    * `THUMBS_DOWN`: Represents the `:-1:` emoji.
    * `THUMBS_UP`: Represents the `:+1:` emoji.
    """

    __schema__ = github_schema
    __choices__ = ("CONFUSED", "EYES", "HEART", "HOORAY", "LAUGH", "ROCKET", "THUMBS_DOWN", "THUMBS_UP")


class ReactionOrderField(sgqlc.types.Enum):
    """A list of fields that reactions can be ordered by.

    Enumeration Choices:

    * `CREATED_AT`: Allows ordering a list of reactions by when they
      were created.
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class RefOrderField(sgqlc.types.Enum):
    """Properties by which ref connections can be ordered.

    Enumeration Choices:

    * `ALPHABETICAL`: Order refs by their alphanumeric name
    * `TAG_COMMIT_DATE`: Order refs by underlying commit date if the
      ref prefix is refs/tags/
    """

    __schema__ = github_schema
    __choices__ = ("ALPHABETICAL", "TAG_COMMIT_DATE")


class ReleaseOrderField(sgqlc.types.Enum):
    """Properties by which release connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order releases by creation time
    * `NAME`: Order releases alphabetically by name
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "NAME")


class RepoAccessAuditEntryVisibility(sgqlc.types.Enum):
    """The privacy of a repository

    Enumeration Choices:

    * `INTERNAL`: The repository is visible only to users in the same
      business.
    * `PRIVATE`: The repository is visible only to those with explicit
      access.
    * `PUBLIC`: The repository is visible to everyone.
    """

    __schema__ = github_schema
    __choices__ = ("INTERNAL", "PRIVATE", "PUBLIC")


class RepoAddMemberAuditEntryVisibility(sgqlc.types.Enum):
    """The privacy of a repository

    Enumeration Choices:

    * `INTERNAL`: The repository is visible only to users in the same
      business.
    * `PRIVATE`: The repository is visible only to those with explicit
      access.
    * `PUBLIC`: The repository is visible to everyone.
    """

    __schema__ = github_schema
    __choices__ = ("INTERNAL", "PRIVATE", "PUBLIC")


class RepoArchivedAuditEntryVisibility(sgqlc.types.Enum):
    """The privacy of a repository

    Enumeration Choices:

    * `INTERNAL`: The repository is visible only to users in the same
      business.
    * `PRIVATE`: The repository is visible only to those with explicit
      access.
    * `PUBLIC`: The repository is visible to everyone.
    """

    __schema__ = github_schema
    __choices__ = ("INTERNAL", "PRIVATE", "PUBLIC")


class RepoChangeMergeSettingAuditEntryMergeType(sgqlc.types.Enum):
    """The merge options available for pull requests to this repository.

    Enumeration Choices:

    * `MERGE`: The pull request is added to the base branch in a merge
      commit.
    * `REBASE`: Commits from the pull request are added onto the base
      branch individually without a merge commit.
    * `SQUASH`: The pull request's commits are squashed into a single
      commit before they are merged to the base branch.
    """

    __schema__ = github_schema
    __choices__ = ("MERGE", "REBASE", "SQUASH")


class RepoCreateAuditEntryVisibility(sgqlc.types.Enum):
    """The privacy of a repository

    Enumeration Choices:

    * `INTERNAL`: The repository is visible only to users in the same
      business.
    * `PRIVATE`: The repository is visible only to those with explicit
      access.
    * `PUBLIC`: The repository is visible to everyone.
    """

    __schema__ = github_schema
    __choices__ = ("INTERNAL", "PRIVATE", "PUBLIC")


class RepoDestroyAuditEntryVisibility(sgqlc.types.Enum):
    """The privacy of a repository

    Enumeration Choices:

    * `INTERNAL`: The repository is visible only to users in the same
      business.
    * `PRIVATE`: The repository is visible only to those with explicit
      access.
    * `PUBLIC`: The repository is visible to everyone.
    """

    __schema__ = github_schema
    __choices__ = ("INTERNAL", "PRIVATE", "PUBLIC")


class RepoRemoveMemberAuditEntryVisibility(sgqlc.types.Enum):
    """The privacy of a repository

    Enumeration Choices:

    * `INTERNAL`: The repository is visible only to users in the same
      business.
    * `PRIVATE`: The repository is visible only to those with explicit
      access.
    * `PUBLIC`: The repository is visible to everyone.
    """

    __schema__ = github_schema
    __choices__ = ("INTERNAL", "PRIVATE", "PUBLIC")


class ReportedContentClassifiers(sgqlc.types.Enum):
    """The reasons a piece of content can be reported or minimized.

    Enumeration Choices:

    * `ABUSE`: An abusive or harassing piece of content
    * `DUPLICATE`: A duplicated piece of content
    * `OFF_TOPIC`: An irrelevant piece of content
    * `OUTDATED`: An outdated piece of content
    * `RESOLVED`: The content has been resolved
    * `SPAM`: A spammy piece of content
    """

    __schema__ = github_schema
    __choices__ = ("ABUSE", "DUPLICATE", "OFF_TOPIC", "OUTDATED", "RESOLVED", "SPAM")


class RepositoryAffiliation(sgqlc.types.Enum):
    """The affiliation of a user to a repository

    Enumeration Choices:

    * `COLLABORATOR`: Repositories that the user has been added to as
      a collaborator.
    * `ORGANIZATION_MEMBER`: Repositories that the user has access to
      through being a member of an organization. This includes every
      repository on every team that the user is on.
    * `OWNER`: Repositories that are owned by the authenticated user.
    """

    __schema__ = github_schema
    __choices__ = ("COLLABORATOR", "ORGANIZATION_MEMBER", "OWNER")


class RepositoryContributionType(sgqlc.types.Enum):
    """The reason a repository is listed as 'contributed'.

    Enumeration Choices:

    * `COMMIT`: Created a commit
    * `ISSUE`: Created an issue
    * `PULL_REQUEST`: Created a pull request
    * `PULL_REQUEST_REVIEW`: Reviewed a pull request
    * `REPOSITORY`: Created the repository
    """

    __schema__ = github_schema
    __choices__ = ("COMMIT", "ISSUE", "PULL_REQUEST", "PULL_REQUEST_REVIEW", "REPOSITORY")


class RepositoryInteractionLimit(sgqlc.types.Enum):
    """A repository interaction limit.

    Enumeration Choices:

    * `COLLABORATORS_ONLY`: Users that are not collaborators will not
      be able to interact with the repository.
    * `CONTRIBUTORS_ONLY`: Users that have not previously committed to
      a repository’s default branch will be unable to interact with
      the repository.
    * `EXISTING_USERS`: Users that have recently created their account
      will be unable to interact with the repository.
    * `NO_LIMIT`: No interaction limits are enabled.
    """

    __schema__ = github_schema
    __choices__ = ("COLLABORATORS_ONLY", "CONTRIBUTORS_ONLY", "EXISTING_USERS", "NO_LIMIT")


class RepositoryInteractionLimitExpiry(sgqlc.types.Enum):
    """The length for a repository interaction limit to be enabled for.

    Enumeration Choices:

    * `ONE_DAY`: The interaction limit will expire after 1 day.
    * `ONE_MONTH`: The interaction limit will expire after 1 month.
    * `ONE_WEEK`: The interaction limit will expire after 1 week.
    * `SIX_MONTHS`: The interaction limit will expire after 6 months.
    * `THREE_DAYS`: The interaction limit will expire after 3 days.
    """

    __schema__ = github_schema
    __choices__ = ("ONE_DAY", "ONE_MONTH", "ONE_WEEK", "SIX_MONTHS", "THREE_DAYS")


class RepositoryInteractionLimitOrigin(sgqlc.types.Enum):
    """Indicates where an interaction limit is configured.

    Enumeration Choices:

    * `ORGANIZATION`: A limit that is configured at the organization
      level.
    * `REPOSITORY`: A limit that is configured at the repository
      level.
    * `USER`: A limit that is configured at the user-wide level.
    """

    __schema__ = github_schema
    __choices__ = ("ORGANIZATION", "REPOSITORY", "USER")


class RepositoryInvitationOrderField(sgqlc.types.Enum):
    """Properties by which repository invitation connections can be
    ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order repository invitations by creation time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class RepositoryLockReason(sgqlc.types.Enum):
    """The possible reasons a given repository could be in a locked
    state.

    Enumeration Choices:

    * `BILLING`: The repository is locked due to a billing related
      reason.
    * `MIGRATING`: The repository is locked due to a migration.
    * `MOVING`: The repository is locked due to a move.
    * `RENAME`: The repository is locked due to a rename.
    * `TRADE_RESTRICTION`: The repository is locked due to a trade
      controls related reason.
    """

    __schema__ = github_schema
    __choices__ = ("BILLING", "MIGRATING", "MOVING", "RENAME", "TRADE_RESTRICTION")


class RepositoryMigrationOrderDirection(sgqlc.types.Enum):
    """Possible directions in which to order a list of repository
    migrations when provided an `orderBy` argument.

    Enumeration Choices:

    * `ASC`: Specifies an ascending order for a given `orderBy`
      argument.
    * `DESC`: Specifies a descending order for a given `orderBy`
      argument.
    """

    __schema__ = github_schema
    __choices__ = ("ASC", "DESC")


class RepositoryMigrationOrderField(sgqlc.types.Enum):
    """Properties by which repository migrations can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order mannequins why when they were created.
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class RepositoryOrderField(sgqlc.types.Enum):
    """Properties by which repository connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order repositories by creation time
    * `NAME`: Order repositories by name
    * `PUSHED_AT`: Order repositories by push time
    * `STARGAZERS`: Order repositories by number of stargazers
    * `UPDATED_AT`: Order repositories by update time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "NAME", "PUSHED_AT", "STARGAZERS", "UPDATED_AT")


class RepositoryPermission(sgqlc.types.Enum):
    """The access level to a repository

    Enumeration Choices:

    * `ADMIN`: Can read, clone, and push to this repository. Can also
      manage issues, pull requests, and repository settings, including
      adding collaborators
    * `MAINTAIN`: Can read, clone, and push to this repository. They
      can also manage issues, pull requests, and some repository
      settings
    * `READ`: Can read and clone this repository. Can also open and
      comment on issues and pull requests
    * `TRIAGE`: Can read and clone this repository. Can also manage
      issues and pull requests
    * `WRITE`: Can read, clone, and push to this repository. Can also
      manage issues and pull requests
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "MAINTAIN", "READ", "TRIAGE", "WRITE")


class RepositoryPrivacy(sgqlc.types.Enum):
    """The privacy of a repository

    Enumeration Choices:

    * `PRIVATE`: Private
    * `PUBLIC`: Public
    """

    __schema__ = github_schema
    __choices__ = ("PRIVATE", "PUBLIC")


class RepositoryRuleType(sgqlc.types.Enum):
    """The rule types supported in rulesets

    Enumeration Choices:

    * `BRANCH_NAME_PATTERN`: Branch name pattern
    * `COMMITTER_EMAIL_PATTERN`: Committer email pattern
    * `COMMIT_AUTHOR_EMAIL_PATTERN`: Commit author email pattern
    * `COMMIT_MESSAGE_PATTERN`: Commit message pattern
    * `CREATION`: Only allow users with bypass permission to create
      matching refs.
    * `DELETION`: Only allow users with bypass permissions to delete
      matching refs.
    * `NON_FAST_FORWARD`: Prevent users with push access from force
      pushing to branches.
    * `PULL_REQUEST`: Require all commits be made to a non-target
      branch and submitted via a pull request before they can be
      merged.
    * `REQUIRED_DEPLOYMENTS`: Choose which environments must be
      successfully deployed to before branches can be merged into a
      branch that matches this rule.
    * `REQUIRED_LINEAR_HISTORY`: Prevent merge commits from being
      pushed to matching branches.
    * `REQUIRED_SIGNATURES`: Commits pushed to matching branches must
      have verified signatures.
    * `REQUIRED_STATUS_CHECKS`: Choose which status checks must pass
      before branches can be merged into a branch that matches this
      rule. When enabled, commits must first be pushed to another
      branch, then merged or pushed directly to a branch that matches
      this rule after status checks have passed.
    * `TAG_NAME_PATTERN`: Tag name pattern
    * `UPDATE`: Only allow users with bypass permission to update
      matching refs.
    """

    __schema__ = github_schema
    __choices__ = (
        "BRANCH_NAME_PATTERN",
        "COMMITTER_EMAIL_PATTERN",
        "COMMIT_AUTHOR_EMAIL_PATTERN",
        "COMMIT_MESSAGE_PATTERN",
        "CREATION",
        "DELETION",
        "NON_FAST_FORWARD",
        "PULL_REQUEST",
        "REQUIRED_DEPLOYMENTS",
        "REQUIRED_LINEAR_HISTORY",
        "REQUIRED_SIGNATURES",
        "REQUIRED_STATUS_CHECKS",
        "TAG_NAME_PATTERN",
        "UPDATE",
    )


class RepositoryRulesetTarget(sgqlc.types.Enum):
    """The targets supported for rulesets

    Enumeration Choices:

    * `BRANCH`: Branch
    * `TAG`: Tag
    """

    __schema__ = github_schema
    __choices__ = ("BRANCH", "TAG")


class RepositoryVisibility(sgqlc.types.Enum):
    """The repository's visibility level.

    Enumeration Choices:

    * `INTERNAL`: The repository is visible only to users in the same
      business.
    * `PRIVATE`: The repository is visible only to those with explicit
      access.
    * `PUBLIC`: The repository is visible to everyone.
    """

    __schema__ = github_schema
    __choices__ = ("INTERNAL", "PRIVATE", "PUBLIC")


class RepositoryVulnerabilityAlertDependencyScope(sgqlc.types.Enum):
    """The possible scopes of an alert's dependency.

    Enumeration Choices:

    * `DEVELOPMENT`: A dependency that is only used in development
    * `RUNTIME`: A dependency that is leveraged during application
      runtime
    """

    __schema__ = github_schema
    __choices__ = ("DEVELOPMENT", "RUNTIME")


class RepositoryVulnerabilityAlertState(sgqlc.types.Enum):
    """The possible states of an alert

    Enumeration Choices:

    * `AUTO_DISMISSED`: An alert that has been automatically closed by
      Dependabot.
    * `DISMISSED`: An alert that has been manually closed by a user.
    * `FIXED`: An alert that has been resolved by a code change.
    * `OPEN`: An alert that is still open.
    """

    __schema__ = github_schema
    __choices__ = ("AUTO_DISMISSED", "DISMISSED", "FIXED", "OPEN")


class RequestableCheckStatusState(sgqlc.types.Enum):
    """The possible states that can be requested when creating a check
    run.

    Enumeration Choices:

    * `COMPLETED`: The check suite or run has been completed.
    * `IN_PROGRESS`: The check suite or run is in progress.
    * `PENDING`: The check suite or run is in pending state.
    * `QUEUED`: The check suite or run has been queued.
    * `WAITING`: The check suite or run is in waiting state.
    """

    __schema__ = github_schema
    __choices__ = ("COMPLETED", "IN_PROGRESS", "PENDING", "QUEUED", "WAITING")


class RoleInOrganization(sgqlc.types.Enum):
    """Possible roles a user may have in relation to an organization.

    Enumeration Choices:

    * `DIRECT_MEMBER`: A user who is a direct member of the
      organization.
    * `OWNER`: A user with full administrative access to the
      organization.
    * `UNAFFILIATED`: A user who is unaffiliated with the
      organization.
    """

    __schema__ = github_schema
    __choices__ = ("DIRECT_MEMBER", "OWNER", "UNAFFILIATED")


class RuleBypassMode(sgqlc.types.Enum):
    """The bypass mode for a rule or ruleset.

    Enumeration Choices:

    * `NONE`: Bypassing is disabled
    * `ORGANIZATION`: Those with bypass permission at the organization
      level can bypass
    * `ORGANIZATION_ALWAYS`: Those with bypass permission at the
      organization level can always bypass
    * `ORGANIZATION_NONE`: Bypassing is disabled
    * `ORGANIZATION_PRS_ONLY`: Those with bypass permission at the
      organization level can bypass for pull requests only
    * `REPOSITORY`: Those with bypass permission at the repository
      level can bypass
    """

    __schema__ = github_schema
    __choices__ = ("NONE", "ORGANIZATION", "ORGANIZATION_ALWAYS", "ORGANIZATION_NONE", "ORGANIZATION_PRS_ONLY", "REPOSITORY")


class RuleEnforcement(sgqlc.types.Enum):
    """The level of enforcement for a rule or ruleset.

    Enumeration Choices:

    * `ACTIVE`: Rules will be enforced
    * `DISABLED`: Do not evaluate or enforce rules
    * `EVALUATE`: Allow admins to test rules before enforcing them.
      Admins can view insights on the Rule Insights page (`evaluate`
      is only available with GitHub Enterprise).
    """

    __schema__ = github_schema
    __choices__ = ("ACTIVE", "DISABLED", "EVALUATE")


class SamlDigestAlgorithm(sgqlc.types.Enum):
    """The possible digest algorithms used to sign SAML requests for an
    identity provider.

    Enumeration Choices:

    * `SHA1`: SHA1
    * `SHA256`: SHA256
    * `SHA384`: SHA384
    * `SHA512`: SHA512
    """

    __schema__ = github_schema
    __choices__ = ("SHA1", "SHA256", "SHA384", "SHA512")


class SamlSignatureAlgorithm(sgqlc.types.Enum):
    """The possible signature algorithms used to sign SAML requests for a
    Identity Provider.

    Enumeration Choices:

    * `RSA_SHA1`: RSA-SHA1
    * `RSA_SHA256`: RSA-SHA256
    * `RSA_SHA384`: RSA-SHA384
    * `RSA_SHA512`: RSA-SHA512
    """

    __schema__ = github_schema
    __choices__ = ("RSA_SHA1", "RSA_SHA256", "RSA_SHA384", "RSA_SHA512")


class SavedReplyOrderField(sgqlc.types.Enum):
    """Properties by which saved reply connections can be ordered.

    Enumeration Choices:

    * `UPDATED_AT`: Order saved reply by when they were updated.
    """

    __schema__ = github_schema
    __choices__ = ("UPDATED_AT",)


class SearchType(sgqlc.types.Enum):
    """Represents the individual results of a search.

    Enumeration Choices:

    * `DISCUSSION`: Returns matching discussions in repositories.
    * `ISSUE`: Returns results matching issues in repositories.
    * `REPOSITORY`: Returns results matching repositories.
    * `USER`: Returns results matching users and organizations on
      GitHub.
    """

    __schema__ = github_schema
    __choices__ = ("DISCUSSION", "ISSUE", "REPOSITORY", "USER")


class SecurityAdvisoryClassification(sgqlc.types.Enum):
    """Classification of the advisory.

    Enumeration Choices:

    * `GENERAL`: Classification of general advisories.
    * `MALWARE`: Classification of malware advisories.
    """

    __schema__ = github_schema
    __choices__ = ("GENERAL", "MALWARE")


class SecurityAdvisoryEcosystem(sgqlc.types.Enum):
    """The possible ecosystems of a security vulnerability's package.

    Enumeration Choices:

    * `ACTIONS`: GitHub Actions
    * `COMPOSER`: PHP packages hosted at packagist.org
    * `ERLANG`: Erlang/Elixir packages hosted at hex.pm
    * `GO`: Go modules
    * `MAVEN`: Java artifacts hosted at the Maven central repository
    * `NPM`: JavaScript packages hosted at npmjs.com
    * `NUGET`: .NET packages hosted at the NuGet Gallery
    * `PIP`: Python packages hosted at PyPI.org
    * `PUB`: Dart packages hosted at pub.dev
    * `RUBYGEMS`: Ruby gems hosted at RubyGems.org
    * `RUST`: Rust crates
    * `SWIFT`: Swift packages
    """

    __schema__ = github_schema
    __choices__ = ("ACTIONS", "COMPOSER", "ERLANG", "GO", "MAVEN", "NPM", "NUGET", "PIP", "PUB", "RUBYGEMS", "RUST", "SWIFT")


class SecurityAdvisoryIdentifierType(sgqlc.types.Enum):
    """Identifier formats available for advisories.

    Enumeration Choices:

    * `CVE`: Common Vulnerabilities and Exposures Identifier.
    * `GHSA`: GitHub Security Advisory ID.
    """

    __schema__ = github_schema
    __choices__ = ("CVE", "GHSA")


class SecurityAdvisoryOrderField(sgqlc.types.Enum):
    """Properties by which security advisory connections can be ordered.

    Enumeration Choices:

    * `PUBLISHED_AT`: Order advisories by publication time
    * `UPDATED_AT`: Order advisories by update time
    """

    __schema__ = github_schema
    __choices__ = ("PUBLISHED_AT", "UPDATED_AT")


class SecurityAdvisorySeverity(sgqlc.types.Enum):
    """Severity of the vulnerability.

    Enumeration Choices:

    * `CRITICAL`: Critical.
    * `HIGH`: High.
    * `LOW`: Low.
    * `MODERATE`: Moderate.
    """

    __schema__ = github_schema
    __choices__ = ("CRITICAL", "HIGH", "LOW", "MODERATE")


class SecurityVulnerabilityOrderField(sgqlc.types.Enum):
    """Properties by which security vulnerability connections can be
    ordered.

    Enumeration Choices:

    * `UPDATED_AT`: Order vulnerability by update time
    """

    __schema__ = github_schema
    __choices__ = ("UPDATED_AT",)


class SocialAccountProvider(sgqlc.types.Enum):
    """Software or company that hosts social media accounts.

    Enumeration Choices:

    * `FACEBOOK`: Social media and networking website.
    * `GENERIC`: Catch-all for social media providers that do not yet
      have specific handling.
    * `HOMETOWN`: Fork of Mastodon with a greater focus on local
      posting.
    * `INSTAGRAM`: Social media website with a focus on photo and
      video sharing.
    * `LINKEDIN`: Professional networking website.
    * `MASTODON`: Open-source federated microblogging service.
    * `REDDIT`: Social news aggregation and discussion website.
    * `TWITCH`: Live-streaming service.
    * `TWITTER`: Microblogging website.
    * `YOUTUBE`: Online video platform.
    """

    __schema__ = github_schema
    __choices__ = ("FACEBOOK", "GENERIC", "HOMETOWN", "INSTAGRAM", "LINKEDIN", "MASTODON", "REDDIT", "TWITCH", "TWITTER", "YOUTUBE")


class SponsorOrderField(sgqlc.types.Enum):
    """Properties by which sponsor connections can be ordered.

    Enumeration Choices:

    * `LOGIN`: Order sponsorable entities by login (username).
    * `RELEVANCE`: Order sponsors by their relevance to the viewer.
    """

    __schema__ = github_schema
    __choices__ = ("LOGIN", "RELEVANCE")


class SponsorableOrderField(sgqlc.types.Enum):
    """Properties by which sponsorable connections can be ordered.

    Enumeration Choices:

    * `LOGIN`: Order sponsorable entities by login (username).
    """

    __schema__ = github_schema
    __choices__ = ("LOGIN",)


class SponsorsActivityAction(sgqlc.types.Enum):
    """The possible actions that GitHub Sponsors activities can
    represent.

    Enumeration Choices:

    * `CANCELLED_SPONSORSHIP`: The activity was cancelling a
      sponsorship.
    * `NEW_SPONSORSHIP`: The activity was starting a sponsorship.
    * `PENDING_CHANGE`: The activity was scheduling a downgrade or
      cancellation.
    * `REFUND`: The activity was funds being refunded to the sponsor
      or GitHub.
    * `SPONSOR_MATCH_DISABLED`: The activity was disabling matching
      for a previously matched sponsorship.
    * `TIER_CHANGE`: The activity was changing the sponsorship tier,
      either directly by the sponsor or by a scheduled/pending change.
    """

    __schema__ = github_schema
    __choices__ = ("CANCELLED_SPONSORSHIP", "NEW_SPONSORSHIP", "PENDING_CHANGE", "REFUND", "SPONSOR_MATCH_DISABLED", "TIER_CHANGE")


class SponsorsActivityOrderField(sgqlc.types.Enum):
    """Properties by which GitHub Sponsors activity connections can be
    ordered.

    Enumeration Choices:

    * `TIMESTAMP`: Order activities by when they happened.
    """

    __schema__ = github_schema
    __choices__ = ("TIMESTAMP",)


class SponsorsActivityPeriod(sgqlc.types.Enum):
    """The possible time periods for which Sponsors activities can be
    requested.

    Enumeration Choices:

    * `ALL`: Don't restrict the activity to any date range, include
      all activity.
    * `DAY`: The previous calendar day.
    * `MONTH`: The previous thirty days.
    * `WEEK`: The previous seven days.
    """

    __schema__ = github_schema
    __choices__ = ("ALL", "DAY", "MONTH", "WEEK")


class SponsorsCountryOrRegionCode(sgqlc.types.Enum):
    """Represents countries or regions for billing and residence for a
    GitHub Sponsors profile.

    Enumeration Choices:

    * `AD`: Andorra
    * `AE`: United Arab Emirates
    * `AF`: Afghanistan
    * `AG`: Antigua and Barbuda
    * `AI`: Anguilla
    * `AL`: Albania
    * `AM`: Armenia
    * `AO`: Angola
    * `AQ`: Antarctica
    * `AR`: Argentina
    * `AS`: American Samoa
    * `AT`: Austria
    * `AU`: Australia
    * `AW`: Aruba
    * `AX`: Åland
    * `AZ`: Azerbaijan
    * `BA`: Bosnia and Herzegovina
    * `BB`: Barbados
    * `BD`: Bangladesh
    * `BE`: Belgium
    * `BF`: Burkina Faso
    * `BG`: Bulgaria
    * `BH`: Bahrain
    * `BI`: Burundi
    * `BJ`: Benin
    * `BL`: Saint Barthélemy
    * `BM`: Bermuda
    * `BN`: Brunei Darussalam
    * `BO`: Bolivia
    * `BQ`: Bonaire, Sint Eustatius and Saba
    * `BR`: Brazil
    * `BS`: Bahamas
    * `BT`: Bhutan
    * `BV`: Bouvet Island
    * `BW`: Botswana
    * `BY`: Belarus
    * `BZ`: Belize
    * `CA`: Canada
    * `CC`: Cocos (Keeling) Islands
    * `CD`: Congo (Kinshasa)
    * `CF`: Central African Republic
    * `CG`: Congo (Brazzaville)
    * `CH`: Switzerland
    * `CI`: Côte d'Ivoire
    * `CK`: Cook Islands
    * `CL`: Chile
    * `CM`: Cameroon
    * `CN`: China
    * `CO`: Colombia
    * `CR`: Costa Rica
    * `CV`: Cape Verde
    * `CW`: Curaçao
    * `CX`: Christmas Island
    * `CY`: Cyprus
    * `CZ`: Czech Republic
    * `DE`: Germany
    * `DJ`: Djibouti
    * `DK`: Denmark
    * `DM`: Dominica
    * `DO`: Dominican Republic
    * `DZ`: Algeria
    * `EC`: Ecuador
    * `EE`: Estonia
    * `EG`: Egypt
    * `EH`: Western Sahara
    * `ER`: Eritrea
    * `ES`: Spain
    * `ET`: Ethiopia
    * `FI`: Finland
    * `FJ`: Fiji
    * `FK`: Falkland Islands
    * `FM`: Micronesia
    * `FO`: Faroe Islands
    * `FR`: France
    * `GA`: Gabon
    * `GB`: United Kingdom
    * `GD`: Grenada
    * `GE`: Georgia
    * `GF`: French Guiana
    * `GG`: Guernsey
    * `GH`: Ghana
    * `GI`: Gibraltar
    * `GL`: Greenland
    * `GM`: Gambia
    * `GN`: Guinea
    * `GP`: Guadeloupe
    * `GQ`: Equatorial Guinea
    * `GR`: Greece
    * `GS`: South Georgia and South Sandwich Islands
    * `GT`: Guatemala
    * `GU`: Guam
    * `GW`: Guinea-Bissau
    * `GY`: Guyana
    * `HK`: Hong Kong
    * `HM`: Heard and McDonald Islands
    * `HN`: Honduras
    * `HR`: Croatia
    * `HT`: Haiti
    * `HU`: Hungary
    * `ID`: Indonesia
    * `IE`: Ireland
    * `IL`: Israel
    * `IM`: Isle of Man
    * `IN`: India
    * `IO`: British Indian Ocean Territory
    * `IQ`: Iraq
    * `IR`: Iran
    * `IS`: Iceland
    * `IT`: Italy
    * `JE`: Jersey
    * `JM`: Jamaica
    * `JO`: Jordan
    * `JP`: Japan
    * `KE`: Kenya
    * `KG`: Kyrgyzstan
    * `KH`: Cambodia
    * `KI`: Kiribati
    * `KM`: Comoros
    * `KN`: Saint Kitts and Nevis
    * `KR`: Korea, South
    * `KW`: Kuwait
    * `KY`: Cayman Islands
    * `KZ`: Kazakhstan
    * `LA`: Laos
    * `LB`: Lebanon
    * `LC`: Saint Lucia
    * `LI`: Liechtenstein
    * `LK`: Sri Lanka
    * `LR`: Liberia
    * `LS`: Lesotho
    * `LT`: Lithuania
    * `LU`: Luxembourg
    * `LV`: Latvia
    * `LY`: Libya
    * `MA`: Morocco
    * `MC`: Monaco
    * `MD`: Moldova
    * `ME`: Montenegro
    * `MF`: Saint Martin (French part)
    * `MG`: Madagascar
    * `MH`: Marshall Islands
    * `MK`: Macedonia
    * `ML`: Mali
    * `MM`: Myanmar
    * `MN`: Mongolia
    * `MO`: Macau
    * `MP`: Northern Mariana Islands
    * `MQ`: Martinique
    * `MR`: Mauritania
    * `MS`: Montserrat
    * `MT`: Malta
    * `MU`: Mauritius
    * `MV`: Maldives
    * `MW`: Malawi
    * `MX`: Mexico
    * `MY`: Malaysia
    * `MZ`: Mozambique
    * `NA`: Namibia
    * `NC`: New Caledonia
    * `NE`: Niger
    * `NF`: Norfolk Island
    * `NG`: Nigeria
    * `NI`: Nicaragua
    * `NL`: Netherlands
    * `NO`: Norway
    * `NP`: Nepal
    * `NR`: Nauru
    * `NU`: Niue
    * `NZ`: New Zealand
    * `OM`: Oman
    * `PA`: Panama
    * `PE`: Peru
    * `PF`: French Polynesia
    * `PG`: Papua New Guinea
    * `PH`: Philippines
    * `PK`: Pakistan
    * `PL`: Poland
    * `PM`: Saint Pierre and Miquelon
    * `PN`: Pitcairn
    * `PR`: Puerto Rico
    * `PS`: Palestine
    * `PT`: Portugal
    * `PW`: Palau
    * `PY`: Paraguay
    * `QA`: Qatar
    * `RE`: Reunion
    * `RO`: Romania
    * `RS`: Serbia
    * `RU`: Russian Federation
    * `RW`: Rwanda
    * `SA`: Saudi Arabia
    * `SB`: Solomon Islands
    * `SC`: Seychelles
    * `SD`: Sudan
    * `SE`: Sweden
    * `SG`: Singapore
    * `SH`: Saint Helena
    * `SI`: Slovenia
    * `SJ`: Svalbard and Jan Mayen Islands
    * `SK`: Slovakia
    * `SL`: Sierra Leone
    * `SM`: San Marino
    * `SN`: Senegal
    * `SO`: Somalia
    * `SR`: Suriname
    * `SS`: South Sudan
    * `ST`: Sao Tome and Principe
    * `SV`: El Salvador
    * `SX`: Sint Maarten (Dutch part)
    * `SZ`: Swaziland
    * `TC`: Turks and Caicos Islands
    * `TD`: Chad
    * `TF`: French Southern Lands
    * `TG`: Togo
    * `TH`: Thailand
    * `TJ`: Tajikistan
    * `TK`: Tokelau
    * `TL`: Timor-Leste
    * `TM`: Turkmenistan
    * `TN`: Tunisia
    * `TO`: Tonga
    * `TR`: Turkey
    * `TT`: Trinidad and Tobago
    * `TV`: Tuvalu
    * `TW`: Taiwan
    * `TZ`: Tanzania
    * `UA`: Ukraine
    * `UG`: Uganda
    * `UM`: United States Minor Outlying Islands
    * `US`: United States of America
    * `UY`: Uruguay
    * `UZ`: Uzbekistan
    * `VA`: Vatican City
    * `VC`: Saint Vincent and the Grenadines
    * `VE`: Venezuela
    * `VG`: Virgin Islands, British
    * `VI`: Virgin Islands, U.S.
    * `VN`: Vietnam
    * `VU`: Vanuatu
    * `WF`: Wallis and Futuna Islands
    * `WS`: Samoa
    * `YE`: Yemen
    * `YT`: Mayotte
    * `ZA`: South Africa
    * `ZM`: Zambia
    * `ZW`: Zimbabwe
    """

    __schema__ = github_schema
    __choices__ = (
        "AD",
        "AE",
        "AF",
        "AG",
        "AI",
        "AL",
        "AM",
        "AO",
        "AQ",
        "AR",
        "AS",
        "AT",
        "AU",
        "AW",
        "AX",
        "AZ",
        "BA",
        "BB",
        "BD",
        "BE",
        "BF",
        "BG",
        "BH",
        "BI",
        "BJ",
        "BL",
        "BM",
        "BN",
        "BO",
        "BQ",
        "BR",
        "BS",
        "BT",
        "BV",
        "BW",
        "BY",
        "BZ",
        "CA",
        "CC",
        "CD",
        "CF",
        "CG",
        "CH",
        "CI",
        "CK",
        "CL",
        "CM",
        "CN",
        "CO",
        "CR",
        "CV",
        "CW",
        "CX",
        "CY",
        "CZ",
        "DE",
        "DJ",
        "DK",
        "DM",
        "DO",
        "DZ",
        "EC",
        "EE",
        "EG",
        "EH",
        "ER",
        "ES",
        "ET",
        "FI",
        "FJ",
        "FK",
        "FM",
        "FO",
        "FR",
        "GA",
        "GB",
        "GD",
        "GE",
        "GF",
        "GG",
        "GH",
        "GI",
        "GL",
        "GM",
        "GN",
        "GP",
        "GQ",
        "GR",
        "GS",
        "GT",
        "GU",
        "GW",
        "GY",
        "HK",
        "HM",
        "HN",
        "HR",
        "HT",
        "HU",
        "ID",
        "IE",
        "IL",
        "IM",
        "IN",
        "IO",
        "IQ",
        "IR",
        "IS",
        "IT",
        "JE",
        "JM",
        "JO",
        "JP",
        "KE",
        "KG",
        "KH",
        "KI",
        "KM",
        "KN",
        "KR",
        "KW",
        "KY",
        "KZ",
        "LA",
        "LB",
        "LC",
        "LI",
        "LK",
        "LR",
        "LS",
        "LT",
        "LU",
        "LV",
        "LY",
        "MA",
        "MC",
        "MD",
        "ME",
        "MF",
        "MG",
        "MH",
        "MK",
        "ML",
        "MM",
        "MN",
        "MO",
        "MP",
        "MQ",
        "MR",
        "MS",
        "MT",
        "MU",
        "MV",
        "MW",
        "MX",
        "MY",
        "MZ",
        "NA",
        "NC",
        "NE",
        "NF",
        "NG",
        "NI",
        "NL",
        "NO",
        "NP",
        "NR",
        "NU",
        "NZ",
        "OM",
        "PA",
        "PE",
        "PF",
        "PG",
        "PH",
        "PK",
        "PL",
        "PM",
        "PN",
        "PR",
        "PS",
        "PT",
        "PW",
        "PY",
        "QA",
        "RE",
        "RO",
        "RS",
        "RU",
        "RW",
        "SA",
        "SB",
        "SC",
        "SD",
        "SE",
        "SG",
        "SH",
        "SI",
        "SJ",
        "SK",
        "SL",
        "SM",
        "SN",
        "SO",
        "SR",
        "SS",
        "ST",
        "SV",
        "SX",
        "SZ",
        "TC",
        "TD",
        "TF",
        "TG",
        "TH",
        "TJ",
        "TK",
        "TL",
        "TM",
        "TN",
        "TO",
        "TR",
        "TT",
        "TV",
        "TW",
        "TZ",
        "UA",
        "UG",
        "UM",
        "US",
        "UY",
        "UZ",
        "VA",
        "VC",
        "VE",
        "VG",
        "VI",
        "VN",
        "VU",
        "WF",
        "WS",
        "YE",
        "YT",
        "ZA",
        "ZM",
        "ZW",
    )


class SponsorsGoalKind(sgqlc.types.Enum):
    """The different kinds of goals a GitHub Sponsors member can have.

    Enumeration Choices:

    * `MONTHLY_SPONSORSHIP_AMOUNT`: The goal is about getting a
      certain amount in USD from sponsorships each month.
    * `TOTAL_SPONSORS_COUNT`: The goal is about reaching a certain
      number of sponsors.
    """

    __schema__ = github_schema
    __choices__ = ("MONTHLY_SPONSORSHIP_AMOUNT", "TOTAL_SPONSORS_COUNT")


class SponsorsListingFeaturedItemFeatureableType(sgqlc.types.Enum):
    """The different kinds of records that can be featured on a GitHub
    Sponsors profile page.

    Enumeration Choices:

    * `REPOSITORY`: A repository owned by the user or organization
      with the GitHub Sponsors profile.
    * `USER`: A user who belongs to the organization with the GitHub
      Sponsors profile.
    """

    __schema__ = github_schema
    __choices__ = ("REPOSITORY", "USER")


class SponsorsTierOrderField(sgqlc.types.Enum):
    """Properties by which Sponsors tiers connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order tiers by creation time.
    * `MONTHLY_PRICE_IN_CENTS`: Order tiers by their monthly price in
      cents
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "MONTHLY_PRICE_IN_CENTS")


class SponsorshipNewsletterOrderField(sgqlc.types.Enum):
    """Properties by which sponsorship update connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order sponsorship newsletters by when they were
      created.
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class SponsorshipOrderField(sgqlc.types.Enum):
    """Properties by which sponsorship connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order sponsorship by creation time.
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class SponsorshipPrivacy(sgqlc.types.Enum):
    """The privacy of a sponsorship

    Enumeration Choices:

    * `PRIVATE`: Private
    * `PUBLIC`: Public
    """

    __schema__ = github_schema
    __choices__ = ("PRIVATE", "PUBLIC")


class SquashMergeCommitMessage(sgqlc.types.Enum):
    """The possible default commit messages for squash merges.

    Enumeration Choices:

    * `BLANK`: Default to a blank commit message.
    * `COMMIT_MESSAGES`: Default to the branch's commit messages.
    * `PR_BODY`: Default to the pull request's body.
    """

    __schema__ = github_schema
    __choices__ = ("BLANK", "COMMIT_MESSAGES", "PR_BODY")


class SquashMergeCommitTitle(sgqlc.types.Enum):
    """The possible default commit titles for squash merges.

    Enumeration Choices:

    * `COMMIT_OR_PR_TITLE`: Default to the commit's title (if only one
      commit) or the pull request's title (when more than one commit).
    * `PR_TITLE`: Default to the pull request's title.
    """

    __schema__ = github_schema
    __choices__ = ("COMMIT_OR_PR_TITLE", "PR_TITLE")


class StarOrderField(sgqlc.types.Enum):
    """Properties by which star connections can be ordered.

    Enumeration Choices:

    * `STARRED_AT`: Allows ordering a list of stars by when they were
      created.
    """

    __schema__ = github_schema
    __choices__ = ("STARRED_AT",)


class StatusState(sgqlc.types.Enum):
    """The possible commit status states.

    Enumeration Choices:

    * `ERROR`: Status is errored.
    * `EXPECTED`: Status is expected.
    * `FAILURE`: Status is failing.
    * `PENDING`: Status is pending.
    * `SUCCESS`: Status is successful.
    """

    __schema__ = github_schema
    __choices__ = ("ERROR", "EXPECTED", "FAILURE", "PENDING", "SUCCESS")


String = sgqlc.types.String


class SubscriptionState(sgqlc.types.Enum):
    """The possible states of a subscription.

    Enumeration Choices:

    * `IGNORED`: The User is never notified.
    * `SUBSCRIBED`: The User is notified of all conversations.
    * `UNSUBSCRIBED`: The User is only notified when participating or
      @mentioned.
    """

    __schema__ = github_schema
    __choices__ = ("IGNORED", "SUBSCRIBED", "UNSUBSCRIBED")


class TeamDiscussionCommentOrderField(sgqlc.types.Enum):
    """Properties by which team discussion comment connections can be
    ordered.

    Enumeration Choices:

    * `NUMBER`: Allows sequential ordering of team discussion comments
      (which is equivalent to chronological ordering).
    """

    __schema__ = github_schema
    __choices__ = ("NUMBER",)


class TeamDiscussionOrderField(sgqlc.types.Enum):
    """Properties by which team discussion connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Allows chronological ordering of team discussions.
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class TeamMemberOrderField(sgqlc.types.Enum):
    """Properties by which team member connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order team members by creation time
    * `LOGIN`: Order team members by login
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "LOGIN")


class TeamMemberRole(sgqlc.types.Enum):
    """The possible team member roles; either 'maintainer' or 'member'.

    Enumeration Choices:

    * `MAINTAINER`: A team maintainer has permission to add and remove
      team members.
    * `MEMBER`: A team member has no administrative permissions on the
      team.
    """

    __schema__ = github_schema
    __choices__ = ("MAINTAINER", "MEMBER")


class TeamMembershipType(sgqlc.types.Enum):
    """Defines which types of team members are included in the returned
    list. Can be one of IMMEDIATE, CHILD_TEAM or ALL.

    Enumeration Choices:

    * `ALL`: Includes immediate and child team members for the team.
    * `CHILD_TEAM`: Includes only child team members for the team.
    * `IMMEDIATE`: Includes only immediate members of the team.
    """

    __schema__ = github_schema
    __choices__ = ("ALL", "CHILD_TEAM", "IMMEDIATE")


class TeamNotificationSetting(sgqlc.types.Enum):
    """The possible team notification values.

    Enumeration Choices:

    * `NOTIFICATIONS_DISABLED`: No one will receive notifications.
    * `NOTIFICATIONS_ENABLED`: Everyone will receive notifications
      when the team is @mentioned.
    """

    __schema__ = github_schema
    __choices__ = ("NOTIFICATIONS_DISABLED", "NOTIFICATIONS_ENABLED")


class TeamOrderField(sgqlc.types.Enum):
    """Properties by which team connections can be ordered.

    Enumeration Choices:

    * `NAME`: Allows ordering a list of teams by name.
    """

    __schema__ = github_schema
    __choices__ = ("NAME",)


class TeamPrivacy(sgqlc.types.Enum):
    """The possible team privacy values.

    Enumeration Choices:

    * `SECRET`: A secret team can only be seen by its members.
    * `VISIBLE`: A visible team can be seen and @mentioned by every
      member of the organization.
    """

    __schema__ = github_schema
    __choices__ = ("SECRET", "VISIBLE")


class TeamRepositoryOrderField(sgqlc.types.Enum):
    """Properties by which team repository connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order repositories by creation time
    * `NAME`: Order repositories by name
    * `PERMISSION`: Order repositories by permission
    * `PUSHED_AT`: Order repositories by push time
    * `STARGAZERS`: Order repositories by number of stargazers
    * `UPDATED_AT`: Order repositories by update time
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "NAME", "PERMISSION", "PUSHED_AT", "STARGAZERS", "UPDATED_AT")


class TeamRole(sgqlc.types.Enum):
    """The role of a user on a team.

    Enumeration Choices:

    * `ADMIN`: User has admin rights on the team.
    * `MEMBER`: User is a member of the team.
    """

    __schema__ = github_schema
    __choices__ = ("ADMIN", "MEMBER")


class TopicSuggestionDeclineReason(sgqlc.types.Enum):
    """Reason that the suggested topic is declined.

    Enumeration Choices:

    * `NOT_RELEVANT`: The suggested topic is not relevant to the
      repository.
    * `PERSONAL_PREFERENCE`: The viewer does not like the suggested
      topic.
    * `TOO_GENERAL`: The suggested topic is too general for the
      repository.
    * `TOO_SPECIFIC`: The suggested topic is too specific for the
      repository (e.g. #ruby-on-rails-version-4-2-1).
    """

    __schema__ = github_schema
    __choices__ = ("NOT_RELEVANT", "PERSONAL_PREFERENCE", "TOO_GENERAL", "TOO_SPECIFIC")


class TrackedIssueStates(sgqlc.types.Enum):
    """The possible states of a tracked issue.

    Enumeration Choices:

    * `CLOSED`: The tracked issue is closed
    * `OPEN`: The tracked issue is open
    """

    __schema__ = github_schema
    __choices__ = ("CLOSED", "OPEN")


class URI(sgqlc.types.Scalar):
    """An RFC 3986, RFC 3987, and RFC 6570 (level 4) compliant URI
    string.
    """

    __schema__ = github_schema


class UserBlockDuration(sgqlc.types.Enum):
    """The possible durations that a user can be blocked for.

    Enumeration Choices:

    * `ONE_DAY`: The user was blocked for 1 day
    * `ONE_MONTH`: The user was blocked for 30 days
    * `ONE_WEEK`: The user was blocked for 7 days
    * `PERMANENT`: The user was blocked permanently
    * `THREE_DAYS`: The user was blocked for 3 days
    """

    __schema__ = github_schema
    __choices__ = ("ONE_DAY", "ONE_MONTH", "ONE_WEEK", "PERMANENT", "THREE_DAYS")


class UserStatusOrderField(sgqlc.types.Enum):
    """Properties by which user status connections can be ordered.

    Enumeration Choices:

    * `UPDATED_AT`: Order user statuses by when they were updated.
    """

    __schema__ = github_schema
    __choices__ = ("UPDATED_AT",)


class VerifiableDomainOrderField(sgqlc.types.Enum):
    """Properties by which verifiable domain connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order verifiable domains by their creation date.
    * `DOMAIN`: Order verifiable domains by the domain name.
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT", "DOMAIN")


class WorkflowRunOrderField(sgqlc.types.Enum):
    """Properties by which workflow run connections can be ordered.

    Enumeration Choices:

    * `CREATED_AT`: Order workflow runs by most recently created
    """

    __schema__ = github_schema
    __choices__ = ("CREATED_AT",)


class WorkflowState(sgqlc.types.Enum):
    """The possible states for a workflow.

    Enumeration Choices:

    * `ACTIVE`: The workflow is active.
    * `DELETED`: The workflow was deleted from the git repository.
    * `DISABLED_FORK`: The workflow was disabled by default on a fork.
    * `DISABLED_INACTIVITY`: The workflow was disabled for inactivity
      in the repository.
    * `DISABLED_MANUALLY`: The workflow was disabled manually.
    """

    __schema__ = github_schema
    __choices__ = ("ACTIVE", "DELETED", "DISABLED_FORK", "DISABLED_INACTIVITY", "DISABLED_MANUALLY")


class X509Certificate(sgqlc.types.Scalar):
    """A valid x509 certificate string"""

    __schema__ = github_schema


########################################################################
# Input Objects
########################################################################
class AbortQueuedMigrationsInput(sgqlc.types.Input):
    """Autogenerated input type of AbortQueuedMigrations"""

    __schema__ = github_schema
    __field_names__ = ("owner_id", "client_mutation_id")
    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The ID of the organization that is running the migrations."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AcceptEnterpriseAdministratorInvitationInput(sgqlc.types.Input):
    """Autogenerated input type of
    AcceptEnterpriseAdministratorInvitation
    """

    __schema__ = github_schema
    __field_names__ = ("invitation_id", "client_mutation_id")
    invitation_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="invitationId")
    """The id of the invitation being accepted"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AcceptTopicSuggestionInput(sgqlc.types.Input):
    """Autogenerated input type of AcceptTopicSuggestion"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "name", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the repository."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the suggested topic."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddAssigneesToAssignableInput(sgqlc.types.Input):
    """Autogenerated input type of AddAssigneesToAssignable"""

    __schema__ = github_schema
    __field_names__ = ("assignable_id", "assignee_ids", "client_mutation_id")
    assignable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="assignableId")
    """The id of the assignable object to add assignees to."""

    assignee_ids = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ID))), graphql_name="assigneeIds")
    """The id of users to add as assignees."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddCommentInput(sgqlc.types.Input):
    """Autogenerated input type of AddComment"""

    __schema__ = github_schema
    __field_names__ = ("subject_id", "body", "client_mutation_id")
    subject_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="subjectId")
    """The Node ID of the subject to modify."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The contents of the comment."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddDiscussionCommentInput(sgqlc.types.Input):
    """Autogenerated input type of AddDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("discussion_id", "reply_to_id", "body", "client_mutation_id")
    discussion_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="discussionId")
    """The Node ID of the discussion to comment on."""

    reply_to_id = sgqlc.types.Field(ID, graphql_name="replyToId")
    """The Node ID of the discussion comment within this discussion to
    reply to.
    """

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The contents of the comment."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddDiscussionPollVoteInput(sgqlc.types.Input):
    """Autogenerated input type of AddDiscussionPollVote"""

    __schema__ = github_schema
    __field_names__ = ("poll_option_id", "client_mutation_id")
    poll_option_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pollOptionId")
    """The Node ID of the discussion poll option to vote for."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddEnterpriseOrganizationMemberInput(sgqlc.types.Input):
    """Autogenerated input type of AddEnterpriseOrganizationMember"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "organization_id", "user_ids", "role", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise which owns the organization."""

    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization the users will be added to."""

    user_ids = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ID))), graphql_name="userIds")
    """The IDs of the enterprise members to add."""

    role = sgqlc.types.Field(OrganizationMemberRole, graphql_name="role")
    """The role to assign the users in the organization"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddEnterpriseSupportEntitlementInput(sgqlc.types.Input):
    """Autogenerated input type of AddEnterpriseSupportEntitlement"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "login", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the Enterprise which the admin belongs to."""

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The login of a member who will receive the support entitlement."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddLabelsToLabelableInput(sgqlc.types.Input):
    """Autogenerated input type of AddLabelsToLabelable"""

    __schema__ = github_schema
    __field_names__ = ("labelable_id", "label_ids", "client_mutation_id")
    labelable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="labelableId")
    """The id of the labelable object to add labels to."""

    label_ids = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ID))), graphql_name="labelIds")
    """The ids of the labels to add."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddProjectCardInput(sgqlc.types.Input):
    """Autogenerated input type of AddProjectCard"""

    __schema__ = github_schema
    __field_names__ = ("project_column_id", "content_id", "note", "client_mutation_id")
    project_column_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectColumnId")
    """The Node ID of the ProjectColumn."""

    content_id = sgqlc.types.Field(ID, graphql_name="contentId")
    """The content of the card. Must be a member of the ProjectCardItem
    union
    """

    note = sgqlc.types.Field(String, graphql_name="note")
    """The note on the card."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddProjectColumnInput(sgqlc.types.Input):
    """Autogenerated input type of AddProjectColumn"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "name", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The Node ID of the project."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the column."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddProjectV2DraftIssueInput(sgqlc.types.Input):
    """Autogenerated input type of AddProjectV2DraftIssue"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "title", "body", "assignee_ids", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to add the draft issue to."""

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """The title of the draft issue. A project item can also be created
    by providing the URL of an Issue or Pull Request if you have
    access.
    """

    body = sgqlc.types.Field(String, graphql_name="body")
    """The body of the draft issue."""

    assignee_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="assigneeIds")
    """The IDs of the assignees of the draft issue."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddProjectV2ItemByIdInput(sgqlc.types.Input):
    """Autogenerated input type of AddProjectV2ItemById"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "content_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to add the item to."""

    content_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="contentId")
    """The id of the Issue or Pull Request to add."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddPullRequestReviewCommentInput(sgqlc.types.Input):
    """Autogenerated input type of AddPullRequestReviewComment"""

    __schema__ = github_schema
    __field_names__ = (
        "pull_request_id",
        "pull_request_review_id",
        "commit_oid",
        "body",
        "path",
        "position",
        "in_reply_to",
        "client_mutation_id",
    )
    pull_request_id = sgqlc.types.Field(ID, graphql_name="pullRequestId")
    """The node ID of the pull request reviewing  **Upcoming Change on
    2023-10-01 UTC** **Description:** `pullRequestId` will be removed.
    use addPullRequestReviewThread or addPullRequestReviewThreadReply
    instead **Reason:** We are deprecating the
    addPullRequestReviewComment mutation
    """

    pull_request_review_id = sgqlc.types.Field(ID, graphql_name="pullRequestReviewId")
    """The Node ID of the review to modify.  **Upcoming Change on
    2023-10-01 UTC** **Description:** `pullRequestReviewId` will be
    removed. use addPullRequestReviewThread or
    addPullRequestReviewThreadReply instead **Reason:** We are
    deprecating the addPullRequestReviewComment mutation
    """

    commit_oid = sgqlc.types.Field(GitObjectID, graphql_name="commitOID")
    """The SHA of the commit to comment on.  **Upcoming Change on
    2023-10-01 UTC** **Description:** `commitOID` will be removed. use
    addPullRequestReviewThread or addPullRequestReviewThreadReply
    instead **Reason:** We are deprecating the
    addPullRequestReviewComment mutation
    """

    body = sgqlc.types.Field(String, graphql_name="body")
    """The text of the comment. This field is required  **Upcoming Change
    on 2023-10-01 UTC** **Description:** `body` will be removed. use
    addPullRequestReviewThread or addPullRequestReviewThreadReply
    instead **Reason:** We are deprecating the
    addPullRequestReviewComment mutation
    """

    path = sgqlc.types.Field(String, graphql_name="path")
    """The relative path of the file to comment on.  **Upcoming Change on
    2023-10-01 UTC** **Description:** `path` will be removed. use
    addPullRequestReviewThread or addPullRequestReviewThreadReply
    instead **Reason:** We are deprecating the
    addPullRequestReviewComment mutation
    """

    position = sgqlc.types.Field(Int, graphql_name="position")
    """The line index in the diff to comment on.  **Upcoming Change on
    2023-10-01 UTC** **Description:** `position` will be removed. use
    addPullRequestReviewThread or addPullRequestReviewThreadReply
    instead **Reason:** We are deprecating the
    addPullRequestReviewComment mutation
    """

    in_reply_to = sgqlc.types.Field(ID, graphql_name="inReplyTo")
    """The comment id to reply to.  **Upcoming Change on 2023-10-01 UTC**
    **Description:** `inReplyTo` will be removed. use
    addPullRequestReviewThread or addPullRequestReviewThreadReply
    instead **Reason:** We are deprecating the
    addPullRequestReviewComment mutation
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddPullRequestReviewInput(sgqlc.types.Input):
    """Autogenerated input type of AddPullRequestReview"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "commit_oid", "body", "event", "comments", "threads", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """The Node ID of the pull request to modify."""

    commit_oid = sgqlc.types.Field(GitObjectID, graphql_name="commitOID")
    """The commit OID the review pertains to."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The contents of the review body comment."""

    event = sgqlc.types.Field(PullRequestReviewEvent, graphql_name="event")
    """The event to perform on the pull request review."""

    comments = sgqlc.types.Field(sgqlc.types.list_of("DraftPullRequestReviewComment"), graphql_name="comments")
    """The review line comments.  **Upcoming Change on 2023-10-01 UTC**
    **Description:** `comments` will be removed. use the `threads`
    argument instead **Reason:** We are deprecating comment fields
    that use diff-relative positioning
    """

    threads = sgqlc.types.Field(sgqlc.types.list_of("DraftPullRequestReviewThread"), graphql_name="threads")
    """The review line comment threads."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddPullRequestReviewThreadInput(sgqlc.types.Input):
    """Autogenerated input type of AddPullRequestReviewThread"""

    __schema__ = github_schema
    __field_names__ = (
        "path",
        "body",
        "pull_request_id",
        "pull_request_review_id",
        "line",
        "side",
        "start_line",
        "start_side",
        "subject_type",
        "client_mutation_id",
    )
    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """Path to the file being commented on."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """Body of the thread's first comment."""

    pull_request_id = sgqlc.types.Field(ID, graphql_name="pullRequestId")
    """The node ID of the pull request reviewing"""

    pull_request_review_id = sgqlc.types.Field(ID, graphql_name="pullRequestReviewId")
    """The Node ID of the review to modify."""

    line = sgqlc.types.Field(Int, graphql_name="line")
    """The line of the blob to which the thread refers, required for
    line-level threads. The end of the line range for multi-line
    comments.
    """

    side = sgqlc.types.Field(DiffSide, graphql_name="side")
    """The side of the diff on which the line resides. For multi-line
    comments, this is the side for the end of the line range.
    """

    start_line = sgqlc.types.Field(Int, graphql_name="startLine")
    """The first line of the range to which the comment refers."""

    start_side = sgqlc.types.Field(DiffSide, graphql_name="startSide")
    """The side of the diff on which the start line resides."""

    subject_type = sgqlc.types.Field(PullRequestReviewThreadSubjectType, graphql_name="subjectType")
    """The level at which the comments in the corresponding thread are
    targeted, can be a diff line or a file
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddReactionInput(sgqlc.types.Input):
    """Autogenerated input type of AddReaction"""

    __schema__ = github_schema
    __field_names__ = ("subject_id", "content", "client_mutation_id")
    subject_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="subjectId")
    """The Node ID of the subject to modify."""

    content = sgqlc.types.Field(sgqlc.types.non_null(ReactionContent), graphql_name="content")
    """The name of the emoji to react with."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddStarInput(sgqlc.types.Input):
    """Autogenerated input type of AddStar"""

    __schema__ = github_schema
    __field_names__ = ("starrable_id", "client_mutation_id")
    starrable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="starrableId")
    """The Starrable ID to star."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddUpvoteInput(sgqlc.types.Input):
    """Autogenerated input type of AddUpvote"""

    __schema__ = github_schema
    __field_names__ = ("subject_id", "client_mutation_id")
    subject_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="subjectId")
    """The Node ID of the discussion or comment to upvote."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddVerifiableDomainInput(sgqlc.types.Input):
    """Autogenerated input type of AddVerifiableDomain"""

    __schema__ = github_schema
    __field_names__ = ("owner_id", "domain", "client_mutation_id")
    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The ID of the owner to add the domain to"""

    domain = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="domain")
    """The URL of the domain"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ApproveDeploymentsInput(sgqlc.types.Input):
    """Autogenerated input type of ApproveDeployments"""

    __schema__ = github_schema
    __field_names__ = ("workflow_run_id", "environment_ids", "comment", "client_mutation_id")
    workflow_run_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="workflowRunId")
    """The node ID of the workflow run containing the pending
    deployments.
    """

    environment_ids = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ID))), graphql_name="environmentIds")
    """The ids of environments to reject deployments"""

    comment = sgqlc.types.Field(String, graphql_name="comment")
    """Optional comment for approving deployments"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ApproveVerifiableDomainInput(sgqlc.types.Input):
    """Autogenerated input type of ApproveVerifiableDomain"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the verifiable domain to approve."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ArchiveProjectV2ItemInput(sgqlc.types.Input):
    """Autogenerated input type of ArchiveProjectV2Item"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "item_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to archive the item from."""

    item_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="itemId")
    """The ID of the ProjectV2Item to archive."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ArchiveRepositoryInput(sgqlc.types.Input):
    """Autogenerated input type of ArchiveRepository"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the repository to mark as archived."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AuditLogOrder(sgqlc.types.Input):
    """Ordering options for Audit Log connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(AuditLogOrderField, graphql_name="field")
    """The field to order Audit Logs by."""

    direction = sgqlc.types.Field(OrderDirection, graphql_name="direction")
    """The ordering direction."""


class BranchNamePatternParametersInput(sgqlc.types.Input):
    """Parameters to be used for the branch_name_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(Boolean, graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class BulkSponsorship(sgqlc.types.Input):
    """Information about a sponsorship to make for a user or organization
    with a GitHub Sponsors profile, as part of sponsoring many users
    or organizations at once.
    """

    __schema__ = github_schema
    __field_names__ = ("sponsorable_id", "sponsorable_login", "amount")
    sponsorable_id = sgqlc.types.Field(ID, graphql_name="sponsorableId")
    """The ID of the user or organization who is receiving the
    sponsorship. Required if sponsorableLogin is not given.
    """

    sponsorable_login = sgqlc.types.Field(String, graphql_name="sponsorableLogin")
    """The username of the user or organization who is receiving the
    sponsorship. Required if sponsorableId is not given.
    """

    amount = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="amount")
    """The amount to pay to the sponsorable in US dollars. Valid values:
    1-12000.
    """


class CancelEnterpriseAdminInvitationInput(sgqlc.types.Input):
    """Autogenerated input type of CancelEnterpriseAdminInvitation"""

    __schema__ = github_schema
    __field_names__ = ("invitation_id", "client_mutation_id")
    invitation_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="invitationId")
    """The Node ID of the pending enterprise administrator invitation."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CancelSponsorshipInput(sgqlc.types.Input):
    """Autogenerated input type of CancelSponsorship"""

    __schema__ = github_schema
    __field_names__ = ("sponsor_id", "sponsor_login", "sponsorable_id", "sponsorable_login", "client_mutation_id")
    sponsor_id = sgqlc.types.Field(ID, graphql_name="sponsorId")
    """The ID of the user or organization who is acting as the sponsor,
    paying for the sponsorship. Required if sponsorLogin is not given.
    """

    sponsor_login = sgqlc.types.Field(String, graphql_name="sponsorLogin")
    """The username of the user or organization who is acting as the
    sponsor, paying for the sponsorship. Required if sponsorId is not
    given.
    """

    sponsorable_id = sgqlc.types.Field(ID, graphql_name="sponsorableId")
    """The ID of the user or organization who is receiving the
    sponsorship. Required if sponsorableLogin is not given.
    """

    sponsorable_login = sgqlc.types.Field(String, graphql_name="sponsorableLogin")
    """The username of the user or organization who is receiving the
    sponsorship. Required if sponsorableId is not given.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ChangeUserStatusInput(sgqlc.types.Input):
    """Autogenerated input type of ChangeUserStatus"""

    __schema__ = github_schema
    __field_names__ = ("emoji", "message", "organization_id", "limited_availability", "expires_at", "client_mutation_id")
    emoji = sgqlc.types.Field(String, graphql_name="emoji")
    """The emoji to represent your status. Can either be a native Unicode
    emoji or an emoji name with colons, e.g., :grinning:.
    """

    message = sgqlc.types.Field(String, graphql_name="message")
    """A short description of your current status."""

    organization_id = sgqlc.types.Field(ID, graphql_name="organizationId")
    """The ID of the organization whose members will be allowed to see
    the status. If omitted, the status will be publicly visible.
    """

    limited_availability = sgqlc.types.Field(Boolean, graphql_name="limitedAvailability")
    """Whether this status should indicate you are not fully available on
    GitHub, e.g., you are away.
    """

    expires_at = sgqlc.types.Field(DateTime, graphql_name="expiresAt")
    """If set, the user status will not be shown after this date."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CheckAnnotationData(sgqlc.types.Input):
    """Information from a check run analysis to specific lines of code."""

    __schema__ = github_schema
    __field_names__ = ("path", "location", "annotation_level", "message", "title", "raw_details")
    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """The path of the file to add an annotation to."""

    location = sgqlc.types.Field(sgqlc.types.non_null("CheckAnnotationRange"), graphql_name="location")
    """The location of the annotation"""

    annotation_level = sgqlc.types.Field(sgqlc.types.non_null(CheckAnnotationLevel), graphql_name="annotationLevel")
    """Represents an annotation's information level"""

    message = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="message")
    """A short description of the feedback for these lines of code."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """The title that represents the annotation."""

    raw_details = sgqlc.types.Field(String, graphql_name="rawDetails")
    """Details about this annotation."""


class CheckAnnotationRange(sgqlc.types.Input):
    """Information from a check run analysis to specific lines of code."""

    __schema__ = github_schema
    __field_names__ = ("start_line", "start_column", "end_line", "end_column")
    start_line = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="startLine")
    """The starting line of the range."""

    start_column = sgqlc.types.Field(Int, graphql_name="startColumn")
    """The starting column of the range."""

    end_line = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="endLine")
    """The ending line of the range."""

    end_column = sgqlc.types.Field(Int, graphql_name="endColumn")
    """The ending column of the range."""


class CheckRunAction(sgqlc.types.Input):
    """Possible further actions the integrator can perform."""

    __schema__ = github_schema
    __field_names__ = ("label", "description", "identifier")
    label = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="label")
    """The text to be displayed on a button in the web UI."""

    description = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="description")
    """A short explanation of what this action would do."""

    identifier = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="identifier")
    """A reference for the action on the integrator's system."""


class CheckRunFilter(sgqlc.types.Input):
    """The filters that are available when fetching check runs."""

    __schema__ = github_schema
    __field_names__ = ("check_type", "app_id", "check_name", "status", "statuses", "conclusions")
    check_type = sgqlc.types.Field(CheckRunType, graphql_name="checkType")
    """Filters the check runs by this type."""

    app_id = sgqlc.types.Field(Int, graphql_name="appId")
    """Filters the check runs created by this application ID."""

    check_name = sgqlc.types.Field(String, graphql_name="checkName")
    """Filters the check runs by this name."""

    status = sgqlc.types.Field(CheckStatusState, graphql_name="status")
    """Filters the check runs by this status. Superceded by statuses."""

    statuses = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(CheckStatusState)), graphql_name="statuses")
    """Filters the check runs by this status. Overrides status."""

    conclusions = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(CheckConclusionState)), graphql_name="conclusions")
    """Filters the check runs by these conclusions."""


class CheckRunOutput(sgqlc.types.Input):
    """Descriptive details about the check run."""

    __schema__ = github_schema
    __field_names__ = ("title", "summary", "text", "annotations", "images")
    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """A title to provide for this check run."""

    summary = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="summary")
    """The summary of the check run (supports Commonmark)."""

    text = sgqlc.types.Field(String, graphql_name="text")
    """The details of the check run (supports Commonmark)."""

    annotations = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(CheckAnnotationData)), graphql_name="annotations")
    """The annotations that are made as part of the check run."""

    images = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("CheckRunOutputImage")), graphql_name="images")
    """Images attached to the check run output displayed in the GitHub
    pull request UI.
    """


class CheckRunOutputImage(sgqlc.types.Input):
    """Images attached to the check run output displayed in the GitHub
    pull request UI.
    """

    __schema__ = github_schema
    __field_names__ = ("alt", "image_url", "caption")
    alt = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="alt")
    """The alternative text for the image."""

    image_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="imageUrl")
    """The full URL of the image."""

    caption = sgqlc.types.Field(String, graphql_name="caption")
    """A short image description."""


class CheckSuiteAutoTriggerPreference(sgqlc.types.Input):
    """The auto-trigger preferences that are available for check suites."""

    __schema__ = github_schema
    __field_names__ = ("app_id", "setting")
    app_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="appId")
    """The node ID of the application that owns the check suite."""

    setting = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="setting")
    """Set to `true` to enable automatic creation of CheckSuite events
    upon pushes to the repository.
    """


class CheckSuiteFilter(sgqlc.types.Input):
    """The filters that are available when fetching check suites."""

    __schema__ = github_schema
    __field_names__ = ("app_id", "check_name")
    app_id = sgqlc.types.Field(Int, graphql_name="appId")
    """Filters the check suites created by this application ID."""

    check_name = sgqlc.types.Field(String, graphql_name="checkName")
    """Filters the check suites by this name."""


class ClearLabelsFromLabelableInput(sgqlc.types.Input):
    """Autogenerated input type of ClearLabelsFromLabelable"""

    __schema__ = github_schema
    __field_names__ = ("labelable_id", "client_mutation_id")
    labelable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="labelableId")
    """The id of the labelable object to clear the labels from."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ClearProjectV2ItemFieldValueInput(sgqlc.types.Input):
    """Autogenerated input type of ClearProjectV2ItemFieldValue"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "item_id", "field_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project."""

    item_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="itemId")
    """The ID of the item to be cleared."""

    field_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="fieldId")
    """The ID of the field to be cleared."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CloneProjectInput(sgqlc.types.Input):
    """Autogenerated input type of CloneProject"""

    __schema__ = github_schema
    __field_names__ = ("target_owner_id", "source_id", "include_workflows", "name", "body", "public", "client_mutation_id")
    target_owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="targetOwnerId")
    """The owner ID to create the project under."""

    source_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="sourceId")
    """The source project to clone."""

    include_workflows = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="includeWorkflows")
    """Whether or not to clone the source project's workflows."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the project."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The description of the project."""

    public = sgqlc.types.Field(Boolean, graphql_name="public")
    """The visibility of the project, defaults to false (private)."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CloneTemplateRepositoryInput(sgqlc.types.Input):
    """Autogenerated input type of CloneTemplateRepository"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "name", "owner_id", "description", "visibility", "include_all_branches", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the template repository."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the new repository."""

    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The ID of the owner for the new repository."""

    description = sgqlc.types.Field(String, graphql_name="description")
    """A short description of the new repository."""

    visibility = sgqlc.types.Field(sgqlc.types.non_null(RepositoryVisibility), graphql_name="visibility")
    """Indicates the repository's visibility level."""

    include_all_branches = sgqlc.types.Field(Boolean, graphql_name="includeAllBranches")
    """Whether to copy all branches from the template to the new
    repository. Defaults to copying only the default branch of the
    template.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CloseDiscussionInput(sgqlc.types.Input):
    """Autogenerated input type of CloseDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("discussion_id", "reason", "client_mutation_id")
    discussion_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="discussionId")
    """ID of the discussion to be closed."""

    reason = sgqlc.types.Field(DiscussionCloseReason, graphql_name="reason")
    """The reason why the discussion is being closed."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CloseIssueInput(sgqlc.types.Input):
    """Autogenerated input type of CloseIssue"""

    __schema__ = github_schema
    __field_names__ = ("issue_id", "state_reason", "client_mutation_id")
    issue_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="issueId")
    """ID of the issue to be closed."""

    state_reason = sgqlc.types.Field(IssueClosedStateReason, graphql_name="stateReason")
    """The reason the issue is to be closed."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ClosePullRequestInput(sgqlc.types.Input):
    """Autogenerated input type of ClosePullRequest"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """ID of the pull request to be closed."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CommitAuthor(sgqlc.types.Input):
    """Specifies an author for filtering Git commits."""

    __schema__ = github_schema
    __field_names__ = ("id", "emails")
    id = sgqlc.types.Field(ID, graphql_name="id")
    """ID of a User to filter by. If non-null, only commits authored by
    this user will be returned. This field takes precedence over
    emails.
    """

    emails = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="emails")
    """Email addresses to filter by. Commits authored by any of the
    specified email addresses will be returned.
    """


class CommitAuthorEmailPatternParametersInput(sgqlc.types.Input):
    """Parameters to be used for the commit_author_email_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(Boolean, graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class CommitContributionOrder(sgqlc.types.Input):
    """Ordering options for commit contribution connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(CommitContributionOrderField), graphql_name="field")
    """The field by which to order commit contributions."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class CommitMessage(sgqlc.types.Input):
    """A message to include with a new commit"""

    __schema__ = github_schema
    __field_names__ = ("headline", "body")
    headline = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="headline")
    """The headline of the message."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The body of the message."""


class CommitMessagePatternParametersInput(sgqlc.types.Input):
    """Parameters to be used for the commit_message_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(Boolean, graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class CommittableBranch(sgqlc.types.Input):
    """A git ref for a commit to be appended to.  The ref must be a
    branch, i.e. its fully qualified name must start with
    `refs/heads/` (although the input is not required to be fully
    qualified).  The Ref may be specified by its global node ID or by
    the `repositoryNameWithOwner` and `branchName`.  ### Examples
    Specify a branch using a global node ID:      { "id":
    "MDM6UmVmMTpyZWZzL2hlYWRzL21haW4=" }  Specify a branch using
    `repositoryNameWithOwner` and `branchName`:      {
    "repositoryNameWithOwner": "github/graphql-client",
    "branchName": "main"     }
    """

    __schema__ = github_schema
    __field_names__ = ("id", "repository_name_with_owner", "branch_name")
    id = sgqlc.types.Field(ID, graphql_name="id")
    """The Node ID of the Ref to be updated."""

    repository_name_with_owner = sgqlc.types.Field(String, graphql_name="repositoryNameWithOwner")
    """The nameWithOwner of the repository to commit to."""

    branch_name = sgqlc.types.Field(String, graphql_name="branchName")
    """The unqualified name of the branch to append the commit to."""


class CommitterEmailPatternParametersInput(sgqlc.types.Input):
    """Parameters to be used for the committer_email_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(Boolean, graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class ContributionOrder(sgqlc.types.Input):
    """Ordering options for contribution connections."""

    __schema__ = github_schema
    __field_names__ = ("direction",)
    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class ConvertProjectCardNoteToIssueInput(sgqlc.types.Input):
    """Autogenerated input type of ConvertProjectCardNoteToIssue"""

    __schema__ = github_schema
    __field_names__ = ("project_card_id", "repository_id", "title", "body", "client_mutation_id")
    project_card_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectCardId")
    """The ProjectCard ID to convert."""

    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the repository to create the issue in."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """The title of the newly created issue. Defaults to the card's note
    text.
    """

    body = sgqlc.types.Field(String, graphql_name="body")
    """The body of the newly created issue."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ConvertPullRequestToDraftInput(sgqlc.types.Input):
    """Autogenerated input type of ConvertPullRequestToDraft"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """ID of the pull request to convert to draft"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CopyProjectV2Input(sgqlc.types.Input):
    """Autogenerated input type of CopyProjectV2"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "owner_id", "title", "include_draft_issues", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the source Project to copy."""

    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The owner ID of the new project."""

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """The title of the project."""

    include_draft_issues = sgqlc.types.Field(Boolean, graphql_name="includeDraftIssues")
    """Include draft issues in the new project"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateAttributionInvitationInput(sgqlc.types.Input):
    """Autogenerated input type of CreateAttributionInvitation"""

    __schema__ = github_schema
    __field_names__ = ("owner_id", "source_id", "target_id", "client_mutation_id")
    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The Node ID of the owner scoping the reattributable data."""

    source_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="sourceId")
    """The Node ID of the account owning the data to reattribute."""

    target_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="targetId")
    """The Node ID of the account which may claim the data."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateBranchProtectionRuleInput(sgqlc.types.Input):
    """Autogenerated input type of CreateBranchProtectionRule"""

    __schema__ = github_schema
    __field_names__ = (
        "repository_id",
        "pattern",
        "requires_approving_reviews",
        "required_approving_review_count",
        "requires_commit_signatures",
        "requires_linear_history",
        "blocks_creations",
        "allows_force_pushes",
        "allows_deletions",
        "is_admin_enforced",
        "requires_status_checks",
        "requires_strict_status_checks",
        "requires_code_owner_reviews",
        "dismisses_stale_reviews",
        "restricts_review_dismissals",
        "review_dismissal_actor_ids",
        "bypass_pull_request_actor_ids",
        "bypass_force_push_actor_ids",
        "restricts_pushes",
        "push_actor_ids",
        "required_status_check_contexts",
        "required_status_checks",
        "requires_deployments",
        "required_deployment_environments",
        "requires_conversation_resolution",
        "require_last_push_approval",
        "lock_branch",
        "lock_allows_fetch_and_merge",
        "client_mutation_id",
    )
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The global relay id of the repository in which a new branch
    protection rule should be created in.
    """

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The glob-like pattern used to determine matching branches."""

    requires_approving_reviews = sgqlc.types.Field(Boolean, graphql_name="requiresApprovingReviews")
    """Are approving reviews required to update matching branches."""

    required_approving_review_count = sgqlc.types.Field(Int, graphql_name="requiredApprovingReviewCount")
    """Number of approving reviews required to update matching branches."""

    requires_commit_signatures = sgqlc.types.Field(Boolean, graphql_name="requiresCommitSignatures")
    """Are commits required to be signed."""

    requires_linear_history = sgqlc.types.Field(Boolean, graphql_name="requiresLinearHistory")
    """Are merge commits prohibited from being pushed to this branch."""

    blocks_creations = sgqlc.types.Field(Boolean, graphql_name="blocksCreations")
    """Is branch creation a protected operation."""

    allows_force_pushes = sgqlc.types.Field(Boolean, graphql_name="allowsForcePushes")
    """Are force pushes allowed on this branch."""

    allows_deletions = sgqlc.types.Field(Boolean, graphql_name="allowsDeletions")
    """Can this branch be deleted."""

    is_admin_enforced = sgqlc.types.Field(Boolean, graphql_name="isAdminEnforced")
    """Can admins overwrite branch protection."""

    requires_status_checks = sgqlc.types.Field(Boolean, graphql_name="requiresStatusChecks")
    """Are status checks required to update matching branches."""

    requires_strict_status_checks = sgqlc.types.Field(Boolean, graphql_name="requiresStrictStatusChecks")
    """Are branches required to be up to date before merging."""

    requires_code_owner_reviews = sgqlc.types.Field(Boolean, graphql_name="requiresCodeOwnerReviews")
    """Are reviews from code owners required to update matching branches."""

    dismisses_stale_reviews = sgqlc.types.Field(Boolean, graphql_name="dismissesStaleReviews")
    """Will new commits pushed to matching branches dismiss pull request
    review approvals.
    """

    restricts_review_dismissals = sgqlc.types.Field(Boolean, graphql_name="restrictsReviewDismissals")
    """Is dismissal of pull request reviews restricted."""

    review_dismissal_actor_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="reviewDismissalActorIds")
    """A list of User, Team, or App IDs allowed to dismiss reviews on
    pull requests targeting matching branches.
    """

    bypass_pull_request_actor_ids = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="bypassPullRequestActorIds"
    )
    """A list of User, Team, or App IDs allowed to bypass pull requests
    targeting matching branches.
    """

    bypass_force_push_actor_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="bypassForcePushActorIds")
    """A list of User, Team, or App IDs allowed to bypass force push
    targeting matching branches.
    """

    restricts_pushes = sgqlc.types.Field(Boolean, graphql_name="restrictsPushes")
    """Is pushing to matching branches restricted."""

    push_actor_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="pushActorIds")
    """A list of User, Team, or App IDs allowed to push to matching
    branches.
    """

    required_status_check_contexts = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="requiredStatusCheckContexts"
    )
    """List of required status check contexts that must pass for commits
    to be accepted to matching branches.
    """

    required_status_checks = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null("RequiredStatusCheckInput")), graphql_name="requiredStatusChecks"
    )
    """The list of required status checks"""

    requires_deployments = sgqlc.types.Field(Boolean, graphql_name="requiresDeployments")
    """Are successful deployments required before merging."""

    required_deployment_environments = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="requiredDeploymentEnvironments"
    )
    """The list of required deployment environments"""

    requires_conversation_resolution = sgqlc.types.Field(Boolean, graphql_name="requiresConversationResolution")
    """Are conversations required to be resolved before merging."""

    require_last_push_approval = sgqlc.types.Field(Boolean, graphql_name="requireLastPushApproval")
    """Whether the most recent push must be approved by someone other
    than the person who pushed it
    """

    lock_branch = sgqlc.types.Field(Boolean, graphql_name="lockBranch")
    """Whether to set the branch as read-only. If this is true, users
    will not be able to push to the branch.
    """

    lock_allows_fetch_and_merge = sgqlc.types.Field(Boolean, graphql_name="lockAllowsFetchAndMerge")
    """Whether users can pull changes from upstream when the branch is
    locked. Set to `true` to allow fork syncing. Set to `false` to
    prevent fork syncing.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateCheckRunInput(sgqlc.types.Input):
    """Autogenerated input type of CreateCheckRun"""

    __schema__ = github_schema
    __field_names__ = (
        "repository_id",
        "name",
        "head_sha",
        "details_url",
        "external_id",
        "status",
        "started_at",
        "conclusion",
        "completed_at",
        "output",
        "actions",
        "client_mutation_id",
    )
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The node ID of the repository."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the check."""

    head_sha = sgqlc.types.Field(sgqlc.types.non_null(GitObjectID), graphql_name="headSha")
    """The SHA of the head commit."""

    details_url = sgqlc.types.Field(URI, graphql_name="detailsUrl")
    """The URL of the integrator's site that has the full details of the
    check.
    """

    external_id = sgqlc.types.Field(String, graphql_name="externalId")
    """A reference for the run on the integrator's system."""

    status = sgqlc.types.Field(RequestableCheckStatusState, graphql_name="status")
    """The current status."""

    started_at = sgqlc.types.Field(DateTime, graphql_name="startedAt")
    """The time that the check run began."""

    conclusion = sgqlc.types.Field(CheckConclusionState, graphql_name="conclusion")
    """The final conclusion of the check."""

    completed_at = sgqlc.types.Field(DateTime, graphql_name="completedAt")
    """The time that the check run finished."""

    output = sgqlc.types.Field(CheckRunOutput, graphql_name="output")
    """Descriptive details about the run."""

    actions = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(CheckRunAction)), graphql_name="actions")
    """Possible further actions the integrator can perform, which a user
    may trigger.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateCheckSuiteInput(sgqlc.types.Input):
    """Autogenerated input type of CreateCheckSuite"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "head_sha", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the repository."""

    head_sha = sgqlc.types.Field(sgqlc.types.non_null(GitObjectID), graphql_name="headSha")
    """The SHA of the head commit."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateCommitOnBranchInput(sgqlc.types.Input):
    """Autogenerated input type of CreateCommitOnBranch"""

    __schema__ = github_schema
    __field_names__ = ("branch", "file_changes", "message", "expected_head_oid", "client_mutation_id")
    branch = sgqlc.types.Field(sgqlc.types.non_null(CommittableBranch), graphql_name="branch")
    """The Ref to be updated.  Must be a branch."""

    file_changes = sgqlc.types.Field("FileChanges", graphql_name="fileChanges")
    """A description of changes to files in this commit."""

    message = sgqlc.types.Field(sgqlc.types.non_null(CommitMessage), graphql_name="message")
    """The commit message the be included with the commit."""

    expected_head_oid = sgqlc.types.Field(sgqlc.types.non_null(GitObjectID), graphql_name="expectedHeadOid")
    """The git commit oid expected at the head of the branch prior to the
    commit
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateDiscussionInput(sgqlc.types.Input):
    """Autogenerated input type of CreateDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "title", "body", "category_id", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The id of the repository on which to create the discussion."""

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """The title of the discussion."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The body of the discussion."""

    category_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="categoryId")
    """The id of the discussion category to associate with this
    discussion.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateEnterpriseOrganizationInput(sgqlc.types.Input):
    """Autogenerated input type of CreateEnterpriseOrganization"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "login", "profile_name", "billing_email", "admin_logins", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise owning the new organization."""

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The login of the new organization."""

    profile_name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="profileName")
    """The profile name of the new organization."""

    billing_email = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="billingEmail")
    """The email used for sending billing receipts."""

    admin_logins = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="adminLogins")
    """The logins for the administrators of the new organization."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateEnvironmentInput(sgqlc.types.Input):
    """Autogenerated input type of CreateEnvironment"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "name", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The node ID of the repository."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the environment."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateIpAllowListEntryInput(sgqlc.types.Input):
    """Autogenerated input type of CreateIpAllowListEntry"""

    __schema__ = github_schema
    __field_names__ = ("owner_id", "allow_list_value", "name", "is_active", "client_mutation_id")
    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The ID of the owner for which to create the new IP allow list
    entry.
    """

    allow_list_value = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="allowListValue")
    """An IP address or range of addresses in CIDR notation."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """An optional name for the IP allow list entry."""

    is_active = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isActive")
    """Whether the IP allow list entry is active when an IP allow list is
    enabled.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateIssueInput(sgqlc.types.Input):
    """Autogenerated input type of CreateIssue"""

    __schema__ = github_schema
    __field_names__ = (
        "repository_id",
        "title",
        "body",
        "assignee_ids",
        "milestone_id",
        "label_ids",
        "project_ids",
        "issue_template",
        "client_mutation_id",
    )
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the repository."""

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """The title for the issue."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The body for the issue description."""

    assignee_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="assigneeIds")
    """The Node ID for the user assignee for this issue."""

    milestone_id = sgqlc.types.Field(ID, graphql_name="milestoneId")
    """The Node ID of the milestone for this issue."""

    label_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="labelIds")
    """An array of Node IDs of labels for this issue."""

    project_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="projectIds")
    """An array of Node IDs for projects associated with this issue."""

    issue_template = sgqlc.types.Field(String, graphql_name="issueTemplate")
    """The name of an issue template in the repository, assigns labels
    and assignees from the template to the issue
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateLinkedBranchInput(sgqlc.types.Input):
    """Autogenerated input type of CreateLinkedBranch"""

    __schema__ = github_schema
    __field_names__ = ("issue_id", "oid", "name", "repository_id", "client_mutation_id")
    issue_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="issueId")
    """ID of the issue to link to."""

    oid = sgqlc.types.Field(sgqlc.types.non_null(GitObjectID), graphql_name="oid")
    """The commit SHA to base the new branch on."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """The name of the new branch. Defaults to issue number and title."""

    repository_id = sgqlc.types.Field(ID, graphql_name="repositoryId")
    """ID of the repository to create the branch in. Defaults to the
    issue repository.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateMigrationSourceInput(sgqlc.types.Input):
    """Autogenerated input type of CreateMigrationSource"""

    __schema__ = github_schema
    __field_names__ = ("name", "url", "access_token", "type", "owner_id", "github_pat", "client_mutation_id")
    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The migration source name."""

    url = sgqlc.types.Field(String, graphql_name="url")
    """The migration source URL, for example `https://github.com` or
    `https://monalisa.ghe.com`.
    """

    access_token = sgqlc.types.Field(String, graphql_name="accessToken")
    """The migration source access token."""

    type = sgqlc.types.Field(sgqlc.types.non_null(MigrationSourceType), graphql_name="type")
    """The migration source type."""

    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The ID of the organization that will own the migration source."""

    github_pat = sgqlc.types.Field(String, graphql_name="githubPat")
    """The GitHub personal access token of the user importing to the
    target repository.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateProjectInput(sgqlc.types.Input):
    """Autogenerated input type of CreateProject"""

    __schema__ = github_schema
    __field_names__ = ("owner_id", "name", "body", "template", "repository_ids", "client_mutation_id")
    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The owner ID to create the project under."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of project."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The description of project."""

    template = sgqlc.types.Field(ProjectTemplate, graphql_name="template")
    """The name of the GitHub-provided template."""

    repository_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="repositoryIds")
    """A list of repository IDs to create as linked repositories for the
    project
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateProjectV2FieldInput(sgqlc.types.Input):
    """Autogenerated input type of CreateProjectV2Field"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "data_type", "name", "single_select_options", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to create the field in."""

    data_type = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2CustomFieldType), graphql_name="dataType")
    """The data type of the field."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the field."""

    single_select_options = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null("ProjectV2SingleSelectFieldOptionInput")), graphql_name="singleSelectOptions"
    )
    """Options for a single select field. At least one value is required
    if data_type is SINGLE_SELECT
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateProjectV2Input(sgqlc.types.Input):
    """Autogenerated input type of CreateProjectV2"""

    __schema__ = github_schema
    __field_names__ = ("owner_id", "title", "repository_id", "team_id", "client_mutation_id")
    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The owner ID to create the project under."""

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """The title of the project."""

    repository_id = sgqlc.types.Field(ID, graphql_name="repositoryId")
    """The repository to link the project to."""

    team_id = sgqlc.types.Field(ID, graphql_name="teamId")
    """The team to link the project to. The team will be granted read
    permissions.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreatePullRequestInput(sgqlc.types.Input):
    """Autogenerated input type of CreatePullRequest"""

    __schema__ = github_schema
    __field_names__ = (
        "repository_id",
        "base_ref_name",
        "head_ref_name",
        "head_repository_id",
        "title",
        "body",
        "maintainer_can_modify",
        "draft",
        "client_mutation_id",
    )
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the repository."""

    base_ref_name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="baseRefName")
    """The name of the branch you want your changes pulled into. This
    should be an existing branch on the current repository. You cannot
    update the base branch on a pull request to point to another
    repository.
    """

    head_ref_name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="headRefName")
    """The name of the branch where your changes are implemented. For
    cross-repository pull requests in the same network, namespace
    `head_ref_name` with a user like this: `username:branch`.
    """

    head_repository_id = sgqlc.types.Field(ID, graphql_name="headRepositoryId")
    """The Node ID of the head repository."""

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """The title of the pull request."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The contents of the pull request."""

    maintainer_can_modify = sgqlc.types.Field(Boolean, graphql_name="maintainerCanModify")
    """Indicates whether maintainers can modify the pull request."""

    draft = sgqlc.types.Field(Boolean, graphql_name="draft")
    """Indicates whether this pull request should be a draft."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateRefInput(sgqlc.types.Input):
    """Autogenerated input type of CreateRef"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "name", "oid", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the Repository to create the Ref in."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The fully qualified name of the new Ref (ie:
    `refs/heads/my_new_branch`).
    """

    oid = sgqlc.types.Field(sgqlc.types.non_null(GitObjectID), graphql_name="oid")
    """The GitObjectID that the new Ref shall target. Must point to a
    commit.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateRepositoryInput(sgqlc.types.Input):
    """Autogenerated input type of CreateRepository"""

    __schema__ = github_schema
    __field_names__ = (
        "name",
        "owner_id",
        "description",
        "visibility",
        "template",
        "homepage_url",
        "has_wiki_enabled",
        "has_issues_enabled",
        "team_id",
        "client_mutation_id",
    )
    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the new repository."""

    owner_id = sgqlc.types.Field(ID, graphql_name="ownerId")
    """The ID of the owner for the new repository."""

    description = sgqlc.types.Field(String, graphql_name="description")
    """A short description of the new repository."""

    visibility = sgqlc.types.Field(sgqlc.types.non_null(RepositoryVisibility), graphql_name="visibility")
    """Indicates the repository's visibility level."""

    template = sgqlc.types.Field(Boolean, graphql_name="template")
    """Whether this repository should be marked as a template such that
    anyone who can access it can create new repositories with the same
    files and directory structure.
    """

    homepage_url = sgqlc.types.Field(URI, graphql_name="homepageUrl")
    """The URL for a web page about this repository."""

    has_wiki_enabled = sgqlc.types.Field(Boolean, graphql_name="hasWikiEnabled")
    """Indicates if the repository should have the wiki feature enabled."""

    has_issues_enabled = sgqlc.types.Field(Boolean, graphql_name="hasIssuesEnabled")
    """Indicates if the repository should have the issues feature
    enabled.
    """

    team_id = sgqlc.types.Field(ID, graphql_name="teamId")
    """When an organization is specified as the owner, this ID identifies
    the team that should be granted access to the new repository.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateRepositoryRulesetInput(sgqlc.types.Input):
    """Autogenerated input type of CreateRepositoryRuleset"""

    __schema__ = github_schema
    __field_names__ = (
        "source_id",
        "name",
        "target",
        "rules",
        "conditions",
        "enforcement",
        "bypass_mode",
        "bypass_actor_ids",
        "client_mutation_id",
    )
    source_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="sourceId")
    """The global relay id of the source in which a new ruleset should be
    created in.
    """

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the ruleset."""

    target = sgqlc.types.Field(RepositoryRulesetTarget, graphql_name="target")
    """The target of the ruleset."""

    rules = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("RepositoryRuleInput")), graphql_name="rules")
    """The list of rules for this ruleset"""

    conditions = sgqlc.types.Field(sgqlc.types.non_null("RepositoryRuleConditionsInput"), graphql_name="conditions")
    """The set of conditions for this ruleset"""

    enforcement = sgqlc.types.Field(sgqlc.types.non_null(RuleEnforcement), graphql_name="enforcement")
    """The enforcement level for this ruleset"""

    bypass_mode = sgqlc.types.Field(RuleBypassMode, graphql_name="bypassMode")
    """The bypass mode for this ruleset"""

    bypass_actor_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="bypassActorIds")
    """A list of Team or App IDs allowed to bypass rules in this ruleset."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateSponsorsListingInput(sgqlc.types.Input):
    """Autogenerated input type of CreateSponsorsListing"""

    __schema__ = github_schema
    __field_names__ = (
        "sponsorable_login",
        "fiscal_host_login",
        "fiscally_hosted_project_profile_url",
        "billing_country_or_region_code",
        "residence_country_or_region_code",
        "contact_email",
        "full_description",
        "client_mutation_id",
    )
    sponsorable_login = sgqlc.types.Field(String, graphql_name="sponsorableLogin")
    """The username of the organization to create a GitHub Sponsors
    profile for, if desired. Defaults to creating a GitHub Sponsors
    profile for the authenticated user if omitted.
    """

    fiscal_host_login = sgqlc.types.Field(String, graphql_name="fiscalHostLogin")
    """The username of the supported fiscal host's GitHub organization,
    if you want to receive sponsorship payouts through a fiscal host
    rather than directly to a bank account. For example, 'Open-Source-
    Collective' for Open Source Collective or 'numfocus' for numFOCUS.
    Case insensitive. See https://docs.github.com/sponsors/receiving-
    sponsorships-through-github-sponsors/using-a-fiscal-host-to-
    receive-github-sponsors-payouts for more information.
    """

    fiscally_hosted_project_profile_url = sgqlc.types.Field(String, graphql_name="fiscallyHostedProjectProfileUrl")
    """The URL for your profile page on the fiscal host's website, e.g.,
    https://opencollective.com/babel or
    https://numfocus.org/project/bokeh. Required if fiscalHostLogin is
    specified.
    """

    billing_country_or_region_code = sgqlc.types.Field(SponsorsCountryOrRegionCode, graphql_name="billingCountryOrRegionCode")
    """The country or region where the sponsorable's bank account is
    located. Required if fiscalHostLogin is not specified, ignored
    when fiscalHostLogin is specified.
    """

    residence_country_or_region_code = sgqlc.types.Field(SponsorsCountryOrRegionCode, graphql_name="residenceCountryOrRegionCode")
    """The country or region where the sponsorable resides. This is for
    tax purposes. Required if the sponsorable is yourself, ignored
    when sponsorableLogin specifies an organization.
    """

    contact_email = sgqlc.types.Field(String, graphql_name="contactEmail")
    """The email address we should use to contact you about the GitHub
    Sponsors profile being created. This will not be shared publicly.
    Must be a verified email address already on your GitHub account.
    Only relevant when the sponsorable is yourself. Defaults to your
    primary email address on file if omitted.
    """

    full_description = sgqlc.types.Field(String, graphql_name="fullDescription")
    """Provide an introduction to serve as the main focus that appears on
    your GitHub Sponsors profile. It's a great opportunity to help
    potential sponsors learn more about you, your work, and why their
    sponsorship is important to you. GitHub-flavored Markdown is
    supported.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateSponsorsTierInput(sgqlc.types.Input):
    """Autogenerated input type of CreateSponsorsTier"""

    __schema__ = github_schema
    __field_names__ = (
        "sponsorable_id",
        "sponsorable_login",
        "amount",
        "is_recurring",
        "repository_id",
        "repository_owner_login",
        "repository_name",
        "welcome_message",
        "description",
        "publish",
        "client_mutation_id",
    )
    sponsorable_id = sgqlc.types.Field(ID, graphql_name="sponsorableId")
    """The ID of the user or organization who owns the GitHub Sponsors
    profile. Defaults to the current user if omitted and
    sponsorableLogin is not given.
    """

    sponsorable_login = sgqlc.types.Field(String, graphql_name="sponsorableLogin")
    """The username of the user or organization who owns the GitHub
    Sponsors profile. Defaults to the current user if omitted and
    sponsorableId is not given.
    """

    amount = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="amount")
    """The value of the new tier in US dollars. Valid values: 1-12000."""

    is_recurring = sgqlc.types.Field(Boolean, graphql_name="isRecurring")
    """Whether sponsorships using this tier should happen monthly/yearly
    or just once.
    """

    repository_id = sgqlc.types.Field(ID, graphql_name="repositoryId")
    """Optional ID of the private repository that sponsors at this tier
    should gain read-only access to. Must be owned by an organization.
    """

    repository_owner_login = sgqlc.types.Field(String, graphql_name="repositoryOwnerLogin")
    """Optional login of the organization owner of the private repository
    that sponsors at this tier should gain read-only access to.
    Necessary if repositoryName is given. Will be ignored if
    repositoryId is given.
    """

    repository_name = sgqlc.types.Field(String, graphql_name="repositoryName")
    """Optional name of the private repository that sponsors at this tier
    should gain read-only access to. Must be owned by an organization.
    Necessary if repositoryOwnerLogin is given. Will be ignored if
    repositoryId is given.
    """

    welcome_message = sgqlc.types.Field(String, graphql_name="welcomeMessage")
    """Optional message new sponsors at this tier will receive."""

    description = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="description")
    """A description of what this tier is, what perks sponsors might
    receive, what a sponsorship at this tier means for you, etc.
    """

    publish = sgqlc.types.Field(Boolean, graphql_name="publish")
    """Whether to make the tier available immediately for sponsors to
    choose. Defaults to creating a draft tier that will not be
    publicly visible.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateSponsorshipInput(sgqlc.types.Input):
    """Autogenerated input type of CreateSponsorship"""

    __schema__ = github_schema
    __field_names__ = (
        "sponsor_id",
        "sponsor_login",
        "sponsorable_id",
        "sponsorable_login",
        "tier_id",
        "amount",
        "is_recurring",
        "receive_emails",
        "privacy_level",
        "client_mutation_id",
    )
    sponsor_id = sgqlc.types.Field(ID, graphql_name="sponsorId")
    """The ID of the user or organization who is acting as the sponsor,
    paying for the sponsorship. Required if sponsorLogin is not given.
    """

    sponsor_login = sgqlc.types.Field(String, graphql_name="sponsorLogin")
    """The username of the user or organization who is acting as the
    sponsor, paying for the sponsorship. Required if sponsorId is not
    given.
    """

    sponsorable_id = sgqlc.types.Field(ID, graphql_name="sponsorableId")
    """The ID of the user or organization who is receiving the
    sponsorship. Required if sponsorableLogin is not given.
    """

    sponsorable_login = sgqlc.types.Field(String, graphql_name="sponsorableLogin")
    """The username of the user or organization who is receiving the
    sponsorship. Required if sponsorableId is not given.
    """

    tier_id = sgqlc.types.Field(ID, graphql_name="tierId")
    """The ID of one of sponsorable's existing tiers to sponsor at.
    Required if amount is not specified.
    """

    amount = sgqlc.types.Field(Int, graphql_name="amount")
    """The amount to pay to the sponsorable in US dollars. Required if a
    tierId is not specified. Valid values: 1-12000.
    """

    is_recurring = sgqlc.types.Field(Boolean, graphql_name="isRecurring")
    """Whether the sponsorship should happen monthly/yearly or just this
    one time. Required if a tierId is not specified.
    """

    receive_emails = sgqlc.types.Field(Boolean, graphql_name="receiveEmails")
    """Whether the sponsor should receive email updates from the
    sponsorable.
    """

    privacy_level = sgqlc.types.Field(SponsorshipPrivacy, graphql_name="privacyLevel")
    """Specify whether others should be able to see that the sponsor is
    sponsoring the sponsorable. Public visibility still does not
    reveal which tier is used.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateSponsorshipsInput(sgqlc.types.Input):
    """Autogenerated input type of CreateSponsorships"""

    __schema__ = github_schema
    __field_names__ = ("sponsor_login", "sponsorships", "receive_emails", "privacy_level", "client_mutation_id")
    sponsor_login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="sponsorLogin")
    """The username of the user or organization who is acting as the
    sponsor, paying for the sponsorships.
    """

    sponsorships = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(BulkSponsorship))), graphql_name="sponsorships"
    )
    """The list of maintainers to sponsor and for how much apiece."""

    receive_emails = sgqlc.types.Field(Boolean, graphql_name="receiveEmails")
    """Whether the sponsor should receive email updates from the
    sponsorables.
    """

    privacy_level = sgqlc.types.Field(SponsorshipPrivacy, graphql_name="privacyLevel")
    """Specify whether others should be able to see that the sponsor is
    sponsoring the sponsorables. Public visibility still does not
    reveal the dollar value of the sponsorship.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateTeamDiscussionCommentInput(sgqlc.types.Input):
    """Autogenerated input type of CreateTeamDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("discussion_id", "body", "client_mutation_id")
    discussion_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="discussionId")
    """The ID of the discussion to which the comment belongs."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The content of the comment."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateTeamDiscussionInput(sgqlc.types.Input):
    """Autogenerated input type of CreateTeamDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("team_id", "title", "body", "private", "client_mutation_id")
    team_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="teamId")
    """The ID of the team to which the discussion belongs."""

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """The title of the discussion."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The content of the discussion."""

    private = sgqlc.types.Field(Boolean, graphql_name="private")
    """If true, restricts the visibility of this discussion to team
    members and organization admins. If false or not specified, allows
    any organization member to view this discussion.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeclineTopicSuggestionInput(sgqlc.types.Input):
    """Autogenerated input type of DeclineTopicSuggestion"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "name", "reason", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the repository."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the suggested topic."""

    reason = sgqlc.types.Field(sgqlc.types.non_null(TopicSuggestionDeclineReason), graphql_name="reason")
    """The reason why the suggested topic is declined."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteBranchProtectionRuleInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteBranchProtectionRule"""

    __schema__ = github_schema
    __field_names__ = ("branch_protection_rule_id", "client_mutation_id")
    branch_protection_rule_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="branchProtectionRuleId")
    """The global relay id of the branch protection rule to be deleted."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteDeploymentInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteDeployment"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The Node ID of the deployment to be deleted."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteDiscussionCommentInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The Node id of the discussion comment to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteDiscussionInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The id of the discussion to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteEnvironmentInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteEnvironment"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The Node ID of the environment to be deleted."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteIpAllowListEntryInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteIpAllowListEntry"""

    __schema__ = github_schema
    __field_names__ = ("ip_allow_list_entry_id", "client_mutation_id")
    ip_allow_list_entry_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ipAllowListEntryId")
    """The ID of the IP allow list entry to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteIssueCommentInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteIssueComment"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the comment to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteIssueInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteIssue"""

    __schema__ = github_schema
    __field_names__ = ("issue_id", "client_mutation_id")
    issue_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="issueId")
    """The ID of the issue to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteLinkedBranchInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteLinkedBranch"""

    __schema__ = github_schema
    __field_names__ = ("linked_branch_id", "client_mutation_id")
    linked_branch_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="linkedBranchId")
    """The ID of the linked branch"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteProjectCardInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteProjectCard"""

    __schema__ = github_schema
    __field_names__ = ("card_id", "client_mutation_id")
    card_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="cardId")
    """The id of the card to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteProjectColumnInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteProjectColumn"""

    __schema__ = github_schema
    __field_names__ = ("column_id", "client_mutation_id")
    column_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="columnId")
    """The id of the column to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteProjectInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteProject"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The Project ID to update."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteProjectV2FieldInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteProjectV2Field"""

    __schema__ = github_schema
    __field_names__ = ("field_id", "client_mutation_id")
    field_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="fieldId")
    """The ID of the field to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteProjectV2Input(sgqlc.types.Input):
    """Autogenerated input type of DeleteProjectV2"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteProjectV2ItemInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteProjectV2Item"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "item_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project from which the item should be removed."""

    item_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="itemId")
    """The ID of the item to be removed."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteProjectV2WorkflowInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteProjectV2Workflow"""

    __schema__ = github_schema
    __field_names__ = ("workflow_id", "client_mutation_id")
    workflow_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="workflowId")
    """The ID of the workflow to be removed."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeletePullRequestReviewCommentInput(sgqlc.types.Input):
    """Autogenerated input type of DeletePullRequestReviewComment"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the comment to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeletePullRequestReviewInput(sgqlc.types.Input):
    """Autogenerated input type of DeletePullRequestReview"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_review_id", "client_mutation_id")
    pull_request_review_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestReviewId")
    """The Node ID of the pull request review to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteRefInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteRef"""

    __schema__ = github_schema
    __field_names__ = ("ref_id", "client_mutation_id")
    ref_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="refId")
    """The Node ID of the Ref to be deleted."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteRepositoryRulesetInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteRepositoryRuleset"""

    __schema__ = github_schema
    __field_names__ = ("repository_ruleset_id", "client_mutation_id")
    repository_ruleset_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryRulesetId")
    """The global relay id of the repository ruleset to be deleted."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteTeamDiscussionCommentInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteTeamDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the comment to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteTeamDiscussionInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteTeamDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The discussion ID to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteVerifiableDomainInput(sgqlc.types.Input):
    """Autogenerated input type of DeleteVerifiableDomain"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the verifiable domain to delete."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeploymentOrder(sgqlc.types.Input):
    """Ordering options for deployment connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(DeploymentOrderField), graphql_name="field")
    """The field to order deployments by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class DequeuePullRequestInput(sgqlc.types.Input):
    """Autogenerated input type of DequeuePullRequest"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the pull request to be dequeued."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DisablePullRequestAutoMergeInput(sgqlc.types.Input):
    """Autogenerated input type of DisablePullRequestAutoMerge"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """ID of the pull request to disable auto merge on."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DiscussionOrder(sgqlc.types.Input):
    """Ways in which lists of discussions can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(DiscussionOrderField), graphql_name="field")
    """The field by which to order discussions."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order discussions by the specified
    field.
    """


class DiscussionPollOptionOrder(sgqlc.types.Input):
    """Ordering options for discussion poll option connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(DiscussionPollOptionOrderField), graphql_name="field")
    """The field to order poll options by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class DismissPullRequestReviewInput(sgqlc.types.Input):
    """Autogenerated input type of DismissPullRequestReview"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_review_id", "message", "client_mutation_id")
    pull_request_review_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestReviewId")
    """The Node ID of the pull request review to modify."""

    message = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="message")
    """The contents of the pull request review dismissal message."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DismissRepositoryVulnerabilityAlertInput(sgqlc.types.Input):
    """Autogenerated input type of DismissRepositoryVulnerabilityAlert"""

    __schema__ = github_schema
    __field_names__ = ("repository_vulnerability_alert_id", "dismiss_reason", "client_mutation_id")
    repository_vulnerability_alert_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryVulnerabilityAlertId")
    """The Dependabot alert ID to dismiss."""

    dismiss_reason = sgqlc.types.Field(sgqlc.types.non_null(DismissReason), graphql_name="dismissReason")
    """The reason the Dependabot alert is being dismissed."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DraftPullRequestReviewComment(sgqlc.types.Input):
    """Specifies a review comment to be left with a Pull Request Review."""

    __schema__ = github_schema
    __field_names__ = ("path", "position", "body")
    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """Path to the file being commented on."""

    position = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="position")
    """Position in the file to leave a comment on."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """Body of the comment to leave."""


class DraftPullRequestReviewThread(sgqlc.types.Input):
    """Specifies a review comment thread to be left with a Pull Request
    Review.
    """

    __schema__ = github_schema
    __field_names__ = ("path", "line", "side", "start_line", "start_side", "body")
    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """Path to the file being commented on."""

    line = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="line")
    """The line of the blob to which the thread refers. The end of the
    line range for multi-line comments.
    """

    side = sgqlc.types.Field(DiffSide, graphql_name="side")
    """The side of the diff on which the line resides. For multi-line
    comments, this is the side for the end of the line range.
    """

    start_line = sgqlc.types.Field(Int, graphql_name="startLine")
    """The first line of the range to which the comment refers."""

    start_side = sgqlc.types.Field(DiffSide, graphql_name="startSide")
    """The side of the diff on which the start line resides."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """Body of the comment to leave."""


class EnablePullRequestAutoMergeInput(sgqlc.types.Input):
    """Autogenerated input type of EnablePullRequestAutoMerge"""

    __schema__ = github_schema
    __field_names__ = (
        "pull_request_id",
        "commit_headline",
        "commit_body",
        "merge_method",
        "author_email",
        "expected_head_oid",
        "client_mutation_id",
    )
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """ID of the pull request to enable auto-merge on."""

    commit_headline = sgqlc.types.Field(String, graphql_name="commitHeadline")
    """Commit headline to use for the commit when the PR is mergable; if
    omitted, a default message will be used. NOTE: when merging with a
    merge queue any input value for commit headline is ignored.
    """

    commit_body = sgqlc.types.Field(String, graphql_name="commitBody")
    """Commit body to use for the commit when the PR is mergable; if
    omitted, a default message will be used. NOTE: when merging with a
    merge queue any input value for commit message is ignored.
    """

    merge_method = sgqlc.types.Field(PullRequestMergeMethod, graphql_name="mergeMethod")
    """The merge method to use. If omitted, defaults to `MERGE`. NOTE:
    when merging with a merge queue any input value for merge method
    is ignored.
    """

    author_email = sgqlc.types.Field(String, graphql_name="authorEmail")
    """The email address to associate with this merge."""

    expected_head_oid = sgqlc.types.Field(GitObjectID, graphql_name="expectedHeadOid")
    """The expected head OID of the pull request."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class EnqueuePullRequestInput(sgqlc.types.Input):
    """Autogenerated input type of EnqueuePullRequest"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "jump", "expected_head_oid", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """The ID of the pull request to enqueue."""

    jump = sgqlc.types.Field(Boolean, graphql_name="jump")
    """Add the pull request to the front of the queue."""

    expected_head_oid = sgqlc.types.Field(GitObjectID, graphql_name="expectedHeadOid")
    """The expected head OID of the pull request."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class EnterpriseAdministratorInvitationOrder(sgqlc.types.Input):
    """Ordering options for enterprise administrator invitation
    connections
    """

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseAdministratorInvitationOrderField), graphql_name="field")
    """The field to order enterprise administrator invitations by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class EnterpriseMemberOrder(sgqlc.types.Input):
    """Ordering options for enterprise member connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseMemberOrderField), graphql_name="field")
    """The field to order enterprise members by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class EnterpriseServerInstallationOrder(sgqlc.types.Input):
    """Ordering options for Enterprise Server installation connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseServerInstallationOrderField), graphql_name="field")
    """The field to order Enterprise Server installations by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class EnterpriseServerUserAccountEmailOrder(sgqlc.types.Input):
    """Ordering options for Enterprise Server user account email
    connections.
    """

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseServerUserAccountEmailOrderField), graphql_name="field")
    """The field to order emails by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class EnterpriseServerUserAccountOrder(sgqlc.types.Input):
    """Ordering options for Enterprise Server user account connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseServerUserAccountOrderField), graphql_name="field")
    """The field to order user accounts by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class EnterpriseServerUserAccountsUploadOrder(sgqlc.types.Input):
    """Ordering options for Enterprise Server user accounts upload
    connections.
    """

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseServerUserAccountsUploadOrderField), graphql_name="field")
    """The field to order user accounts uploads by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class FileAddition(sgqlc.types.Input):
    """A command to add a file at the given path with the given contents
    as part of a commit.  Any existing file at that that path will be
    replaced.
    """

    __schema__ = github_schema
    __field_names__ = ("path", "contents")
    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """The path in the repository where the file will be located"""

    contents = sgqlc.types.Field(sgqlc.types.non_null(Base64String), graphql_name="contents")
    """The base64 encoded contents of the file"""


class FileChanges(sgqlc.types.Input):
    """A description of a set of changes to a file tree to be made as
    part of a git commit, modeled as zero or more file `additions` and
    zero or more file `deletions`.  Both fields are optional; omitting
    both will produce a commit with no file changes.  `deletions` and
    `additions` describe changes to files identified by their path in
    the git tree using unix-style path separators, i.e. `/`.  The root
    of a git tree is an empty string, so paths are not slash-prefixed.
    `path` values must be unique across all `additions` and
    `deletions` provided.  Any duplication will result in a validation
    error.  ### Encoding  File contents must be provided in full for
    each `FileAddition`.  The `contents` of a `FileAddition` must be
    encoded using RFC 4648 compliant base64, i.e. correct padding is
    required and no characters outside the standard alphabet may be
    used.  Invalid base64 encoding will be rejected with a validation
    error.  The encoded contents may be binary.  For text files, no
    assumptions are made about the character encoding of the file
    contents (after base64 decoding).  No charset transcoding or line-
    ending normalization will be performed; it is the client's
    responsibility to manage the character encoding of files they
    provide. However, for maximum compatibility we recommend using
    UTF-8 encoding and ensuring that all files in a repository use a
    consistent line-ending convention (`\n` or `\r\n`), and that all
    files end with a newline.  ### Modeling file changes  Each of the
    the five types of conceptual changes that can be made in a git
    commit can be described using the `FileChanges` type as follows:
    1. New file addition: create file `hello world\n` at path
    `docs/README.txt`:         {          "additions" [            {
    "path": "docs/README.txt",              "contents":
    base64encode("hello world\n")            }          ]        }  2.
    Existing file modification: change existing `docs/README.txt` to
    have new    content `new content here\n`:         {
    "additions" [            {              "path": "docs/README.txt",
    "contents": base64encode("new content here\n")            }
    ]        }  3. Existing file deletion: remove existing file
    `docs/README.txt`.    Note that the path is required to exist --
    specifying a    path that does not exist on the given branch will
    abort the    commit and return an error.         {
    "deletions" [            {              "path": "docs/README.txt"
    }          ]        }   4. File rename with no changes: rename
    `docs/README.txt` with    previous content `hello world\n` to the
    same content at    `newdocs/README.txt`:         {
    "deletions" [            {              "path": "docs/README.txt",
    }          ],          "additions" [            {
    "path": "newdocs/README.txt",              "contents":
    base64encode("hello world\n")            }          ]        }
    5. File rename with changes: rename `docs/README.txt` with
    previous content `hello world\n` to a file at path
    `newdocs/README.txt` with content `new contents\n`:         {
    "deletions" [            {              "path": "docs/README.txt",
    }          ],          "additions" [            {
    "path": "newdocs/README.txt",              "contents":
    base64encode("new contents\n")            }          ]        }
    """

    __schema__ = github_schema
    __field_names__ = ("deletions", "additions")
    deletions = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("FileDeletion")), graphql_name="deletions")
    """Files to delete."""

    additions = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(FileAddition)), graphql_name="additions")
    """File to add or change."""


class FileDeletion(sgqlc.types.Input):
    """A command to delete the file at the given path as part of a
    commit.
    """

    __schema__ = github_schema
    __field_names__ = ("path",)
    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """The path to delete"""


class FollowOrganizationInput(sgqlc.types.Input):
    """Autogenerated input type of FollowOrganization"""

    __schema__ = github_schema
    __field_names__ = ("organization_id", "client_mutation_id")
    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """ID of the organization to follow."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class FollowUserInput(sgqlc.types.Input):
    """Autogenerated input type of FollowUser"""

    __schema__ = github_schema
    __field_names__ = ("user_id", "client_mutation_id")
    user_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="userId")
    """ID of the user to follow."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class GistOrder(sgqlc.types.Input):
    """Ordering options for gist connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(GistOrderField), graphql_name="field")
    """The field to order repositories by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class GrantEnterpriseOrganizationsMigratorRoleInput(sgqlc.types.Input):
    """Autogenerated input type of
    GrantEnterpriseOrganizationsMigratorRole
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "login", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise to which all organizations managed by it
    will be granted the migrator role.
    """

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The login of the user to grant the migrator role"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class GrantMigratorRoleInput(sgqlc.types.Input):
    """Autogenerated input type of GrantMigratorRole"""

    __schema__ = github_schema
    __field_names__ = ("organization_id", "actor", "actor_type", "client_mutation_id")
    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization that the user/team belongs to."""

    actor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="actor")
    """The user login or Team slug to grant the migrator role."""

    actor_type = sgqlc.types.Field(sgqlc.types.non_null(ActorType), graphql_name="actorType")
    """Specifies the type of the actor, can be either USER or TEAM."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class InviteEnterpriseAdminInput(sgqlc.types.Input):
    """Autogenerated input type of InviteEnterpriseAdmin"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "invitee", "email", "role", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise to which you want to invite an
    administrator.
    """

    invitee = sgqlc.types.Field(String, graphql_name="invitee")
    """The login of a user to invite as an administrator."""

    email = sgqlc.types.Field(String, graphql_name="email")
    """The email of the person to invite as an administrator."""

    role = sgqlc.types.Field(EnterpriseAdministratorRole, graphql_name="role")
    """The role of the administrator."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class IpAllowListEntryOrder(sgqlc.types.Input):
    """Ordering options for IP allow list entry connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(IpAllowListEntryOrderField), graphql_name="field")
    """The field to order IP allow list entries by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class IssueCommentOrder(sgqlc.types.Input):
    """Ways in which lists of issue comments can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(IssueCommentOrderField), graphql_name="field")
    """The field in which to order issue comments by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order issue comments by the specified
    field.
    """


class IssueFilters(sgqlc.types.Input):
    """Ways in which to filter lists of issues."""

    __schema__ = github_schema
    __field_names__ = (
        "assignee",
        "created_by",
        "labels",
        "mentioned",
        "milestone",
        "milestone_number",
        "since",
        "states",
        "viewer_subscribed",
    )
    assignee = sgqlc.types.Field(String, graphql_name="assignee")
    """List issues assigned to given name. Pass in `null` for issues with
    no assigned user, and `*` for issues assigned to any user.
    """

    created_by = sgqlc.types.Field(String, graphql_name="createdBy")
    """List issues created by given name."""

    labels = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="labels")
    """List issues where the list of label names exist on the issue."""

    mentioned = sgqlc.types.Field(String, graphql_name="mentioned")
    """List issues where the given name is mentioned in the issue."""

    milestone = sgqlc.types.Field(String, graphql_name="milestone")
    """List issues by given milestone argument. If an string
    representation of an integer is passed, it should refer to a
    milestone by its database ID. Pass in `null` for issues with no
    milestone, and `*` for issues that are assigned to any milestone.
    """

    milestone_number = sgqlc.types.Field(String, graphql_name="milestoneNumber")
    """List issues by given milestone argument. If an string
    representation of an integer is passed, it should refer to a
    milestone by its number field. Pass in `null` for issues with no
    milestone, and `*` for issues that are assigned to any milestone.
    """

    since = sgqlc.types.Field(DateTime, graphql_name="since")
    """List issues that have been updated at or after the given date."""

    states = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(IssueState)), graphql_name="states")
    """List issues filtered by the list of states given."""

    viewer_subscribed = sgqlc.types.Field(Boolean, graphql_name="viewerSubscribed")
    """List issues subscribed to by viewer."""


class IssueOrder(sgqlc.types.Input):
    """Ways in which lists of issues can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(IssueOrderField), graphql_name="field")
    """The field in which to order issues by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order issues by the specified field."""


class LabelOrder(sgqlc.types.Input):
    """Ways in which lists of labels can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(LabelOrderField), graphql_name="field")
    """The field in which to order labels by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order labels by the specified field."""


class LanguageOrder(sgqlc.types.Input):
    """Ordering options for language connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(LanguageOrderField), graphql_name="field")
    """The field to order languages by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class LinkProjectV2ToRepositoryInput(sgqlc.types.Input):
    """Autogenerated input type of LinkProjectV2ToRepository"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "repository_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the project to link to the repository."""

    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the repository to link to the project."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class LinkProjectV2ToTeamInput(sgqlc.types.Input):
    """Autogenerated input type of LinkProjectV2ToTeam"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "team_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the project to link to the team."""

    team_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="teamId")
    """The ID of the team to link to the project."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class LinkRepositoryToProjectInput(sgqlc.types.Input):
    """Autogenerated input type of LinkRepositoryToProject"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "repository_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to link to a Repository"""

    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the Repository to link to a Project."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class LockLockableInput(sgqlc.types.Input):
    """Autogenerated input type of LockLockable"""

    __schema__ = github_schema
    __field_names__ = ("lockable_id", "lock_reason", "client_mutation_id")
    lockable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="lockableId")
    """ID of the item to be locked."""

    lock_reason = sgqlc.types.Field(LockReason, graphql_name="lockReason")
    """A reason for why the item will be locked."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MannequinOrder(sgqlc.types.Input):
    """Ordering options for mannequins."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(MannequinOrderField), graphql_name="field")
    """The field to order mannequins by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class MarkDiscussionCommentAsAnswerInput(sgqlc.types.Input):
    """Autogenerated input type of MarkDiscussionCommentAsAnswer"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The Node ID of the discussion comment to mark as an answer."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MarkFileAsViewedInput(sgqlc.types.Input):
    """Autogenerated input type of MarkFileAsViewed"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "path", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """The Node ID of the pull request."""

    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """The path of the file to mark as viewed"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MarkProjectV2AsTemplateInput(sgqlc.types.Input):
    """Autogenerated input type of MarkProjectV2AsTemplate"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to mark as a template."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MarkPullRequestReadyForReviewInput(sgqlc.types.Input):
    """Autogenerated input type of MarkPullRequestReadyForReview"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """ID of the pull request to be marked as ready for review."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MergeBranchInput(sgqlc.types.Input):
    """Autogenerated input type of MergeBranch"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "base", "head", "commit_message", "author_email", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the Repository containing the base branch that will
    be modified.
    """

    base = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="base")
    """The name of the base branch that the provided head will be merged
    into.
    """

    head = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="head")
    """The head to merge into the base branch. This can be a branch name
    or a commit GitObjectID.
    """

    commit_message = sgqlc.types.Field(String, graphql_name="commitMessage")
    """Message to use for the merge commit. If omitted, a default will be
    used.
    """

    author_email = sgqlc.types.Field(String, graphql_name="authorEmail")
    """The email address to associate with this commit."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MergePullRequestInput(sgqlc.types.Input):
    """Autogenerated input type of MergePullRequest"""

    __schema__ = github_schema
    __field_names__ = (
        "pull_request_id",
        "commit_headline",
        "commit_body",
        "expected_head_oid",
        "merge_method",
        "author_email",
        "client_mutation_id",
    )
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """ID of the pull request to be merged."""

    commit_headline = sgqlc.types.Field(String, graphql_name="commitHeadline")
    """Commit headline to use for the merge commit; if omitted, a default
    message will be used.
    """

    commit_body = sgqlc.types.Field(String, graphql_name="commitBody")
    """Commit body to use for the merge commit; if omitted, a default
    message will be used
    """

    expected_head_oid = sgqlc.types.Field(GitObjectID, graphql_name="expectedHeadOid")
    """OID that the pull request head ref must match to allow merge; if
    omitted, no check is performed.
    """

    merge_method = sgqlc.types.Field(PullRequestMergeMethod, graphql_name="mergeMethod")
    """The merge method to use. If omitted, defaults to 'MERGE' """

    author_email = sgqlc.types.Field(String, graphql_name="authorEmail")
    """The email address to associate with this merge."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MilestoneOrder(sgqlc.types.Input):
    """Ordering options for milestone connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(MilestoneOrderField), graphql_name="field")
    """The field to order milestones by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class MinimizeCommentInput(sgqlc.types.Input):
    """Autogenerated input type of MinimizeComment"""

    __schema__ = github_schema
    __field_names__ = ("subject_id", "classifier", "client_mutation_id")
    subject_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="subjectId")
    """The Node ID of the subject to modify."""

    classifier = sgqlc.types.Field(sgqlc.types.non_null(ReportedContentClassifiers), graphql_name="classifier")
    """The classification of comment"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MoveProjectCardInput(sgqlc.types.Input):
    """Autogenerated input type of MoveProjectCard"""

    __schema__ = github_schema
    __field_names__ = ("card_id", "column_id", "after_card_id", "client_mutation_id")
    card_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="cardId")
    """The id of the card to move."""

    column_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="columnId")
    """The id of the column to move it into."""

    after_card_id = sgqlc.types.Field(ID, graphql_name="afterCardId")
    """Place the new card after the card with this id. Pass null to place
    it at the top.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MoveProjectColumnInput(sgqlc.types.Input):
    """Autogenerated input type of MoveProjectColumn"""

    __schema__ = github_schema
    __field_names__ = ("column_id", "after_column_id", "client_mutation_id")
    column_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="columnId")
    """The id of the column to move."""

    after_column_id = sgqlc.types.Field(ID, graphql_name="afterColumnId")
    """Place the new column after the column with this id. Pass null to
    place it at the front.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class OrgEnterpriseOwnerOrder(sgqlc.types.Input):
    """Ordering options for an organization's enterprise owner
    connections.
    """

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(OrgEnterpriseOwnerOrderField), graphql_name="field")
    """The field to order enterprise owners by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class OrganizationOrder(sgqlc.types.Input):
    """Ordering options for organization connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(OrganizationOrderField), graphql_name="field")
    """The field to order organizations by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class PackageFileOrder(sgqlc.types.Input):
    """Ways in which lists of package files can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(PackageFileOrderField, graphql_name="field")
    """The field in which to order package files by."""

    direction = sgqlc.types.Field(OrderDirection, graphql_name="direction")
    """The direction in which to order package files by the specified
    field.
    """


class PackageOrder(sgqlc.types.Input):
    """Ways in which lists of packages can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(PackageOrderField, graphql_name="field")
    """The field in which to order packages by."""

    direction = sgqlc.types.Field(OrderDirection, graphql_name="direction")
    """The direction in which to order packages by the specified field."""


class PackageVersionOrder(sgqlc.types.Input):
    """Ways in which lists of package versions can be ordered upon
    return.
    """

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(PackageVersionOrderField, graphql_name="field")
    """The field in which to order package versions by."""

    direction = sgqlc.types.Field(OrderDirection, graphql_name="direction")
    """The direction in which to order package versions by the specified
    field.
    """


class PinIssueInput(sgqlc.types.Input):
    """Autogenerated input type of PinIssue"""

    __schema__ = github_schema
    __field_names__ = ("issue_id", "client_mutation_id")
    issue_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="issueId")
    """The ID of the issue to be pinned"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ProjectOrder(sgqlc.types.Input):
    """Ways in which lists of projects can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(ProjectOrderField), graphql_name="field")
    """The field in which to order projects by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order projects by the specified field."""


class ProjectV2Collaborator(sgqlc.types.Input):
    """A collaborator to update on a project. Only one of the userId or
    teamId should be provided.
    """

    __schema__ = github_schema
    __field_names__ = ("user_id", "team_id", "role")
    user_id = sgqlc.types.Field(ID, graphql_name="userId")
    """The ID of the user as a collaborator."""

    team_id = sgqlc.types.Field(ID, graphql_name="teamId")
    """The ID of the team as a collaborator."""

    role = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2Roles), graphql_name="role")
    """The role to grant the collaborator"""


class ProjectV2FieldOrder(sgqlc.types.Input):
    """Ordering options for project v2 field connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2FieldOrderField), graphql_name="field")
    """The field to order the project v2 fields by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class ProjectV2FieldValue(sgqlc.types.Input):
    """The values that can be used to update a field of an item inside a
    Project. Only 1 value can be updated at a time.
    """

    __schema__ = github_schema
    __field_names__ = ("text", "number", "date", "single_select_option_id", "iteration_id")
    text = sgqlc.types.Field(String, graphql_name="text")
    """The text to set on the field."""

    number = sgqlc.types.Field(Float, graphql_name="number")
    """The number to set on the field."""

    date = sgqlc.types.Field(Date, graphql_name="date")
    """The ISO 8601 date to set on the field."""

    single_select_option_id = sgqlc.types.Field(String, graphql_name="singleSelectOptionId")
    """The id of the single select option to set on the field."""

    iteration_id = sgqlc.types.Field(String, graphql_name="iterationId")
    """The id of the iteration to set on the field."""


class ProjectV2Filters(sgqlc.types.Input):
    """Ways in which to filter lists of projects."""

    __schema__ = github_schema
    __field_names__ = ("state",)
    state = sgqlc.types.Field(ProjectV2State, graphql_name="state")
    """List project v2 filtered by the state given."""


class ProjectV2ItemFieldValueOrder(sgqlc.types.Input):
    """Ordering options for project v2 item field value connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2ItemFieldValueOrderField), graphql_name="field")
    """The field to order the project v2 item field values by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class ProjectV2ItemOrder(sgqlc.types.Input):
    """Ordering options for project v2 item connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2ItemOrderField), graphql_name="field")
    """The field to order the project v2 items by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class ProjectV2Order(sgqlc.types.Input):
    """Ways in which lists of projects can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2OrderField), graphql_name="field")
    """The field in which to order projects by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order projects by the specified field."""


class ProjectV2SingleSelectFieldOptionInput(sgqlc.types.Input):
    """Represents a single select field option"""

    __schema__ = github_schema
    __field_names__ = ("name", "color", "description")
    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the option"""

    color = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2SingleSelectFieldOptionColor), graphql_name="color")
    """The display color of the option"""

    description = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="description")
    """The description text of the option"""


class ProjectV2ViewOrder(sgqlc.types.Input):
    """Ordering options for project v2 view connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2ViewOrderField), graphql_name="field")
    """The field to order the project v2 views by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class ProjectV2WorkflowOrder(sgqlc.types.Input):
    """Ordering options for project v2 workflows connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2WorkflowsOrderField), graphql_name="field")
    """The field to order the project v2 workflows by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class PublishSponsorsTierInput(sgqlc.types.Input):
    """Autogenerated input type of PublishSponsorsTier"""

    __schema__ = github_schema
    __field_names__ = ("tier_id", "client_mutation_id")
    tier_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="tierId")
    """The ID of the draft tier to publish."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class PullRequestOrder(sgqlc.types.Input):
    """Ways in which lists of issues can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(PullRequestOrderField), graphql_name="field")
    """The field in which to order pull requests by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order pull requests by the specified
    field.
    """


class PullRequestParametersInput(sgqlc.types.Input):
    """Require all commits be made to a non-target branch and submitted
    via a pull request before they can be merged.
    """

    __schema__ = github_schema
    __field_names__ = (
        "dismiss_stale_reviews_on_push",
        "require_code_owner_review",
        "require_last_push_approval",
        "required_approving_review_count",
        "required_review_thread_resolution",
    )
    dismiss_stale_reviews_on_push = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="dismissStaleReviewsOnPush")
    """New, reviewable commits pushed will dismiss previous pull request
    review approvals.
    """

    require_code_owner_review = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requireCodeOwnerReview")
    """Require an approving review in pull requests that modify files
    that have a designated code owner.
    """

    require_last_push_approval = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requireLastPushApproval")
    """Whether the most recent reviewable push must be approved by
    someone other than the person who pushed it.
    """

    required_approving_review_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="requiredApprovingReviewCount")
    """The number of approving reviews that are required before a pull
    request can be merged.
    """

    required_review_thread_resolution = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requiredReviewThreadResolution")
    """All conversations on code must be resolved before a pull request
    can be merged.
    """


class ReactionOrder(sgqlc.types.Input):
    """Ways in which lists of reactions can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(ReactionOrderField), graphql_name="field")
    """The field in which to order reactions by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order reactions by the specified field."""


class RefNameConditionTargetInput(sgqlc.types.Input):
    """Parameters to be used for the ref_name condition"""

    __schema__ = github_schema
    __field_names__ = ("exclude", "include")
    exclude = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="exclude")
    """Array of ref names or patterns to exclude. The condition will not
    pass if any of these patterns match.
    """

    include = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="include")
    """Array of ref names or patterns to include. One of these patterns
    must match for the condition to pass. Also accepts
    `~DEFAULT_BRANCH` to include the default branch or `~ALL` to
    include all branches.
    """


class RefOrder(sgqlc.types.Input):
    """Ways in which lists of git refs can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(RefOrderField), graphql_name="field")
    """The field in which to order refs by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order refs by the specified field."""


class RegenerateEnterpriseIdentityProviderRecoveryCodesInput(sgqlc.types.Input):
    """Autogenerated input type of
    RegenerateEnterpriseIdentityProviderRecoveryCodes
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set an identity provider."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RegenerateVerifiableDomainTokenInput(sgqlc.types.Input):
    """Autogenerated input type of RegenerateVerifiableDomainToken"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the verifiable domain to regenerate the verification
    token of.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RejectDeploymentsInput(sgqlc.types.Input):
    """Autogenerated input type of RejectDeployments"""

    __schema__ = github_schema
    __field_names__ = ("workflow_run_id", "environment_ids", "comment", "client_mutation_id")
    workflow_run_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="workflowRunId")
    """The node ID of the workflow run containing the pending
    deployments.
    """

    environment_ids = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ID))), graphql_name="environmentIds")
    """The ids of environments to reject deployments"""

    comment = sgqlc.types.Field(String, graphql_name="comment")
    """Optional comment for rejecting deployments"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ReleaseOrder(sgqlc.types.Input):
    """Ways in which lists of releases can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(ReleaseOrderField), graphql_name="field")
    """The field in which to order releases by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order releases by the specified field."""


class RemoveAssigneesFromAssignableInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveAssigneesFromAssignable"""

    __schema__ = github_schema
    __field_names__ = ("assignable_id", "assignee_ids", "client_mutation_id")
    assignable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="assignableId")
    """The id of the assignable object to remove assignees from."""

    assignee_ids = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ID))), graphql_name="assigneeIds")
    """The id of users to remove as assignees."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveEnterpriseAdminInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveEnterpriseAdmin"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "login", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The Enterprise ID from which to remove the administrator."""

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The login of the user to remove as an administrator."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveEnterpriseIdentityProviderInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveEnterpriseIdentityProvider"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise from which to remove the identity
    provider.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveEnterpriseMemberInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveEnterpriseMember"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "user_id", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise from which the user should be removed."""

    user_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="userId")
    """The ID of the user to remove from the enterprise."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveEnterpriseOrganizationInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveEnterpriseOrganization"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "organization_id", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise from which the organization should be
    removed.
    """

    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization to remove from the enterprise."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveEnterpriseSupportEntitlementInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveEnterpriseSupportEntitlement"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "login", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the Enterprise which the admin belongs to."""

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The login of a member who will lose the support entitlement."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveLabelsFromLabelableInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveLabelsFromLabelable"""

    __schema__ = github_schema
    __field_names__ = ("labelable_id", "label_ids", "client_mutation_id")
    labelable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="labelableId")
    """The id of the Labelable to remove labels from."""

    label_ids = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ID))), graphql_name="labelIds")
    """The ids of labels to remove."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveOutsideCollaboratorInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveOutsideCollaborator"""

    __schema__ = github_schema
    __field_names__ = ("user_id", "organization_id", "client_mutation_id")
    user_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="userId")
    """The ID of the outside collaborator to remove."""

    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization to remove the outside collaborator
    from.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveReactionInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveReaction"""

    __schema__ = github_schema
    __field_names__ = ("subject_id", "content", "client_mutation_id")
    subject_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="subjectId")
    """The Node ID of the subject to modify."""

    content = sgqlc.types.Field(sgqlc.types.non_null(ReactionContent), graphql_name="content")
    """The name of the emoji reaction to remove."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveStarInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveStar"""

    __schema__ = github_schema
    __field_names__ = ("starrable_id", "client_mutation_id")
    starrable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="starrableId")
    """The Starrable ID to unstar."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveUpvoteInput(sgqlc.types.Input):
    """Autogenerated input type of RemoveUpvote"""

    __schema__ = github_schema
    __field_names__ = ("subject_id", "client_mutation_id")
    subject_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="subjectId")
    """The Node ID of the discussion or comment to remove upvote."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ReopenDiscussionInput(sgqlc.types.Input):
    """Autogenerated input type of ReopenDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("discussion_id", "client_mutation_id")
    discussion_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="discussionId")
    """ID of the discussion to be reopened."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ReopenIssueInput(sgqlc.types.Input):
    """Autogenerated input type of ReopenIssue"""

    __schema__ = github_schema
    __field_names__ = ("issue_id", "client_mutation_id")
    issue_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="issueId")
    """ID of the issue to be opened."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ReopenPullRequestInput(sgqlc.types.Input):
    """Autogenerated input type of ReopenPullRequest"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """ID of the pull request to be reopened."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RepositoryInvitationOrder(sgqlc.types.Input):
    """Ordering options for repository invitation connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(RepositoryInvitationOrderField), graphql_name="field")
    """The field to order repository invitations by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class RepositoryMigrationOrder(sgqlc.types.Input):
    """Ordering options for repository migrations."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(RepositoryMigrationOrderField), graphql_name="field")
    """The field to order repository migrations by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(RepositoryMigrationOrderDirection), graphql_name="direction")
    """The ordering direction."""


class RepositoryNameConditionTargetInput(sgqlc.types.Input):
    """Parameters to be used for the repository_name condition"""

    __schema__ = github_schema
    __field_names__ = ("exclude", "include", "protected")
    exclude = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="exclude")
    """Array of repository names or patterns to exclude. The condition
    will not pass if any of these patterns match.
    """

    include = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="include")
    """Array of repository names or patterns to include. One of these
    patterns must match for the condition to pass. Also accepts `~ALL`
    to include all repositories.
    """

    protected = sgqlc.types.Field(Boolean, graphql_name="protected")
    """Target changes that match these patterns will be prevented except
    by those with bypass permissions.
    """


class RepositoryOrder(sgqlc.types.Input):
    """Ordering options for repository connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(RepositoryOrderField), graphql_name="field")
    """The field to order repositories by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class RepositoryRuleConditionsInput(sgqlc.types.Input):
    """Specifies the conditions required for a ruleset to evaluate"""

    __schema__ = github_schema
    __field_names__ = ("ref_name", "repository_name")
    ref_name = sgqlc.types.Field(RefNameConditionTargetInput, graphql_name="refName")
    """Configuration for the ref_name condition"""

    repository_name = sgqlc.types.Field(RepositoryNameConditionTargetInput, graphql_name="repositoryName")
    """Configuration for the repository_name condition"""


class RepositoryRuleInput(sgqlc.types.Input):
    """Specifies the attributes for a new or updated rule."""

    __schema__ = github_schema
    __field_names__ = ("id", "type", "parameters")
    id = sgqlc.types.Field(ID, graphql_name="id")
    """Optional ID of this rule when updating"""

    type = sgqlc.types.Field(sgqlc.types.non_null(RepositoryRuleType), graphql_name="type")
    """The type of rule to create."""

    parameters = sgqlc.types.Field("RuleParametersInput", graphql_name="parameters")
    """The parameters for the rule."""


class RequestReviewsInput(sgqlc.types.Input):
    """Autogenerated input type of RequestReviews"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "user_ids", "team_ids", "union", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """The Node ID of the pull request to modify."""

    user_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="userIds")
    """The Node IDs of the user to request."""

    team_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="teamIds")
    """The Node IDs of the team to request."""

    union = sgqlc.types.Field(Boolean, graphql_name="union")
    """Add users to the set rather than replace."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RequiredDeploymentsParametersInput(sgqlc.types.Input):
    """Choose which environments must be successfully deployed to before
    branches can be merged into a branch that matches this rule.
    """

    __schema__ = github_schema
    __field_names__ = ("required_deployment_environments",)
    required_deployment_environments = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="requiredDeploymentEnvironments"
    )
    """The environments that must be successfully deployed to before
    branches can be merged.
    """


class RequiredStatusCheckInput(sgqlc.types.Input):
    """Specifies the attributes for a new or updated required status
    check.
    """

    __schema__ = github_schema
    __field_names__ = ("context", "app_id")
    context = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="context")
    """Status check context that must pass for commits to be accepted to
    the matching branch.
    """

    app_id = sgqlc.types.Field(ID, graphql_name="appId")
    """The ID of the App that must set the status in order for it to be
    accepted. Omit this value to use whichever app has recently been
    setting this status, or use "any" to allow any app to set the
    status.
    """


class RequiredStatusChecksParametersInput(sgqlc.types.Input):
    """Choose which status checks must pass before branches can be merged
    into a branch that matches this rule. When enabled, commits must
    first be pushed to another branch, then merged or pushed directly
    to a branch that matches this rule after status checks have
    passed.
    """

    __schema__ = github_schema
    __field_names__ = ("required_status_checks", "strict_required_status_checks_policy")
    required_status_checks = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("StatusCheckConfigurationInput"))),
        graphql_name="requiredStatusChecks",
    )
    """Status checks that are required."""

    strict_required_status_checks_policy = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="strictRequiredStatusChecksPolicy")
    """Whether pull requests targeting a matching branch must be tested
    with the latest code. This setting will not take effect unless at
    least one status check is enabled.
    """


class RerequestCheckSuiteInput(sgqlc.types.Input):
    """Autogenerated input type of RerequestCheckSuite"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "check_suite_id", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the repository."""

    check_suite_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="checkSuiteId")
    """The Node ID of the check suite."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ResolveReviewThreadInput(sgqlc.types.Input):
    """Autogenerated input type of ResolveReviewThread"""

    __schema__ = github_schema
    __field_names__ = ("thread_id", "client_mutation_id")
    thread_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="threadId")
    """The ID of the thread to resolve"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RetireSponsorsTierInput(sgqlc.types.Input):
    """Autogenerated input type of RetireSponsorsTier"""

    __schema__ = github_schema
    __field_names__ = ("tier_id", "client_mutation_id")
    tier_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="tierId")
    """The ID of the published tier to retire."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RevertPullRequestInput(sgqlc.types.Input):
    """Autogenerated input type of RevertPullRequest"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "title", "body", "draft", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """The ID of the pull request to revert."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """The title of the revert pull request."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The description of the revert pull request."""

    draft = sgqlc.types.Field(Boolean, graphql_name="draft")
    """Indicates whether the revert pull request should be a draft."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RevokeEnterpriseOrganizationsMigratorRoleInput(sgqlc.types.Input):
    """Autogenerated input type of
    RevokeEnterpriseOrganizationsMigratorRole
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "login", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise to which all organizations managed by it
    will be granted the migrator role.
    """

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The login of the user to revoke the migrator role"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RevokeMigratorRoleInput(sgqlc.types.Input):
    """Autogenerated input type of RevokeMigratorRole"""

    __schema__ = github_schema
    __field_names__ = ("organization_id", "actor", "actor_type", "client_mutation_id")
    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization that the user/team belongs to."""

    actor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="actor")
    """The user login or Team slug to revoke the migrator role from."""

    actor_type = sgqlc.types.Field(sgqlc.types.non_null(ActorType), graphql_name="actorType")
    """Specifies the type of the actor, can be either USER or TEAM."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RuleParametersInput(sgqlc.types.Input):
    """Specifies the parameters for a `RepositoryRule` object. Only one
    of the fields should be specified.
    """

    __schema__ = github_schema
    __field_names__ = (
        "update",
        "required_deployments",
        "pull_request",
        "required_status_checks",
        "commit_message_pattern",
        "commit_author_email_pattern",
        "committer_email_pattern",
        "branch_name_pattern",
        "tag_name_pattern",
    )
    update = sgqlc.types.Field("UpdateParametersInput", graphql_name="update")
    """Parameters used for the `update` rule type"""

    required_deployments = sgqlc.types.Field(RequiredDeploymentsParametersInput, graphql_name="requiredDeployments")
    """Parameters used for the `required_deployments` rule type"""

    pull_request = sgqlc.types.Field(PullRequestParametersInput, graphql_name="pullRequest")
    """Parameters used for the `pull_request` rule type"""

    required_status_checks = sgqlc.types.Field(RequiredStatusChecksParametersInput, graphql_name="requiredStatusChecks")
    """Parameters used for the `required_status_checks` rule type"""

    commit_message_pattern = sgqlc.types.Field(CommitMessagePatternParametersInput, graphql_name="commitMessagePattern")
    """Parameters used for the `commit_message_pattern` rule type"""

    commit_author_email_pattern = sgqlc.types.Field(CommitAuthorEmailPatternParametersInput, graphql_name="commitAuthorEmailPattern")
    """Parameters used for the `commit_author_email_pattern` rule type"""

    committer_email_pattern = sgqlc.types.Field(CommitterEmailPatternParametersInput, graphql_name="committerEmailPattern")
    """Parameters used for the `committer_email_pattern` rule type"""

    branch_name_pattern = sgqlc.types.Field(BranchNamePatternParametersInput, graphql_name="branchNamePattern")
    """Parameters used for the `branch_name_pattern` rule type"""

    tag_name_pattern = sgqlc.types.Field("TagNamePatternParametersInput", graphql_name="tagNamePattern")
    """Parameters used for the `tag_name_pattern` rule type"""


class SavedReplyOrder(sgqlc.types.Input):
    """Ordering options for saved reply connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(SavedReplyOrderField), graphql_name="field")
    """The field to order saved replies by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class SecurityAdvisoryIdentifierFilter(sgqlc.types.Input):
    """An advisory identifier to filter results on."""

    __schema__ = github_schema
    __field_names__ = ("type", "value")
    type = sgqlc.types.Field(sgqlc.types.non_null(SecurityAdvisoryIdentifierType), graphql_name="type")
    """The identifier type."""

    value = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="value")
    """The identifier string. Supports exact or partial matching."""


class SecurityAdvisoryOrder(sgqlc.types.Input):
    """Ordering options for security advisory connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(SecurityAdvisoryOrderField), graphql_name="field")
    """The field to order security advisories by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class SecurityVulnerabilityOrder(sgqlc.types.Input):
    """Ordering options for security vulnerability connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(SecurityVulnerabilityOrderField), graphql_name="field")
    """The field to order security vulnerabilities by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class SetEnterpriseIdentityProviderInput(sgqlc.types.Input):
    """Autogenerated input type of SetEnterpriseIdentityProvider"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "sso_url", "issuer", "idp_certificate", "signature_method", "digest_method", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set an identity provider."""

    sso_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="ssoUrl")
    """The URL endpoint for the identity provider's SAML SSO."""

    issuer = sgqlc.types.Field(String, graphql_name="issuer")
    """The Issuer Entity ID for the SAML identity provider"""

    idp_certificate = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="idpCertificate")
    """The x509 certificate used by the identity provider to sign
    assertions and responses.
    """

    signature_method = sgqlc.types.Field(sgqlc.types.non_null(SamlSignatureAlgorithm), graphql_name="signatureMethod")
    """The signature algorithm used to sign SAML requests for the
    identity provider.
    """

    digest_method = sgqlc.types.Field(sgqlc.types.non_null(SamlDigestAlgorithm), graphql_name="digestMethod")
    """The digest algorithm used to sign SAML requests for the identity
    provider.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class SetOrganizationInteractionLimitInput(sgqlc.types.Input):
    """Autogenerated input type of SetOrganizationInteractionLimit"""

    __schema__ = github_schema
    __field_names__ = ("organization_id", "limit", "expiry", "client_mutation_id")
    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization to set a limit for."""

    limit = sgqlc.types.Field(sgqlc.types.non_null(RepositoryInteractionLimit), graphql_name="limit")
    """The limit to set."""

    expiry = sgqlc.types.Field(RepositoryInteractionLimitExpiry, graphql_name="expiry")
    """When this limit should expire."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class SetRepositoryInteractionLimitInput(sgqlc.types.Input):
    """Autogenerated input type of SetRepositoryInteractionLimit"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "limit", "expiry", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the repository to set a limit for."""

    limit = sgqlc.types.Field(sgqlc.types.non_null(RepositoryInteractionLimit), graphql_name="limit")
    """The limit to set."""

    expiry = sgqlc.types.Field(RepositoryInteractionLimitExpiry, graphql_name="expiry")
    """When this limit should expire."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class SetUserInteractionLimitInput(sgqlc.types.Input):
    """Autogenerated input type of SetUserInteractionLimit"""

    __schema__ = github_schema
    __field_names__ = ("user_id", "limit", "expiry", "client_mutation_id")
    user_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="userId")
    """The ID of the user to set a limit for."""

    limit = sgqlc.types.Field(sgqlc.types.non_null(RepositoryInteractionLimit), graphql_name="limit")
    """The limit to set."""

    expiry = sgqlc.types.Field(RepositoryInteractionLimitExpiry, graphql_name="expiry")
    """When this limit should expire."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class SponsorOrder(sgqlc.types.Input):
    """Ordering options for connections to get sponsor entities for
    GitHub Sponsors.
    """

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(SponsorOrderField), graphql_name="field")
    """The field to order sponsor entities by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class SponsorableOrder(sgqlc.types.Input):
    """Ordering options for connections to get sponsorable entities for
    GitHub Sponsors.
    """

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(SponsorableOrderField), graphql_name="field")
    """The field to order sponsorable entities by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class SponsorsActivityOrder(sgqlc.types.Input):
    """Ordering options for GitHub Sponsors activity connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(SponsorsActivityOrderField), graphql_name="field")
    """The field to order activity by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class SponsorsTierOrder(sgqlc.types.Input):
    """Ordering options for Sponsors tiers connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(SponsorsTierOrderField), graphql_name="field")
    """The field to order tiers by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class SponsorshipNewsletterOrder(sgqlc.types.Input):
    """Ordering options for sponsorship newsletter connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(SponsorshipNewsletterOrderField), graphql_name="field")
    """The field to order sponsorship newsletters by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class SponsorshipOrder(sgqlc.types.Input):
    """Ordering options for sponsorship connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(SponsorshipOrderField), graphql_name="field")
    """The field to order sponsorship by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class StarOrder(sgqlc.types.Input):
    """Ways in which star connections can be ordered."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(StarOrderField), graphql_name="field")
    """The field in which to order nodes by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order nodes."""


class StartOrganizationMigrationInput(sgqlc.types.Input):
    """Autogenerated input type of StartOrganizationMigration"""

    __schema__ = github_schema
    __field_names__ = ("source_org_url", "target_org_name", "target_enterprise_id", "source_access_token", "client_mutation_id")
    source_org_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="sourceOrgUrl")
    """The URL of the organization to migrate."""

    target_org_name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="targetOrgName")
    """The name of the target organization."""

    target_enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="targetEnterpriseId")
    """The ID of the enterprise the target organization belongs to."""

    source_access_token = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="sourceAccessToken")
    """The migration source access token."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class StartRepositoryMigrationInput(sgqlc.types.Input):
    """Autogenerated input type of StartRepositoryMigration"""

    __schema__ = github_schema
    __field_names__ = (
        "source_id",
        "owner_id",
        "source_repository_url",
        "repository_name",
        "continue_on_error",
        "git_archive_url",
        "metadata_archive_url",
        "access_token",
        "github_pat",
        "skip_releases",
        "target_repo_visibility",
        "lock_source",
        "client_mutation_id",
    )
    source_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="sourceId")
    """The ID of the migration source."""

    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The ID of the organization that will own the imported repository."""

    source_repository_url = sgqlc.types.Field(URI, graphql_name="sourceRepositoryUrl")
    """The URL of the source repository."""

    repository_name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="repositoryName")
    """The name of the imported repository."""

    continue_on_error = sgqlc.types.Field(Boolean, graphql_name="continueOnError")
    """Whether to continue the migration on error. Defaults to `false`."""

    git_archive_url = sgqlc.types.Field(String, graphql_name="gitArchiveUrl")
    """The signed URL to access the user-uploaded git archive."""

    metadata_archive_url = sgqlc.types.Field(String, graphql_name="metadataArchiveUrl")
    """The signed URL to access the user-uploaded metadata archive."""

    access_token = sgqlc.types.Field(String, graphql_name="accessToken")
    """The migration source access token."""

    github_pat = sgqlc.types.Field(String, graphql_name="githubPat")
    """The GitHub personal access token of the user importing to the
    target repository.
    """

    skip_releases = sgqlc.types.Field(Boolean, graphql_name="skipReleases")
    """Whether to skip migrating releases for the repository."""

    target_repo_visibility = sgqlc.types.Field(String, graphql_name="targetRepoVisibility")
    """The visibility of the imported repository."""

    lock_source = sgqlc.types.Field(Boolean, graphql_name="lockSource")
    """Whether to lock the source repository."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class StatusCheckConfigurationInput(sgqlc.types.Input):
    """Required status check"""

    __schema__ = github_schema
    __field_names__ = ("context", "integration_id")
    context = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="context")
    """The status check context name that must be present on the commit."""

    integration_id = sgqlc.types.Field(Int, graphql_name="integrationId")
    """The optional integration ID that this status check must originate
    from.
    """


class SubmitPullRequestReviewInput(sgqlc.types.Input):
    """Autogenerated input type of SubmitPullRequestReview"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "pull_request_review_id", "event", "body", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(ID, graphql_name="pullRequestId")
    """The Pull Request ID to submit any pending reviews."""

    pull_request_review_id = sgqlc.types.Field(ID, graphql_name="pullRequestReviewId")
    """The Pull Request Review ID to submit."""

    event = sgqlc.types.Field(sgqlc.types.non_null(PullRequestReviewEvent), graphql_name="event")
    """The event to send to the Pull Request Review."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The text field to set on the Pull Request Review."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class TagNamePatternParametersInput(sgqlc.types.Input):
    """Parameters to be used for the tag_name_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(Boolean, graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class TeamDiscussionCommentOrder(sgqlc.types.Input):
    """Ways in which team discussion comment connections can be ordered."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(TeamDiscussionCommentOrderField), graphql_name="field")
    """The field by which to order nodes."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order nodes."""


class TeamDiscussionOrder(sgqlc.types.Input):
    """Ways in which team discussion connections can be ordered."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(TeamDiscussionOrderField), graphql_name="field")
    """The field by which to order nodes."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order nodes."""


class TeamMemberOrder(sgqlc.types.Input):
    """Ordering options for team member connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(TeamMemberOrderField), graphql_name="field")
    """The field to order team members by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class TeamOrder(sgqlc.types.Input):
    """Ways in which team connections can be ordered."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(TeamOrderField), graphql_name="field")
    """The field in which to order nodes by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order nodes."""


class TeamRepositoryOrder(sgqlc.types.Input):
    """Ordering options for team repository connections"""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(TeamRepositoryOrderField), graphql_name="field")
    """The field to order repositories by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class TransferEnterpriseOrganizationInput(sgqlc.types.Input):
    """Autogenerated input type of TransferEnterpriseOrganization"""

    __schema__ = github_schema
    __field_names__ = ("organization_id", "destination_enterprise_id", "client_mutation_id")
    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization to transfer."""

    destination_enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="destinationEnterpriseId")
    """The ID of the enterprise where the organization should be
    transferred.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class TransferIssueInput(sgqlc.types.Input):
    """Autogenerated input type of TransferIssue"""

    __schema__ = github_schema
    __field_names__ = ("issue_id", "repository_id", "create_labels_if_missing", "client_mutation_id")
    issue_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="issueId")
    """The Node ID of the issue to be transferred"""

    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the repository the issue should be transferred to"""

    create_labels_if_missing = sgqlc.types.Field(Boolean, graphql_name="createLabelsIfMissing")
    """Whether to create labels if they don't exist in the target
    repository (matched by name)
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnarchiveProjectV2ItemInput(sgqlc.types.Input):
    """Autogenerated input type of UnarchiveProjectV2Item"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "item_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to archive the item from."""

    item_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="itemId")
    """The ID of the ProjectV2Item to unarchive."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnarchiveRepositoryInput(sgqlc.types.Input):
    """Autogenerated input type of UnarchiveRepository"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the repository to unarchive."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnfollowOrganizationInput(sgqlc.types.Input):
    """Autogenerated input type of UnfollowOrganization"""

    __schema__ = github_schema
    __field_names__ = ("organization_id", "client_mutation_id")
    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """ID of the organization to unfollow."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnfollowUserInput(sgqlc.types.Input):
    """Autogenerated input type of UnfollowUser"""

    __schema__ = github_schema
    __field_names__ = ("user_id", "client_mutation_id")
    user_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="userId")
    """ID of the user to unfollow."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnlinkProjectV2FromRepositoryInput(sgqlc.types.Input):
    """Autogenerated input type of UnlinkProjectV2FromRepository"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "repository_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the project to unlink from the repository."""

    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the repository to unlink from the project."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnlinkProjectV2FromTeamInput(sgqlc.types.Input):
    """Autogenerated input type of UnlinkProjectV2FromTeam"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "team_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the project to unlink from the team."""

    team_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="teamId")
    """The ID of the team to unlink from the project."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnlinkRepositoryFromProjectInput(sgqlc.types.Input):
    """Autogenerated input type of UnlinkRepositoryFromProject"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "repository_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project linked to the Repository."""

    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the Repository linked to the Project."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnlockLockableInput(sgqlc.types.Input):
    """Autogenerated input type of UnlockLockable"""

    __schema__ = github_schema
    __field_names__ = ("lockable_id", "client_mutation_id")
    lockable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="lockableId")
    """ID of the item to be unlocked."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnmarkDiscussionCommentAsAnswerInput(sgqlc.types.Input):
    """Autogenerated input type of UnmarkDiscussionCommentAsAnswer"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The Node ID of the discussion comment to unmark as an answer."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnmarkFileAsViewedInput(sgqlc.types.Input):
    """Autogenerated input type of UnmarkFileAsViewed"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "path", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """The Node ID of the pull request."""

    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """The path of the file to mark as unviewed"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnmarkIssueAsDuplicateInput(sgqlc.types.Input):
    """Autogenerated input type of UnmarkIssueAsDuplicate"""

    __schema__ = github_schema
    __field_names__ = ("duplicate_id", "canonical_id", "client_mutation_id")
    duplicate_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="duplicateId")
    """ID of the issue or pull request currently marked as a duplicate."""

    canonical_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="canonicalId")
    """ID of the issue or pull request currently considered
    canonical/authoritative/original.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnmarkProjectV2AsTemplateInput(sgqlc.types.Input):
    """Autogenerated input type of UnmarkProjectV2AsTemplate"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to unmark as a template."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnminimizeCommentInput(sgqlc.types.Input):
    """Autogenerated input type of UnminimizeComment"""

    __schema__ = github_schema
    __field_names__ = ("subject_id", "client_mutation_id")
    subject_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="subjectId")
    """The Node ID of the subject to modify."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnpinIssueInput(sgqlc.types.Input):
    """Autogenerated input type of UnpinIssue"""

    __schema__ = github_schema
    __field_names__ = ("issue_id", "client_mutation_id")
    issue_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="issueId")
    """The ID of the issue to be unpinned"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UnresolveReviewThreadInput(sgqlc.types.Input):
    """Autogenerated input type of UnresolveReviewThread"""

    __schema__ = github_schema
    __field_names__ = ("thread_id", "client_mutation_id")
    thread_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="threadId")
    """The ID of the thread to unresolve"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateBranchProtectionRuleInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateBranchProtectionRule"""

    __schema__ = github_schema
    __field_names__ = (
        "branch_protection_rule_id",
        "pattern",
        "requires_approving_reviews",
        "required_approving_review_count",
        "requires_commit_signatures",
        "requires_linear_history",
        "blocks_creations",
        "allows_force_pushes",
        "allows_deletions",
        "is_admin_enforced",
        "requires_status_checks",
        "requires_strict_status_checks",
        "requires_code_owner_reviews",
        "dismisses_stale_reviews",
        "restricts_review_dismissals",
        "review_dismissal_actor_ids",
        "bypass_pull_request_actor_ids",
        "bypass_force_push_actor_ids",
        "restricts_pushes",
        "push_actor_ids",
        "required_status_check_contexts",
        "required_status_checks",
        "requires_deployments",
        "required_deployment_environments",
        "requires_conversation_resolution",
        "require_last_push_approval",
        "lock_branch",
        "lock_allows_fetch_and_merge",
        "client_mutation_id",
    )
    branch_protection_rule_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="branchProtectionRuleId")
    """The global relay id of the branch protection rule to be updated."""

    pattern = sgqlc.types.Field(String, graphql_name="pattern")
    """The glob-like pattern used to determine matching branches."""

    requires_approving_reviews = sgqlc.types.Field(Boolean, graphql_name="requiresApprovingReviews")
    """Are approving reviews required to update matching branches."""

    required_approving_review_count = sgqlc.types.Field(Int, graphql_name="requiredApprovingReviewCount")
    """Number of approving reviews required to update matching branches."""

    requires_commit_signatures = sgqlc.types.Field(Boolean, graphql_name="requiresCommitSignatures")
    """Are commits required to be signed."""

    requires_linear_history = sgqlc.types.Field(Boolean, graphql_name="requiresLinearHistory")
    """Are merge commits prohibited from being pushed to this branch."""

    blocks_creations = sgqlc.types.Field(Boolean, graphql_name="blocksCreations")
    """Is branch creation a protected operation."""

    allows_force_pushes = sgqlc.types.Field(Boolean, graphql_name="allowsForcePushes")
    """Are force pushes allowed on this branch."""

    allows_deletions = sgqlc.types.Field(Boolean, graphql_name="allowsDeletions")
    """Can this branch be deleted."""

    is_admin_enforced = sgqlc.types.Field(Boolean, graphql_name="isAdminEnforced")
    """Can admins overwrite branch protection."""

    requires_status_checks = sgqlc.types.Field(Boolean, graphql_name="requiresStatusChecks")
    """Are status checks required to update matching branches."""

    requires_strict_status_checks = sgqlc.types.Field(Boolean, graphql_name="requiresStrictStatusChecks")
    """Are branches required to be up to date before merging."""

    requires_code_owner_reviews = sgqlc.types.Field(Boolean, graphql_name="requiresCodeOwnerReviews")
    """Are reviews from code owners required to update matching branches."""

    dismisses_stale_reviews = sgqlc.types.Field(Boolean, graphql_name="dismissesStaleReviews")
    """Will new commits pushed to matching branches dismiss pull request
    review approvals.
    """

    restricts_review_dismissals = sgqlc.types.Field(Boolean, graphql_name="restrictsReviewDismissals")
    """Is dismissal of pull request reviews restricted."""

    review_dismissal_actor_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="reviewDismissalActorIds")
    """A list of User, Team, or App IDs allowed to dismiss reviews on
    pull requests targeting matching branches.
    """

    bypass_pull_request_actor_ids = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="bypassPullRequestActorIds"
    )
    """A list of User, Team, or App IDs allowed to bypass pull requests
    targeting matching branches.
    """

    bypass_force_push_actor_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="bypassForcePushActorIds")
    """A list of User, Team, or App IDs allowed to bypass force push
    targeting matching branches.
    """

    restricts_pushes = sgqlc.types.Field(Boolean, graphql_name="restrictsPushes")
    """Is pushing to matching branches restricted."""

    push_actor_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="pushActorIds")
    """A list of User, Team, or App IDs allowed to push to matching
    branches.
    """

    required_status_check_contexts = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="requiredStatusCheckContexts"
    )
    """List of required status check contexts that must pass for commits
    to be accepted to matching branches.
    """

    required_status_checks = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null(RequiredStatusCheckInput)), graphql_name="requiredStatusChecks"
    )
    """The list of required status checks"""

    requires_deployments = sgqlc.types.Field(Boolean, graphql_name="requiresDeployments")
    """Are successful deployments required before merging."""

    required_deployment_environments = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="requiredDeploymentEnvironments"
    )
    """The list of required deployment environments"""

    requires_conversation_resolution = sgqlc.types.Field(Boolean, graphql_name="requiresConversationResolution")
    """Are conversations required to be resolved before merging."""

    require_last_push_approval = sgqlc.types.Field(Boolean, graphql_name="requireLastPushApproval")
    """Whether the most recent push must be approved by someone other
    than the person who pushed it
    """

    lock_branch = sgqlc.types.Field(Boolean, graphql_name="lockBranch")
    """Whether to set the branch as read-only. If this is true, users
    will not be able to push to the branch.
    """

    lock_allows_fetch_and_merge = sgqlc.types.Field(Boolean, graphql_name="lockAllowsFetchAndMerge")
    """Whether users can pull changes from upstream when the branch is
    locked. Set to `true` to allow fork syncing. Set to `false` to
    prevent fork syncing.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateCheckRunInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateCheckRun"""

    __schema__ = github_schema
    __field_names__ = (
        "repository_id",
        "check_run_id",
        "name",
        "details_url",
        "external_id",
        "status",
        "started_at",
        "conclusion",
        "completed_at",
        "output",
        "actions",
        "client_mutation_id",
    )
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The node ID of the repository."""

    check_run_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="checkRunId")
    """The node of the check."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """The name of the check."""

    details_url = sgqlc.types.Field(URI, graphql_name="detailsUrl")
    """The URL of the integrator's site that has the full details of the
    check.
    """

    external_id = sgqlc.types.Field(String, graphql_name="externalId")
    """A reference for the run on the integrator's system."""

    status = sgqlc.types.Field(RequestableCheckStatusState, graphql_name="status")
    """The current status."""

    started_at = sgqlc.types.Field(DateTime, graphql_name="startedAt")
    """The time that the check run began."""

    conclusion = sgqlc.types.Field(CheckConclusionState, graphql_name="conclusion")
    """The final conclusion of the check."""

    completed_at = sgqlc.types.Field(DateTime, graphql_name="completedAt")
    """The time that the check run finished."""

    output = sgqlc.types.Field(CheckRunOutput, graphql_name="output")
    """Descriptive details about the run."""

    actions = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(CheckRunAction)), graphql_name="actions")
    """Possible further actions the integrator can perform, which a user
    may trigger.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateCheckSuitePreferencesInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateCheckSuitePreferences"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "auto_trigger_preferences", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the repository."""

    auto_trigger_preferences = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(CheckSuiteAutoTriggerPreference))),
        graphql_name="autoTriggerPreferences",
    )
    """The check suite preferences to modify."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateDiscussionCommentInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("comment_id", "body", "client_mutation_id")
    comment_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="commentId")
    """The Node ID of the discussion comment to update."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The new contents of the comment body."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateDiscussionInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("discussion_id", "title", "body", "category_id", "client_mutation_id")
    discussion_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="discussionId")
    """The Node ID of the discussion to update."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """The new discussion title."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The new contents of the discussion body."""

    category_id = sgqlc.types.Field(ID, graphql_name="categoryId")
    """The Node ID of a discussion category within the same repository to
    change this discussion to.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseAdministratorRoleInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateEnterpriseAdministratorRole"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "login", "role", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the Enterprise which the admin belongs to."""

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The login of a administrator whose role is being changed."""

    role = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseAdministratorRole), graphql_name="role")
    """The new role for the Enterprise administrator."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseAllowPrivateRepositoryForkingSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseAllowPrivateRepositoryForkingSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "policy_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the allow private
    repository forking setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the allow private repository forking setting on the
    enterprise.
    """

    policy_value = sgqlc.types.Field(EnterpriseAllowPrivateRepositoryForkingPolicyValue, graphql_name="policyValue")
    """The value for the allow private repository forking policy on the
    enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseDefaultRepositoryPermissionSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseDefaultRepositoryPermissionSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the base repository
    permission setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseDefaultRepositoryPermissionSettingValue), graphql_name="settingValue")
    """The value for the base repository permission setting on the
    enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseMembersCanChangeRepositoryVisibilitySettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseMembersCanChangeRepositoryVisibilitySetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the members can change
    repository visibility setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the members can change repository visibility setting
    on the enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseMembersCanCreateRepositoriesSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseMembersCanCreateRepositoriesSetting
    """

    __schema__ = github_schema
    __field_names__ = (
        "enterprise_id",
        "setting_value",
        "members_can_create_repositories_policy_enabled",
        "members_can_create_public_repositories",
        "members_can_create_private_repositories",
        "members_can_create_internal_repositories",
        "client_mutation_id",
    )
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the members can create
    repositories setting.
    """

    setting_value = sgqlc.types.Field(EnterpriseMembersCanCreateRepositoriesSettingValue, graphql_name="settingValue")
    """Value for the members can create repositories setting on the
    enterprise. This or the granular public/private/internal allowed
    fields (but not both) must be provided.
    """

    members_can_create_repositories_policy_enabled = sgqlc.types.Field(Boolean, graphql_name="membersCanCreateRepositoriesPolicyEnabled")
    """When false, allow member organizations to set their own repository
    creation member privileges.
    """

    members_can_create_public_repositories = sgqlc.types.Field(Boolean, graphql_name="membersCanCreatePublicRepositories")
    """Allow members to create public repositories. Defaults to current
    value.
    """

    members_can_create_private_repositories = sgqlc.types.Field(Boolean, graphql_name="membersCanCreatePrivateRepositories")
    """Allow members to create private repositories. Defaults to current
    value.
    """

    members_can_create_internal_repositories = sgqlc.types.Field(Boolean, graphql_name="membersCanCreateInternalRepositories")
    """Allow members to create internal repositories. Defaults to current
    value.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseMembersCanDeleteIssuesSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseMembersCanDeleteIssuesSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the members can delete
    issues setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the members can delete issues setting on the
    enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseMembersCanDeleteRepositoriesSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseMembersCanDeleteRepositoriesSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the members can delete
    repositories setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the members can delete repositories setting on the
    enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseMembersCanInviteCollaboratorsSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseMembersCanInviteCollaboratorsSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the members can invite
    collaborators setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the members can invite collaborators setting on the
    enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseMembersCanMakePurchasesSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseMembersCanMakePurchasesSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the members can make
    purchases setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseMembersCanMakePurchasesSettingValue), graphql_name="settingValue")
    """The value for the members can make purchases setting on the
    enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseMembersCanUpdateProtectedBranchesSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseMembersCanUpdateProtectedBranchesSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the members can update
    protected branches setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the members can update protected branches setting on
    the enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseMembersCanViewDependencyInsightsSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseMembersCanViewDependencyInsightsSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the members can view
    dependency insights setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the members can view dependency insights setting on
    the enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseOrganizationProjectsSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseOrganizationProjectsSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the organization projects
    setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the organization projects setting on the enterprise."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseOwnerOrganizationRoleInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateEnterpriseOwnerOrganizationRole"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "organization_id", "organization_role", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the Enterprise which the owner belongs to."""

    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization for membership change."""

    organization_role = sgqlc.types.Field(sgqlc.types.non_null(RoleInOrganization), graphql_name="organizationRole")
    """The role to assume in the organization."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseProfileInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateEnterpriseProfile"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "name", "description", "website_url", "location", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The Enterprise ID to update."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """The name of the enterprise."""

    description = sgqlc.types.Field(String, graphql_name="description")
    """The description of the enterprise."""

    website_url = sgqlc.types.Field(String, graphql_name="websiteUrl")
    """The URL of the enterprise's website."""

    location = sgqlc.types.Field(String, graphql_name="location")
    """The location of the enterprise."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseRepositoryProjectsSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseRepositoryProjectsSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the repository projects
    setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the repository projects setting on the enterprise."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseTeamDiscussionsSettingInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateEnterpriseTeamDiscussionsSetting"""

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the team discussions
    setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="settingValue")
    """The value for the team discussions setting on the enterprise."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnterpriseTwoFactorAuthenticationRequiredSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateEnterpriseTwoFactorAuthenticationRequiredSetting
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_id", "setting_value", "client_mutation_id")
    enterprise_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="enterpriseId")
    """The ID of the enterprise on which to set the two factor
    authentication required setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseEnabledSettingValue), graphql_name="settingValue")
    """The value for the two factor authentication required setting on
    the enterprise.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateEnvironmentInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateEnvironment"""

    __schema__ = github_schema
    __field_names__ = ("environment_id", "wait_timer", "reviewers", "client_mutation_id")
    environment_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="environmentId")
    """The node ID of the environment."""

    wait_timer = sgqlc.types.Field(Int, graphql_name="waitTimer")
    """The wait timer in minutes."""

    reviewers = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="reviewers")
    """The ids of users or teams that can approve deployments to this
    environment
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateIpAllowListEnabledSettingInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateIpAllowListEnabledSetting"""

    __schema__ = github_schema
    __field_names__ = ("owner_id", "setting_value", "client_mutation_id")
    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The ID of the owner on which to set the IP allow list enabled
    setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(IpAllowListEnabledSettingValue), graphql_name="settingValue")
    """The value for the IP allow list enabled setting."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateIpAllowListEntryInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateIpAllowListEntry"""

    __schema__ = github_schema
    __field_names__ = ("ip_allow_list_entry_id", "allow_list_value", "name", "is_active", "client_mutation_id")
    ip_allow_list_entry_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ipAllowListEntryId")
    """The ID of the IP allow list entry to update."""

    allow_list_value = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="allowListValue")
    """An IP address or range of addresses in CIDR notation."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """An optional name for the IP allow list entry."""

    is_active = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isActive")
    """Whether the IP allow list entry is active when an IP allow list is
    enabled.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateIpAllowListForInstalledAppsEnabledSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateIpAllowListForInstalledAppsEnabledSetting
    """

    __schema__ = github_schema
    __field_names__ = ("owner_id", "setting_value", "client_mutation_id")
    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The ID of the owner."""

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(IpAllowListForInstalledAppsEnabledSettingValue), graphql_name="settingValue")
    """The value for the IP allow list configuration for installed GitHub
    Apps setting.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateIssueCommentInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateIssueComment"""

    __schema__ = github_schema
    __field_names__ = ("id", "body", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the IssueComment to modify."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The updated text of the comment."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateIssueInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateIssue"""

    __schema__ = github_schema
    __field_names__ = ("id", "title", "body", "assignee_ids", "milestone_id", "label_ids", "state", "project_ids", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the Issue to modify."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """The title for the issue."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The body for the issue description."""

    assignee_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="assigneeIds")
    """An array of Node IDs of users for this issue."""

    milestone_id = sgqlc.types.Field(ID, graphql_name="milestoneId")
    """The Node ID of the milestone for this issue."""

    label_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="labelIds")
    """An array of Node IDs of labels for this issue."""

    state = sgqlc.types.Field(IssueState, graphql_name="state")
    """The desired issue state."""

    project_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="projectIds")
    """An array of Node IDs for projects associated with this issue."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateNotificationRestrictionSettingInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateNotificationRestrictionSetting"""

    __schema__ = github_schema
    __field_names__ = ("owner_id", "setting_value", "client_mutation_id")
    owner_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="ownerId")
    """The ID of the owner on which to set the restrict notifications
    setting.
    """

    setting_value = sgqlc.types.Field(sgqlc.types.non_null(NotificationRestrictionSettingValue), graphql_name="settingValue")
    """The value for the restrict notifications setting."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateOrganizationAllowPrivateRepositoryForkingSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateOrganizationAllowPrivateRepositoryForkingSetting
    """

    __schema__ = github_schema
    __field_names__ = ("organization_id", "forking_enabled", "client_mutation_id")
    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization on which to set the allow private
    repository forking setting.
    """

    forking_enabled = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="forkingEnabled")
    """Enable forking of private repositories in the organization?"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateOrganizationWebCommitSignoffSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateOrganizationWebCommitSignoffSetting
    """

    __schema__ = github_schema
    __field_names__ = ("organization_id", "web_commit_signoff_required", "client_mutation_id")
    organization_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="organizationId")
    """The ID of the organization on which to set the web commit signoff
    setting.
    """

    web_commit_signoff_required = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="webCommitSignoffRequired")
    """Enable signoff on web-based commits for repositories in the
    organization?
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateParametersInput(sgqlc.types.Input):
    """Only allow users with bypass permission to update matching refs."""

    __schema__ = github_schema
    __field_names__ = ("update_allows_fetch_and_merge",)
    update_allows_fetch_and_merge = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="updateAllowsFetchAndMerge")
    """Branch can pull changes from its upstream repository"""


class UpdateProjectCardInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateProjectCard"""

    __schema__ = github_schema
    __field_names__ = ("project_card_id", "is_archived", "note", "client_mutation_id")
    project_card_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectCardId")
    """The ProjectCard ID to update."""

    is_archived = sgqlc.types.Field(Boolean, graphql_name="isArchived")
    """Whether or not the ProjectCard should be archived"""

    note = sgqlc.types.Field(String, graphql_name="note")
    """The note of ProjectCard."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateProjectColumnInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateProjectColumn"""

    __schema__ = github_schema
    __field_names__ = ("project_column_id", "name", "client_mutation_id")
    project_column_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectColumnId")
    """The ProjectColumn ID to update."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of project column."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateProjectInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateProject"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "name", "body", "state", "public", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The Project ID to update."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """The name of project."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The description of project."""

    state = sgqlc.types.Field(ProjectState, graphql_name="state")
    """Whether the project is open or closed."""

    public = sgqlc.types.Field(Boolean, graphql_name="public")
    """Whether the project is public or not."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateProjectV2CollaboratorsInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateProjectV2Collaborators"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "collaborators", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the project to update the collaborators for."""

    collaborators = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ProjectV2Collaborator))), graphql_name="collaborators"
    )
    """The collaborators to update."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateProjectV2DraftIssueInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateProjectV2DraftIssue"""

    __schema__ = github_schema
    __field_names__ = ("draft_issue_id", "title", "body", "assignee_ids", "client_mutation_id")
    draft_issue_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="draftIssueId")
    """The ID of the draft issue to update."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """The title of the draft issue."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The body of the draft issue."""

    assignee_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="assigneeIds")
    """The IDs of the assignees of the draft issue."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateProjectV2Input(sgqlc.types.Input):
    """Autogenerated input type of UpdateProjectV2"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "title", "short_description", "readme", "closed", "public", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project to update."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """Set the title of the project."""

    short_description = sgqlc.types.Field(String, graphql_name="shortDescription")
    """Set the short description of the project."""

    readme = sgqlc.types.Field(String, graphql_name="readme")
    """Set the readme description of the project."""

    closed = sgqlc.types.Field(Boolean, graphql_name="closed")
    """Set the project to closed or open."""

    public = sgqlc.types.Field(Boolean, graphql_name="public")
    """Set the project to public or private."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateProjectV2ItemFieldValueInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateProjectV2ItemFieldValue"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "item_id", "field_id", "value", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project."""

    item_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="itemId")
    """The ID of the item to be updated."""

    field_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="fieldId")
    """The ID of the field to be updated."""

    value = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2FieldValue), graphql_name="value")
    """The value which will be set on the field."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateProjectV2ItemPositionInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateProjectV2ItemPosition"""

    __schema__ = github_schema
    __field_names__ = ("project_id", "item_id", "after_id", "client_mutation_id")
    project_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="projectId")
    """The ID of the Project."""

    item_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="itemId")
    """The ID of the item to be moved."""

    after_id = sgqlc.types.Field(ID, graphql_name="afterId")
    """The ID of the item to position this item after. If omitted or set
    to null the item will be moved to top.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdatePullRequestBranchInput(sgqlc.types.Input):
    """Autogenerated input type of UpdatePullRequestBranch"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_id", "expected_head_oid", "client_mutation_id")
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """The Node ID of the pull request."""

    expected_head_oid = sgqlc.types.Field(GitObjectID, graphql_name="expectedHeadOid")
    """The head ref oid for the upstream branch."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdatePullRequestInput(sgqlc.types.Input):
    """Autogenerated input type of UpdatePullRequest"""

    __schema__ = github_schema
    __field_names__ = (
        "pull_request_id",
        "base_ref_name",
        "title",
        "body",
        "state",
        "maintainer_can_modify",
        "assignee_ids",
        "milestone_id",
        "label_ids",
        "project_ids",
        "client_mutation_id",
    )
    pull_request_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestId")
    """The Node ID of the pull request."""

    base_ref_name = sgqlc.types.Field(String, graphql_name="baseRefName")
    """The name of the branch you want your changes pulled into. This
    should be an existing branch on the current repository.
    """

    title = sgqlc.types.Field(String, graphql_name="title")
    """The title of the pull request."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The contents of the pull request."""

    state = sgqlc.types.Field(PullRequestUpdateState, graphql_name="state")
    """The target state of the pull request."""

    maintainer_can_modify = sgqlc.types.Field(Boolean, graphql_name="maintainerCanModify")
    """Indicates whether maintainers can modify the pull request."""

    assignee_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="assigneeIds")
    """An array of Node IDs of users for this pull request."""

    milestone_id = sgqlc.types.Field(ID, graphql_name="milestoneId")
    """The Node ID of the milestone for this pull request."""

    label_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="labelIds")
    """An array of Node IDs of labels for this pull request."""

    project_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="projectIds")
    """An array of Node IDs for projects associated with this pull
    request.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdatePullRequestReviewCommentInput(sgqlc.types.Input):
    """Autogenerated input type of UpdatePullRequestReviewComment"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_review_comment_id", "body", "client_mutation_id")
    pull_request_review_comment_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestReviewCommentId")
    """The Node ID of the comment to modify."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The text of the comment."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdatePullRequestReviewInput(sgqlc.types.Input):
    """Autogenerated input type of UpdatePullRequestReview"""

    __schema__ = github_schema
    __field_names__ = ("pull_request_review_id", "body", "client_mutation_id")
    pull_request_review_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="pullRequestReviewId")
    """The Node ID of the pull request review to modify."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The contents of the pull request review body."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateRefInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateRef"""

    __schema__ = github_schema
    __field_names__ = ("ref_id", "oid", "force", "client_mutation_id")
    ref_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="refId")
    """The Node ID of the Ref to be updated."""

    oid = sgqlc.types.Field(sgqlc.types.non_null(GitObjectID), graphql_name="oid")
    """The GitObjectID that the Ref shall be updated to target."""

    force = sgqlc.types.Field(Boolean, graphql_name="force")
    """Permit updates of branch Refs that are not fast-forwards?"""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateRepositoryInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateRepository"""

    __schema__ = github_schema
    __field_names__ = (
        "repository_id",
        "name",
        "description",
        "template",
        "homepage_url",
        "has_wiki_enabled",
        "has_issues_enabled",
        "has_projects_enabled",
        "has_discussions_enabled",
        "client_mutation_id",
    )
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the repository to update."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """The new name of the repository."""

    description = sgqlc.types.Field(String, graphql_name="description")
    """A new description for the repository. Pass an empty string to
    erase the existing description.
    """

    template = sgqlc.types.Field(Boolean, graphql_name="template")
    """Whether this repository should be marked as a template such that
    anyone who can access it can create new repositories with the same
    files and directory structure.
    """

    homepage_url = sgqlc.types.Field(URI, graphql_name="homepageUrl")
    """The URL for a web page about this repository. Pass an empty string
    to erase the existing URL.
    """

    has_wiki_enabled = sgqlc.types.Field(Boolean, graphql_name="hasWikiEnabled")
    """Indicates if the repository should have the wiki feature enabled."""

    has_issues_enabled = sgqlc.types.Field(Boolean, graphql_name="hasIssuesEnabled")
    """Indicates if the repository should have the issues feature
    enabled.
    """

    has_projects_enabled = sgqlc.types.Field(Boolean, graphql_name="hasProjectsEnabled")
    """Indicates if the repository should have the project boards feature
    enabled.
    """

    has_discussions_enabled = sgqlc.types.Field(Boolean, graphql_name="hasDiscussionsEnabled")
    """Indicates if the repository should have the discussions feature
    enabled.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateRepositoryRulesetInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateRepositoryRuleset"""

    __schema__ = github_schema
    __field_names__ = (
        "repository_ruleset_id",
        "name",
        "target",
        "rules",
        "conditions",
        "enforcement",
        "bypass_mode",
        "bypass_actor_ids",
        "client_mutation_id",
    )
    repository_ruleset_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryRulesetId")
    """The global relay id of the repository ruleset to be updated."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """The name of the ruleset."""

    target = sgqlc.types.Field(RepositoryRulesetTarget, graphql_name="target")
    """The target of the ruleset."""

    rules = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(RepositoryRuleInput)), graphql_name="rules")
    """The list of rules for this ruleset"""

    conditions = sgqlc.types.Field(RepositoryRuleConditionsInput, graphql_name="conditions")
    """The list of conditions for this ruleset"""

    enforcement = sgqlc.types.Field(RuleEnforcement, graphql_name="enforcement")
    """The enforcement level for this ruleset"""

    bypass_mode = sgqlc.types.Field(RuleBypassMode, graphql_name="bypassMode")
    """The bypass mode for this ruleset"""

    bypass_actor_ids = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ID)), graphql_name="bypassActorIds")
    """A list of Team or App IDs allowed to bypass rules in this ruleset."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateRepositoryWebCommitSignoffSettingInput(sgqlc.types.Input):
    """Autogenerated input type of
    UpdateRepositoryWebCommitSignoffSetting
    """

    __schema__ = github_schema
    __field_names__ = ("repository_id", "web_commit_signoff_required", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The ID of the repository to update."""

    web_commit_signoff_required = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="webCommitSignoffRequired")
    """Indicates if the repository should require signoff on web-based
    commits.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateSponsorshipPreferencesInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateSponsorshipPreferences"""

    __schema__ = github_schema
    __field_names__ = (
        "sponsor_id",
        "sponsor_login",
        "sponsorable_id",
        "sponsorable_login",
        "receive_emails",
        "privacy_level",
        "client_mutation_id",
    )
    sponsor_id = sgqlc.types.Field(ID, graphql_name="sponsorId")
    """The ID of the user or organization who is acting as the sponsor,
    paying for the sponsorship. Required if sponsorLogin is not given.
    """

    sponsor_login = sgqlc.types.Field(String, graphql_name="sponsorLogin")
    """The username of the user or organization who is acting as the
    sponsor, paying for the sponsorship. Required if sponsorId is not
    given.
    """

    sponsorable_id = sgqlc.types.Field(ID, graphql_name="sponsorableId")
    """The ID of the user or organization who is receiving the
    sponsorship. Required if sponsorableLogin is not given.
    """

    sponsorable_login = sgqlc.types.Field(String, graphql_name="sponsorableLogin")
    """The username of the user or organization who is receiving the
    sponsorship. Required if sponsorableId is not given.
    """

    receive_emails = sgqlc.types.Field(Boolean, graphql_name="receiveEmails")
    """Whether the sponsor should receive email updates from the
    sponsorable.
    """

    privacy_level = sgqlc.types.Field(SponsorshipPrivacy, graphql_name="privacyLevel")
    """Specify whether others should be able to see that the sponsor is
    sponsoring the sponsorable. Public visibility still does not
    reveal which tier is used.
    """

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateSubscriptionInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateSubscription"""

    __schema__ = github_schema
    __field_names__ = ("subscribable_id", "state", "client_mutation_id")
    subscribable_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="subscribableId")
    """The Node ID of the subscribable object to modify."""

    state = sgqlc.types.Field(sgqlc.types.non_null(SubscriptionState), graphql_name="state")
    """The new state of the subscription."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateTeamDiscussionCommentInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateTeamDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("id", "body", "body_version", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the comment to modify."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The updated text of the comment."""

    body_version = sgqlc.types.Field(String, graphql_name="bodyVersion")
    """The current version of the body content."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateTeamDiscussionInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateTeamDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("id", "title", "body", "body_version", "pinned", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The Node ID of the discussion to modify."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """The updated title of the discussion."""

    body = sgqlc.types.Field(String, graphql_name="body")
    """The updated text of the discussion."""

    body_version = sgqlc.types.Field(String, graphql_name="bodyVersion")
    """The current version of the body content. If provided, this update
    operation will be rejected if the given version does not match the
    latest version on the server.
    """

    pinned = sgqlc.types.Field(Boolean, graphql_name="pinned")
    """If provided, sets the pinned state of the updated discussion."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateTeamsRepositoryInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateTeamsRepository"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "team_ids", "permission", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """Repository ID being granted access to."""

    team_ids = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ID))), graphql_name="teamIds")
    """A list of teams being granted access. Limit: 10"""

    permission = sgqlc.types.Field(sgqlc.types.non_null(RepositoryPermission), graphql_name="permission")
    """Permission that should be granted to the teams."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UpdateTopicsInput(sgqlc.types.Input):
    """Autogenerated input type of UpdateTopics"""

    __schema__ = github_schema
    __field_names__ = ("repository_id", "topic_names", "client_mutation_id")
    repository_id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="repositoryId")
    """The Node ID of the repository."""

    topic_names = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="topicNames")
    """An array of topic names."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class UserStatusOrder(sgqlc.types.Input):
    """Ordering options for user status connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(UserStatusOrderField), graphql_name="field")
    """The field to order user statuses by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class VerifiableDomainOrder(sgqlc.types.Input):
    """Ordering options for verifiable domain connections."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(VerifiableDomainOrderField), graphql_name="field")
    """The field to order verifiable domains by."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The ordering direction."""


class VerifyVerifiableDomainInput(sgqlc.types.Input):
    """Autogenerated input type of VerifyVerifiableDomain"""

    __schema__ = github_schema
    __field_names__ = ("id", "client_mutation_id")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """The ID of the verifiable domain to verify."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class WorkflowRunOrder(sgqlc.types.Input):
    """Ways in which lists of workflow runs can be ordered upon return."""

    __schema__ = github_schema
    __field_names__ = ("field", "direction")
    field = sgqlc.types.Field(sgqlc.types.non_null(WorkflowRunOrderField), graphql_name="field")
    """The field by which to order workflows."""

    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction in which to order workflow runs by the specified
    field.
    """


########################################################################
# Output Objects and Interfaces
########################################################################
class Actor(sgqlc.types.Interface):
    """Represents an object which can take actions on GitHub. Typically a
    User or Bot.
    """

    __schema__ = github_schema
    __field_names__ = ("avatar_url", "login", "resource_path", "url")
    avatar_url = sgqlc.types.Field(
        sgqlc.types.non_null(URI),
        graphql_name="avatarUrl",
        args=sgqlc.types.ArgDict((("size", sgqlc.types.Arg(Int, graphql_name="size", default=None)),)),
    )
    """A URL pointing to the actor's public avatar.

    Arguments:

    * `size` (`Int`): The size of the resulting square image.
    """

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The username of the actor."""

    resource_path = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="resourcePath")
    """The HTTP path for this actor."""

    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """The HTTP URL for this actor."""


class AnnouncementBanner(sgqlc.types.Interface):
    """Represents an announcement banner."""

    __schema__ = github_schema
    __field_names__ = ("announcement", "announcement_expires_at", "announcement_user_dismissible")
    announcement = sgqlc.types.Field(String, graphql_name="announcement")
    """The text of the announcement"""

    announcement_expires_at = sgqlc.types.Field(DateTime, graphql_name="announcementExpiresAt")
    """The expiration date of the announcement, if any"""

    announcement_user_dismissible = sgqlc.types.Field(Boolean, graphql_name="announcementUserDismissible")
    """Whether the announcement can be dismissed by the user"""


class Assignable(sgqlc.types.Interface):
    """An object that can have users assigned to it."""

    __schema__ = github_schema
    __field_names__ = ("assignees",)
    assignees = sgqlc.types.Field(
        sgqlc.types.non_null("UserConnection"),
        graphql_name="assignees",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of Users assigned to this object.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class AuditEntry(sgqlc.types.Interface):
    """An entry in the audit log."""

    __schema__ = github_schema
    __field_names__ = (
        "action",
        "actor",
        "actor_ip",
        "actor_location",
        "actor_login",
        "actor_resource_path",
        "actor_url",
        "created_at",
        "operation_type",
        "user",
        "user_login",
        "user_resource_path",
        "user_url",
    )
    action = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="action")
    """The action name"""

    actor = sgqlc.types.Field("AuditEntryActor", graphql_name="actor")
    """The user who initiated the action"""

    actor_ip = sgqlc.types.Field(String, graphql_name="actorIp")
    """The IP address of the actor"""

    actor_location = sgqlc.types.Field("ActorLocation", graphql_name="actorLocation")
    """A readable representation of the actor's location"""

    actor_login = sgqlc.types.Field(String, graphql_name="actorLogin")
    """The username of the user who initiated the action"""

    actor_resource_path = sgqlc.types.Field(URI, graphql_name="actorResourcePath")
    """The HTTP path for the actor."""

    actor_url = sgqlc.types.Field(URI, graphql_name="actorUrl")
    """The HTTP URL for the actor."""

    created_at = sgqlc.types.Field(sgqlc.types.non_null(PreciseDateTime), graphql_name="createdAt")
    """The time the action was initiated"""

    operation_type = sgqlc.types.Field(OperationType, graphql_name="operationType")
    """The corresponding operation type for the action"""

    user = sgqlc.types.Field("User", graphql_name="user")
    """The user affected by the action"""

    user_login = sgqlc.types.Field(String, graphql_name="userLogin")
    """For actions involving two users, the actor is the initiator and
    the user is the affected user.
    """

    user_resource_path = sgqlc.types.Field(URI, graphql_name="userResourcePath")
    """The HTTP path for the user."""

    user_url = sgqlc.types.Field(URI, graphql_name="userUrl")
    """The HTTP URL for the user."""


class Closable(sgqlc.types.Interface):
    """An object that can be closed"""

    __schema__ = github_schema
    __field_names__ = ("closed", "closed_at", "viewer_can_close", "viewer_can_reopen")
    closed = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="closed")
    """Indicates if the object is closed (definition of closed may depend
    on type)
    """

    closed_at = sgqlc.types.Field(DateTime, graphql_name="closedAt")
    """Identifies the date and time when the object was closed."""

    viewer_can_close = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanClose")
    """Indicates if the object can be closed by the viewer."""

    viewer_can_reopen = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanReopen")
    """Indicates if the object can be reopened by the viewer."""


class Comment(sgqlc.types.Interface):
    """Represents a comment."""

    __schema__ = github_schema
    __field_names__ = (
        "author",
        "author_association",
        "body",
        "body_html",
        "body_text",
        "created_at",
        "created_via_email",
        "editor",
        "id",
        "includes_created_edit",
        "last_edited_at",
        "published_at",
        "updated_at",
        "user_content_edits",
        "viewer_did_author",
    )
    author = sgqlc.types.Field(Actor, graphql_name="author")
    """The actor who authored the comment."""

    author_association = sgqlc.types.Field(sgqlc.types.non_null(CommentAuthorAssociation), graphql_name="authorAssociation")
    """Author's association with the subject of the comment."""

    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The body as Markdown."""

    body_html = sgqlc.types.Field(sgqlc.types.non_null(HTML), graphql_name="bodyHTML")
    """The body rendered to HTML."""

    body_text = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="bodyText")
    """The body rendered to text."""

    created_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="createdAt")
    """Identifies the date and time when the object was created."""

    created_via_email = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="createdViaEmail")
    """Check if this comment was created via an email reply."""

    editor = sgqlc.types.Field(Actor, graphql_name="editor")
    """The actor who edited the comment."""

    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    includes_created_edit = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="includesCreatedEdit")
    """Check if this comment was edited and includes an edit with the
    creation data
    """

    last_edited_at = sgqlc.types.Field(DateTime, graphql_name="lastEditedAt")
    """The moment the editor made the last edit"""

    published_at = sgqlc.types.Field(DateTime, graphql_name="publishedAt")
    """Identifies when the comment was published at."""

    updated_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="updatedAt")
    """Identifies the date and time when the object was last updated."""

    user_content_edits = sgqlc.types.Field(
        "UserContentEditConnection",
        graphql_name="userContentEdits",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of edits to this content.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    viewer_did_author = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerDidAuthor")
    """Did the viewer author this comment."""


class Contribution(sgqlc.types.Interface):
    """Represents a contribution a user made on GitHub, such as opening
    an issue.
    """

    __schema__ = github_schema
    __field_names__ = ("is_restricted", "occurred_at", "resource_path", "url", "user")
    is_restricted = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isRestricted")
    """Whether this contribution is associated with a record you do not
    have access to. For example, your own 'first issue' contribution
    may have been made on a repository you can no longer access.
    """

    occurred_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="occurredAt")
    """When this contribution was made."""

    resource_path = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="resourcePath")
    """The HTTP path for this contribution."""

    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """The HTTP URL for this contribution."""

    user = sgqlc.types.Field(sgqlc.types.non_null("User"), graphql_name="user")
    """The user who made this contribution."""


class Deletable(sgqlc.types.Interface):
    """Entities that can be deleted."""

    __schema__ = github_schema
    __field_names__ = ("viewer_can_delete",)
    viewer_can_delete = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanDelete")
    """Check if the current viewer can delete this object."""


class EnterpriseAuditEntryData(sgqlc.types.Interface):
    """Metadata for an audit entry containing enterprise account
    information.
    """

    __schema__ = github_schema
    __field_names__ = ("enterprise_resource_path", "enterprise_slug", "enterprise_url")
    enterprise_resource_path = sgqlc.types.Field(URI, graphql_name="enterpriseResourcePath")
    """The HTTP path for this enterprise."""

    enterprise_slug = sgqlc.types.Field(String, graphql_name="enterpriseSlug")
    """The slug of the enterprise."""

    enterprise_url = sgqlc.types.Field(URI, graphql_name="enterpriseUrl")
    """The HTTP URL for this enterprise."""


class GitObject(sgqlc.types.Interface):
    """Represents a Git object."""

    __schema__ = github_schema
    __field_names__ = ("abbreviated_oid", "commit_resource_path", "commit_url", "id", "oid", "repository")
    abbreviated_oid = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="abbreviatedOid")
    """An abbreviated version of the Git object ID"""

    commit_resource_path = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="commitResourcePath")
    """The HTTP path for this Git object"""

    commit_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="commitUrl")
    """The HTTP URL for this Git object"""

    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    oid = sgqlc.types.Field(sgqlc.types.non_null(GitObjectID), graphql_name="oid")
    """The Git object ID"""

    repository = sgqlc.types.Field(sgqlc.types.non_null("Repository"), graphql_name="repository")
    """The Repository the Git object belongs to"""


class GitSignature(sgqlc.types.Interface):
    """Information about a signature (GPG or S/MIME) on a Commit or Tag."""

    __schema__ = github_schema
    __field_names__ = ("email", "is_valid", "payload", "signature", "signer", "state", "was_signed_by_git_hub")
    email = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="email")
    """Email used to sign this object."""

    is_valid = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isValid")
    """True if the signature is valid and verified by GitHub."""

    payload = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="payload")
    """Payload for GPG signing object. Raw ODB object without the
    signature header.
    """

    signature = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="signature")
    """ASCII-armored signature header from object."""

    signer = sgqlc.types.Field("User", graphql_name="signer")
    """GitHub user corresponding to the email signing this commit."""

    state = sgqlc.types.Field(sgqlc.types.non_null(GitSignatureState), graphql_name="state")
    """The state of this signature. `VALID` if signature is valid and
    verified by GitHub, otherwise represents reason why signature is
    considered invalid.
    """

    was_signed_by_git_hub = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="wasSignedByGitHub")
    """True if the signature was made with GitHub's signing key."""


class HovercardContext(sgqlc.types.Interface):
    """An individual line of a hovercard"""

    __schema__ = github_schema
    __field_names__ = ("message", "octicon")
    message = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="message")
    """A string describing this context"""

    octicon = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="octicon")
    """An octicon to accompany this context"""


class Labelable(sgqlc.types.Interface):
    """An object that can have labels assigned to it."""

    __schema__ = github_schema
    __field_names__ = ("labels",)
    labels = sgqlc.types.Field(
        "LabelConnection",
        graphql_name="labels",
        args=sgqlc.types.ArgDict(
            (
                ("order_by", sgqlc.types.Arg(LabelOrder, graphql_name="orderBy", default={"field": "CREATED_AT", "direction": "ASC"})),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of labels associated with the object.

    Arguments:

    * `order_by` (`LabelOrder`): Ordering options for labels returned
      from the connection. (default: `{field: CREATED_AT, direction:
      ASC}`)
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class Lockable(sgqlc.types.Interface):
    """An object that can be locked."""

    __schema__ = github_schema
    __field_names__ = ("active_lock_reason", "locked")
    active_lock_reason = sgqlc.types.Field(LockReason, graphql_name="activeLockReason")
    """Reason that the conversation was locked."""

    locked = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="locked")
    """`true` if the object is locked"""


class MemberStatusable(sgqlc.types.Interface):
    """Entities that have members who can set status messages."""

    __schema__ = github_schema
    __field_names__ = ("member_statuses",)
    member_statuses = sgqlc.types.Field(
        sgqlc.types.non_null("UserStatusConnection"),
        graphql_name="memberStatuses",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(UserStatusOrder, graphql_name="orderBy", default={"field": "UPDATED_AT", "direction": "DESC"}),
                ),
            )
        ),
    )
    """Get the status messages members of this entity have set that are
    either public or visible only to the organization.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`UserStatusOrder`): Ordering options for user
      statuses returned from the connection. (default: `{field:
      UPDATED_AT, direction: DESC}`)
    """


class Migration(sgqlc.types.Interface):
    """Represents a GitHub Enterprise Importer (GEI) migration."""

    __schema__ = github_schema
    __field_names__ = (
        "continue_on_error",
        "created_at",
        "database_id",
        "failure_reason",
        "id",
        "migration_log_url",
        "migration_source",
        "repository_name",
        "source_url",
        "state",
        "warnings_count",
    )
    continue_on_error = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="continueOnError")
    """The migration flag to continue on error."""

    created_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="createdAt")
    """Identifies the date and time when the object was created."""

    database_id = sgqlc.types.Field(String, graphql_name="databaseId")
    """Identifies the primary key from the database."""

    failure_reason = sgqlc.types.Field(String, graphql_name="failureReason")
    """The reason the migration failed."""

    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    migration_log_url = sgqlc.types.Field(URI, graphql_name="migrationLogUrl")
    """The URL for the migration log (expires 1 day after migration
    completes).
    """

    migration_source = sgqlc.types.Field(sgqlc.types.non_null("MigrationSource"), graphql_name="migrationSource")
    """The migration source."""

    repository_name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="repositoryName")
    """The target repository name."""

    source_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="sourceUrl")
    """The migration source URL, for example `https://github.com` or
    `https://monalisa.ghe.com`.
    """

    state = sgqlc.types.Field(sgqlc.types.non_null(MigrationState), graphql_name="state")
    """The migration state."""

    warnings_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="warningsCount")
    """The number of warnings encountered for this migration. To review
    the warnings, check the [Migration
    Log](https://docs.github.com/en/migrations/using-github-
    enterprise-importer/completing-your-migration-with-github-
    enterprise-importer/accessing-your-migration-logs-for-github-
    enterprise-importer).
    """


class Minimizable(sgqlc.types.Interface):
    """Entities that can be minimized."""

    __schema__ = github_schema
    __field_names__ = ("is_minimized", "minimized_reason", "viewer_can_minimize")
    is_minimized = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isMinimized")
    """Returns whether or not a comment has been minimized."""

    minimized_reason = sgqlc.types.Field(String, graphql_name="minimizedReason")
    """Returns why the comment was minimized. One of `abuse`, `off-
    topic`, `outdated`, `resolved`, `duplicate` and `spam`. Note that
    the case and formatting of these values differs from the inputs to
    the `MinimizeComment` mutation.
    """

    viewer_can_minimize = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanMinimize")
    """Check if the current viewer can minimize this object."""


class Node(sgqlc.types.Interface):
    """An object with an ID."""

    __schema__ = github_schema
    __field_names__ = ("id",)
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")
    """ID of the object."""


class OauthApplicationAuditEntryData(sgqlc.types.Interface):
    """Metadata for an audit entry with action oauth_application.*"""

    __schema__ = github_schema
    __field_names__ = ("oauth_application_name", "oauth_application_resource_path", "oauth_application_url")
    oauth_application_name = sgqlc.types.Field(String, graphql_name="oauthApplicationName")
    """The name of the OAuth application."""

    oauth_application_resource_path = sgqlc.types.Field(URI, graphql_name="oauthApplicationResourcePath")
    """The HTTP path for the OAuth application"""

    oauth_application_url = sgqlc.types.Field(URI, graphql_name="oauthApplicationUrl")
    """The HTTP URL for the OAuth application"""


class OrganizationAuditEntryData(sgqlc.types.Interface):
    """Metadata for an audit entry with action org.*"""

    __schema__ = github_schema
    __field_names__ = ("organization", "organization_name", "organization_resource_path", "organization_url")
    organization = sgqlc.types.Field("Organization", graphql_name="organization")
    """The Organization associated with the Audit Entry."""

    organization_name = sgqlc.types.Field(String, graphql_name="organizationName")
    """The name of the Organization."""

    organization_resource_path = sgqlc.types.Field(URI, graphql_name="organizationResourcePath")
    """The HTTP path for the organization"""

    organization_url = sgqlc.types.Field(URI, graphql_name="organizationUrl")
    """The HTTP URL for the organization"""


class PackageOwner(sgqlc.types.Interface):
    """Represents an owner of a package."""

    __schema__ = github_schema
    __field_names__ = ("id", "packages")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    packages = sgqlc.types.Field(
        sgqlc.types.non_null("PackageConnection"),
        graphql_name="packages",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("names", sgqlc.types.Arg(sgqlc.types.list_of(String), graphql_name="names", default=None)),
                ("repository_id", sgqlc.types.Arg(ID, graphql_name="repositoryId", default=None)),
                ("package_type", sgqlc.types.Arg(PackageType, graphql_name="packageType", default=None)),
                ("order_by", sgqlc.types.Arg(PackageOrder, graphql_name="orderBy", default={"field": "CREATED_AT", "direction": "DESC"})),
            )
        ),
    )
    """A list of packages under the owner.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `names` (`[String]`): Find packages by their names.
    * `repository_id` (`ID`): Find packages in a repository by ID.
    * `package_type` (`PackageType`): Filter registry package by type.
    * `order_by` (`PackageOrder`): Ordering of the returned packages.
      (default: `{field: CREATED_AT, direction: DESC}`)
    """


class ProfileOwner(sgqlc.types.Interface):
    """Represents any entity on GitHub that has a profile page."""

    __schema__ = github_schema
    __field_names__ = (
        "any_pinnable_items",
        "email",
        "id",
        "item_showcase",
        "location",
        "login",
        "name",
        "pinnable_items",
        "pinned_items",
        "pinned_items_remaining",
        "viewer_can_change_pinned_items",
        "website_url",
    )
    any_pinnable_items = sgqlc.types.Field(
        sgqlc.types.non_null(Boolean),
        graphql_name="anyPinnableItems",
        args=sgqlc.types.ArgDict((("type", sgqlc.types.Arg(PinnableItemType, graphql_name="type", default=None)),)),
    )
    """Determine if this repository owner has any items that can be
    pinned to their profile.

    Arguments:

    * `type` (`PinnableItemType`): Filter to only a particular kind of
      pinnable item.
    """

    email = sgqlc.types.Field(String, graphql_name="email")
    """The public profile email."""

    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    item_showcase = sgqlc.types.Field(sgqlc.types.non_null("ProfileItemShowcase"), graphql_name="itemShowcase")
    """Showcases a selection of repositories and gists that the profile
    owner has either curated or that have been selected automatically
    based on popularity.
    """

    location = sgqlc.types.Field(String, graphql_name="location")
    """The public profile location."""

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The username used to login."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """The public profile name."""

    pinnable_items = sgqlc.types.Field(
        sgqlc.types.non_null("PinnableItemConnection"),
        graphql_name="pinnableItems",
        args=sgqlc.types.ArgDict(
            (
                ("types", sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(PinnableItemType)), graphql_name="types", default=None)),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of repositories and gists this profile owner can pin to
    their profile.

    Arguments:

    * `types` (`[PinnableItemType!]`): Filter the types of pinnable
      items that are returned.
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    pinned_items = sgqlc.types.Field(
        sgqlc.types.non_null("PinnableItemConnection"),
        graphql_name="pinnedItems",
        args=sgqlc.types.ArgDict(
            (
                ("types", sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(PinnableItemType)), graphql_name="types", default=None)),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of repositories and gists this profile owner has pinned to
    their profile

    Arguments:

    * `types` (`[PinnableItemType!]`): Filter the types of pinned
      items that are returned.
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    pinned_items_remaining = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="pinnedItemsRemaining")
    """Returns how many more items this profile owner can pin to their
    profile.
    """

    viewer_can_change_pinned_items = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanChangePinnedItems")
    """Can the viewer pin repositories and gists to the profile?"""

    website_url = sgqlc.types.Field(URI, graphql_name="websiteUrl")
    """The public profile website URL."""


class ProjectOwner(sgqlc.types.Interface):
    """Represents an owner of a Project."""

    __schema__ = github_schema
    __field_names__ = ("id", "project", "projects", "projects_resource_path", "projects_url", "viewer_can_create_projects")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    project = sgqlc.types.Field(
        "Project",
        graphql_name="project",
        args=sgqlc.types.ArgDict((("number", sgqlc.types.Arg(sgqlc.types.non_null(Int), graphql_name="number", default=None)),)),
    )
    """Find project by number.

    Arguments:

    * `number` (`Int!`): The project number to find.
    """

    projects = sgqlc.types.Field(
        sgqlc.types.non_null("ProjectConnection"),
        graphql_name="projects",
        args=sgqlc.types.ArgDict(
            (
                ("order_by", sgqlc.types.Arg(ProjectOrder, graphql_name="orderBy", default=None)),
                ("search", sgqlc.types.Arg(String, graphql_name="search", default=None)),
                ("states", sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(ProjectState)), graphql_name="states", default=None)),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of projects under the owner.

    Arguments:

    * `order_by` (`ProjectOrder`): Ordering options for projects
      returned from the connection
    * `search` (`String`): Query to search projects by, currently only
      searching by name.
    * `states` (`[ProjectState!]`): A list of states to filter the
      projects by.
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    projects_resource_path = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="projectsResourcePath")
    """The HTTP path listing owners projects"""

    projects_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="projectsUrl")
    """The HTTP URL listing owners projects"""

    viewer_can_create_projects = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanCreateProjects")
    """Can the current viewer create new projects on this owner."""


class ProjectV2FieldCommon(sgqlc.types.Interface):
    """Common fields across different project field types"""

    __schema__ = github_schema
    __field_names__ = ("created_at", "data_type", "database_id", "id", "name", "project", "updated_at")
    created_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="createdAt")
    """Identifies the date and time when the object was created."""

    data_type = sgqlc.types.Field(sgqlc.types.non_null(ProjectV2FieldType), graphql_name="dataType")
    """The field's type."""

    database_id = sgqlc.types.Field(Int, graphql_name="databaseId")
    """Identifies the primary key from the database."""

    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The project field's name."""

    project = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2"), graphql_name="project")
    """The project that contains this field."""

    updated_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="updatedAt")
    """Identifies the date and time when the object was last updated."""


class ProjectV2ItemFieldValueCommon(sgqlc.types.Interface):
    """Common fields across different project field value types"""

    __schema__ = github_schema
    __field_names__ = ("created_at", "creator", "database_id", "field", "id", "item", "updated_at")
    created_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="createdAt")
    """Identifies the date and time when the object was created."""

    creator = sgqlc.types.Field(Actor, graphql_name="creator")
    """The actor who created the item."""

    database_id = sgqlc.types.Field(Int, graphql_name="databaseId")
    """Identifies the primary key from the database."""

    field = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2FieldConfiguration"), graphql_name="field")
    """The project field that contains this value."""

    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    item = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2Item"), graphql_name="item")
    """The project item that contains this value."""

    updated_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="updatedAt")
    """Identifies the date and time when the object was last updated."""


class ProjectV2Owner(sgqlc.types.Interface):
    """Represents an owner of a project (beta)."""

    __schema__ = github_schema
    __field_names__ = ("id", "project_v2", "projects_v2")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    project_v2 = sgqlc.types.Field(
        "ProjectV2",
        graphql_name="projectV2",
        args=sgqlc.types.ArgDict((("number", sgqlc.types.Arg(sgqlc.types.non_null(Int), graphql_name="number", default=None)),)),
    )
    """Find a project by number.

    Arguments:

    * `number` (`Int!`): The project number.
    """

    projects_v2 = sgqlc.types.Field(
        sgqlc.types.non_null("ProjectV2Connection"),
        graphql_name="projectsV2",
        args=sgqlc.types.ArgDict(
            (
                ("query", sgqlc.types.Arg(String, graphql_name="query", default=None)),
                ("order_by", sgqlc.types.Arg(ProjectV2Order, graphql_name="orderBy", default={"field": "NUMBER", "direction": "DESC"})),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of projects under the owner.

    Arguments:

    * `query` (`String`): A project to search for under the the owner.
    * `order_by` (`ProjectV2Order`): How to order the returned
      projects. (default: `{field: NUMBER, direction: DESC}`)
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class ProjectV2Recent(sgqlc.types.Interface):
    """Recent projects for the owner."""

    __schema__ = github_schema
    __field_names__ = ("recent_projects",)
    recent_projects = sgqlc.types.Field(
        sgqlc.types.non_null("ProjectV2Connection"),
        graphql_name="recentProjects",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """Recent projects that this user has modified in the context of the
    owner.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class Reactable(sgqlc.types.Interface):
    """Represents a subject that can be reacted on."""

    __schema__ = github_schema
    __field_names__ = ("database_id", "id", "reaction_groups", "reactions", "viewer_can_react")
    database_id = sgqlc.types.Field(Int, graphql_name="databaseId")
    """Identifies the primary key from the database."""

    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    reaction_groups = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("ReactionGroup")), graphql_name="reactionGroups")
    """A list of reactions grouped by content left on the subject."""

    reactions = sgqlc.types.Field(
        sgqlc.types.non_null("ReactionConnection"),
        graphql_name="reactions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("content", sgqlc.types.Arg(ReactionContent, graphql_name="content", default=None)),
                ("order_by", sgqlc.types.Arg(ReactionOrder, graphql_name="orderBy", default=None)),
            )
        ),
    )
    """A list of Reactions left on the Issue.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `content` (`ReactionContent`): Allows filtering Reactions by
      emoji.
    * `order_by` (`ReactionOrder`): Allows specifying the order in
      which reactions are returned.
    """

    viewer_can_react = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanReact")
    """Can user react to this subject"""


class RepositoryAuditEntryData(sgqlc.types.Interface):
    """Metadata for an audit entry with action repo.*"""

    __schema__ = github_schema
    __field_names__ = ("repository", "repository_name", "repository_resource_path", "repository_url")
    repository = sgqlc.types.Field("Repository", graphql_name="repository")
    """The repository associated with the action"""

    repository_name = sgqlc.types.Field(String, graphql_name="repositoryName")
    """The name of the repository"""

    repository_resource_path = sgqlc.types.Field(URI, graphql_name="repositoryResourcePath")
    """The HTTP path for the repository"""

    repository_url = sgqlc.types.Field(URI, graphql_name="repositoryUrl")
    """The HTTP URL for the repository"""


class RepositoryDiscussionAuthor(sgqlc.types.Interface):
    """Represents an author of discussions in repositories."""

    __schema__ = github_schema
    __field_names__ = ("repository_discussions",)
    repository_discussions = sgqlc.types.Field(
        sgqlc.types.non_null("DiscussionConnection"),
        graphql_name="repositoryDiscussions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(DiscussionOrder, graphql_name="orderBy", default={"field": "CREATED_AT", "direction": "DESC"}),
                ),
                ("repository_id", sgqlc.types.Arg(ID, graphql_name="repositoryId", default=None)),
                ("answered", sgqlc.types.Arg(Boolean, graphql_name="answered", default=None)),
                ("states", sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(DiscussionState)), graphql_name="states", default=())),
            )
        ),
    )
    """Discussions this user has started.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`DiscussionOrder`): Ordering options for discussions
      returned from the connection. (default: `{field: CREATED_AT,
      direction: DESC}`)
    * `repository_id` (`ID`): Filter discussions to only those in a
      specific repository.
    * `answered` (`Boolean`): Filter discussions to only those that
      have been answered or not. Defaults to including both answered
      and unanswered discussions. (default: `null`)
    * `states` (`[DiscussionState!]`): A list of states to filter the
      discussions by. (default: `[]`)
    """


class RepositoryDiscussionCommentAuthor(sgqlc.types.Interface):
    """Represents an author of discussion comments in repositories."""

    __schema__ = github_schema
    __field_names__ = ("repository_discussion_comments",)
    repository_discussion_comments = sgqlc.types.Field(
        sgqlc.types.non_null("DiscussionCommentConnection"),
        graphql_name="repositoryDiscussionComments",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("repository_id", sgqlc.types.Arg(ID, graphql_name="repositoryId", default=None)),
                ("only_answers", sgqlc.types.Arg(Boolean, graphql_name="onlyAnswers", default=False)),
            )
        ),
    )
    """Discussion comments this user has authored.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `repository_id` (`ID`): Filter discussion comments to only those
      in a specific repository.
    * `only_answers` (`Boolean`): Filter discussion comments to only
      those that were marked as the answer (default: `false`)
    """


class RepositoryInfo(sgqlc.types.Interface):
    """A subset of repository info."""

    __schema__ = github_schema
    __field_names__ = (
        "archived_at",
        "created_at",
        "description",
        "description_html",
        "fork_count",
        "has_discussions_enabled",
        "has_issues_enabled",
        "has_projects_enabled",
        "has_wiki_enabled",
        "homepage_url",
        "is_archived",
        "is_fork",
        "is_in_organization",
        "is_locked",
        "is_mirror",
        "is_private",
        "is_template",
        "license_info",
        "lock_reason",
        "mirror_url",
        "name",
        "name_with_owner",
        "open_graph_image_url",
        "owner",
        "pushed_at",
        "resource_path",
        "short_description_html",
        "updated_at",
        "url",
        "uses_custom_open_graph_image",
        "visibility",
    )
    archived_at = sgqlc.types.Field(DateTime, graphql_name="archivedAt")
    """Identifies the date and time when the repository was archived."""

    created_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="createdAt")
    """Identifies the date and time when the object was created."""

    description = sgqlc.types.Field(String, graphql_name="description")
    """The description of the repository."""

    description_html = sgqlc.types.Field(sgqlc.types.non_null(HTML), graphql_name="descriptionHTML")
    """The description of the repository rendered to HTML."""

    fork_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="forkCount")
    """Returns how many forks there are of this repository in the whole
    network.
    """

    has_discussions_enabled = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasDiscussionsEnabled")
    """Indicates if the repository has the Discussions feature enabled."""

    has_issues_enabled = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasIssuesEnabled")
    """Indicates if the repository has issues feature enabled."""

    has_projects_enabled = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasProjectsEnabled")
    """Indicates if the repository has the Projects feature enabled."""

    has_wiki_enabled = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasWikiEnabled")
    """Indicates if the repository has wiki feature enabled."""

    homepage_url = sgqlc.types.Field(URI, graphql_name="homepageUrl")
    """The repository's URL."""

    is_archived = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isArchived")
    """Indicates if the repository is unmaintained."""

    is_fork = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isFork")
    """Identifies if the repository is a fork."""

    is_in_organization = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isInOrganization")
    """Indicates if a repository is either owned by an organization, or
    is a private fork of an organization repository.
    """

    is_locked = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isLocked")
    """Indicates if the repository has been locked or not."""

    is_mirror = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isMirror")
    """Identifies if the repository is a mirror."""

    is_private = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isPrivate")
    """Identifies if the repository is private or internal."""

    is_template = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isTemplate")
    """Identifies if the repository is a template that can be used to
    generate new repositories.
    """

    license_info = sgqlc.types.Field("License", graphql_name="licenseInfo")
    """The license associated with the repository"""

    lock_reason = sgqlc.types.Field(RepositoryLockReason, graphql_name="lockReason")
    """The reason the repository has been locked."""

    mirror_url = sgqlc.types.Field(URI, graphql_name="mirrorUrl")
    """The repository's original mirror URL."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the repository."""

    name_with_owner = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="nameWithOwner")
    """The repository's name with owner."""

    open_graph_image_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="openGraphImageUrl")
    """The image used to represent this repository in Open Graph data."""

    owner = sgqlc.types.Field(sgqlc.types.non_null("RepositoryOwner"), graphql_name="owner")
    """The User owner of the repository."""

    pushed_at = sgqlc.types.Field(DateTime, graphql_name="pushedAt")
    """Identifies the date and time when the repository was last pushed
    to.
    """

    resource_path = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="resourcePath")
    """The HTTP path for this repository"""

    short_description_html = sgqlc.types.Field(
        sgqlc.types.non_null(HTML),
        graphql_name="shortDescriptionHTML",
        args=sgqlc.types.ArgDict((("limit", sgqlc.types.Arg(Int, graphql_name="limit", default=200)),)),
    )
    """A description of the repository, rendered to HTML without any
    links in it.

    Arguments:

    * `limit` (`Int`): How many characters to return. (default: `200`)
    """

    updated_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="updatedAt")
    """Identifies the date and time when the object was last updated."""

    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """The HTTP URL for this repository"""

    uses_custom_open_graph_image = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="usesCustomOpenGraphImage")
    """Whether this repository has a custom image to use with Open Graph
    as opposed to being represented by the owner's avatar.
    """

    visibility = sgqlc.types.Field(sgqlc.types.non_null(RepositoryVisibility), graphql_name="visibility")
    """Indicates the repository's visibility level."""


class RepositoryNode(sgqlc.types.Interface):
    """Represents a object that belongs to a repository."""

    __schema__ = github_schema
    __field_names__ = ("repository",)
    repository = sgqlc.types.Field(sgqlc.types.non_null("Repository"), graphql_name="repository")
    """The repository associated with this node."""


class RepositoryOwner(sgqlc.types.Interface):
    """Represents an owner of a Repository."""

    __schema__ = github_schema
    __field_names__ = ("avatar_url", "id", "login", "repositories", "repository", "resource_path", "url")
    avatar_url = sgqlc.types.Field(
        sgqlc.types.non_null(URI),
        graphql_name="avatarUrl",
        args=sgqlc.types.ArgDict((("size", sgqlc.types.Arg(Int, graphql_name="size", default=None)),)),
    )
    """A URL pointing to the owner's public avatar.

    Arguments:

    * `size` (`Int`): The size of the resulting square image.
    """

    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    login = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="login")
    """The username used to login."""

    repositories = sgqlc.types.Field(
        sgqlc.types.non_null("RepositoryConnection"),
        graphql_name="repositories",
        args=sgqlc.types.ArgDict(
            (
                ("privacy", sgqlc.types.Arg(RepositoryPrivacy, graphql_name="privacy", default=None)),
                ("order_by", sgqlc.types.Arg(RepositoryOrder, graphql_name="orderBy", default=None)),
                ("affiliations", sgqlc.types.Arg(sgqlc.types.list_of(RepositoryAffiliation), graphql_name="affiliations", default=None)),
                (
                    "owner_affiliations",
                    sgqlc.types.Arg(
                        sgqlc.types.list_of(RepositoryAffiliation), graphql_name="ownerAffiliations", default=("OWNER", "COLLABORATOR")
                    ),
                ),
                ("is_locked", sgqlc.types.Arg(Boolean, graphql_name="isLocked", default=None)),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("is_fork", sgqlc.types.Arg(Boolean, graphql_name="isFork", default=None)),
            )
        ),
    )
    """A list of repositories that the user owns.

    Arguments:

    * `privacy` (`RepositoryPrivacy`): If non-null, filters
      repositories according to privacy
    * `order_by` (`RepositoryOrder`): Ordering options for
      repositories returned from the connection
    * `affiliations` (`[RepositoryAffiliation]`): Array of viewer's
      affiliation options for repositories returned from the
      connection. For example, OWNER will include only repositories
      that the current viewer owns.
    * `owner_affiliations` (`[RepositoryAffiliation]`): Array of
      owner's affiliation options for repositories returned from the
      connection. For example, OWNER will include only repositories
      that the organization or user being viewed owns. (default:
      `[OWNER, COLLABORATOR]`)
    * `is_locked` (`Boolean`): If non-null, filters repositories
      according to whether they have been locked
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `is_fork` (`Boolean`): If non-null, filters repositories
      according to whether they are forks of another repository
    """

    repository = sgqlc.types.Field(
        "Repository",
        graphql_name="repository",
        args=sgqlc.types.ArgDict(
            (
                ("name", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="name", default=None)),
                ("follow_renames", sgqlc.types.Arg(Boolean, graphql_name="followRenames", default=True)),
            )
        ),
    )
    """Find Repository.

    Arguments:

    * `name` (`String!`): Name of Repository to find.
    * `follow_renames` (`Boolean`): Follow repository renames. If
      disabled, a repository referenced by its old name will return an
      error. (default: `true`)
    """

    resource_path = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="resourcePath")
    """The HTTP URL for the owner."""

    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """The HTTP URL for the owner."""


class RequirableByPullRequest(sgqlc.types.Interface):
    """Represents a type that can be required by a pull request for
    merging.
    """

    __schema__ = github_schema
    __field_names__ = ("is_required",)
    is_required = sgqlc.types.Field(
        sgqlc.types.non_null(Boolean),
        graphql_name="isRequired",
        args=sgqlc.types.ArgDict(
            (
                ("pull_request_id", sgqlc.types.Arg(ID, graphql_name="pullRequestId", default=None)),
                ("pull_request_number", sgqlc.types.Arg(Int, graphql_name="pullRequestNumber", default=None)),
            )
        ),
    )
    """Whether this is required to pass before merging for a specific
    pull request.

    Arguments:

    * `pull_request_id` (`ID`): The id of the pull request this is
      required for
    * `pull_request_number` (`Int`): The number of the pull request
      this is required for
    """


class Sponsorable(sgqlc.types.Interface):
    """Entities that can sponsor or be sponsored through GitHub Sponsors."""

    __schema__ = github_schema
    __field_names__ = (
        "estimated_next_sponsors_payout_in_cents",
        "has_sponsors_listing",
        "is_sponsored_by",
        "is_sponsoring_viewer",
        "monthly_estimated_sponsors_income_in_cents",
        "sponsoring",
        "sponsors",
        "sponsors_activities",
        "sponsors_listing",
        "sponsorship_for_viewer_as_sponsor",
        "sponsorship_for_viewer_as_sponsorable",
        "sponsorship_newsletters",
        "sponsorships_as_maintainer",
        "sponsorships_as_sponsor",
        "total_sponsorship_amount_as_sponsor_in_cents",
        "viewer_can_sponsor",
        "viewer_is_sponsoring",
    )
    estimated_next_sponsors_payout_in_cents = sgqlc.types.Field(
        sgqlc.types.non_null(Int), graphql_name="estimatedNextSponsorsPayoutInCents"
    )
    """The estimated next GitHub Sponsors payout for this
    user/organization in cents (USD).
    """

    has_sponsors_listing = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasSponsorsListing")
    """True if this user/organization has a GitHub Sponsors listing."""

    is_sponsored_by = sgqlc.types.Field(
        sgqlc.types.non_null(Boolean),
        graphql_name="isSponsoredBy",
        args=sgqlc.types.ArgDict(
            (("account_login", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="accountLogin", default=None)),)
        ),
    )
    """Whether the given account is sponsoring this user/organization.

    Arguments:

    * `account_login` (`String!`): The target account's login.
    """

    is_sponsoring_viewer = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isSponsoringViewer")
    """True if the viewer is sponsored by this user/organization."""

    monthly_estimated_sponsors_income_in_cents = sgqlc.types.Field(
        sgqlc.types.non_null(Int), graphql_name="monthlyEstimatedSponsorsIncomeInCents"
    )
    """The estimated monthly GitHub Sponsors income for this
    user/organization in cents (USD).
    """

    sponsoring = sgqlc.types.Field(
        sgqlc.types.non_null("SponsorConnection"),
        graphql_name="sponsoring",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("order_by", sgqlc.types.Arg(SponsorOrder, graphql_name="orderBy", default={"field": "RELEVANCE", "direction": "DESC"})),
            )
        ),
    )
    """List of users and organizations this entity is sponsoring.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`SponsorOrder`): Ordering options for the users and
      organizations returned from the connection. (default: `{field:
      RELEVANCE, direction: DESC}`)
    """

    sponsors = sgqlc.types.Field(
        sgqlc.types.non_null("SponsorConnection"),
        graphql_name="sponsors",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("tier_id", sgqlc.types.Arg(ID, graphql_name="tierId", default=None)),
                ("order_by", sgqlc.types.Arg(SponsorOrder, graphql_name="orderBy", default={"field": "RELEVANCE", "direction": "DESC"})),
            )
        ),
    )
    """List of sponsors for this user or organization.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `tier_id` (`ID`): If given, will filter for sponsors at the
      given tier. Will only return sponsors whose tier the viewer is
      permitted to see.
    * `order_by` (`SponsorOrder`): Ordering options for sponsors
      returned from the connection. (default: `{field: RELEVANCE,
      direction: DESC}`)
    """

    sponsors_activities = sgqlc.types.Field(
        sgqlc.types.non_null("SponsorsActivityConnection"),
        graphql_name="sponsorsActivities",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("period", sgqlc.types.Arg(SponsorsActivityPeriod, graphql_name="period", default="MONTH")),
                ("since", sgqlc.types.Arg(DateTime, graphql_name="since", default=None)),
                ("until", sgqlc.types.Arg(DateTime, graphql_name="until", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(SponsorsActivityOrder, graphql_name="orderBy", default={"field": "TIMESTAMP", "direction": "DESC"}),
                ),
                (
                    "actions",
                    sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(SponsorsActivityAction)), graphql_name="actions", default=()),
                ),
                ("include_as_sponsor", sgqlc.types.Arg(Boolean, graphql_name="includeAsSponsor", default=False)),
            )
        ),
    )
    """Events involving this sponsorable, such as new sponsorships.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `period` (`SponsorsActivityPeriod`): Filter activities returned
      to only those that occurred in the most recent specified time
      period. Set to ALL to avoid filtering by when the activity
      occurred. Will be ignored if `since` or `until` is given.
      (default: `MONTH`)
    * `since` (`DateTime`): Filter activities to those that occurred
      on or after this time.
    * `until` (`DateTime`): Filter activities to those that occurred
      before this time.
    * `order_by` (`SponsorsActivityOrder`): Ordering options for
      activity returned from the connection. (default: `{field:
      TIMESTAMP, direction: DESC}`)
    * `actions` (`[SponsorsActivityAction!]`): Filter activities to
      only the specified actions. (default: `[]`)
    * `include_as_sponsor` (`Boolean`): Whether to include those
      events where this sponsorable acted as the sponsor. Defaults to
      only including events where this sponsorable was the recipient
      of a sponsorship. (default: `false`)
    """

    sponsors_listing = sgqlc.types.Field("SponsorsListing", graphql_name="sponsorsListing")
    """The GitHub Sponsors listing for this user or organization."""

    sponsorship_for_viewer_as_sponsor = sgqlc.types.Field(
        "Sponsorship",
        graphql_name="sponsorshipForViewerAsSponsor",
        args=sgqlc.types.ArgDict((("active_only", sgqlc.types.Arg(Boolean, graphql_name="activeOnly", default=True)),)),
    )
    """The sponsorship from the viewer to this user/organization; that
    is, the sponsorship where you're the sponsor.

    Arguments:

    * `active_only` (`Boolean`): Whether to return the sponsorship
      only if it's still active. Pass false to get the viewer's
      sponsorship back even if it has been cancelled. (default:
      `true`)
    """

    sponsorship_for_viewer_as_sponsorable = sgqlc.types.Field(
        "Sponsorship",
        graphql_name="sponsorshipForViewerAsSponsorable",
        args=sgqlc.types.ArgDict((("active_only", sgqlc.types.Arg(Boolean, graphql_name="activeOnly", default=True)),)),
    )
    """The sponsorship from this user/organization to the viewer; that
    is, the sponsorship you're receiving.

    Arguments:

    * `active_only` (`Boolean`): Whether to return the sponsorship
      only if it's still active. Pass false to get the sponsorship
      back even if it has been cancelled. (default: `true`)
    """

    sponsorship_newsletters = sgqlc.types.Field(
        sgqlc.types.non_null("SponsorshipNewsletterConnection"),
        graphql_name="sponsorshipNewsletters",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(
                        SponsorshipNewsletterOrder, graphql_name="orderBy", default={"field": "CREATED_AT", "direction": "DESC"}
                    ),
                ),
            )
        ),
    )
    """List of sponsorship updates sent from this sponsorable to
    sponsors.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`SponsorshipNewsletterOrder`): Ordering options for
      sponsorship updates returned from the connection. (default:
      `{field: CREATED_AT, direction: DESC}`)
    """

    sponsorships_as_maintainer = sgqlc.types.Field(
        sgqlc.types.non_null("SponsorshipConnection"),
        graphql_name="sponsorshipsAsMaintainer",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("include_private", sgqlc.types.Arg(Boolean, graphql_name="includePrivate", default=False)),
                ("order_by", sgqlc.types.Arg(SponsorshipOrder, graphql_name="orderBy", default=None)),
                ("active_only", sgqlc.types.Arg(Boolean, graphql_name="activeOnly", default=True)),
            )
        ),
    )
    """The sponsorships where this user or organization is the maintainer
    receiving the funds.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `include_private` (`Boolean`): Whether or not to include private
      sponsorships in the result set (default: `false`)
    * `order_by` (`SponsorshipOrder`): Ordering options for
      sponsorships returned from this connection. If left blank, the
      sponsorships will be ordered based on relevancy to the viewer.
    * `active_only` (`Boolean`): Whether to include only sponsorships
      that are active right now, versus all sponsorships this
      maintainer has ever received. (default: `true`)
    """

    sponsorships_as_sponsor = sgqlc.types.Field(
        sgqlc.types.non_null("SponsorshipConnection"),
        graphql_name="sponsorshipsAsSponsor",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("order_by", sgqlc.types.Arg(SponsorshipOrder, graphql_name="orderBy", default=None)),
                (
                    "maintainer_logins",
                    sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="maintainerLogins", default=None),
                ),
                ("active_only", sgqlc.types.Arg(Boolean, graphql_name="activeOnly", default=True)),
            )
        ),
    )
    """The sponsorships where this user or organization is the funder.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`SponsorshipOrder`): Ordering options for
      sponsorships returned from this connection. If left blank, the
      sponsorships will be ordered based on relevancy to the viewer.
    * `maintainer_logins` (`[String!]`): Filter sponsorships returned
      to those for the specified maintainers. That is, the recipient
      of the sponsorship is a user or organization with one of the
      given logins.
    * `active_only` (`Boolean`): Whether to include only sponsorships
      that are active right now, versus all sponsorships this sponsor
      has ever made. (default: `true`)
    """

    total_sponsorship_amount_as_sponsor_in_cents = sgqlc.types.Field(
        Int,
        graphql_name="totalSponsorshipAmountAsSponsorInCents",
        args=sgqlc.types.ArgDict(
            (
                ("since", sgqlc.types.Arg(DateTime, graphql_name="since", default=None)),
                ("until", sgqlc.types.Arg(DateTime, graphql_name="until", default=None)),
                (
                    "sponsorable_logins",
                    sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="sponsorableLogins", default=()),
                ),
            )
        ),
    )
    """The amount in United States cents (e.g., 500 = $5.00 USD) that
    this entity has spent on GitHub to fund sponsorships. Only returns
    a value when viewed by the user themselves or by a user who can
    manage sponsorships for the requested organization.

    Arguments:

    * `since` (`DateTime`): Filter payments to those that occurred on
      or after this time.
    * `until` (`DateTime`): Filter payments to those that occurred
      before this time.
    * `sponsorable_logins` (`[String!]`): Filter payments to those
      made to the users or organizations with the specified usernames.
      (default: `[]`)
    """

    viewer_can_sponsor = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanSponsor")
    """Whether or not the viewer is able to sponsor this
    user/organization.
    """

    viewer_is_sponsoring = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerIsSponsoring")
    """True if the viewer is sponsoring this user/organization."""


class Starrable(sgqlc.types.Interface):
    """Things that can be starred."""

    __schema__ = github_schema
    __field_names__ = ("id", "stargazer_count", "stargazers", "viewer_has_starred")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    stargazer_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="stargazerCount")
    """Returns a count of how many stargazers there are on this object"""

    stargazers = sgqlc.types.Field(
        sgqlc.types.non_null("StargazerConnection"),
        graphql_name="stargazers",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("order_by", sgqlc.types.Arg(StarOrder, graphql_name="orderBy", default=None)),
            )
        ),
    )
    """A list of users who have starred this starrable.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`StarOrder`): Order for connection
    """

    viewer_has_starred = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerHasStarred")
    """Returns a boolean indicating whether the viewing user has starred
    this starrable.
    """


class Subscribable(sgqlc.types.Interface):
    """Entities that can be subscribed to for web and email
    notifications.
    """

    __schema__ = github_schema
    __field_names__ = ("id", "viewer_can_subscribe", "viewer_subscription")
    id = sgqlc.types.Field(sgqlc.types.non_null(ID), graphql_name="id")

    viewer_can_subscribe = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanSubscribe")
    """Check if the viewer is able to change their subscription status
    for the repository.
    """

    viewer_subscription = sgqlc.types.Field(SubscriptionState, graphql_name="viewerSubscription")
    """Identifies if the viewer is watching, not watching, or ignoring
    the subscribable entity.
    """


class TeamAuditEntryData(sgqlc.types.Interface):
    """Metadata for an audit entry with action team.*"""

    __schema__ = github_schema
    __field_names__ = ("team", "team_name", "team_resource_path", "team_url")
    team = sgqlc.types.Field("Team", graphql_name="team")
    """The team associated with the action"""

    team_name = sgqlc.types.Field(String, graphql_name="teamName")
    """The name of the team"""

    team_resource_path = sgqlc.types.Field(URI, graphql_name="teamResourcePath")
    """The HTTP path for this team"""

    team_url = sgqlc.types.Field(URI, graphql_name="teamUrl")
    """The HTTP URL for this team"""


class TopicAuditEntryData(sgqlc.types.Interface):
    """Metadata for an audit entry with a topic."""

    __schema__ = github_schema
    __field_names__ = ("topic", "topic_name")
    topic = sgqlc.types.Field("Topic", graphql_name="topic")
    """The name of the topic added to the repository"""

    topic_name = sgqlc.types.Field(String, graphql_name="topicName")
    """The name of the topic added to the repository"""


class UniformResourceLocatable(sgqlc.types.Interface):
    """Represents a type that can be retrieved by a URL."""

    __schema__ = github_schema
    __field_names__ = ("resource_path", "url")
    resource_path = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="resourcePath")
    """The HTML path to this resource."""

    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """The URL to this resource."""


class Updatable(sgqlc.types.Interface):
    """Entities that can be updated."""

    __schema__ = github_schema
    __field_names__ = ("viewer_can_update",)
    viewer_can_update = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanUpdate")
    """Check if the current viewer can update this object."""


class UpdatableComment(sgqlc.types.Interface):
    """Comments that can be updated."""

    __schema__ = github_schema
    __field_names__ = ("viewer_cannot_update_reasons",)
    viewer_cannot_update_reasons = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(CommentCannotUpdateReason))), graphql_name="viewerCannotUpdateReasons"
    )
    """Reasons why the current viewer can not update this comment."""


class Votable(sgqlc.types.Interface):
    """A subject that may be upvoted."""

    __schema__ = github_schema
    __field_names__ = ("upvote_count", "viewer_can_upvote", "viewer_has_upvoted")
    upvote_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="upvoteCount")
    """Number of upvotes that this subject has received."""

    viewer_can_upvote = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanUpvote")
    """Whether or not the current user can add or remove an upvote on
    this subject.
    """

    viewer_has_upvoted = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerHasUpvoted")
    """Whether or not the current user has already upvoted this subject."""


class AbortQueuedMigrationsPayload(sgqlc.types.Type):
    """Autogenerated return type of AbortQueuedMigrations"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "success")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    success = sgqlc.types.Field(Boolean, graphql_name="success")
    """Did the operation succeed?"""


class AcceptEnterpriseAdministratorInvitationPayload(sgqlc.types.Type):
    """Autogenerated return type of
    AcceptEnterpriseAdministratorInvitation
    """

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "invitation", "message")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    invitation = sgqlc.types.Field("EnterpriseAdministratorInvitation", graphql_name="invitation")
    """The invitation that was accepted."""

    message = sgqlc.types.Field(String, graphql_name="message")
    """A message confirming the result of accepting an administrator
    invitation.
    """


class AcceptTopicSuggestionPayload(sgqlc.types.Type):
    """Autogenerated return type of AcceptTopicSuggestion"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "topic")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    topic = sgqlc.types.Field("Topic", graphql_name="topic")
    """The accepted topic."""


class ActorLocation(sgqlc.types.Type):
    """Location information for an actor"""

    __schema__ = github_schema
    __field_names__ = ("city", "country", "country_code", "region", "region_code")
    city = sgqlc.types.Field(String, graphql_name="city")
    """City"""

    country = sgqlc.types.Field(String, graphql_name="country")
    """Country name"""

    country_code = sgqlc.types.Field(String, graphql_name="countryCode")
    """Country code"""

    region = sgqlc.types.Field(String, graphql_name="region")
    """Region name"""

    region_code = sgqlc.types.Field(String, graphql_name="regionCode")
    """Region or state code"""


class AddAssigneesToAssignablePayload(sgqlc.types.Type):
    """Autogenerated return type of AddAssigneesToAssignable"""

    __schema__ = github_schema
    __field_names__ = ("assignable", "client_mutation_id")
    assignable = sgqlc.types.Field(Assignable, graphql_name="assignable")
    """The item that was assigned."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class AddCommentPayload(sgqlc.types.Type):
    """Autogenerated return type of AddComment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "comment_edge", "subject", "timeline_edge")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    comment_edge = sgqlc.types.Field("IssueCommentEdge", graphql_name="commentEdge")
    """The edge from the subject's comment connection."""

    subject = sgqlc.types.Field(Node, graphql_name="subject")
    """The subject"""

    timeline_edge = sgqlc.types.Field("IssueTimelineItemEdge", graphql_name="timelineEdge")
    """The edge from the subject's timeline connection."""


class AddDiscussionCommentPayload(sgqlc.types.Type):
    """Autogenerated return type of AddDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "comment")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    comment = sgqlc.types.Field("DiscussionComment", graphql_name="comment")
    """The newly created discussion comment."""


class AddDiscussionPollVotePayload(sgqlc.types.Type):
    """Autogenerated return type of AddDiscussionPollVote"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "poll_option")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    poll_option = sgqlc.types.Field("DiscussionPollOption", graphql_name="pollOption")
    """The poll option that a vote was added to."""


class AddEnterpriseOrganizationMemberPayload(sgqlc.types.Type):
    """Autogenerated return type of AddEnterpriseOrganizationMember"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "users")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    users = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("User")), graphql_name="users")
    """The users who were added to the organization."""


class AddEnterpriseSupportEntitlementPayload(sgqlc.types.Type):
    """Autogenerated return type of AddEnterpriseSupportEntitlement"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "message")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    message = sgqlc.types.Field(String, graphql_name="message")
    """A message confirming the result of adding the support entitlement."""


class AddLabelsToLabelablePayload(sgqlc.types.Type):
    """Autogenerated return type of AddLabelsToLabelable"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "labelable")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    labelable = sgqlc.types.Field(Labelable, graphql_name="labelable")
    """The item that was labeled."""


class AddProjectCardPayload(sgqlc.types.Type):
    """Autogenerated return type of AddProjectCard"""

    __schema__ = github_schema
    __field_names__ = ("card_edge", "client_mutation_id", "project_column")
    card_edge = sgqlc.types.Field("ProjectCardEdge", graphql_name="cardEdge")
    """The edge from the ProjectColumn's card connection."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_column = sgqlc.types.Field("ProjectColumn", graphql_name="projectColumn")
    """The ProjectColumn"""


class AddProjectColumnPayload(sgqlc.types.Type):
    """Autogenerated return type of AddProjectColumn"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "column_edge", "project")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    column_edge = sgqlc.types.Field("ProjectColumnEdge", graphql_name="columnEdge")
    """The edge from the project's column connection."""

    project = sgqlc.types.Field("Project", graphql_name="project")
    """The project"""


class AddProjectV2DraftIssuePayload(sgqlc.types.Type):
    """Autogenerated return type of AddProjectV2DraftIssue"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project_item")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_item = sgqlc.types.Field("ProjectV2Item", graphql_name="projectItem")
    """The draft issue added to the project."""


class AddProjectV2ItemByIdPayload(sgqlc.types.Type):
    """Autogenerated return type of AddProjectV2ItemById"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "item")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    item = sgqlc.types.Field("ProjectV2Item", graphql_name="item")
    """The item added to the project."""


class AddPullRequestReviewCommentPayload(sgqlc.types.Type):
    """Autogenerated return type of AddPullRequestReviewComment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "comment", "comment_edge")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    comment = sgqlc.types.Field("PullRequestReviewComment", graphql_name="comment")
    """The newly created comment."""

    comment_edge = sgqlc.types.Field("PullRequestReviewCommentEdge", graphql_name="commentEdge")
    """The edge from the review's comment connection."""


class AddPullRequestReviewPayload(sgqlc.types.Type):
    """Autogenerated return type of AddPullRequestReview"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request_review", "review_edge")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request_review = sgqlc.types.Field("PullRequestReview", graphql_name="pullRequestReview")
    """The newly created pull request review."""

    review_edge = sgqlc.types.Field("PullRequestReviewEdge", graphql_name="reviewEdge")
    """The edge from the pull request's review connection."""


class AddPullRequestReviewThreadPayload(sgqlc.types.Type):
    """Autogenerated return type of AddPullRequestReviewThread"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "thread")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    thread = sgqlc.types.Field("PullRequestReviewThread", graphql_name="thread")
    """The newly created thread."""


class AddReactionPayload(sgqlc.types.Type):
    """Autogenerated return type of AddReaction"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "reaction", "reaction_groups", "subject")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    reaction = sgqlc.types.Field("Reaction", graphql_name="reaction")
    """The reaction object."""

    reaction_groups = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("ReactionGroup")), graphql_name="reactionGroups")
    """The reaction groups for the subject."""

    subject = sgqlc.types.Field(Reactable, graphql_name="subject")
    """The reactable subject."""


class AddStarPayload(sgqlc.types.Type):
    """Autogenerated return type of AddStar"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "starrable")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    starrable = sgqlc.types.Field(Starrable, graphql_name="starrable")
    """The starrable."""


class AddUpvotePayload(sgqlc.types.Type):
    """Autogenerated return type of AddUpvote"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "subject")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    subject = sgqlc.types.Field(Votable, graphql_name="subject")
    """The votable subject."""


class AddVerifiableDomainPayload(sgqlc.types.Type):
    """Autogenerated return type of AddVerifiableDomain"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "domain")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    domain = sgqlc.types.Field("VerifiableDomain", graphql_name="domain")
    """The verifiable domain that was added."""


class ApproveDeploymentsPayload(sgqlc.types.Type):
    """Autogenerated return type of ApproveDeployments"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "deployments")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    deployments = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("Deployment")), graphql_name="deployments")
    """The affected deployments."""


class ApproveVerifiableDomainPayload(sgqlc.types.Type):
    """Autogenerated return type of ApproveVerifiableDomain"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "domain")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    domain = sgqlc.types.Field("VerifiableDomain", graphql_name="domain")
    """The verifiable domain that was approved."""


class ArchiveProjectV2ItemPayload(sgqlc.types.Type):
    """Autogenerated return type of ArchiveProjectV2Item"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "item")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    item = sgqlc.types.Field("ProjectV2Item", graphql_name="item")
    """The item archived from the project."""


class ArchiveRepositoryPayload(sgqlc.types.Type):
    """Autogenerated return type of ArchiveRepository"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "repository")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    repository = sgqlc.types.Field("Repository", graphql_name="repository")
    """The repository that was marked as archived."""


class AutoMergeRequest(sgqlc.types.Type):
    """Represents an auto-merge request for a pull request"""

    __schema__ = github_schema
    __field_names__ = ("author_email", "commit_body", "commit_headline", "enabled_at", "enabled_by", "merge_method", "pull_request")
    author_email = sgqlc.types.Field(String, graphql_name="authorEmail")
    """The email address of the author of this auto-merge request."""

    commit_body = sgqlc.types.Field(String, graphql_name="commitBody")
    """The commit message of the auto-merge request. If a merge queue is
    required by the base branch, this value will be set by the merge
    queue when merging.
    """

    commit_headline = sgqlc.types.Field(String, graphql_name="commitHeadline")
    """The commit title of the auto-merge request. If a merge queue is
    required by the base branch, this value will be set by the merge
    queue when merging
    """

    enabled_at = sgqlc.types.Field(DateTime, graphql_name="enabledAt")
    """When was this auto-merge request was enabled."""

    enabled_by = sgqlc.types.Field(Actor, graphql_name="enabledBy")
    """The actor who created the auto-merge request."""

    merge_method = sgqlc.types.Field(sgqlc.types.non_null(PullRequestMergeMethod), graphql_name="mergeMethod")
    """The merge method of the auto-merge request. If a merge queue is
    required by the base branch, this value will be set by the merge
    queue when merging.
    """

    pull_request = sgqlc.types.Field(sgqlc.types.non_null("PullRequest"), graphql_name="pullRequest")
    """The pull request that this auto-merge request is set against."""


class Blame(sgqlc.types.Type):
    """Represents a Git blame."""

    __schema__ = github_schema
    __field_names__ = ("ranges",)
    ranges = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("BlameRange"))), graphql_name="ranges")
    """The list of ranges from a Git blame."""


class BlameRange(sgqlc.types.Type):
    """Represents a range of information from a Git blame."""

    __schema__ = github_schema
    __field_names__ = ("age", "commit", "ending_line", "starting_line")
    age = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="age")
    """Identifies the recency of the change, from 1 (new) to 10 (old).
    This is calculated as a 2-quantile and determines the length of
    distance between the median age of all the changes in the file and
    the recency of the current range's change.
    """

    commit = sgqlc.types.Field(sgqlc.types.non_null("Commit"), graphql_name="commit")
    """Identifies the line author"""

    ending_line = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="endingLine")
    """The ending line for the range"""

    starting_line = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="startingLine")
    """The starting line for the range"""


class BranchNamePatternParameters(sgqlc.types.Type):
    """Parameters to be used for the branch_name_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class BranchProtectionRuleConflict(sgqlc.types.Type):
    """A conflict between two branch protection rules."""

    __schema__ = github_schema
    __field_names__ = ("branch_protection_rule", "conflicting_branch_protection_rule", "ref")
    branch_protection_rule = sgqlc.types.Field("BranchProtectionRule", graphql_name="branchProtectionRule")
    """Identifies the branch protection rule."""

    conflicting_branch_protection_rule = sgqlc.types.Field("BranchProtectionRule", graphql_name="conflictingBranchProtectionRule")
    """Identifies the conflicting branch protection rule."""

    ref = sgqlc.types.Field("Ref", graphql_name="ref")
    """Identifies the branch ref that has conflicting rules"""


class BranchProtectionRuleConflictConnection(sgqlc.types.relay.Connection):
    """The connection type for BranchProtectionRuleConflict."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("BranchProtectionRuleConflictEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(BranchProtectionRuleConflict), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class BranchProtectionRuleConflictEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(BranchProtectionRuleConflict, graphql_name="node")
    """The item at the end of the edge."""


class BranchProtectionRuleConnection(sgqlc.types.relay.Connection):
    """The connection type for BranchProtectionRule."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("BranchProtectionRuleEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("BranchProtectionRule"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class BranchProtectionRuleEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("BranchProtectionRule", graphql_name="node")
    """The item at the end of the edge."""


class BypassForcePushAllowanceConnection(sgqlc.types.relay.Connection):
    """The connection type for BypassForcePushAllowance."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("BypassForcePushAllowanceEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("BypassForcePushAllowance"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class BypassForcePushAllowanceEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("BypassForcePushAllowance", graphql_name="node")
    """The item at the end of the edge."""


class BypassPullRequestAllowanceConnection(sgqlc.types.relay.Connection):
    """The connection type for BypassPullRequestAllowance."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("BypassPullRequestAllowanceEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("BypassPullRequestAllowance"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class BypassPullRequestAllowanceEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("BypassPullRequestAllowance", graphql_name="node")
    """The item at the end of the edge."""


class CVSS(sgqlc.types.Type):
    """The Common Vulnerability Scoring System"""

    __schema__ = github_schema
    __field_names__ = ("score", "vector_string")
    score = sgqlc.types.Field(sgqlc.types.non_null(Float), graphql_name="score")
    """The CVSS score associated with this advisory"""

    vector_string = sgqlc.types.Field(String, graphql_name="vectorString")
    """The CVSS vector string associated with this advisory"""


class CWEConnection(sgqlc.types.relay.Connection):
    """The connection type for CWE."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CWEEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("CWE"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CWEEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("CWE", graphql_name="node")
    """The item at the end of the edge."""


class CancelEnterpriseAdminInvitationPayload(sgqlc.types.Type):
    """Autogenerated return type of CancelEnterpriseAdminInvitation"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "invitation", "message")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    invitation = sgqlc.types.Field("EnterpriseAdministratorInvitation", graphql_name="invitation")
    """The invitation that was canceled."""

    message = sgqlc.types.Field(String, graphql_name="message")
    """A message confirming the result of canceling an administrator
    invitation.
    """


class CancelSponsorshipPayload(sgqlc.types.Type):
    """Autogenerated return type of CancelSponsorship"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "sponsors_tier")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    sponsors_tier = sgqlc.types.Field("SponsorsTier", graphql_name="sponsorsTier")
    """The tier that was being used at the time of cancellation."""


class ChangeUserStatusPayload(sgqlc.types.Type):
    """Autogenerated return type of ChangeUserStatus"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "status")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    status = sgqlc.types.Field("UserStatus", graphql_name="status")
    """Your updated status."""


class CheckAnnotation(sgqlc.types.Type):
    """A single check annotation."""

    __schema__ = github_schema
    __field_names__ = ("annotation_level", "blob_url", "database_id", "location", "message", "path", "raw_details", "title")
    annotation_level = sgqlc.types.Field(CheckAnnotationLevel, graphql_name="annotationLevel")
    """The annotation's severity level."""

    blob_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="blobUrl")
    """The path to the file that this annotation was made on."""

    database_id = sgqlc.types.Field(Int, graphql_name="databaseId")
    """Identifies the primary key from the database."""

    location = sgqlc.types.Field(sgqlc.types.non_null("CheckAnnotationSpan"), graphql_name="location")
    """The position of this annotation."""

    message = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="message")
    """The annotation's message."""

    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """The path that this annotation was made on."""

    raw_details = sgqlc.types.Field(String, graphql_name="rawDetails")
    """Additional information about the annotation."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """The annotation's title"""


class CheckAnnotationConnection(sgqlc.types.relay.Connection):
    """The connection type for CheckAnnotation."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CheckAnnotationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(CheckAnnotation), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CheckAnnotationEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(CheckAnnotation, graphql_name="node")
    """The item at the end of the edge."""


class CheckAnnotationPosition(sgqlc.types.Type):
    """A character position in a check annotation."""

    __schema__ = github_schema
    __field_names__ = ("column", "line")
    column = sgqlc.types.Field(Int, graphql_name="column")
    """Column number (1 indexed)."""

    line = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="line")
    """Line number (1 indexed)."""


class CheckAnnotationSpan(sgqlc.types.Type):
    """An inclusive pair of positions for a check annotation."""

    __schema__ = github_schema
    __field_names__ = ("end", "start")
    end = sgqlc.types.Field(sgqlc.types.non_null(CheckAnnotationPosition), graphql_name="end")
    """End position (inclusive)."""

    start = sgqlc.types.Field(sgqlc.types.non_null(CheckAnnotationPosition), graphql_name="start")
    """Start position (inclusive)."""


class CheckRunConnection(sgqlc.types.relay.Connection):
    """The connection type for CheckRun."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CheckRunEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("CheckRun"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CheckRunEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("CheckRun", graphql_name="node")
    """The item at the end of the edge."""


class CheckRunStateCount(sgqlc.types.Type):
    """Represents a count of the state of a check run."""

    __schema__ = github_schema
    __field_names__ = ("count", "state")
    count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="count")
    """The number of check runs with this state."""

    state = sgqlc.types.Field(sgqlc.types.non_null(CheckRunState), graphql_name="state")
    """The state of a check run."""


class CheckStep(sgqlc.types.Type):
    """A single check step."""

    __schema__ = github_schema
    __field_names__ = ("completed_at", "conclusion", "external_id", "name", "number", "seconds_to_completion", "started_at", "status")
    completed_at = sgqlc.types.Field(DateTime, graphql_name="completedAt")
    """Identifies the date and time when the check step was completed."""

    conclusion = sgqlc.types.Field(CheckConclusionState, graphql_name="conclusion")
    """The conclusion of the check step."""

    external_id = sgqlc.types.Field(String, graphql_name="externalId")
    """A reference for the check step on the integrator's system."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The step's name."""

    number = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="number")
    """The index of the step in the list of steps of the parent check
    run.
    """

    seconds_to_completion = sgqlc.types.Field(Int, graphql_name="secondsToCompletion")
    """Number of seconds to completion."""

    started_at = sgqlc.types.Field(DateTime, graphql_name="startedAt")
    """Identifies the date and time when the check step was started."""

    status = sgqlc.types.Field(sgqlc.types.non_null(CheckStatusState), graphql_name="status")
    """The current status of the check step."""


class CheckStepConnection(sgqlc.types.relay.Connection):
    """The connection type for CheckStep."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CheckStepEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(CheckStep), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CheckStepEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(CheckStep, graphql_name="node")
    """The item at the end of the edge."""


class CheckSuiteConnection(sgqlc.types.relay.Connection):
    """The connection type for CheckSuite."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CheckSuiteEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("CheckSuite"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CheckSuiteEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("CheckSuite", graphql_name="node")
    """The item at the end of the edge."""


class ClearLabelsFromLabelablePayload(sgqlc.types.Type):
    """Autogenerated return type of ClearLabelsFromLabelable"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "labelable")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    labelable = sgqlc.types.Field(Labelable, graphql_name="labelable")
    """The item that was unlabeled."""


class ClearProjectV2ItemFieldValuePayload(sgqlc.types.Type):
    """Autogenerated return type of ClearProjectV2ItemFieldValue"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project_v2_item")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_v2_item = sgqlc.types.Field("ProjectV2Item", graphql_name="projectV2Item")
    """The updated item."""


class CloneProjectPayload(sgqlc.types.Type):
    """Autogenerated return type of CloneProject"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "job_status_id", "project")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    job_status_id = sgqlc.types.Field(String, graphql_name="jobStatusId")
    """The id of the JobStatus for populating cloned fields."""

    project = sgqlc.types.Field("Project", graphql_name="project")
    """The new cloned project."""


class CloneTemplateRepositoryPayload(sgqlc.types.Type):
    """Autogenerated return type of CloneTemplateRepository"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "repository")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    repository = sgqlc.types.Field("Repository", graphql_name="repository")
    """The new repository."""


class CloseDiscussionPayload(sgqlc.types.Type):
    """Autogenerated return type of CloseDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "discussion")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    discussion = sgqlc.types.Field("Discussion", graphql_name="discussion")
    """The discussion that was closed."""


class CloseIssuePayload(sgqlc.types.Type):
    """Autogenerated return type of CloseIssue"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "issue")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    issue = sgqlc.types.Field("Issue", graphql_name="issue")
    """The issue that was closed."""


class ClosePullRequestPayload(sgqlc.types.Type):
    """Autogenerated return type of ClosePullRequest"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The pull request that was closed."""


class CommitAuthorEmailPatternParameters(sgqlc.types.Type):
    """Parameters to be used for the commit_author_email_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class CommitCommentConnection(sgqlc.types.relay.Connection):
    """The connection type for CommitComment."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CommitCommentEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("CommitComment"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CommitCommentEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("CommitComment", graphql_name="node")
    """The item at the end of the edge."""


class CommitConnection(sgqlc.types.relay.Connection):
    """The connection type for Commit."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CommitEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Commit"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CommitContributionsByRepository(sgqlc.types.Type):
    """This aggregates commits made by a user within one repository."""

    __schema__ = github_schema
    __field_names__ = ("contributions", "repository", "resource_path", "url")
    contributions = sgqlc.types.Field(
        sgqlc.types.non_null("CreatedCommitContributionConnection"),
        graphql_name="contributions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(CommitContributionOrder, graphql_name="orderBy", default={"field": "OCCURRED_AT", "direction": "DESC"}),
                ),
            )
        ),
    )
    """The commit contributions, each representing a day.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`CommitContributionOrder`): Ordering options for
      commit contributions returned from the connection. (default:
      `{field: OCCURRED_AT, direction: DESC}`)
    """

    repository = sgqlc.types.Field(sgqlc.types.non_null("Repository"), graphql_name="repository")
    """The repository in which the commits were made."""

    resource_path = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="resourcePath")
    """The HTTP path for the user's commits to the repository in this
    time range.
    """

    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """The HTTP URL for the user's commits to the repository in this time
    range.
    """


class CommitEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Commit", graphql_name="node")
    """The item at the end of the edge."""


class CommitHistoryConnection(sgqlc.types.relay.Connection):
    """The connection type for Commit."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of(CommitEdge), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Commit"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CommitMessagePatternParameters(sgqlc.types.Type):
    """Parameters to be used for the commit_message_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class CommitterEmailPatternParameters(sgqlc.types.Type):
    """Parameters to be used for the committer_email_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class ComparisonCommitConnection(sgqlc.types.relay.Connection):
    """The connection type for Commit."""

    __schema__ = github_schema
    __field_names__ = ("author_count", "edges", "nodes", "page_info", "total_count")
    author_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="authorCount")
    """The total count of authors and co-authors across all commits."""

    edges = sgqlc.types.Field(sgqlc.types.list_of(CommitEdge), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Commit"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ContributionCalendar(sgqlc.types.Type):
    """A calendar of contributions made on GitHub by a user."""

    __schema__ = github_schema
    __field_names__ = ("colors", "is_halloween", "months", "total_contributions", "weeks")
    colors = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="colors")
    """A list of hex color codes used in this calendar. The darker the
    color, the more contributions it represents.
    """

    is_halloween = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isHalloween")
    """Determine if the color set was chosen because it's currently
    Halloween.
    """

    months = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("ContributionCalendarMonth"))), graphql_name="months"
    )
    """A list of the months of contributions in this calendar."""

    total_contributions = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalContributions")
    """The count of total contributions in the calendar."""

    weeks = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("ContributionCalendarWeek"))), graphql_name="weeks"
    )
    """A list of the weeks of contributions in this calendar."""


class ContributionCalendarDay(sgqlc.types.Type):
    """Represents a single day of contributions on GitHub by a user."""

    __schema__ = github_schema
    __field_names__ = ("color", "contribution_count", "contribution_level", "date", "weekday")
    color = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="color")
    """The hex color code that represents how many contributions were
    made on this day compared to others in the calendar.
    """

    contribution_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="contributionCount")
    """How many contributions were made by the user on this day."""

    contribution_level = sgqlc.types.Field(sgqlc.types.non_null(ContributionLevel), graphql_name="contributionLevel")
    """Indication of contributions, relative to other days. Can be used
    to indicate which color to represent this day on a calendar.
    """

    date = sgqlc.types.Field(sgqlc.types.non_null(Date), graphql_name="date")
    """The day this square represents."""

    weekday = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="weekday")
    """A number representing which day of the week this square
    represents, e.g., 1 is Monday.
    """


class ContributionCalendarMonth(sgqlc.types.Type):
    """A month of contributions in a user's contribution graph."""

    __schema__ = github_schema
    __field_names__ = ("first_day", "name", "total_weeks", "year")
    first_day = sgqlc.types.Field(sgqlc.types.non_null(Date), graphql_name="firstDay")
    """The date of the first day of this month."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the month."""

    total_weeks = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalWeeks")
    """How many weeks started in this month."""

    year = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="year")
    """The year the month occurred in."""


class ContributionCalendarWeek(sgqlc.types.Type):
    """A week of contributions in a user's contribution graph."""

    __schema__ = github_schema
    __field_names__ = ("contribution_days", "first_day")
    contribution_days = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ContributionCalendarDay))), graphql_name="contributionDays"
    )
    """The days of contributions in this week."""

    first_day = sgqlc.types.Field(sgqlc.types.non_null(Date), graphql_name="firstDay")
    """The date of the earliest square in this week."""


class ContributionsCollection(sgqlc.types.Type):
    """A contributions collection aggregates contributions such as opened
    issues and commits created by a user.
    """

    __schema__ = github_schema
    __field_names__ = (
        "commit_contributions_by_repository",
        "contribution_calendar",
        "contribution_years",
        "does_end_in_current_month",
        "earliest_restricted_contribution_date",
        "ended_at",
        "first_issue_contribution",
        "first_pull_request_contribution",
        "first_repository_contribution",
        "has_activity_in_the_past",
        "has_any_contributions",
        "has_any_restricted_contributions",
        "is_single_day",
        "issue_contributions",
        "issue_contributions_by_repository",
        "joined_git_hub_contribution",
        "latest_restricted_contribution_date",
        "most_recent_collection_with_activity",
        "most_recent_collection_without_activity",
        "popular_issue_contribution",
        "popular_pull_request_contribution",
        "pull_request_contributions",
        "pull_request_contributions_by_repository",
        "pull_request_review_contributions",
        "pull_request_review_contributions_by_repository",
        "repository_contributions",
        "restricted_contributions_count",
        "started_at",
        "total_commit_contributions",
        "total_issue_contributions",
        "total_pull_request_contributions",
        "total_pull_request_review_contributions",
        "total_repositories_with_contributed_commits",
        "total_repositories_with_contributed_issues",
        "total_repositories_with_contributed_pull_request_reviews",
        "total_repositories_with_contributed_pull_requests",
        "total_repository_contributions",
        "user",
    )
    commit_contributions_by_repository = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(CommitContributionsByRepository))),
        graphql_name="commitContributionsByRepository",
        args=sgqlc.types.ArgDict((("max_repositories", sgqlc.types.Arg(Int, graphql_name="maxRepositories", default=25)),)),
    )
    """Commit contributions made by the user, grouped by repository.

    Arguments:

    * `max_repositories` (`Int`): How many repositories should be
      included. (default: `25`)
    """

    contribution_calendar = sgqlc.types.Field(sgqlc.types.non_null(ContributionCalendar), graphql_name="contributionCalendar")
    """A calendar of this user's contributions on GitHub."""

    contribution_years = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(Int))), graphql_name="contributionYears"
    )
    """The years the user has been making contributions with the most
    recent year first.
    """

    does_end_in_current_month = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="doesEndInCurrentMonth")
    """Determine if this collection's time span ends in the current
    month.
    """

    earliest_restricted_contribution_date = sgqlc.types.Field(Date, graphql_name="earliestRestrictedContributionDate")
    """The date of the first restricted contribution the user made in
    this time period. Can only be non-null when the user has enabled
    private contribution counts.
    """

    ended_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="endedAt")
    """The ending date and time of this collection."""

    first_issue_contribution = sgqlc.types.Field("CreatedIssueOrRestrictedContribution", graphql_name="firstIssueContribution")
    """The first issue the user opened on GitHub. This will be null if
    that issue was opened outside the collection's time range and
    ignoreTimeRange is false. If the issue is not visible but the user
    has opted to show private contributions, a RestrictedContribution
    will be returned.
    """

    first_pull_request_contribution = sgqlc.types.Field(
        "CreatedPullRequestOrRestrictedContribution", graphql_name="firstPullRequestContribution"
    )
    """The first pull request the user opened on GitHub. This will be
    null if that pull request was opened outside the collection's time
    range and ignoreTimeRange is not true. If the pull request is not
    visible but the user has opted to show private contributions, a
    RestrictedContribution will be returned.
    """

    first_repository_contribution = sgqlc.types.Field(
        "CreatedRepositoryOrRestrictedContribution", graphql_name="firstRepositoryContribution"
    )
    """The first repository the user created on GitHub. This will be null
    if that first repository was created outside the collection's time
    range and ignoreTimeRange is false. If the repository is not
    visible, then a RestrictedContribution is returned.
    """

    has_activity_in_the_past = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasActivityInThePast")
    """Does the user have any more activity in the timeline that occurred
    prior to the collection's time range?
    """

    has_any_contributions = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasAnyContributions")
    """Determine if there are any contributions in this collection."""

    has_any_restricted_contributions = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasAnyRestrictedContributions")
    """Determine if the user made any contributions in this time frame
    whose details are not visible because they were made in a private
    repository. Can only be true if the user enabled private
    contribution counts.
    """

    is_single_day = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isSingleDay")
    """Whether or not the collector's time span is all within the same
    day.
    """

    issue_contributions = sgqlc.types.Field(
        sgqlc.types.non_null("CreatedIssueContributionConnection"),
        graphql_name="issueContributions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),
                ("exclude_popular", sgqlc.types.Arg(Boolean, graphql_name="excludePopular", default=False)),
                ("order_by", sgqlc.types.Arg(ContributionOrder, graphql_name="orderBy", default={"direction": "DESC"})),
            )
        ),
    )
    """A list of issues the user opened.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `exclude_first` (`Boolean`): Should the user's first issue ever
      be excluded from the result. (default: `false`)
    * `exclude_popular` (`Boolean`): Should the user's most commented
      issue be excluded from the result. (default: `false`)
    * `order_by` (`ContributionOrder`): Ordering options for
      contributions returned from the connection. (default:
      `{direction: DESC}`)
    """

    issue_contributions_by_repository = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("IssueContributionsByRepository"))),
        graphql_name="issueContributionsByRepository",
        args=sgqlc.types.ArgDict(
            (
                ("max_repositories", sgqlc.types.Arg(Int, graphql_name="maxRepositories", default=25)),
                ("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),
                ("exclude_popular", sgqlc.types.Arg(Boolean, graphql_name="excludePopular", default=False)),
            )
        ),
    )
    """Issue contributions made by the user, grouped by repository.

    Arguments:

    * `max_repositories` (`Int`): How many repositories should be
      included. (default: `25`)
    * `exclude_first` (`Boolean`): Should the user's first issue ever
      be excluded from the result. (default: `false`)
    * `exclude_popular` (`Boolean`): Should the user's most commented
      issue be excluded from the result. (default: `false`)
    """

    joined_git_hub_contribution = sgqlc.types.Field("JoinedGitHubContribution", graphql_name="joinedGitHubContribution")
    """When the user signed up for GitHub. This will be null if that sign
    up date falls outside the collection's time range and
    ignoreTimeRange is false.
    """

    latest_restricted_contribution_date = sgqlc.types.Field(Date, graphql_name="latestRestrictedContributionDate")
    """The date of the most recent restricted contribution the user made
    in this time period. Can only be non-null when the user has
    enabled private contribution counts.
    """

    most_recent_collection_with_activity = sgqlc.types.Field("ContributionsCollection", graphql_name="mostRecentCollectionWithActivity")
    """When this collection's time range does not include any activity
    from the user, use this to get a different collection from an
    earlier time range that does have activity.
    """

    most_recent_collection_without_activity = sgqlc.types.Field(
        "ContributionsCollection", graphql_name="mostRecentCollectionWithoutActivity"
    )
    """Returns a different contributions collection from an earlier time
    range than this one that does not have any contributions.
    """

    popular_issue_contribution = sgqlc.types.Field("CreatedIssueContribution", graphql_name="popularIssueContribution")
    """The issue the user opened on GitHub that received the most
    comments in the specified time frame.
    """

    popular_pull_request_contribution = sgqlc.types.Field("CreatedPullRequestContribution", graphql_name="popularPullRequestContribution")
    """The pull request the user opened on GitHub that received the most
    comments in the specified time frame.
    """

    pull_request_contributions = sgqlc.types.Field(
        sgqlc.types.non_null("CreatedPullRequestContributionConnection"),
        graphql_name="pullRequestContributions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),
                ("exclude_popular", sgqlc.types.Arg(Boolean, graphql_name="excludePopular", default=False)),
                ("order_by", sgqlc.types.Arg(ContributionOrder, graphql_name="orderBy", default={"direction": "DESC"})),
            )
        ),
    )
    """Pull request contributions made by the user.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `exclude_first` (`Boolean`): Should the user's first pull
      request ever be excluded from the result. (default: `false`)
    * `exclude_popular` (`Boolean`): Should the user's most commented
      pull request be excluded from the result. (default: `false`)
    * `order_by` (`ContributionOrder`): Ordering options for
      contributions returned from the connection. (default:
      `{direction: DESC}`)
    """

    pull_request_contributions_by_repository = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("PullRequestContributionsByRepository"))),
        graphql_name="pullRequestContributionsByRepository",
        args=sgqlc.types.ArgDict(
            (
                ("max_repositories", sgqlc.types.Arg(Int, graphql_name="maxRepositories", default=25)),
                ("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),
                ("exclude_popular", sgqlc.types.Arg(Boolean, graphql_name="excludePopular", default=False)),
            )
        ),
    )
    """Pull request contributions made by the user, grouped by
    repository.

    Arguments:

    * `max_repositories` (`Int`): How many repositories should be
      included. (default: `25`)
    * `exclude_first` (`Boolean`): Should the user's first pull
      request ever be excluded from the result. (default: `false`)
    * `exclude_popular` (`Boolean`): Should the user's most commented
      pull request be excluded from the result. (default: `false`)
    """

    pull_request_review_contributions = sgqlc.types.Field(
        sgqlc.types.non_null("CreatedPullRequestReviewContributionConnection"),
        graphql_name="pullRequestReviewContributions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("order_by", sgqlc.types.Arg(ContributionOrder, graphql_name="orderBy", default={"direction": "DESC"})),
            )
        ),
    )
    """Pull request review contributions made by the user. Returns the
    most recently submitted review for each PR reviewed by the user.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`ContributionOrder`): Ordering options for
      contributions returned from the connection. (default:
      `{direction: DESC}`)
    """

    pull_request_review_contributions_by_repository = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("PullRequestReviewContributionsByRepository"))),
        graphql_name="pullRequestReviewContributionsByRepository",
        args=sgqlc.types.ArgDict((("max_repositories", sgqlc.types.Arg(Int, graphql_name="maxRepositories", default=25)),)),
    )
    """Pull request review contributions made by the user, grouped by
    repository.

    Arguments:

    * `max_repositories` (`Int`): How many repositories should be
      included. (default: `25`)
    """

    repository_contributions = sgqlc.types.Field(
        sgqlc.types.non_null("CreatedRepositoryContributionConnection"),
        graphql_name="repositoryContributions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),
                ("order_by", sgqlc.types.Arg(ContributionOrder, graphql_name="orderBy", default={"direction": "DESC"})),
            )
        ),
    )
    """A list of repositories owned by the user that the user created in
    this time range.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `exclude_first` (`Boolean`): Should the user's first repository
      ever be excluded from the result. (default: `false`)
    * `order_by` (`ContributionOrder`): Ordering options for
      contributions returned from the connection. (default:
      `{direction: DESC}`)
    """

    restricted_contributions_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="restrictedContributionsCount")
    """A count of contributions made by the user that the viewer cannot
    access. Only non-zero when the user has chosen to share their
    private contribution counts.
    """

    started_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="startedAt")
    """The beginning date and time of this collection."""

    total_commit_contributions = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCommitContributions")
    """How many commits were made by the user in this time span."""

    total_issue_contributions = sgqlc.types.Field(
        sgqlc.types.non_null(Int),
        graphql_name="totalIssueContributions",
        args=sgqlc.types.ArgDict(
            (
                ("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),
                ("exclude_popular", sgqlc.types.Arg(Boolean, graphql_name="excludePopular", default=False)),
            )
        ),
    )
    """How many issues the user opened.

    Arguments:

    * `exclude_first` (`Boolean`): Should the user's first issue ever
      be excluded from this count. (default: `false`)
    * `exclude_popular` (`Boolean`): Should the user's most commented
      issue be excluded from this count. (default: `false`)
    """

    total_pull_request_contributions = sgqlc.types.Field(
        sgqlc.types.non_null(Int),
        graphql_name="totalPullRequestContributions",
        args=sgqlc.types.ArgDict(
            (
                ("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),
                ("exclude_popular", sgqlc.types.Arg(Boolean, graphql_name="excludePopular", default=False)),
            )
        ),
    )
    """How many pull requests the user opened.

    Arguments:

    * `exclude_first` (`Boolean`): Should the user's first pull
      request ever be excluded from this count. (default: `false`)
    * `exclude_popular` (`Boolean`): Should the user's most commented
      pull request be excluded from this count. (default: `false`)
    """

    total_pull_request_review_contributions = sgqlc.types.Field(
        sgqlc.types.non_null(Int), graphql_name="totalPullRequestReviewContributions"
    )
    """How many pull request reviews the user left."""

    total_repositories_with_contributed_commits = sgqlc.types.Field(
        sgqlc.types.non_null(Int), graphql_name="totalRepositoriesWithContributedCommits"
    )
    """How many different repositories the user committed to."""

    total_repositories_with_contributed_issues = sgqlc.types.Field(
        sgqlc.types.non_null(Int),
        graphql_name="totalRepositoriesWithContributedIssues",
        args=sgqlc.types.ArgDict(
            (
                ("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),
                ("exclude_popular", sgqlc.types.Arg(Boolean, graphql_name="excludePopular", default=False)),
            )
        ),
    )
    """How many different repositories the user opened issues in.

    Arguments:

    * `exclude_first` (`Boolean`): Should the user's first issue ever
      be excluded from this count. (default: `false`)
    * `exclude_popular` (`Boolean`): Should the user's most commented
      issue be excluded from this count. (default: `false`)
    """

    total_repositories_with_contributed_pull_request_reviews = sgqlc.types.Field(
        sgqlc.types.non_null(Int), graphql_name="totalRepositoriesWithContributedPullRequestReviews"
    )
    """How many different repositories the user left pull request reviews
    in.
    """

    total_repositories_with_contributed_pull_requests = sgqlc.types.Field(
        sgqlc.types.non_null(Int),
        graphql_name="totalRepositoriesWithContributedPullRequests",
        args=sgqlc.types.ArgDict(
            (
                ("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),
                ("exclude_popular", sgqlc.types.Arg(Boolean, graphql_name="excludePopular", default=False)),
            )
        ),
    )
    """How many different repositories the user opened pull requests in.

    Arguments:

    * `exclude_first` (`Boolean`): Should the user's first pull
      request ever be excluded from this count. (default: `false`)
    * `exclude_popular` (`Boolean`): Should the user's most commented
      pull request be excluded from this count. (default: `false`)
    """

    total_repository_contributions = sgqlc.types.Field(
        sgqlc.types.non_null(Int),
        graphql_name="totalRepositoryContributions",
        args=sgqlc.types.ArgDict((("exclude_first", sgqlc.types.Arg(Boolean, graphql_name="excludeFirst", default=False)),)),
    )
    """How many repositories the user created.

    Arguments:

    * `exclude_first` (`Boolean`): Should the user's first repository
      ever be excluded from this count. (default: `false`)
    """

    user = sgqlc.types.Field(sgqlc.types.non_null("User"), graphql_name="user")
    """The user who made the contributions in this collection."""


class ConvertProjectCardNoteToIssuePayload(sgqlc.types.Type):
    """Autogenerated return type of ConvertProjectCardNoteToIssue"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project_card")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_card = sgqlc.types.Field("ProjectCard", graphql_name="projectCard")
    """The updated ProjectCard."""


class ConvertPullRequestToDraftPayload(sgqlc.types.Type):
    """Autogenerated return type of ConvertPullRequestToDraft"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The pull request that is now a draft."""


class CopyProjectV2Payload(sgqlc.types.Type):
    """Autogenerated return type of CopyProjectV2"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project_v2")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_v2 = sgqlc.types.Field("ProjectV2", graphql_name="projectV2")
    """The copied project."""


class CreateAttributionInvitationPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateAttributionInvitation"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "owner", "source", "target")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    owner = sgqlc.types.Field("Organization", graphql_name="owner")
    """The owner scoping the reattributable data."""

    source = sgqlc.types.Field("Claimable", graphql_name="source")
    """The account owning the data to reattribute."""

    target = sgqlc.types.Field("Claimable", graphql_name="target")
    """The account which may claim the data."""


class CreateBranchProtectionRulePayload(sgqlc.types.Type):
    """Autogenerated return type of CreateBranchProtectionRule"""

    __schema__ = github_schema
    __field_names__ = ("branch_protection_rule", "client_mutation_id")
    branch_protection_rule = sgqlc.types.Field("BranchProtectionRule", graphql_name="branchProtectionRule")
    """The newly created BranchProtectionRule."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateCheckRunPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateCheckRun"""

    __schema__ = github_schema
    __field_names__ = ("check_run", "client_mutation_id")
    check_run = sgqlc.types.Field("CheckRun", graphql_name="checkRun")
    """The newly created check run."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateCheckSuitePayload(sgqlc.types.Type):
    """Autogenerated return type of CreateCheckSuite"""

    __schema__ = github_schema
    __field_names__ = ("check_suite", "client_mutation_id")
    check_suite = sgqlc.types.Field("CheckSuite", graphql_name="checkSuite")
    """The newly created check suite."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class CreateCommitOnBranchPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateCommitOnBranch"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "commit", "ref")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    commit = sgqlc.types.Field("Commit", graphql_name="commit")
    """The new commit."""

    ref = sgqlc.types.Field("Ref", graphql_name="ref")
    """The ref which has been updated to point to the new commit."""


class CreateDiscussionPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "discussion")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    discussion = sgqlc.types.Field("Discussion", graphql_name="discussion")
    """The discussion that was just created."""


class CreateEnterpriseOrganizationPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateEnterpriseOrganization"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "enterprise", "organization")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    enterprise = sgqlc.types.Field("Enterprise", graphql_name="enterprise")
    """The enterprise that owns the created organization."""

    organization = sgqlc.types.Field("Organization", graphql_name="organization")
    """The organization that was created."""


class CreateEnvironmentPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateEnvironment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "environment")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    environment = sgqlc.types.Field("Environment", graphql_name="environment")
    """The new or existing environment."""


class CreateIpAllowListEntryPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateIpAllowListEntry"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "ip_allow_list_entry")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    ip_allow_list_entry = sgqlc.types.Field("IpAllowListEntry", graphql_name="ipAllowListEntry")
    """The IP allow list entry that was created."""


class CreateIssuePayload(sgqlc.types.Type):
    """Autogenerated return type of CreateIssue"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "issue")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    issue = sgqlc.types.Field("Issue", graphql_name="issue")
    """The new issue."""


class CreateLinkedBranchPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateLinkedBranch"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "linked_branch")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    linked_branch = sgqlc.types.Field("LinkedBranch", graphql_name="linkedBranch")
    """The new branch issue reference."""


class CreateMigrationSourcePayload(sgqlc.types.Type):
    """Autogenerated return type of CreateMigrationSource"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "migration_source")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    migration_source = sgqlc.types.Field("MigrationSource", graphql_name="migrationSource")
    """The created migration source."""


class CreateProjectPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateProject"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project = sgqlc.types.Field("Project", graphql_name="project")
    """The new project."""


class CreateProjectV2FieldPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateProjectV2Field"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project_v2_field")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_v2_field = sgqlc.types.Field("ProjectV2FieldConfiguration", graphql_name="projectV2Field")
    """The new field."""


class CreateProjectV2Payload(sgqlc.types.Type):
    """Autogenerated return type of CreateProjectV2"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project_v2")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_v2 = sgqlc.types.Field("ProjectV2", graphql_name="projectV2")
    """The new project."""


class CreatePullRequestPayload(sgqlc.types.Type):
    """Autogenerated return type of CreatePullRequest"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The new pull request."""


class CreateRefPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateRef"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "ref")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    ref = sgqlc.types.Field("Ref", graphql_name="ref")
    """The newly created ref."""


class CreateRepositoryPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateRepository"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "repository")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    repository = sgqlc.types.Field("Repository", graphql_name="repository")
    """The new repository."""


class CreateRepositoryRulesetPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateRepositoryRuleset"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "ruleset")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    ruleset = sgqlc.types.Field("RepositoryRuleset", graphql_name="ruleset")
    """The newly created Ruleset."""


class CreateSponsorsListingPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateSponsorsListing"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "sponsors_listing")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    sponsors_listing = sgqlc.types.Field("SponsorsListing", graphql_name="sponsorsListing")
    """The new GitHub Sponsors profile."""


class CreateSponsorsTierPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateSponsorsTier"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "sponsors_tier")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    sponsors_tier = sgqlc.types.Field("SponsorsTier", graphql_name="sponsorsTier")
    """The new tier."""


class CreateSponsorshipPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateSponsorship"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "sponsorship")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    sponsorship = sgqlc.types.Field("Sponsorship", graphql_name="sponsorship")
    """The sponsorship that was started."""


class CreateSponsorshipsPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateSponsorships"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "sponsorables")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    sponsorables = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(Sponsorable)), graphql_name="sponsorables")
    """The users and organizations who received a sponsorship."""


class CreateTeamDiscussionCommentPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateTeamDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "team_discussion_comment")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    team_discussion_comment = sgqlc.types.Field("TeamDiscussionComment", graphql_name="teamDiscussionComment")
    """The new comment."""


class CreateTeamDiscussionPayload(sgqlc.types.Type):
    """Autogenerated return type of CreateTeamDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "team_discussion")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    team_discussion = sgqlc.types.Field("TeamDiscussion", graphql_name="teamDiscussion")
    """The new discussion."""


class CreatedCommitContributionConnection(sgqlc.types.relay.Connection):
    """The connection type for CreatedCommitContribution."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CreatedCommitContributionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("CreatedCommitContribution"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of commits across days and repositories
    in the connection.
    """


class CreatedCommitContributionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("CreatedCommitContribution", graphql_name="node")
    """The item at the end of the edge."""


class CreatedIssueContributionConnection(sgqlc.types.relay.Connection):
    """The connection type for CreatedIssueContribution."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CreatedIssueContributionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("CreatedIssueContribution"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CreatedIssueContributionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("CreatedIssueContribution", graphql_name="node")
    """The item at the end of the edge."""


class CreatedPullRequestContributionConnection(sgqlc.types.relay.Connection):
    """The connection type for CreatedPullRequestContribution."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CreatedPullRequestContributionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("CreatedPullRequestContribution"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CreatedPullRequestContributionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("CreatedPullRequestContribution", graphql_name="node")
    """The item at the end of the edge."""


class CreatedPullRequestReviewContributionConnection(sgqlc.types.relay.Connection):
    """The connection type for CreatedPullRequestReviewContribution."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CreatedPullRequestReviewContributionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("CreatedPullRequestReviewContribution"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CreatedPullRequestReviewContributionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("CreatedPullRequestReviewContribution", graphql_name="node")
    """The item at the end of the edge."""


class CreatedRepositoryContributionConnection(sgqlc.types.relay.Connection):
    """The connection type for CreatedRepositoryContribution."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("CreatedRepositoryContributionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("CreatedRepositoryContribution"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class CreatedRepositoryContributionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("CreatedRepositoryContribution", graphql_name="node")
    """The item at the end of the edge."""


class DeclineTopicSuggestionPayload(sgqlc.types.Type):
    """Autogenerated return type of DeclineTopicSuggestion"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "topic")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    topic = sgqlc.types.Field("Topic", graphql_name="topic")
    """The declined topic."""


class DeleteBranchProtectionRulePayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteBranchProtectionRule"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id",)
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteDeploymentPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteDeployment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id",)
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteDiscussionCommentPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "comment")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    comment = sgqlc.types.Field("DiscussionComment", graphql_name="comment")
    """The discussion comment that was just deleted."""


class DeleteDiscussionPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "discussion")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    discussion = sgqlc.types.Field("Discussion", graphql_name="discussion")
    """The discussion that was just deleted."""


class DeleteEnvironmentPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteEnvironment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id",)
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteIpAllowListEntryPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteIpAllowListEntry"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "ip_allow_list_entry")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    ip_allow_list_entry = sgqlc.types.Field("IpAllowListEntry", graphql_name="ipAllowListEntry")
    """The IP allow list entry that was deleted."""


class DeleteIssueCommentPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteIssueComment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id",)
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteIssuePayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteIssue"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "repository")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    repository = sgqlc.types.Field("Repository", graphql_name="repository")
    """The repository the issue belonged to"""


class DeleteLinkedBranchPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteLinkedBranch"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "issue")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    issue = sgqlc.types.Field("Issue", graphql_name="issue")
    """The issue the linked branch was unlinked from."""


class DeleteProjectCardPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteProjectCard"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "column", "deleted_card_id")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    column = sgqlc.types.Field("ProjectColumn", graphql_name="column")
    """The column the deleted card was in."""

    deleted_card_id = sgqlc.types.Field(ID, graphql_name="deletedCardId")
    """The deleted card ID."""


class DeleteProjectColumnPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteProjectColumn"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "deleted_column_id", "project")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    deleted_column_id = sgqlc.types.Field(ID, graphql_name="deletedColumnId")
    """The deleted column ID."""

    project = sgqlc.types.Field("Project", graphql_name="project")
    """The project the deleted column was in."""


class DeleteProjectPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteProject"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "owner")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    owner = sgqlc.types.Field(ProjectOwner, graphql_name="owner")
    """The repository or organization the project was removed from."""


class DeleteProjectV2FieldPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteProjectV2Field"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project_v2_field")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_v2_field = sgqlc.types.Field("ProjectV2FieldConfiguration", graphql_name="projectV2Field")
    """The deleted field."""


class DeleteProjectV2ItemPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteProjectV2Item"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "deleted_item_id")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    deleted_item_id = sgqlc.types.Field(ID, graphql_name="deletedItemId")
    """The ID of the deleted item."""


class DeleteProjectV2Payload(sgqlc.types.Type):
    """Autogenerated return type of DeleteProjectV2"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project_v2")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_v2 = sgqlc.types.Field("ProjectV2", graphql_name="projectV2")
    """The deleted Project."""


class DeleteProjectV2WorkflowPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteProjectV2Workflow"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "deleted_workflow_id", "project_v2")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    deleted_workflow_id = sgqlc.types.Field(ID, graphql_name="deletedWorkflowId")
    """The ID of the deleted workflow."""

    project_v2 = sgqlc.types.Field("ProjectV2", graphql_name="projectV2")
    """The project the deleted workflow was in."""


class DeletePullRequestReviewCommentPayload(sgqlc.types.Type):
    """Autogenerated return type of DeletePullRequestReviewComment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request_review", "pull_request_review_comment")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request_review = sgqlc.types.Field("PullRequestReview", graphql_name="pullRequestReview")
    """The pull request review the deleted comment belonged to."""

    pull_request_review_comment = sgqlc.types.Field("PullRequestReviewComment", graphql_name="pullRequestReviewComment")
    """The deleted pull request review comment."""


class DeletePullRequestReviewPayload(sgqlc.types.Type):
    """Autogenerated return type of DeletePullRequestReview"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request_review")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request_review = sgqlc.types.Field("PullRequestReview", graphql_name="pullRequestReview")
    """The deleted pull request review."""


class DeleteRefPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteRef"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id",)
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteRepositoryRulesetPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteRepositoryRuleset"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id",)
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteTeamDiscussionCommentPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteTeamDiscussionComment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id",)
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteTeamDiscussionPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteTeamDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id",)
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class DeleteVerifiableDomainPayload(sgqlc.types.Type):
    """Autogenerated return type of DeleteVerifiableDomain"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "owner")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    owner = sgqlc.types.Field("VerifiableDomainOwner", graphql_name="owner")
    """The owning account from which the domain was deleted."""


class DependabotUpdateError(sgqlc.types.Type):
    """An error produced from a Dependabot Update"""

    __schema__ = github_schema
    __field_names__ = ("body", "error_type", "title")
    body = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="body")
    """The body of the error"""

    error_type = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="errorType")
    """The error code"""

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """The title of the error"""


class DeployKeyConnection(sgqlc.types.relay.Connection):
    """The connection type for DeployKey."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DeployKeyEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("DeployKey"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DeployKeyEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("DeployKey", graphql_name="node")
    """The item at the end of the edge."""


class DeploymentConnection(sgqlc.types.relay.Connection):
    """The connection type for Deployment."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DeploymentEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Deployment"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DeploymentEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Deployment", graphql_name="node")
    """The item at the end of the edge."""


class DeploymentProtectionRule(sgqlc.types.Type):
    """A protection rule."""

    __schema__ = github_schema
    __field_names__ = ("database_id", "reviewers", "timeout", "type")
    database_id = sgqlc.types.Field(Int, graphql_name="databaseId")
    """Identifies the primary key from the database."""

    reviewers = sgqlc.types.Field(
        sgqlc.types.non_null("DeploymentReviewerConnection"),
        graphql_name="reviewers",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """The teams or users that can review the deployment

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    timeout = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="timeout")
    """The timeout in minutes for this protection rule."""

    type = sgqlc.types.Field(sgqlc.types.non_null(DeploymentProtectionRuleType), graphql_name="type")
    """The type of protection rule."""


class DeploymentProtectionRuleConnection(sgqlc.types.relay.Connection):
    """The connection type for DeploymentProtectionRule."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DeploymentProtectionRuleEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(DeploymentProtectionRule), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DeploymentProtectionRuleEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(DeploymentProtectionRule, graphql_name="node")
    """The item at the end of the edge."""


class DeploymentRequest(sgqlc.types.Type):
    """A request to deploy a workflow run to an environment."""

    __schema__ = github_schema
    __field_names__ = ("current_user_can_approve", "environment", "reviewers", "wait_timer", "wait_timer_started_at")
    current_user_can_approve = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="currentUserCanApprove")
    """Whether or not the current user can approve the deployment"""

    environment = sgqlc.types.Field(sgqlc.types.non_null("Environment"), graphql_name="environment")
    """The target environment of the deployment"""

    reviewers = sgqlc.types.Field(
        sgqlc.types.non_null("DeploymentReviewerConnection"),
        graphql_name="reviewers",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """The teams or users that can review the deployment

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    wait_timer = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="waitTimer")
    """The wait timer in minutes configured in the environment"""

    wait_timer_started_at = sgqlc.types.Field(DateTime, graphql_name="waitTimerStartedAt")
    """The wait timer in minutes configured in the environment"""


class DeploymentRequestConnection(sgqlc.types.relay.Connection):
    """The connection type for DeploymentRequest."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DeploymentRequestEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(DeploymentRequest), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DeploymentRequestEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(DeploymentRequest, graphql_name="node")
    """The item at the end of the edge."""


class DeploymentReviewConnection(sgqlc.types.relay.Connection):
    """The connection type for DeploymentReview."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DeploymentReviewEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("DeploymentReview"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DeploymentReviewEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("DeploymentReview", graphql_name="node")
    """The item at the end of the edge."""


class DeploymentReviewerConnection(sgqlc.types.relay.Connection):
    """The connection type for DeploymentReviewer."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DeploymentReviewerEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("DeploymentReviewer"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DeploymentReviewerEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("DeploymentReviewer", graphql_name="node")
    """The item at the end of the edge."""


class DeploymentStatusConnection(sgqlc.types.relay.Connection):
    """The connection type for DeploymentStatus."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DeploymentStatusEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("DeploymentStatus"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DeploymentStatusEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("DeploymentStatus", graphql_name="node")
    """The item at the end of the edge."""


class DequeuePullRequestPayload(sgqlc.types.Type):
    """Autogenerated return type of DequeuePullRequest"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "merge_queue_entry")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    merge_queue_entry = sgqlc.types.Field("MergeQueueEntry", graphql_name="mergeQueueEntry")
    """The merge queue entry of the dequeued pull request."""


class DisablePullRequestAutoMergePayload(sgqlc.types.Type):
    """Autogenerated return type of DisablePullRequestAutoMerge"""

    __schema__ = github_schema
    __field_names__ = ("actor", "client_mutation_id", "pull_request")
    actor = sgqlc.types.Field(Actor, graphql_name="actor")
    """Identifies the actor who performed the event."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The pull request auto merge was disabled on."""


class DiscussionCategoryConnection(sgqlc.types.relay.Connection):
    """The connection type for DiscussionCategory."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DiscussionCategoryEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("DiscussionCategory"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DiscussionCategoryEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("DiscussionCategory", graphql_name="node")
    """The item at the end of the edge."""


class DiscussionCommentConnection(sgqlc.types.relay.Connection):
    """The connection type for DiscussionComment."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DiscussionCommentEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("DiscussionComment"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DiscussionCommentEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("DiscussionComment", graphql_name="node")
    """The item at the end of the edge."""


class DiscussionConnection(sgqlc.types.relay.Connection):
    """The connection type for Discussion."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DiscussionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Discussion"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DiscussionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Discussion", graphql_name="node")
    """The item at the end of the edge."""


class DiscussionPollOptionConnection(sgqlc.types.relay.Connection):
    """The connection type for DiscussionPollOption."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("DiscussionPollOptionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("DiscussionPollOption"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class DiscussionPollOptionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("DiscussionPollOption", graphql_name="node")
    """The item at the end of the edge."""


class DismissPullRequestReviewPayload(sgqlc.types.Type):
    """Autogenerated return type of DismissPullRequestReview"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request_review")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request_review = sgqlc.types.Field("PullRequestReview", graphql_name="pullRequestReview")
    """The dismissed pull request review."""


class DismissRepositoryVulnerabilityAlertPayload(sgqlc.types.Type):
    """Autogenerated return type of DismissRepositoryVulnerabilityAlert"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "repository_vulnerability_alert")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    repository_vulnerability_alert = sgqlc.types.Field("RepositoryVulnerabilityAlert", graphql_name="repositoryVulnerabilityAlert")
    """The Dependabot alert that was dismissed"""


class EnablePullRequestAutoMergePayload(sgqlc.types.Type):
    """Autogenerated return type of EnablePullRequestAutoMerge"""

    __schema__ = github_schema
    __field_names__ = ("actor", "client_mutation_id", "pull_request")
    actor = sgqlc.types.Field(Actor, graphql_name="actor")
    """Identifies the actor who performed the event."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The pull request auto-merge was enabled on."""


class EnqueuePullRequestPayload(sgqlc.types.Type):
    """Autogenerated return type of EnqueuePullRequest"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "merge_queue_entry")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    merge_queue_entry = sgqlc.types.Field("MergeQueueEntry", graphql_name="mergeQueueEntry")
    """The merge queue entry for the enqueued pull request."""


class EnterpriseAdministratorConnection(sgqlc.types.relay.Connection):
    """The connection type for User."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseAdministratorEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("User"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseAdministratorEdge(sgqlc.types.Type):
    """A User who is an administrator of an enterprise."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "role")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("User", graphql_name="node")
    """The item at the end of the edge."""

    role = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseAdministratorRole), graphql_name="role")
    """The role of the administrator."""


class EnterpriseAdministratorInvitationConnection(sgqlc.types.relay.Connection):
    """The connection type for EnterpriseAdministratorInvitation."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseAdministratorInvitationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseAdministratorInvitation"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseAdministratorInvitationEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("EnterpriseAdministratorInvitation", graphql_name="node")
    """The item at the end of the edge."""


class EnterpriseBillingInfo(sgqlc.types.Type):
    """Enterprise billing information visible to enterprise billing
    managers and owners.
    """

    __schema__ = github_schema
    __field_names__ = (
        "all_licensable_users_count",
        "asset_packs",
        "bandwidth_quota",
        "bandwidth_usage",
        "bandwidth_usage_percentage",
        "storage_quota",
        "storage_usage",
        "storage_usage_percentage",
        "total_available_licenses",
        "total_licenses",
    )
    all_licensable_users_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="allLicensableUsersCount")
    """The number of licenseable users/emails across the enterprise."""

    asset_packs = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="assetPacks")
    """The number of data packs used by all organizations owned by the
    enterprise.
    """

    bandwidth_quota = sgqlc.types.Field(sgqlc.types.non_null(Float), graphql_name="bandwidthQuota")
    """The bandwidth quota in GB for all organizations owned by the
    enterprise.
    """

    bandwidth_usage = sgqlc.types.Field(sgqlc.types.non_null(Float), graphql_name="bandwidthUsage")
    """The bandwidth usage in GB for all organizations owned by the
    enterprise.
    """

    bandwidth_usage_percentage = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="bandwidthUsagePercentage")
    """The bandwidth usage as a percentage of the bandwidth quota."""

    storage_quota = sgqlc.types.Field(sgqlc.types.non_null(Float), graphql_name="storageQuota")
    """The storage quota in GB for all organizations owned by the
    enterprise.
    """

    storage_usage = sgqlc.types.Field(sgqlc.types.non_null(Float), graphql_name="storageUsage")
    """The storage usage in GB for all organizations owned by the
    enterprise.
    """

    storage_usage_percentage = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="storageUsagePercentage")
    """The storage usage as a percentage of the storage quota."""

    total_available_licenses = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalAvailableLicenses")
    """The number of available licenses across all owned organizations
    based on the unique number of billable users.
    """

    total_licenses = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalLicenses")
    """The total number of licenses allocated."""


class EnterpriseFailedInvitationConnection(sgqlc.types.relay.Connection):
    """The connection type for OrganizationInvitation."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count", "total_unique_user_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseFailedInvitationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("OrganizationInvitation"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""

    total_unique_user_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalUniqueUserCount")
    """Identifies the total count of unique users in the connection."""


class EnterpriseFailedInvitationEdge(sgqlc.types.Type):
    """A failed invitation to be a member in an enterprise organization."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("OrganizationInvitation", graphql_name="node")
    """The item at the end of the edge."""


class EnterpriseMemberConnection(sgqlc.types.relay.Connection):
    """The connection type for EnterpriseMember."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseMemberEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseMember"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseMemberEdge(sgqlc.types.Type):
    """A User who is a member of an enterprise through one or more
    organizations.
    """

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("EnterpriseMember", graphql_name="node")
    """The item at the end of the edge."""


class EnterpriseOrganizationMembershipConnection(sgqlc.types.relay.Connection):
    """The connection type for Organization."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseOrganizationMembershipEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Organization"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseOrganizationMembershipEdge(sgqlc.types.Type):
    """An enterprise organization that a user is a member of."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "role")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Organization", graphql_name="node")
    """The item at the end of the edge."""

    role = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseUserAccountMembershipRole), graphql_name="role")
    """The role of the user in the enterprise membership."""


class EnterpriseOutsideCollaboratorConnection(sgqlc.types.relay.Connection):
    """The connection type for User."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseOutsideCollaboratorEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("User"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseOutsideCollaboratorEdge(sgqlc.types.Type):
    """A User who is an outside collaborator of an enterprise through one
    or more organizations.
    """

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "repositories")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("User", graphql_name="node")
    """The item at the end of the edge."""

    repositories = sgqlc.types.Field(
        sgqlc.types.non_null("EnterpriseRepositoryInfoConnection"),
        graphql_name="repositories",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("order_by", sgqlc.types.Arg(RepositoryOrder, graphql_name="orderBy", default={"field": "NAME", "direction": "ASC"})),
            )
        ),
    )
    """The enterprise organization repositories this user is a member of.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`RepositoryOrder`): Ordering options for
      repositories. (default: `{field: NAME, direction: ASC}`)
    """


class EnterpriseOwnerInfo(sgqlc.types.Type):
    """Enterprise information visible to enterprise owners or enterprise
    owners' personal access tokens (classic) with read:enterprise or
    admin:enterprise scope.
    """

    __schema__ = github_schema
    __field_names__ = (
        "admins",
        "affiliated_users_with_two_factor_disabled",
        "affiliated_users_with_two_factor_disabled_exist",
        "allow_private_repository_forking_setting",
        "allow_private_repository_forking_setting_organizations",
        "allow_private_repository_forking_setting_policy_value",
        "default_repository_permission_setting",
        "default_repository_permission_setting_organizations",
        "domains",
        "enterprise_server_installations",
        "failed_invitations",
        "ip_allow_list_enabled_setting",
        "ip_allow_list_entries",
        "ip_allow_list_for_installed_apps_enabled_setting",
        "is_updating_default_repository_permission",
        "is_updating_two_factor_requirement",
        "members_can_change_repository_visibility_setting",
        "members_can_change_repository_visibility_setting_organizations",
        "members_can_create_internal_repositories_setting",
        "members_can_create_private_repositories_setting",
        "members_can_create_public_repositories_setting",
        "members_can_create_repositories_setting",
        "members_can_create_repositories_setting_organizations",
        "members_can_delete_issues_setting",
        "members_can_delete_issues_setting_organizations",
        "members_can_delete_repositories_setting",
        "members_can_delete_repositories_setting_organizations",
        "members_can_invite_collaborators_setting",
        "members_can_invite_collaborators_setting_organizations",
        "members_can_make_purchases_setting",
        "members_can_update_protected_branches_setting",
        "members_can_update_protected_branches_setting_organizations",
        "members_can_view_dependency_insights_setting",
        "members_can_view_dependency_insights_setting_organizations",
        "notification_delivery_restriction_enabled_setting",
        "oidc_provider",
        "organization_projects_setting",
        "organization_projects_setting_organizations",
        "outside_collaborators",
        "pending_admin_invitations",
        "pending_collaborator_invitations",
        "pending_member_invitations",
        "repository_projects_setting",
        "repository_projects_setting_organizations",
        "saml_identity_provider",
        "saml_identity_provider_setting_organizations",
        "support_entitlements",
        "team_discussions_setting",
        "team_discussions_setting_organizations",
        "two_factor_required_setting",
        "two_factor_required_setting_organizations",
    )
    admins = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseAdministratorConnection),
        graphql_name="admins",
        args=sgqlc.types.ArgDict(
            (
                (
                    "organization_logins",
                    sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="organizationLogins", default=None),
                ),
                ("query", sgqlc.types.Arg(String, graphql_name="query", default=None)),
                ("role", sgqlc.types.Arg(EnterpriseAdministratorRole, graphql_name="role", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(EnterpriseMemberOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"}),
                ),
                ("has_two_factor_enabled", sgqlc.types.Arg(Boolean, graphql_name="hasTwoFactorEnabled", default=None)),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of all of the administrators for this enterprise.

    Arguments:

    * `organization_logins` (`[String!]`): Only return members within
      the organizations with these logins
    * `query` (`String`): The search string to look for.
    * `role` (`EnterpriseAdministratorRole`): The role to filter by.
    * `order_by` (`EnterpriseMemberOrder`): Ordering options for
      administrators returned from the connection. (default: `{field:
      LOGIN, direction: ASC}`)
    * `has_two_factor_enabled` (`Boolean`): Only return administrators
      with this two-factor authentication status. (default: `null`)
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    affiliated_users_with_two_factor_disabled = sgqlc.types.Field(
        sgqlc.types.non_null("UserConnection"),
        graphql_name="affiliatedUsersWithTwoFactorDisabled",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of users in the enterprise who currently have two-factor
    authentication disabled.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    affiliated_users_with_two_factor_disabled_exist = sgqlc.types.Field(
        sgqlc.types.non_null(Boolean), graphql_name="affiliatedUsersWithTwoFactorDisabledExist"
    )
    """Whether or not affiliated users with two-factor authentication
    disabled exist in the enterprise.
    """

    allow_private_repository_forking_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="allowPrivateRepositoryForkingSetting"
    )
    """The setting value for whether private repository forking is
    enabled for repositories in organizations in this enterprise.
    """

    allow_private_repository_forking_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="allowPrivateRepositoryForkingSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    private repository forking setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    allow_private_repository_forking_setting_policy_value = sgqlc.types.Field(
        EnterpriseAllowPrivateRepositoryForkingPolicyValue, graphql_name="allowPrivateRepositoryForkingSettingPolicyValue"
    )
    """The value for the allow private repository forking policy on the
    enterprise.
    """

    default_repository_permission_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseDefaultRepositoryPermissionSettingValue), graphql_name="defaultRepositoryPermissionSetting"
    )
    """The setting value for base repository permissions for
    organizations in this enterprise.
    """

    default_repository_permission_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="defaultRepositoryPermissionSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(DefaultRepositoryPermissionField), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    base repository permission.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`DefaultRepositoryPermissionField!`): The permission to
      find organizations for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    domains = sgqlc.types.Field(
        sgqlc.types.non_null("VerifiableDomainConnection"),
        graphql_name="domains",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("is_verified", sgqlc.types.Arg(Boolean, graphql_name="isVerified", default=None)),
                ("is_approved", sgqlc.types.Arg(Boolean, graphql_name="isApproved", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(VerifiableDomainOrder, graphql_name="orderBy", default={"field": "DOMAIN", "direction": "ASC"}),
                ),
            )
        ),
    )
    """A list of domains owned by the enterprise. Visible to enterprise
    owners or enterprise owners' personal access tokens (classic) with
    admin:enterprise scope.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `is_verified` (`Boolean`): Filter whether or not the domain is
      verified. (default: `null`)
    * `is_approved` (`Boolean`): Filter whether or not the domain is
      approved. (default: `null`)
    * `order_by` (`VerifiableDomainOrder`): Ordering options for
      verifiable domains returned. (default: `{field: DOMAIN,
      direction: ASC}`)
    """

    enterprise_server_installations = sgqlc.types.Field(
        sgqlc.types.non_null("EnterpriseServerInstallationConnection"),
        graphql_name="enterpriseServerInstallations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("connected_only", sgqlc.types.Arg(Boolean, graphql_name="connectedOnly", default=False)),
                (
                    "order_by",
                    sgqlc.types.Arg(
                        EnterpriseServerInstallationOrder, graphql_name="orderBy", default={"field": "HOST_NAME", "direction": "ASC"}
                    ),
                ),
            )
        ),
    )
    """Enterprise Server installations owned by the enterprise.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `connected_only` (`Boolean`): Whether or not to only return
      installations discovered via GitHub Connect. (default: `false`)
    * `order_by` (`EnterpriseServerInstallationOrder`): Ordering
      options for Enterprise Server installations returned. (default:
      `{field: HOST_NAME, direction: ASC}`)
    """

    failed_invitations = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseFailedInvitationConnection),
        graphql_name="failedInvitations",
        args=sgqlc.types.ArgDict(
            (
                ("query", sgqlc.types.Arg(String, graphql_name="query", default=None)),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of failed invitations in the enterprise.

    Arguments:

    * `query` (`String`): The search string to look for.
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    ip_allow_list_enabled_setting = sgqlc.types.Field(
        sgqlc.types.non_null(IpAllowListEnabledSettingValue), graphql_name="ipAllowListEnabledSetting"
    )
    """The setting value for whether the enterprise has an IP allow list
    enabled.
    """

    ip_allow_list_entries = sgqlc.types.Field(
        sgqlc.types.non_null("IpAllowListEntryConnection"),
        graphql_name="ipAllowListEntries",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(
                        IpAllowListEntryOrder, graphql_name="orderBy", default={"field": "ALLOW_LIST_VALUE", "direction": "ASC"}
                    ),
                ),
            )
        ),
    )
    """The IP addresses that are allowed to access resources owned by the
    enterprise. Visible to enterprise owners or enterprise owners'
    personal access tokens (classic) with admin:enterprise scope.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`IpAllowListEntryOrder`): Ordering options for IP
      allow list entries returned. (default: `{field:
      ALLOW_LIST_VALUE, direction: ASC}`)
    """

    ip_allow_list_for_installed_apps_enabled_setting = sgqlc.types.Field(
        sgqlc.types.non_null(IpAllowListForInstalledAppsEnabledSettingValue), graphql_name="ipAllowListForInstalledAppsEnabledSetting"
    )
    """The setting value for whether the enterprise has IP allow list
    configuration for installed GitHub Apps enabled.
    """

    is_updating_default_repository_permission = sgqlc.types.Field(
        sgqlc.types.non_null(Boolean), graphql_name="isUpdatingDefaultRepositoryPermission"
    )
    """Whether or not the base repository permission is currently being
    updated.
    """

    is_updating_two_factor_requirement = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isUpdatingTwoFactorRequirement")
    """Whether the two-factor authentication requirement is currently
    being enforced.
    """

    members_can_change_repository_visibility_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="membersCanChangeRepositoryVisibilitySetting"
    )
    """The setting value for whether organization members with admin
    permissions on a repository can change repository visibility.
    """

    members_can_change_repository_visibility_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="membersCanChangeRepositoryVisibilitySettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    can change repository visibility setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    members_can_create_internal_repositories_setting = sgqlc.types.Field(
        Boolean, graphql_name="membersCanCreateInternalRepositoriesSetting"
    )
    """The setting value for whether members of organizations in the
    enterprise can create internal repositories.
    """

    members_can_create_private_repositories_setting = sgqlc.types.Field(Boolean, graphql_name="membersCanCreatePrivateRepositoriesSetting")
    """The setting value for whether members of organizations in the
    enterprise can create private repositories.
    """

    members_can_create_public_repositories_setting = sgqlc.types.Field(Boolean, graphql_name="membersCanCreatePublicRepositoriesSetting")
    """The setting value for whether members of organizations in the
    enterprise can create public repositories.
    """

    members_can_create_repositories_setting = sgqlc.types.Field(
        EnterpriseMembersCanCreateRepositoriesSettingValue, graphql_name="membersCanCreateRepositoriesSetting"
    )
    """The setting value for whether members of organizations in the
    enterprise can create repositories.
    """

    members_can_create_repositories_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="membersCanCreateRepositoriesSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                (
                    "value",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(OrganizationMembersCanCreateRepositoriesSettingValue), graphql_name="value", default=None
                    ),
                ),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    repository creation setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value`
      (`OrganizationMembersCanCreateRepositoriesSettingValue!`): The
      setting to find organizations for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    members_can_delete_issues_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="membersCanDeleteIssuesSetting"
    )
    """The setting value for whether members with admin permissions for
    repositories can delete issues.
    """

    members_can_delete_issues_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="membersCanDeleteIssuesSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    members can delete issues setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    members_can_delete_repositories_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="membersCanDeleteRepositoriesSetting"
    )
    """The setting value for whether members with admin permissions for
    repositories can delete or transfer repositories.
    """

    members_can_delete_repositories_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="membersCanDeleteRepositoriesSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    members can delete repositories setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    members_can_invite_collaborators_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="membersCanInviteCollaboratorsSetting"
    )
    """The setting value for whether members of organizations in the
    enterprise can invite outside collaborators.
    """

    members_can_invite_collaborators_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="membersCanInviteCollaboratorsSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    members can invite collaborators setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    members_can_make_purchases_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseMembersCanMakePurchasesSettingValue), graphql_name="membersCanMakePurchasesSetting"
    )
    """Indicates whether members of this enterprise's organizations can
    purchase additional services for those organizations.
    """

    members_can_update_protected_branches_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="membersCanUpdateProtectedBranchesSetting"
    )
    """The setting value for whether members with admin permissions for
    repositories can update protected branches.
    """

    members_can_update_protected_branches_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="membersCanUpdateProtectedBranchesSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    members can update protected branches setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    members_can_view_dependency_insights_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="membersCanViewDependencyInsightsSetting"
    )
    """The setting value for whether members can view dependency
    insights.
    """

    members_can_view_dependency_insights_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="membersCanViewDependencyInsightsSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    members can view dependency insights setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    notification_delivery_restriction_enabled_setting = sgqlc.types.Field(
        sgqlc.types.non_null(NotificationRestrictionSettingValue), graphql_name="notificationDeliveryRestrictionEnabledSetting"
    )
    """Indicates if email notification delivery for this enterprise is
    restricted to verified or approved domains.
    """

    oidc_provider = sgqlc.types.Field("OIDCProvider", graphql_name="oidcProvider")
    """The OIDC Identity Provider for the enterprise."""

    organization_projects_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="organizationProjectsSetting"
    )
    """The setting value for whether organization projects are enabled
    for organizations in this enterprise.
    """

    organization_projects_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="organizationProjectsSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    organization projects setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    outside_collaborators = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseOutsideCollaboratorConnection),
        graphql_name="outsideCollaborators",
        args=sgqlc.types.ArgDict(
            (
                ("login", sgqlc.types.Arg(String, graphql_name="login", default=None)),
                ("query", sgqlc.types.Arg(String, graphql_name="query", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(EnterpriseMemberOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"}),
                ),
                ("visibility", sgqlc.types.Arg(RepositoryVisibility, graphql_name="visibility", default=None)),
                ("has_two_factor_enabled", sgqlc.types.Arg(Boolean, graphql_name="hasTwoFactorEnabled", default=None)),
                (
                    "organization_logins",
                    sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="organizationLogins", default=None),
                ),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of outside collaborators across the repositories in the
    enterprise.

    Arguments:

    * `login` (`String`): The login of one specific outside
      collaborator.
    * `query` (`String`): The search string to look for.
    * `order_by` (`EnterpriseMemberOrder`): Ordering options for
      outside collaborators returned from the connection. (default:
      `{field: LOGIN, direction: ASC}`)
    * `visibility` (`RepositoryVisibility`): Only return outside
      collaborators on repositories with this visibility.
    * `has_two_factor_enabled` (`Boolean`): Only return outside
      collaborators with this two-factor authentication status.
      (default: `null`)
    * `organization_logins` (`[String!]`): Only return outside
      collaborators within the organizations with these logins
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    pending_admin_invitations = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseAdministratorInvitationConnection),
        graphql_name="pendingAdminInvitations",
        args=sgqlc.types.ArgDict(
            (
                ("query", sgqlc.types.Arg(String, graphql_name="query", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(
                        EnterpriseAdministratorInvitationOrder, graphql_name="orderBy", default={"field": "CREATED_AT", "direction": "DESC"}
                    ),
                ),
                ("role", sgqlc.types.Arg(EnterpriseAdministratorRole, graphql_name="role", default=None)),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of pending administrator invitations for the enterprise.

    Arguments:

    * `query` (`String`): The search string to look for.
    * `order_by` (`EnterpriseAdministratorInvitationOrder`): Ordering
      options for pending enterprise administrator invitations
      returned from the connection. (default: `{field: CREATED_AT,
      direction: DESC}`)
    * `role` (`EnterpriseAdministratorRole`): The role to filter by.
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    pending_collaborator_invitations = sgqlc.types.Field(
        sgqlc.types.non_null("RepositoryInvitationConnection"),
        graphql_name="pendingCollaboratorInvitations",
        args=sgqlc.types.ArgDict(
            (
                ("query", sgqlc.types.Arg(String, graphql_name="query", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(
                        RepositoryInvitationOrder, graphql_name="orderBy", default={"field": "CREATED_AT", "direction": "DESC"}
                    ),
                ),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of pending collaborator invitations across the repositories
    in the enterprise.

    Arguments:

    * `query` (`String`): The search string to look for.
    * `order_by` (`RepositoryInvitationOrder`): Ordering options for
      pending repository collaborator invitations returned from the
      connection. (default: `{field: CREATED_AT, direction: DESC}`)
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    pending_member_invitations = sgqlc.types.Field(
        sgqlc.types.non_null("EnterprisePendingMemberInvitationConnection"),
        graphql_name="pendingMemberInvitations",
        args=sgqlc.types.ArgDict(
            (
                ("query", sgqlc.types.Arg(String, graphql_name="query", default=None)),
                (
                    "organization_logins",
                    sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="organizationLogins", default=None),
                ),
                ("invitation_source", sgqlc.types.Arg(OrganizationInvitationSource, graphql_name="invitationSource", default=None)),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of pending member invitations for organizations in the
    enterprise.

    Arguments:

    * `query` (`String`): The search string to look for.
    * `organization_logins` (`[String!]`): Only return invitations
      within the organizations with these logins
    * `invitation_source` (`OrganizationInvitationSource`): Only
      return invitations matching this invitation source
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    repository_projects_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="repositoryProjectsSetting"
    )
    """The setting value for whether repository projects are enabled in
    this enterprise.
    """

    repository_projects_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="repositoryProjectsSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    repository projects setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    saml_identity_provider = sgqlc.types.Field("EnterpriseIdentityProvider", graphql_name="samlIdentityProvider")
    """The SAML Identity Provider for the enterprise."""

    saml_identity_provider_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="samlIdentityProviderSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(IdentityProviderConfigurationState), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the SAML single
    sign-on setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`IdentityProviderConfigurationState!`): The setting
      value to find organizations for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    support_entitlements = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseMemberConnection),
        graphql_name="supportEntitlements",
        args=sgqlc.types.ArgDict(
            (
                (
                    "order_by",
                    sgqlc.types.Arg(EnterpriseMemberOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"}),
                ),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """A list of members with a support entitlement.

    Arguments:

    * `order_by` (`EnterpriseMemberOrder`): Ordering options for
      support entitlement users returned from the connection.
      (default: `{field: LOGIN, direction: ASC}`)
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    team_discussions_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledDisabledSettingValue), graphql_name="teamDiscussionsSetting"
    )
    """The setting value for whether team discussions are enabled for
    organizations in this enterprise.
    """

    team_discussions_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="teamDiscussionsSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the provided
    team discussions setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """

    two_factor_required_setting = sgqlc.types.Field(
        sgqlc.types.non_null(EnterpriseEnabledSettingValue), graphql_name="twoFactorRequiredSetting"
    )
    """The setting value for whether the enterprise requires two-factor
    authentication for its organizations and users.
    """

    two_factor_required_setting_organizations = sgqlc.types.Field(
        sgqlc.types.non_null("OrganizationConnection"),
        graphql_name="twoFactorRequiredSettingOrganizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("value", sgqlc.types.Arg(sgqlc.types.non_null(Boolean), graphql_name="value", default=None)),
                ("order_by", sgqlc.types.Arg(OrganizationOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
            )
        ),
    )
    """A list of enterprise organizations configured with the two-factor
    authentication setting value.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `value` (`Boolean!`): The setting value to find organizations
      for.
    * `order_by` (`OrganizationOrder`): Ordering options for
      organizations with this setting. (default: `{field: LOGIN,
      direction: ASC}`)
    """


class EnterprisePendingMemberInvitationConnection(sgqlc.types.relay.Connection):
    """The connection type for OrganizationInvitation."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count", "total_unique_user_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterprisePendingMemberInvitationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("OrganizationInvitation"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""

    total_unique_user_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalUniqueUserCount")
    """Identifies the total count of unique users in the connection."""


class EnterprisePendingMemberInvitationEdge(sgqlc.types.Type):
    """An invitation to be a member in an enterprise organization."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("OrganizationInvitation", graphql_name="node")
    """The item at the end of the edge."""


class EnterpriseRepositoryInfoConnection(sgqlc.types.relay.Connection):
    """The connection type for EnterpriseRepositoryInfo."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseRepositoryInfoEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseRepositoryInfo"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseRepositoryInfoEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("EnterpriseRepositoryInfo", graphql_name="node")
    """The item at the end of the edge."""


class EnterpriseServerInstallationConnection(sgqlc.types.relay.Connection):
    """The connection type for EnterpriseServerInstallation."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerInstallationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerInstallation"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseServerInstallationEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("EnterpriseServerInstallation", graphql_name="node")
    """The item at the end of the edge."""


class EnterpriseServerInstallationMembershipConnection(sgqlc.types.relay.Connection):
    """The connection type for EnterpriseServerInstallation."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerInstallationMembershipEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerInstallation"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseServerInstallationMembershipEdge(sgqlc.types.Type):
    """An Enterprise Server installation that a user is a member of."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "role")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("EnterpriseServerInstallation", graphql_name="node")
    """The item at the end of the edge."""

    role = sgqlc.types.Field(sgqlc.types.non_null(EnterpriseUserAccountMembershipRole), graphql_name="role")
    """The role of the user in the enterprise membership."""


class EnterpriseServerUserAccountConnection(sgqlc.types.relay.Connection):
    """The connection type for EnterpriseServerUserAccount."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerUserAccountEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerUserAccount"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseServerUserAccountEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("EnterpriseServerUserAccount", graphql_name="node")
    """The item at the end of the edge."""


class EnterpriseServerUserAccountEmailConnection(sgqlc.types.relay.Connection):
    """The connection type for EnterpriseServerUserAccountEmail."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerUserAccountEmailEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerUserAccountEmail"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseServerUserAccountEmailEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("EnterpriseServerUserAccountEmail", graphql_name="node")
    """The item at the end of the edge."""


class EnterpriseServerUserAccountsUploadConnection(sgqlc.types.relay.Connection):
    """The connection type for EnterpriseServerUserAccountsUpload."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerUserAccountsUploadEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("EnterpriseServerUserAccountsUpload"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnterpriseServerUserAccountsUploadEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("EnterpriseServerUserAccountsUpload", graphql_name="node")
    """The item at the end of the edge."""


class EnvironmentConnection(sgqlc.types.relay.Connection):
    """The connection type for Environment."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("EnvironmentEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Environment"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class EnvironmentEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Environment", graphql_name="node")
    """The item at the end of the edge."""


class ExternalIdentityAttribute(sgqlc.types.Type):
    """An attribute for the External Identity attributes collection"""

    __schema__ = github_schema
    __field_names__ = ("metadata", "name", "value")
    metadata = sgqlc.types.Field(String, graphql_name="metadata")
    """The attribute metadata as JSON"""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The attribute name"""

    value = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="value")
    """The attribute value"""


class ExternalIdentityConnection(sgqlc.types.relay.Connection):
    """The connection type for ExternalIdentity."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ExternalIdentityEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ExternalIdentity"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ExternalIdentityEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ExternalIdentity", graphql_name="node")
    """The item at the end of the edge."""


class ExternalIdentitySamlAttributes(sgqlc.types.Type):
    """SAML attributes for the External Identity"""

    __schema__ = github_schema
    __field_names__ = ("attributes", "emails", "family_name", "given_name", "groups", "name_id", "username")
    attributes = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ExternalIdentityAttribute))), graphql_name="attributes"
    )
    """SAML Identity attributes"""

    emails = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("UserEmailMetadata")), graphql_name="emails")
    """The emails associated with the SAML identity"""

    family_name = sgqlc.types.Field(String, graphql_name="familyName")
    """Family name of the SAML identity"""

    given_name = sgqlc.types.Field(String, graphql_name="givenName")
    """Given name of the SAML identity"""

    groups = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="groups")
    """The groups linked to this identity in IDP"""

    name_id = sgqlc.types.Field(String, graphql_name="nameId")
    """The NameID of the SAML identity"""

    username = sgqlc.types.Field(String, graphql_name="username")
    """The userName of the SAML identity"""


class ExternalIdentityScimAttributes(sgqlc.types.Type):
    """SCIM attributes for the External Identity"""

    __schema__ = github_schema
    __field_names__ = ("emails", "family_name", "given_name", "groups", "username")
    emails = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("UserEmailMetadata")), graphql_name="emails")
    """The emails associated with the SCIM identity"""

    family_name = sgqlc.types.Field(String, graphql_name="familyName")
    """Family name of the SCIM identity"""

    given_name = sgqlc.types.Field(String, graphql_name="givenName")
    """Given name of the SCIM identity"""

    groups = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="groups")
    """The groups linked to this identity in IDP"""

    username = sgqlc.types.Field(String, graphql_name="username")
    """The userName of the SCIM identity"""


class FollowOrganizationPayload(sgqlc.types.Type):
    """Autogenerated return type of FollowOrganization"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "organization")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    organization = sgqlc.types.Field("Organization", graphql_name="organization")
    """The organization that was followed."""


class FollowUserPayload(sgqlc.types.Type):
    """Autogenerated return type of FollowUser"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "user")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    user = sgqlc.types.Field("User", graphql_name="user")
    """The user that was followed."""


class FollowerConnection(sgqlc.types.relay.Connection):
    """The connection type for User."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("UserEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("User"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class FollowingConnection(sgqlc.types.relay.Connection):
    """The connection type for User."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("UserEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("User"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class FundingLink(sgqlc.types.Type):
    """A funding platform link for a repository."""

    __schema__ = github_schema
    __field_names__ = ("platform", "url")
    platform = sgqlc.types.Field(sgqlc.types.non_null(FundingPlatform), graphql_name="platform")
    """The funding platform this link is for."""

    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """The configured URL for this funding link."""


class GistCommentConnection(sgqlc.types.relay.Connection):
    """The connection type for GistComment."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("GistCommentEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("GistComment"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class GistCommentEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("GistComment", graphql_name="node")
    """The item at the end of the edge."""


class GistConnection(sgqlc.types.relay.Connection):
    """The connection type for Gist."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("GistEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Gist"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class GistEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Gist", graphql_name="node")
    """The item at the end of the edge."""


class GistFile(sgqlc.types.Type):
    """A file in a gist."""

    __schema__ = github_schema
    __field_names__ = ("encoded_name", "encoding", "extension", "is_image", "is_truncated", "language", "name", "size", "text")
    encoded_name = sgqlc.types.Field(String, graphql_name="encodedName")
    """The file name encoded to remove characters that are invalid in URL
    paths.
    """

    encoding = sgqlc.types.Field(String, graphql_name="encoding")
    """The gist file encoding."""

    extension = sgqlc.types.Field(String, graphql_name="extension")
    """The file extension from the file name."""

    is_image = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isImage")
    """Indicates if this file is an image."""

    is_truncated = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isTruncated")
    """Whether the file's contents were truncated."""

    language = sgqlc.types.Field("Language", graphql_name="language")
    """The programming language this file is written in."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """The gist file name."""

    size = sgqlc.types.Field(Int, graphql_name="size")
    """The gist file size in bytes."""

    text = sgqlc.types.Field(
        String, graphql_name="text", args=sgqlc.types.ArgDict((("truncate", sgqlc.types.Arg(Int, graphql_name="truncate", default=None)),))
    )
    """UTF8 text data or null if the file is binary

    Arguments:

    * `truncate` (`Int`): Optionally truncate the returned file to
      this length.
    """


class GitActor(sgqlc.types.Type):
    """Represents an actor in a Git commit (ie. an author or committer)."""

    __schema__ = github_schema
    __field_names__ = ("avatar_url", "date", "email", "name", "user")
    avatar_url = sgqlc.types.Field(
        sgqlc.types.non_null(URI),
        graphql_name="avatarUrl",
        args=sgqlc.types.ArgDict((("size", sgqlc.types.Arg(Int, graphql_name="size", default=None)),)),
    )
    """A URL pointing to the author's public avatar.

    Arguments:

    * `size` (`Int`): The size of the resulting square image.
    """

    date = sgqlc.types.Field(GitTimestamp, graphql_name="date")
    """The timestamp of the Git action (authoring or committing)."""

    email = sgqlc.types.Field(String, graphql_name="email")
    """The email in the Git commit."""

    name = sgqlc.types.Field(String, graphql_name="name")
    """The name in the Git commit."""

    user = sgqlc.types.Field("User", graphql_name="user")
    """The GitHub user corresponding to the email field. Null if no such
    user exists.
    """


class GitActorConnection(sgqlc.types.relay.Connection):
    """The connection type for GitActor."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("GitActorEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(GitActor), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class GitActorEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(GitActor, graphql_name="node")
    """The item at the end of the edge."""


class GitHubMetadata(sgqlc.types.Type):
    """Represents information about the GitHub instance."""

    __schema__ = github_schema
    __field_names__ = (
        "git_hub_services_sha",
        "git_ip_addresses",
        "hook_ip_addresses",
        "importer_ip_addresses",
        "is_password_authentication_verifiable",
        "pages_ip_addresses",
    )
    git_hub_services_sha = sgqlc.types.Field(sgqlc.types.non_null(GitObjectID), graphql_name="gitHubServicesSha")
    """Returns a String that's a SHA of `github-services`"""

    git_ip_addresses = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="gitIpAddresses")
    """IP addresses that users connect to for git operations"""

    hook_ip_addresses = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="hookIpAddresses")
    """IP addresses that service hooks are sent from"""

    importer_ip_addresses = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="importerIpAddresses")
    """IP addresses that the importer connects from"""

    is_password_authentication_verifiable = sgqlc.types.Field(
        sgqlc.types.non_null(Boolean), graphql_name="isPasswordAuthenticationVerifiable"
    )
    """Whether or not users are verified"""

    pages_ip_addresses = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="pagesIpAddresses")
    """IP addresses for GitHub Pages' A records"""


class GrantEnterpriseOrganizationsMigratorRolePayload(sgqlc.types.Type):
    """Autogenerated return type of
    GrantEnterpriseOrganizationsMigratorRole
    """

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "organizations")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    organizations = sgqlc.types.Field(
        "OrganizationConnection",
        graphql_name="organizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """The organizations that had the migrator role applied to for the
    given user.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class GrantMigratorRolePayload(sgqlc.types.Type):
    """Autogenerated return type of GrantMigratorRole"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "success")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    success = sgqlc.types.Field(Boolean, graphql_name="success")
    """Did the operation succeed?"""


class Hovercard(sgqlc.types.Type):
    """Detail needed to display a hovercard for a user"""

    __schema__ = github_schema
    __field_names__ = ("contexts",)
    contexts = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(HovercardContext))), graphql_name="contexts")
    """Each of the contexts for this hovercard"""


class InviteEnterpriseAdminPayload(sgqlc.types.Type):
    """Autogenerated return type of InviteEnterpriseAdmin"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "invitation")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    invitation = sgqlc.types.Field("EnterpriseAdministratorInvitation", graphql_name="invitation")
    """The created enterprise administrator invitation."""


class IpAllowListEntryConnection(sgqlc.types.relay.Connection):
    """The connection type for IpAllowListEntry."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("IpAllowListEntryEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("IpAllowListEntry"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class IpAllowListEntryEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("IpAllowListEntry", graphql_name="node")
    """The item at the end of the edge."""


class IssueCommentConnection(sgqlc.types.relay.Connection):
    """The connection type for IssueComment."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("IssueCommentEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("IssueComment"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class IssueCommentEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("IssueComment", graphql_name="node")
    """The item at the end of the edge."""


class IssueConnection(sgqlc.types.relay.Connection):
    """The connection type for Issue."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("IssueEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Issue"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class IssueContributionsByRepository(sgqlc.types.Type):
    """This aggregates issues opened by a user within one repository."""

    __schema__ = github_schema
    __field_names__ = ("contributions", "repository")
    contributions = sgqlc.types.Field(
        sgqlc.types.non_null(CreatedIssueContributionConnection),
        graphql_name="contributions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("order_by", sgqlc.types.Arg(ContributionOrder, graphql_name="orderBy", default={"direction": "DESC"})),
            )
        ),
    )
    """The issue contributions.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`ContributionOrder`): Ordering options for
      contributions returned from the connection. (default:
      `{direction: DESC}`)
    """

    repository = sgqlc.types.Field(sgqlc.types.non_null("Repository"), graphql_name="repository")
    """The repository in which the issues were opened."""


class IssueEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Issue", graphql_name="node")
    """The item at the end of the edge."""


class IssueTemplate(sgqlc.types.Type):
    """A repository issue template."""

    __schema__ = github_schema
    __field_names__ = ("about", "assignees", "body", "filename", "labels", "name", "title")
    about = sgqlc.types.Field(String, graphql_name="about")
    """The template purpose."""

    assignees = sgqlc.types.Field(
        sgqlc.types.non_null("UserConnection"),
        graphql_name="assignees",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """The suggested assignees.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    body = sgqlc.types.Field(String, graphql_name="body")
    """The suggested issue body."""

    filename = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="filename")
    """The template filename."""

    labels = sgqlc.types.Field(
        "LabelConnection",
        graphql_name="labels",
        args=sgqlc.types.ArgDict(
            (
                ("order_by", sgqlc.types.Arg(LabelOrder, graphql_name="orderBy", default={"field": "CREATED_AT", "direction": "ASC"})),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """The suggested issue labels

    Arguments:

    * `order_by` (`LabelOrder`): Ordering options for labels returned
      from the connection. (default: `{field: CREATED_AT, direction:
      ASC}`)
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The template name."""

    title = sgqlc.types.Field(String, graphql_name="title")
    """The suggested issue title."""


class IssueTimelineConnection(sgqlc.types.relay.Connection):
    """The connection type for IssueTimelineItem."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("IssueTimelineItemEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("IssueTimelineItem"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class IssueTimelineItemEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("IssueTimelineItem", graphql_name="node")
    """The item at the end of the edge."""


class IssueTimelineItemsConnection(sgqlc.types.relay.Connection):
    """The connection type for IssueTimelineItems."""

    __schema__ = github_schema
    __field_names__ = ("edges", "filtered_count", "nodes", "page_count", "page_info", "total_count", "updated_at")
    edges = sgqlc.types.Field(sgqlc.types.list_of("IssueTimelineItemsEdge"), graphql_name="edges")
    """A list of edges."""

    filtered_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="filteredCount")
    """Identifies the count of items after applying `before` and `after`
    filters.
    """

    nodes = sgqlc.types.Field(sgqlc.types.list_of("IssueTimelineItems"), graphql_name="nodes")
    """A list of nodes."""

    page_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="pageCount")
    """Identifies the count of items after applying `before`/`after`
    filters and `first`/`last`/`skip` slicing.
    """

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""

    updated_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="updatedAt")
    """Identifies the date and time when the timeline was last updated."""


class IssueTimelineItemsEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("IssueTimelineItems", graphql_name="node")
    """The item at the end of the edge."""


class LabelConnection(sgqlc.types.relay.Connection):
    """The connection type for Label."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("LabelEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Label"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class LabelEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Label", graphql_name="node")
    """The item at the end of the edge."""


class LanguageConnection(sgqlc.types.relay.Connection):
    """A list of languages associated with the parent."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count", "total_size")
    edges = sgqlc.types.Field(sgqlc.types.list_of("LanguageEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Language"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""

    total_size = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalSize")
    """The total size in bytes of files written in that language."""


class LanguageEdge(sgqlc.types.Type):
    """Represents the language of a repository."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "size")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")

    node = sgqlc.types.Field(sgqlc.types.non_null("Language"), graphql_name="node")

    size = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="size")
    """The number of bytes of code written in the language."""


class LicenseRule(sgqlc.types.Type):
    """Describes a License's conditions, permissions, and limitations"""

    __schema__ = github_schema
    __field_names__ = ("description", "key", "label")
    description = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="description")
    """A description of the rule"""

    key = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="key")
    """The machine-readable rule key"""

    label = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="label")
    """The human-readable rule label"""


class LinkProjectV2ToRepositoryPayload(sgqlc.types.Type):
    """Autogenerated return type of LinkProjectV2ToRepository"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "repository")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    repository = sgqlc.types.Field("Repository", graphql_name="repository")
    """The repository the project is linked to."""


class LinkProjectV2ToTeamPayload(sgqlc.types.Type):
    """Autogenerated return type of LinkProjectV2ToTeam"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "team")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    team = sgqlc.types.Field("Team", graphql_name="team")
    """The team the project is linked to"""


class LinkRepositoryToProjectPayload(sgqlc.types.Type):
    """Autogenerated return type of LinkRepositoryToProject"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project", "repository")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project = sgqlc.types.Field("Project", graphql_name="project")
    """The linked Project."""

    repository = sgqlc.types.Field("Repository", graphql_name="repository")
    """The linked Repository."""


class LinkedBranchConnection(sgqlc.types.relay.Connection):
    """The connection type for LinkedBranch."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("LinkedBranchEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("LinkedBranch"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class LinkedBranchEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("LinkedBranch", graphql_name="node")
    """The item at the end of the edge."""


class LockLockablePayload(sgqlc.types.Type):
    """Autogenerated return type of LockLockable"""

    __schema__ = github_schema
    __field_names__ = ("actor", "client_mutation_id", "locked_record")
    actor = sgqlc.types.Field(Actor, graphql_name="actor")
    """Identifies the actor who performed the event."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    locked_record = sgqlc.types.Field(Lockable, graphql_name="lockedRecord")
    """The item that was locked."""


class MannequinConnection(sgqlc.types.relay.Connection):
    """The connection type for Mannequin."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("MannequinEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Mannequin"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class MannequinEdge(sgqlc.types.Type):
    """Represents a mannequin."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Mannequin", graphql_name="node")
    """The item at the end of the edge."""


class MarkDiscussionCommentAsAnswerPayload(sgqlc.types.Type):
    """Autogenerated return type of MarkDiscussionCommentAsAnswer"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "discussion")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    discussion = sgqlc.types.Field("Discussion", graphql_name="discussion")
    """The discussion that includes the chosen comment."""


class MarkFileAsViewedPayload(sgqlc.types.Type):
    """Autogenerated return type of MarkFileAsViewed"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The updated pull request."""


class MarkProjectV2AsTemplatePayload(sgqlc.types.Type):
    """Autogenerated return type of MarkProjectV2AsTemplate"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "project_v2")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    project_v2 = sgqlc.types.Field("ProjectV2", graphql_name="projectV2")
    """The project."""


class MarkPullRequestReadyForReviewPayload(sgqlc.types.Type):
    """Autogenerated return type of MarkPullRequestReadyForReview"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The pull request that is ready for review."""


class MarketplaceListingConnection(sgqlc.types.relay.Connection):
    """Look up Marketplace Listings"""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("MarketplaceListingEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("MarketplaceListing"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class MarketplaceListingEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("MarketplaceListing", graphql_name="node")
    """The item at the end of the edge."""


class MergeBranchPayload(sgqlc.types.Type):
    """Autogenerated return type of MergeBranch"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "merge_commit")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    merge_commit = sgqlc.types.Field("Commit", graphql_name="mergeCommit")
    """The resulting merge Commit."""


class MergePullRequestPayload(sgqlc.types.Type):
    """Autogenerated return type of MergePullRequest"""

    __schema__ = github_schema
    __field_names__ = ("actor", "client_mutation_id", "pull_request")
    actor = sgqlc.types.Field(Actor, graphql_name="actor")
    """Identifies the actor who performed the event."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The pull request that was merged."""


class MergeQueueConfiguration(sgqlc.types.Type):
    """Configuration for a MergeQueue"""

    __schema__ = github_schema
    __field_names__ = (
        "check_response_timeout",
        "maximum_entries_to_build",
        "maximum_entries_to_merge",
        "merge_method",
        "merging_strategy",
        "minimum_entries_to_merge",
        "minimum_entries_to_merge_wait_time",
    )
    check_response_timeout = sgqlc.types.Field(Int, graphql_name="checkResponseTimeout")
    """The amount of time in minutes to wait for a check response before
    considering it a failure.
    """

    maximum_entries_to_build = sgqlc.types.Field(Int, graphql_name="maximumEntriesToBuild")
    """The maximum number of entries to build at once."""

    maximum_entries_to_merge = sgqlc.types.Field(Int, graphql_name="maximumEntriesToMerge")
    """The maximum number of entries to merge at once."""

    merge_method = sgqlc.types.Field(PullRequestMergeMethod, graphql_name="mergeMethod")
    """The merge method to use for this queue."""

    merging_strategy = sgqlc.types.Field(MergeQueueMergingStrategy, graphql_name="mergingStrategy")
    """The strategy to use when merging entries."""

    minimum_entries_to_merge = sgqlc.types.Field(Int, graphql_name="minimumEntriesToMerge")
    """The minimum number of entries required to merge at once."""

    minimum_entries_to_merge_wait_time = sgqlc.types.Field(Int, graphql_name="minimumEntriesToMergeWaitTime")
    """The amount of time in minutes to wait before ignoring the minumum
    number of entries in the queue requirement and merging a
    collection of entries
    """


class MergeQueueEntryConnection(sgqlc.types.relay.Connection):
    """The connection type for MergeQueueEntry."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("MergeQueueEntryEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("MergeQueueEntry"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class MergeQueueEntryEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("MergeQueueEntry", graphql_name="node")
    """The item at the end of the edge."""


class MilestoneConnection(sgqlc.types.relay.Connection):
    """The connection type for Milestone."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("MilestoneEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Milestone"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class MilestoneEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Milestone", graphql_name="node")
    """The item at the end of the edge."""


class MinimizeCommentPayload(sgqlc.types.Type):
    """Autogenerated return type of MinimizeComment"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "minimized_comment")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    minimized_comment = sgqlc.types.Field(Minimizable, graphql_name="minimizedComment")
    """The comment that was minimized."""


class MoveProjectCardPayload(sgqlc.types.Type):
    """Autogenerated return type of MoveProjectCard"""

    __schema__ = github_schema
    __field_names__ = ("card_edge", "client_mutation_id")
    card_edge = sgqlc.types.Field("ProjectCardEdge", graphql_name="cardEdge")
    """The new edge of the moved card."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class MoveProjectColumnPayload(sgqlc.types.Type):
    """Autogenerated return type of MoveProjectColumn"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "column_edge")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    column_edge = sgqlc.types.Field("ProjectColumnEdge", graphql_name="columnEdge")
    """The new edge of the moved column."""


class Mutation(sgqlc.types.Type):
    """The root query for implementing GraphQL mutations."""

    __schema__ = github_schema
    __field_names__ = (
        "abort_queued_migrations",
        "accept_enterprise_administrator_invitation",
        "accept_topic_suggestion",
        "add_assignees_to_assignable",
        "add_comment",
        "add_discussion_comment",
        "add_discussion_poll_vote",
        "add_enterprise_organization_member",
        "add_enterprise_support_entitlement",
        "add_labels_to_labelable",
        "add_project_card",
        "add_project_column",
        "add_project_v2_draft_issue",
        "add_project_v2_item_by_id",
        "add_pull_request_review",
        "add_pull_request_review_comment",
        "add_pull_request_review_thread",
        "add_reaction",
        "add_star",
        "add_upvote",
        "add_verifiable_domain",
        "approve_deployments",
        "approve_verifiable_domain",
        "archive_project_v2_item",
        "archive_repository",
        "cancel_enterprise_admin_invitation",
        "cancel_sponsorship",
        "change_user_status",
        "clear_labels_from_labelable",
        "clear_project_v2_item_field_value",
        "clone_project",
        "clone_template_repository",
        "close_discussion",
        "close_issue",
        "close_pull_request",
        "convert_project_card_note_to_issue",
        "convert_pull_request_to_draft",
        "copy_project_v2",
        "create_attribution_invitation",
        "create_branch_protection_rule",
        "create_check_run",
        "create_check_suite",
        "create_commit_on_branch",
        "create_discussion",
        "create_enterprise_organization",
        "create_environment",
        "create_ip_allow_list_entry",
        "create_issue",
        "create_linked_branch",
        "create_migration_source",
        "create_project",
        "create_project_v2",
        "create_project_v2_field",
        "create_pull_request",
        "create_ref",
        "create_repository",
        "create_repository_ruleset",
        "create_sponsors_listing",
        "create_sponsors_tier",
        "create_sponsorship",
        "create_sponsorships",
        "create_team_discussion",
        "create_team_discussion_comment",
        "decline_topic_suggestion",
        "delete_branch_protection_rule",
        "delete_deployment",
        "delete_discussion",
        "delete_discussion_comment",
        "delete_environment",
        "delete_ip_allow_list_entry",
        "delete_issue",
        "delete_issue_comment",
        "delete_linked_branch",
        "delete_project",
        "delete_project_card",
        "delete_project_column",
        "delete_project_v2",
        "delete_project_v2_field",
        "delete_project_v2_item",
        "delete_project_v2_workflow",
        "delete_pull_request_review",
        "delete_pull_request_review_comment",
        "delete_ref",
        "delete_repository_ruleset",
        "delete_team_discussion",
        "delete_team_discussion_comment",
        "delete_verifiable_domain",
        "dequeue_pull_request",
        "disable_pull_request_auto_merge",
        "dismiss_pull_request_review",
        "dismiss_repository_vulnerability_alert",
        "enable_pull_request_auto_merge",
        "enqueue_pull_request",
        "follow_organization",
        "follow_user",
        "grant_enterprise_organizations_migrator_role",
        "grant_migrator_role",
        "invite_enterprise_admin",
        "link_project_v2_to_repository",
        "link_project_v2_to_team",
        "link_repository_to_project",
        "lock_lockable",
        "mark_discussion_comment_as_answer",
        "mark_file_as_viewed",
        "mark_project_v2_as_template",
        "mark_pull_request_ready_for_review",
        "merge_branch",
        "merge_pull_request",
        "minimize_comment",
        "move_project_card",
        "move_project_column",
        "pin_issue",
        "publish_sponsors_tier",
        "regenerate_enterprise_identity_provider_recovery_codes",
        "regenerate_verifiable_domain_token",
        "reject_deployments",
        "remove_assignees_from_assignable",
        "remove_enterprise_admin",
        "remove_enterprise_identity_provider",
        "remove_enterprise_member",
        "remove_enterprise_organization",
        "remove_enterprise_support_entitlement",
        "remove_labels_from_labelable",
        "remove_outside_collaborator",
        "remove_reaction",
        "remove_star",
        "remove_upvote",
        "reopen_discussion",
        "reopen_issue",
        "reopen_pull_request",
        "request_reviews",
        "rerequest_check_suite",
        "resolve_review_thread",
        "retire_sponsors_tier",
        "revert_pull_request",
        "revoke_enterprise_organizations_migrator_role",
        "revoke_migrator_role",
        "set_enterprise_identity_provider",
        "set_organization_interaction_limit",
        "set_repository_interaction_limit",
        "set_user_interaction_limit",
        "start_organization_migration",
        "start_repository_migration",
        "submit_pull_request_review",
        "transfer_enterprise_organization",
        "transfer_issue",
        "unarchive_project_v2_item",
        "unarchive_repository",
        "unfollow_organization",
        "unfollow_user",
        "unlink_project_v2_from_repository",
        "unlink_project_v2_from_team",
        "unlink_repository_from_project",
        "unlock_lockable",
        "unmark_discussion_comment_as_answer",
        "unmark_file_as_viewed",
        "unmark_issue_as_duplicate",
        "unmark_project_v2_as_template",
        "unminimize_comment",
        "unpin_issue",
        "unresolve_review_thread",
        "update_branch_protection_rule",
        "update_check_run",
        "update_check_suite_preferences",
        "update_discussion",
        "update_discussion_comment",
        "update_enterprise_administrator_role",
        "update_enterprise_allow_private_repository_forking_setting",
        "update_enterprise_default_repository_permission_setting",
        "update_enterprise_members_can_change_repository_visibility_setting",
        "update_enterprise_members_can_create_repositories_setting",
        "update_enterprise_members_can_delete_issues_setting",
        "update_enterprise_members_can_delete_repositories_setting",
        "update_enterprise_members_can_invite_collaborators_setting",
        "update_enterprise_members_can_make_purchases_setting",
        "update_enterprise_members_can_update_protected_branches_setting",
        "update_enterprise_members_can_view_dependency_insights_setting",
        "update_enterprise_organization_projects_setting",
        "update_enterprise_owner_organization_role",
        "update_enterprise_profile",
        "update_enterprise_repository_projects_setting",
        "update_enterprise_team_discussions_setting",
        "update_enterprise_two_factor_authentication_required_setting",
        "update_environment",
        "update_ip_allow_list_enabled_setting",
        "update_ip_allow_list_entry",
        "update_ip_allow_list_for_installed_apps_enabled_setting",
        "update_issue",
        "update_issue_comment",
        "update_notification_restriction_setting",
        "update_organization_allow_private_repository_forking_setting",
        "update_organization_web_commit_signoff_setting",
        "update_project",
        "update_project_card",
        "update_project_column",
        "update_project_v2",
        "update_project_v2_collaborators",
        "update_project_v2_draft_issue",
        "update_project_v2_item_field_value",
        "update_project_v2_item_position",
        "update_pull_request",
        "update_pull_request_branch",
        "update_pull_request_review",
        "update_pull_request_review_comment",
        "update_ref",
        "update_repository",
        "update_repository_ruleset",
        "update_repository_web_commit_signoff_setting",
        "update_sponsorship_preferences",
        "update_subscription",
        "update_team_discussion",
        "update_team_discussion_comment",
        "update_teams_repository",
        "update_topics",
        "verify_verifiable_domain",
    )
    abort_queued_migrations = sgqlc.types.Field(
        AbortQueuedMigrationsPayload,
        graphql_name="abortQueuedMigrations",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AbortQueuedMigrationsInput), graphql_name="input", default=None)),)
        ),
    )
    """Clear all of a customer's queued migrations

    Arguments:

    * `input` (`AbortQueuedMigrationsInput!`): Parameters for
      AbortQueuedMigrations
    """

    accept_enterprise_administrator_invitation = sgqlc.types.Field(
        AcceptEnterpriseAdministratorInvitationPayload,
        graphql_name="acceptEnterpriseAdministratorInvitation",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(sgqlc.types.non_null(AcceptEnterpriseAdministratorInvitationInput), graphql_name="input", default=None),
                ),
            )
        ),
    )
    """Accepts a pending invitation for a user to become an administrator
    of an enterprise.

    Arguments:

    * `input` (`AcceptEnterpriseAdministratorInvitationInput!`):
      Parameters for AcceptEnterpriseAdministratorInvitation
    """

    accept_topic_suggestion = sgqlc.types.Field(
        AcceptTopicSuggestionPayload,
        graphql_name="acceptTopicSuggestion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AcceptTopicSuggestionInput), graphql_name="input", default=None)),)
        ),
    )
    """Applies a suggested topic to the repository.

    Arguments:

    * `input` (`AcceptTopicSuggestionInput!`): Parameters for
      AcceptTopicSuggestion
    """

    add_assignees_to_assignable = sgqlc.types.Field(
        AddAssigneesToAssignablePayload,
        graphql_name="addAssigneesToAssignable",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddAssigneesToAssignableInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds assignees to an assignable object.

    Arguments:

    * `input` (`AddAssigneesToAssignableInput!`): Parameters for
      AddAssigneesToAssignable
    """

    add_comment = sgqlc.types.Field(
        AddCommentPayload,
        graphql_name="addComment",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(AddCommentInput), graphql_name="input", default=None)),)),
    )
    """Adds a comment to an Issue or Pull Request.

    Arguments:

    * `input` (`AddCommentInput!`): Parameters for AddComment
    """

    add_discussion_comment = sgqlc.types.Field(
        AddDiscussionCommentPayload,
        graphql_name="addDiscussionComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddDiscussionCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds a comment to a Discussion, possibly as a reply to another
    comment.

    Arguments:

    * `input` (`AddDiscussionCommentInput!`): Parameters for
      AddDiscussionComment
    """

    add_discussion_poll_vote = sgqlc.types.Field(
        AddDiscussionPollVotePayload,
        graphql_name="addDiscussionPollVote",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddDiscussionPollVoteInput), graphql_name="input", default=None)),)
        ),
    )
    """Vote for an option in a discussion poll.

    Arguments:

    * `input` (`AddDiscussionPollVoteInput!`): Parameters for
      AddDiscussionPollVote
    """

    add_enterprise_organization_member = sgqlc.types.Field(
        AddEnterpriseOrganizationMemberPayload,
        graphql_name="addEnterpriseOrganizationMember",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddEnterpriseOrganizationMemberInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds enterprise members to an organization within the enterprise.

    Arguments:

    * `input` (`AddEnterpriseOrganizationMemberInput!`): Parameters
      for AddEnterpriseOrganizationMember
    """

    add_enterprise_support_entitlement = sgqlc.types.Field(
        AddEnterpriseSupportEntitlementPayload,
        graphql_name="addEnterpriseSupportEntitlement",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddEnterpriseSupportEntitlementInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds a support entitlement to an enterprise member.

    Arguments:

    * `input` (`AddEnterpriseSupportEntitlementInput!`): Parameters
      for AddEnterpriseSupportEntitlement
    """

    add_labels_to_labelable = sgqlc.types.Field(
        AddLabelsToLabelablePayload,
        graphql_name="addLabelsToLabelable",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddLabelsToLabelableInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds labels to a labelable object.

    Arguments:

    * `input` (`AddLabelsToLabelableInput!`): Parameters for
      AddLabelsToLabelable
    """

    add_project_card = sgqlc.types.Field(
        AddProjectCardPayload,
        graphql_name="addProjectCard",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddProjectCardInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds a card to a ProjectColumn. Either `contentId` or `note` must
    be provided but **not** both.

    Arguments:

    * `input` (`AddProjectCardInput!`): Parameters for AddProjectCard
    """

    add_project_column = sgqlc.types.Field(
        AddProjectColumnPayload,
        graphql_name="addProjectColumn",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddProjectColumnInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds a column to a Project.

    Arguments:

    * `input` (`AddProjectColumnInput!`): Parameters for
      AddProjectColumn
    """

    add_project_v2_draft_issue = sgqlc.types.Field(
        AddProjectV2DraftIssuePayload,
        graphql_name="addProjectV2DraftIssue",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddProjectV2DraftIssueInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates a new draft issue and add it to a Project.

    Arguments:

    * `input` (`AddProjectV2DraftIssueInput!`): Parameters for
      AddProjectV2DraftIssue
    """

    add_project_v2_item_by_id = sgqlc.types.Field(
        AddProjectV2ItemByIdPayload,
        graphql_name="addProjectV2ItemById",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddProjectV2ItemByIdInput), graphql_name="input", default=None)),)
        ),
    )
    """Links an existing content instance to a Project.

    Arguments:

    * `input` (`AddProjectV2ItemByIdInput!`): Parameters for
      AddProjectV2ItemById
    """

    add_pull_request_review = sgqlc.types.Field(
        AddPullRequestReviewPayload,
        graphql_name="addPullRequestReview",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddPullRequestReviewInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds a review to a Pull Request.

    Arguments:

    * `input` (`AddPullRequestReviewInput!`): Parameters for
      AddPullRequestReview
    """

    add_pull_request_review_comment = sgqlc.types.Field(
        AddPullRequestReviewCommentPayload,
        graphql_name="addPullRequestReviewComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddPullRequestReviewCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds a comment to a review.

    Arguments:

    * `input` (`AddPullRequestReviewCommentInput!`): Parameters for
      AddPullRequestReviewComment
    """

    add_pull_request_review_thread = sgqlc.types.Field(
        AddPullRequestReviewThreadPayload,
        graphql_name="addPullRequestReviewThread",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddPullRequestReviewThreadInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds a new thread to a pending Pull Request Review.

    Arguments:

    * `input` (`AddPullRequestReviewThreadInput!`): Parameters for
      AddPullRequestReviewThread
    """

    add_reaction = sgqlc.types.Field(
        AddReactionPayload,
        graphql_name="addReaction",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(AddReactionInput), graphql_name="input", default=None)),)),
    )
    """Adds a reaction to a subject.

    Arguments:

    * `input` (`AddReactionInput!`): Parameters for AddReaction
    """

    add_star = sgqlc.types.Field(
        AddStarPayload,
        graphql_name="addStar",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(AddStarInput), graphql_name="input", default=None)),)),
    )
    """Adds a star to a Starrable.

    Arguments:

    * `input` (`AddStarInput!`): Parameters for AddStar
    """

    add_upvote = sgqlc.types.Field(
        AddUpvotePayload,
        graphql_name="addUpvote",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(AddUpvoteInput), graphql_name="input", default=None)),)),
    )
    """Add an upvote to a discussion or discussion comment.

    Arguments:

    * `input` (`AddUpvoteInput!`): Parameters for AddUpvote
    """

    add_verifiable_domain = sgqlc.types.Field(
        AddVerifiableDomainPayload,
        graphql_name="addVerifiableDomain",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(AddVerifiableDomainInput), graphql_name="input", default=None)),)
        ),
    )
    """Adds a verifiable domain to an owning account.

    Arguments:

    * `input` (`AddVerifiableDomainInput!`): Parameters for
      AddVerifiableDomain
    """

    approve_deployments = sgqlc.types.Field(
        ApproveDeploymentsPayload,
        graphql_name="approveDeployments",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ApproveDeploymentsInput), graphql_name="input", default=None)),)
        ),
    )
    """Approve all pending deployments under one or more environments

    Arguments:

    * `input` (`ApproveDeploymentsInput!`): Parameters for
      ApproveDeployments
    """

    approve_verifiable_domain = sgqlc.types.Field(
        ApproveVerifiableDomainPayload,
        graphql_name="approveVerifiableDomain",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ApproveVerifiableDomainInput), graphql_name="input", default=None)),)
        ),
    )
    """Approve a verifiable domain for notification delivery.

    Arguments:

    * `input` (`ApproveVerifiableDomainInput!`): Parameters for
      ApproveVerifiableDomain
    """

    archive_project_v2_item = sgqlc.types.Field(
        ArchiveProjectV2ItemPayload,
        graphql_name="archiveProjectV2Item",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ArchiveProjectV2ItemInput), graphql_name="input", default=None)),)
        ),
    )
    """Archives a ProjectV2Item

    Arguments:

    * `input` (`ArchiveProjectV2ItemInput!`): Parameters for
      ArchiveProjectV2Item
    """

    archive_repository = sgqlc.types.Field(
        ArchiveRepositoryPayload,
        graphql_name="archiveRepository",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ArchiveRepositoryInput), graphql_name="input", default=None)),)
        ),
    )
    """Marks a repository as archived.

    Arguments:

    * `input` (`ArchiveRepositoryInput!`): Parameters for
      ArchiveRepository
    """

    cancel_enterprise_admin_invitation = sgqlc.types.Field(
        CancelEnterpriseAdminInvitationPayload,
        graphql_name="cancelEnterpriseAdminInvitation",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CancelEnterpriseAdminInvitationInput), graphql_name="input", default=None)),)
        ),
    )
    """Cancels a pending invitation for an administrator to join an
    enterprise.

    Arguments:

    * `input` (`CancelEnterpriseAdminInvitationInput!`): Parameters
      for CancelEnterpriseAdminInvitation
    """

    cancel_sponsorship = sgqlc.types.Field(
        CancelSponsorshipPayload,
        graphql_name="cancelSponsorship",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CancelSponsorshipInput), graphql_name="input", default=None)),)
        ),
    )
    """Cancel an active sponsorship.

    Arguments:

    * `input` (`CancelSponsorshipInput!`): Parameters for
      CancelSponsorship
    """

    change_user_status = sgqlc.types.Field(
        ChangeUserStatusPayload,
        graphql_name="changeUserStatus",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ChangeUserStatusInput), graphql_name="input", default=None)),)
        ),
    )
    """Update your status on GitHub.

    Arguments:

    * `input` (`ChangeUserStatusInput!`): Parameters for
      ChangeUserStatus
    """

    clear_labels_from_labelable = sgqlc.types.Field(
        ClearLabelsFromLabelablePayload,
        graphql_name="clearLabelsFromLabelable",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ClearLabelsFromLabelableInput), graphql_name="input", default=None)),)
        ),
    )
    """Clears all labels from a labelable object.

    Arguments:

    * `input` (`ClearLabelsFromLabelableInput!`): Parameters for
      ClearLabelsFromLabelable
    """

    clear_project_v2_item_field_value = sgqlc.types.Field(
        ClearProjectV2ItemFieldValuePayload,
        graphql_name="clearProjectV2ItemFieldValue",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ClearProjectV2ItemFieldValueInput), graphql_name="input", default=None)),)
        ),
    )
    """This mutation clears the value of a field for an item in a
    Project. Currently only text, number, date, assignees, labels,
    single-select, iteration and milestone fields are supported.

    Arguments:

    * `input` (`ClearProjectV2ItemFieldValueInput!`): Parameters for
      ClearProjectV2ItemFieldValue
    """

    clone_project = sgqlc.types.Field(
        CloneProjectPayload,
        graphql_name="cloneProject",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CloneProjectInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates a new project by cloning configuration from an existing
    project.

    Arguments:

    * `input` (`CloneProjectInput!`): Parameters for CloneProject
    """

    clone_template_repository = sgqlc.types.Field(
        CloneTemplateRepositoryPayload,
        graphql_name="cloneTemplateRepository",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CloneTemplateRepositoryInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a new repository with the same files and directory
    structure as a template repository.

    Arguments:

    * `input` (`CloneTemplateRepositoryInput!`): Parameters for
      CloneTemplateRepository
    """

    close_discussion = sgqlc.types.Field(
        CloseDiscussionPayload,
        graphql_name="closeDiscussion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CloseDiscussionInput), graphql_name="input", default=None)),)
        ),
    )
    """Close a discussion.

    Arguments:

    * `input` (`CloseDiscussionInput!`): Parameters for
      CloseDiscussion
    """

    close_issue = sgqlc.types.Field(
        CloseIssuePayload,
        graphql_name="closeIssue",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(CloseIssueInput), graphql_name="input", default=None)),)),
    )
    """Close an issue.

    Arguments:

    * `input` (`CloseIssueInput!`): Parameters for CloseIssue
    """

    close_pull_request = sgqlc.types.Field(
        ClosePullRequestPayload,
        graphql_name="closePullRequest",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ClosePullRequestInput), graphql_name="input", default=None)),)
        ),
    )
    """Close a pull request.

    Arguments:

    * `input` (`ClosePullRequestInput!`): Parameters for
      ClosePullRequest
    """

    convert_project_card_note_to_issue = sgqlc.types.Field(
        ConvertProjectCardNoteToIssuePayload,
        graphql_name="convertProjectCardNoteToIssue",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ConvertProjectCardNoteToIssueInput), graphql_name="input", default=None)),)
        ),
    )
    """Convert a project note card to one associated with a newly created
    issue.

    Arguments:

    * `input` (`ConvertProjectCardNoteToIssueInput!`): Parameters for
      ConvertProjectCardNoteToIssue
    """

    convert_pull_request_to_draft = sgqlc.types.Field(
        ConvertPullRequestToDraftPayload,
        graphql_name="convertPullRequestToDraft",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ConvertPullRequestToDraftInput), graphql_name="input", default=None)),)
        ),
    )
    """Converts a pull request to draft

    Arguments:

    * `input` (`ConvertPullRequestToDraftInput!`): Parameters for
      ConvertPullRequestToDraft
    """

    copy_project_v2 = sgqlc.types.Field(
        CopyProjectV2Payload,
        graphql_name="copyProjectV2",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CopyProjectV2Input), graphql_name="input", default=None)),)
        ),
    )
    """Copy a project.

    Arguments:

    * `input` (`CopyProjectV2Input!`): Parameters for CopyProjectV2
    """

    create_attribution_invitation = sgqlc.types.Field(
        CreateAttributionInvitationPayload,
        graphql_name="createAttributionInvitation",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateAttributionInvitationInput), graphql_name="input", default=None)),)
        ),
    )
    """Invites a user to claim reattributable data

    Arguments:

    * `input` (`CreateAttributionInvitationInput!`): Parameters for
      CreateAttributionInvitation
    """

    create_branch_protection_rule = sgqlc.types.Field(
        CreateBranchProtectionRulePayload,
        graphql_name="createBranchProtectionRule",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateBranchProtectionRuleInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a new branch protection rule

    Arguments:

    * `input` (`CreateBranchProtectionRuleInput!`): Parameters for
      CreateBranchProtectionRule
    """

    create_check_run = sgqlc.types.Field(
        CreateCheckRunPayload,
        graphql_name="createCheckRun",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateCheckRunInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a check run.

    Arguments:

    * `input` (`CreateCheckRunInput!`): Parameters for CreateCheckRun
    """

    create_check_suite = sgqlc.types.Field(
        CreateCheckSuitePayload,
        graphql_name="createCheckSuite",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateCheckSuiteInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a check suite

    Arguments:

    * `input` (`CreateCheckSuiteInput!`): Parameters for
      CreateCheckSuite
    """

    create_commit_on_branch = sgqlc.types.Field(
        CreateCommitOnBranchPayload,
        graphql_name="createCommitOnBranch",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateCommitOnBranchInput), graphql_name="input", default=None)),)
        ),
    )
    """Appends a commit to the given branch as the authenticated user.
    This mutation creates a commit whose parent is the HEAD of the
    provided branch and also updates that branch to point to the new
    commit. It can be thought of as similar to `git commit`.  ###
    Locating a Branch  Commits are appended to a `branch` of type
    `Ref`. This must refer to a git branch (i.e.  the fully qualified
    path must begin with `refs/heads/`, although including this prefix
    is optional.  Callers may specify the `branch` to commit to either
    by its global node ID or by passing both of
    `repositoryNameWithOwner` and `refName`.  For more details see the
    documentation for `CommittableBranch`.  ### Describing Changes
    `fileChanges` are specified as a `FilesChanges` object describing
    `FileAdditions` and `FileDeletions`.  Please see the documentation
    for `FileChanges` for more information on how to use this argument
    to describe any set of file changes.  ### Authorship  Similar to
    the web commit interface, this mutation does not support
    specifying the author or committer of the commit and will not add
    support for this in the future.  A commit created by a successful
    execution of this mutation will be authored by the owner of the
    credential which authenticates the API request.  The committer
    will be identical to that of commits authored using the web
    interface.  If you need full control over author and committer
    information, please use the Git Database REST API instead.  ###
    Commit Signing  Commits made using this mutation are automatically
    signed by GitHub if supported and will be marked as verified in
    the user interface.

    Arguments:

    * `input` (`CreateCommitOnBranchInput!`): Parameters for
      CreateCommitOnBranch
    """

    create_discussion = sgqlc.types.Field(
        CreateDiscussionPayload,
        graphql_name="createDiscussion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateDiscussionInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a discussion.

    Arguments:

    * `input` (`CreateDiscussionInput!`): Parameters for
      CreateDiscussion
    """

    create_enterprise_organization = sgqlc.types.Field(
        CreateEnterpriseOrganizationPayload,
        graphql_name="createEnterpriseOrganization",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateEnterpriseOrganizationInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates an organization as part of an enterprise account.

    Arguments:

    * `input` (`CreateEnterpriseOrganizationInput!`): Parameters for
      CreateEnterpriseOrganization
    """

    create_environment = sgqlc.types.Field(
        CreateEnvironmentPayload,
        graphql_name="createEnvironment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateEnvironmentInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates an environment or simply returns it if already exists.

    Arguments:

    * `input` (`CreateEnvironmentInput!`): Parameters for
      CreateEnvironment
    """

    create_ip_allow_list_entry = sgqlc.types.Field(
        CreateIpAllowListEntryPayload,
        graphql_name="createIpAllowListEntry",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateIpAllowListEntryInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates a new IP allow list entry.

    Arguments:

    * `input` (`CreateIpAllowListEntryInput!`): Parameters for
      CreateIpAllowListEntry
    """

    create_issue = sgqlc.types.Field(
        CreateIssuePayload,
        graphql_name="createIssue",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateIssueInput), graphql_name="input", default=None)),)),
    )
    """Creates a new issue.

    Arguments:

    * `input` (`CreateIssueInput!`): Parameters for CreateIssue
    """

    create_linked_branch = sgqlc.types.Field(
        CreateLinkedBranchPayload,
        graphql_name="createLinkedBranch",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateLinkedBranchInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a branch linked to an issue.

    Arguments:

    * `input` (`CreateLinkedBranchInput!`): Parameters for
      CreateLinkedBranch
    """

    create_migration_source = sgqlc.types.Field(
        CreateMigrationSourcePayload,
        graphql_name="createMigrationSource",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateMigrationSourceInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates a GitHub Enterprise Importer (GEI) migration source.

    Arguments:

    * `input` (`CreateMigrationSourceInput!`): Parameters for
      CreateMigrationSource
    """

    create_project = sgqlc.types.Field(
        CreateProjectPayload,
        graphql_name="createProject",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateProjectInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates a new project.

    Arguments:

    * `input` (`CreateProjectInput!`): Parameters for CreateProject
    """

    create_project_v2 = sgqlc.types.Field(
        CreateProjectV2Payload,
        graphql_name="createProjectV2",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateProjectV2Input), graphql_name="input", default=None)),)
        ),
    )
    """Creates a new project.

    Arguments:

    * `input` (`CreateProjectV2Input!`): Parameters for
      CreateProjectV2
    """

    create_project_v2_field = sgqlc.types.Field(
        CreateProjectV2FieldPayload,
        graphql_name="createProjectV2Field",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateProjectV2FieldInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a new project field.

    Arguments:

    * `input` (`CreateProjectV2FieldInput!`): Parameters for
      CreateProjectV2Field
    """

    create_pull_request = sgqlc.types.Field(
        CreatePullRequestPayload,
        graphql_name="createPullRequest",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreatePullRequestInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a new pull request

    Arguments:

    * `input` (`CreatePullRequestInput!`): Parameters for
      CreatePullRequest
    """

    create_ref = sgqlc.types.Field(
        CreateRefPayload,
        graphql_name="createRef",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateRefInput), graphql_name="input", default=None)),)),
    )
    """Create a new Git Ref.

    Arguments:

    * `input` (`CreateRefInput!`): Parameters for CreateRef
    """

    create_repository = sgqlc.types.Field(
        CreateRepositoryPayload,
        graphql_name="createRepository",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateRepositoryInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a new repository.

    Arguments:

    * `input` (`CreateRepositoryInput!`): Parameters for
      CreateRepository
    """

    create_repository_ruleset = sgqlc.types.Field(
        CreateRepositoryRulesetPayload,
        graphql_name="createRepositoryRuleset",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateRepositoryRulesetInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a repository ruleset

    Arguments:

    * `input` (`CreateRepositoryRulesetInput!`): Parameters for
      CreateRepositoryRuleset
    """

    create_sponsors_listing = sgqlc.types.Field(
        CreateSponsorsListingPayload,
        graphql_name="createSponsorsListing",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateSponsorsListingInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a GitHub Sponsors profile to allow others to sponsor you or
    your organization.

    Arguments:

    * `input` (`CreateSponsorsListingInput!`): Parameters for
      CreateSponsorsListing
    """

    create_sponsors_tier = sgqlc.types.Field(
        CreateSponsorsTierPayload,
        graphql_name="createSponsorsTier",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateSponsorsTierInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a new payment tier for your GitHub Sponsors profile.

    Arguments:

    * `input` (`CreateSponsorsTierInput!`): Parameters for
      CreateSponsorsTier
    """

    create_sponsorship = sgqlc.types.Field(
        CreateSponsorshipPayload,
        graphql_name="createSponsorship",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateSponsorshipInput), graphql_name="input", default=None)),)
        ),
    )
    """Start a new sponsorship of a maintainer in GitHub Sponsors, or
    reactivate a past sponsorship.

    Arguments:

    * `input` (`CreateSponsorshipInput!`): Parameters for
      CreateSponsorship
    """

    create_sponsorships = sgqlc.types.Field(
        CreateSponsorshipsPayload,
        graphql_name="createSponsorships",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateSponsorshipsInput), graphql_name="input", default=None)),)
        ),
    )
    """Make many one-time sponsorships for different sponsorable users or
    organizations at once. Can only sponsor those who have a public
    GitHub Sponsors profile.

    Arguments:

    * `input` (`CreateSponsorshipsInput!`): Parameters for
      CreateSponsorships
    """

    create_team_discussion = sgqlc.types.Field(
        CreateTeamDiscussionPayload,
        graphql_name="createTeamDiscussion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateTeamDiscussionInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates a new team discussion.

    Arguments:

    * `input` (`CreateTeamDiscussionInput!`): Parameters for
      CreateTeamDiscussion
    """

    create_team_discussion_comment = sgqlc.types.Field(
        CreateTeamDiscussionCommentPayload,
        graphql_name="createTeamDiscussionComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(CreateTeamDiscussionCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates a new team discussion comment.

    Arguments:

    * `input` (`CreateTeamDiscussionCommentInput!`): Parameters for
      CreateTeamDiscussionComment
    """

    decline_topic_suggestion = sgqlc.types.Field(
        DeclineTopicSuggestionPayload,
        graphql_name="declineTopicSuggestion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeclineTopicSuggestionInput), graphql_name="input", default=None)),)
        ),
    )
    """Rejects a suggested topic for the repository.

    Arguments:

    * `input` (`DeclineTopicSuggestionInput!`): Parameters for
      DeclineTopicSuggestion
    """

    delete_branch_protection_rule = sgqlc.types.Field(
        DeleteBranchProtectionRulePayload,
        graphql_name="deleteBranchProtectionRule",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteBranchProtectionRuleInput), graphql_name="input", default=None)),)
        ),
    )
    """Delete a branch protection rule

    Arguments:

    * `input` (`DeleteBranchProtectionRuleInput!`): Parameters for
      DeleteBranchProtectionRule
    """

    delete_deployment = sgqlc.types.Field(
        DeleteDeploymentPayload,
        graphql_name="deleteDeployment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteDeploymentInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a deployment.

    Arguments:

    * `input` (`DeleteDeploymentInput!`): Parameters for
      DeleteDeployment
    """

    delete_discussion = sgqlc.types.Field(
        DeleteDiscussionPayload,
        graphql_name="deleteDiscussion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteDiscussionInput), graphql_name="input", default=None)),)
        ),
    )
    """Delete a discussion and all of its replies.

    Arguments:

    * `input` (`DeleteDiscussionInput!`): Parameters for
      DeleteDiscussion
    """

    delete_discussion_comment = sgqlc.types.Field(
        DeleteDiscussionCommentPayload,
        graphql_name="deleteDiscussionComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteDiscussionCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Delete a discussion comment. If it has replies, wipe it instead.

    Arguments:

    * `input` (`DeleteDiscussionCommentInput!`): Parameters for
      DeleteDiscussionComment
    """

    delete_environment = sgqlc.types.Field(
        DeleteEnvironmentPayload,
        graphql_name="deleteEnvironment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteEnvironmentInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes an environment

    Arguments:

    * `input` (`DeleteEnvironmentInput!`): Parameters for
      DeleteEnvironment
    """

    delete_ip_allow_list_entry = sgqlc.types.Field(
        DeleteIpAllowListEntryPayload,
        graphql_name="deleteIpAllowListEntry",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteIpAllowListEntryInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes an IP allow list entry.

    Arguments:

    * `input` (`DeleteIpAllowListEntryInput!`): Parameters for
      DeleteIpAllowListEntry
    """

    delete_issue = sgqlc.types.Field(
        DeleteIssuePayload,
        graphql_name="deleteIssue",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteIssueInput), graphql_name="input", default=None)),)),
    )
    """Deletes an Issue object.

    Arguments:

    * `input` (`DeleteIssueInput!`): Parameters for DeleteIssue
    """

    delete_issue_comment = sgqlc.types.Field(
        DeleteIssueCommentPayload,
        graphql_name="deleteIssueComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteIssueCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes an IssueComment object.

    Arguments:

    * `input` (`DeleteIssueCommentInput!`): Parameters for
      DeleteIssueComment
    """

    delete_linked_branch = sgqlc.types.Field(
        DeleteLinkedBranchPayload,
        graphql_name="deleteLinkedBranch",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteLinkedBranchInput), graphql_name="input", default=None)),)
        ),
    )
    """Unlink a branch from an issue.

    Arguments:

    * `input` (`DeleteLinkedBranchInput!`): Parameters for
      DeleteLinkedBranch
    """

    delete_project = sgqlc.types.Field(
        DeleteProjectPayload,
        graphql_name="deleteProject",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteProjectInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a project.

    Arguments:

    * `input` (`DeleteProjectInput!`): Parameters for DeleteProject
    """

    delete_project_card = sgqlc.types.Field(
        DeleteProjectCardPayload,
        graphql_name="deleteProjectCard",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteProjectCardInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a project card.

    Arguments:

    * `input` (`DeleteProjectCardInput!`): Parameters for
      DeleteProjectCard
    """

    delete_project_column = sgqlc.types.Field(
        DeleteProjectColumnPayload,
        graphql_name="deleteProjectColumn",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteProjectColumnInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a project column.

    Arguments:

    * `input` (`DeleteProjectColumnInput!`): Parameters for
      DeleteProjectColumn
    """

    delete_project_v2 = sgqlc.types.Field(
        DeleteProjectV2Payload,
        graphql_name="deleteProjectV2",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteProjectV2Input), graphql_name="input", default=None)),)
        ),
    )
    """Delete a project.

    Arguments:

    * `input` (`DeleteProjectV2Input!`): Parameters for
      DeleteProjectV2
    """

    delete_project_v2_field = sgqlc.types.Field(
        DeleteProjectV2FieldPayload,
        graphql_name="deleteProjectV2Field",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteProjectV2FieldInput), graphql_name="input", default=None)),)
        ),
    )
    """Delete a project field.

    Arguments:

    * `input` (`DeleteProjectV2FieldInput!`): Parameters for
      DeleteProjectV2Field
    """

    delete_project_v2_item = sgqlc.types.Field(
        DeleteProjectV2ItemPayload,
        graphql_name="deleteProjectV2Item",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteProjectV2ItemInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes an item from a Project.

    Arguments:

    * `input` (`DeleteProjectV2ItemInput!`): Parameters for
      DeleteProjectV2Item
    """

    delete_project_v2_workflow = sgqlc.types.Field(
        DeleteProjectV2WorkflowPayload,
        graphql_name="deleteProjectV2Workflow",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteProjectV2WorkflowInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a project workflow.

    Arguments:

    * `input` (`DeleteProjectV2WorkflowInput!`): Parameters for
      DeleteProjectV2Workflow
    """

    delete_pull_request_review = sgqlc.types.Field(
        DeletePullRequestReviewPayload,
        graphql_name="deletePullRequestReview",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeletePullRequestReviewInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a pull request review.

    Arguments:

    * `input` (`DeletePullRequestReviewInput!`): Parameters for
      DeletePullRequestReview
    """

    delete_pull_request_review_comment = sgqlc.types.Field(
        DeletePullRequestReviewCommentPayload,
        graphql_name="deletePullRequestReviewComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeletePullRequestReviewCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a pull request review comment.

    Arguments:

    * `input` (`DeletePullRequestReviewCommentInput!`): Parameters for
      DeletePullRequestReviewComment
    """

    delete_ref = sgqlc.types.Field(
        DeleteRefPayload,
        graphql_name="deleteRef",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteRefInput), graphql_name="input", default=None)),)),
    )
    """Delete a Git Ref.

    Arguments:

    * `input` (`DeleteRefInput!`): Parameters for DeleteRef
    """

    delete_repository_ruleset = sgqlc.types.Field(
        DeleteRepositoryRulesetPayload,
        graphql_name="deleteRepositoryRuleset",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteRepositoryRulesetInput), graphql_name="input", default=None)),)
        ),
    )
    """Delete a repository ruleset

    Arguments:

    * `input` (`DeleteRepositoryRulesetInput!`): Parameters for
      DeleteRepositoryRuleset
    """

    delete_team_discussion = sgqlc.types.Field(
        DeleteTeamDiscussionPayload,
        graphql_name="deleteTeamDiscussion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteTeamDiscussionInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a team discussion.

    Arguments:

    * `input` (`DeleteTeamDiscussionInput!`): Parameters for
      DeleteTeamDiscussion
    """

    delete_team_discussion_comment = sgqlc.types.Field(
        DeleteTeamDiscussionCommentPayload,
        graphql_name="deleteTeamDiscussionComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteTeamDiscussionCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a team discussion comment.

    Arguments:

    * `input` (`DeleteTeamDiscussionCommentInput!`): Parameters for
      DeleteTeamDiscussionComment
    """

    delete_verifiable_domain = sgqlc.types.Field(
        DeleteVerifiableDomainPayload,
        graphql_name="deleteVerifiableDomain",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DeleteVerifiableDomainInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a verifiable domain.

    Arguments:

    * `input` (`DeleteVerifiableDomainInput!`): Parameters for
      DeleteVerifiableDomain
    """

    dequeue_pull_request = sgqlc.types.Field(
        DequeuePullRequestPayload,
        graphql_name="dequeuePullRequest",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DequeuePullRequestInput), graphql_name="input", default=None)),)
        ),
    )
    """Remove a pull request from the merge queue.

    Arguments:

    * `input` (`DequeuePullRequestInput!`): Parameters for
      DequeuePullRequest
    """

    disable_pull_request_auto_merge = sgqlc.types.Field(
        DisablePullRequestAutoMergePayload,
        graphql_name="disablePullRequestAutoMerge",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DisablePullRequestAutoMergeInput), graphql_name="input", default=None)),)
        ),
    )
    """Disable auto merge on the given pull request

    Arguments:

    * `input` (`DisablePullRequestAutoMergeInput!`): Parameters for
      DisablePullRequestAutoMerge
    """

    dismiss_pull_request_review = sgqlc.types.Field(
        DismissPullRequestReviewPayload,
        graphql_name="dismissPullRequestReview",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(DismissPullRequestReviewInput), graphql_name="input", default=None)),)
        ),
    )
    """Dismisses an approved or rejected pull request review.

    Arguments:

    * `input` (`DismissPullRequestReviewInput!`): Parameters for
      DismissPullRequestReview
    """

    dismiss_repository_vulnerability_alert = sgqlc.types.Field(
        DismissRepositoryVulnerabilityAlertPayload,
        graphql_name="dismissRepositoryVulnerabilityAlert",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(sgqlc.types.non_null(DismissRepositoryVulnerabilityAlertInput), graphql_name="input", default=None),
                ),
            )
        ),
    )
    """Dismisses the Dependabot alert.

    Arguments:

    * `input` (`DismissRepositoryVulnerabilityAlertInput!`):
      Parameters for DismissRepositoryVulnerabilityAlert
    """

    enable_pull_request_auto_merge = sgqlc.types.Field(
        EnablePullRequestAutoMergePayload,
        graphql_name="enablePullRequestAutoMerge",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(EnablePullRequestAutoMergeInput), graphql_name="input", default=None)),)
        ),
    )
    """Enable the default auto-merge on a pull request.

    Arguments:

    * `input` (`EnablePullRequestAutoMergeInput!`): Parameters for
      EnablePullRequestAutoMerge
    """

    enqueue_pull_request = sgqlc.types.Field(
        EnqueuePullRequestPayload,
        graphql_name="enqueuePullRequest",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(EnqueuePullRequestInput), graphql_name="input", default=None)),)
        ),
    )
    """Add a pull request to the merge queue.

    Arguments:

    * `input` (`EnqueuePullRequestInput!`): Parameters for
      EnqueuePullRequest
    """

    follow_organization = sgqlc.types.Field(
        FollowOrganizationPayload,
        graphql_name="followOrganization",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(FollowOrganizationInput), graphql_name="input", default=None)),)
        ),
    )
    """Follow an organization.

    Arguments:

    * `input` (`FollowOrganizationInput!`): Parameters for
      FollowOrganization
    """

    follow_user = sgqlc.types.Field(
        FollowUserPayload,
        graphql_name="followUser",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(FollowUserInput), graphql_name="input", default=None)),)),
    )
    """Follow a user.

    Arguments:

    * `input` (`FollowUserInput!`): Parameters for FollowUser
    """

    grant_enterprise_organizations_migrator_role = sgqlc.types.Field(
        GrantEnterpriseOrganizationsMigratorRolePayload,
        graphql_name="grantEnterpriseOrganizationsMigratorRole",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(GrantEnterpriseOrganizationsMigratorRoleInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Grant the migrator role to a user for all organizations under an
    enterprise account.

    Arguments:

    * `input` (`GrantEnterpriseOrganizationsMigratorRoleInput!`):
      Parameters for GrantEnterpriseOrganizationsMigratorRole
    """

    grant_migrator_role = sgqlc.types.Field(
        GrantMigratorRolePayload,
        graphql_name="grantMigratorRole",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(GrantMigratorRoleInput), graphql_name="input", default=None)),)
        ),
    )
    """Grant the migrator role to a user or a team.

    Arguments:

    * `input` (`GrantMigratorRoleInput!`): Parameters for
      GrantMigratorRole
    """

    invite_enterprise_admin = sgqlc.types.Field(
        InviteEnterpriseAdminPayload,
        graphql_name="inviteEnterpriseAdmin",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(InviteEnterpriseAdminInput), graphql_name="input", default=None)),)
        ),
    )
    """Invite someone to become an administrator of the enterprise.

    Arguments:

    * `input` (`InviteEnterpriseAdminInput!`): Parameters for
      InviteEnterpriseAdmin
    """

    link_project_v2_to_repository = sgqlc.types.Field(
        LinkProjectV2ToRepositoryPayload,
        graphql_name="linkProjectV2ToRepository",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(LinkProjectV2ToRepositoryInput), graphql_name="input", default=None)),)
        ),
    )
    """Links a project to a repository.

    Arguments:

    * `input` (`LinkProjectV2ToRepositoryInput!`): Parameters for
      LinkProjectV2ToRepository
    """

    link_project_v2_to_team = sgqlc.types.Field(
        LinkProjectV2ToTeamPayload,
        graphql_name="linkProjectV2ToTeam",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(LinkProjectV2ToTeamInput), graphql_name="input", default=None)),)
        ),
    )
    """Links a project to a team.

    Arguments:

    * `input` (`LinkProjectV2ToTeamInput!`): Parameters for
      LinkProjectV2ToTeam
    """

    link_repository_to_project = sgqlc.types.Field(
        LinkRepositoryToProjectPayload,
        graphql_name="linkRepositoryToProject",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(LinkRepositoryToProjectInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates a repository link for a project.

    Arguments:

    * `input` (`LinkRepositoryToProjectInput!`): Parameters for
      LinkRepositoryToProject
    """

    lock_lockable = sgqlc.types.Field(
        LockLockablePayload,
        graphql_name="lockLockable",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(LockLockableInput), graphql_name="input", default=None)),)
        ),
    )
    """Lock a lockable object

    Arguments:

    * `input` (`LockLockableInput!`): Parameters for LockLockable
    """

    mark_discussion_comment_as_answer = sgqlc.types.Field(
        MarkDiscussionCommentAsAnswerPayload,
        graphql_name="markDiscussionCommentAsAnswer",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(MarkDiscussionCommentAsAnswerInput), graphql_name="input", default=None)),)
        ),
    )
    """Mark a discussion comment as the chosen answer for discussions in
    an answerable category.

    Arguments:

    * `input` (`MarkDiscussionCommentAsAnswerInput!`): Parameters for
      MarkDiscussionCommentAsAnswer
    """

    mark_file_as_viewed = sgqlc.types.Field(
        MarkFileAsViewedPayload,
        graphql_name="markFileAsViewed",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(MarkFileAsViewedInput), graphql_name="input", default=None)),)
        ),
    )
    """Mark a pull request file as viewed

    Arguments:

    * `input` (`MarkFileAsViewedInput!`): Parameters for
      MarkFileAsViewed
    """

    mark_project_v2_as_template = sgqlc.types.Field(
        MarkProjectV2AsTemplatePayload,
        graphql_name="markProjectV2AsTemplate",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(MarkProjectV2AsTemplateInput), graphql_name="input", default=None)),)
        ),
    )
    """Mark a project as a template. Note that only projects which are
    owned by an Organization can be marked as a template.

    Arguments:

    * `input` (`MarkProjectV2AsTemplateInput!`): Parameters for
      MarkProjectV2AsTemplate
    """

    mark_pull_request_ready_for_review = sgqlc.types.Field(
        MarkPullRequestReadyForReviewPayload,
        graphql_name="markPullRequestReadyForReview",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(MarkPullRequestReadyForReviewInput), graphql_name="input", default=None)),)
        ),
    )
    """Marks a pull request ready for review.

    Arguments:

    * `input` (`MarkPullRequestReadyForReviewInput!`): Parameters for
      MarkPullRequestReadyForReview
    """

    merge_branch = sgqlc.types.Field(
        MergeBranchPayload,
        graphql_name="mergeBranch",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(MergeBranchInput), graphql_name="input", default=None)),)),
    )
    """Merge a head into a branch.

    Arguments:

    * `input` (`MergeBranchInput!`): Parameters for MergeBranch
    """

    merge_pull_request = sgqlc.types.Field(
        MergePullRequestPayload,
        graphql_name="mergePullRequest",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(MergePullRequestInput), graphql_name="input", default=None)),)
        ),
    )
    """Merge a pull request.

    Arguments:

    * `input` (`MergePullRequestInput!`): Parameters for
      MergePullRequest
    """

    minimize_comment = sgqlc.types.Field(
        MinimizeCommentPayload,
        graphql_name="minimizeComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(MinimizeCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Minimizes a comment on an Issue, Commit, Pull Request, or Gist

    Arguments:

    * `input` (`MinimizeCommentInput!`): Parameters for
      MinimizeComment
    """

    move_project_card = sgqlc.types.Field(
        MoveProjectCardPayload,
        graphql_name="moveProjectCard",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(MoveProjectCardInput), graphql_name="input", default=None)),)
        ),
    )
    """Moves a project card to another place.

    Arguments:

    * `input` (`MoveProjectCardInput!`): Parameters for
      MoveProjectCard
    """

    move_project_column = sgqlc.types.Field(
        MoveProjectColumnPayload,
        graphql_name="moveProjectColumn",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(MoveProjectColumnInput), graphql_name="input", default=None)),)
        ),
    )
    """Moves a project column to another place.

    Arguments:

    * `input` (`MoveProjectColumnInput!`): Parameters for
      MoveProjectColumn
    """

    pin_issue = sgqlc.types.Field(
        "PinIssuePayload",
        graphql_name="pinIssue",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(PinIssueInput), graphql_name="input", default=None)),)),
    )
    """Pin an issue to a repository

    Arguments:

    * `input` (`PinIssueInput!`): Parameters for PinIssue
    """

    publish_sponsors_tier = sgqlc.types.Field(
        "PublishSponsorsTierPayload",
        graphql_name="publishSponsorsTier",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(PublishSponsorsTierInput), graphql_name="input", default=None)),)
        ),
    )
    """Publish an existing sponsorship tier that is currently still a
    draft to a GitHub Sponsors profile.

    Arguments:

    * `input` (`PublishSponsorsTierInput!`): Parameters for
      PublishSponsorsTier
    """

    regenerate_enterprise_identity_provider_recovery_codes = sgqlc.types.Field(
        "RegenerateEnterpriseIdentityProviderRecoveryCodesPayload",
        graphql_name="regenerateEnterpriseIdentityProviderRecoveryCodes",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(RegenerateEnterpriseIdentityProviderRecoveryCodesInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Regenerates the identity provider recovery codes for an enterprise

    Arguments:

    * `input`
      (`RegenerateEnterpriseIdentityProviderRecoveryCodesInput!`):
      Parameters for RegenerateEnterpriseIdentityProviderRecoveryCodes
    """

    regenerate_verifiable_domain_token = sgqlc.types.Field(
        "RegenerateVerifiableDomainTokenPayload",
        graphql_name="regenerateVerifiableDomainToken",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RegenerateVerifiableDomainTokenInput), graphql_name="input", default=None)),)
        ),
    )
    """Regenerates a verifiable domain's verification token.

    Arguments:

    * `input` (`RegenerateVerifiableDomainTokenInput!`): Parameters
      for RegenerateVerifiableDomainToken
    """

    reject_deployments = sgqlc.types.Field(
        "RejectDeploymentsPayload",
        graphql_name="rejectDeployments",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RejectDeploymentsInput), graphql_name="input", default=None)),)
        ),
    )
    """Reject all pending deployments under one or more environments

    Arguments:

    * `input` (`RejectDeploymentsInput!`): Parameters for
      RejectDeployments
    """

    remove_assignees_from_assignable = sgqlc.types.Field(
        "RemoveAssigneesFromAssignablePayload",
        graphql_name="removeAssigneesFromAssignable",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveAssigneesFromAssignableInput), graphql_name="input", default=None)),)
        ),
    )
    """Removes assignees from an assignable object.

    Arguments:

    * `input` (`RemoveAssigneesFromAssignableInput!`): Parameters for
      RemoveAssigneesFromAssignable
    """

    remove_enterprise_admin = sgqlc.types.Field(
        "RemoveEnterpriseAdminPayload",
        graphql_name="removeEnterpriseAdmin",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveEnterpriseAdminInput), graphql_name="input", default=None)),)
        ),
    )
    """Removes an administrator from the enterprise.

    Arguments:

    * `input` (`RemoveEnterpriseAdminInput!`): Parameters for
      RemoveEnterpriseAdmin
    """

    remove_enterprise_identity_provider = sgqlc.types.Field(
        "RemoveEnterpriseIdentityProviderPayload",
        graphql_name="removeEnterpriseIdentityProvider",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveEnterpriseIdentityProviderInput), graphql_name="input", default=None)),)
        ),
    )
    """Removes the identity provider from an enterprise

    Arguments:

    * `input` (`RemoveEnterpriseIdentityProviderInput!`): Parameters
      for RemoveEnterpriseIdentityProvider
    """

    remove_enterprise_member = sgqlc.types.Field(
        "RemoveEnterpriseMemberPayload",
        graphql_name="removeEnterpriseMember",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveEnterpriseMemberInput), graphql_name="input", default=None)),)
        ),
    )
    """Removes a user from all organizations within the enterprise

    Arguments:

    * `input` (`RemoveEnterpriseMemberInput!`): Parameters for
      RemoveEnterpriseMember
    """

    remove_enterprise_organization = sgqlc.types.Field(
        "RemoveEnterpriseOrganizationPayload",
        graphql_name="removeEnterpriseOrganization",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveEnterpriseOrganizationInput), graphql_name="input", default=None)),)
        ),
    )
    """Removes an organization from the enterprise

    Arguments:

    * `input` (`RemoveEnterpriseOrganizationInput!`): Parameters for
      RemoveEnterpriseOrganization
    """

    remove_enterprise_support_entitlement = sgqlc.types.Field(
        "RemoveEnterpriseSupportEntitlementPayload",
        graphql_name="removeEnterpriseSupportEntitlement",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveEnterpriseSupportEntitlementInput), graphql_name="input", default=None)),)
        ),
    )
    """Removes a support entitlement from an enterprise member.

    Arguments:

    * `input` (`RemoveEnterpriseSupportEntitlementInput!`): Parameters
      for RemoveEnterpriseSupportEntitlement
    """

    remove_labels_from_labelable = sgqlc.types.Field(
        "RemoveLabelsFromLabelablePayload",
        graphql_name="removeLabelsFromLabelable",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveLabelsFromLabelableInput), graphql_name="input", default=None)),)
        ),
    )
    """Removes labels from a Labelable object.

    Arguments:

    * `input` (`RemoveLabelsFromLabelableInput!`): Parameters for
      RemoveLabelsFromLabelable
    """

    remove_outside_collaborator = sgqlc.types.Field(
        "RemoveOutsideCollaboratorPayload",
        graphql_name="removeOutsideCollaborator",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveOutsideCollaboratorInput), graphql_name="input", default=None)),)
        ),
    )
    """Removes outside collaborator from all repositories in an
    organization.

    Arguments:

    * `input` (`RemoveOutsideCollaboratorInput!`): Parameters for
      RemoveOutsideCollaborator
    """

    remove_reaction = sgqlc.types.Field(
        "RemoveReactionPayload",
        graphql_name="removeReaction",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveReactionInput), graphql_name="input", default=None)),)
        ),
    )
    """Removes a reaction from a subject.

    Arguments:

    * `input` (`RemoveReactionInput!`): Parameters for RemoveReaction
    """

    remove_star = sgqlc.types.Field(
        "RemoveStarPayload",
        graphql_name="removeStar",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveStarInput), graphql_name="input", default=None)),)),
    )
    """Removes a star from a Starrable.

    Arguments:

    * `input` (`RemoveStarInput!`): Parameters for RemoveStar
    """

    remove_upvote = sgqlc.types.Field(
        "RemoveUpvotePayload",
        graphql_name="removeUpvote",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RemoveUpvoteInput), graphql_name="input", default=None)),)
        ),
    )
    """Remove an upvote to a discussion or discussion comment.

    Arguments:

    * `input` (`RemoveUpvoteInput!`): Parameters for RemoveUpvote
    """

    reopen_discussion = sgqlc.types.Field(
        "ReopenDiscussionPayload",
        graphql_name="reopenDiscussion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ReopenDiscussionInput), graphql_name="input", default=None)),)
        ),
    )
    """Reopen a discussion.

    Arguments:

    * `input` (`ReopenDiscussionInput!`): Parameters for
      ReopenDiscussion
    """

    reopen_issue = sgqlc.types.Field(
        "ReopenIssuePayload",
        graphql_name="reopenIssue",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(ReopenIssueInput), graphql_name="input", default=None)),)),
    )
    """Reopen a issue.

    Arguments:

    * `input` (`ReopenIssueInput!`): Parameters for ReopenIssue
    """

    reopen_pull_request = sgqlc.types.Field(
        "ReopenPullRequestPayload",
        graphql_name="reopenPullRequest",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ReopenPullRequestInput), graphql_name="input", default=None)),)
        ),
    )
    """Reopen a pull request.

    Arguments:

    * `input` (`ReopenPullRequestInput!`): Parameters for
      ReopenPullRequest
    """

    request_reviews = sgqlc.types.Field(
        "RequestReviewsPayload",
        graphql_name="requestReviews",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RequestReviewsInput), graphql_name="input", default=None)),)
        ),
    )
    """Set review requests on a pull request.

    Arguments:

    * `input` (`RequestReviewsInput!`): Parameters for RequestReviews
    """

    rerequest_check_suite = sgqlc.types.Field(
        "RerequestCheckSuitePayload",
        graphql_name="rerequestCheckSuite",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RerequestCheckSuiteInput), graphql_name="input", default=None)),)
        ),
    )
    """Rerequests an existing check suite.

    Arguments:

    * `input` (`RerequestCheckSuiteInput!`): Parameters for
      RerequestCheckSuite
    """

    resolve_review_thread = sgqlc.types.Field(
        "ResolveReviewThreadPayload",
        graphql_name="resolveReviewThread",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(ResolveReviewThreadInput), graphql_name="input", default=None)),)
        ),
    )
    """Marks a review thread as resolved.

    Arguments:

    * `input` (`ResolveReviewThreadInput!`): Parameters for
      ResolveReviewThread
    """

    retire_sponsors_tier = sgqlc.types.Field(
        "RetireSponsorsTierPayload",
        graphql_name="retireSponsorsTier",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RetireSponsorsTierInput), graphql_name="input", default=None)),)
        ),
    )
    """Retire a published payment tier from your GitHub Sponsors profile
    so it cannot be used to start new sponsorships.

    Arguments:

    * `input` (`RetireSponsorsTierInput!`): Parameters for
      RetireSponsorsTier
    """

    revert_pull_request = sgqlc.types.Field(
        "RevertPullRequestPayload",
        graphql_name="revertPullRequest",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RevertPullRequestInput), graphql_name="input", default=None)),)
        ),
    )
    """Create a pull request that reverts the changes from a merged pull
    request.

    Arguments:

    * `input` (`RevertPullRequestInput!`): Parameters for
      RevertPullRequest
    """

    revoke_enterprise_organizations_migrator_role = sgqlc.types.Field(
        "RevokeEnterpriseOrganizationsMigratorRolePayload",
        graphql_name="revokeEnterpriseOrganizationsMigratorRole",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(RevokeEnterpriseOrganizationsMigratorRoleInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Revoke the migrator role to a user for all organizations under an
    enterprise account.

    Arguments:

    * `input` (`RevokeEnterpriseOrganizationsMigratorRoleInput!`):
      Parameters for RevokeEnterpriseOrganizationsMigratorRole
    """

    revoke_migrator_role = sgqlc.types.Field(
        "RevokeMigratorRolePayload",
        graphql_name="revokeMigratorRole",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(RevokeMigratorRoleInput), graphql_name="input", default=None)),)
        ),
    )
    """Revoke the migrator role from a user or a team.

    Arguments:

    * `input` (`RevokeMigratorRoleInput!`): Parameters for
      RevokeMigratorRole
    """

    set_enterprise_identity_provider = sgqlc.types.Field(
        "SetEnterpriseIdentityProviderPayload",
        graphql_name="setEnterpriseIdentityProvider",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(SetEnterpriseIdentityProviderInput), graphql_name="input", default=None)),)
        ),
    )
    """Creates or updates the identity provider for an enterprise.

    Arguments:

    * `input` (`SetEnterpriseIdentityProviderInput!`): Parameters for
      SetEnterpriseIdentityProvider
    """

    set_organization_interaction_limit = sgqlc.types.Field(
        "SetOrganizationInteractionLimitPayload",
        graphql_name="setOrganizationInteractionLimit",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(SetOrganizationInteractionLimitInput), graphql_name="input", default=None)),)
        ),
    )
    """Set an organization level interaction limit for an organization's
    public repositories.

    Arguments:

    * `input` (`SetOrganizationInteractionLimitInput!`): Parameters
      for SetOrganizationInteractionLimit
    """

    set_repository_interaction_limit = sgqlc.types.Field(
        "SetRepositoryInteractionLimitPayload",
        graphql_name="setRepositoryInteractionLimit",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(SetRepositoryInteractionLimitInput), graphql_name="input", default=None)),)
        ),
    )
    """Sets an interaction limit setting for a repository.

    Arguments:

    * `input` (`SetRepositoryInteractionLimitInput!`): Parameters for
      SetRepositoryInteractionLimit
    """

    set_user_interaction_limit = sgqlc.types.Field(
        "SetUserInteractionLimitPayload",
        graphql_name="setUserInteractionLimit",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(SetUserInteractionLimitInput), graphql_name="input", default=None)),)
        ),
    )
    """Set a user level interaction limit for an user's public
    repositories.

    Arguments:

    * `input` (`SetUserInteractionLimitInput!`): Parameters for
      SetUserInteractionLimit
    """

    start_organization_migration = sgqlc.types.Field(
        "StartOrganizationMigrationPayload",
        graphql_name="startOrganizationMigration",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(StartOrganizationMigrationInput), graphql_name="input", default=None)),)
        ),
    )
    """Starts a GitHub Enterprise Importer organization migration.

    Arguments:

    * `input` (`StartOrganizationMigrationInput!`): Parameters for
      StartOrganizationMigration
    """

    start_repository_migration = sgqlc.types.Field(
        "StartRepositoryMigrationPayload",
        graphql_name="startRepositoryMigration",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(StartRepositoryMigrationInput), graphql_name="input", default=None)),)
        ),
    )
    """Starts a GitHub Enterprise Importer (GEI) repository migration.

    Arguments:

    * `input` (`StartRepositoryMigrationInput!`): Parameters for
      StartRepositoryMigration
    """

    submit_pull_request_review = sgqlc.types.Field(
        "SubmitPullRequestReviewPayload",
        graphql_name="submitPullRequestReview",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(SubmitPullRequestReviewInput), graphql_name="input", default=None)),)
        ),
    )
    """Submits a pending pull request review.

    Arguments:

    * `input` (`SubmitPullRequestReviewInput!`): Parameters for
      SubmitPullRequestReview
    """

    transfer_enterprise_organization = sgqlc.types.Field(
        "TransferEnterpriseOrganizationPayload",
        graphql_name="transferEnterpriseOrganization",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(TransferEnterpriseOrganizationInput), graphql_name="input", default=None)),)
        ),
    )
    """Transfer an organization from one enterprise to another
    enterprise.

    Arguments:

    * `input` (`TransferEnterpriseOrganizationInput!`): Parameters for
      TransferEnterpriseOrganization
    """

    transfer_issue = sgqlc.types.Field(
        "TransferIssuePayload",
        graphql_name="transferIssue",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(TransferIssueInput), graphql_name="input", default=None)),)
        ),
    )
    """Transfer an issue to a different repository

    Arguments:

    * `input` (`TransferIssueInput!`): Parameters for TransferIssue
    """

    unarchive_project_v2_item = sgqlc.types.Field(
        "UnarchiveProjectV2ItemPayload",
        graphql_name="unarchiveProjectV2Item",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnarchiveProjectV2ItemInput), graphql_name="input", default=None)),)
        ),
    )
    """Unarchives a ProjectV2Item

    Arguments:

    * `input` (`UnarchiveProjectV2ItemInput!`): Parameters for
      UnarchiveProjectV2Item
    """

    unarchive_repository = sgqlc.types.Field(
        "UnarchiveRepositoryPayload",
        graphql_name="unarchiveRepository",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnarchiveRepositoryInput), graphql_name="input", default=None)),)
        ),
    )
    """Unarchives a repository.

    Arguments:

    * `input` (`UnarchiveRepositoryInput!`): Parameters for
      UnarchiveRepository
    """

    unfollow_organization = sgqlc.types.Field(
        "UnfollowOrganizationPayload",
        graphql_name="unfollowOrganization",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnfollowOrganizationInput), graphql_name="input", default=None)),)
        ),
    )
    """Unfollow an organization.

    Arguments:

    * `input` (`UnfollowOrganizationInput!`): Parameters for
      UnfollowOrganization
    """

    unfollow_user = sgqlc.types.Field(
        "UnfollowUserPayload",
        graphql_name="unfollowUser",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnfollowUserInput), graphql_name="input", default=None)),)
        ),
    )
    """Unfollow a user.

    Arguments:

    * `input` (`UnfollowUserInput!`): Parameters for UnfollowUser
    """

    unlink_project_v2_from_repository = sgqlc.types.Field(
        "UnlinkProjectV2FromRepositoryPayload",
        graphql_name="unlinkProjectV2FromRepository",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnlinkProjectV2FromRepositoryInput), graphql_name="input", default=None)),)
        ),
    )
    """Unlinks a project from a repository.

    Arguments:

    * `input` (`UnlinkProjectV2FromRepositoryInput!`): Parameters for
      UnlinkProjectV2FromRepository
    """

    unlink_project_v2_from_team = sgqlc.types.Field(
        "UnlinkProjectV2FromTeamPayload",
        graphql_name="unlinkProjectV2FromTeam",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnlinkProjectV2FromTeamInput), graphql_name="input", default=None)),)
        ),
    )
    """Unlinks a project to a team.

    Arguments:

    * `input` (`UnlinkProjectV2FromTeamInput!`): Parameters for
      UnlinkProjectV2FromTeam
    """

    unlink_repository_from_project = sgqlc.types.Field(
        "UnlinkRepositoryFromProjectPayload",
        graphql_name="unlinkRepositoryFromProject",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnlinkRepositoryFromProjectInput), graphql_name="input", default=None)),)
        ),
    )
    """Deletes a repository link from a project.

    Arguments:

    * `input` (`UnlinkRepositoryFromProjectInput!`): Parameters for
      UnlinkRepositoryFromProject
    """

    unlock_lockable = sgqlc.types.Field(
        "UnlockLockablePayload",
        graphql_name="unlockLockable",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnlockLockableInput), graphql_name="input", default=None)),)
        ),
    )
    """Unlock a lockable object

    Arguments:

    * `input` (`UnlockLockableInput!`): Parameters for UnlockLockable
    """

    unmark_discussion_comment_as_answer = sgqlc.types.Field(
        "UnmarkDiscussionCommentAsAnswerPayload",
        graphql_name="unmarkDiscussionCommentAsAnswer",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnmarkDiscussionCommentAsAnswerInput), graphql_name="input", default=None)),)
        ),
    )
    """Unmark a discussion comment as the chosen answer for discussions
    in an answerable category.

    Arguments:

    * `input` (`UnmarkDiscussionCommentAsAnswerInput!`): Parameters
      for UnmarkDiscussionCommentAsAnswer
    """

    unmark_file_as_viewed = sgqlc.types.Field(
        "UnmarkFileAsViewedPayload",
        graphql_name="unmarkFileAsViewed",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnmarkFileAsViewedInput), graphql_name="input", default=None)),)
        ),
    )
    """Unmark a pull request file as viewed

    Arguments:

    * `input` (`UnmarkFileAsViewedInput!`): Parameters for
      UnmarkFileAsViewed
    """

    unmark_issue_as_duplicate = sgqlc.types.Field(
        "UnmarkIssueAsDuplicatePayload",
        graphql_name="unmarkIssueAsDuplicate",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnmarkIssueAsDuplicateInput), graphql_name="input", default=None)),)
        ),
    )
    """Unmark an issue as a duplicate of another issue.

    Arguments:

    * `input` (`UnmarkIssueAsDuplicateInput!`): Parameters for
      UnmarkIssueAsDuplicate
    """

    unmark_project_v2_as_template = sgqlc.types.Field(
        "UnmarkProjectV2AsTemplatePayload",
        graphql_name="unmarkProjectV2AsTemplate",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnmarkProjectV2AsTemplateInput), graphql_name="input", default=None)),)
        ),
    )
    """Unmark a project as a template.

    Arguments:

    * `input` (`UnmarkProjectV2AsTemplateInput!`): Parameters for
      UnmarkProjectV2AsTemplate
    """

    unminimize_comment = sgqlc.types.Field(
        "UnminimizeCommentPayload",
        graphql_name="unminimizeComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnminimizeCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Unminimizes a comment on an Issue, Commit, Pull Request, or Gist

    Arguments:

    * `input` (`UnminimizeCommentInput!`): Parameters for
      UnminimizeComment
    """

    unpin_issue = sgqlc.types.Field(
        "UnpinIssuePayload",
        graphql_name="unpinIssue",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(UnpinIssueInput), graphql_name="input", default=None)),)),
    )
    """Unpin a pinned issue from a repository

    Arguments:

    * `input` (`UnpinIssueInput!`): Parameters for UnpinIssue
    """

    unresolve_review_thread = sgqlc.types.Field(
        "UnresolveReviewThreadPayload",
        graphql_name="unresolveReviewThread",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UnresolveReviewThreadInput), graphql_name="input", default=None)),)
        ),
    )
    """Marks a review thread as unresolved.

    Arguments:

    * `input` (`UnresolveReviewThreadInput!`): Parameters for
      UnresolveReviewThread
    """

    update_branch_protection_rule = sgqlc.types.Field(
        "UpdateBranchProtectionRulePayload",
        graphql_name="updateBranchProtectionRule",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateBranchProtectionRuleInput), graphql_name="input", default=None)),)
        ),
    )
    """Update a branch protection rule

    Arguments:

    * `input` (`UpdateBranchProtectionRuleInput!`): Parameters for
      UpdateBranchProtectionRule
    """

    update_check_run = sgqlc.types.Field(
        "UpdateCheckRunPayload",
        graphql_name="updateCheckRun",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateCheckRunInput), graphql_name="input", default=None)),)
        ),
    )
    """Update a check run

    Arguments:

    * `input` (`UpdateCheckRunInput!`): Parameters for UpdateCheckRun
    """

    update_check_suite_preferences = sgqlc.types.Field(
        "UpdateCheckSuitePreferencesPayload",
        graphql_name="updateCheckSuitePreferences",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateCheckSuitePreferencesInput), graphql_name="input", default=None)),)
        ),
    )
    """Modifies the settings of an existing check suite

    Arguments:

    * `input` (`UpdateCheckSuitePreferencesInput!`): Parameters for
      UpdateCheckSuitePreferences
    """

    update_discussion = sgqlc.types.Field(
        "UpdateDiscussionPayload",
        graphql_name="updateDiscussion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateDiscussionInput), graphql_name="input", default=None)),)
        ),
    )
    """Update a discussion

    Arguments:

    * `input` (`UpdateDiscussionInput!`): Parameters for
      UpdateDiscussion
    """

    update_discussion_comment = sgqlc.types.Field(
        "UpdateDiscussionCommentPayload",
        graphql_name="updateDiscussionComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateDiscussionCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Update the contents of a comment on a Discussion

    Arguments:

    * `input` (`UpdateDiscussionCommentInput!`): Parameters for
      UpdateDiscussionComment
    """

    update_enterprise_administrator_role = sgqlc.types.Field(
        "UpdateEnterpriseAdministratorRolePayload",
        graphql_name="updateEnterpriseAdministratorRole",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateEnterpriseAdministratorRoleInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates the role of an enterprise administrator.

    Arguments:

    * `input` (`UpdateEnterpriseAdministratorRoleInput!`): Parameters
      for UpdateEnterpriseAdministratorRole
    """

    update_enterprise_allow_private_repository_forking_setting = sgqlc.types.Field(
        "UpdateEnterpriseAllowPrivateRepositoryForkingSettingPayload",
        graphql_name="updateEnterpriseAllowPrivateRepositoryForkingSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseAllowPrivateRepositoryForkingSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets whether private repository forks are enabled for an
    enterprise.

    Arguments:

    * `input`
      (`UpdateEnterpriseAllowPrivateRepositoryForkingSettingInput!`):
      Parameters for
      UpdateEnterpriseAllowPrivateRepositoryForkingSetting
    """

    update_enterprise_default_repository_permission_setting = sgqlc.types.Field(
        "UpdateEnterpriseDefaultRepositoryPermissionSettingPayload",
        graphql_name="updateEnterpriseDefaultRepositoryPermissionSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseDefaultRepositoryPermissionSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets the base repository permission for organizations in an
    enterprise.

    Arguments:

    * `input`
      (`UpdateEnterpriseDefaultRepositoryPermissionSettingInput!`):
      Parameters for
      UpdateEnterpriseDefaultRepositoryPermissionSetting
    """

    update_enterprise_members_can_change_repository_visibility_setting = sgqlc.types.Field(
        "UpdateEnterpriseMembersCanChangeRepositoryVisibilitySettingPayload",
        graphql_name="updateEnterpriseMembersCanChangeRepositoryVisibilitySetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseMembersCanChangeRepositoryVisibilitySettingInput),
                        graphql_name="input",
                        default=None,
                    ),
                ),
            )
        ),
    )
    """Sets whether organization members with admin permissions on a
    repository can change repository visibility.

    Arguments:

    * `input` (`UpdateEnterpriseMembersCanChangeRepositoryVisibilitySe
      ttingInput!`): Parameters for
      UpdateEnterpriseMembersCanChangeRepositoryVisibilitySetting
    """

    update_enterprise_members_can_create_repositories_setting = sgqlc.types.Field(
        "UpdateEnterpriseMembersCanCreateRepositoriesSettingPayload",
        graphql_name="updateEnterpriseMembersCanCreateRepositoriesSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseMembersCanCreateRepositoriesSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets the members can create repositories setting for an
    enterprise.

    Arguments:

    * `input`
      (`UpdateEnterpriseMembersCanCreateRepositoriesSettingInput!`):
      Parameters for
      UpdateEnterpriseMembersCanCreateRepositoriesSetting
    """

    update_enterprise_members_can_delete_issues_setting = sgqlc.types.Field(
        "UpdateEnterpriseMembersCanDeleteIssuesSettingPayload",
        graphql_name="updateEnterpriseMembersCanDeleteIssuesSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseMembersCanDeleteIssuesSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets the members can delete issues setting for an enterprise.

    Arguments:

    * `input` (`UpdateEnterpriseMembersCanDeleteIssuesSettingInput!`):
      Parameters for UpdateEnterpriseMembersCanDeleteIssuesSetting
    """

    update_enterprise_members_can_delete_repositories_setting = sgqlc.types.Field(
        "UpdateEnterpriseMembersCanDeleteRepositoriesSettingPayload",
        graphql_name="updateEnterpriseMembersCanDeleteRepositoriesSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseMembersCanDeleteRepositoriesSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets the members can delete repositories setting for an
    enterprise.

    Arguments:

    * `input`
      (`UpdateEnterpriseMembersCanDeleteRepositoriesSettingInput!`):
      Parameters for
      UpdateEnterpriseMembersCanDeleteRepositoriesSetting
    """

    update_enterprise_members_can_invite_collaborators_setting = sgqlc.types.Field(
        "UpdateEnterpriseMembersCanInviteCollaboratorsSettingPayload",
        graphql_name="updateEnterpriseMembersCanInviteCollaboratorsSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseMembersCanInviteCollaboratorsSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets whether members can invite collaborators are enabled for an
    enterprise.

    Arguments:

    * `input`
      (`UpdateEnterpriseMembersCanInviteCollaboratorsSettingInput!`):
      Parameters for
      UpdateEnterpriseMembersCanInviteCollaboratorsSetting
    """

    update_enterprise_members_can_make_purchases_setting = sgqlc.types.Field(
        "UpdateEnterpriseMembersCanMakePurchasesSettingPayload",
        graphql_name="updateEnterpriseMembersCanMakePurchasesSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseMembersCanMakePurchasesSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets whether or not an organization admin can make purchases.

    Arguments:

    * `input`
      (`UpdateEnterpriseMembersCanMakePurchasesSettingInput!`):
      Parameters for UpdateEnterpriseMembersCanMakePurchasesSetting
    """

    update_enterprise_members_can_update_protected_branches_setting = sgqlc.types.Field(
        "UpdateEnterpriseMembersCanUpdateProtectedBranchesSettingPayload",
        graphql_name="updateEnterpriseMembersCanUpdateProtectedBranchesSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseMembersCanUpdateProtectedBranchesSettingInput),
                        graphql_name="input",
                        default=None,
                    ),
                ),
            )
        ),
    )
    """Sets the members can update protected branches setting for an
    enterprise.

    Arguments:

    * `input` (`UpdateEnterpriseMembersCanUpdateProtectedBranchesSetti
      ngInput!`): Parameters for
      UpdateEnterpriseMembersCanUpdateProtectedBranchesSetting
    """

    update_enterprise_members_can_view_dependency_insights_setting = sgqlc.types.Field(
        "UpdateEnterpriseMembersCanViewDependencyInsightsSettingPayload",
        graphql_name="updateEnterpriseMembersCanViewDependencyInsightsSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseMembersCanViewDependencyInsightsSettingInput),
                        graphql_name="input",
                        default=None,
                    ),
                ),
            )
        ),
    )
    """Sets the members can view dependency insights for an enterprise.

    Arguments:

    * `input` (`UpdateEnterpriseMembersCanViewDependencyInsightsSettin
      gInput!`): Parameters for
      UpdateEnterpriseMembersCanViewDependencyInsightsSetting
    """

    update_enterprise_organization_projects_setting = sgqlc.types.Field(
        "UpdateEnterpriseOrganizationProjectsSettingPayload",
        graphql_name="updateEnterpriseOrganizationProjectsSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseOrganizationProjectsSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets whether organization projects are enabled for an enterprise.

    Arguments:

    * `input` (`UpdateEnterpriseOrganizationProjectsSettingInput!`):
      Parameters for UpdateEnterpriseOrganizationProjectsSetting
    """

    update_enterprise_owner_organization_role = sgqlc.types.Field(
        "UpdateEnterpriseOwnerOrganizationRolePayload",
        graphql_name="updateEnterpriseOwnerOrganizationRole",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(sgqlc.types.non_null(UpdateEnterpriseOwnerOrganizationRoleInput), graphql_name="input", default=None),
                ),
            )
        ),
    )
    """Updates the role of an enterprise owner with an organization.

    Arguments:

    * `input` (`UpdateEnterpriseOwnerOrganizationRoleInput!`):
      Parameters for UpdateEnterpriseOwnerOrganizationRole
    """

    update_enterprise_profile = sgqlc.types.Field(
        "UpdateEnterpriseProfilePayload",
        graphql_name="updateEnterpriseProfile",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateEnterpriseProfileInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates an enterprise's profile.

    Arguments:

    * `input` (`UpdateEnterpriseProfileInput!`): Parameters for
      UpdateEnterpriseProfile
    """

    update_enterprise_repository_projects_setting = sgqlc.types.Field(
        "UpdateEnterpriseRepositoryProjectsSettingPayload",
        graphql_name="updateEnterpriseRepositoryProjectsSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseRepositoryProjectsSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets whether repository projects are enabled for a enterprise.

    Arguments:

    * `input` (`UpdateEnterpriseRepositoryProjectsSettingInput!`):
      Parameters for UpdateEnterpriseRepositoryProjectsSetting
    """

    update_enterprise_team_discussions_setting = sgqlc.types.Field(
        "UpdateEnterpriseTeamDiscussionsSettingPayload",
        graphql_name="updateEnterpriseTeamDiscussionsSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(sgqlc.types.non_null(UpdateEnterpriseTeamDiscussionsSettingInput), graphql_name="input", default=None),
                ),
            )
        ),
    )
    """Sets whether team discussions are enabled for an enterprise.

    Arguments:

    * `input` (`UpdateEnterpriseTeamDiscussionsSettingInput!`):
      Parameters for UpdateEnterpriseTeamDiscussionsSetting
    """

    update_enterprise_two_factor_authentication_required_setting = sgqlc.types.Field(
        "UpdateEnterpriseTwoFactorAuthenticationRequiredSettingPayload",
        graphql_name="updateEnterpriseTwoFactorAuthenticationRequiredSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateEnterpriseTwoFactorAuthenticationRequiredSettingInput),
                        graphql_name="input",
                        default=None,
                    ),
                ),
            )
        ),
    )
    """Sets whether two factor authentication is required for all users
    in an enterprise.

    Arguments:

    * `input` (`UpdateEnterpriseTwoFactorAuthenticationRequiredSetting
      Input!`): Parameters for
      UpdateEnterpriseTwoFactorAuthenticationRequiredSetting
    """

    update_environment = sgqlc.types.Field(
        "UpdateEnvironmentPayload",
        graphql_name="updateEnvironment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateEnvironmentInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates an environment.

    Arguments:

    * `input` (`UpdateEnvironmentInput!`): Parameters for
      UpdateEnvironment
    """

    update_ip_allow_list_enabled_setting = sgqlc.types.Field(
        "UpdateIpAllowListEnabledSettingPayload",
        graphql_name="updateIpAllowListEnabledSetting",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateIpAllowListEnabledSettingInput), graphql_name="input", default=None)),)
        ),
    )
    """Sets whether an IP allow list is enabled on an owner.

    Arguments:

    * `input` (`UpdateIpAllowListEnabledSettingInput!`): Parameters
      for UpdateIpAllowListEnabledSetting
    """

    update_ip_allow_list_entry = sgqlc.types.Field(
        "UpdateIpAllowListEntryPayload",
        graphql_name="updateIpAllowListEntry",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateIpAllowListEntryInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates an IP allow list entry.

    Arguments:

    * `input` (`UpdateIpAllowListEntryInput!`): Parameters for
      UpdateIpAllowListEntry
    """

    update_ip_allow_list_for_installed_apps_enabled_setting = sgqlc.types.Field(
        "UpdateIpAllowListForInstalledAppsEnabledSettingPayload",
        graphql_name="updateIpAllowListForInstalledAppsEnabledSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateIpAllowListForInstalledAppsEnabledSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets whether IP allow list configuration for installed GitHub Apps
    is enabled on an owner.

    Arguments:

    * `input`
      (`UpdateIpAllowListForInstalledAppsEnabledSettingInput!`):
      Parameters for UpdateIpAllowListForInstalledAppsEnabledSetting
    """

    update_issue = sgqlc.types.Field(
        "UpdateIssuePayload",
        graphql_name="updateIssue",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateIssueInput), graphql_name="input", default=None)),)),
    )
    """Updates an Issue.

    Arguments:

    * `input` (`UpdateIssueInput!`): Parameters for UpdateIssue
    """

    update_issue_comment = sgqlc.types.Field(
        "UpdateIssueCommentPayload",
        graphql_name="updateIssueComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateIssueCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates an IssueComment object.

    Arguments:

    * `input` (`UpdateIssueCommentInput!`): Parameters for
      UpdateIssueComment
    """

    update_notification_restriction_setting = sgqlc.types.Field(
        "UpdateNotificationRestrictionSettingPayload",
        graphql_name="updateNotificationRestrictionSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(sgqlc.types.non_null(UpdateNotificationRestrictionSettingInput), graphql_name="input", default=None),
                ),
            )
        ),
    )
    """Update the setting to restrict notifications to only verified or
    approved domains available to an owner.

    Arguments:

    * `input` (`UpdateNotificationRestrictionSettingInput!`):
      Parameters for UpdateNotificationRestrictionSetting
    """

    update_organization_allow_private_repository_forking_setting = sgqlc.types.Field(
        "UpdateOrganizationAllowPrivateRepositoryForkingSettingPayload",
        graphql_name="updateOrganizationAllowPrivateRepositoryForkingSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateOrganizationAllowPrivateRepositoryForkingSettingInput),
                        graphql_name="input",
                        default=None,
                    ),
                ),
            )
        ),
    )
    """Sets whether private repository forks are enabled for an
    organization.

    Arguments:

    * `input` (`UpdateOrganizationAllowPrivateRepositoryForkingSetting
      Input!`): Parameters for
      UpdateOrganizationAllowPrivateRepositoryForkingSetting
    """

    update_organization_web_commit_signoff_setting = sgqlc.types.Field(
        "UpdateOrganizationWebCommitSignoffSettingPayload",
        graphql_name="updateOrganizationWebCommitSignoffSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(
                        sgqlc.types.non_null(UpdateOrganizationWebCommitSignoffSettingInput), graphql_name="input", default=None
                    ),
                ),
            )
        ),
    )
    """Sets whether contributors are required to sign off on web-based
    commits for repositories in an organization.

    Arguments:

    * `input` (`UpdateOrganizationWebCommitSignoffSettingInput!`):
      Parameters for UpdateOrganizationWebCommitSignoffSetting
    """

    update_project = sgqlc.types.Field(
        "UpdateProjectPayload",
        graphql_name="updateProject",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateProjectInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates an existing project.

    Arguments:

    * `input` (`UpdateProjectInput!`): Parameters for UpdateProject
    """

    update_project_card = sgqlc.types.Field(
        "UpdateProjectCardPayload",
        graphql_name="updateProjectCard",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateProjectCardInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates an existing project card.

    Arguments:

    * `input` (`UpdateProjectCardInput!`): Parameters for
      UpdateProjectCard
    """

    update_project_column = sgqlc.types.Field(
        "UpdateProjectColumnPayload",
        graphql_name="updateProjectColumn",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateProjectColumnInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates an existing project column.

    Arguments:

    * `input` (`UpdateProjectColumnInput!`): Parameters for
      UpdateProjectColumn
    """

    update_project_v2 = sgqlc.types.Field(
        "UpdateProjectV2Payload",
        graphql_name="updateProjectV2",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateProjectV2Input), graphql_name="input", default=None)),)
        ),
    )
    """Updates an existing project (beta).

    Arguments:

    * `input` (`UpdateProjectV2Input!`): Parameters for
      UpdateProjectV2
    """

    update_project_v2_collaborators = sgqlc.types.Field(
        "UpdateProjectV2CollaboratorsPayload",
        graphql_name="updateProjectV2Collaborators",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateProjectV2CollaboratorsInput), graphql_name="input", default=None)),)
        ),
    )
    """Update the collaborators on a team or a project

    Arguments:

    * `input` (`UpdateProjectV2CollaboratorsInput!`): Parameters for
      UpdateProjectV2Collaborators
    """

    update_project_v2_draft_issue = sgqlc.types.Field(
        "UpdateProjectV2DraftIssuePayload",
        graphql_name="updateProjectV2DraftIssue",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateProjectV2DraftIssueInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates a draft issue within a Project.

    Arguments:

    * `input` (`UpdateProjectV2DraftIssueInput!`): Parameters for
      UpdateProjectV2DraftIssue
    """

    update_project_v2_item_field_value = sgqlc.types.Field(
        "UpdateProjectV2ItemFieldValuePayload",
        graphql_name="updateProjectV2ItemFieldValue",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateProjectV2ItemFieldValueInput), graphql_name="input", default=None)),)
        ),
    )
    """This mutation updates the value of a field for an item in a
    Project. Currently only single-select, text, number, date, and
    iteration fields are supported.

    Arguments:

    * `input` (`UpdateProjectV2ItemFieldValueInput!`): Parameters for
      UpdateProjectV2ItemFieldValue
    """

    update_project_v2_item_position = sgqlc.types.Field(
        "UpdateProjectV2ItemPositionPayload",
        graphql_name="updateProjectV2ItemPosition",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateProjectV2ItemPositionInput), graphql_name="input", default=None)),)
        ),
    )
    """This mutation updates the position of the item in the project,
    where the position represents the priority of an item.

    Arguments:

    * `input` (`UpdateProjectV2ItemPositionInput!`): Parameters for
      UpdateProjectV2ItemPosition
    """

    update_pull_request = sgqlc.types.Field(
        "UpdatePullRequestPayload",
        graphql_name="updatePullRequest",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdatePullRequestInput), graphql_name="input", default=None)),)
        ),
    )
    """Update a pull request

    Arguments:

    * `input` (`UpdatePullRequestInput!`): Parameters for
      UpdatePullRequest
    """

    update_pull_request_branch = sgqlc.types.Field(
        "UpdatePullRequestBranchPayload",
        graphql_name="updatePullRequestBranch",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdatePullRequestBranchInput), graphql_name="input", default=None)),)
        ),
    )
    """Merge or Rebase HEAD from upstream branch into pull request branch

    Arguments:

    * `input` (`UpdatePullRequestBranchInput!`): Parameters for
      UpdatePullRequestBranch
    """

    update_pull_request_review = sgqlc.types.Field(
        "UpdatePullRequestReviewPayload",
        graphql_name="updatePullRequestReview",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdatePullRequestReviewInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates the body of a pull request review.

    Arguments:

    * `input` (`UpdatePullRequestReviewInput!`): Parameters for
      UpdatePullRequestReview
    """

    update_pull_request_review_comment = sgqlc.types.Field(
        "UpdatePullRequestReviewCommentPayload",
        graphql_name="updatePullRequestReviewComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdatePullRequestReviewCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates a pull request review comment.

    Arguments:

    * `input` (`UpdatePullRequestReviewCommentInput!`): Parameters for
      UpdatePullRequestReviewComment
    """

    update_ref = sgqlc.types.Field(
        "UpdateRefPayload",
        graphql_name="updateRef",
        args=sgqlc.types.ArgDict((("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateRefInput), graphql_name="input", default=None)),)),
    )
    """Update a Git Ref.

    Arguments:

    * `input` (`UpdateRefInput!`): Parameters for UpdateRef
    """

    update_repository = sgqlc.types.Field(
        "UpdateRepositoryPayload",
        graphql_name="updateRepository",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateRepositoryInput), graphql_name="input", default=None)),)
        ),
    )
    """Update information about a repository.

    Arguments:

    * `input` (`UpdateRepositoryInput!`): Parameters for
      UpdateRepository
    """

    update_repository_ruleset = sgqlc.types.Field(
        "UpdateRepositoryRulesetPayload",
        graphql_name="updateRepositoryRuleset",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateRepositoryRulesetInput), graphql_name="input", default=None)),)
        ),
    )
    """Update a repository ruleset

    Arguments:

    * `input` (`UpdateRepositoryRulesetInput!`): Parameters for
      UpdateRepositoryRuleset
    """

    update_repository_web_commit_signoff_setting = sgqlc.types.Field(
        "UpdateRepositoryWebCommitSignoffSettingPayload",
        graphql_name="updateRepositoryWebCommitSignoffSetting",
        args=sgqlc.types.ArgDict(
            (
                (
                    "input",
                    sgqlc.types.Arg(sgqlc.types.non_null(UpdateRepositoryWebCommitSignoffSettingInput), graphql_name="input", default=None),
                ),
            )
        ),
    )
    """Sets whether contributors are required to sign off on web-based
    commits for a repository.

    Arguments:

    * `input` (`UpdateRepositoryWebCommitSignoffSettingInput!`):
      Parameters for UpdateRepositoryWebCommitSignoffSetting
    """

    update_sponsorship_preferences = sgqlc.types.Field(
        "UpdateSponsorshipPreferencesPayload",
        graphql_name="updateSponsorshipPreferences",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateSponsorshipPreferencesInput), graphql_name="input", default=None)),)
        ),
    )
    """Change visibility of your sponsorship and opt in or out of email
    updates from the maintainer.

    Arguments:

    * `input` (`UpdateSponsorshipPreferencesInput!`): Parameters for
      UpdateSponsorshipPreferences
    """

    update_subscription = sgqlc.types.Field(
        "UpdateSubscriptionPayload",
        graphql_name="updateSubscription",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateSubscriptionInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates the state for subscribable subjects.

    Arguments:

    * `input` (`UpdateSubscriptionInput!`): Parameters for
      UpdateSubscription
    """

    update_team_discussion = sgqlc.types.Field(
        "UpdateTeamDiscussionPayload",
        graphql_name="updateTeamDiscussion",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateTeamDiscussionInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates a team discussion.

    Arguments:

    * `input` (`UpdateTeamDiscussionInput!`): Parameters for
      UpdateTeamDiscussion
    """

    update_team_discussion_comment = sgqlc.types.Field(
        "UpdateTeamDiscussionCommentPayload",
        graphql_name="updateTeamDiscussionComment",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateTeamDiscussionCommentInput), graphql_name="input", default=None)),)
        ),
    )
    """Updates a discussion comment.

    Arguments:

    * `input` (`UpdateTeamDiscussionCommentInput!`): Parameters for
      UpdateTeamDiscussionComment
    """

    update_teams_repository = sgqlc.types.Field(
        "UpdateTeamsRepositoryPayload",
        graphql_name="updateTeamsRepository",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateTeamsRepositoryInput), graphql_name="input", default=None)),)
        ),
    )
    """Update team repository.

    Arguments:

    * `input` (`UpdateTeamsRepositoryInput!`): Parameters for
      UpdateTeamsRepository
    """

    update_topics = sgqlc.types.Field(
        "UpdateTopicsPayload",
        graphql_name="updateTopics",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(UpdateTopicsInput), graphql_name="input", default=None)),)
        ),
    )
    """Replaces the repository's topics with the given topics.

    Arguments:

    * `input` (`UpdateTopicsInput!`): Parameters for UpdateTopics
    """

    verify_verifiable_domain = sgqlc.types.Field(
        "VerifyVerifiableDomainPayload",
        graphql_name="verifyVerifiableDomain",
        args=sgqlc.types.ArgDict(
            (("input", sgqlc.types.Arg(sgqlc.types.non_null(VerifyVerifiableDomainInput), graphql_name="input", default=None)),)
        ),
    )
    """Verify that a verifiable domain has the expected DNS record.

    Arguments:

    * `input` (`VerifyVerifiableDomainInput!`): Parameters for
      VerifyVerifiableDomain
    """


class OrganizationAuditEntryConnection(sgqlc.types.relay.Connection):
    """The connection type for OrganizationAuditEntry."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("OrganizationAuditEntryEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("OrganizationAuditEntry"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class OrganizationAuditEntryEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("OrganizationAuditEntry", graphql_name="node")
    """The item at the end of the edge."""


class OrganizationConnection(sgqlc.types.relay.Connection):
    """A list of organizations managed by an enterprise."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("OrganizationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Organization"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class OrganizationEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Organization", graphql_name="node")
    """The item at the end of the edge."""


class OrganizationEnterpriseOwnerConnection(sgqlc.types.relay.Connection):
    """The connection type for User."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("OrganizationEnterpriseOwnerEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("User"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class OrganizationEnterpriseOwnerEdge(sgqlc.types.Type):
    """An enterprise owner in the context of an organization that is part
    of the enterprise.
    """

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "organization_role")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("User", graphql_name="node")
    """The item at the end of the edge."""

    organization_role = sgqlc.types.Field(sgqlc.types.non_null(RoleInOrganization), graphql_name="organizationRole")
    """The role of the owner with respect to the organization."""


class OrganizationInvitationConnection(sgqlc.types.relay.Connection):
    """The connection type for OrganizationInvitation."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("OrganizationInvitationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("OrganizationInvitation"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class OrganizationInvitationEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("OrganizationInvitation", graphql_name="node")
    """The item at the end of the edge."""


class OrganizationMemberConnection(sgqlc.types.relay.Connection):
    """The connection type for User."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("OrganizationMemberEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("User"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class OrganizationMemberEdge(sgqlc.types.Type):
    """Represents a user within an organization."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "has_two_factor_enabled", "node", "role")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    has_two_factor_enabled = sgqlc.types.Field(Boolean, graphql_name="hasTwoFactorEnabled")
    """Whether the organization member has two factor enabled or not.
    Returns null if information is not available to viewer.
    """

    node = sgqlc.types.Field("User", graphql_name="node")
    """The item at the end of the edge."""

    role = sgqlc.types.Field(OrganizationMemberRole, graphql_name="role")
    """The role this user has in the organization."""


class PackageConnection(sgqlc.types.relay.Connection):
    """The connection type for Package."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PackageEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Package"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PackageEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Package", graphql_name="node")
    """The item at the end of the edge."""


class PackageFileConnection(sgqlc.types.relay.Connection):
    """The connection type for PackageFile."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PackageFileEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PackageFile"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PackageFileEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PackageFile", graphql_name="node")
    """The item at the end of the edge."""


class PackageStatistics(sgqlc.types.Type):
    """Represents a object that contains package activity statistics such
    as downloads.
    """

    __schema__ = github_schema
    __field_names__ = ("downloads_total_count",)
    downloads_total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="downloadsTotalCount")
    """Number of times the package was downloaded since it was created."""


class PackageVersionConnection(sgqlc.types.relay.Connection):
    """The connection type for PackageVersion."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PackageVersionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PackageVersion"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null("PageInfo"), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PackageVersionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PackageVersion", graphql_name="node")
    """The item at the end of the edge."""


class PackageVersionStatistics(sgqlc.types.Type):
    """Represents a object that contains package version activity
    statistics such as downloads.
    """

    __schema__ = github_schema
    __field_names__ = ("downloads_total_count",)
    downloads_total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="downloadsTotalCount")
    """Number of times the package was downloaded since it was created."""


class PageInfo(sgqlc.types.Type):
    """Information about pagination in a connection."""

    __schema__ = github_schema
    __field_names__ = ("end_cursor", "has_next_page", "has_previous_page", "start_cursor")
    end_cursor = sgqlc.types.Field(String, graphql_name="endCursor")
    """When paginating forwards, the cursor to continue."""

    has_next_page = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasNextPage")
    """When paginating forwards, are there more items?"""

    has_previous_page = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasPreviousPage")
    """When paginating backwards, are there more items?"""

    start_cursor = sgqlc.types.Field(String, graphql_name="startCursor")
    """When paginating backwards, the cursor to continue."""


class PermissionSource(sgqlc.types.Type):
    """A level of permission and source for a user's access to a
    repository.
    """

    __schema__ = github_schema
    __field_names__ = ("organization", "permission", "source")
    organization = sgqlc.types.Field(sgqlc.types.non_null("Organization"), graphql_name="organization")
    """The organization the repository belongs to."""

    permission = sgqlc.types.Field(sgqlc.types.non_null(DefaultRepositoryPermissionField), graphql_name="permission")
    """The level of access this source has granted to the user."""

    source = sgqlc.types.Field(sgqlc.types.non_null("PermissionGranter"), graphql_name="source")
    """The source of this permission."""


class PinIssuePayload(sgqlc.types.Type):
    """Autogenerated return type of PinIssue"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "issue")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    issue = sgqlc.types.Field("Issue", graphql_name="issue")
    """The issue that was pinned"""


class PinnableItemConnection(sgqlc.types.relay.Connection):
    """The connection type for PinnableItem."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PinnableItemEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PinnableItem"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PinnableItemEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PinnableItem", graphql_name="node")
    """The item at the end of the edge."""


class PinnedDiscussionConnection(sgqlc.types.relay.Connection):
    """The connection type for PinnedDiscussion."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PinnedDiscussionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PinnedDiscussion"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PinnedDiscussionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PinnedDiscussion", graphql_name="node")
    """The item at the end of the edge."""


class PinnedIssueConnection(sgqlc.types.relay.Connection):
    """The connection type for PinnedIssue."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PinnedIssueEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PinnedIssue"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PinnedIssueEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PinnedIssue", graphql_name="node")
    """The item at the end of the edge."""


class ProfileItemShowcase(sgqlc.types.Type):
    """A curatable list of repositories relating to a repository owner,
    which defaults to showing the most popular repositories they own.
    """

    __schema__ = github_schema
    __field_names__ = ("has_pinned_items", "items")
    has_pinned_items = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="hasPinnedItems")
    """Whether or not the owner has pinned any repositories or gists."""

    items = sgqlc.types.Field(
        sgqlc.types.non_null(PinnableItemConnection),
        graphql_name="items",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """The repositories and gists in the showcase. If the profile owner
    has any pinned items, those will be returned. Otherwise, the
    profile owner's popular repositories will be returned.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class ProjectCardConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectCard."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectCardEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectCard"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectCardEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectCard", graphql_name="node")
    """The item at the end of the edge."""


class ProjectColumnConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectColumn."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectColumnEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectColumn"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectColumnEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectColumn", graphql_name="node")
    """The item at the end of the edge."""


class ProjectConnection(sgqlc.types.relay.Connection):
    """A list of projects associated with the owner."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Project"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Project", graphql_name="node")
    """The item at the end of the edge."""


class ProjectProgress(sgqlc.types.Type):
    """Project progress stats."""

    __schema__ = github_schema
    __field_names__ = (
        "done_count",
        "done_percentage",
        "enabled",
        "in_progress_count",
        "in_progress_percentage",
        "todo_count",
        "todo_percentage",
    )
    done_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="doneCount")
    """The number of done cards."""

    done_percentage = sgqlc.types.Field(sgqlc.types.non_null(Float), graphql_name="donePercentage")
    """The percentage of done cards."""

    enabled = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="enabled")
    """Whether progress tracking is enabled and cards with purpose exist
    for this project
    """

    in_progress_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="inProgressCount")
    """The number of in-progress cards."""

    in_progress_percentage = sgqlc.types.Field(sgqlc.types.non_null(Float), graphql_name="inProgressPercentage")
    """The percentage of in-progress cards."""

    todo_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="todoCount")
    """The number of to do cards."""

    todo_percentage = sgqlc.types.Field(sgqlc.types.non_null(Float), graphql_name="todoPercentage")
    """The percentage of to do cards."""


class ProjectV2ActorConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2Actor."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2ActorEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2Actor"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2ActorEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectV2Actor", graphql_name="node")
    """The item at the end of the edge."""


class ProjectV2Connection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2Edge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2Edge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectV2", graphql_name="node")
    """The item at the end of the edge."""


class ProjectV2FieldConfigurationConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2FieldConfiguration."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2FieldConfigurationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2FieldConfiguration"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2FieldConfigurationEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectV2FieldConfiguration", graphql_name="node")
    """The item at the end of the edge."""


class ProjectV2FieldConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2Field."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2FieldEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2Field"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2FieldEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectV2Field", graphql_name="node")
    """The item at the end of the edge."""


class ProjectV2ItemConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2Item."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2ItemEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2Item"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2ItemEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectV2Item", graphql_name="node")
    """The item at the end of the edge."""


class ProjectV2ItemFieldLabelValue(sgqlc.types.Type):
    """The value of the labels field in a Project item."""

    __schema__ = github_schema
    __field_names__ = ("field", "labels")
    field = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2FieldConfiguration"), graphql_name="field")
    """The field that contains this value."""

    labels = sgqlc.types.Field(
        LabelConnection,
        graphql_name="labels",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """Labels value of a field

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class ProjectV2ItemFieldMilestoneValue(sgqlc.types.Type):
    """The value of a milestone field in a Project item."""

    __schema__ = github_schema
    __field_names__ = ("field", "milestone")
    field = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2FieldConfiguration"), graphql_name="field")
    """The field that contains this value."""

    milestone = sgqlc.types.Field("Milestone", graphql_name="milestone")
    """Milestone value of a field"""


class ProjectV2ItemFieldPullRequestValue(sgqlc.types.Type):
    """The value of a pull request field in a Project item."""

    __schema__ = github_schema
    __field_names__ = ("field", "pull_requests")
    field = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2FieldConfiguration"), graphql_name="field")
    """The field that contains this value."""

    pull_requests = sgqlc.types.Field(
        "PullRequestConnection",
        graphql_name="pullRequests",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                (
                    "order_by",
                    sgqlc.types.Arg(PullRequestOrder, graphql_name="orderBy", default={"field": "CREATED_AT", "direction": "ASC"}),
                ),
            )
        ),
    )
    """The pull requests for this field

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`PullRequestOrder`): Ordering options for pull
      requests. (default: `{field: CREATED_AT, direction: ASC}`)
    """


class ProjectV2ItemFieldRepositoryValue(sgqlc.types.Type):
    """The value of a repository field in a Project item."""

    __schema__ = github_schema
    __field_names__ = ("field", "repository")
    field = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2FieldConfiguration"), graphql_name="field")
    """The field that contains this value."""

    repository = sgqlc.types.Field("Repository", graphql_name="repository")
    """The repository for this field."""


class ProjectV2ItemFieldReviewerValue(sgqlc.types.Type):
    """The value of a reviewers field in a Project item."""

    __schema__ = github_schema
    __field_names__ = ("field", "reviewers")
    field = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2FieldConfiguration"), graphql_name="field")
    """The field that contains this value."""

    reviewers = sgqlc.types.Field(
        "RequestedReviewerConnection",
        graphql_name="reviewers",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """The reviewers for this field.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class ProjectV2ItemFieldUserValue(sgqlc.types.Type):
    """The value of a user field in a Project item."""

    __schema__ = github_schema
    __field_names__ = ("field", "users")
    field = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2FieldConfiguration"), graphql_name="field")
    """The field that contains this value."""

    users = sgqlc.types.Field(
        "UserConnection",
        graphql_name="users",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """The users for this field

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class ProjectV2ItemFieldValueConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2ItemFieldValue."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2ItemFieldValueEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2ItemFieldValue"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2ItemFieldValueEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectV2ItemFieldValue", graphql_name="node")
    """The item at the end of the edge."""


class ProjectV2IterationFieldConfiguration(sgqlc.types.Type):
    """Iteration field configuration for a project."""

    __schema__ = github_schema
    __field_names__ = ("completed_iterations", "duration", "iterations", "start_day")
    completed_iterations = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("ProjectV2IterationFieldIteration"))),
        graphql_name="completedIterations",
    )
    """The iteration's completed iterations"""

    duration = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="duration")
    """The iteration's duration in days"""

    iterations = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("ProjectV2IterationFieldIteration"))), graphql_name="iterations"
    )
    """The iteration's iterations"""

    start_day = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="startDay")
    """The iteration's start day of the week"""


class ProjectV2IterationFieldIteration(sgqlc.types.Type):
    """Iteration field iteration settings for a project."""

    __schema__ = github_schema
    __field_names__ = ("duration", "id", "start_date", "title", "title_html")
    duration = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="duration")
    """The iteration's duration in days"""

    id = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="id")
    """The iteration's ID."""

    start_date = sgqlc.types.Field(sgqlc.types.non_null(Date), graphql_name="startDate")
    """The iteration's start date"""

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """The iteration's title."""

    title_html = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="titleHTML")
    """The iteration's html title."""


class ProjectV2SingleSelectFieldOption(sgqlc.types.Type):
    """Single select field option for a configuration for a project."""

    __schema__ = github_schema
    __field_names__ = ("id", "name", "name_html")
    id = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="id")
    """The option's ID."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The option's name."""

    name_html = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="nameHTML")
    """The option's html name."""


class ProjectV2SortBy(sgqlc.types.Type):
    """Represents a sort by field and direction."""

    __schema__ = github_schema
    __field_names__ = ("direction", "field")
    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction of the sorting. Possible values are ASC and DESC."""

    field = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2Field"), graphql_name="field")
    """The field by which items are sorted."""


class ProjectV2SortByConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2SortBy."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2SortByEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(ProjectV2SortBy), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2SortByEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(ProjectV2SortBy, graphql_name="node")
    """The item at the end of the edge."""


class ProjectV2SortByField(sgqlc.types.Type):
    """Represents a sort by field and direction."""

    __schema__ = github_schema
    __field_names__ = ("direction", "field")
    direction = sgqlc.types.Field(sgqlc.types.non_null(OrderDirection), graphql_name="direction")
    """The direction of the sorting. Possible values are ASC and DESC."""

    field = sgqlc.types.Field(sgqlc.types.non_null("ProjectV2FieldConfiguration"), graphql_name="field")
    """The field by which items are sorted."""


class ProjectV2SortByFieldConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2SortByField."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2SortByFieldEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(ProjectV2SortByField), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2SortByFieldEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(ProjectV2SortByField, graphql_name="node")
    """The item at the end of the edge."""


class ProjectV2ViewConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2View."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2ViewEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2View"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2ViewEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectV2View", graphql_name="node")
    """The item at the end of the edge."""


class ProjectV2WorkflowConnection(sgqlc.types.relay.Connection):
    """The connection type for ProjectV2Workflow."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2WorkflowEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ProjectV2Workflow"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ProjectV2WorkflowEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ProjectV2Workflow", graphql_name="node")
    """The item at the end of the edge."""


class PublicKeyConnection(sgqlc.types.relay.Connection):
    """The connection type for PublicKey."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PublicKeyEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PublicKey"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PublicKeyEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PublicKey", graphql_name="node")
    """The item at the end of the edge."""


class PublishSponsorsTierPayload(sgqlc.types.Type):
    """Autogenerated return type of PublishSponsorsTier"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "sponsors_tier")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    sponsors_tier = sgqlc.types.Field("SponsorsTier", graphql_name="sponsorsTier")
    """The tier that was published."""


class PullRequestChangedFile(sgqlc.types.Type):
    """A file changed in a pull request."""

    __schema__ = github_schema
    __field_names__ = ("additions", "change_type", "deletions", "path", "viewer_viewed_state")
    additions = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="additions")
    """The number of additions to the file."""

    change_type = sgqlc.types.Field(sgqlc.types.non_null(PatchStatus), graphql_name="changeType")
    """How the file was changed in this PullRequest"""

    deletions = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="deletions")
    """The number of deletions to the file."""

    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """The path of the file."""

    viewer_viewed_state = sgqlc.types.Field(sgqlc.types.non_null(FileViewedState), graphql_name="viewerViewedState")
    """The state of the file for the viewer."""


class PullRequestChangedFileConnection(sgqlc.types.relay.Connection):
    """The connection type for PullRequestChangedFile."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PullRequestChangedFileEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(PullRequestChangedFile), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PullRequestChangedFileEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(PullRequestChangedFile, graphql_name="node")
    """The item at the end of the edge."""


class PullRequestCommitConnection(sgqlc.types.relay.Connection):
    """The connection type for PullRequestCommit."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PullRequestCommitEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PullRequestCommit"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PullRequestCommitEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PullRequestCommit", graphql_name="node")
    """The item at the end of the edge."""


class PullRequestConnection(sgqlc.types.relay.Connection):
    """The connection type for PullRequest."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PullRequestEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PullRequest"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PullRequestContributionsByRepository(sgqlc.types.Type):
    """This aggregates pull requests opened by a user within one
    repository.
    """

    __schema__ = github_schema
    __field_names__ = ("contributions", "repository")
    contributions = sgqlc.types.Field(
        sgqlc.types.non_null(CreatedPullRequestContributionConnection),
        graphql_name="contributions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("order_by", sgqlc.types.Arg(ContributionOrder, graphql_name="orderBy", default={"direction": "DESC"})),
            )
        ),
    )
    """The pull request contributions.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`ContributionOrder`): Ordering options for
      contributions returned from the connection. (default:
      `{direction: DESC}`)
    """

    repository = sgqlc.types.Field(sgqlc.types.non_null("Repository"), graphql_name="repository")
    """The repository in which the pull requests were opened."""


class PullRequestEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PullRequest", graphql_name="node")
    """The item at the end of the edge."""


class PullRequestParameters(sgqlc.types.Type):
    """Require all commits be made to a non-target branch and submitted
    via a pull request before they can be merged.
    """

    __schema__ = github_schema
    __field_names__ = (
        "dismiss_stale_reviews_on_push",
        "require_code_owner_review",
        "require_last_push_approval",
        "required_approving_review_count",
        "required_review_thread_resolution",
    )
    dismiss_stale_reviews_on_push = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="dismissStaleReviewsOnPush")
    """New, reviewable commits pushed will dismiss previous pull request
    review approvals.
    """

    require_code_owner_review = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requireCodeOwnerReview")
    """Require an approving review in pull requests that modify files
    that have a designated code owner.
    """

    require_last_push_approval = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requireLastPushApproval")
    """Whether the most recent reviewable push must be approved by
    someone other than the person who pushed it.
    """

    required_approving_review_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="requiredApprovingReviewCount")
    """The number of approving reviews that are required before a pull
    request can be merged.
    """

    required_review_thread_resolution = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requiredReviewThreadResolution")
    """All conversations on code must be resolved before a pull request
    can be merged.
    """


class PullRequestReviewCommentConnection(sgqlc.types.relay.Connection):
    """The connection type for PullRequestReviewComment."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PullRequestReviewCommentEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PullRequestReviewComment"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PullRequestReviewCommentEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PullRequestReviewComment", graphql_name="node")
    """The item at the end of the edge."""


class PullRequestReviewConnection(sgqlc.types.relay.Connection):
    """The connection type for PullRequestReview."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PullRequestReviewEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PullRequestReview"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PullRequestReviewContributionsByRepository(sgqlc.types.Type):
    """This aggregates pull request reviews made by a user within one
    repository.
    """

    __schema__ = github_schema
    __field_names__ = ("contributions", "repository")
    contributions = sgqlc.types.Field(
        sgqlc.types.non_null(CreatedPullRequestReviewContributionConnection),
        graphql_name="contributions",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("order_by", sgqlc.types.Arg(ContributionOrder, graphql_name="orderBy", default={"direction": "DESC"})),
            )
        ),
    )
    """The pull request review contributions.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`ContributionOrder`): Ordering options for
      contributions returned from the connection. (default:
      `{direction: DESC}`)
    """

    repository = sgqlc.types.Field(sgqlc.types.non_null("Repository"), graphql_name="repository")
    """The repository in which the pull request reviews were made."""


class PullRequestReviewEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PullRequestReview", graphql_name="node")
    """The item at the end of the edge."""


class PullRequestReviewThreadConnection(sgqlc.types.relay.Connection):
    """Review comment threads for a pull request review."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PullRequestReviewThreadEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PullRequestReviewThread"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PullRequestReviewThreadEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PullRequestReviewThread", graphql_name="node")
    """The item at the end of the edge."""


class PullRequestRevisionMarker(sgqlc.types.Type):
    """Represents the latest point in the pull request timeline for which
    the viewer has seen the pull request's commits.
    """

    __schema__ = github_schema
    __field_names__ = ("created_at", "last_seen_commit", "pull_request")
    created_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="createdAt")
    """Identifies the date and time when the object was created."""

    last_seen_commit = sgqlc.types.Field(sgqlc.types.non_null("Commit"), graphql_name="lastSeenCommit")
    """The last commit the viewer has seen."""

    pull_request = sgqlc.types.Field(sgqlc.types.non_null("PullRequest"), graphql_name="pullRequest")
    """The pull request to which the marker belongs."""


class PullRequestTemplate(sgqlc.types.Type):
    """A repository pull request template."""

    __schema__ = github_schema
    __field_names__ = ("body", "filename", "repository")
    body = sgqlc.types.Field(String, graphql_name="body")
    """The body of the template"""

    filename = sgqlc.types.Field(String, graphql_name="filename")
    """The filename of the template"""

    repository = sgqlc.types.Field(sgqlc.types.non_null("Repository"), graphql_name="repository")
    """The repository the template belongs to"""


class PullRequestTimelineConnection(sgqlc.types.relay.Connection):
    """The connection type for PullRequestTimelineItem."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PullRequestTimelineItemEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PullRequestTimelineItem"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PullRequestTimelineItemEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PullRequestTimelineItem", graphql_name="node")
    """The item at the end of the edge."""


class PullRequestTimelineItemsConnection(sgqlc.types.relay.Connection):
    """The connection type for PullRequestTimelineItems."""

    __schema__ = github_schema
    __field_names__ = ("edges", "filtered_count", "nodes", "page_count", "page_info", "total_count", "updated_at")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PullRequestTimelineItemsEdge"), graphql_name="edges")
    """A list of edges."""

    filtered_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="filteredCount")
    """Identifies the count of items after applying `before` and `after`
    filters.
    """

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PullRequestTimelineItems"), graphql_name="nodes")
    """A list of nodes."""

    page_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="pageCount")
    """Identifies the count of items after applying `before`/`after`
    filters and `first`/`last`/`skip` slicing.
    """

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""

    updated_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="updatedAt")
    """Identifies the date and time when the timeline was last updated."""


class PullRequestTimelineItemsEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PullRequestTimelineItems", graphql_name="node")
    """The item at the end of the edge."""


class PushAllowanceConnection(sgqlc.types.relay.Connection):
    """The connection type for PushAllowance."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("PushAllowanceEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("PushAllowance"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class PushAllowanceEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("PushAllowance", graphql_name="node")
    """The item at the end of the edge."""


class Query(sgqlc.types.Type):
    """The query root of GitHub's GraphQL interface."""

    __schema__ = github_schema
    __field_names__ = (
        "code_of_conduct",
        "codes_of_conduct",
        "enterprise",
        "enterprise_administrator_invitation",
        "enterprise_administrator_invitation_by_token",
        "license",
        "licenses",
        "marketplace_categories",
        "marketplace_category",
        "marketplace_listing",
        "marketplace_listings",
        "meta",
        "node",
        "nodes",
        "organization",
        "rate_limit",
        "relay",
        "repository",
        "repository_owner",
        "resource",
        "search",
        "security_advisories",
        "security_advisory",
        "security_vulnerabilities",
        "sponsorables",
        "topic",
        "user",
        "viewer",
    )
    code_of_conduct = sgqlc.types.Field(
        "CodeOfConduct",
        graphql_name="codeOfConduct",
        args=sgqlc.types.ArgDict((("key", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="key", default=None)),)),
    )
    """Look up a code of conduct by its key

    Arguments:

    * `key` (`String!`): The code of conduct's key
    """

    codes_of_conduct = sgqlc.types.Field(sgqlc.types.list_of("CodeOfConduct"), graphql_name="codesOfConduct")
    """Look up a code of conduct by its key"""

    enterprise = sgqlc.types.Field(
        "Enterprise",
        graphql_name="enterprise",
        args=sgqlc.types.ArgDict(
            (
                ("slug", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="slug", default=None)),
                ("invitation_token", sgqlc.types.Arg(String, graphql_name="invitationToken", default=None)),
            )
        ),
    )
    """Look up an enterprise by URL slug.

    Arguments:

    * `slug` (`String!`): The enterprise URL slug.
    * `invitation_token` (`String`): The enterprise invitation token.
    """

    enterprise_administrator_invitation = sgqlc.types.Field(
        "EnterpriseAdministratorInvitation",
        graphql_name="enterpriseAdministratorInvitation",
        args=sgqlc.types.ArgDict(
            (
                ("user_login", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="userLogin", default=None)),
                ("enterprise_slug", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="enterpriseSlug", default=None)),
                ("role", sgqlc.types.Arg(sgqlc.types.non_null(EnterpriseAdministratorRole), graphql_name="role", default=None)),
            )
        ),
    )
    """Look up a pending enterprise administrator invitation by invitee,
    enterprise and role.

    Arguments:

    * `user_login` (`String!`): The login of the user invited to join
      the business.
    * `enterprise_slug` (`String!`): The slug of the enterprise the
      user was invited to join.
    * `role` (`EnterpriseAdministratorRole!`): The role for the
      business member invitation.
    """

    enterprise_administrator_invitation_by_token = sgqlc.types.Field(
        "EnterpriseAdministratorInvitation",
        graphql_name="enterpriseAdministratorInvitationByToken",
        args=sgqlc.types.ArgDict(
            (("invitation_token", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="invitationToken", default=None)),)
        ),
    )
    """Look up a pending enterprise administrator invitation by
    invitation token.

    Arguments:

    * `invitation_token` (`String!`): The invitation token sent with
      the invitation email.
    """

    license = sgqlc.types.Field(
        "License",
        graphql_name="license",
        args=sgqlc.types.ArgDict((("key", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="key", default=None)),)),
    )
    """Look up an open source license by its key

    Arguments:

    * `key` (`String!`): The license's downcased SPDX ID
    """

    licenses = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of("License")), graphql_name="licenses")
    """Return a list of known open source licenses"""

    marketplace_categories = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("MarketplaceCategory"))),
        graphql_name="marketplaceCategories",
        args=sgqlc.types.ArgDict(
            (
                (
                    "include_categories",
                    sgqlc.types.Arg(sgqlc.types.list_of(sgqlc.types.non_null(String)), graphql_name="includeCategories", default=None),
                ),
                ("exclude_empty", sgqlc.types.Arg(Boolean, graphql_name="excludeEmpty", default=None)),
                ("exclude_subcategories", sgqlc.types.Arg(Boolean, graphql_name="excludeSubcategories", default=None)),
            )
        ),
    )
    """Get alphabetically sorted list of Marketplace categories

    Arguments:

    * `include_categories` (`[String!]`): Return only the specified
      categories.
    * `exclude_empty` (`Boolean`): Exclude categories with no
      listings.
    * `exclude_subcategories` (`Boolean`): Returns top level
      categories only, excluding any subcategories.
    """

    marketplace_category = sgqlc.types.Field(
        "MarketplaceCategory",
        graphql_name="marketplaceCategory",
        args=sgqlc.types.ArgDict(
            (
                ("slug", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="slug", default=None)),
                ("use_topic_aliases", sgqlc.types.Arg(Boolean, graphql_name="useTopicAliases", default=None)),
            )
        ),
    )
    """Look up a Marketplace category by its slug.

    Arguments:

    * `slug` (`String!`): The URL slug of the category.
    * `use_topic_aliases` (`Boolean`): Also check topic aliases for
      the category slug
    """

    marketplace_listing = sgqlc.types.Field(
        "MarketplaceListing",
        graphql_name="marketplaceListing",
        args=sgqlc.types.ArgDict((("slug", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="slug", default=None)),)),
    )
    """Look up a single Marketplace listing

    Arguments:

    * `slug` (`String!`): Select the listing that matches this slug.
      It's the short name of the listing used in its URL.
    """

    marketplace_listings = sgqlc.types.Field(
        sgqlc.types.non_null(MarketplaceListingConnection),
        graphql_name="marketplaceListings",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("category_slug", sgqlc.types.Arg(String, graphql_name="categorySlug", default=None)),
                ("use_topic_aliases", sgqlc.types.Arg(Boolean, graphql_name="useTopicAliases", default=None)),
                ("viewer_can_admin", sgqlc.types.Arg(Boolean, graphql_name="viewerCanAdmin", default=None)),
                ("admin_id", sgqlc.types.Arg(ID, graphql_name="adminId", default=None)),
                ("organization_id", sgqlc.types.Arg(ID, graphql_name="organizationId", default=None)),
                ("all_states", sgqlc.types.Arg(Boolean, graphql_name="allStates", default=None)),
                ("slugs", sgqlc.types.Arg(sgqlc.types.list_of(String), graphql_name="slugs", default=None)),
                ("primary_category_only", sgqlc.types.Arg(Boolean, graphql_name="primaryCategoryOnly", default=False)),
                ("with_free_trials_only", sgqlc.types.Arg(Boolean, graphql_name="withFreeTrialsOnly", default=False)),
            )
        ),
    )
    """Look up Marketplace listings

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `category_slug` (`String`): Select only listings with the given
      category.
    * `use_topic_aliases` (`Boolean`): Also check topic aliases for
      the category slug
    * `viewer_can_admin` (`Boolean`): Select listings to which user
      has admin access. If omitted, listings visible to the viewer are
      returned.
    * `admin_id` (`ID`): Select listings that can be administered by
      the specified user.
    * `organization_id` (`ID`): Select listings for products owned by
      the specified organization.
    * `all_states` (`Boolean`): Select listings visible to the viewer
      even if they are not approved. If omitted or false, only
      approved listings will be returned.
    * `slugs` (`[String]`): Select the listings with these slugs, if
      they are visible to the viewer.
    * `primary_category_only` (`Boolean`): Select only listings where
      the primary category matches the given category slug. (default:
      `false`)
    * `with_free_trials_only` (`Boolean`): Select only listings that
      offer a free trial. (default: `false`)
    """

    meta = sgqlc.types.Field(sgqlc.types.non_null(GitHubMetadata), graphql_name="meta")
    """Return information about the GitHub instance"""

    node = sgqlc.types.Field(
        Node,
        graphql_name="node",
        args=sgqlc.types.ArgDict((("id", sgqlc.types.Arg(sgqlc.types.non_null(ID), graphql_name="id", default=None)),)),
    )
    """Fetches an object given its ID.

    Arguments:

    * `id` (`ID!`): ID of the object.
    """

    nodes = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(Node)),
        graphql_name="nodes",
        args=sgqlc.types.ArgDict(
            (
                (
                    "ids",
                    sgqlc.types.Arg(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(ID))), graphql_name="ids", default=None),
                ),
            )
        ),
    )
    """Lookup nodes by a list of IDs.

    Arguments:

    * `ids` (`[ID!]!`): The list of node IDs.
    """

    organization = sgqlc.types.Field(
        "Organization",
        graphql_name="organization",
        args=sgqlc.types.ArgDict((("login", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="login", default=None)),)),
    )
    """Lookup a organization by login.

    Arguments:

    * `login` (`String!`): The organization's login.
    """

    rate_limit = sgqlc.types.Field(
        "RateLimit",
        graphql_name="rateLimit",
        args=sgqlc.types.ArgDict((("dry_run", sgqlc.types.Arg(Boolean, graphql_name="dryRun", default=False)),)),
    )
    """The client's rate limit information.

    Arguments:

    * `dry_run` (`Boolean`): If true, calculate the cost for the query
      without evaluating it (default: `false`)
    """

    relay = sgqlc.types.Field(sgqlc.types.non_null("Query"), graphql_name="relay")
    """Workaround for re-exposing the root query object. (Refer to
    https://github.com/facebook/relay/issues/112 for more
    information.)
    """

    repository = sgqlc.types.Field(
        "Repository",
        graphql_name="repository",
        args=sgqlc.types.ArgDict(
            (
                ("owner", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="owner", default=None)),
                ("name", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="name", default=None)),
                ("follow_renames", sgqlc.types.Arg(Boolean, graphql_name="followRenames", default=True)),
            )
        ),
    )
    """Lookup a given repository by the owner and repository name.

    Arguments:

    * `owner` (`String!`): The login field of a user or organization
    * `name` (`String!`): The name of the repository
    * `follow_renames` (`Boolean`): Follow repository renames. If
      disabled, a repository referenced by its old name will return an
      error. (default: `true`)
    """

    repository_owner = sgqlc.types.Field(
        RepositoryOwner,
        graphql_name="repositoryOwner",
        args=sgqlc.types.ArgDict((("login", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="login", default=None)),)),
    )
    """Lookup a repository owner (ie. either a User or an Organization)
    by login.

    Arguments:

    * `login` (`String!`): The username to lookup the owner by.
    """

    resource = sgqlc.types.Field(
        UniformResourceLocatable,
        graphql_name="resource",
        args=sgqlc.types.ArgDict((("url", sgqlc.types.Arg(sgqlc.types.non_null(URI), graphql_name="url", default=None)),)),
    )
    """Lookup resource by a URL.

    Arguments:

    * `url` (`URI!`): The URL.
    """

    search = sgqlc.types.Field(
        sgqlc.types.non_null("SearchResultItemConnection"),
        graphql_name="search",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("query", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="query", default=None)),
                ("type", sgqlc.types.Arg(sgqlc.types.non_null(SearchType), graphql_name="type", default=None)),
            )
        ),
    )
    """Perform a search across resources, returning a maximum of 1,000
    results.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `query` (`String!`): The search string to look for.
    * `type` (`SearchType!`): The types of search items to search
      within.
    """

    security_advisories = sgqlc.types.Field(
        sgqlc.types.non_null("SecurityAdvisoryConnection"),
        graphql_name="securityAdvisories",
        args=sgqlc.types.ArgDict(
            (
                (
                    "order_by",
                    sgqlc.types.Arg(SecurityAdvisoryOrder, graphql_name="orderBy", default={"field": "UPDATED_AT", "direction": "DESC"}),
                ),
                ("identifier", sgqlc.types.Arg(SecurityAdvisoryIdentifierFilter, graphql_name="identifier", default=None)),
                ("published_since", sgqlc.types.Arg(DateTime, graphql_name="publishedSince", default=None)),
                ("updated_since", sgqlc.types.Arg(DateTime, graphql_name="updatedSince", default=None)),
                (
                    "classifications",
                    sgqlc.types.Arg(
                        sgqlc.types.list_of(sgqlc.types.non_null(SecurityAdvisoryClassification)),
                        graphql_name="classifications",
                        default=None,
                    ),
                ),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """GitHub Security Advisories

    Arguments:

    * `order_by` (`SecurityAdvisoryOrder`): Ordering options for the
      returned topics. (default: `{field: UPDATED_AT, direction:
      DESC}`)
    * `identifier` (`SecurityAdvisoryIdentifierFilter`): Filter
      advisories by identifier, e.g. GHSA or CVE.
    * `published_since` (`DateTime`): Filter advisories to those
      published since a time in the past.
    * `updated_since` (`DateTime`): Filter advisories to those updated
      since a time in the past.
    * `classifications` (`[SecurityAdvisoryClassification!]`): A list
      of classifications to filter advisories by.
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    security_advisory = sgqlc.types.Field(
        "SecurityAdvisory",
        graphql_name="securityAdvisory",
        args=sgqlc.types.ArgDict((("ghsa_id", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="ghsaId", default=None)),)),
    )
    """Fetch a Security Advisory by its GHSA ID

    Arguments:

    * `ghsa_id` (`String!`): GitHub Security Advisory ID.
    """

    security_vulnerabilities = sgqlc.types.Field(
        sgqlc.types.non_null("SecurityVulnerabilityConnection"),
        graphql_name="securityVulnerabilities",
        args=sgqlc.types.ArgDict(
            (
                (
                    "order_by",
                    sgqlc.types.Arg(
                        SecurityVulnerabilityOrder, graphql_name="orderBy", default={"field": "UPDATED_AT", "direction": "DESC"}
                    ),
                ),
                ("ecosystem", sgqlc.types.Arg(SecurityAdvisoryEcosystem, graphql_name="ecosystem", default=None)),
                ("package", sgqlc.types.Arg(String, graphql_name="package", default=None)),
                (
                    "severities",
                    sgqlc.types.Arg(
                        sgqlc.types.list_of(sgqlc.types.non_null(SecurityAdvisorySeverity)), graphql_name="severities", default=None
                    ),
                ),
                (
                    "classifications",
                    sgqlc.types.Arg(
                        sgqlc.types.list_of(sgqlc.types.non_null(SecurityAdvisoryClassification)),
                        graphql_name="classifications",
                        default=None,
                    ),
                ),
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """Software Vulnerabilities documented by GitHub Security Advisories

    Arguments:

    * `order_by` (`SecurityVulnerabilityOrder`): Ordering options for
      the returned topics. (default: `{field: UPDATED_AT, direction:
      DESC}`)
    * `ecosystem` (`SecurityAdvisoryEcosystem`): An ecosystem to
      filter vulnerabilities by.
    * `package` (`String`): A package name to filter vulnerabilities
      by.
    * `severities` (`[SecurityAdvisorySeverity!]`): A list of
      severities to filter vulnerabilities by.
    * `classifications` (`[SecurityAdvisoryClassification!]`): A list
      of advisory classifications to filter vulnerabilities by.
    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    sponsorables = sgqlc.types.Field(
        sgqlc.types.non_null("SponsorableItemConnection"),
        graphql_name="sponsorables",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("order_by", sgqlc.types.Arg(SponsorableOrder, graphql_name="orderBy", default={"field": "LOGIN", "direction": "ASC"})),
                ("only_dependencies", sgqlc.types.Arg(Boolean, graphql_name="onlyDependencies", default=False)),
                ("org_login_for_dependencies", sgqlc.types.Arg(String, graphql_name="orgLoginForDependencies", default=None)),
                ("dependency_ecosystem", sgqlc.types.Arg(SecurityAdvisoryEcosystem, graphql_name="dependencyEcosystem", default=None)),
                ("ecosystem", sgqlc.types.Arg(DependencyGraphEcosystem, graphql_name="ecosystem", default=None)),
            )
        ),
    )
    """Users and organizations who can be sponsored via GitHub Sponsors.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `order_by` (`SponsorableOrder`): Ordering options for users and
      organizations returned from the connection. (default: `{field:
      LOGIN, direction: ASC}`)
    * `only_dependencies` (`Boolean`): Whether only sponsorables who
      own the viewer's dependencies will be returned. Must be
      authenticated to use. Can check an organization instead for
      their dependencies owned by sponsorables by passing
      orgLoginForDependencies. (default: `false`)
    * `org_login_for_dependencies` (`String`): Optional organization
      username for whose dependencies should be checked. Used when
      onlyDependencies = true. Omit to check your own dependencies. If
      you are not an administrator of the organization, only
      dependencies from its public repositories will be considered.
    * `dependency_ecosystem` (`SecurityAdvisoryEcosystem`): Optional
      filter for which dependencies should be checked for sponsorable
      owners. Only sponsorable owners of dependencies in this
      ecosystem will be included. Used when onlyDependencies = true.
      **Upcoming Change on 2022-07-01 UTC** **Description:**
      `dependencyEcosystem` will be removed. Use the ecosystem
      argument instead. **Reason:** The type is switching from
      SecurityAdvisoryEcosystem to DependencyGraphEcosystem.
    * `ecosystem` (`DependencyGraphEcosystem`): Optional filter for
      which dependencies should be checked for sponsorable owners.
      Only sponsorable owners of dependencies in this ecosystem will
      be included. Used when onlyDependencies = true.
    """

    topic = sgqlc.types.Field(
        "Topic",
        graphql_name="topic",
        args=sgqlc.types.ArgDict((("name", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="name", default=None)),)),
    )
    """Look up a topic by name.

    Arguments:

    * `name` (`String!`): The topic's name.
    """

    user = sgqlc.types.Field(
        "User",
        graphql_name="user",
        args=sgqlc.types.ArgDict((("login", sgqlc.types.Arg(sgqlc.types.non_null(String), graphql_name="login", default=None)),)),
    )
    """Lookup a user by login.

    Arguments:

    * `login` (`String!`): The user's login.
    """

    viewer = sgqlc.types.Field(sgqlc.types.non_null("User"), graphql_name="viewer")
    """The currently authenticated user."""


class RateLimit(sgqlc.types.Type):
    """Represents the client's rate limit."""

    __schema__ = github_schema
    __field_names__ = ("cost", "limit", "node_count", "remaining", "reset_at", "used")
    cost = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="cost")
    """The point cost for the current query counting against the rate
    limit.
    """

    limit = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="limit")
    """The maximum number of points the client is permitted to consume in
    a 60 minute window.
    """

    node_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="nodeCount")
    """The maximum number of nodes this query may return"""

    remaining = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="remaining")
    """The number of points remaining in the current rate limit window."""

    reset_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="resetAt")
    """The time at which the current rate limit window resets in UTC
    epoch seconds.
    """

    used = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="used")
    """The number of points used in the current rate limit window."""


class ReactingUserConnection(sgqlc.types.relay.Connection):
    """The connection type for User."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ReactingUserEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("User"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ReactingUserEdge(sgqlc.types.Type):
    """Represents a user that's made a reaction."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "reacted_at")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(sgqlc.types.non_null("User"), graphql_name="node")

    reacted_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="reactedAt")
    """The moment when the user made the reaction."""


class ReactionConnection(sgqlc.types.relay.Connection):
    """A list of reactions that have been left on the subject."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count", "viewer_has_reacted")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ReactionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Reaction"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""

    viewer_has_reacted = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerHasReacted")
    """Whether or not the authenticated user has left a reaction on the
    subject.
    """


class ReactionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Reaction", graphql_name="node")
    """The item at the end of the edge."""


class ReactionGroup(sgqlc.types.Type):
    """A group of emoji reactions to a particular piece of content."""

    __schema__ = github_schema
    __field_names__ = ("content", "created_at", "reactors", "subject", "viewer_has_reacted")
    content = sgqlc.types.Field(sgqlc.types.non_null(ReactionContent), graphql_name="content")
    """Identifies the emoji reaction."""

    created_at = sgqlc.types.Field(DateTime, graphql_name="createdAt")
    """Identifies when the reaction was created."""

    reactors = sgqlc.types.Field(
        sgqlc.types.non_null("ReactorConnection"),
        graphql_name="reactors",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """Reactors to the reaction subject with the emotion represented by
    this reaction group.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """

    subject = sgqlc.types.Field(sgqlc.types.non_null(Reactable), graphql_name="subject")
    """The subject that was reacted to."""

    viewer_has_reacted = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerHasReacted")
    """Whether or not the authenticated user has left a reaction on the
    subject.
    """


class ReactorConnection(sgqlc.types.relay.Connection):
    """The connection type for Reactor."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ReactorEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Reactor"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ReactorEdge(sgqlc.types.Type):
    """Represents an author of a reaction."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "reacted_at")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(sgqlc.types.non_null("Reactor"), graphql_name="node")
    """The author of the reaction."""

    reacted_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="reactedAt")
    """The moment when the user made the reaction."""


class RefConnection(sgqlc.types.relay.Connection):
    """The connection type for Ref."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RefEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Ref"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RefEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Ref", graphql_name="node")
    """The item at the end of the edge."""


class RefNameConditionTarget(sgqlc.types.Type):
    """Parameters to be used for the ref_name condition"""

    __schema__ = github_schema
    __field_names__ = ("exclude", "include")
    exclude = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="exclude")
    """Array of ref names or patterns to exclude. The condition will not
    pass if any of these patterns match.
    """

    include = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="include")
    """Array of ref names or patterns to include. One of these patterns
    must match for the condition to pass. Also accepts
    `~DEFAULT_BRANCH` to include the default branch or `~ALL` to
    include all branches.
    """


class RefUpdateRule(sgqlc.types.Type):
    """A ref update rules for a viewer."""

    __schema__ = github_schema
    __field_names__ = (
        "allows_deletions",
        "allows_force_pushes",
        "blocks_creations",
        "pattern",
        "required_approving_review_count",
        "required_status_check_contexts",
        "requires_code_owner_reviews",
        "requires_conversation_resolution",
        "requires_linear_history",
        "requires_signatures",
        "viewer_allowed_to_dismiss_reviews",
        "viewer_can_push",
    )
    allows_deletions = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="allowsDeletions")
    """Can this branch be deleted."""

    allows_force_pushes = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="allowsForcePushes")
    """Are force pushes allowed on this branch."""

    blocks_creations = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="blocksCreations")
    """Can matching branches be created."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """Identifies the protection rule pattern."""

    required_approving_review_count = sgqlc.types.Field(Int, graphql_name="requiredApprovingReviewCount")
    """Number of approving reviews required to update matching branches."""

    required_status_check_contexts = sgqlc.types.Field(sgqlc.types.list_of(String), graphql_name="requiredStatusCheckContexts")
    """List of required status check contexts that must pass for commits
    to be accepted to matching branches.
    """

    requires_code_owner_reviews = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requiresCodeOwnerReviews")
    """Are reviews from code owners required to update matching branches."""

    requires_conversation_resolution = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requiresConversationResolution")
    """Are conversations required to be resolved before merging."""

    requires_linear_history = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requiresLinearHistory")
    """Are merge commits prohibited from being pushed to this branch."""

    requires_signatures = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="requiresSignatures")
    """Are commits required to be signed."""

    viewer_allowed_to_dismiss_reviews = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerAllowedToDismissReviews")
    """Is the viewer allowed to dismiss reviews."""

    viewer_can_push = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="viewerCanPush")
    """Can the viewer push to the branch"""


class RegenerateEnterpriseIdentityProviderRecoveryCodesPayload(sgqlc.types.Type):
    """Autogenerated return type of
    RegenerateEnterpriseIdentityProviderRecoveryCodes
    """

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "identity_provider")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    identity_provider = sgqlc.types.Field("EnterpriseIdentityProvider", graphql_name="identityProvider")
    """The identity provider for the enterprise."""


class RegenerateVerifiableDomainTokenPayload(sgqlc.types.Type):
    """Autogenerated return type of RegenerateVerifiableDomainToken"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "verification_token")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    verification_token = sgqlc.types.Field(String, graphql_name="verificationToken")
    """The verification token that was generated."""


class RejectDeploymentsPayload(sgqlc.types.Type):
    """Autogenerated return type of RejectDeployments"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "deployments")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    deployments = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null("Deployment")), graphql_name="deployments")
    """The affected deployments."""


class ReleaseAssetConnection(sgqlc.types.relay.Connection):
    """The connection type for ReleaseAsset."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ReleaseAssetEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ReleaseAsset"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ReleaseAssetEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ReleaseAsset", graphql_name="node")
    """The item at the end of the edge."""


class ReleaseConnection(sgqlc.types.relay.Connection):
    """The connection type for Release."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ReleaseEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Release"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ReleaseEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Release", graphql_name="node")
    """The item at the end of the edge."""


class RemoveAssigneesFromAssignablePayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveAssigneesFromAssignable"""

    __schema__ = github_schema
    __field_names__ = ("assignable", "client_mutation_id")
    assignable = sgqlc.types.Field(Assignable, graphql_name="assignable")
    """The item that was unassigned."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class RemoveEnterpriseAdminPayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveEnterpriseAdmin"""

    __schema__ = github_schema
    __field_names__ = ("admin", "client_mutation_id", "enterprise", "message", "viewer")
    admin = sgqlc.types.Field("User", graphql_name="admin")
    """The user who was removed as an administrator."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    enterprise = sgqlc.types.Field("Enterprise", graphql_name="enterprise")
    """The updated enterprise."""

    message = sgqlc.types.Field(String, graphql_name="message")
    """A message confirming the result of removing an administrator."""

    viewer = sgqlc.types.Field("User", graphql_name="viewer")
    """The viewer performing the mutation."""


class RemoveEnterpriseIdentityProviderPayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveEnterpriseIdentityProvider"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "identity_provider")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    identity_provider = sgqlc.types.Field("EnterpriseIdentityProvider", graphql_name="identityProvider")
    """The identity provider that was removed from the enterprise."""


class RemoveEnterpriseMemberPayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveEnterpriseMember"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "enterprise", "user", "viewer")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    enterprise = sgqlc.types.Field("Enterprise", graphql_name="enterprise")
    """The updated enterprise."""

    user = sgqlc.types.Field("User", graphql_name="user")
    """The user that was removed from the enterprise."""

    viewer = sgqlc.types.Field("User", graphql_name="viewer")
    """The viewer performing the mutation."""


class RemoveEnterpriseOrganizationPayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveEnterpriseOrganization"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "enterprise", "organization", "viewer")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    enterprise = sgqlc.types.Field("Enterprise", graphql_name="enterprise")
    """The updated enterprise."""

    organization = sgqlc.types.Field("Organization", graphql_name="organization")
    """The organization that was removed from the enterprise."""

    viewer = sgqlc.types.Field("User", graphql_name="viewer")
    """The viewer performing the mutation."""


class RemoveEnterpriseSupportEntitlementPayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveEnterpriseSupportEntitlement"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "message")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    message = sgqlc.types.Field(String, graphql_name="message")
    """A message confirming the result of removing the support
    entitlement.
    """


class RemoveLabelsFromLabelablePayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveLabelsFromLabelable"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "labelable")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    labelable = sgqlc.types.Field(Labelable, graphql_name="labelable")
    """The Labelable the labels were removed from."""


class RemoveOutsideCollaboratorPayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveOutsideCollaborator"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "removed_user")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    removed_user = sgqlc.types.Field("User", graphql_name="removedUser")
    """The user that was removed as an outside collaborator."""


class RemoveReactionPayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveReaction"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "reaction", "reaction_groups", "subject")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    reaction = sgqlc.types.Field("Reaction", graphql_name="reaction")
    """The reaction object."""

    reaction_groups = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(ReactionGroup)), graphql_name="reactionGroups")
    """The reaction groups for the subject."""

    subject = sgqlc.types.Field(Reactable, graphql_name="subject")
    """The reactable subject."""


class RemoveStarPayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveStar"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "starrable")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    starrable = sgqlc.types.Field(Starrable, graphql_name="starrable")
    """The starrable."""


class RemoveUpvotePayload(sgqlc.types.Type):
    """Autogenerated return type of RemoveUpvote"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "subject")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    subject = sgqlc.types.Field(Votable, graphql_name="subject")
    """The votable subject."""


class ReopenDiscussionPayload(sgqlc.types.Type):
    """Autogenerated return type of ReopenDiscussion"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "discussion")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    discussion = sgqlc.types.Field("Discussion", graphql_name="discussion")
    """The discussion that was reopened."""


class ReopenIssuePayload(sgqlc.types.Type):
    """Autogenerated return type of ReopenIssue"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "issue")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    issue = sgqlc.types.Field("Issue", graphql_name="issue")
    """The issue that was opened."""


class ReopenPullRequestPayload(sgqlc.types.Type):
    """Autogenerated return type of ReopenPullRequest"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The pull request that was reopened."""


class RepositoryCodeowners(sgqlc.types.Type):
    """Information extracted from a repository's `CODEOWNERS` file."""

    __schema__ = github_schema
    __field_names__ = ("errors",)
    errors = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("RepositoryCodeownersError"))), graphql_name="errors"
    )
    """Any problems that were encountered while parsing the `CODEOWNERS`
    file.
    """


class RepositoryCodeownersError(sgqlc.types.Type):
    """An error in a `CODEOWNERS` file."""

    __schema__ = github_schema
    __field_names__ = ("column", "kind", "line", "message", "path", "source", "suggestion")
    column = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="column")
    """The column number where the error occurs."""

    kind = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="kind")
    """A short string describing the type of error."""

    line = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="line")
    """The line number where the error occurs."""

    message = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="message")
    """A complete description of the error, combining information from
    other fields.
    """

    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """The path to the file when the error occurs."""

    source = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="source")
    """The content of the line where the error occurs."""

    suggestion = sgqlc.types.Field(String, graphql_name="suggestion")
    """A suggestion of how to fix the error."""


class RepositoryCollaboratorConnection(sgqlc.types.relay.Connection):
    """The connection type for User."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RepositoryCollaboratorEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("User"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RepositoryCollaboratorEdge(sgqlc.types.Type):
    """Represents a user who is a collaborator of a repository."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "permission", "permission_sources")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(sgqlc.types.non_null("User"), graphql_name="node")

    permission = sgqlc.types.Field(sgqlc.types.non_null(RepositoryPermission), graphql_name="permission")
    """The permission the user has on the repository."""

    permission_sources = sgqlc.types.Field(sgqlc.types.list_of(sgqlc.types.non_null(PermissionSource)), graphql_name="permissionSources")
    """A list of sources for the user's access to the repository."""


class RepositoryConnection(sgqlc.types.relay.Connection):
    """A list of repositories owned by the subject."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count", "total_disk_usage")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RepositoryEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Repository"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""

    total_disk_usage = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalDiskUsage")
    """The total size in kilobytes of all repositories in the connection."""


class RepositoryContactLink(sgqlc.types.Type):
    """A repository contact link."""

    __schema__ = github_schema
    __field_names__ = ("about", "name", "url")
    about = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="about")
    """The contact link purpose."""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The contact link name."""

    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """The contact link URL."""


class RepositoryEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Repository", graphql_name="node")
    """The item at the end of the edge."""


class RepositoryInteractionAbility(sgqlc.types.Type):
    """Repository interaction limit that applies to this object."""

    __schema__ = github_schema
    __field_names__ = ("expires_at", "limit", "origin")
    expires_at = sgqlc.types.Field(DateTime, graphql_name="expiresAt")
    """The time the currently active limit expires."""

    limit = sgqlc.types.Field(sgqlc.types.non_null(RepositoryInteractionLimit), graphql_name="limit")
    """The current limit that is enabled on this object."""

    origin = sgqlc.types.Field(sgqlc.types.non_null(RepositoryInteractionLimitOrigin), graphql_name="origin")
    """The origin of the currently active interaction limit."""


class RepositoryInvitationConnection(sgqlc.types.relay.Connection):
    """A list of repository invitations."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RepositoryInvitationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("RepositoryInvitation"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RepositoryInvitationEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("RepositoryInvitation", graphql_name="node")
    """The item at the end of the edge."""


class RepositoryMigrationConnection(sgqlc.types.relay.Connection):
    """The connection type for RepositoryMigration."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RepositoryMigrationEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("RepositoryMigration"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RepositoryMigrationEdge(sgqlc.types.Type):
    """Represents a repository migration."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("RepositoryMigration", graphql_name="node")
    """The item at the end of the edge."""


class RepositoryNameConditionTarget(sgqlc.types.Type):
    """Parameters to be used for the repository_name condition"""

    __schema__ = github_schema
    __field_names__ = ("exclude", "include", "protected")
    exclude = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="exclude")
    """Array of repository names or patterns to exclude. The condition
    will not pass if any of these patterns match.
    """

    include = sgqlc.types.Field(sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="include")
    """Array of repository names or patterns to include. One of these
    patterns must match for the condition to pass. Also accepts `~ALL`
    to include all repositories.
    """

    protected = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="protected")
    """Target changes that match these patterns will be prevented except
    by those with bypass permissions.
    """


class RepositoryRuleConditions(sgqlc.types.Type):
    """Set of conditions that determine if a ruleset will evaluate"""

    __schema__ = github_schema
    __field_names__ = ("ref_name", "repository_name")
    ref_name = sgqlc.types.Field(RefNameConditionTarget, graphql_name="refName")
    """Configuration for the ref_name condition"""

    repository_name = sgqlc.types.Field(RepositoryNameConditionTarget, graphql_name="repositoryName")
    """Configuration for the repository_name condition"""


class RepositoryRuleConnection(sgqlc.types.relay.Connection):
    """The connection type for RepositoryRule."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RepositoryRuleEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("RepositoryRule"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RepositoryRuleEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("RepositoryRule", graphql_name="node")
    """The item at the end of the edge."""


class RepositoryRulesetBypassActorConnection(sgqlc.types.relay.Connection):
    """The connection type for RepositoryRulesetBypassActor."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RepositoryRulesetBypassActorEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("RepositoryRulesetBypassActor"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RepositoryRulesetBypassActorEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("RepositoryRulesetBypassActor", graphql_name="node")
    """The item at the end of the edge."""


class RepositoryRulesetConnection(sgqlc.types.relay.Connection):
    """The connection type for RepositoryRuleset."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RepositoryRulesetEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("RepositoryRuleset"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RepositoryRulesetEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("RepositoryRuleset", graphql_name="node")
    """The item at the end of the edge."""


class RepositoryTopicConnection(sgqlc.types.relay.Connection):
    """The connection type for RepositoryTopic."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RepositoryTopicEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("RepositoryTopic"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RepositoryTopicEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("RepositoryTopic", graphql_name="node")
    """The item at the end of the edge."""


class RepositoryVulnerabilityAlertConnection(sgqlc.types.relay.Connection):
    """The connection type for RepositoryVulnerabilityAlert."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RepositoryVulnerabilityAlertEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("RepositoryVulnerabilityAlert"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RepositoryVulnerabilityAlertEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("RepositoryVulnerabilityAlert", graphql_name="node")
    """The item at the end of the edge."""


class RequestReviewsPayload(sgqlc.types.Type):
    """Autogenerated return type of RequestReviews"""

    __schema__ = github_schema
    __field_names__ = ("actor", "client_mutation_id", "pull_request", "requested_reviewers_edge")
    actor = sgqlc.types.Field(Actor, graphql_name="actor")
    """Identifies the actor who performed the event."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The pull request that is getting requests."""

    requested_reviewers_edge = sgqlc.types.Field("UserEdge", graphql_name="requestedReviewersEdge")
    """The edge from the pull request to the requested reviewers."""


class RequestedReviewerConnection(sgqlc.types.relay.Connection):
    """The connection type for RequestedReviewer."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("RequestedReviewerEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("RequestedReviewer"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class RequestedReviewerEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("RequestedReviewer", graphql_name="node")
    """The item at the end of the edge."""


class RequiredDeploymentsParameters(sgqlc.types.Type):
    """Choose which environments must be successfully deployed to before
    branches can be merged into a branch that matches this rule.
    """

    __schema__ = github_schema
    __field_names__ = ("required_deployment_environments",)
    required_deployment_environments = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null(String))), graphql_name="requiredDeploymentEnvironments"
    )
    """The environments that must be successfully deployed to before
    branches can be merged.
    """


class RequiredStatusCheckDescription(sgqlc.types.Type):
    """Represents a required status check for a protected branch, but not
    any specific run of that check.
    """

    __schema__ = github_schema
    __field_names__ = ("app", "context")
    app = sgqlc.types.Field("App", graphql_name="app")
    """The App that must provide this status in order for it to be
    accepted.
    """

    context = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="context")
    """The name of this status."""


class RequiredStatusChecksParameters(sgqlc.types.Type):
    """Choose which status checks must pass before branches can be merged
    into a branch that matches this rule. When enabled, commits must
    first be pushed to another branch, then merged or pushed directly
    to a branch that matches this rule after status checks have
    passed.
    """

    __schema__ = github_schema
    __field_names__ = ("required_status_checks", "strict_required_status_checks_policy")
    required_status_checks = sgqlc.types.Field(
        sgqlc.types.non_null(sgqlc.types.list_of(sgqlc.types.non_null("StatusCheckConfiguration"))), graphql_name="requiredStatusChecks"
    )
    """Status checks that are required."""

    strict_required_status_checks_policy = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="strictRequiredStatusChecksPolicy")
    """Whether pull requests targeting a matching branch must be tested
    with the latest code. This setting will not take effect unless at
    least one status check is enabled.
    """


class RerequestCheckSuitePayload(sgqlc.types.Type):
    """Autogenerated return type of RerequestCheckSuite"""

    __schema__ = github_schema
    __field_names__ = ("check_suite", "client_mutation_id")
    check_suite = sgqlc.types.Field("CheckSuite", graphql_name="checkSuite")
    """The requested check suite."""

    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""


class ResolveReviewThreadPayload(sgqlc.types.Type):
    """Autogenerated return type of ResolveReviewThread"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "thread")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    thread = sgqlc.types.Field("PullRequestReviewThread", graphql_name="thread")
    """The thread to resolve."""


class RetireSponsorsTierPayload(sgqlc.types.Type):
    """Autogenerated return type of RetireSponsorsTier"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "sponsors_tier")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    sponsors_tier = sgqlc.types.Field("SponsorsTier", graphql_name="sponsorsTier")
    """The tier that was retired."""


class RevertPullRequestPayload(sgqlc.types.Type):
    """Autogenerated return type of RevertPullRequest"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request", "revert_pull_request")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request = sgqlc.types.Field("PullRequest", graphql_name="pullRequest")
    """The pull request that was reverted."""

    revert_pull_request = sgqlc.types.Field("PullRequest", graphql_name="revertPullRequest")
    """The new pull request that reverts the input pull request."""


class ReviewDismissalAllowanceConnection(sgqlc.types.relay.Connection):
    """The connection type for ReviewDismissalAllowance."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ReviewDismissalAllowanceEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ReviewDismissalAllowance"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ReviewDismissalAllowanceEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ReviewDismissalAllowance", graphql_name="node")
    """The item at the end of the edge."""


class ReviewRequestConnection(sgqlc.types.relay.Connection):
    """The connection type for ReviewRequest."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("ReviewRequestEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("ReviewRequest"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class ReviewRequestEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("ReviewRequest", graphql_name="node")
    """The item at the end of the edge."""


class RevokeEnterpriseOrganizationsMigratorRolePayload(sgqlc.types.Type):
    """Autogenerated return type of
    RevokeEnterpriseOrganizationsMigratorRole
    """

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "organizations")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    organizations = sgqlc.types.Field(
        OrganizationConnection,
        graphql_name="organizations",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
            )
        ),
    )
    """The organizations that had the migrator role revoked for the given
    user.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    """


class RevokeMigratorRolePayload(sgqlc.types.Type):
    """Autogenerated return type of RevokeMigratorRole"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "success")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    success = sgqlc.types.Field(Boolean, graphql_name="success")
    """Did the operation succeed?"""


class SavedReplyConnection(sgqlc.types.relay.Connection):
    """The connection type for SavedReply."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SavedReplyEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("SavedReply"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SavedReplyEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("SavedReply", graphql_name="node")
    """The item at the end of the edge."""


class SearchResultItemConnection(sgqlc.types.relay.Connection):
    """A list of results that matched against a search query. Regardless
    of the number of matches, a maximum of 1,000 results will be
    available across all types, potentially split across many pages.
    """

    __schema__ = github_schema
    __field_names__ = (
        "code_count",
        "discussion_count",
        "edges",
        "issue_count",
        "nodes",
        "page_info",
        "repository_count",
        "user_count",
        "wiki_count",
    )
    code_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="codeCount")
    """The total number of pieces of code that matched the search query.
    Regardless of the total number of matches, a maximum of 1,000
    results will be available across all types.
    """

    discussion_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="discussionCount")
    """The total number of discussions that matched the search query.
    Regardless of the total number of matches, a maximum of 1,000
    results will be available across all types.
    """

    edges = sgqlc.types.Field(sgqlc.types.list_of("SearchResultItemEdge"), graphql_name="edges")
    """A list of edges."""

    issue_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="issueCount")
    """The total number of issues that matched the search query.
    Regardless of the total number of matches, a maximum of 1,000
    results will be available across all types.
    """

    nodes = sgqlc.types.Field(sgqlc.types.list_of("SearchResultItem"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    repository_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="repositoryCount")
    """The total number of repositories that matched the search query.
    Regardless of the total number of matches, a maximum of 1,000
    results will be available across all types.
    """

    user_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="userCount")
    """The total number of users that matched the search query.
    Regardless of the total number of matches, a maximum of 1,000
    results will be available across all types.
    """

    wiki_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="wikiCount")
    """The total number of wiki pages that matched the search query.
    Regardless of the total number of matches, a maximum of 1,000
    results will be available across all types.
    """


class SearchResultItemEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "text_matches")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("SearchResultItem", graphql_name="node")
    """The item at the end of the edge."""

    text_matches = sgqlc.types.Field(sgqlc.types.list_of("TextMatch"), graphql_name="textMatches")
    """Text matches on the result found."""


class SecurityAdvisoryConnection(sgqlc.types.relay.Connection):
    """The connection type for SecurityAdvisory."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SecurityAdvisoryEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("SecurityAdvisory"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SecurityAdvisoryEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("SecurityAdvisory", graphql_name="node")
    """The item at the end of the edge."""


class SecurityAdvisoryIdentifier(sgqlc.types.Type):
    """A GitHub Security Advisory Identifier"""

    __schema__ = github_schema
    __field_names__ = ("type", "value")
    type = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="type")
    """The identifier type, e.g. GHSA, CVE"""

    value = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="value")
    """The identifier"""


class SecurityAdvisoryPackage(sgqlc.types.Type):
    """An individual package"""

    __schema__ = github_schema
    __field_names__ = ("ecosystem", "name")
    ecosystem = sgqlc.types.Field(sgqlc.types.non_null(SecurityAdvisoryEcosystem), graphql_name="ecosystem")
    """The ecosystem the package belongs to, e.g. RUBYGEMS, NPM"""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The package name"""


class SecurityAdvisoryPackageVersion(sgqlc.types.Type):
    """An individual package version"""

    __schema__ = github_schema
    __field_names__ = ("identifier",)
    identifier = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="identifier")
    """The package name or version"""


class SecurityAdvisoryReference(sgqlc.types.Type):
    """A GitHub Security Advisory Reference"""

    __schema__ = github_schema
    __field_names__ = ("url",)
    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """A publicly accessible reference"""


class SecurityVulnerability(sgqlc.types.Type):
    """An individual vulnerability within an Advisory"""

    __schema__ = github_schema
    __field_names__ = ("advisory", "first_patched_version", "package", "severity", "updated_at", "vulnerable_version_range")
    advisory = sgqlc.types.Field(sgqlc.types.non_null("SecurityAdvisory"), graphql_name="advisory")
    """The Advisory associated with this Vulnerability"""

    first_patched_version = sgqlc.types.Field(SecurityAdvisoryPackageVersion, graphql_name="firstPatchedVersion")
    """The first version containing a fix for the vulnerability"""

    package = sgqlc.types.Field(sgqlc.types.non_null(SecurityAdvisoryPackage), graphql_name="package")
    """A description of the vulnerable package"""

    severity = sgqlc.types.Field(sgqlc.types.non_null(SecurityAdvisorySeverity), graphql_name="severity")
    """The severity of the vulnerability within this package"""

    updated_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="updatedAt")
    """When the vulnerability was last updated"""

    vulnerable_version_range = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="vulnerableVersionRange")
    """A string that describes the vulnerable package versions. This
    string follows a basic syntax with a few forms. + `= 0.2.0`
    denotes a single vulnerable version. + `<= 1.0.8` denotes a
    version range up to and including the specified version + `<
    0.1.11` denotes a version range up to, but excluding, the
    specified version + `>= 4.3.0, < 4.3.5` denotes a version range
    with a known minimum and maximum version. + `>= 0.0.1` denotes a
    version range with a known minimum, but no known maximum
    """


class SecurityVulnerabilityConnection(sgqlc.types.relay.Connection):
    """The connection type for SecurityVulnerability."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SecurityVulnerabilityEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(SecurityVulnerability), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SecurityVulnerabilityEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(SecurityVulnerability, graphql_name="node")
    """The item at the end of the edge."""


class SetEnterpriseIdentityProviderPayload(sgqlc.types.Type):
    """Autogenerated return type of SetEnterpriseIdentityProvider"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "identity_provider")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    identity_provider = sgqlc.types.Field("EnterpriseIdentityProvider", graphql_name="identityProvider")
    """The identity provider for the enterprise."""


class SetOrganizationInteractionLimitPayload(sgqlc.types.Type):
    """Autogenerated return type of SetOrganizationInteractionLimit"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "organization")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    organization = sgqlc.types.Field("Organization", graphql_name="organization")
    """The organization that the interaction limit was set for."""


class SetRepositoryInteractionLimitPayload(sgqlc.types.Type):
    """Autogenerated return type of SetRepositoryInteractionLimit"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "repository")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    repository = sgqlc.types.Field("Repository", graphql_name="repository")
    """The repository that the interaction limit was set for."""


class SetUserInteractionLimitPayload(sgqlc.types.Type):
    """Autogenerated return type of SetUserInteractionLimit"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "user")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    user = sgqlc.types.Field("User", graphql_name="user")
    """The user that the interaction limit was set for."""


class SocialAccount(sgqlc.types.Type):
    """Social media profile associated with a user."""

    __schema__ = github_schema
    __field_names__ = ("display_name", "provider", "url")
    display_name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="displayName")
    """Name of the social media account as it appears on the profile."""

    provider = sgqlc.types.Field(sgqlc.types.non_null(SocialAccountProvider), graphql_name="provider")
    """Software or company that hosts the social media account."""

    url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="url")
    """URL of the social media account."""


class SocialAccountConnection(sgqlc.types.relay.Connection):
    """The connection type for SocialAccount."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SocialAccountEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(SocialAccount), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SocialAccountEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(SocialAccount, graphql_name="node")
    """The item at the end of the edge."""


class SponsorConnection(sgqlc.types.relay.Connection):
    """The connection type for Sponsor."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SponsorEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Sponsor"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SponsorEdge(sgqlc.types.Type):
    """Represents a user or organization who is sponsoring someone in
    GitHub Sponsors.
    """

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Sponsor", graphql_name="node")
    """The item at the end of the edge."""


class SponsorableItemConnection(sgqlc.types.relay.Connection):
    """The connection type for SponsorableItem."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SponsorableItemEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("SponsorableItem"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SponsorableItemEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("SponsorableItem", graphql_name="node")
    """The item at the end of the edge."""


class SponsorsActivityConnection(sgqlc.types.relay.Connection):
    """The connection type for SponsorsActivity."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SponsorsActivityEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("SponsorsActivity"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SponsorsActivityEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("SponsorsActivity", graphql_name="node")
    """The item at the end of the edge."""


class SponsorsGoal(sgqlc.types.Type):
    """A goal associated with a GitHub Sponsors listing, representing a
    target the sponsored maintainer would like to attain.
    """

    __schema__ = github_schema
    __field_names__ = ("description", "kind", "percent_complete", "target_value", "title")
    description = sgqlc.types.Field(String, graphql_name="description")
    """A description of the goal from the maintainer."""

    kind = sgqlc.types.Field(sgqlc.types.non_null(SponsorsGoalKind), graphql_name="kind")
    """What the objective of this goal is."""

    percent_complete = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="percentComplete")
    """The percentage representing how complete this goal is, between
    0-100.
    """

    target_value = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="targetValue")
    """What the goal amount is. Represents an amount in USD for monthly
    sponsorship amount goals. Represents a count of unique sponsors
    for total sponsors count goals.
    """

    title = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="title")
    """A brief summary of the kind and target value of this goal."""


class SponsorsTierAdminInfo(sgqlc.types.Type):
    """SponsorsTier information only visible to users that can administer
    the associated Sponsors listing.
    """

    __schema__ = github_schema
    __field_names__ = ("is_draft", "is_published", "is_retired", "sponsorships")
    is_draft = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isDraft")
    """Indicates whether this tier is still a work in progress by the
    sponsorable and not yet published to the associated GitHub
    Sponsors profile. Draft tiers cannot be used for new sponsorships
    and will not be in use on existing sponsorships. Draft tiers
    cannot be seen by anyone but the admins of the GitHub Sponsors
    profile.
    """

    is_published = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isPublished")
    """Indicates whether this tier is published to the associated GitHub
    Sponsors profile. Published tiers are visible to anyone who can
    see the GitHub Sponsors profile, and are available for use in
    sponsorships if the GitHub Sponsors profile is publicly visible.
    """

    is_retired = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isRetired")
    """Indicates whether this tier has been retired from the associated
    GitHub Sponsors profile. Retired tiers are no longer shown on the
    GitHub Sponsors profile and cannot be chosen for new sponsorships.
    Existing sponsorships may still use retired tiers if the sponsor
    selected the tier before it was retired.
    """

    sponsorships = sgqlc.types.Field(
        sgqlc.types.non_null("SponsorshipConnection"),
        graphql_name="sponsorships",
        args=sgqlc.types.ArgDict(
            (
                ("after", sgqlc.types.Arg(String, graphql_name="after", default=None)),
                ("before", sgqlc.types.Arg(String, graphql_name="before", default=None)),
                ("first", sgqlc.types.Arg(Int, graphql_name="first", default=None)),
                ("last", sgqlc.types.Arg(Int, graphql_name="last", default=None)),
                ("include_private", sgqlc.types.Arg(Boolean, graphql_name="includePrivate", default=False)),
                ("order_by", sgqlc.types.Arg(SponsorshipOrder, graphql_name="orderBy", default=None)),
            )
        ),
    )
    """The sponsorships using this tier.

    Arguments:

    * `after` (`String`): Returns the elements in the list that come
      after the specified cursor.
    * `before` (`String`): Returns the elements in the list that come
      before the specified cursor.
    * `first` (`Int`): Returns the first _n_ elements from the list.
    * `last` (`Int`): Returns the last _n_ elements from the list.
    * `include_private` (`Boolean`): Whether or not to return private
      sponsorships using this tier. Defaults to only returning public
      sponsorships on this tier. (default: `false`)
    * `order_by` (`SponsorshipOrder`): Ordering options for
      sponsorships returned from this connection. If left blank, the
      sponsorships will be ordered based on relevancy to the viewer.
    """


class SponsorsTierConnection(sgqlc.types.relay.Connection):
    """The connection type for SponsorsTier."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SponsorsTierEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("SponsorsTier"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SponsorsTierEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("SponsorsTier", graphql_name="node")
    """The item at the end of the edge."""


class SponsorshipConnection(sgqlc.types.relay.Connection):
    """The connection type for Sponsorship."""

    __schema__ = github_schema
    __field_names__ = (
        "edges",
        "nodes",
        "page_info",
        "total_count",
        "total_recurring_monthly_price_in_cents",
        "total_recurring_monthly_price_in_dollars",
    )
    edges = sgqlc.types.Field(sgqlc.types.list_of("SponsorshipEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Sponsorship"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""

    total_recurring_monthly_price_in_cents = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalRecurringMonthlyPriceInCents")
    """The total amount in cents of all recurring sponsorships in the
    connection whose amount you can view. Does not include one-time
    sponsorships.
    """

    total_recurring_monthly_price_in_dollars = sgqlc.types.Field(
        sgqlc.types.non_null(Int), graphql_name="totalRecurringMonthlyPriceInDollars"
    )
    """The total amount in USD of all recurring sponsorships in the
    connection whose amount you can view. Does not include one-time
    sponsorships.
    """


class SponsorshipEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Sponsorship", graphql_name="node")
    """The item at the end of the edge."""


class SponsorshipNewsletterConnection(sgqlc.types.relay.Connection):
    """The connection type for SponsorshipNewsletter."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SponsorshipNewsletterEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("SponsorshipNewsletter"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SponsorshipNewsletterEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("SponsorshipNewsletter", graphql_name="node")
    """The item at the end of the edge."""


class StargazerConnection(sgqlc.types.relay.Connection):
    """The connection type for User."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("StargazerEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("User"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class StargazerEdge(sgqlc.types.Type):
    """Represents a user that's starred a repository."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "starred_at")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(sgqlc.types.non_null("User"), graphql_name="node")

    starred_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="starredAt")
    """Identifies when the item was starred."""


class StarredRepositoryConnection(sgqlc.types.relay.Connection):
    """The connection type for Repository."""

    __schema__ = github_schema
    __field_names__ = ("edges", "is_over_limit", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("StarredRepositoryEdge"), graphql_name="edges")
    """A list of edges."""

    is_over_limit = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isOverLimit")
    """Is the list of stars for this user truncated? This is true for
    users that have many stars.
    """

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Repository"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class StarredRepositoryEdge(sgqlc.types.Type):
    """Represents a starred repository."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node", "starred_at")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(sgqlc.types.non_null("Repository"), graphql_name="node")

    starred_at = sgqlc.types.Field(sgqlc.types.non_null(DateTime), graphql_name="starredAt")
    """Identifies when the item was starred."""


class StartOrganizationMigrationPayload(sgqlc.types.Type):
    """Autogenerated return type of StartOrganizationMigration"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "org_migration")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    org_migration = sgqlc.types.Field("OrganizationMigration", graphql_name="orgMigration")
    """The new organization migration."""


class StartRepositoryMigrationPayload(sgqlc.types.Type):
    """Autogenerated return type of StartRepositoryMigration"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "repository_migration")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    repository_migration = sgqlc.types.Field("RepositoryMigration", graphql_name="repositoryMigration")
    """The new repository migration."""


class StatusCheckConfiguration(sgqlc.types.Type):
    """Required status check"""

    __schema__ = github_schema
    __field_names__ = ("context", "integration_id")
    context = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="context")
    """The status check context name that must be present on the commit."""

    integration_id = sgqlc.types.Field(Int, graphql_name="integrationId")
    """The optional integration ID that this status check must originate
    from.
    """


class StatusCheckRollupContextConnection(sgqlc.types.relay.Connection):
    """The connection type for StatusCheckRollupContext."""

    __schema__ = github_schema
    __field_names__ = (
        "check_run_count",
        "check_run_counts_by_state",
        "edges",
        "nodes",
        "page_info",
        "status_context_count",
        "status_context_counts_by_state",
        "total_count",
    )
    check_run_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="checkRunCount")
    """The number of check runs in this rollup."""

    check_run_counts_by_state = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null(CheckRunStateCount)), graphql_name="checkRunCountsByState"
    )
    """Counts of check runs by state."""

    edges = sgqlc.types.Field(sgqlc.types.list_of("StatusCheckRollupContextEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("StatusCheckRollupContext"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    status_context_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="statusContextCount")
    """The number of status contexts in this rollup."""

    status_context_counts_by_state = sgqlc.types.Field(
        sgqlc.types.list_of(sgqlc.types.non_null("StatusContextStateCount")), graphql_name="statusContextCountsByState"
    )
    """Counts of status contexts by state."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class StatusCheckRollupContextEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("StatusCheckRollupContext", graphql_name="node")
    """The item at the end of the edge."""


class StatusContextStateCount(sgqlc.types.Type):
    """Represents a count of the state of a status context."""

    __schema__ = github_schema
    __field_names__ = ("count", "state")
    count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="count")
    """The number of statuses with this state."""

    state = sgqlc.types.Field(sgqlc.types.non_null(StatusState), graphql_name="state")
    """The state of a status context."""


class StripeConnectAccount(sgqlc.types.Type):
    """A Stripe Connect account for receiving sponsorship funds from
    GitHub Sponsors.
    """

    __schema__ = github_schema
    __field_names__ = (
        "account_id",
        "billing_country_or_region",
        "country_or_region",
        "is_active",
        "sponsors_listing",
        "stripe_dashboard_url",
    )
    account_id = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="accountId")
    """The account number used to identify this Stripe Connect account."""

    billing_country_or_region = sgqlc.types.Field(String, graphql_name="billingCountryOrRegion")
    """The name of the country or region of an external account, such as
    a bank account, tied to the Stripe Connect account. Will only
    return a value when queried by the maintainer of the associated
    GitHub Sponsors profile themselves, or by an admin of the
    sponsorable organization.
    """

    country_or_region = sgqlc.types.Field(String, graphql_name="countryOrRegion")
    """The name of the country or region of the Stripe Connect account.
    Will only return a value when queried by the maintainer of the
    associated GitHub Sponsors profile themselves, or by an admin of
    the sponsorable organization.
    """

    is_active = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isActive")
    """Whether this Stripe Connect account is currently in use for the
    associated GitHub Sponsors profile.
    """

    sponsors_listing = sgqlc.types.Field(sgqlc.types.non_null("SponsorsListing"), graphql_name="sponsorsListing")
    """The GitHub Sponsors profile associated with this Stripe Connect
    account.
    """

    stripe_dashboard_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="stripeDashboardUrl")
    """The URL to access this Stripe Connect account on Stripe's website."""


class SubmitPullRequestReviewPayload(sgqlc.types.Type):
    """Autogenerated return type of SubmitPullRequestReview"""

    __schema__ = github_schema
    __field_names__ = ("client_mutation_id", "pull_request_review")
    client_mutation_id = sgqlc.types.Field(String, graphql_name="clientMutationId")
    """A unique identifier for the client performing the mutation."""

    pull_request_review = sgqlc.types.Field("PullRequestReview", graphql_name="pullRequestReview")
    """The submitted pull request review."""


class Submodule(sgqlc.types.Type):
    """A pointer to a repository at a specific revision embedded inside
    another repository.
    """

    __schema__ = github_schema
    __field_names__ = ("branch", "git_url", "name", "name_raw", "path", "path_raw", "subproject_commit_oid")
    branch = sgqlc.types.Field(String, graphql_name="branch")
    """The branch of the upstream submodule for tracking updates"""

    git_url = sgqlc.types.Field(sgqlc.types.non_null(URI), graphql_name="gitUrl")
    """The git URL of the submodule repository"""

    name = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="name")
    """The name of the submodule in .gitmodules"""

    name_raw = sgqlc.types.Field(sgqlc.types.non_null(Base64String), graphql_name="nameRaw")
    """The name of the submodule in .gitmodules (Base64-encoded)"""

    path = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="path")
    """The path in the superproject that this submodule is located in"""

    path_raw = sgqlc.types.Field(sgqlc.types.non_null(Base64String), graphql_name="pathRaw")
    """The path in the superproject that this submodule is located in
    (Base64-encoded)
    """

    subproject_commit_oid = sgqlc.types.Field(GitObjectID, graphql_name="subprojectCommitOid")
    """The commit revision of the subproject repository being tracked by
    the submodule
    """


class SubmoduleConnection(sgqlc.types.relay.Connection):
    """The connection type for Submodule."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("SubmoduleEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of(Submodule), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class SubmoduleEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field(Submodule, graphql_name="node")
    """The item at the end of the edge."""


class SuggestedReviewer(sgqlc.types.Type):
    """A suggestion to review a pull request based on a user's commit
    history and review comments.
    """

    __schema__ = github_schema
    __field_names__ = ("is_author", "is_commenter", "reviewer")
    is_author = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isAuthor")
    """Is this suggestion based on past commits?"""

    is_commenter = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="isCommenter")
    """Is this suggestion based on past review comments?"""

    reviewer = sgqlc.types.Field(sgqlc.types.non_null("User"), graphql_name="reviewer")
    """Identifies the user suggested to review the pull request."""


class TagNamePatternParameters(sgqlc.types.Type):
    """Parameters to be used for the tag_name_pattern rule"""

    __schema__ = github_schema
    __field_names__ = ("name", "negate", "operator", "pattern")
    name = sgqlc.types.Field(String, graphql_name="name")
    """How this rule will appear to users."""

    negate = sgqlc.types.Field(sgqlc.types.non_null(Boolean), graphql_name="negate")
    """If true, the rule will fail if the pattern matches."""

    operator = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="operator")
    """The operator to use for matching."""

    pattern = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="pattern")
    """The pattern to match with."""


class TeamConnection(sgqlc.types.relay.Connection):
    """The connection type for Team."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("TeamEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("Team"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class TeamDiscussionCommentConnection(sgqlc.types.relay.Connection):
    """The connection type for TeamDiscussionComment."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("TeamDiscussionCommentEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("TeamDiscussionComment"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class TeamDiscussionCommentEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("TeamDiscussionComment", graphql_name="node")
    """The item at the end of the edge."""


class TeamDiscussionConnection(sgqlc.types.relay.Connection):
    """The connection type for TeamDiscussion."""

    __schema__ = github_schema
    __field_names__ = ("edges", "nodes", "page_info", "total_count")
    edges = sgqlc.types.Field(sgqlc.types.list_of("TeamDiscussionEdge"), graphql_name="edges")
    """A list of edges."""

    nodes = sgqlc.types.Field(sgqlc.types.list_of("TeamDiscussion"), graphql_name="nodes")
    """A list of nodes."""

    page_info = sgqlc.types.Field(sgqlc.types.non_null(PageInfo), graphql_name="pageInfo")
    """Information to aid in pagination."""

    total_count = sgqlc.types.Field(sgqlc.types.non_null(Int), graphql_name="totalCount")
    """Identifies the total count of items in the connection."""


class TeamDiscussionEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("TeamDiscussion", graphql_name="node")
    """The item at the end of the edge."""


class TeamEdge(sgqlc.types.Type):
    """An edge in a connection."""

    __schema__ = github_schema
    __field_names__ = ("cursor", "node")
    cursor = sgqlc.types.Field(sgqlc.types.non_null(String), graphql_name="cursor")
    """A cursor for use in pagination."""

    node = sgqlc.types.Field("Team", graphql_name="node")
    """The item at the end of the edge."""


class TeamMemberConnection(sgqlc.types.relay.Connection):
    """The connection type for User."