import { ISchedule, ScheduleTargetConfig, ScheduleTargetInput } from '@aws-cdk/aws-scheduler-alpha';
import { Duration } from 'aws-cdk-lib';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as sqs from 'aws-cdk-lib/aws-sqs';
/**
 * Base properties for a Schedule Target
 */
export interface ScheduleTargetBaseProps {
    /**
     * An execution role is an IAM role that EventBridge Scheduler assumes in order to interact with other AWS services on your behalf.
     *
     * If none provided templates target will automatically create an IAM role with all the minimum necessary
     * permissions to interact with the templated target. If you wish you may specify your own IAM role, then the templated targets
     * will grant minimal required permissions.
     *
     * Universal target automatically create an IAM role if you do not specify your own IAM role.
     * However, in comparison with templated targets, for universal targets you must grant the required
     * IAM permissions yourself.
     *
     * @default - created by target
     */
    readonly role?: iam.IRole;
    /**
     * The SQS queue to be used as deadLetterQueue.
     *
     * The events not successfully delivered are automatically retried for a specified period of time,
     * depending on the retry policy of the target.
     * If an event is not delivered before all retry attempts are exhausted, it will be sent to the dead letter queue.
     *
     * @default - no dead-letter queue
     */
    readonly deadLetterQueue?: sqs.IQueue;
    /**
     * Input passed to the target.
     *
     * @default - no input.
     */
    readonly input?: ScheduleTargetInput;
    /**
     * The maximum age of a request that Scheduler sends to a target for processing.
     *
     * Minimum value of 60.
     * Maximum value of 86400.
     *
     * @default Duration.hours(24)
     */
    readonly maxEventAge?: Duration;
    /**
     * The maximum number of times to retry when the target returns an error.
     *
     * Minimum value of 0.
     * Maximum value of 185.
     *
     * @default 185
     */
    readonly retryAttempts?: number;
}
/**
 * Base class for Schedule Targets
 */
export declare abstract class ScheduleTargetBase {
    private readonly baseProps;
    protected readonly targetArn: string;
    constructor(baseProps: ScheduleTargetBaseProps, targetArn: string);
    protected abstract addTargetActionToRole(schedule: ISchedule, role: iam.IRole): void;
    protected bindBaseTargetConfig(_schedule: ISchedule): ScheduleTargetConfig;
    /**
     * Create a return a Schedule Target Configuration for the given schedule
     * @param schedule
     * @returns a Schedule Target Configuration
     */
    bind(schedule: ISchedule): ScheduleTargetConfig;
    /**
     * Obtain the Role for the EventBridge Scheduler event
     *
     * If a role already exists, it will be returned. This ensures that if multiple
     * events have the same target, they will share a role.
     */
    private singletonScheduleRole;
    /**
     * Allow a schedule to send events with failed invocation to an Amazon SQS queue.
     * @param schedule schedule to add DLQ to
     * @param queue the DLQ
     */
    private addToDeadLetterQueueResourcePolicy;
    private renderRetryPolicy;
}
