"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScheduleTargetBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const helpers_internal_1 = require("aws-cdk-lib/core/lib/helpers-internal");
const util_1 = require("./util");
/**
 * Base class for Schedule Targets
 */
class ScheduleTargetBase {
    constructor(baseProps, targetArn) {
        this.baseProps = baseProps;
        this.targetArn = targetArn;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_scheduler_targets_alpha_ScheduleTargetBaseProps(baseProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ScheduleTargetBase);
            }
            throw error;
        }
    }
    bindBaseTargetConfig(_schedule) {
        const role = this.baseProps.role ?? this.singletonScheduleRole(_schedule, this.targetArn);
        this.addTargetActionToRole(_schedule, role);
        if (this.baseProps.deadLetterQueue) {
            this.addToDeadLetterQueueResourcePolicy(_schedule, this.baseProps.deadLetterQueue);
        }
        return {
            arn: this.targetArn,
            role: role,
            deadLetterConfig: this.baseProps.deadLetterQueue ? {
                arn: this.baseProps.deadLetterQueue.queueArn,
            } : undefined,
            retryPolicy: this.renderRetryPolicy(this.baseProps.maxEventAge, this.baseProps.retryAttempts),
            input: this.baseProps.input,
        };
    }
    /**
     * Create a return a Schedule Target Configuration for the given schedule
     * @param schedule
     * @returns a Schedule Target Configuration
     */
    bind(schedule) {
        return this.bindBaseTargetConfig(schedule);
    }
    /**
     * Obtain the Role for the EventBridge Scheduler event
     *
     * If a role already exists, it will be returned. This ensures that if multiple
     * events have the same target, they will share a role.
     */
    singletonScheduleRole(schedule, targetArn) {
        const stack = aws_cdk_lib_1.Stack.of(schedule);
        const arn = aws_cdk_lib_1.Token.isUnresolved(targetArn) ? stack.resolve(targetArn).toString() : targetArn;
        const hash = (0, helpers_internal_1.md5hash)(arn).slice(0, 6);
        const id = 'SchedulerRoleForTarget-' + hash;
        const existingRole = stack.node.tryFindChild(id);
        const principal = new iam.PrincipalWithConditions(new iam.ServicePrincipal('scheduler.amazonaws.com'), {
            StringEquals: {
                'aws:SourceAccount': schedule.env.account,
            },
        });
        if (existingRole) {
            existingRole.assumeRolePolicy?.addStatements(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [principal],
                actions: ['sts:AssumeRole'],
            }));
            return existingRole;
        }
        const role = new iam.Role(stack, id, {
            roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: principal,
        });
        return role;
    }
    /**
     * Allow a schedule to send events with failed invocation to an Amazon SQS queue.
     * @param schedule schedule to add DLQ to
     * @param queue the DLQ
     */
    addToDeadLetterQueueResourcePolicy(schedule, queue) {
        if (!(0, util_1.sameEnvDimension)(schedule.env.region, queue.env.region)) {
            throw new Error(`Cannot assign Dead Letter Queue in region ${queue.env.region} to the schedule ${aws_cdk_lib_1.Names.nodeUniqueId(schedule.node)} in region ${schedule.env.region}. Both the queue and the schedule must be in the same region.`);
        }
        // Skip Resource Policy creation if the Queue is not in the same account.
        // There is no way to add a target onto an imported schedule, so we can assume we will run the following code only
        // in the account where the schedule is created.
        if ((0, util_1.sameEnvDimension)(schedule.env.account, queue.env.account)) {
            const policyStatementId = `AllowSchedule${aws_cdk_lib_1.Names.nodeUniqueId(schedule.node)}`;
            queue.addToResourcePolicy(new iam.PolicyStatement({
                sid: policyStatementId,
                principals: [new iam.ServicePrincipal('scheduler.amazonaws.com')],
                effect: iam.Effect.ALLOW,
                actions: ['sqs:SendMessage'],
                resources: [queue.queueArn],
            }));
        }
        else {
            aws_cdk_lib_1.Annotations.of(schedule).addWarning(`Cannot add a resource policy to your dead letter queue associated with schedule ${schedule.scheduleName} because the queue is in a different account. You must add the resource policy manually to the dead letter queue in account ${queue.env.account}.`);
        }
    }
    renderRetryPolicy(maximumEventAge, maximumRetryAttempts) {
        const maxMaxAge = aws_cdk_lib_1.Duration.days(1).toSeconds();
        const minMaxAge = aws_cdk_lib_1.Duration.minutes(15).toSeconds();
        let maxAge = maxMaxAge;
        if (maximumEventAge) {
            maxAge = maximumEventAge.toSeconds({ integral: true });
            if (maxAge > maxMaxAge) {
                throw new Error('Maximum event age is 1 day');
            }
            if (maxAge < minMaxAge) {
                throw new Error('Minimum event age is 15 minutes');
            }
        }
        ;
        let maxAttempts = 185;
        if (typeof maximumRetryAttempts != 'undefined') {
            if (maximumRetryAttempts < 0) {
                throw Error('Number of retry attempts should be greater or equal than 0');
            }
            if (maximumRetryAttempts > 185) {
                throw Error('Number of retry attempts should be less or equal than 185');
            }
            maxAttempts = maximumRetryAttempts;
        }
        return {
            maximumEventAgeInSeconds: maxAge,
            maximumRetryAttempts: maxAttempts,
        };
    }
}
exports.ScheduleTargetBase = ScheduleTargetBase;
_a = JSII_RTTI_SYMBOL_1;
ScheduleTargetBase[_a] = { fqn: "@aws-cdk/aws-scheduler-targets-alpha.ScheduleTargetBase", version: "2.131.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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