import unittest as ut
from itertools import chain
from typing import Generator, Sequence

import numpy as np

from seismicrna.relate.aux.iterrel import iter_relvecs_q53, iter_relvecs_all
from seismicrna.core.rel import (DELET,
                                 INS_5,
                                 INS_3,
                                 SUB_A,
                                 SUB_C,
                                 SUB_G,
                                 SUB_T,
                                 ANY_B,
                                 ANY_D,
                                 ANY_H,
                                 ANY_V,
                                 ANY_N)
from seismicrna.core.seq import DNA, expand_degenerate_seq


class TestIterRelvecsQ53(ut.TestCase):
    """ Test function `iter_relvecs_q53`. """

    @staticmethod
    def list_rels(seq: str,
                  low_qual: Sequence[int] = (),
                  end5: int | None = None,
                  end3: int | None = None,
                  insert3: bool = True,
                  max_ins: int | None = None):
        """ Convenience function to run `rel.iter_relvecs_q53` from a
        sequence of str and return a list of lists of ints. """
        return list(iter_relvecs_q53(DNA(seq),
                                     low_qual,
                                     end5,
                                     end3,
                                     insert3,
                                     max_ins))

    def test_type(self):
        """ Test that the function yields (int, int, dict) tuples. """
        self.assertTrue(isinstance(iter_relvecs_q53(DNA("ACGT")), Generator))
        relvecs = list(iter_relvecs_q53(DNA("ACGT")))
        self.assertTrue(all(isinstance(relvec, tuple) for relvec in relvecs))
        self.assertTrue(all(len(relvec) == 3 for relvec in relvecs))
        self.assertTrue(all(isinstance(relvec[0], int) for relvec in relvecs))
        self.assertTrue(all(isinstance(relvec[1], int) for relvec in relvecs))
        self.assertTrue(all(isinstance(relvec[2], dict) for relvec in relvecs))

    def test_a(self):
        """ Test with sequence 'A'. """
        self.assertEqual(self.list_rels("A"),
                         [(1, 1, {}),
                          (1, 1, {1: SUB_C}),
                          (1, 1, {1: SUB_G}),
                          (1, 1, {1: SUB_T})])

    def test_c(self):
        """ Test with sequence 'C'. """
        self.assertEqual(self.list_rels("C"),
                         [(1, 1, {1: SUB_A}),
                          (1, 1, {}),
                          (1, 1, {1: SUB_G}),
                          (1, 1, {1: SUB_T})])

    def test_g(self):
        """ Test with sequence 'G'. """
        self.assertEqual(self.list_rels("G"),
                         [(1, 1, {1: SUB_A}),
                          (1, 1, {1: SUB_C}),
                          (1, 1, {}),
                          (1, 1, {1: SUB_T})])

    def test_t(self):
        """ Test with sequence 'T'. """
        self.assertEqual(self.list_rels("T"),
                         [(1, 1, {1: SUB_A}),
                          (1, 1, {1: SUB_C}),
                          (1, 1, {1: SUB_G}),
                          (1, 1, {})])

    def test_n(self):
        """ Test with sequence 'N'. """
        self.assertEqual(self.list_rels("N"),
                         [(1, 1, {1: ANY_N}),
                          (1, 1, {1: ANY_N}),
                          (1, 1, {1: ANY_N}),
                          (1, 1, {1: ANY_N})])

    def test_aa_insert5(self):
        """ Test with sequence 'AA'. """
        self.assertEqual(self.list_rels("AA", insert3=False),
                         [(1, 2, {}),
                          (1, 2, {1: INS_5}),
                          (1, 2, {2: SUB_C}),
                          (1, 2, {1: INS_5,
                                  2: SUB_C}),
                          (1, 2, {2: SUB_G}),
                          (1, 2, {1: INS_5,
                                  2: SUB_G}),
                          (1, 2, {2: SUB_T}),
                          (1, 2, {1: INS_5,
                                  2: SUB_T}),
                          (1, 2, {1: SUB_C}),
                          (1, 2, {1: SUB_C + INS_5}),
                          (1, 2, {1: SUB_C,
                                  2: SUB_C}),
                          (1, 2, {1: SUB_C + INS_5,
                                  2: SUB_C}),
                          (1, 2, {1: SUB_C,
                                  2: SUB_G}),
                          (1, 2, {1: SUB_C + INS_5,
                                  2: SUB_G}),
                          (1, 2, {1: SUB_C,
                                  2: SUB_T}),
                          (1, 2, {1: SUB_C + INS_5,
                                  2: SUB_T}),
                          (1, 2, {1: SUB_G}),
                          (1, 2, {1: SUB_G + INS_5}),
                          (1, 2, {1: SUB_G,
                                  2: SUB_C}),
                          (1, 2, {1: SUB_G + INS_5,
                                  2: SUB_C}),
                          (1, 2, {1: SUB_G,
                                  2: SUB_G}),
                          (1, 2, {1: SUB_G + INS_5,
                                  2: SUB_G}),
                          (1, 2, {1: SUB_G,
                                  2: SUB_T}),
                          (1, 2, {1: SUB_G + INS_5,
                                  2: SUB_T}),
                          (1, 2, {1: SUB_T}),
                          (1, 2, {1: SUB_T + INS_5}),
                          (1, 2, {1: SUB_T,
                                  2: SUB_C}),
                          (1, 2, {1: SUB_T + INS_5,
                                  2: SUB_C}),
                          (1, 2, {1: SUB_T,
                                  2: SUB_G}),
                          (1, 2, {1: SUB_T + INS_5,
                                  2: SUB_G}),
                          (1, 2, {1: SUB_T,
                                  2: SUB_T}),
                          (1, 2, {1: SUB_T + INS_5,
                                  2: SUB_T})])

    def test_aa_insert3(self):
        """ Test with sequence 'AA'. """
        self.assertEqual(self.list_rels("AA", insert3=True),
                         [(1, 2, {}),
                          (1, 2, {2: INS_3}),
                          (1, 2, {2: SUB_C}),
                          (1, 2, {2: INS_3 + SUB_C}),
                          (1, 2, {2: SUB_G}),
                          (1, 2, {2: INS_3 + SUB_G}),
                          (1, 2, {2: SUB_T}),
                          (1, 2, {2: INS_3 + SUB_T}),
                          (1, 2, {1: SUB_C}),
                          (1, 2, {1: SUB_C,
                                  2: INS_3}),
                          (1, 2, {1: SUB_C,
                                  2: SUB_C}),
                          (1, 2, {1: SUB_C,
                                  2: INS_3 + SUB_C}),
                          (1, 2, {1: SUB_C,
                                  2: SUB_G}),
                          (1, 2, {1: SUB_C,
                                  2: INS_3 + SUB_G}),
                          (1, 2, {1: SUB_C,
                                  2: SUB_T}),
                          (1, 2, {1: SUB_C,
                                  2: INS_3 + SUB_T}),
                          (1, 2, {1: SUB_G}),
                          (1, 2, {1: SUB_G,
                                  2: INS_3}),
                          (1, 2, {1: SUB_G,
                                  2: SUB_C}),
                          (1, 2, {1: SUB_G,
                                  2: INS_3 + SUB_C}),
                          (1, 2, {1: SUB_G,
                                  2: SUB_G}),
                          (1, 2, {1: SUB_G,
                                  2: INS_3 + SUB_G}),
                          (1, 2, {1: SUB_G,
                                  2: SUB_T}),
                          (1, 2, {1: SUB_G,
                                  2: INS_3 + SUB_T}),
                          (1, 2, {1: SUB_T}),
                          (1, 2, {1: SUB_T,
                                  2: INS_3}),
                          (1, 2, {1: SUB_T,
                                  2: SUB_C}),
                          (1, 2, {1: SUB_T,
                                  2: INS_3 + SUB_C}),
                          (1, 2, {1: SUB_T,
                                  2: SUB_G}),
                          (1, 2, {1: SUB_T,
                                  2: INS_3 + SUB_G}),
                          (1, 2, {1: SUB_T,
                                  2: SUB_T}),
                          (1, 2, {1: SUB_T,
                                  2: INS_3 + SUB_T})])

    def test_low_qual(self):
        """ Test with each low-quality base. """
        for base, low_qual in zip("ACGTN",
                                  [ANY_B, ANY_D, ANY_H, ANY_V, ANY_N],
                                  strict=True):
            self.assertEqual(self.list_rels(base, [1]),
                             [(1, 1, {1: low_qual})])

    def test_low_qual_invalid(self):
        """ Test that invalid low-qual positions raise ValueError. """
        seq = "ACGTN"
        for n in range(1, len(seq) + 1):
            self.assertRaises(ValueError, self.list_rels, seq[: n], [0])
            self.assertRaises(ValueError, self.list_rels, seq[: n], [n + 1])

    def test_xaax_insert5(self):
        """ Test that bases with no coverage are marked. """
        self.assertEqual(self.list_rels("TAAG", end5=2, end3=3, insert3=False),
                         [(2, 3, {}),
                          (2, 3, {2: INS_5}),
                          (2, 3, {3: SUB_C}),
                          (2, 3, {2: INS_5,
                                  3: SUB_C}),
                          (2, 3, {3: SUB_G}),
                          (2, 3, {2: INS_5,
                                  3: SUB_G}),
                          (2, 3, {3: SUB_T}),
                          (2, 3, {2: INS_5,
                                  3: SUB_T}),
                          (2, 3, {2: SUB_C}),
                          (2, 3, {2: SUB_C + INS_5}),
                          (2, 3, {2: SUB_C,
                                  3: SUB_C}),
                          (2, 3, {2: SUB_C + INS_5,
                                  3: SUB_C}),
                          (2, 3, {2: SUB_C,
                                  3: SUB_G}),
                          (2, 3, {2: SUB_C + INS_5,
                                  3: SUB_G}),
                          (2, 3, {2: SUB_C,
                                  3: SUB_T}),
                          (2, 3, {2: SUB_C + INS_5,
                                  3: SUB_T}),
                          (2, 3, {2: SUB_G}),
                          (2, 3, {2: SUB_G + INS_5}),
                          (2, 3, {2: SUB_G,
                                  3: SUB_C}),
                          (2, 3, {2: SUB_G + INS_5,
                                  3: SUB_C}),
                          (2, 3, {2: SUB_G,
                                  3: SUB_G}),
                          (2, 3, {2: SUB_G + INS_5,
                                  3: SUB_G}),
                          (2, 3, {2: SUB_G,
                                  3: SUB_T}),
                          (2, 3, {2: SUB_G + INS_5,
                                  3: SUB_T}),
                          (2, 3, {2: SUB_T}),
                          (2, 3, {2: SUB_T + INS_5}),
                          (2, 3, {2: SUB_T,
                                  3: SUB_C}),
                          (2, 3, {2: SUB_T + INS_5,
                                  3: SUB_C}),
                          (2, 3, {2: SUB_T,
                                  3: SUB_G}),
                          (2, 3, {2: SUB_T + INS_5,
                                  3: SUB_G}),
                          (2, 3, {2: SUB_T,
                                  3: SUB_T}),
                          (2, 3, {2: SUB_T + INS_5,
                                  3: SUB_T})])

    def test_xaax_insert3(self):
        """ Test that bases with no coverage are marked. """
        self.assertEqual(self.list_rels("TAAG", end5=2, end3=3, insert3=True),
                         [(2, 3, {}),
                          (2, 3, {3: INS_3}),
                          (2, 3, {3: SUB_C}),
                          (2, 3, {3: INS_3 + SUB_C}),
                          (2, 3, {3: SUB_G}),
                          (2, 3, {3: INS_3 + SUB_G}),
                          (2, 3, {3: SUB_T}),
                          (2, 3, {3: INS_3 + SUB_T}),
                          (2, 3, {2: SUB_C}),
                          (2, 3, {2: SUB_C,
                                  3: INS_3}),
                          (2, 3, {2: SUB_C,
                                  3: SUB_C}),
                          (2, 3, {2: SUB_C,
                                  3: INS_3 + SUB_C}),
                          (2, 3, {2: SUB_C,
                                  3: SUB_G}),
                          (2, 3, {2: SUB_C,
                                  3: INS_3 + SUB_G}),
                          (2, 3, {2: SUB_C,
                                  3: SUB_T}),
                          (2, 3, {2: SUB_C,
                                  3: INS_3 + SUB_T}),
                          (2, 3, {2: SUB_G}),
                          (2, 3, {2: SUB_G,
                                  3: INS_3}),
                          (2, 3, {2: SUB_G,
                                  3: SUB_C}),
                          (2, 3, {2: SUB_G,
                                  3: INS_3 + SUB_C}),
                          (2, 3, {2: SUB_G,
                                  3: SUB_G}),
                          (2, 3, {2: SUB_G,
                                  3: INS_3 + SUB_G}),
                          (2, 3, {2: SUB_G,
                                  3: SUB_T}),
                          (2, 3, {2: SUB_G,
                                  3: INS_3 + SUB_T}),
                          (2, 3, {2: SUB_T}),
                          (2, 3, {2: SUB_T,
                                  3: INS_3}),
                          (2, 3, {2: SUB_T,
                                  3: SUB_C}),
                          (2, 3, {2: SUB_T,
                                  3: INS_3 + SUB_C}),
                          (2, 3, {2: SUB_T,
                                  3: SUB_G}),
                          (2, 3, {2: SUB_T,
                                  3: INS_3 + SUB_G}),
                          (2, 3, {2: SUB_T,
                                  3: SUB_T}),
                          (2, 3, {2: SUB_T,
                                  3: INS_3 + SUB_T})])

    def test_agg_insert5(self):
        """ Test with sequence 'AGG'. """
        rels = self.list_rels("AGG", insert3=False)
        self.assertEqual(rels,
                         [(1, 3, {2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: INS_5,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {2: SUB_A + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_A}),
                          (1, 3, {2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: INS_5,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {2: SUB_A + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_C}),
                          (1, 3, {2: SUB_A}),
                          (1, 3, {1: INS_5,
                                  2: SUB_A}),
                          (1, 3, {2: SUB_A + INS_5}),
                          (1, 3, {1: INS_5,
                                  2: SUB_A + INS_5}),
                          (1, 3, {2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: INS_5,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {2: SUB_A + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_T}),
                          (1, 3, {2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: INS_5,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {2: SUB_C + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_A}),
                          (1, 3, {2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: INS_5,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {2: SUB_C + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_C}),
                          (1, 3, {2: SUB_C}),
                          (1, 3, {1: INS_5,
                                  2: SUB_C}),
                          (1, 3, {2: SUB_C + INS_5}),
                          (1, 3, {1: INS_5,
                                  2: SUB_C + INS_5}),
                          (1, 3, {2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: INS_5,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {2: SUB_C + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_T}),
                          (1, 3, {3: SUB_A}),
                          (1, 3, {1: INS_5,
                                  3: SUB_A}),
                          (1, 3, {2: INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: INS_5,
                                  2: INS_5,
                                  3: SUB_A}),
                          (1, 3, {3: SUB_C}),
                          (1, 3, {1: INS_5,
                                  3: SUB_C}),
                          (1, 3, {2: INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: INS_5,
                                  2: INS_5,
                                  3: SUB_C}),
                          (1, 3, {}),
                          (1, 3, {1: INS_5}),
                          (1, 3, {2: INS_5}),
                          (1, 3, {1: INS_5,
                                  2: INS_5}),
                          (1, 3, {3: SUB_T}),
                          (1, 3, {1: INS_5,
                                  3: SUB_T}),
                          (1, 3, {2: INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: INS_5,
                                  2: INS_5,
                                  3: SUB_T}),
                          (1, 3, {2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: INS_5,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {2: SUB_T + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_A}),
                          (1, 3, {2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: INS_5,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {2: SUB_T + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_C}),
                          (1, 3, {2: SUB_T}),
                          (1, 3, {1: INS_5,
                                  2: SUB_T}),
                          (1, 3, {2: SUB_T + INS_5}),
                          (1, 3, {1: INS_5,
                                  2: SUB_T + INS_5}),
                          (1, 3, {2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: INS_5,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {2: SUB_T + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_T}),
                          (1, 3, {2: DELET,
                                  3: SUB_A}),
                          (1, 3, {2: DELET,
                                  3: SUB_C}),
                          (1, 3, {2: DELET}),
                          (1, 3, {2: DELET,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A + INS_5}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_A + INS_5}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C + INS_5}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_C + INS_5}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5}),
                          (1, 3, {1: SUB_C,
                                  2: INS_5}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: INS_5}),
                          (1, 3, {1: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T + INS_5}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_T + INS_5}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C + INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: DELET,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: DELET,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: DELET}),
                          (1, 3, {1: SUB_C,
                                  2: DELET,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A + INS_5}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_A + INS_5}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C + INS_5}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_C + INS_5}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G}),
                          (1, 3, {1: SUB_G + INS_5}),
                          (1, 3, {1: SUB_G,
                                  2: INS_5}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: INS_5}),
                          (1, 3, {1: SUB_G,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T + INS_5}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_T + INS_5}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G + INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: DELET,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: DELET,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: DELET}),
                          (1, 3, {1: SUB_G,
                                  2: DELET,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A + INS_5}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_A + INS_5}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_A + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C + INS_5}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_C + INS_5}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_C + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5}),
                          (1, 3, {1: SUB_T,
                                  2: INS_5}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: INS_5}),
                          (1, 3, {1: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T + INS_5}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_T + INS_5}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T + INS_5,
                                  2: SUB_T + INS_5,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: DELET,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: DELET,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: DELET}),
                          (1, 3, {1: SUB_T,
                                  2: DELET,
                                  3: SUB_T})])

    def test_agg_insert3(self):
        """ Test with sequence 'AGG'. """
        rels = self.list_rels("AGG", insert3=True)
        self.assertEqual(rels,
                         [(1, 3, {2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {2: INS_3 + SUB_A,
                                  3: SUB_A}),
                          (1, 3, {2: SUB_A,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {2: INS_3 + SUB_A,
                                  3: SUB_C}),
                          (1, 3, {2: SUB_A,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {2: SUB_A}),
                          (1, 3, {2: INS_3 + SUB_A}),
                          (1, 3, {2: SUB_A,
                                  3: INS_3}),
                          (1, 3, {2: INS_3 + SUB_A,
                                  3: INS_3}),
                          (1, 3, {2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {2: INS_3 + SUB_A,
                                  3: SUB_T}),
                          (1, 3, {2: SUB_A,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {2: INS_3 + SUB_C,
                                  3: SUB_A}),
                          (1, 3, {2: SUB_C,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {2: INS_3 + SUB_C,
                                  3: SUB_C}),
                          (1, 3, {2: SUB_C,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {2: SUB_C}),
                          (1, 3, {2: INS_3 + SUB_C}),
                          (1, 3, {2: SUB_C,
                                  3: INS_3}),
                          (1, 3, {2: INS_3 + SUB_C,
                                  3: INS_3}),
                          (1, 3, {2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {2: INS_3 + SUB_C,
                                  3: SUB_T}),
                          (1, 3, {2: SUB_C,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {3: SUB_A}),
                          (1, 3, {2: INS_3,
                                  3: SUB_A}),
                          (1, 3, {3: INS_3 + SUB_A}),
                          (1, 3, {2: INS_3,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {3: SUB_C}),
                          (1, 3, {2: INS_3,
                                  3: SUB_C}),
                          (1, 3, {3: INS_3 + SUB_C}),
                          (1, 3, {2: INS_3,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {}),
                          (1, 3, {2: INS_3}),
                          (1, 3, {3: INS_3}),
                          (1, 3, {2: INS_3,
                                  3: INS_3}),
                          (1, 3, {3: SUB_T}),
                          (1, 3, {2: INS_3,
                                  3: SUB_T}),
                          (1, 3, {3: INS_3 + SUB_T}),
                          (1, 3, {2: INS_3,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {2: INS_3 + SUB_T,
                                  3: SUB_A}),
                          (1, 3, {2: SUB_T,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {2: INS_3 + SUB_T,
                                  3: SUB_C}),
                          (1, 3, {2: SUB_T,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {2: SUB_T}),
                          (1, 3, {2: INS_3 + SUB_T}),
                          (1, 3, {2: SUB_T,
                                  3: INS_3}),
                          (1, 3, {2: INS_3 + SUB_T,
                                  3: INS_3}),
                          (1, 3, {2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {2: INS_3 + SUB_T,
                                  3: SUB_T}),
                          (1, 3, {2: SUB_T,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {2: DELET,
                                  3: SUB_A}),
                          (1, 3, {2: DELET,
                                  3: SUB_C}),
                          (1, 3, {2: DELET}),
                          (1, 3, {2: DELET,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: INS_3}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_A,
                                  3: INS_3}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_A,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: INS_3}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_C,
                                  3: INS_3}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_C,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3}),
                          (1, 3, {1: SUB_C,
                                  3: INS_3}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3,
                                  3: INS_3}),
                          (1, 3, {1: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: INS_3}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_T,
                                  3: INS_3}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: SUB_T,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_C,
                                  2: DELET,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_C,
                                  2: DELET,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_C,
                                  2: DELET}),
                          (1, 3, {1: SUB_C,
                                  2: DELET,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: INS_3}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_A,
                                  3: INS_3}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_A,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: INS_3}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_C,
                                  3: INS_3}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_C,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_G,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_G,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_G}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3}),
                          (1, 3, {1: SUB_G,
                                  3: INS_3}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3,
                                  3: INS_3}),
                          (1, 3, {1: SUB_G,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: INS_3}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_T,
                                  3: INS_3}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: SUB_T,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_G,
                                  2: DELET,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_G,
                                  2: DELET,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_G,
                                  2: DELET}),
                          (1, 3, {1: SUB_G,
                                  2: DELET,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_A,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_A,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: INS_3}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_A,
                                  3: INS_3}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_A,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_A,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_A,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_C,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_C,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: INS_3}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_C,
                                  3: INS_3}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_C,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_C,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_C,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3}),
                          (1, 3, {1: SUB_T,
                                  3: INS_3}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3,
                                  3: INS_3}),
                          (1, 3, {1: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_T,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_T,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: INS_3}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_T,
                                  3: INS_3}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_T,
                                  3: SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: SUB_T,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: INS_3 + SUB_T,
                                  3: INS_3 + SUB_T}),
                          (1, 3, {1: SUB_T,
                                  2: DELET,
                                  3: SUB_A}),
                          (1, 3, {1: SUB_T,
                                  2: DELET,
                                  3: SUB_C}),
                          (1, 3, {1: SUB_T,
                                  2: DELET}),
                          (1, 3, {1: SUB_T,
                                  2: DELET,
                                  3: SUB_T})])


class TestIterRelvecsAll(ut.TestCase):

    def assert_equal(self, ref: DNA, expects: list, insert3: bool):
        """ Check that the expected and actual results match. """
        for exp, res in zip(chain(*expects),
                            iter_relvecs_all(ref, insert3),
                            strict=True):
            with self.subTest(exp=exp, res=res):
                self.assertTrue(np.all(exp == res))

    def test_length_1(self):
        """ Test with all length-1 DNA sequences. """
        for insert3 in [False, True]:
            for ref in expand_degenerate_seq(DNA("N")):
                expects = [
                    iter_relvecs_q53(ref, [], 1, 1, insert3),
                    iter_relvecs_q53(ref, [1], 1, 1, insert3),
                ]
                self.assert_equal(ref, expects, insert3)

    def test_length_2(self):
        """ Test with all length-2 DNA sequences. """
        for insert3 in [False, True]:
            for ref in expand_degenerate_seq(DNA("NN")):
                expects = [
                    iter_relvecs_q53(ref, [], 1, 1, insert3),
                    iter_relvecs_q53(ref, [1], 1, 1, insert3),
                    iter_relvecs_q53(ref, [], 1, 2, insert3),
                    iter_relvecs_q53(ref, [1], 1, 2, insert3),
                    iter_relvecs_q53(ref, [2], 1, 2, insert3),
                    iter_relvecs_q53(ref, [1, 2], 1, 2, insert3),
                    iter_relvecs_q53(ref, [], 2, 2, insert3),
                    iter_relvecs_q53(ref, [2], 2, 2, insert3),
                ]
                self.assert_equal(ref, expects, insert3)

    def test_length_3(self):
        """ Test with all length-3 DNA sequences. """
        for insert3 in [False, True]:
            for ref in expand_degenerate_seq(DNA("NNN")):
                expects = [
                    iter_relvecs_q53(ref, [], 1, 1, insert3),
                    iter_relvecs_q53(ref, [1], 1, 1, insert3),
                    iter_relvecs_q53(ref, [], 1, 2, insert3),
                    iter_relvecs_q53(ref, [1], 1, 2, insert3),
                    iter_relvecs_q53(ref, [2], 1, 2, insert3),
                    iter_relvecs_q53(ref, [1, 2], 1, 2, insert3),
                    iter_relvecs_q53(ref, [], 1, 3, insert3),
                    iter_relvecs_q53(ref, [1], 1, 3, insert3),
                    iter_relvecs_q53(ref, [2], 1, 3, insert3),
                    iter_relvecs_q53(ref, [3], 1, 3, insert3),
                    iter_relvecs_q53(ref, [1, 2], 1, 3, insert3),
                    iter_relvecs_q53(ref, [1, 3], 1, 3, insert3),
                    iter_relvecs_q53(ref, [2, 3], 1, 3, insert3),
                    iter_relvecs_q53(ref, [1, 2, 3], 1, 3, insert3),
                    iter_relvecs_q53(ref, [], 2, 2, insert3),
                    iter_relvecs_q53(ref, [2], 2, 2, insert3),
                    iter_relvecs_q53(ref, [], 2, 3, insert3),
                    iter_relvecs_q53(ref, [2], 2, 3, insert3),
                    iter_relvecs_q53(ref, [3], 2, 3, insert3),
                    iter_relvecs_q53(ref, [2, 3], 2, 3, insert3),
                    iter_relvecs_q53(ref, [], 3, 3, insert3),
                    iter_relvecs_q53(ref, [3], 3, 3, insert3),
                ]
                self.assert_equal(ref, expects, insert3)


if __name__ == "__main__":
    ut.main(verbosity=2)
