"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeDependable = exports.Grant = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const policy_statement_1 = require("./policy-statement");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return props.policyDependable ? cdk.DependableTrait.get(props.policyDependable).dependencyRoots : [];
            },
        });
    }
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        var _c;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_GrantWithResourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipalOrResource);
            }
            throw error;
        }
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const resourceAndPrincipalAccountComparison = options.grantee.grantPrincipal.principalAccount
            ? cdk.Token.compareStrings(options.resource.env.account, options.grantee.grantPrincipal.principalAccount)
            : undefined;
        // if both accounts are tokens, we assume here they are the same
        const equalOrBothUnresolved = resourceAndPrincipalAccountComparison === cdk.TokenComparison.SAME
            || resourceAndPrincipalAccountComparison == cdk.TokenComparison.BOTH_UNRESOLVED;
        const sameAccount = resourceAndPrincipalAccountComparison
            ? equalOrBothUnresolved
            // if the principal doesn't have an account (for example, a service principal),
            // we should modify the resource's trust policy
            : false;
        // If we added to the principal AND we're in the same account, then we're done.
        // If not, it's a different account and we must also add a trust policy on the resource.
        if (result.success && sameAccount) {
            return result;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        return new Grant({
            resourceStatement: statement,
            options,
            policyDependable: resourceResult.statementAdded ? (_c = resourceResult.policyDependable) !== null && _c !== void 0 ? _c : options.resource : undefined,
        });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_GrantOnPrincipalOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipal);
            }
            throw error;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: options.resourceArns,
        });
        const addedToPrincipal = options.grantee.grantPrincipal.addToPrincipalPolicy(statement);
        if (!addedToPrincipal.statementAdded) {
            return new Grant({ principalStatement: undefined, options });
        }
        if (!addedToPrincipal.policyDependable) {
            throw new Error('Contract violation: when Principal returns statementAdded=true, it should return a dependable');
        }
        return new Grant({ principalStatement: statement, options, policyDependable: addedToPrincipal.policyDependable });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the principal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        var _c;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_GrantOnPrincipalAndResourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipalAndResource);
            }
            throw error;
        }
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.resourcePolicyPrincipal || options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        const resourceDependable = resourceResult.statementAdded ? (_c = resourceResult.policyDependable) !== null && _c !== void 0 ? _c : options.resource : undefined;
        return new Grant({
            principalStatement: statement,
            resourceStatement: result.resourceStatement,
            options,
            policyDependable: resourceDependable ? new CompositeDependable(result, resourceDependable) : result,
        });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.drop);
            }
            throw error;
        }
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] },
        });
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // eslint-disable-next-line max-len
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
    /**
     * Make sure this grant is applied before the given constructs are deployed
     *
     * The same as construct.node.addDependency(grant), but slightly nicer to read.
     */
    applyBefore(...constructs) {
        for (const construct of constructs) {
            construct.node.addDependency(this);
        }
    }
}
exports.Grant = Grant;
_a = JSII_RTTI_SYMBOL_1;
Grant[_a] = { fqn: "@aws-cdk/aws-iam.Grant", version: "1.158.0" };
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
/**
 * Composite dependable
 *
 * Not as simple as eagerly getting the dependency roots from the
 * inner dependables, as they may be mutable so we need to defer
 * the query.
 */
class CompositeDependable {
    constructor(...dependables) {
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return Array.prototype.concat.apply([], dependables.map(d => cdk.DependableTrait.get(d).dependencyRoots));
            },
        });
    }
}
exports.CompositeDependable = CompositeDependable;
_b = JSII_RTTI_SYMBOL_1;
CompositeDependable[_b] = { fqn: "@aws-cdk/aws-iam.CompositeDependable", version: "1.158.0" };
//# sourceMappingURL=data:application/json;base64,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