"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.deriveEstimateSizeOptions = exports.Effect = exports.PolicyStatement = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const group_1 = require("./group");
const principals_1 = require("./principals");
const postprocess_policy_document_1 = require("./private/postprocess-policy-document");
const util_1 = require("./util");
const ensureArrayOrUndefined = (field) => {
    if (field === undefined) {
        return undefined;
    }
    if (typeof (field) !== 'string' && !Array.isArray(field)) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    if (Array.isArray(field) && !!field.find((f) => typeof (f) !== 'string')) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    return Array.isArray(field) ? field : [field];
};
/**
 * An estimate on how long ARNs typically are
 *
 * This is used to decide when to start splitting statements into new Managed Policies.
 * Because we often can't know the length of an ARN (it may be a token and only
 * available at deployment time) we'll have to estimate it.
 *
 * The estimate can be overridden by setting the `@aws-cdk/aws-iam.arnSizeEstimate` context key.
 */
const DEFAULT_ARN_SIZE_ESTIMATE = 150;
/**
 * Context key which can be used to override the estimated length of unresolved ARNs.
 */
const ARN_SIZE_ESTIMATE_CONTEXT_KEY = '@aws-cdk/aws-iam.arnSizeEstimate';
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement {
    constructor(props = {}) {
        this._action = new Array();
        this._notAction = new Array();
        this._principal = {};
        this._notPrincipal = {};
        this._resource = new Array();
        this._notResource = new Array();
        this._condition = {};
        // Hold on to those principals
        this._principals = new Array();
        this._notPrincipals = new Array();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_PolicyStatementProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        // Validate actions
        for (const action of [...props.actions || [], ...props.notActions || []]) {
            if (!/^(\*|[a-zA-Z0-9-]+:[a-zA-Z0-9*]+)$/.test(action) && !cdk.Token.isUnresolved(action)) {
                throw new Error(`Action '${action}' is invalid. An action string consists of a service namespace, a colon, and the name of an action. Action names can include wildcards.`);
            }
        }
        this.sid = props.sid;
        this.effect = props.effect || Effect.ALLOW;
        this.addActions(...props.actions || []);
        this.addNotActions(...props.notActions || []);
        this.addPrincipals(...props.principals || []);
        this.addNotPrincipals(...props.notPrincipals || []);
        this.addResources(...props.resources || []);
        this.addNotResources(...props.notResources || []);
        if (props.conditions !== undefined) {
            this.addConditions(props.conditions);
        }
    }
    /**
     * Creates a new PolicyStatement based on the object provided.
     * This will accept an object created from the `.toJSON()` call
     * @param obj the PolicyStatement in object form.
     */
    static fromJson(obj) {
        const ret = new PolicyStatement({
            sid: obj.Sid,
            actions: ensureArrayOrUndefined(obj.Action),
            resources: ensureArrayOrUndefined(obj.Resource),
            conditions: obj.Condition,
            effect: obj.Effect,
            notActions: ensureArrayOrUndefined(obj.NotAction),
            notResources: ensureArrayOrUndefined(obj.NotResource),
            principals: obj.Principal ? [new JsonPrincipal(obj.Principal)] : undefined,
            notPrincipals: obj.NotPrincipal ? [new JsonPrincipal(obj.NotPrincipal)] : undefined,
        });
        // validate that the PolicyStatement has the correct shape
        const errors = ret.validateForAnyPolicy();
        if (errors.length > 0) {
            throw new Error('Incorrect Policy Statement: ' + errors.join('\n'));
        }
        return ret;
    }
    //
    // Actions
    //
    /**
     * Specify allowed actions into the "Action" section of the policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html
     *
     * @param actions actions that will be allowed.
     */
    addActions(...actions) {
        if (actions.length > 0 && this._notAction.length > 0) {
            throw new Error('Cannot add \'Actions\' to policy statement if \'NotActions\' have been added');
        }
        this._action.push(...actions);
    }
    /**
     * Explicitly allow all actions except the specified list of actions into the "NotAction" section
     * of the policy document.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html
     *
     * @param notActions actions that will be denied. All other actions will be permitted.
     */
    addNotActions(...notActions) {
        if (notActions.length > 0 && this._action.length > 0) {
            throw new Error('Cannot add \'NotActions\' to policy statement if \'Actions\' have been added');
        }
        this._notAction.push(...notActions);
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return this._principals.length + this._notPrincipals.length > 0;
    }
    /**
     * Adds principals to the "Principal" section of a policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
     *
     * @param principals IAM principals that will be added
     */
    addPrincipals(...principals) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_IPrincipal(principals);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPrincipals);
            }
            throw error;
        }
        this._principals.push(...principals);
        if (Object.keys(principals).length > 0 && Object.keys(this._notPrincipal).length > 0) {
            throw new Error('Cannot add \'Principals\' to policy statement if \'NotPrincipals\' have been added');
        }
        for (const principal of principals) {
            this.validatePolicyPrincipal(principal);
            const fragment = principal.policyFragment;
            util_1.mergePrincipal(this._principal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    /**
     * Specify principals that is not allowed or denied access to the "NotPrincipal" section of
     * a policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html
     *
     * @param notPrincipals IAM principals that will be denied access
     */
    addNotPrincipals(...notPrincipals) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_IPrincipal(notPrincipals);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addNotPrincipals);
            }
            throw error;
        }
        this._notPrincipals.push(...notPrincipals);
        if (Object.keys(notPrincipals).length > 0 && Object.keys(this._principal).length > 0) {
            throw new Error('Cannot add \'NotPrincipals\' to policy statement if \'Principals\' have been added');
        }
        for (const notPrincipal of notPrincipals) {
            this.validatePolicyPrincipal(notPrincipal);
            const fragment = notPrincipal.policyFragment;
            util_1.mergePrincipal(this._notPrincipal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    validatePolicyPrincipal(principal) {
        if (principal instanceof group_1.Group) {
            throw new Error('Cannot use an IAM Group as the \'Principal\' or \'NotPrincipal\' in an IAM Policy');
        }
    }
    /**
     * Specify AWS account ID as the principal entity to the "Principal" section of a policy statement.
     */
    addAwsAccountPrincipal(accountId) {
        this.addPrincipals(new principals_1.AccountPrincipal(accountId));
    }
    /**
     * Specify a principal using the ARN  identifier of the principal.
     * You cannot specify IAM groups and instance profiles as principals.
     *
     * @param arn ARN identifier of AWS account, IAM user, or IAM role (i.e. arn:aws:iam::123456789012:user/user-name)
     */
    addArnPrincipal(arn) {
        this.addPrincipals(new principals_1.ArnPrincipal(arn));
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service, opts) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_ServicePrincipalOpts(opts);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addServicePrincipal);
            }
            throw error;
        }
        this.addPrincipals(new principals_1.ServicePrincipal(service, opts));
    }
    /**
     * Adds a federated identity provider such as Amazon Cognito to this policy statement.
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com')
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    addFederatedPrincipal(federated, conditions) {
        this.addPrincipals(new principals_1.FederatedPrincipal(federated, conditions));
    }
    /**
     * Adds an AWS account root user principal to this policy statement
     */
    addAccountRootPrincipal() {
        this.addPrincipals(new principals_1.AccountRootPrincipal());
    }
    /**
     * Adds a canonical user ID principal to this policy document
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account
     */
    addCanonicalUserPrincipal(canonicalUserId) {
        this.addPrincipals(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    /**
     * Adds all identities in all accounts ("*") to this policy statement
     */
    addAnyPrincipal() {
        this.addPrincipals(new principals_1.AnyPrincipal());
    }
    //
    // Resources
    //
    /**
     * Specify resources that this policy statement applies into the "Resource" section of
     * this policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_resource.html
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement applies to
     */
    addResources(...arns) {
        if (arns.length > 0 && this._notResource.length > 0) {
            throw new Error('Cannot add \'Resources\' to policy statement if \'NotResources\' have been added');
        }
        this._resource.push(...arns);
    }
    /**
     * Specify resources that this policy statement will not apply to in the "NotResource" section
     * of this policy statement. All resources except the specified list will be matched.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement does not apply to
     */
    addNotResources(...arns) {
        if (arns.length > 0 && this._resource.length > 0) {
            throw new Error('Cannot add \'NotResources\' to policy statement if \'Resources\' have been added');
        }
        this._notResource.push(...arns);
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        this.addResources('*');
    }
    /**
     * Indicates if this permission has at least one resource associated with it.
     */
    get hasResource() {
        return this._resource && this._resource.length > 0;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     *
     * If multiple calls are made to add a condition with the same operator and field, only
     * the last one wins. For example:
     *
     * ```ts
     * declare const stmt: iam.PolicyStatement;
     *
     * stmt.addCondition('StringEquals', { 'aws:SomeField': '1' });
     * stmt.addCondition('StringEquals', { 'aws:SomeField': '2' });
     * ```
     *
     * Will end up with the single condition `StringEquals: { 'aws:SomeField': '2' }`.
     *
     * If you meant to add a condition to say that the field can be *either* `1` or `2`, write
     * this:
     *
     * ```ts
     * declare const stmt: iam.PolicyStatement;
     *
     * stmt.addCondition('StringEquals', { 'aws:SomeField': ['1', '2'] });
     * ```
     */
    addCondition(key, value) {
        const existingValue = this._condition[key];
        this._condition[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Add multiple conditions to the Policy
     *
     * See the `addCondition` function for a caveat on calling this method multiple times.
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
    }
    /**
     * Add a condition that limits to a given account
     *
     * This method can only be called once: subsequent calls will overwrite earlier calls.
     */
    addAccountCondition(accountId) {
        this.addCondition('StringEquals', { 'sts:ExternalId': accountId });
    }
    /**
     * Create a new `PolicyStatement` with the same exact properties
     * as this one, except for the overrides
     */
    copy(overrides = {}) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_PolicyStatementProps(overrides);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.copy);
            }
            throw error;
        }
        return new PolicyStatement({
            sid: (_b = overrides.sid) !== null && _b !== void 0 ? _b : this.sid,
            effect: (_c = overrides.effect) !== null && _c !== void 0 ? _c : this.effect,
            actions: (_d = overrides.actions) !== null && _d !== void 0 ? _d : this.actions,
            notActions: (_e = overrides.notActions) !== null && _e !== void 0 ? _e : this.notActions,
            principals: (_f = overrides.principals) !== null && _f !== void 0 ? _f : this.principals,
            notPrincipals: (_g = overrides.notPrincipals) !== null && _g !== void 0 ? _g : this.notPrincipals,
            resources: (_h = overrides.resources) !== null && _h !== void 0 ? _h : this.resources,
            notResources: (_j = overrides.notResources) !== null && _j !== void 0 ? _j : this.notResources,
            conditions: (_k = overrides.conditions) !== null && _k !== void 0 ? _k : this.conditions,
        });
    }
    /**
     * JSON-ify the policy statement
     *
     * Used when JSON.stringify() is called
     */
    toStatementJson() {
        return postprocess_policy_document_1.normalizeStatement({
            Action: this._action,
            NotAction: this._notAction,
            Condition: this._condition,
            Effect: this.effect,
            Principal: this._principal,
            NotPrincipal: this._notPrincipal,
            Resource: this._resource,
            NotResource: this._notResource,
            Sid: this.sid,
        });
    }
    /**
     * String representation of this policy statement
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyStatement',
        });
    }
    /**
     * JSON-ify the statement
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.toStatementJson();
    }
    /**
     * Add a principal's conditions
     *
     * For convenience, principals have been modeled as both a principal
     * and a set of conditions. This makes it possible to have a single
     * object represent e.g. an "SNS Topic" (SNS service principal + aws:SourcArn
     * condition) or an Organization member (* + aws:OrgId condition).
     *
     * However, when using multiple principals in the same policy statement,
     * they must all have the same conditions or the OR samentics
     * implied by a list of principals cannot be guaranteed (user needs to
     * add multiple statements in that case).
     */
    addPrincipalConditions(conditions) {
        // Stringifying the conditions is an easy way to do deep equality
        const theseConditions = JSON.stringify(conditions);
        if (this.principalConditionsJson === undefined) {
            // First principal, anything goes
            this.principalConditionsJson = theseConditions;
        }
        else {
            if (this.principalConditionsJson !== theseConditions) {
                throw new Error(`All principals in a PolicyStatement must have the same Conditions (got '${this.principalConditionsJson}' and '${theseConditions}'). Use multiple statements instead.`);
            }
        }
        this.addConditions(conditions);
    }
    /**
     * Validate that the policy statement satisfies base requirements for a policy.
     *
     * @returns An array of validation error messages, or an empty array if the statement is valid.
     */
    validateForAnyPolicy() {
        const errors = new Array();
        if (this._action.length === 0 && this._notAction.length === 0) {
            errors.push('A PolicyStatement must specify at least one \'action\' or \'notAction\'.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for a resource-based policy.
     *
     * @returns An array of validation error messages, or an empty array if the statement is valid.
     */
    validateForResourcePolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this._principal).length === 0 && Object.keys(this._notPrincipal).length === 0) {
            errors.push('A PolicyStatement used in a resource-based policy must specify at least one IAM principal.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for an identity-based policy.
     *
     * @returns An array of validation error messages, or an empty array if the statement is valid.
     */
    validateForIdentityPolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this._principal).length > 0 || Object.keys(this._notPrincipal).length > 0) {
            errors.push('A PolicyStatement used in an identity-based policy cannot specify any IAM principals.');
        }
        if (Object.keys(this._resource).length === 0 && Object.keys(this._notResource).length === 0) {
            errors.push('A PolicyStatement used in an identity-based policy must specify at least one resource.');
        }
        return errors;
    }
    /**
     * The Actions added to this statement
     */
    get actions() {
        return [...this._action];
    }
    /**
     * The NotActions added to this statement
     */
    get notActions() {
        return [...this._notAction];
    }
    /**
     * The Principals added to this statement
     */
    get principals() {
        return [...this._principals];
    }
    /**
     * The NotPrincipals added to this statement
     */
    get notPrincipals() {
        return [...this._notPrincipals];
    }
    /**
     * The Resources added to this statement
     */
    get resources() {
        return [...this._resource];
    }
    /**
     * The NotResources added to this statement
     */
    get notResources() {
        return [...this._notResource];
    }
    /**
     * The conditions added to this statement
     */
    get conditions() {
        return { ...this._condition };
    }
    /**
     * Estimate the size of this policy statement
     *
     * By necessity, this will not be accurate. We'll do our best to overestimate
     * so we won't have nasty surprises.
     *
     * @internal
     */
    _estimateSize(options) {
        let ret = 0;
        const { actionEstimate, arnEstimate } = options;
        ret += `"Effect": "${this.effect}",`.length;
        count('Action', this.actions, actionEstimate);
        count('NotAction', this.notActions, actionEstimate);
        count('Resource', this.resources, arnEstimate);
        count('NotResource', this.notResources, arnEstimate);
        ret += this.principals.length * arnEstimate;
        ret += this.notPrincipals.length * arnEstimate;
        ret += JSON.stringify(this.conditions).length;
        return ret;
        function count(key, values, tokenSize) {
            if (values.length > 0) {
                ret += key.length + 5 /* quotes, colon, brackets */ +
                    util_1.sum(values.map(v => (cdk.Token.isUnresolved(v) ? tokenSize : v.length) + 3 /* quotes, separator */));
            }
        }
    }
}
exports.PolicyStatement = PolicyStatement;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatement[_a] = { fqn: "@aws-cdk/aws-iam.PolicyStatement", version: "1.158.0" };
/**
 * The Effect element of an IAM policy
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_effect.html
 */
var Effect;
(function (Effect) {
    /**
     * Allows access to a resource in an IAM policy statement. By default, access to resources are denied.
     */
    Effect["ALLOW"] = "Allow";
    /**
     * Explicitly deny access to a resource. By default, all requests are denied implicitly.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html
     */
    Effect["DENY"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
class JsonPrincipal extends principals_1.PrincipalBase {
    constructor(json = {}) {
        super();
        // special case: if principal is a string, turn it into a "LiteralString" principal,
        // so we render the exact same string back out.
        if (typeof (json) === 'string') {
            json = { [util_1.LITERAL_STRING_KEY]: [json] };
        }
        if (typeof (json) !== 'object') {
            throw new Error(`JSON IAM principal should be an object, got ${JSON.stringify(json)}`);
        }
        this.policyFragment = {
            principalJson: json,
            conditions: {},
        };
    }
    dedupeString() {
        return JSON.stringify(this.policyFragment);
    }
}
/**
 * Derive the size estimation options from context
 *
 * @internal
 */
function deriveEstimateSizeOptions(scope) {
    var _b;
    const actionEstimate = 20;
    const arnEstimate = (_b = scope.node.tryGetContext(ARN_SIZE_ESTIMATE_CONTEXT_KEY)) !== null && _b !== void 0 ? _b : DEFAULT_ARN_SIZE_ESTIMATE;
    if (typeof arnEstimate !== 'number') {
        throw new Error(`Context value ${ARN_SIZE_ESTIMATE_CONTEXT_KEY} should be a number, got ${JSON.stringify(arnEstimate)}`);
    }
    return { actionEstimate, arnEstimate };
}
exports.deriveEstimateSizeOptions = deriveEstimateSizeOptions;
//# sourceMappingURL=data:application/json;base64,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