"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Role = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const grant_1 = require("./grant");
const iam_generated_1 = require("./iam.generated");
const managed_policy_1 = require("./managed-policy");
const policy_1 = require("./policy");
const policy_document_1 = require("./policy-document");
const principals_1 = require("./principals");
const assume_role_policy_1 = require("./private/assume-role-policy");
const immutable_role_1 = require("./private/immutable-role");
const policydoc_adapter_1 = require("./private/policydoc-adapter");
const util_1 = require("./util");
const MAX_INLINE_SIZE = 10000;
const MAX_MANAGEDPOL_SIZE = 6000;
/**
 * IAM Role
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 */
class Role extends core_1.Resource {
    constructor(scope, id, props) {
        var _b;
        super(scope, id, {
            physicalName: props.roleName,
        });
        this.grantPrincipal = this;
        this.principalAccount = this.env.account;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.managedPolicies = [];
        this.attachedPolicies = new util_1.AttachedPolicies();
        this.dependables = new Map();
        this._didSplit = false;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_RoleProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const externalIds = props.externalIds || [];
        if (props.externalId) {
            externalIds.push(props.externalId);
        }
        this.assumeRolePolicy = createAssumeRolePolicy(props.assumedBy, externalIds);
        this.managedPolicies.push(...props.managedPolicies || []);
        this.inlinePolicies = props.inlinePolicies || {};
        this.permissionsBoundary = props.permissionsBoundary;
        const maxSessionDuration = props.maxSessionDuration && props.maxSessionDuration.toSeconds();
        validateMaxSessionDuration(maxSessionDuration);
        const description = (props.description && ((_b = props.description) === null || _b === void 0 ? void 0 : _b.length) > 0) ? props.description : undefined;
        if (description && description.length > 1000) {
            throw new Error('Role description must be no longer than 1000 characters.');
        }
        validateRolePath(props.path);
        const role = new iam_generated_1.CfnRole(this, 'Resource', {
            assumeRolePolicyDocument: this.assumeRolePolicy,
            managedPolicyArns: util_1.UniqueStringSet.from(() => this.managedPolicies.map(p => p.managedPolicyArn)),
            policies: _flatten(this.inlinePolicies),
            path: props.path,
            permissionsBoundary: this.permissionsBoundary ? this.permissionsBoundary.managedPolicyArn : undefined,
            roleName: this.physicalName,
            maxSessionDuration,
            description,
        });
        this.roleId = role.attrRoleId;
        this.roleArn = this.getResourceArnAttribute(role.attrArn, {
            region: '',
            service: 'iam',
            resource: 'role',
            // Removes leading slash from path
            resourceName: `${props.path ? props.path.substr(props.path.charAt(0) === '/' ? 1 : 0) : ''}${this.physicalName}`,
        });
        this.roleName = this.getResourceNameAttribute(role.ref);
        this.policyFragment = new principals_1.ArnPrincipal(this.roleArn).policyFragment;
        function _flatten(policies) {
            if (policies == null || Object.keys(policies).length === 0) {
                return undefined;
            }
            const result = new Array();
            for (const policyName of Object.keys(policies)) {
                const policyDocument = policies[policyName];
                result.push({ policyName, policyDocument });
            }
            return result;
        }
        core_1.Aspects.of(this).add({
            visit: (c) => {
                if (c === this) {
                    this.splitLargePolicy();
                }
            },
        });
    }
    /**
     * Import an external role by ARN.
     *
     * If the imported Role ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * role has a `path` (like `arn:...:role/AdminRoles/Alice`), the
     * `roleName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Role ARN should be supplied without the
     * `path` in order to resolve the correct role resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param roleArn the ARN of the role to import
     * @param options allow customizing the behavior of the returned role
     */
    static fromRoleArn(scope, id, roleArn, options = {}) {
        var _b;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_FromRoleArnOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromRoleArn);
            }
            throw error;
        }
        const scopeStack = core_1.Stack.of(scope);
        const parsedArn = scopeStack.splitArn(roleArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const resourceName = parsedArn.resourceName;
        const roleAccount = parsedArn.account;
        // service roles have an ARN like 'arn:aws:iam::<account>:role/service-role/<roleName>'
        // or 'arn:aws:iam::<account>:role/service-role/servicename.amazonaws.com/service-role/<roleName>'
        // we want to support these as well, so we just use the element after the last slash as role name
        const roleName = resourceName.split('/').pop();
        class Import extends core_1.Resource {
            constructor(_scope, _id) {
                super(_scope, _id, {
                    account: roleAccount,
                });
                this.grantPrincipal = this;
                this.principalAccount = roleAccount;
                this.assumeRoleAction = 'sts:AssumeRole';
                this.policyFragment = new principals_1.ArnPrincipal(roleArn).policyFragment;
                this.roleArn = roleArn;
                this.roleName = roleName;
                this.attachedPolicies = new util_1.AttachedPolicies();
            }
            addToPolicy(statement) {
                return this.addToPrincipalPolicy(statement).statementAdded;
            }
            addToPrincipalPolicy(statement) {
                if (!this.defaultPolicy) {
                    this.defaultPolicy = new policy_1.Policy(this, 'Policy');
                    this.attachInlinePolicy(this.defaultPolicy);
                }
                this.defaultPolicy.addStatements(statement);
                return { statementAdded: true, policyDependable: this.defaultPolicy };
            }
            attachInlinePolicy(policy) {
                const thisAndPolicyAccountComparison = core_1.Token.compareStrings(this.env.account, policy.env.account);
                const equalOrAnyUnresolved = thisAndPolicyAccountComparison === core_1.TokenComparison.SAME ||
                    thisAndPolicyAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
                    thisAndPolicyAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
                if (equalOrAnyUnresolved) {
                    this.attachedPolicies.attach(policy);
                    policy.attachToRole(this);
                }
            }
            addManagedPolicy(_policy) {
                // FIXME: Add warning that we're ignoring this
            }
            /**
             * Grant permissions to the given principal to pass this role.
             */
            grantPassRole(identity) {
                return this.grant(identity, 'iam:PassRole');
            }
            /**
             * Grant permissions to the given principal to pass this role.
             */
            grantAssumeRole(identity) {
                return this.grant(identity, 'sts:AssumeRole');
            }
            /**
             * Grant the actions defined in actions to the identity Principal on this resource.
             */
            grant(grantee, ...actions) {
                return grant_1.Grant.addToPrincipal({
                    grantee,
                    actions,
                    resourceArns: [this.roleArn],
                    scope: this,
                });
            }
            dedupeString() {
                return `ImportedRole:${roleArn}`;
            }
        }
        if (options.addGrantsToResources !== undefined && options.mutable !== false) {
            throw new Error('\'addGrantsToResources\' can only be passed if \'mutable: false\'');
        }
        const importedRole = new Import(scope, id);
        const roleArnAndScopeStackAccountComparison = core_1.Token.compareStrings(importedRole.env.account, scopeStack.account);
        const equalOrAnyUnresolved = roleArnAndScopeStackAccountComparison === core_1.TokenComparison.SAME ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
        // we only return an immutable Role if both accounts were explicitly provided, and different
        return options.mutable !== false && equalOrAnyUnresolved
            ? importedRole
            : new immutable_role_1.ImmutableRole(scope, `ImmutableRole${id}`, importedRole, (_b = options.addGrantsToResources) !== null && _b !== void 0 ? _b : false);
    }
    /**
     * Import an external role by name.
     *
     * The imported role is assumed to exist in the same account as the account
     * the scope's containing Stack is being deployed to.
     */
    static fromRoleName(scope, id, roleName) {
        return Role.fromRoleArn(scope, id, core_1.Stack.of(scope).formatArn({
            region: '',
            service: 'iam',
            resource: 'role',
            resourceName: roleName,
        }));
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPrincipalPolicy(statement) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipalPolicy);
            }
            throw error;
        }
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.attachInlinePolicy(this.defaultPolicy);
        }
        this.defaultPolicy.addStatements(statement);
        // We might split this statement off into a different policy, so we'll need to
        // late-bind the dependable.
        const policyDependable = new core_1.ConcreteDependable();
        this.dependables.set(statement, policyDependable);
        return { statementAdded: true, policyDependable };
    }
    addToPolicy(statement) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPolicy);
            }
            throw error;
        }
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The the managed policy to attach.
     */
    addManagedPolicy(policy) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_IManagedPolicy(policy);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addManagedPolicy);
            }
            throw error;
        }
        if (this.managedPolicies.find(mp => mp === policy)) {
            return;
        }
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_Policy(policy);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attachInlinePolicy);
            }
            throw error;
        }
        this.attachedPolicies.attach(policy);
        policy.attachToRole(this);
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(grantee, ...actions) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_IPrincipal(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grant);
            }
            throw error;
        }
        return grant_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.roleArn],
            scope: this,
        });
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_IPrincipal(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPassRole);
            }
            throw error;
        }
        return this.grant(identity, 'iam:PassRole');
    }
    /**
     * Grant permissions to the given principal to assume this role.
     */
    grantAssumeRole(identity) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_IPrincipal(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantAssumeRole);
            }
            throw error;
        }
        return this.grant(identity, 'sts:AssumeRole');
    }
    /**
     * Return a copy of this Role object whose Policies will not be updated
     *
     * Use the object returned by this method if you want this Role to be used by
     * a construct without it automatically updating the Role's Policies.
     *
     * If you do, you are responsible for adding the correct statements to the
     * Role's policies yourself.
     */
    withoutPolicyUpdates(options = {}) {
        var _b;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_WithoutPolicyUpdatesOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.withoutPolicyUpdates);
            }
            throw error;
        }
        if (!this.immutableRole) {
            this.immutableRole = new immutable_role_1.ImmutableRole(constructs_1.Node.of(this).scope, `ImmutableRole${this.node.id}`, this, (_b = options.addGrantsToResources) !== null && _b !== void 0 ? _b : false);
        }
        return this.immutableRole;
    }
    validate() {
        var _b;
        const errors = super.validate();
        errors.push(...((_b = this.assumeRolePolicy) === null || _b === void 0 ? void 0 : _b.validateForResourcePolicy()) || []);
        for (const policy of Object.values(this.inlinePolicies)) {
            errors.push(...policy.validateForIdentityPolicy());
        }
        return errors;
    }
    /**
     * Split large inline policies into managed policies
     *
     * This gets around the 10k bytes limit on role policies.
     */
    splitLargePolicy() {
        if (!this.defaultPolicy || this._didSplit) {
            return;
        }
        this._didSplit = true;
        const self = this;
        const originalDoc = this.defaultPolicy.document;
        const splitOffDocs = originalDoc._splitDocument(this, MAX_INLINE_SIZE, MAX_MANAGEDPOL_SIZE);
        // Includes the "current" document
        const mpCount = this.managedPolicies.length + (splitOffDocs.size - 1);
        if (mpCount > 20) {
            core_1.Annotations.of(this).addWarning(`Policy too large: ${mpCount} exceeds the maximum of 20 managed policies attached to a Role`);
        }
        else if (mpCount > 10) {
            core_1.Annotations.of(this).addWarning(`Policy large: ${mpCount} exceeds 10 managed policies attached to a Role, this requires a quota increase`);
        }
        // Create the managed policies and fix up the dependencies
        markDeclaringConstruct(originalDoc, this.defaultPolicy);
        let i = 1;
        for (const newDoc of splitOffDocs.keys()) {
            if (newDoc === originalDoc) {
                continue;
            }
            const mp = new managed_policy_1.ManagedPolicy(this, `OverflowPolicy${i++}`, {
                description: `Part of the policies for ${this.node.path}`,
                document: newDoc,
                roles: [this],
            });
            markDeclaringConstruct(newDoc, mp);
        }
        /**
         * Update the Dependables for the statements in the given PolicyDocument to point to the actual declaring construct
         */
        function markDeclaringConstruct(doc, declaringConstruct) {
            var _b, _c;
            for (const original of (_b = splitOffDocs.get(doc)) !== null && _b !== void 0 ? _b : []) {
                (_c = self.dependables.get(original)) === null || _c === void 0 ? void 0 : _c.add(declaringConstruct);
            }
        }
    }
}
exports.Role = Role;
_a = JSII_RTTI_SYMBOL_1;
Role[_a] = { fqn: "@aws-cdk/aws-iam.Role", version: "1.158.0" };
function createAssumeRolePolicy(principal, externalIds) {
    const actualDoc = new policy_document_1.PolicyDocument();
    // If requested, add externalIds to every statement added to this doc
    const addDoc = externalIds.length === 0
        ? actualDoc
        : new policydoc_adapter_1.MutatingPolicyDocumentAdapter(actualDoc, (statement) => {
            statement.addCondition('StringEquals', {
                'sts:ExternalId': externalIds.length === 1 ? externalIds[0] : externalIds,
            });
            return statement;
        });
    assume_role_policy_1.defaultAddPrincipalToAssumeRole(principal, addDoc);
    return actualDoc;
}
function validateRolePath(path) {
    if (path === undefined || core_1.Token.isUnresolved(path)) {
        return;
    }
    const validRolePath = /^(\/|\/[\u0021-\u007F]+\/)$/;
    if (path.length == 0 || path.length > 512) {
        throw new Error(`Role path must be between 1 and 512 characters. The provided role path is ${path.length} characters.`);
    }
    else if (!validRolePath.test(path)) {
        throw new Error('Role path must be either a slash or valid characters (alphanumerics and symbols) surrounded by slashes. '
            + `Valid characters are unicode characters in [\\u0021-\\u007F]. However, ${path} is provided.`);
    }
}
function validateMaxSessionDuration(duration) {
    if (duration === undefined) {
        return;
    }
    if (duration < 3600 || duration > 43200) {
        throw new Error(`maxSessionDuration is set to ${duration}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`);
    }
}
//# sourceMappingURL=data:application/json;base64,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