# Test the simulator with randomly generated OriginIR code.

import numpy as np
from qpandalite.originir.random_originir import random_originir
from qpandalite.originir.originir_spec import (available_originir_gates, 
                                               available_originir_error_channels,
                                               generate_sub_gateset_originir,
                                               generate_sub_error_channel_originir)

from qpandalite.simulator import OriginIR_NoisySimulator
from qpandalite.simulator.originir_simulator import OriginIR_Simulator
from qpandalite.test._utils import qpandalite_test
class NotMatchError(Exception):
    pass

def compare_density_operator(circuit, backend_1 = 'density_operator', backend_2 = 'density_operator_qutip', noisy = False):
    # This test compares two density operators generated by QuTip and QPanda-lite.
    # The test is based on the following steps:
    # 1. Simulate the circuit using QuTip and QPanda-lite.
    # 2. Compare the results.
    # 3. If the results are not the same, raise an error.

    # Step 1: Simulate the circuit using QuTip and QPanda-lite
    if noisy:
        sim1 = OriginIR_NoisySimulator(backend_type=backend_1)
        sim2 = OriginIR_NoisySimulator(backend_type=backend_2)
    else:
        sim1 = OriginIR_Simulator(backend_type=backend_1)
        sim2 = OriginIR_Simulator(backend_type=backend_2)

    #print('sim1')
    mat1 = sim1.simulate_density_matrix(circuit)
    #print('sim2')
    mat2 = sim2.simulate_density_matrix(circuit)
    #print('finish')

    # Step 2: Compare the results
    if not np.allclose(mat1, mat2):
        return NotMatchError(
            '---------------\n'
            f'{circuit}\n'
            '---------------\n'
            'Result not match!\n'
            f'{backend_1} Result = {mat1}\n'
            f'{backend_2} Result = {mat2}\n'
        )


def test_random_originir_compare_density_operator(backend_1 = 'density_operator',
                                                  backend_2 = 'density_operator_qutip',
                                                  random_batchsize = 100,
                                                  n_qubits = 5, n_gates = 20, 
                                                  gate_set = available_originir_gates,
                                                  error_channel = available_originir_error_channels):
    
    err_list = []    
    good_circuit_list = []
    bad_circuit_list = []
    for i in range(random_batchsize):
        originir_code = random_originir(
            n_qubits=n_qubits, 
            n_gates=n_gates, 
            instruction_set=gate_set, 
            channel_set=error_channel,
            allow_control=False,
            allow_dagger=True)
        
        #print(f'{originir_code}')
        noisy = False if not error_channel else True
        err = compare_density_operator(originir_code, backend_1, backend_2, noisy=noisy)    
        
        if err:
            print('Test failed!')
            err_list.append(err)
            bad_circuit_list.append((originir_code, err))
        else:
            print('Test passed!')
            good_circuit_list.append((originir_code, None))


    print(len(err_list), 'circuits failed')
    print(random_batchsize - len(err_list), 'circuits passed')

    # log good and bad circuits
    with open('good_circuits.txt', 'w') as f:
        for circuit, result in good_circuit_list:
            f.write(circuit + '\n----Result----\n' + str(result) + '\n-----------------\n\n')

    with open('bad_circuits.txt', 'w') as f:
        for e in err_list:
            f.write(str(e) + '\n')

    if len(err_list) > 0:
        raise ValueError('Some circuits failed!')
    

@qpandalite_test('Test Random OriginIR Density Operator')
def test_random_originir_density_operator():
    
    gate_set =  ['H', 'X', 'Y', 'Z', 'S', 'SX', 'T',
                 'RX', 'RY', 'RZ', 'U1', 'RPhi90', 'RPhi180',
                 'RPhi', 'U2',
                 'U3',
                 'CNOT', 'CZ', 'ISWAP', 
                 'XX', 'YY', 'ZZ', 'XY', 
                 'PHASE2Q', 
                 'UU15',
                 'TOFFOLI', 'CSWAP']
    
    gate_set =  ['H', 'X', 'Y', 'Z', 'S', 'SX', 'T',
                 'RX', 'RY', 'RZ', 'U1', 'RPhi90', 'RPhi180',
                 'RPhi', 'U2',
                 'U3',
                 'CNOT', 'CZ', 'ISWAP', 
                 'XX', 'YY', 'ZZ', 
                 'XY', 
                 'PHASE2Q', 
                 'UU15',
                 'TOFFOLI', 'CSWAP']
    
    gate_set = generate_sub_gateset_originir(gate_set)

    error_channel = ['PauliError1Q', 
                    'Depolarizing', 
                    'BitFlip', 
                    'PhaseFlip', 
                    'AmplitudeDamping',
                    'PauliError2Q', 
                    'TwoQubitDepolarizing']
    
    error_channel = ['PauliError1Q', 
                    'Depolarizing', 
                    'BitFlip', 
                    'PhaseFlip', 
                    'AmplitudeDamping',
                    'PauliError2Q', 
                    'TwoQubitDepolarizing']

    error_channel = generate_sub_error_channel_originir(error_channel)
   
    test_random_originir_compare_density_operator(
        backend_1 = 'density_operator',
        backend_2 = 'density_operator_qutip',
        random_batchsize = 100,
        n_qubits = 5,
        n_gates = 20,
        gate_set = gate_set,
        error_channel = error_channel)
    
    test_random_originir_compare_density_operator(
        backend_1 = 'density_operator',
        backend_2 = 'statevector',
        random_batchsize = 100,
        n_qubits = 5,
        n_gates = 20,
        gate_set = gate_set,
        error_channel = None)


if __name__ == '__main__':
    test_random_originir_density_operator()
