# Copyright 2020-2021 Akretion France (http://www.akretion.com/)
# @author: Alexis de Lattre <alexis.delattre@akretion.com>
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl).

import base64
import logging
from datetime import datetime

from markupsafe import Markup
from pyfrdas2 import (
    format_street_block,
    generate_file,
    get_partner_declaration_threshold,
)
from pyfrdas2.pyfrdas2 import logger as pyfrdas2logger
from stdnum.fr.siret import is_valid

from odoo import _, api, fields, models, tools
from odoo.exceptions import UserError, ValidationError
from odoo.osv import expression
from odoo.tools.misc import format_amount, format_date

try:
    from pyfrdas2 import __version__ as pyfrdas2version
except (OSError, ImportError):
    pyfrdas2version = "too old"

LOGLEVELS = {
    "debug": logging.DEBUG,
    "info": logging.INFO,
    "warn": logging.WARN,
    "error": logging.ERROR,
}

logger = logging.getLogger(__name__)
pyfrdas2logger.setLevel(
    LOGLEVELS.get(tools.config.get("log_level", "info"), logging.INFO)
)


PCG_DAS2_WARN_ACCOUNTS = [
    "6221",
    "6222",
    "6226",
    "6228",
    "653",
    "6516",
]

FRANCE_CODES = (
    "FR",
    "GP",
    "MQ",
    "GF",
    "RE",
    "YT",
    "NC",
    "PF",
    "TF",
    "MF",
    "BL",
    "PM",
    "WF",
)
AMOUNT_FIELDS = [
    "fee_amount",
    "commission_amount",
    "brokerage_amount",
    "discount_amount",
    "attendance_fee_amount",
    "copyright_royalties_amount",
    "licence_royalties_amount",
    "other_income_amount",
    "allowance_amount",
    "benefits_in_kind_amount",
    "withholding_tax_amount",
]


class L10nFrDas2(models.Model):
    _name = "l10n.fr.das2"
    _inherit = ["mail.thread", "mail.activity.mixin"]
    _order = "year desc"
    _description = "DAS2"
    _check_company_auto = True

    year = fields.Integer(
        required=True,
        tracking=True,
        default=lambda self: self._default_year(),
        index=True,
    )
    state = fields.Selection(
        [
            ("draft", "Draft"),
            ("done", "Done"),
        ],
        default="draft",
        readonly=True,
        tracking=True,
        index=True,
    )
    company_id = fields.Many2one(
        "res.company",
        ondelete="cascade",
        required=True,
        default=lambda self: self.env.company,
    )
    currency_id = fields.Many2one(
        related="company_id.currency_id",
        store=True,
        string="Company Currency",
    )
    payment_journal_ids = fields.Many2many(
        "account.journal",
        string="Payment Journals",
        required=True,
        default=lambda self: self._default_payment_journals(),
        domain="[('company_id', '=', company_id)]",
        check_company=True,
    )
    line_ids = fields.One2many(
        "l10n.fr.das2.line",
        "parent_id",
        string="Lines",
    )
    partner_declare_threshold = fields.Integer(
        compute="_compute_partner_declare_threshold",
        store=True,
        precompute=True,
        string="Partner Declaration Threshold",
    )
    dads_type = fields.Selection(
        [
            ("4", "La société verse des salaires"),
            ("1", "La société ne verse pas de salaires"),
        ],
        "DADS Type",
        required=True,
        tracking=True,
        default=lambda self: self._default_dads_type(),
    )
    # option for draft moves ?
    contact_id = fields.Many2one(
        "res.partner",
        string="Administrative Contact",
        default=lambda self: self.env.user.partner_id.id,
        tracking=True,
        help="Contact in the company for the fiscal administration: the name, "
        "email and phone number of this partner will be used in the file.",
    )
    attachment_id = fields.Many2one("ir.attachment", readonly=True)
    attachment_datas = fields.Binary(
        related="attachment_id.datas", string="Declaration File"
    )
    attachment_name = fields.Char(related="attachment_id.name", string="Filename")
    unencrypted_attachment_id = fields.Many2one(
        "ir.attachment", string="Unencrypted Attachment", readonly=True
    )
    # The only drawback of the warning_msg solution is that I didn't find a way
    # to put a link to partners inside it
    warning_msg = fields.Html(readonly=True)

    _sql_constraints = [
        (
            "year_company_uniq",
            "unique(company_id, year)",
            "A DAS2 already exists for that year!",
        )
    ]

    @api.depends("year")
    def _compute_partner_declare_threshold(self):
        for rec in self:
            if rec.year:
                rec.partner_declare_threshold = get_partner_declaration_threshold(
                    rec.year
                )

    @api.model
    def _default_dads_type(self):
        previous_decl = self.search(
            [("dads_type", "!=", False)], order="year desc", limit=1
        )
        if previous_decl:
            return previous_decl.dads_type
        else:
            return "4"

    @api.model
    def _default_payment_journals(self):
        res = []
        pay_journals = self.env["account.journal"].search(
            [
                ("type", "in", ("bank", "cash", "credit")),
                ("company_id", "=", self.env.company.id),
            ]
        )
        if pay_journals:
            res = pay_journals.ids
        return res

    @api.model
    def _default_year(self):
        last_year = datetime.today().year - 1
        return last_year

    @api.depends("year")
    def _compute_display_name(self):
        for rec in self:
            rec.display_name = f"DAS2 {rec.year}"

    def done(self):
        self.ensure_one()
        vals = {"state": "done"}
        if self.line_ids:
            running_env = tools.config.get("running_env")
            if running_env in ("test", "dev"):
                encryption = "test"
            else:
                encryption = "prod"
            msg = Markup(
                _(
                    "DAS2 file generated. Encrypted with DGFiP's"
                    " <b>%(encryption)s</b> PGP key using pyfrdas2 "
                    "version %(pyfrdas2version)s.",
                    encryption=encryption,
                    pyfrdas2version=pyfrdas2version,
                )
            )

            attach = self.generate_file_and_attach(encryption=encryption)
            self.message_post(body=msg)
            # also generate a clear file, for audit purposes
            unencrypted_attach = self.generate_file_and_attach(encryption="none")
            vals.update(
                {
                    "attachment_id": attach.id,
                    "unencrypted_attachment_id": unencrypted_attach.id,
                }
            )
        self.write(vals)
        return

    def back2draft(self):
        self.ensure_one()
        if self.attachment_id:
            self.attachment_id.unlink()
            self.message_post(body=_("DAS2 file deleted."))
        if self.unencrypted_attachment_id:
            self.unencrypted_attachment_id.unlink()
        self.write({"state": "draft"})
        return

    def unlink(self):
        for rec in self:
            if rec.state == "done":
                raise UserError(
                    _("Cannot delete declaration %s in done state.") % rec.display_name
                )
        return super().unlink()

    def generate_lines(self):
        self.ensure_one()
        lfdlo = self.env["l10n.fr.das2.line"]
        company = self.company_id
        if not company.country_id:
            raise UserError(
                _("Country not set on company '%s'.") % company.display_name
            )
        if company.country_id.code not in FRANCE_CODES:
            raise UserError(
                _(
                    "Company '%(company)s' is configured in country '%(country)s'. "
                    "The DAS2 is only for France and it's oversea territories.",
                    company=company.display_name,
                    country=company.country_id.name,
                )
            )
        if company.currency_id != self.env.ref("base.EUR"):
            raise UserError(
                _(
                    "Company '%(company)s' is configured with currency '%(currency)s'. "
                    "It should be EUR.",
                    company=company.display_name,
                    currency=company.currency_id.name,
                )
            )
        das2_partners = self.env["res.partner"].search(
            [("parent_id", "=", False), ("fr_das2_type", "!=", False)]
        )
        if not das2_partners:
            raise UserError(_("There are no partners configured for DAS2."))
        self.line_ids.unlink()
        base_domain = [
            ("company_id", "=", self.company_id.id),
            ("date", ">=", "%d-01-01" % self.year),
            ("date", "<=", "%d-12-31" % self.year),
            ("journal_id", "in", self.payment_journal_ids.ids),
            ("balance", "!=", 0),
            ("parent_state", "=", "posted"),
        ]
        for partner in das2_partners:
            vals = self._prepare_line(partner, base_domain)
            if vals:
                lfdlo.create(vals)
        self.generate_warning_msg(das2_partners)

    def _prepare_line(self, partner, base_domain):
        amlo = self.env["account.move.line"]
        mlines = amlo.search(
            base_domain
            + [
                ("partner_id", "=", partner.id),
                ("account_id", "=", partner.property_account_payable_id.id),
            ]
        )
        note = ""
        amount = 0.0
        for mline in mlines:
            amount += mline.balance
            note_text = _(
                "Payment dated <b>%(date)s</b> in journal <b>%(journal)s</b>: "
                "%(amount)s (journal entry %(move_name)s)",
                date=format_date(self.env, mline.date),
                journal=mline.journal_id.display_name,
                amount=format_amount(self.env, mline.balance, self.currency_id),
                move_name=mline.move_id.name,
            )
            note += f"<li>{note_text}</li>"
        res = False
        amount_int = int(round(amount))
        if note and amount_int > 0:
            field_name = f"{partner.fr_das2_type}_amount"
            res = {
                field_name: amount_int,
                "parent_id": self.id,
                "partner_id": partner.id,
                "note": f"<ul>{note}</ul>",
            }
        return res

    def generate_warning_msg(self, das2_partners):
        amlo = self.env["account.move.line"]
        aao = self.env["account.account"]
        ajo = self.env["account.journal"]
        company = self.company_id
        purchase_journals = ajo.search(
            [("type", "=", "purchase"), ("company_id", "=", company.id)]
        )
        acc_domain = expression.OR(
            [
                ("code", "=like", f"{acc_code}%"),
            ]
            for acc_code in PCG_DAS2_WARN_ACCOUNTS
        )
        das2_accounts = aao.with_company(self.company_id.id).search(acc_domain)
        rg_res = amlo._read_group(
            [
                ("company_id", "=", company.id),
                ("date", ">=", "%d-10-01" % (self.year - 1)),
                ("date", "<=", "%d-12-31" % self.year),
                ("journal_id", "in", purchase_journals.ids),
                ("partner_id", "!=", False),
                ("partner_id", "not in", das2_partners.ids),
                ("account_id", "in", das2_accounts.ids),
                ("balance", "!=", 0),
                ("parent_state", "=", "posted"),
                ("display_type", "=", "product"),
            ],
            ["partner_id"],
            ["__count"],
        )
        msg = False
        msg_post = _("DAS2 lines generated. ")
        if rg_res:
            msg = _(
                "The following partners are not configured for DAS2 but "
                "they have expenses in some accounts that indicate "
                "they should probably be configured for DAS2:<ul>"
            )
            msg_post += msg
            for rg_re in rg_res:
                partner = rg_re[0]
                msg_post += (
                    '<li><a href="#" data-oe-model="res.partner" '
                    'data-oe-id="%d">%s</a></li>' % (partner.id, partner.display_name)
                )
                msg += f"<li>{partner.display_name}</li>"
            msg_post += "</ul>"
            msg += "</ul>"
        self.message_post(body=Markup(msg_post))
        self.write({"warning_msg": msg})

    @api.model
    def _prepare_field(
        self, field_name, partner, value, size, required=False, numeric=False
    ):
        """This function is designed to be inherited."""
        if numeric:
            if not value:
                value = 0
            if not isinstance(value, int):
                try:
                    value = int(value)
                except Exception as e:
                    raise UserError(
                        _(
                            "Failed to convert field '%(field_name)s' "
                            "(partner %(partner)s) to integer.",
                            field_name=field_name,
                            partner=partner.display_name,
                        )
                    ) from e
            value = str(value)
            if len(value) > size:
                raise UserError(
                    _(
                        "Field %(field_name)s (partner %(partner)s) has value "
                        "%(value)s: it is bigger than the maximum size "
                        "(%(size)d characters).",
                        field_name=field_name,
                        partner=partner.display_name,
                        value=value,
                        size=size,
                    )
                )
            if len(value) < size:
                value = value.rjust(size, "0")
            return value
        if required and not value:
            raise UserError(
                _(
                    "The field '%(field_name)s' (partner %(partner)s) is empty or 0. "
                    "It should have a non-null value.",
                    field_name=field_name,
                    partner=partner.display_name,
                )
            )
        if not value:
            value = " " * size
        # Cut if too long
        value = value[0:size]
        # enlarge if too small
        if len(value) < size:
            value = value.ljust(size, " ")
        return value

    def _prepare_address(self, partner):
        cstreet2 = self._prepare_field("Street2", partner, partner.street2, 32)
        cstreet = format_street_block(partner.street)
        # specs : only bureau distributeur and code postal are
        # required. And they say it is tolerated to set city as
        # bureau distributeur
        # And we don't set the commune field because we put the same info
        # in bureau distributeur
        # specs 2024 : we now have to have structured adresses with a separation
        # for the number, bis/ter and street name.
        # We'll do our best to comply with that.
        if not partner.city:
            raise UserError(_("Missing city on partner '%s'.") % partner.display_name)
        if partner.country_id and partner.country_id.code not in FRANCE_CODES:
            if not partner.country_id.fr_cog:
                raise UserError(
                    _("Missing Code Officiel Géographique on country '%s'.")
                    % partner.country_id.display_name
                )
            cog = self._prepare_field(
                "COG", partner, partner.country_id.fr_cog, 5, True, numeric=True
            )
            raw_country_name = partner.with_context(lang="fr_FR").country_id.name
            country_name = self._prepare_field(
                "Nom du pays", partner, raw_country_name, 26, True
            )
            raw_commune = partner.city
            if partner.zip:
                raw_commune = f"{partner.zip} {partner.city}"
            commune = self._prepare_field("Commune", partner, raw_commune, 26, True)
            caddress = (
                cstreet2
                + " "
                + cstreet
                + cog
                + " "
                + commune
                + cog
                + " "
                + country_name
            )
        # According to the specs, we should have some code specific for DOM-TOM
        # But it's not easy, because we are supposed to give the INSEE code
        # of the city, not of the territory => we don't handle that for the
        # moment
        else:
            ccity = self._prepare_field("City", partner, partner.city, 26, True)
            czip = self._prepare_field("Zip", partner, partner.zip, 5, True)
            caddress = (
                cstreet2 + " " + cstreet + "0" * 5 + " " + " " * 26 + czip + " " + ccity
            )
        assert len(caddress) == 129
        return caddress

    def _prepare_file(self):
        company = self.company_id
        cpartner = company.partner_id
        contact = self.contact_id
        eu_countries = self.env.ref("base.europe").country_ids
        csiren = self._prepare_field("SIREN", cpartner, cpartner.siren, 9, True)
        csiret = self._prepare_field("SIRET", cpartner, cpartner.siret, 14, True)
        cape = self._prepare_field("APE", cpartner, company.ape, 5, True)
        cname = self._prepare_field("Name", cpartner, company.name, 50, True)
        file_type = "X"  # tous déclarants honoraires seuls
        year = str(self.year)
        assert len(year) == 4
        caddress = self._prepare_address(cpartner)
        cprefix = csiret + "01" + year + self.dads_type
        # line 010 Company header
        flines = []
        flines.append(
            csiren
            + "0" * 12
            + "010"
            + " " * 14
            + cape
            + " " * 4
            + cname
            + caddress
            + " " * 8
            + file_type
            + csiret
            + " " * 5
            + caddress
            + " "
            + " " * 288
        )

        # ligne 020 Etablissement header
        # We don't add a field for profession on the company
        # because it's not even a field on the paper form!
        # We only set profession for suppliers
        flines.append(
            cprefix
            + "020"
            + " " * 14
            + cape
            + "0" * 14
            + " " * 41
            + cname
            + caddress
            + " " * 40
            + " " * 53
            + "N"
            + " " * 3
            + "N"
            + " " * 297
        )

        i = 0
        for line in self.line_ids.filtered(lambda x: x.to_declare):
            i += 1
            partner = line.partner_id
            if (
                partner.country_id
                and partner.country_id.code in FRANCE_CODES
                and not line.partner_siret
            ):
                raise UserError(
                    _("Missing SIRET for french partner %s.") % partner.display_name
                )
            if (
                not partner.is_company
                and partner.country_id
                and partner.country_id.code not in FRANCE_CODES
                and partner.country_id in eu_countries
            ):
                if not hasattr(partner, "birthdate_date"):
                    raise UserError(
                        _(
                            "Partner '%(partner_name)s' is a physical person "
                            "in country %(country)s which is a foreign EU country. "
                            "So you must install the OCA module "
                            "'partner_contact_birthdate' and set the birth date "
                            "of this partner.",
                            partner_name=partner.name,
                            country=partner.country_id.name,
                        )
                    )
                if not partner.birthdate_date:
                    raise UserError(
                        _(
                            "Missing birth date on partner '%s'. This information is "
                            "required for physical persons in foreign EU countries.",
                            partner.name,
                        )
                    )

            # ligne 210 honoraire
            birthdate = " " * 8
            if partner.is_company:
                partner_name = self._prepare_field(
                    "Partner name", partner, partner.name, 50, True
                )
                lastname = " " * 30
                firstname = " " * 20
            else:
                partner_name = " " * 50
                if hasattr(partner, "firstname") and partner.firstname:
                    lastname = self._prepare_field(
                        "Lastname", partner, partner.lastname, 30, True
                    )
                    firstname = self._prepare_field(
                        "Firstname", partner, partner.firstname, 20, True
                    )
                else:
                    lastname = self._prepare_field(
                        "Partner name", partner, partner.name, 30, True
                    )
                    firstname = " " * 20
                if (
                    partner.country_id
                    and partner.country_id in eu_countries
                    and partner.country_id.code not in FRANCE_CODES
                ):
                    birthdate = partner.birthdate_date.strftime("%d%m%Y")
            address = self._prepare_address(partner)
            partner_siret = self._prepare_field(
                "SIRET", partner, line.partner_siret, 14
            )
            job = self._prepare_field("Profession", partner, line.job, 30)
            amount_fields_list = [
                self._prepare_field(x, partner, line[x], 10, numeric=True)
                for x in AMOUNT_FIELDS
            ]
            if line.benefits_in_kind_amount:
                bik_letters = ""
                bik_letters += line.benefits_in_kind_food and "N" or " "
                bik_letters += line.benefits_in_kind_accomodation and "L" or " "
                bik_letters += line.benefits_in_kind_car and "V" or " "
                bik_letters += line.benefits_in_kind_other and "A" or " "
                bik_letters += line.benefits_in_kind_nict and "T" or " "
            else:
                bik_letters = " " * 5
            if line.allowance_amount:
                allow_letters = ""
                allow_letters += line.allowance_fixed and "F" or " "
                allow_letters += line.allowance_real and "R" or " "
                allow_letters += line.allowance_employer and "P" or " "
            else:
                allow_letters = " " * 3
            flines.append(
                cprefix
                + "210"
                + partner_siret
                + lastname
                + firstname
                + partner_name
                + job
                + address
                + "".join(amount_fields_list)
                + bik_letters
                + allow_letters
                + " " * 2
                + "0" * 10
                + birthdate
                + " " * 237
            )
        rg = self.env["l10n.fr.das2.line"]._read_group(
            [("parent_id", "=", self.id)], [], [f"{x}:sum" for x in AMOUNT_FIELDS]
        )[0]
        total_fields_list = [
            self._prepare_field(x, cpartner, x, 12, numeric=True) for x in rg
        ]
        contact_name = self._prepare_field(
            "Administrative contact name", contact, contact.name, 50
        )
        contact_email = self._prepare_field(
            "Administrative contact email", contact, contact.email, 60
        )
        phone = contact.phone or contact.mobile
        phone = (
            phone.replace(" ", "")
            .replace(".", "")
            .replace("-", "")
            .replace("\u00a0", "")
        )
        if phone.startswith("+33"):
            phone = f"0{phone[3:]}"
        contact_phone = self._prepare_field(
            "Administrative contact phone", contact, phone, 10
        )
        # Total Etablissement
        flines.append(
            cprefix
            + "300"
            + " " * 36
            + "0" * 12 * 9
            + "".join(total_fields_list)
            + " " * 12
            + "0" * 12 * 2
            + "0" * 6
            + " " * 48
            + "0" * 12
            + " " * 74
            + contact_name
            + contact_phone
            + contact_email
            + csiren
            + " " * 67
        )

        lines_number = self._prepare_field(
            "Number of lines", cpartner, i, 6, numeric=True
        )
        # Total Entreprise
        flines.append(
            csiren
            + "9" * 12
            + "310"
            + "00001"
            + "0" * 6
            + lines_number
            + " " * 36
            + "0" * 12 * 9
            + "".join(total_fields_list)
            + " " * 12
            + "0" * 12 * 2
            + "0" * 6
            + " " * 48
            + "0" * 12
            + " " * 253
        )
        for fline in flines:
            if len(fline) != 672:
                raise UserError(
                    _(
                        "One of the lines has a length of %(length)d. "
                        "All lines should have a length of 672. "
                        "Line: %(line_content)s.",
                        length=len(fline),
                        line_content=fline,
                    )
                )
        file_content = "\r\n".join(flines) + "\r\n"
        return file_content

    def generate_file_and_attach(self, encryption="prod"):
        self.ensure_one()
        assert encryption in ("prod", "test", "none")
        company = self.company_id
        if not self.line_ids:
            raise UserError(_("The DAS2 has no lines."))
        if not company.siret:
            raise UserError(_("Missing SIRET on company '%s'.") % company.display_name)
        if not company.siren:
            raise UserError(_("Missing SIREN on company '%s'.") % company.display_name)
        if not company.ape:
            raise UserError(_("Missing APE on company '%s'.") % company.display_name)
        if not company.street:
            raise UserError(_("Missing Street on company '%s'") % company.display_name)
        contact = self.contact_id
        if not contact:
            raise UserError(_("Missing administrative contact."))
        if not contact.email:
            raise UserError(
                _("The email is not set on the administrative contact " "partner '%s'.")
                % contact.display_name
            )
        if not contact.phone and not contact.mobile:
            raise UserError(
                _(
                    "The phone number is not set on the administrative contact "
                    "partner '%s'."
                )
                % contact.display_name
            )
        if self.attachment_id:
            raise UserError(
                _(
                    "A declaration file already exists. First, delete it via the "
                    "attachments and then re-generate it."
                )
            )

        file_content = self._prepare_file()
        # In 2025, they made it clear (at last !) that the file must be in utf-8
        file_content_encoded = file_content.encode("utf-8")

        try:
            file_bytes_result, filename = generate_file(
                file_content_encoded, self.year, company.siren, encryption=encryption
            )
        except Exception as e:
            raise UserError(e) from e

        attach = self.env["ir.attachment"].create(
            {
                "name": filename,
                "res_id": self.id,
                "res_model": self._name,
                "datas": base64.encodebytes(file_bytes_result),
            }
        )
        return attach

    def button_lines_fullscreen(self):
        self.ensure_one()
        action = self.env.ref("l10n_fr_das2.l10n_fr_das2_line_action").sudo().read()[0]
        action.update(
            {
                "domain": [("parent_id", "=", self.id)],
                "views": False,
            }
        )
        return action


class L10nFrDas2Line(models.Model):
    _name = "l10n.fr.das2.line"
    _description = "DAS2 line"

    parent_id = fields.Many2one(
        "l10n.fr.das2", string="DAS2 Report", ondelete="cascade"
    )
    partner_id = fields.Many2one(
        "res.partner",
        string="Supplier",
        ondelete="restrict",
        domain=[("parent_id", "=", False)],
        required=True,
    )
    partner_siret = fields.Char(
        compute="_compute_partner_siret",
        store=True,
        precompute=True,
        readonly=False,
        string="SIRET",
        size=14,
    )
    company_id = fields.Many2one(related="parent_id.company_id", store=True)
    currency_id = fields.Many2one(
        related="parent_id.company_id.currency_id",
        store=True,
        string="Company Currency",
    )
    fee_amount = fields.Integer(string="Honoraires et vacations")
    commission_amount = fields.Integer(string="Commissions")
    brokerage_amount = fields.Integer(string="Courtages")
    discount_amount = fields.Integer(string="Ristournes")
    attendance_fee_amount = fields.Integer(string="Jetons de présence")
    copyright_royalties_amount = fields.Integer(string="Droits d'auteur")
    licence_royalties_amount = fields.Integer(string="Droits d'inventeur")
    other_income_amount = fields.Integer(string="Autres rémunérations")
    allowance_amount = fields.Integer(string="Indemnités et remboursements")
    benefits_in_kind_amount = fields.Integer(string="Avantages en nature")
    withholding_tax_amount = fields.Integer(string="Retenue à la source")
    total_amount = fields.Integer(
        compute="_compute_total_amount",
        store=True,
        readonly=True,
    )
    to_declare = fields.Boolean(
        compute="_compute_total_amount", readonly=True, store=True
    )
    allowance_fixed = fields.Boolean("Allocation forfaitaire")
    allowance_real = fields.Boolean("Sur frais réels")
    allowance_employer = fields.Boolean("Prise en charge directe par l'employeur")
    benefits_in_kind_food = fields.Boolean("Nourriture")
    benefits_in_kind_accomodation = fields.Boolean("Logement")
    benefits_in_kind_car = fields.Boolean("Voiture")
    benefits_in_kind_other = fields.Boolean("Autres")
    benefits_in_kind_nict = fields.Boolean("Outils issus des NTIC")
    state = fields.Selection(related="parent_id.state", store=True)
    note = fields.Html()
    job = fields.Char(
        compute="_compute_job",
        store=True,
        precompute=True,
        readonly=False,
        string="Profession",
        size=30,
    )

    _sql_constraints = [
        (
            "partner_parent_unique",
            "unique(partner_id, parent_id)",
            "Same partner used on several lines!",
        ),
        (
            "fee_amount_positive",
            "CHECK(fee_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "commission_amount_positive",
            "CHECK(commission_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "brokerage_amount_positive",
            "CHECK(brokerage_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "discount_amount_positive",
            "CHECK(discount_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "attendance_fee_amount_positive",
            "CHECK(attendance_fee_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "copyright_royalties_amount_positive",
            "CHECK(copyright_royalties_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "licence_royalties_amount_positive",
            "CHECK(licence_royalties_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "other_income_amount_positive",
            "CHECK(other_income_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "allowance_amount_positive",
            "CHECK(allowance_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "benefits_in_kind_amount_positive",
            "CHECK(benefits_in_kind_amount >= 0)",
            "Negative amounts not allowed!",
        ),
        (
            "withholding_tax_amount_positive",
            "CHECK(withholding_tax_amount >= 0)",
            "Negative amounts not allowed!",
        ),
    ]

    @api.depends(
        "parent_id.partner_declare_threshold",
        "fee_amount",
        "commission_amount",
        "brokerage_amount",
        "discount_amount",
        "attendance_fee_amount",
        "copyright_royalties_amount",
        "licence_royalties_amount",
        "other_income_amount",
        "allowance_amount",
        "benefits_in_kind_amount",
        "withholding_tax_amount",
    )
    def _compute_total_amount(self):
        for line in self:
            total_amount = 0
            for field_name in AMOUNT_FIELDS:
                total_amount += line[field_name]
            to_declare = False
            if line.parent_id:
                if total_amount >= line.parent_id.partner_declare_threshold:
                    to_declare = True
            line.to_declare = to_declare
            line.total_amount = total_amount

    @api.depends("partner_id")
    def _compute_partner_siret(self):
        for line in self:
            if line.partner_id and line.partner_id.siren and line.partner_id.nic:
                line.partner_siret = line.partner_id.siret

    @api.depends("partner_id")
    def _compute_job(self):
        for line in self:
            if line.partner_id and line.partner_id.fr_das2_job:
                line.job = line.partner_id.fr_das2_job

    @api.constrains("partner_siret")
    def _check_siret(self):
        for line in self:
            if line.partner_siret and not is_valid(line.partner_siret):
                raise ValidationError(
                    _(
                        "SIRET '%(siret)s' of supplier '%(partner)s' is invalid.",
                        siret=line.partner_siret,
                        partner=line.partner_id.display_name,
                    )
                )
