import os
import pdb

from mitmproxy.http import HTTPFlow as MitmproxyHTTPFlow
from mitmproxy.http import Headers, Request as MitmproxyRequest, Response as MitmproxyResponse

from stoobly_agent.app.proxy.handle_mock_service import handle_request_mock
from stoobly_agent.app.proxy.handle_replay_service import handle_request_replay, handle_response_replay
from stoobly_agent.app.proxy.handle_record_service import handle_response_record
from stoobly_agent.app.proxy.handle_test_service import handle_request_test, handle_response_test
from stoobly_agent.app.proxy.intercept_settings import InterceptSettings
from stoobly_agent.app.proxy.mock.context import MockContext
from stoobly_agent.app.proxy.replay.context import ReplayContext
from stoobly_agent.app.proxy.record.context import RecordContext
from stoobly_agent.app.proxy.utils.response_handler import bad_request
from stoobly_agent.app.settings import Settings
from stoobly_agent.config.constants import mode
from stoobly_agent.lib.logger import Logger

# Disable proxy settings in urllib
os.environ['no_proxy'] = '*'

LOG_ID = 'InterceptHandler'

def request(flow: MitmproxyHTTPFlow):
    request: MitmproxyRequest = flow.request

    __patch_cookie(request)

    intercept_settings = InterceptSettings(Settings.instance(), request)

    active_mode = intercept_settings.mode
    Logger.instance().debug(f"{LOG_ID}:ProxyMode: {active_mode}")

    if active_mode == mode.MOCK:
        context = MockContext(flow, intercept_settings)
        handle_request_mock(context)
    elif active_mode == mode.RECORD:
        __disable_web_cache(request)
    elif active_mode == mode.REPLAY:
        context = ReplayContext(flow, intercept_settings)
        handle_request_replay(context)
    elif active_mode == mode.TEST:
        context = ReplayContext(flow, intercept_settings)
        handle_request_test(context)
    else:
        if active_mode != mode.NONE:
            bad_request(
                flow,
                "Valid env MODES: %s, Got: %s" % ([mode.MOCK, mode.RECORD, mode.REPLAY, mode.TEST], active_mode)
            )

def response(flow: MitmproxyHTTPFlow):
    request: MitmproxyRequest = flow.request

    intercept_settings = InterceptSettings(Settings.instance(), request)
    intercept_settings.for_response()

    active_mode = intercept_settings.mode

    if active_mode == mode.RECORD:
        context = RecordContext(flow, intercept_settings)
        return handle_response_record(context)
    elif active_mode == mode.REPLAY:
        context = ReplayContext(flow, intercept_settings)
        return handle_response_replay(context)
    elif active_mode == mode.TEST:
        context = ReplayContext(flow, intercept_settings)
        return handle_response_test(context)

### PRIVATE

def __disable_web_cache(request: MitmproxyRequest) -> None:
    request.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    request.headers['Expires'] = '0'
    request.headers['Pragma'] = 'no-cache'

    if 'IF-NONE-MATCH' in request.headers:
        del request.headers['IF-NONE-MATCH']

    if 'IF-MODIFIED-SINCE' in request.headers:
        del request.headers['IF-MODIFIED-SINCE']

# Fix issue where multi-value cookies become comma separated
def __patch_cookie(request: MitmproxyRequest):
    header_name = 'cookie'

    if len(request.headers.get_all(header_name)) > 1:
        __combine_header(request.headers, header_name, '; ')
        
def __combine_header(headers: Headers, header_name: str, delimitter: str):
    values = headers.get_all(header_name)
    headers[header_name] = delimitter.join(values)