# If you publish results that make use of this software or the Birth Annotator
# for Budding Yeast algorithm, please cite:
# Pietsch, J.M.J., Muñoz, A.F., Adjavon, D.-Y.A., Farquhar, I., Clark, I.B.N.,
# and Swain, P.S. (2023). Determining growth rates from bright-field images of
# budding cells through identifying overlaps. eLife. 12:e79812.
# https://doi.org/10.7554/eLife.79812
# 
# 
# The MIT License (MIT)
# 
# Copyright (c) Julian Pietsch, Alán Muñoz and Diane Adjavon 2023
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to
# deal in the Software without restriction, including without limitation the
# rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
# sell copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
# IN THE SOFTWARE.
import numpy as np
import itertools

from scipy import ndimage

from typing import (Union, Sequence, Any, Optional, List, NamedTuple, Literal,
                    Tuple)

from .errors import BadParam
from baby import segmentation
from .segmentation import (
    iterative_erosion, threshold_segmentation, binary_edge,
    single_region_prop, mask_to_knots, get_edge_scores, refine_radial_grouped,
    iterative_dilation, draw_radial)
from .utils import EncodableNamedTuple
from .volume import volume


BROADCASTABLE_PARAMS = {'edge_sub_dilations', 'interior_threshold',
                        'nclosing', 'nopening', 'connectivity',
                        'min_area', 'bbox_padding', 'pedge_thresh',
                        'group_thresh_expansion'}


@EncodableNamedTuple
class SegmentationParameters(NamedTuple):
    """Defines parameters for segmentation from CNN outputs.

    Designed to be used with :py:class:`MorphSegGrouped` and descriptions of
    the attributes below should be read in conjunction with that class.

    Certain parameters can be specified group-wise, that is, with a different
    value for each of the groupings defined by ``cellgroups``. If specified as
    a singleton, the same parameter value will be replicated for each group. If
    specified as a list, the length of the list should match the number of
    cell groupings.

    This class has been decorated with :py:func:`utils.EncodableNamedTuple`,
    so can be reversibly encoded to JSON using :py:func:`utils.jsonify`.

    Attributes:
        cellgroups: By default, all groups defined by the ``flattener``
            (except ``budonly`` groups) will be included in descending size
            order. Optionally specify a sequence of ``str`` or a sequence of
            tuples of ``str`` naming specific flattener groups or respectively
            combinations thereof.
        edge_sub_dilations: Number of iterations of grayscale morphological
            dilation to apply to the edge probability image when it is used
            to dampen the interior probability image before thresholding.
            Specify ``None`` to disable dampening, or ``0`` to dampen with
            the (undilated) edge probability image. Can be specified
            group-wise.
        interior_threshold: Threshold to apply to interior probability image
            for locating cell instances (see
            :py:func:`segmentation.threshold_segmentation`). Can be specified
            group-wise.
        connectivity: Connectivity for labelling connected regions (see
            :py:func:`segmentation.threshold_segmentation`). Can be specified
            group-wise.
        nclosing: Number of closing iterations to apply to identified masks
            (see :py:func:`segmentation.threshold_segmentation`). Can be
            specified group-wise.
        nopening: Number of opening iterations to apply to identified masks
            (see :py:func:`segmentation.threshold_segmentation`). Can be
            specified group-wise.
        min_area: Minimum area (in square pixels) below which a mask will be
            discarded. Has precedence over a smaller minimum specified by
            any of the ``cellgroups``. Can be specified group-wise.
        fit_radial: If ``True``, edges are generated by splines with knots
            defined in radial coordinates; if ``False``, edges are taken
            directly from the dilated masks.
        cartesian_spline: If ``True``, splines are interpolated in cartesian
            coordinates; if ``False``, they are interpolated in radial
            coordinates.
        bbox_padding: The number of pixels of padding that should be
            included when limiting mask processing to a bounding box.
        nrays_thresh_map: See :py:func:`segmentation.guess_radial_edge`. Only
            applies when ``cartesian_spline=False`` and ``curvy_knots=False``.
        nrays_max: See :py:func:`segmentation.guess_radial_edge`. Only applies
            when ``cartesian_spline=False`` and ``curvy_knots=False``.
        n_trials: See :py:func:`segmentation.guess_cartesian_edge`. Only
            applies when ``cartesian_spline=True`` and ``curvy_knots=False``.
        pxperknot: See :py:func:`segmentation.guess_cartesian_edge`. Only
            applies when ``cartesian_spline=True`` and ``curvy_knots=False``.
        squishiness: See :py:func:`segmentation.guess_cartesian_edge`. Only
            applies when ``cartesian_spline=True`` and ``curvy_knots=False``.
        alignedness: See :py:func:`segmentation.guess_cartesian_edge`. Only
            applies when ``cartesian_spline=True`` and ``curvy_knots=False``.
        curvy_knots: If ``True``, knots for splines are placed at points of
            estimated high curvature
            (:py:func:`segmentation.curvy_knots_from_outline`); if ``False``,
            knots are placed at equally spaced locations in radial space
            (``cartesian_spline=False``) or approximately equally spaced
            locations along the perimeter (``cartesian_spline=True``).
        n_knots_fraction: Determines the number of knots as a fraction of edge
            pixel count for the case where ``curvy_knots=True``
            (see:py:func:`segmentation.curvy_knots_from_outline`).
        pedge_thresh: Threshold on the edge probability scores of guessed
            outlines, below which outlines will be discarded. Can be specified
            group-wise. If any are specified as ``None``, then the decision on
            which contained outline to keep in neighbouring groups based on
            area rather than edge score.
        use_group_thresh: If ``True``, discard masks if their areas fall
            outside the size limits as defined for the group they originated
            in; if ``False``, keep all generated masks (except those below the
            ``min_area`` threshold).
        group_thresh_expansion: Expansion factor for the size limits defined
            by each group. Can be specified group-wise.
        containment_func: Specifies the metric to use to decide whether cells
            in adjacent groups should be considered identical or not.
            Currently the only valid choices are 'mask_containment' and
            'mask_iou' (see :py:func:`segmentation.mask_containment` and
            :py:func:`segmentation.mask_iou`).
        containment_thresh: Threshold above which to consider cells in
            adjacent groups identical.
    """
    cellgroups: Union[None, List[str], List[Tuple[str, ...]]] = None
    edge_sub_dilations: Union[None, int, List[Optional[int]]] = 0
    interior_threshold: Union[float, List[float]] = 0.5
    nclosing: Union[int, List[int]] = 0
    nopening: Union[int, List[int]] = 0
    connectivity: Union[int, List[int]] = 2
    min_area: Union[int, List[int]] = 10
    fit_radial: bool = True
    cartesian_spline: int = False
    bbox_padding: Union[int, List[int]] = 10
    nrays_thresh_map: List[Tuple[float, int]] = [(5., 4), (20., 6)]
    nrays_max: int = 8
    n_trials: int = 10
    pxperknot: float = 4.
    squishiness: float = 0.1
    alignedness: float = 10.
    curvy_knots: bool = False
    n_knots_fraction: float = 0.5
    pedge_thresh: Union[None, float, List[Optional[float]]] = None
    use_group_thresh: bool = False
    group_thresh_expansion: Union[float, List[float]] = 0.
    containment_func: Literal['mask_containment', 'mask_iou'] = 'mask_containment'
    containment_thresh: int = 0.8


# class ContainmentFunction:
#     def __init__(self, threshold: float = .8):
#         self.threshold = threshold
#
#     def __call__(self, *args, **kwargs):
#         return mask_containment(*args, **kwargs) > self.threshold


# Making a callable object instance:
#
# a = lambda a, b: a + b
#
# def a(a, b):
#     return a + b
#
# class A:
#     def __call__(self, a, b):
#         return a + b
# a = A()

class Cell:
    def __init__(self, area, mask, predicted_edge, border_rect, params):
        self.area = area
        self.mask = mask
        self.predicted_edge = predicted_edge
        self.params = params

        self._coords = None
        self._edge = None
        self._edge_score = None

    @property
    def edge_score(self):
        if self._edge_score is None:
            self._edge_score = get_edge_scores((self.edge,),
                                               self.predicted_edge)[0]
        return self._edge_score

    def _calculate_properties(self):
        if self.params.fit_radial:
            coords, edge = mask_to_knots(self.mask,
                                         p_edge=self.predicted_edge,
                                         **self.params._asdict())
            self.mask = ndimage.binary_fill_holes(edge)
        else:
            edge = binary_edge(self.mask) | (self.border_rect & self.mask)
            coords = tuple()
        self._coords = coords
        self._edge = edge

    @property
    def edge(self):
        if self._edge is None:
            self._calculate_properties()
        return self._edge

    @property
    def coords(self):
        if self._coords is None:
            self._calculate_properties()
        return self._coords

    @property
    def volume(self, method='conical'):
        return volume(self.edge, method=method)


class Target:
    def __init__(self, name, flattener, desired_targets=None):
        self.name = name
        self.flattener = flattener
        self._definition = None
        self._available_targets = dict()

        if desired_targets is None:
            desired_targets = ['interior', 'filled', 'edge', 'overlap']
        self.__desired_targets = desired_targets

    @property
    def available_targets(self):
        if not self._available_targets:
            self._calculate_targets()
        return self._available_targets

    def __getitem__(self, item):
        if item in self.available_targets and self.available_targets[item] \
                is None:
            self._calculate_targets()
        return self.available_targets[item]

    def __contains__(self, item):
        return item in self.available_targets

    def _calculate_targets(self):
        targets = self.flattener.getGroupTargets(self.name,
                                                 self.__desired_targets)
        self._available_targets.update(dict(zip(
            self.__desired_targets,
            targets)))

        none_targets = [k for k, v in self._available_targets.items()
                        if v is None]
        for none_target in none_targets:
            del self._available_targets[none_target]

        if 'interior' not in self._available_targets:
            if 'filled' in self._available_targets:
                self._available_targets['interior'] = self._available_targets[
                    'filled']
            else:
                raise ValueError('No interior or filled target specified in '
                                 f'available targets {self.available_targets}')

    @property
    def definition(self):
        if self._definition is None:
            self._definition = self.flattener.getTargetDef(
                self['interior'])
        return self._definition

    def prediction(self, pred, target_name):
        if target_name in self:
            result = pred[self.flattener.names().index(self[target_name])]
        else:
            result = np.zeros((pred.shape[1], pred.shape[2]))
        return result


class Group:

    def __init__(self, targets, params):

        self.params = params
        self.targets = targets
        self.cells = []

        # Computed members
        self._n_erode = None
        self._lower = None
        self._upper = None

    def _calculate_bounds(self):
        # get bounds
        if self.params.use_group_thresh:
            lower = min(target.definition.get('lower', 1.) for target in
                        self.targets)
            upper = max(target.definition.get('upper', float('inf')) for
                        target in self.targets)
            expansion = self.params.group_thresh_expansion * (
                lower if upper == float('inf') else upper - lower)
            lower = max(lower - expansion, self.params.min_area)
            upper += expansion
        else:
            lower, upper = self.params.min_area, float('inf')
        self._lower = lower
        self._upper = upper

    @property
    def lower(self):
        if self._lower is None:
            self._calculate_bounds()
        return self._lower

    @property
    def upper(self):
        if self._upper is None:
            self._calculate_bounds()
        return self._upper

    @property
    def n_erode(self):
        if self._n_erode is None:
            self._n_erode = [target.definition.get('nerode', 0)
                             for target in self.targets]
        return self._n_erode

    @property
    def max_n_erode(self):
        max_n_erode = max(self.n_erode)
        if self.params.edge_sub_dilations is not None and max_n_erode == 0:
            return 1
        else:
            return max_n_erode  # Todo: add number of dilations?

    @property
    def n_cells(self):
        return len(self.cells)

    @property
    def interior_threshold(self):
        return self.params.interior_threshold or 0.5

    def prediction(self, pred, target_name, erode=False):
        predictions = [target.prediction(pred, target_name)
                       for target in self.targets]
        if erode:
            predictions = self.erode_predictions(predictions)
        result = np.dstack(predictions)
        return result.max(axis=2)

    def erode_predictions(self, predictions: List) -> List:
        # Perform morphological erosion on any group members that do not
        # match the maximum (NB: erosion here is non-binary):
        predictions = [iterative_erosion(p, self.max_n_erode - n_erode)
                       for p, n_erode in zip(predictions, self.n_erode)]
        return predictions

    def segment(self, pred, border_rect):
        """Find cell instances for this group from interior probability

        Given raw CNN output ``pred``, extract the interior probability image
        defined for this group, then apply threshold segmentation as per
        :py:func:`segmentation.threshold_segmentation` and create a
        :py:class:`Cell` for each identified mask. Depending on the
        :py:class:`SegmentationParameters`, some filtering by mask area and/or
        probability of the mask edge on the edge probability image may be
        applied.
        
        Args:
            pred (ndarray): The neural network's prediction with shape
                (targets, X, Y, 1).
            border_rect (ndarray): An image of shape (X, Y) which is False
                except for a one-pixel border of True values.
        """

        # TODO this is the bit where the use of the overlap informations
        #  from morph_thresh_masks should be added
        pred_interior = self.prediction(pred, 'interior', erode=True)
        pred_edge = self.prediction(pred, 'edge', erode=False)

        if self.params.edge_sub_dilations is not None:
            pred_edge = iterative_dilation(pred_edge,
                                           self.params.edge_sub_dilations)
            pred_interior *= (1 - pred_edge)

        masks_areas = [
            (m, a) for m, a in threshold_segmentation(
                pred_interior,
                interior_threshold=self.params.interior_threshold or 0.5,
                nclosing=self.params.nclosing or 0,
                nopening=self.params.nopening or 0,
                ndilate=self.max_n_erode,
                return_area=True,
                connectivity=self.params.connectivity)
            if self.lower <= a < self.upper
        ]
        self.cells = [
            Cell(a, m, pred_edge, border_rect, self.params)
            for m, a in masks_areas
        ]
        # Remove cells that do not exceed the p_edge threshold
        if self.params.pedge_thresh is not None:
            self.cells = [cell for cell in self.cells
                          if cell.edge_score > self.params.pedge_thresh]


def broadcast_arg(arg: Union[Sequence, Any],
                  argname: Optional[str] = None,
                  n_groups: Optional[int] = 3):
    if argname is None:
        argname = 'Argument'
    try:
        if len(arg) != n_groups:
            raise ValueError(f'"{argname}" is of incorrect length')
        else:
            return arg
    except TypeError:
        return [arg] * n_groups



class MorphSegGrouped:
    """Provides an instance segmentation method given CNN target definitions.

    Args:
        flattener (preprocessing.SegmentationFlattening): Target definitions
            for the CNN predictions that will be used as input to
            :py:meth:`segment`.
        params (SegmentationParameters): Parameters used for segmentation. See
            :py:class:`SegmentationParameters` for more details.
        return_masks (bool): Whether masks should be returned by
            :py:meth:`segment`.
        return_coords (bool): Whether knot coordinates should be returned by
            :py:meth:`segment`.

    Attributes:
        groups (List[Group]): Segmentation processing/results for each group.
        flattener (preprocessing.SegmentationFlattening): As for the
            ``flattener`` argument.
        params (SegmentationParameters): Parameters used for segmentation
            including any broadcasting for the number of cell groups.
        return_masks: Whether masks should be returned by :py:meth:`segment`.
        return_coords: Whether knot coordinates should be returned by
            :py:meth:`segment`.
    """

    def __init__(self, flattener, params=SegmentationParameters(),
                 return_masks=False, return_coords=False):
        self.flattener = flattener
        self.return_masks = return_masks
        self.return_coords = return_coords

        # TODO: assertions about valid options
        #  (e.g. 0 < interior_threshold < 1)

        # Define group parameters
        cellgroups = params.cellgroups
        if cellgroups is None:
            cellgroups = [t.group for t in flattener.targets
                          if t.prop in {'interior', 'filled'}]
        cellgroups = [(g,) if isinstance(g, str) else g for g in cellgroups]
        n_groups = len(cellgroups)

        # Broadcast relevant parameters
        params = params._replace(**{
            p: broadcast_arg(getattr(params, p), p, n_groups)
            for p in BROADCASTABLE_PARAMS
        })

        # Minimum area must be larger than 1 to avoid generating cells with
        # no size:
        params = params._replace(
            min_area=[np.max([a, 1]) for a in params.min_area])

        self.params = params

        self.containment_func = getattr(segmentation, params.containment_func)
        self.containment_thresh = params.containment_thresh

        # Initialize the different groups and their targets
        self.groups = []
        for i, target_names in enumerate(cellgroups):
            targets = [Target(name, flattener) for name in target_names]
            self.groups.append(Group(targets, params=params._replace(**{
                p: getattr(params, p)[i] for p in BROADCASTABLE_PARAMS
            })))

    # Todo: This is ideally the form of the input argument
    def contains(self, a, b):
        return self.containment_func(a, b) > self.containment_thresh

    def remove_duplicates(self):
        """Resolve any cells duplicated across adjacent groups.
        """

        if all([t is None for t in self.params.pedge_thresh]):
            def accessor(cell):
                return cell.area
        else:
            def accessor(cell):
                return cell.edge_score

        # def accessor(group):
        #     return getattr(group, 'a' if self.pedge_thresh else 'b')

        for lower_group, upper_group in zip(self.groups,
                                            self.groups[1:]):
            pairs = [(lower, upper)
                     for lower in lower_group.cells
                     for upper in upper_group.cells
                     if self.contains(lower.mask, upper.mask)]

            for lower, upper in pairs:
                # Check that one of the cells hasn't already been removed in
                # previous pairs
                if lower in lower_group.cells and upper in upper_group.cells:
                    if accessor(lower) < accessor(upper):
                        lower_group.cells.remove(lower)
                    else:
                        upper_group.cells.remove(upper)

    # Todo: rename
    def extract_edges(self, pred, shape, refine_outlines, return_volume):
        masks = [[]]
        if refine_outlines:
            if not self.params.fit_radial:
                raise BadParam(
                    '"refine_outlines" requires "fit_radial" to have been specified'
                )
            # Refine outlines using edge predictions
            grouped_coords = [[cell.coords for cell in group.cells]
                              for group in self.groups]
            predicted_edges = [group.prediction(pred, 'edge') for group in
                               self.groups]

            if predicted_edges:
                coords = list(itertools.chain.from_iterable(
                    refine_radial_grouped(
                        grouped_coords, predicted_edges,
                        cartesian_spline=self.params.cartesian_spline)))
            else:
                coords = tuple()
            edges = [draw_radial(
                        radii, angles, centre, shape,
                        cartesian_spline=self.params.cartesian_spline)
                     for centre, radii, angles in coords]
            if self.return_masks:
                masks = [ndimage.binary_fill_holes(e) for e in edges]
            if return_volume:
                volumes = [volume(edge, method='conical') for edge in edges]
                return edges, masks, coords, volumes
            return edges, masks, coords
        else:
            # Extract edges, masks and AC coordinates from initial segmentation
            if return_volume:
                outputs = [(cell.edge, cell.mask, cell.coords, cell.volume)
                           for group in self.groups
                           for cell in group.cells]
            else:
                outputs = [(cell.edge, cell.mask, cell.coords)
                           for group in self.groups
                           for cell in group.cells]

            if len(outputs) > 0:
                return (list(o) for o in zip(*outputs))
            else:
                return 4 * [[]] if return_volume else 3 * [[]]

    def segment(self, pred, refine_outlines=False, return_volume=False):
        """Returns segmented instances based on the output of the CNN.

        Args:
            pred: A list of prediction images (ndarray with shape (x, y))
                matching the names of :py:attr:`flattener` (see
                :py:meth:`preprocessing.SegmentationFlattening.names`).

        Returns:
            A list of boolean edge images (ndarray shape (x, y)), one for each
            cell identified. If ``return_masks`` and/or ``return_coords`` are
            true, the output will be a tuple of edge images, filled masks,
            and/or radial coordinates.
        """
        if len(pred) != len(self.flattener.names()):
            raise BadParam(
                '"pred" arg does not match number of flattener targets')
        shape = np.squeeze(pred[0]).shape
        border_rect = np.pad(
            np.zeros(tuple(x - 2 for x in shape), dtype='bool'),
            pad_width=1, mode='constant',
            constant_values=True)

        for group in self.groups:
            group.segment(pred, border_rect)

        # Remove cells that are duplicated in several groups
        self.remove_duplicates()
        # edges, masks, coords, volumes = \
        result = self.extract_edges(pred, shape,
                                    refine_outlines=refine_outlines,
                                    return_volume=return_volume)

        # Todo: return_masks and return_coords seem useless as always set
        #  necessary for brain.segment and tracker
        output = SegmentationOutput(*result)
        # [True,
        #  self.return_masks,
        #  self.return_coords,
        #  self.return_volume]))

        return output
        # if len(output) > 1:
        #     return output
        # else:
        #     return output[0]


class SegmentationOutput(NamedTuple):
    edges: list
    masks: list = []
    coords: list = []
    volume: list = []
