from datetime import datetime
from typing import List, Optional

from pydantic import BaseModel, Field, field_validator


class Currency(BaseModel):
    currencyCodeA: int
    currencyCodeB: int
    date: int
    rateSell: float = None
    rateBuy: float = None
    rateCross: float = None

    def utc_date(self):
        return datetime.utcfromtimestamp(self.date)


class Account(BaseModel):
    id: str
    sendId: str
    balance: float
    credit_limit: float = Field(validation_alias='creditLimit')
    type: str
    currency_code: int = Field(validation_alias='currencyCode')
    cashback_type: str = Field(validation_alias='cashbackType')
    masked_pan: Optional[List[str]] = Field(validation_alias='maskedPan')
    iban: Optional[str] = None

    @field_validator('balance', 'credit_limit', mode='before')
    @classmethod
    def convert_to_money(cls, value) -> float:
        return value / 100


class ClientInfo(BaseModel):
    id: str = Field(validation_alias='clientId')
    name: str
    webhook_url: str = Field(validation_alias='webHookUrl')
    permissions: str
    accounts: List[Account]


class StatementItem(BaseModel):
    id: str
    time: int
    description: str
    mcc: int
    hold: bool
    amount: float
    operation_amount: float = Field(validation_alias='operationAmount')
    currency_code: int = Field(validation_alias='currencyCode')
    commission_rate: float = Field(validation_alias='commissionRate')
    cashback_amount: float = Field(validation_alias='cashbackAmount')
    balance: float
    comment: str = None
    receipt_id: str = Field(default=None, validation_alias='receiptId')
    counter_edrpou: str = Field(default=None, validation_alias='counterEdrpou')
    counter_iban: str = Field(default=None, validation_alias='counterIban')

    @field_validator('balance', 'amount', 'operation_amount', 'commission_rate', 'cashback_amount', mode='before')
    @classmethod
    def convert_to_money(cls, value) -> float:
        return value / 100

    class Config:
        fields = {
            'operation_amount': 'operationAmount',
            'currency_code': 'currencyCode',
            'commission_rate': 'commissionRate',
            'cashback_amount': 'cashbackAmount',
            'receipt_id': 'receiptId',
            'counter_edrpou': 'counterEdrpou',
            'counter_iban': 'counterIban',
        }
