import click
import sys
import subprocess
import importlib.metadata
from .config import init_config, load_config
from .deps import resolve_dependencies
from .security import check_security
from .licenses import check_license_compliance
from .approvals import is_approved, approve_package
from .keys import generate_key_pair, load_private_key, load_public_key, sign_data, verify_data

def install_package(package_spec):
    """Install the specified package using pip."""
    try:
        click.echo(f"Predator Drone: Attempting to install: {package_spec}")
        result = subprocess.run(
            ["pip", "install", package_spec],
            check=True,
            capture_output=True,
            text=True
        )
        click.echo(f"Predator Drone: pip install output: {result.stdout}")
    except subprocess.CalledProcessError as e:
        click.echo(f"Predator Drone: pip install error: {e.stderr}", err=True)
        raise Exception(f"Failed to install {package_spec}: {e}")

@click.group()
def cli():
    """OceanCrow_PDG: Python Dependency Governance Tool."""
    pass

@cli.command()
def init():
    """Initialize the configuration and generate key pair."""
    passphrase = click.prompt("Enter passphrase", hide_input=True)
    confirm_passphrase = click.prompt("Repeat for confirmation", hide_input=True)
    if passphrase != confirm_passphrase:
        raise ValueError("Passphrases do not match.")
    init_config()
    generate_key_pair(passphrase)
    click.echo("Predator Drone: Configuration and keys initialized.")

@cli.command()
@click.argument("package_spec")
def review(package_spec):
    """Review the specified package for dependencies, security, and licenses."""
    try:
        package_name = package_spec.split("==")[0]
        try:
            importlib.metadata.version(package_name)
        except importlib.metadata.PackageNotFoundError:
            click.echo(f"Predator Drone Warning: Package {package_name} not found in environment", err=True)
            return
        config = load_config()
        deps = resolve_dependencies(package_spec)
        if not deps or all(not d for d in deps):
            click.echo("Predator Drone Warning: No valid dependencies resolved", err=True)
            deps = [package_spec]
        vulnerabilities = check_security(deps) if deps else []
        compliant = check_license_compliance(deps) if deps else True
        click.echo(f"Predator Drone: Dependencies: {deps}")
        click.echo(f"Predator Drone: Vulnerabilities: {vulnerabilities}")
        click.echo(f"Predator Drone: License Compliance: {compliant}")
    except Exception as e:
        click.echo(f"Predator Drone: Error reviewing package: {e}", err=True)
        sys.exit(1)

@cli.command()
@click.argument("package_spec")
def approve(package_spec):
    """Approve the specified package version."""
    try:
        if "==" not in package_spec:
            raise ValueError("Predator Drone Warning: Invalid package specification (e.g., 'name==version')")
        passphrase = click.prompt("Enter passphrase", hide_input=True)
        private_key = load_private_key(passphrase)
        name, version = package_spec.split("==")
        approve_package(name, version, private_key)
        click.echo(f"Predator Drone: Approved {package_spec}")
    except Exception as e:
        click.echo(f"Predator Drone: Error approving package: {e}", err=True)
        sys.exit(1)

@cli.command()
@click.argument("package_spec")
def install(package_spec):
    """Install the specified package if approved."""
    try:
        if not package_spec or "==" not in package_spec:
            raise ValueError("Predator Drone Warning: Invalid package specification (e.g., 'name==version')")
        passphrase = click.prompt("Enter passphrase", hide_input=True)
        load_private_key(passphrase)
        public_key = load_public_key()
        name, version = package_spec.split("==")
        if is_approved(name, version, public_key):
            install_package(package_spec)
            click.echo(f"Predator Drone: Installed {package_spec}")
        else:
            click.echo(f"Predator Drone: {package_spec} not approved.", err=True)
            sys.exit(1)
    except Exception as e:
        click.echo(f"Predator Drone: Error installing package: {e}", err=True)
        sys.exit(1)

if __name__ == "__main__":
    cli() # Invoke the CLI group