import subprocess
import os
import click
import importlib.metadata

def resolve_dependencies(package_spec):
    """Resolve dependency tree for a given package specification with enhanced oversight."""
    try:
        pip_tools_version = importlib.metadata.version("pip-tools")
        click.echo(f"Predator Drone: Using pip-tools version: {pip_tools_version}", err=True)
        click.echo(f"Predator Drone: Processing package spec: {package_spec}", err=True)
        
        with open("temp.in", "w") as f:
            f.write(f"{package_spec}\n")
        result = subprocess.run(
            ["pip-compile", "--output-file=temp.out", "temp.in"],
            capture_output=True, text=True
        )
        os.remove("temp.in")
        if os.path.exists("temp.out"):
            with open("temp.out", "r") as f:
                deps = [line.strip() for line in f if line.strip() and not line.startswith("#")]
            os.remove("temp.out")
            if not deps:
                click.echo("Predator Drone Warning: No dependencies resolved by pip-compile", err=True)
            else:
                click.echo(f"Predator Drone: pip-compile deps: {deps}", err=True)
                valid_deps = [d for d in deps if "==" in d and not d.startswith("#")]
                if not valid_deps:
                    click.echo("Predator Drone Warning: No valid dependencies after filtering", err=True)
                    return [package_spec]
                return valid_deps
        if result.returncode != 0:
            click.echo(f"Predator Drone: pip-compile error: {result.stderr}", err=True)
            package_name = package_spec.split("==")[0]
            show_result = subprocess.run(["pip", "show", package_name], capture_output=True, text=True)
            if show_result.returncode == 0:
                for line in show_result.stdout.splitlines():
                    if line.startswith("Version: "):
                        version = line.split("Version: ")[1].strip()
                        click.echo(f"Predator Drone: pip show fallback: {package_name}=={version}", err=True)
                        return [f"{package_name}=={version}"]
        click.echo("Predator Drone Warning: Falling back to input package spec", err=True)
        return [package_spec]

    except Exception as e:
        click.echo(f"Predator Drone: Dependency resolution failed: {e}", err=True)
        return [package_spec]