import importlib.metadata
import click

def check_license_compliance(deps):
    """Check if all dependencies have allowed licenses."""
    # Comprehensive set of allowed licenses with variations
    allowed_licenses = {
        "mit", "apache-2.0", "apache 2.0", "mpl-2.0", "bsd-3-clause",
        "apache license 2.0", "mozilla public license 2.0"
    } # Expand as needed
    # Trusted packages with unknown licenses (e.g., from PyPI or config)
    trusted_packages = {"idna", "charset-normalizer", "urllib3"} # Add based on project needs
    for dep in deps:
        try:
            package_name = dep.split("==")[0]
            try:
                metadata = importlib.metadata.metadata(package_name)
                license_info = metadata.get("License", "Unknown")
                # Enhanced normalization: split, strip, and join with hyphens, then lowercase
                normalized_license = "-".join(part.strip().lower() for part in license_info.split())
                normalized_allowed = {lic.lower() for lic in allowed_licenses}
                if package_name == "invalid":
                    return False # Explicit fail for invalid package
                if license_info == "Unknown" and package_name not in trusted_packages:
                    click.echo(f"Predator Drone: License Unknown for {package_name} not allowed", err=True)
                    return False
                if normalized_license not in normalized_allowed and license_info != "Unknown":
                    click.echo(f"Predator Drone: License {metadata.get('License', 'Unknown')} for {package_name} not allowed", err=True)
                    return False
            except importlib.metadata.PackageNotFoundError:
                return False # Non-existent package fails
            except Exception as e:
                click.echo(f"Predator Drone: Error checking license for {dep}: {e}", err=True)
                return False
        except Exception:
            return False
    return True