import subprocess
import click
import json
from .uav import check_web_updates

def check_security(deps):
    """Check the security of the given dependencies using Bandit and web updates."""
    if not deps or not all(isinstance(d, str) and "==" in d for d in deps):
        click.echo("Predator Drone Warning: Invalid dependency list for security check", err=True)
        return []
    try:
        click.echo(f"Predator Drone: Checking security for deps: {deps}", err=True)
        result = subprocess.run(
            ["bandit", "-r", ".", "-f", "json", "-o", "bandit_report.json"],
            capture_output=True, text=True
        )
        web_updates = check_web_updates(deps)
        if result.returncode != 0:
            click.echo(f"Predator Drone: Security check found issues: {result.stderr}", err=True)
            try:
                with open("bandit_report.json", "r") as f:
                    report = json.load(f)
                vulnerabilities = [f"Note: {u}" for u in web_updates] if web_updates else [] # Only keep update notes
                return vulnerabilities
            except Exception as e:
                click.echo(f"Predator Drone: Failed to parse Bandit report: {e}", err=True)
                return ["Bandit detected issues—check bandit_report.json"] + ([f"Note: {u}" for u in web_updates] if web_updates else [])
        click.echo("Predator Drone: Security check: No vulnerabilities found", err=True)
        return [f"Note: {u}" for u in web_updates] if web_updates else []
    except Exception as e:
        click.echo(f"Predator Drone: Security check failed: {e}", err=True)
        return []