import concurrent.futures as cf
import glob
import logging
import re
import zipfile
from collections.abc import Iterable
from pathlib import Path
from typing import Optional, List, Tuple, Set

from locuaz.fileutils import DirHandle, FileHandle, catenate_pdbs
from locuaz.gromacsutils import fix_gromacs_pdb


def extract_pdbs(
        zip_file: Path,
        out_prefix: str,
        *,
        nprocs: int = 1,
        new_chainID: Optional[str] = None,
        start: int,
        end: int,
        allowed_nonstandard_residues: Set,
        delete_zip_file: bool = True,
        log: Optional[logging.Logger] = None,
) -> Tuple[int, int]:
    """extract_pdbs Extracts `zip_file`, fixes the PDBs and, optionally, deletes
    the original zipfile.

    Args:
        zip_file (Path): zip file with a bunch of PDBs named "output{i}.pdb"
            generated by trjconv.
        out_prefix (str): prefix for each PDB file. Will be followed by its frame number.
        new_chainID (Optional[str], optional): new chainID for the full PDB. Defaults to None.
        delete_zip_file (bool, optional):. Defaults to True.

    Returns:
        nframes(int): number of PDBs in the input `zip_file`.
    """
    zip_file = Path(zip_file)
    current_dir = zip_file.parent
    zipped_pdbs = zipfile.ZipFile(zip_file)
    regex_get_number = r"\d+"
    nframes = 1 + max(
        [
            int(re.findall(regex_get_number, nombre.filename)[0])
            for nombre in zipped_pdbs.filelist
        ]
    )
    end = nframes if end == -1 else end
    assert (end - start) <= nframes, f"Bad start ({start}) and end ({end}) frames. " \
                                     f"Only got {nframes} out of the trajectory"

    with zipped_pdbs as sipesipe:
        sipesipe.extractall(current_dir)

    with cf.ProcessPoolExecutor(max_workers=nprocs) as ex:
        futuros: List[cf.Future] = []
        for i in range(nframes):
            old_pdb = Path(current_dir / (f"output{i}.pdb"))
            new_pdb = Path(current_dir / (f"{out_prefix}-{i}.pdb"))

            if i >= start and i < end:
                futuros.append(
                    ex.submit(fix_gromacs_pdb, old_pdb, new_pdb, new_chainID=new_chainID,
                              allowed_nonstandard_residues = allowed_nonstandard_residues)
                )
            else:
                # Discard this frame
                old_pdb.unlink()

        for futu in cf.as_completed(futuros):
            if futu.exception():
                if log:
                    log.error("Exception while fixing PDBs for scoring.")
                raise RuntimeError(futu.exception())
            old_pdb = futu.result()
            old_pdb.unlink()

    if delete_zip_file:
        zip_file.unlink()

    return start, end


def join_target_binder(
        pdbs_path: Path,
        *,
        start: int,
        end: int,
        tar: str = "target",
        bin: str = "binder",
        cpx: str = "complex",
) -> None:
    for i in range(start, end):
        tmp_fn = pdbs_path / f"tmp-{i}.pdb"
        catenate_pdbs(
            tmp_fn,
            FileHandle(pdbs_path / f"{tar}-{i}.pdb"),
            FileHandle(pdbs_path / f"{bin}-{i}.pdb"),
        )
        fix_gromacs_pdb(tmp_fn, pdbs_path / f"{cpx}-{i}.pdb")
        tmp_fn.unlink()


def rm_aux_scoring_files(
        frames_path: DirHandle, scorers: Iterable, *, start: int, end: int
) -> None:
    for i in range(start, end):
        Path(frames_path, f"target-{i}.pdb").unlink()
        Path(frames_path, f"binder-{i}.pdb").unlink()
        Path(frames_path, f"complex-{i}.pdb").unlink()

    if "bluues" in scorers:
        bluues_dir = frames_path / "bluues"
        for solv_nrg_file in glob.glob(str(Path(bluues_dir, "*solv_nrg"))):
            Path(solv_nrg_file).unlink()
        for pqr_file in glob.glob(str(Path(bluues_dir, "*pqr"))):
            Path(pqr_file).unlink()
        for log_file in glob.glob(str(Path(bluues_dir, "*log"))):
            Path(log_file).unlink()

    if "haddock" in scorers:
        haddock_dir = frames_path / "haddock"
        for pdb_file in glob.glob(str(Path(haddock_dir, "*pdb"))):
            Path(pdb_file).unlink()
        for psf_file in glob.glob(str(Path(haddock_dir, "*psf"))):
            Path(psf_file).unlink()
        for out_file in glob.glob(str(Path(haddock_dir, "*out"))):
            Path(out_file).unlink()
        for inp_file in glob.glob(str(Path(haddock_dir, "*inp"))):
            Path(inp_file).unlink()

    if "gmxmmpbsa" in scorers:
        haddock_dir = frames_path / "gmxmmpbsa"
        for xtc_file in glob.glob(str(Path(haddock_dir, "*xtc"))):
            Path(xtc_file).unlink()
        for mdcrd_file in glob.glob(str(Path(haddock_dir, "*mdcrd*"))):
            Path(mdcrd_file).unlink()
