# -*- coding: utf-8 -*-
"""Configuration options for Azure discovery migration."""
import ipaddress
import json
import pathlib
import typing as t

import yaml
import pydantic
from pydantic import constr  # Constrained string type.
from pydantic import Field, ValidationError, root_validator, validator
from naterminator.core.Globals import Globals

if not hasattr(t, "Literal"):
    from typing_extensions import Literal

    t.Literal = Literal


CIDRList = t.List[ipaddress.IPv4Network]
Tag = t.Dict[str, str]
Tags = t.List[Tag]
_str = constr(strip_whitespace=True)


def _default_network() -> CIDRList:
    return [ipaddress.IPv4Network("0.0.0.0/0")]

class _BaseModel(pydantic.BaseModel):
    discovery_only: t.ClassVar[bool] = False

    class Config:
        json_encoders = {
            ipaddress.IPv4Address: str,
            ipaddress.IPv4Network: str,
        }
        extra = "forbid"

class ModeIdConfig(pydantic.BaseModel):
    id: _str

    @validator("id")
    def validate_id(cls, v):
        if not v in Globals.ModeIds:
            raise ValueError(f"id must be one of {Globals.ModeIds}")
        return v
    
class WorkflowAccountConfig(_BaseModel):
    iam_prefix: t.Optional[_str] = None
    role_name: _str
    aws_account_id: t.Optional[_str] = None
    region: _str

class ControllerConfig(_BaseModel):
    controller_ip: ipaddress.IPv4Address
    controller_user: _str
    controller_password_env: _str
    aws_account_name: _str

class TagConfig(_BaseModel):
    key: _str
    value: _str

class TestConfig(_BaseModel):
    input_test_config: t.Optional[_str] = "test.config"
    output_test_config: t.Optional[_str] = "config.json"
    enable_test: t.Optional[bool] = False

class VpcConfig(_BaseModel):
    vpc_id: _str
    aws_account_id: _str
    iam_prefix: t.Optional[_str] = ""
    role_name: _str
    spoke_gw_name: _str
    spoke_gw_size: t.Optional[_str] = None
    region: _str
    tag_route_table: t.Optional[TagConfig] = None

class BackupConfig(_BaseModel):
    backup_folder: t.Optional[_str] = "./backups"
    workflow_account: t.Optional[WorkflowAccountConfig] = None
    s3_bucket: _str

class ReplaceNatGwWithNewEipConfig(ModeIdConfig):
    hs_mode: t.Optional[int] = 2
    controller: ControllerConfig
    test: TestConfig = Field(default_factory=TestConfig)
    vpc: VpcConfig
    horizontal_scaling: t.Optional[_str] = 2
    backup: BackupConfig
    log_output_path: t.Optional[_str] = "./"


def load_from_dict(config_dict: t.Dict):
    """Load discovery migration settings from a python dictionary.

    Args:
        config_dict: Python dictionary in which to load configuration
            settings from.

    Returns:
        Parsed discovery migration settings.
    """

    try:
        config = None
        mode = ModeIdConfig(**config_dict)
        if mode.id in [ "replace_natgw_with_new_eip", "replace_natgw_retain_eip_no_tmp_natgw" ]:
            config = ReplaceNatGwWithNewEipConfig(**config_dict)
    except ValidationError as e:
        print(e.json())
        raise SystemExit(1) from e
    if config != None:
        config = dump_to_dict(config)
    return config


def dump_to_dict(config) -> t.Dict:
    """Dump discovery migration settings to a python dictionary.

    Args:
        config: Discovery migration settings.

    Returns:
        Configuration dictionary.
    """
    json_data = config.json()
    data = json.loads(json_data)

    return data


def load_from_yaml(yml_path: pathlib.Path, discovery_only: bool = False):
    """Load discovery migration settings from a yaml.

    Args:
        yml_path: Path to location of discovery migration yaml.

    Returns:
        Parsed discovery migration settings.
    """
    with open(yml_path, "r") as fh:
        data = yaml.load(fh, Loader=yaml.FullLoader)
    
    return load_from_dict(data)


def dump_to_yaml(config, dest: pathlib.Path) -> pathlib.Path:
    """Dump discovery migration settings to a yaml file.

    Args:
        config: Discovery migration settings.
        dest: Path to destination location of discovery migration yaml.

    Returns:
        Path to destination location of discovery migration yaml.
    """
