import base64
from io import BytesIO
from typing import Any, Union, Optional

import PIL.Image
import PIL.ImageDraw
import aiohttp
from aiohttp import ClientResponseError
from cachetools import TTLCache
from qrcode_styled import QRCodeStyled

# Initialize cache for storing generated QR codes for 24 hours
CACHE: TTLCache = TTLCache(maxsize=10_000, ttl=86000)

DEFAULT_WALLET_IMAGE = "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"  # noqa


async def download_image_from_url(image_url: str) -> BytesIO:
    """
    Downloads an image from a given URL.

    :param image_url: URL of the image
    :return: BytesIO object containing the image data
    :raises ValueError: If image download fails, content type is not an image, or image format is unsupported
    """
    # Check if the image is already in the cache
    image_cache = CACHE.get(image_url)
    if image_cache:
        return BytesIO(image_cache)

    try:
        async with aiohttp.ClientSession() as session:
            async with session.get(image_url) as response:
                response.raise_for_status()

                # Check if the content type is an image
                content_type = response.headers.get('Content-Type', '').lower()

                # Check if the image format is supported (jpeg or png)
                if not content_type.startswith('image/jpeg') and not content_type.startswith('image/png'):
                    raise ValueError(f"Unsupported image format: {content_type}")

                image_data = await response.read()
                CACHE.setdefault(image_url, image_data)
                return BytesIO(image_data)

    except ClientResponseError:
        # Return default wallet image in case of download error
        return BytesIO(base64.b64decode(DEFAULT_WALLET_IMAGE, validate=True))

    except Exception as e:
        raise ValueError(f"Failed to download image from URL: {image_url}. Error: {e}")


async def generate_qrcode(
        data: str,
        border: int,
        box_size: int,
        image_url: Union[str, None] = None,
        image_padding: int = 10,
        image_round: int = 50,
) -> bytes:
    """
    Generates a styled QR code from data with an optional image inclusion.

    :param data: Data to be encoded in the QR code
    :param border: Border size of the QR code
    :param box_size: Size of each box in the QR code
    :param image_url: URL of the image to be included in the QR code (optional)
    :param image_padding: Padding around the optional image in the QR code (optional)
    :param image_round: Radius for rounding corners of the optional image in the QR code (optional)
    :return: BytesIO object containing the generated QR code image
    :raises ValueError: If there is an error in generating the QR code
    """
    try:
        image_id = f"{data}_{border}_{box_size}_{image_url}_{image_padding}_{image_round}"
        # Check if the image URL is in the cache
        if image_id in CACHE:
            return CACHE.get(image_id)  # type: ignore

        # Download and process the optional image
        logo_image_stream = await download_image_from_url(image_url) if image_url else None
        padded_image = process_optional_image(logo_image_stream, image_padding, image_round)

        # Generate the QR code using qrcode_styled library
        qr = QRCodeStyled(border=border, box_size=box_size)
        qrcode_image = await qr.get_buffer_async(data=data, image=padded_image)
        qrcode_image_data = qrcode_image.getvalue()

        # Cache the generated QR code
        CACHE.setdefault(image_id, qrcode_image_data)

        return qrcode_image_data

    except (Exception,):
        raise


def process_optional_image(
        image_stream: Optional[BytesIO],
        image_padding: int = 10,
        image_round: int = 50,
) -> Any:
    """
    Processes an optional image to be included in the QR code.

    :param image_stream: BytesIO object containing the image data
    :param image_padding: Padding around the optional image in the QR code (optional)
    :param image_round: Radius for rounding corners of the optional image in the QR code (optional)
    :return: Processed Image object
    """
    if image_stream:
        logo_image = PIL.Image.open(image_stream).convert("RGBA")

        # If the image has an alpha channel (transparency), replace transparent areas with white
        if logo_image.mode == 'RGBA':
            alpha = logo_image.split()[3]
            white_background = PIL.Image.new("RGBA", logo_image.size, (255, 255, 255, 255))
            white_background.paste(logo_image, (0, 0), alpha)
            logo_image = white_background

        # Create a rounded rectangle mask for the logo image
        mask = PIL.Image.new("L", logo_image.size, 0)
        draw = PIL.ImageDraw.Draw(mask)
        draw.rounded_rectangle((0, 0, logo_image.width, logo_image.height), image_round, fill=255)

        # Apply the mask to the logo image
        logo_image.putalpha(mask)
        logo_image = PIL.Image.alpha_composite(PIL.Image.new("RGBA", logo_image.size, (255, 255, 255, 255)), logo_image)

        # Add padding to the logo image
        padded_size = (logo_image.width + 2 * image_padding, logo_image.height + 2 * image_padding)
        padded_image = PIL.Image.new("RGBA", padded_size, (255, 255, 255, 255))
        padded_image.paste(logo_image, (image_padding, image_padding))

        return padded_image

    return None
