"""
Main entry point for updating the football dataset from command line.
This script orchestrates the dataset update process by calling update_dataset.py
"""

import argparse
import sys
import logging
from typing import Optional
import pandas as pd

from utils.config import get_settings
from updating.update_dataset import DatasetUpdater
from predicting.predicting import Predicting

# Setup logging
logging.basicConfig(
    level=logging.INFO, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s"
)
logger = logging.getLogger(__name__)


def data_update_dataset(
    competition: Optional[str] = None, date: Optional[str] = None
) -> pd.DataFrame:
    """
    Fetch latest matches (including future matches), refresh rankings/odds,
    append to dataset and build features only for those rows.

    Args:
        competition: Specific competition to update (None for all competitions)
        date: Date to process (format: YYYY-MM-DD). If None, uses today.
              Must be Tuesday or Friday.

    Returns:
        DataFrame with updated match data and features

    Raises:
        ValueError: If the date is not a Tuesday or Friday
    """
    # Get settings from configuration
    s = get_settings()

    # Initialize DatasetUpdater with settings
    calc_data = DatasetUpdater(
        circular_max_values=s.circular_max_values,
        features_config=s.features_config,
        competitions_config=s.competitions_config,
        paths=s.paths,
    )

    # Run the update
    calc_data.update_dataset(date=date, competition=competition)

    # Initialize Predicting with settings
    calc_predicting = Predicting(
        circular_max_values=s.circular_max_values,
        features_config=s.features_config,
        competitions_config=s.competitions_config,
        paths=s.paths,
    )

    # Run the update
    calc_predicting.update_dataset(date=date, competition=competition)


def main():
    """
    Main entry point for command-line execution.
    Parses arguments and calls data_update_dataset.
    """
    parser = argparse.ArgumentParser(
        description="Update football dataset with latest matches",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Update all competitions for today (must be Tuesday or Friday)
  python main.py
  
  # Update specific competition
  python main.py --competition spanish_league
  python main.py -c spanish_league
  
  # Update with specific date (must be Tuesday or Friday)
  python main.py --date 2025-10-10
  python main.py -d 2025-10-10
  
  # Update specific competition on specific date
  python main.py --competition spanish_league --date 2025-10-10
  python main.py -c spanish_league -d 2025-10-10
        """,
    )

    parser.add_argument(
        "--competition",
        "-c",
        type=str,
        default=None,
        help="Specific competition to update (e.g., spanish_league). If not provided, updates all competitions.",
    )

    parser.add_argument(
        "--date",
        "-d",
        type=str,
        default=None,
        help="Date to process in YYYY-MM-DD format (e.g., 2025-10-10). Must be Tuesday or Friday. If not provided, uses today.",
    )

    parser.add_argument(
        "--verbose", "-v", action="store_true", help="Enable verbose output"
    )

    parser.add_argument(
        "--list-competitions",
        action="store_true",
        help="List all available competitions and exit",
    )

    args = parser.parse_args()

    # Set logging level based on verbose flag
    if args.verbose:
        logging.getLogger().setLevel(logging.DEBUG)

    # List competitions if requested
    if args.list_competitions:
        s = get_settings()
        print("\n" + "=" * 80)
        print("AVAILABLE COMPETITIONS")
        print("=" * 80)
        for comp_name, comp_config in s.competitions_config.items():
            country = comp_config.get("country", "N/A")
            print(f"  • {comp_name:<30} (Country: {country})")
        print("=" * 80 + "\n")
        return 0

    try:
        print("\n" + "=" * 80)
        print("FOOTBALL DATASET UPDATER")
        print("=" * 80)

        if args.competition:
            print(f"📊 Competition: {args.competition}")
        else:
            print(f"📊 Competition: ALL")

        if args.date:
            print(f"📅 Date: {args.date}")
        else:
            print(f"📅 Date: TODAY")

        print("=" * 80 + "\n")

        # Run the update
        data_update_dataset(competition=args.competition, date=args.date)

        print("\n" + "=" * 80)
        # print(f"🎉 SUCCESS! Updated {len(df)} matches")
        print("=" * 80 + "\n")

        return 0

    except ValueError as e:
        print("\n" + "=" * 80)
        print("❌ VALIDATION ERROR")
        print("=" * 80)
        print(f"{e}")
        print("=" * 80 + "\n")
        logger.error(f"Validation error: {e}")
        return 1

    except KeyboardInterrupt:
        print("\n\n⚠️  Process interrupted by user")
        logger.warning("Process interrupted by user")
        return 1

    except Exception as e:
        print("\n" + "=" * 80)
        print("❌ UNEXPECTED ERROR")
        print("=" * 80)
        print(f"{e}")
        print("=" * 80 + "\n")
        logger.error(f"Unexpected error: {e}", exc_info=True)
        return 1


if __name__ == "__main__":
    sys.exit(main())
