"""
Enrichment parser for LLM-generated enrichment reports.

This module parses Markdown enrichment reports generated by LLMs during
the dual-stack enrichment workflow and applies them to plan bundles.
"""

from __future__ import annotations

import re
from contextlib import suppress
from pathlib import Path
from typing import Any

from beartype import beartype
from icontract import ensure, require

from specfact_cli.models.plan import Feature, PlanBundle, Story


class EnrichmentReport:
    """Parsed enrichment report from LLM."""

    def __init__(self) -> None:
        """Initialize empty enrichment report."""
        self.missing_features: list[dict[str, Any]] = []
        self.confidence_adjustments: dict[str, float] = {}
        self.business_context: dict[str, list[str]] = {
            "priorities": [],
            "constraints": [],
            "unknowns": [],
        }

    @beartype
    @require(lambda feature: isinstance(feature, dict), "Feature must be dictionary")
    def add_missing_feature(self, feature: dict[str, Any]) -> None:
        """Add a missing feature discovered by LLM."""
        self.missing_features.append(feature)

    @beartype
    @require(lambda feature_key: isinstance(feature_key, str), "Feature key must be string")
    @require(lambda confidence: 0.0 <= confidence <= 1.0, "Confidence must be 0.0-1.0")
    def adjust_confidence(self, feature_key: str, confidence: float) -> None:
        """Adjust confidence score for a feature."""
        self.confidence_adjustments[feature_key] = confidence

    @beartype
    @require(lambda category: isinstance(category, str), "Category must be string")
    @require(lambda items: isinstance(items, list), "Items must be list")
    def add_business_context(self, category: str, items: list[str]) -> None:
        """Add business context items."""
        if category in self.business_context:
            self.business_context[category].extend(items)


class EnrichmentParser:
    """Parser for Markdown enrichment reports."""

    @beartype
    @require(
        lambda report_path: isinstance(report_path, (Path, str)) and bool(str(report_path).strip()),
        "Report path must be non-empty Path or str",
    )
    @ensure(lambda result: isinstance(result, EnrichmentReport), "Must return EnrichmentReport")
    def parse(self, report_path: Path | str) -> EnrichmentReport:
        """
        Parse Markdown enrichment report.

        Args:
            report_path: Path to Markdown enrichment report (must be non-empty)

        Returns:
            Parsed EnrichmentReport

        Raises:
            FileNotFoundError: If report file doesn't exist
            ValueError: If report_path is empty or invalid
        """
        report_path = Path(report_path)
        if not str(report_path).strip():
            raise ValueError("Report path cannot be empty")
        if not report_path.exists():
            raise FileNotFoundError(f"Enrichment report not found: {report_path}")
        if report_path.is_dir():
            raise ValueError(f"Report path must be a file, not a directory: {report_path}")

        content = report_path.read_text(encoding="utf-8")
        report = EnrichmentReport()

        # Parse missing features section
        self._parse_missing_features(content, report)

        # Parse confidence adjustments section
        self._parse_confidence_adjustments(content, report)

        # Parse business context section
        self._parse_business_context(content, report)

        return report

    @beartype
    @require(lambda content: isinstance(content, str), "Content must be string")
    @require(lambda report: isinstance(report, EnrichmentReport), "Report must be EnrichmentReport")
    def _parse_missing_features(self, content: str, report: EnrichmentReport) -> None:
        """Parse missing features section from enrichment report."""
        # Look for "Missing Features" or "Missing features" section
        pattern = r"##\s*(?:Missing\s+)?Features?\s*(?:\(.*?\))?\s*\n(.*?)(?=##|\Z)"
        match = re.search(pattern, content, re.IGNORECASE | re.DOTALL)
        if not match:
            return

        section = match.group(1)

        # Extract individual features (numbered or bulleted)
        # Stop at next feature (numbered item at start of line, optionally followed by bold text)
        # This avoids stopping at story numbers which are indented
        feature_pattern = r"(?:^|\n)(?:\d+\.|\*|\-)\s*(.+?)(?=\n(?:^\d+\.\s*\*\*|^\d+\.\s+[A-Z]|\*|\-|\Z))"
        features = re.findall(feature_pattern, section, re.MULTILINE | re.DOTALL)

        for feature_text in features:
            feature = self._parse_feature_block(feature_text)
            if feature:
                report.add_missing_feature(feature)

    @beartype
    @require(lambda feature_text: isinstance(feature_text, str), "Feature text must be string")
    @ensure(lambda result: result is None or isinstance(result, dict), "Must return None or dict")
    def _parse_feature_block(self, feature_text: str) -> dict[str, Any] | None:
        """Parse a single feature block from enrichment report."""
        feature: dict[str, Any] = {
            "key": "",
            "title": "",
            "confidence": 0.5,
            "outcomes": [],
            "stories": [],
        }

        # Extract title first (from bold text: "**Title** (Key: ...)" or "1. **Title** (Key: ...)")
        # Feature text may or may not include the leading number (depends on extraction pattern)
        title_match = re.search(r"^\*\*([^*]+)\*\*", feature_text, re.MULTILINE)
        if not title_match:
            # Try with optional number prefix
            title_match = re.search(r"^\d+\.\s*\*\*([^*]+)\*\*", feature_text, re.MULTILINE)
        if title_match:
            feature["title"] = title_match.group(1).strip()

        # Extract key (e.g., "FEATURE-IDEINTEGRATION" or "(Key: FEATURE-IDEINTEGRATION)")
        # Try parentheses format first: (Key: FEATURE-XXX)
        key_match = re.search(r"\(Key:\s*([A-Z0-9_-]+)\)", feature_text, re.IGNORECASE)
        if not key_match:
            # Try without parentheses: Key: FEATURE-XXX
            key_match = re.search(r"(?:key|Key):\s*([A-Z0-9_-]+)", feature_text, re.IGNORECASE)
        if key_match:
            feature["key"] = key_match.group(1)
        else:
            # Generate key from title if we have one
            if feature["title"]:
                feature["key"] = f"FEATURE-{feature['title'].upper().replace(' ', '').replace('-', '')[:20]}"

        # Extract title from "Title:" keyword if not found in bold text
        if not feature["title"]:
            title_match = re.search(r"(?:title|Title):\s*(.+?)(?:\n|$)", feature_text, re.IGNORECASE)
            if title_match:
                feature["title"] = title_match.group(1).strip()

        # Extract confidence
        confidence_match = re.search(r"(?:confidence|Confidence):\s*([0-9.]+)", feature_text, re.IGNORECASE)
        if confidence_match:
            with suppress(ValueError):
                feature["confidence"] = float(confidence_match.group(1))

        # Extract outcomes (stop at Stories: section to avoid capturing story text)
        outcomes_match = re.search(
            r"(?:outcomes?|Outcomes?):\s*(.+?)(?:\n\s*(?:stories?|Stories?):|\Z)",
            feature_text,
            re.IGNORECASE | re.DOTALL,
        )
        if outcomes_match:
            outcomes_text = outcomes_match.group(1).strip()
            # Split by lines or commas, filter out empty strings and story markers
            outcomes = [
                o.strip()
                for o in re.split(r"\n|,", outcomes_text)
                if o.strip() and not o.strip().startswith("- Stories:")
            ]
            feature["outcomes"] = outcomes

        # Extract business value or reason
        reason_match = re.search(
            r"(?:reason|Reason|Business value):\s*(.+?)(?:\n(?:stories?|Stories?)|$)",
            feature_text,
            re.IGNORECASE | re.DOTALL,
        )
        if reason_match:
            reason = reason_match.group(1).strip()
            if reason and reason not in feature["outcomes"]:
                feature["outcomes"].append(reason)

        # Extract stories (REQUIRED for features to pass promotion validation)
        # Stop at next feature (numbered with bold title) or section header
        stories_match = re.search(
            r"(?:stories?|Stories?):\s*(.+?)(?=\n\d+\.\s*\*\*|\n##|\Z)", feature_text, re.IGNORECASE | re.DOTALL
        )
        if stories_match:
            stories_text = stories_match.group(1).strip()
            stories = self._parse_stories_from_text(stories_text, feature.get("key", ""))
            feature["stories"] = stories

        # Only return if we have at least a key or title
        if feature["key"] or feature["title"]:
            return feature

        return None

    @beartype
    @require(lambda stories_text: isinstance(stories_text, str), "Stories text must be string")
    @require(lambda feature_key: isinstance(feature_key, str), "Feature key must be string")
    @ensure(lambda result: isinstance(result, list), "Must return list of story dicts")
    def _parse_stories_from_text(self, stories_text: str, feature_key: str) -> list[dict[str, Any]]:
        """Parse stories from enrichment report text."""
        stories: list[dict[str, Any]] = []

        # Extract individual stories (numbered, bulleted, or sub-headers)
        # Pattern matches: "1. Story title", "- Story title", "### Story title", etc.
        # Handle indented stories (common in nested lists)
        # Match numbered stories with optional indentation: "    1. Story title" or "1. Story title"
        story_pattern = r"(?:^|\n)(?:\s*)(?:\d+\.)\s*(.+?)(?=\n(?:\s*)(?:\d+\.)|\Z)"
        story_matches = re.findall(story_pattern, stories_text, re.MULTILINE | re.DOTALL)

        # If no matches with numbered pattern, try bulleted pattern
        if not story_matches:
            story_pattern = r"(?:^|\n)(?:\s*)(?:\*|\-)\s*(.+?)(?=\n(?:\s*)(?:\*|\-|\d+\.)|\Z)"
            story_matches = re.findall(story_pattern, stories_text, re.MULTILINE | re.DOTALL)

        for idx, story_text in enumerate(story_matches, start=1):
            story = self._parse_story_block(story_text, feature_key, idx)
            if story:
                stories.append(story)

        return stories

    @beartype
    @require(lambda story_text: isinstance(story_text, str), "Story text must be string")
    @require(lambda feature_key: isinstance(feature_key, str), "Feature key must be string")
    @require(
        lambda story_number: isinstance(story_number, int) and story_number > 0, "Story number must be positive int"
    )
    @ensure(lambda result: result is None or isinstance(result, dict), "Must return None or story dict")
    def _parse_story_block(self, story_text: str, feature_key: str, story_number: int) -> dict[str, Any] | None:
        """Parse a single story block from enrichment report."""
        story: dict[str, Any] = {
            "key": "",
            "title": "",
            "acceptance": [],
            "story_points": None,
            "value_points": None,
            "tasks": [],
            "confidence": 0.8,
        }

        # Generate story key from feature key and number
        if feature_key:
            # Extract base from feature key (e.g., "FEATURE-DUALSTACK" -> "DUALSTACK")
            base = feature_key.replace("FEATURE-", "").upper()
            story["key"] = f"STORY-{base}-{story_number:03d}"
        else:
            story["key"] = f"STORY-{story_number:03d}"

        # Extract title (first line or after "Title:")
        title_match = re.search(r"(?:title|Title):\s*(.+?)(?:\n|$)", story_text, re.IGNORECASE)
        if title_match:
            story["title"] = title_match.group(1).strip()
        else:
            # Use first line as title (remove leading number/bullet if present)
            first_line = story_text.split("\n")[0].strip()
            # Remove leading number/bullet: "1. Title" -> "Title" or "- Title" -> "Title"
            first_line = re.sub(r"^(?:\d+\.|\*|\-)\s*", "", first_line).strip()
            # Remove story key prefix if present: "STORY-XXX: Title" -> "Title"
            first_line = re.sub(r"^STORY-[A-Z0-9-]+:\s*", "", first_line, flags=re.IGNORECASE).strip()
            if first_line and not first_line.startswith("#") and not first_line.startswith("-"):
                story["title"] = first_line

        # Extract acceptance criteria
        # Handle both "- Acceptance: ..." and "Acceptance: ..." formats
        # Pattern matches: "- Acceptance: ..." or "Acceptance: ..." (with optional indentation and dash)
        # Use simple pattern that matches "Acceptance:" and captures until end or next numbered item
        acceptance_match = re.search(
            r"(?:acceptance|Acceptance|criteria|Criteria):\s*(.+?)(?=\n\s*\d+\.|\n\s*(?:tasks?|Tasks?|points?|Points?|##)|\Z)",
            story_text,
            re.IGNORECASE | re.DOTALL,
        )
        if acceptance_match:
            acceptance_text = acceptance_match.group(1).strip()
            # Split by commas (common format: "criterion1, criterion2, criterion3")
            # Use lookahead to split on comma-space before capital letter (sentence boundaries)
            # Also split on newlines for multi-line format
            acceptance = [
                a.strip()
                for a in re.split(r",\s+(?=[A-Z][a-z])|\n", acceptance_text)
                if a.strip() and not a.strip().startswith("-") and not a.strip().startswith("Acceptance:")
            ]
            # If splitting didn't work well, try simpler comma split
            if not acceptance or len(acceptance) == 1:
                acceptance = [
                    a.strip() for a in acceptance_text.split(",") if a.strip() and not a.strip().startswith("-")
                ]
            # If still empty after splitting, use the whole text as one criterion
            if not acceptance:
                acceptance = [acceptance_text]
            story["acceptance"] = acceptance
        else:
            # Default acceptance if none found
            story["acceptance"] = [f"{story.get('title', 'Story')} works as expected"]

        # Extract tasks
        tasks_match = re.search(
            r"(?:tasks?|Tasks?):\s*(.+?)(?:\n(?:points?|Points?|$))", story_text, re.IGNORECASE | re.DOTALL
        )
        if tasks_match:
            tasks_text = tasks_match.group(1)
            tasks = [t.strip() for t in re.split(r"\n|,", tasks_text) if t.strip()]
            story["tasks"] = tasks

        # Extract story points
        story_points_match = re.search(r"(?:story\s+points?|Story\s+Points?):\s*(\d+)", story_text, re.IGNORECASE)
        if story_points_match:
            with suppress(ValueError):
                story["story_points"] = int(story_points_match.group(1))

        # Extract value points
        value_points_match = re.search(r"(?:value\s+points?|Value\s+Points?):\s*(\d+)", story_text, re.IGNORECASE)
        if value_points_match:
            with suppress(ValueError):
                story["value_points"] = int(value_points_match.group(1))

        # Only return if we have at least a title
        if story["title"]:
            return story

        return None

    @beartype
    @require(lambda content: isinstance(content, str), "Content must be string")
    @require(lambda report: isinstance(report, EnrichmentReport), "Report must be EnrichmentReport")
    def _parse_confidence_adjustments(self, content: str, report: EnrichmentReport) -> None:
        """Parse confidence adjustments section from enrichment report."""
        # Look for "Confidence Adjustments" or "Confidence adjustments" section
        pattern = r"##\s*Confidence\s+Adjustments?\s*\n(.*?)(?=##|\Z)"
        match = re.search(pattern, content, re.IGNORECASE | re.DOTALL)
        if not match:
            return

        section = match.group(1)

        # Extract adjustments (format: "FEATURE-KEY → 0.95" or "FEATURE-KEY: 0.95")
        adjustment_pattern = r"([A-Z0-9_-]+)\s*(?:→|:)\s*([0-9.]+)"
        adjustments = re.findall(adjustment_pattern, section, re.IGNORECASE)

        for feature_key, confidence_str in adjustments:
            try:
                confidence = float(confidence_str)
                if 0.0 <= confidence <= 1.0:
                    report.adjust_confidence(feature_key.upper(), confidence)
            except ValueError:
                pass

    @beartype
    @require(lambda content: isinstance(content, str), "Content must be string")
    @require(lambda report: isinstance(report, EnrichmentReport), "Report must be EnrichmentReport")
    def _parse_business_context(self, content: str, report: EnrichmentReport) -> None:
        """Parse business context section from enrichment report."""
        # Look for "Business Context" section
        pattern = r"##\s*Business\s+Context\s*\n(.*?)(?=##|\Z)"
        match = re.search(pattern, content, re.IGNORECASE | re.DOTALL)
        if not match:
            return

        section = match.group(1)

        # Extract priorities
        priorities_match = re.search(
            r"(?:Priorities?|Priority):\s*(.+?)(?:\n(?:Constraints?|Unknowns?)|$)", section, re.IGNORECASE | re.DOTALL
        )
        if priorities_match:
            priorities_text = priorities_match.group(1)
            priorities = [
                p.strip() for p in re.split(r"\n|,", priorities_text) if p.strip() and not p.strip().startswith("-")
            ]
            report.add_business_context("priorities", priorities)

        # Extract constraints
        constraints_match = re.search(
            r"(?:Constraints?|Constraint):\s*(.+?)(?:\n(?:Unknowns?|Priorities?)|$)", section, re.IGNORECASE | re.DOTALL
        )
        if constraints_match:
            constraints_text = constraints_match.group(1)
            constraints = [
                c.strip() for c in re.split(r"\n|,", constraints_text) if c.strip() and not c.strip().startswith("-")
            ]
            report.add_business_context("constraints", constraints)

        # Extract unknowns
        unknowns_match = re.search(
            r"(?:Unknowns?|Unknown):\s*(.+?)(?:\n(?:Priorities?|Constraints?)|$)", section, re.IGNORECASE | re.DOTALL
        )
        if unknowns_match:
            unknowns_text = unknowns_match.group(1)
            unknowns = [
                u.strip() for u in re.split(r"\n|,", unknowns_text) if u.strip() and not u.strip().startswith("-")
            ]
            report.add_business_context("unknowns", unknowns)


@beartype
@require(lambda plan_bundle: isinstance(plan_bundle, PlanBundle), "Plan bundle must be PlanBundle")
@require(lambda enrichment: isinstance(enrichment, EnrichmentReport), "Enrichment must be EnrichmentReport")
@ensure(lambda result: isinstance(result, PlanBundle), "Must return PlanBundle")
def apply_enrichment(plan_bundle: PlanBundle, enrichment: EnrichmentReport) -> PlanBundle:
    """
    Apply enrichment report to plan bundle.

    Args:
        plan_bundle: Original plan bundle from CLI
        enrichment: Parsed enrichment report

    Returns:
        Enriched plan bundle
    """
    # Create a copy to avoid mutating the original
    enriched = plan_bundle.model_copy(deep=True)

    # Apply confidence adjustments
    feature_keys = {f.key: i for i, f in enumerate(enriched.features)}
    for feature_key, new_confidence in enrichment.confidence_adjustments.items():
        if feature_key in feature_keys:
            enriched.features[feature_keys[feature_key]].confidence = new_confidence

    # Add missing features
    for missing_feature_data in enrichment.missing_features:
        # Check if feature already exists
        feature_key = missing_feature_data.get("key", "")
        if feature_key and feature_key in feature_keys:
            # Update existing feature instead of adding duplicate
            existing_idx = feature_keys[feature_key]
            existing_feature = enriched.features[existing_idx]
            # Update confidence if provided
            if "confidence" in missing_feature_data:
                existing_feature.confidence = missing_feature_data["confidence"]
            # Update title if provided and empty
            if "title" in missing_feature_data and missing_feature_data["title"] and not existing_feature.title:
                existing_feature.title = missing_feature_data["title"]
            # Merge outcomes
            if "outcomes" in missing_feature_data:
                for outcome in missing_feature_data["outcomes"]:
                    if outcome not in existing_feature.outcomes:
                        existing_feature.outcomes.append(outcome)
            # Merge stories (add new stories that don't already exist)
            stories_data = missing_feature_data.get("stories", [])
            if stories_data:
                existing_story_keys = {s.key for s in existing_feature.stories}
                for story_data in stories_data:
                    if isinstance(story_data, dict):
                        story_key = story_data.get("key", "")
                        # Only add story if it doesn't already exist
                        if story_key and story_key not in existing_story_keys:
                            story = Story(
                                key=story_key,
                                title=story_data.get("title", "Untitled Story"),
                                acceptance=story_data.get("acceptance", []),
                                story_points=story_data.get("story_points"),
                                value_points=story_data.get("value_points"),
                                tasks=story_data.get("tasks", []),
                                confidence=story_data.get("confidence", 0.8),
                                draft=False,
                                scenarios=None,
                                contracts=None,
                            )
                            existing_feature.stories.append(story)
                            existing_story_keys.add(story_key)
        else:
            # Create new feature with stories (if provided)
            stories_data = missing_feature_data.get("stories", [])
            stories: list[Story] = []
            for story_data in stories_data:
                if isinstance(story_data, dict):
                    story = Story(
                        key=story_data.get("key", f"STORY-{len(stories) + 1:03d}"),
                        title=story_data.get("title", "Untitled Story"),
                        acceptance=story_data.get("acceptance", []),
                        story_points=story_data.get("story_points"),
                        value_points=story_data.get("value_points"),
                        tasks=story_data.get("tasks", []),
                        confidence=story_data.get("confidence", 0.8),
                        draft=False,
                        scenarios=None,
                        contracts=None,
                    )
                    stories.append(story)

            feature = Feature(
                key=missing_feature_data.get("key", f"FEATURE-{len(enriched.features) + 1:03d}"),
                title=missing_feature_data.get("title", "Untitled Feature"),
                outcomes=missing_feature_data.get("outcomes", []),
                acceptance=[],
                constraints=[],
                stories=stories,  # Include parsed stories
                confidence=missing_feature_data.get("confidence", 0.5),
                draft=False,
                source_tracking=None,
                contract=None,
                protocol=None,
            )
            enriched.features.append(feature)

    # Apply business context to idea if present
    if enriched.idea and enrichment.business_context and enrichment.business_context.get("constraints"):
        if enriched.idea.constraints is None:
            enriched.idea.constraints = []
        enriched.idea.constraints.extend(enrichment.business_context["constraints"])

    return enriched
