"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f, _g;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Bundling image derived from runtime bundling image (lambci)
        const image = cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
            buildArgs: {
                ...(_b = options.buildArgs) !== null && _b !== void 0 ? _b : {},
                IMAGE: options.runtime.bundlingDockerImage.image,
                PARCEL_VERSION: (_c = options.parcelVersion) !== null && _c !== void 0 ? _c : '2.0.0-beta.1',
            },
        });
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_d = options.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_e = options.nodeModules) !== null && _e !== void 0 ? _e : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            'cdk-lambda': `${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/index.js`,
            'targets': {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_f = options.sourceMaps) !== null && _f !== void 0 ? _f : false,
                    minify: (_g = options.minify) !== null && _g !== void 0 ? _g : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(projectRoot, path.resolve(options.entry)));
        const parcelCommand = [
            '$(node -p "require.resolve(\'parcel\')")',
            'build', containerEntryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', '/parcel-cache']
                : [],
        ].join(' ');
        let installer = Installer.NPM;
        let lockfile;
        let depsCommand = '';
        if (dependencies) {
            // Create a dummy package.json for dependencies that we need to install
            fs.writeFileSync(path.join(projectRoot, '.package.json'), JSON.stringify({ dependencies }));
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, LockFile.YARN))) {
                installer = Installer.YARN;
                lockfile = LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, LockFile.NPM))) {
                lockfile = LockFile.NPM;
            }
            // Move dummy package.json and lock file then install
            depsCommand = chain([
                `mv ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/.package.json ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/package.json`,
                lockfile ? `cp ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/${lockfile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${lockfile}` : '',
                `cd ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR} && ${installer} install`,
            ]);
        }
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                image,
                command: ['bash', '-c', chain([parcelCommand, depsCommand])],
                environment: options.parcelEnvironment,
                volumes: options.cacheDir
                    ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                    : [],
                workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
            },
        });
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile || (LockFile = {}));
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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