"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const openapi_helper_1 = require("../lib/openapi-helper");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const path = require("path");
const assertions_1 = require("aws-cdk-lib/assertions");
const core_1 = require("@aws-solutions-constructs/core");
const defaults = require("@aws-solutions-constructs/core");
const inlineJsonApiDefinition = require("./openapi/apiDefinition.json");
test('Simple deployment works', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiGateway).toBeDefined();
    expect(construct.apiGatewayCloudWatchRole).toBeDefined();
    expect(construct.apiGatewayLogGroup).toBeDefined();
    expect(construct.apiLambdaFunctions.length).toEqual(1);
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from Asset', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from S3 Bucket and Key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = (0, core_1.CreateScrapBucket)(stack, "scrapBucket");
    const apiDefinitionKey = 'api.yaml';
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionBucket,
        apiDefinitionKey,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from an apiDefinitionJson ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionJson: inlineJsonApiDefinition,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
const incorrectDefinitionMessage = 'Exactly one of apiDefinitionAsset, apiDefinitionJson or (apiDefinitionBucket/apiDefinitionKey) must be provided';
test('Throws error when both api definition asset and s3 object are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionBucket: new s3.Bucket(stack, 'ApiDefinitionBucket'),
            apiDefinitionKey: 'key',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Multiple Lambda Functions can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
    template.resourceCountIs("AWS::Lambda::Alias", 0);
});
test('Existing lambda function can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        functionName: 'ExistingLambdaFunction',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                existingLambdaObj
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    functionName: 'NewLambdaFunction',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'ExistingLambdaFunction'
    });
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'NewLambdaFunction'
    });
});
test('Throws error when neither existingLambdaObj or lambdaFunctionProps is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiIntegrations: [
                {
                    id: 'MessagesHandler'
                }
            ]
        });
    };
    expect(app).toThrowError(`ApiIntegration id:MessagesHandler must have exactly one of lambdaFunctionProps or existingLambdaObj\n`);
});
test('Two Constructs create APIs with different names', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const secondApiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'SecondOpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition-withCognitoAuth.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'second-test', {
        apiDefinitionAsset: secondApiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::ApiGateway::RestApi", 2);
    template.hasResourceProperties("AWS::ApiGateway::RestApi", {
        Name: {
            "Fn::Join": [
                "",
                [
                    "test-apigateway-lambda-",
                    {
                        "Fn::Select": [
                            2,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        Ref: "AWS::StackId"
                                    }
                                ]
                            }
                        ]
                    }
                ]
            ]
        }
    });
    template.hasResourceProperties("AWS::ApiGateway::RestApi", {
        Name: {
            "Fn::Join": [
                "",
                [
                    "second-test-",
                    {
                        "Fn::Select": [
                            2,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        Ref: "AWS::StackId"
                                    }
                                ]
                            }
                        ]
                    }
                ]
            ]
        }
    });
});
test('Confirm API definition uri is added to function name', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    const resources = JSON.parse(JSON.stringify(template)).Resources;
    expect(Object.keys(resources).find((element) => {
        return element.includes("MessagesHandler");
    })).toBeTruthy();
});
/*
 * openapi-helper.ts tests
*/
test('Throws error when no api definition is specified', () => {
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Throws error when no api integration is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionAsset,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('At least one ApiIntegration must be specified in the apiIntegrations property');
});
test('Throws error when api definition s3 bucket is specified but s3 object key is missing', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = (0, core_1.CreateScrapBucket)(stack, "scrapBucket");
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionBucket,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('apiDefinitionBucket and apiDefinitionKey must be specified together.');
});
test('Throws error when api definition s3 object key is specified but s3 bucket is missing', () => {
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('apiDefinitionBucket and apiDefinitionKey must be specified together.');
});
test('Throws error when api is defined as asset and s3 bucket is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const apiDefinitionBucket = (0, core_1.CreateScrapBucket)(stack, "scrapBucket");
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionAsset,
            apiDefinitionBucket,
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Throws error when api is defined as asset and s3 key is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const apiDefinitionBucket = (0, core_1.CreateScrapBucket)(stack, "scrapBucket");
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionAsset,
            apiDefinitionBucket,
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Throws error when both api definition inline and api definition asset are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionAsset,
            apiDefinitionJson: inlineJsonApiDefinition,
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Throws error when both api definition inline and s3 object are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionJson: inlineJsonApiDefinition,
            apiDefinitionBucket: new s3.Bucket(stack, 'ApiDefinitionBucket'),
            apiDefinitionKey: 'key',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('ObtainApiDefinition from local asset', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const testFunction = new lambda.Function(stack, 'test', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiLambdaFunctions = [
        {
            id: 'MessageHandler',
            lambdaFunction: testFunction
        }
    ];
    const api = (0, openapi_helper_1.ObtainApiDefinition)(stack, {
        apiDefinitionAsset,
        tokenToFunctionMap: apiLambdaFunctions
    });
    expect(api).toBeDefined();
    expect(api.bucketName).toBeDefined();
    expect(api.key).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("Custom::TemplateWriter", 1);
});
test('ObtainApiDefinition from inline JSON spec', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiLambdaFunctions = [
        {
            id: 'MessageHandler',
            lambdaFunction: testFunction
        }
    ];
    const api = (0, openapi_helper_1.ObtainApiDefinition)(stack, {
        apiJsonDefinition: inlineJsonApiDefinition,
        tokenToFunctionMap: apiLambdaFunctions
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("Custom::TemplateWriter", 0);
    expect(api).toBeDefined();
    expect(api.definition).toBeDefined();
    expect(api.definition.openapi).toEqual("3.0.1");
    expect(api.definition.info).toBeDefined();
    expect(api.definition.paths).toBeDefined();
});
test('ObtainApiDefinition from S3 bucket/key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiLambdaFunctions = [
        {
            id: 'MessageHandler',
            lambdaFunction: testFunction
        }
    ];
    const api = (0, openapi_helper_1.ObtainApiDefinition)(stack, {
        apiDefinitionBucket: (0, core_1.CreateScrapBucket)(stack, "scrapBucket"),
        apiDefinitionKey: "api.yml",
        tokenToFunctionMap: apiLambdaFunctions
    });
    expect(api).toBeDefined();
    expect(api.bucketName).toBeDefined();
    expect(api.key).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("Custom::TemplateWriter", 1);
});
test('ObtainApiDefinition uses custom properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const testFunction = new lambda.Function(stack, 'test', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiLambdaFunctions = [
        {
            id: 'MessageHandler',
            lambdaFunction: testFunction
        }
    ];
    const api = (0, openapi_helper_1.ObtainApiDefinition)(stack, {
        apiDefinitionAsset,
        tokenToFunctionMap: apiLambdaFunctions,
        internalTransformTimeout: aws_cdk_lib_1.Duration.seconds(385),
        internalTransformMemorySize: 3456
    });
    expect(api).toBeDefined();
    expect(api.bucketName).toBeDefined();
    expect(api.key).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("Custom::TemplateWriter", 1);
    template.hasResource("AWS::Lambda::Function", {
        Properties: {
            Timeout: 385,
            MemorySize: 3456
        }
    });
});
test('Shared lambda function works', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const constructTwo = new lib_1.OpenApiGatewayToLambda(stack, 'two-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                existingLambdaObj: construct.apiLambdaFunctions[0].lambdaFunction
            }
        ]
    });
    expect(construct.apiGateway).toBeDefined();
    expect(construct.apiGatewayCloudWatchRole).toBeDefined();
    expect(construct.apiGatewayLogGroup).toBeDefined();
    expect(construct.apiLambdaFunctions.length).toEqual(1);
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
    expect(construct.apiLambdaFunctions[0].lambdaFunction.functionArn).toEqual(constructTwo.apiLambdaFunctions[0].lambdaFunction.functionArn);
});
test('Deploys API based on Alias correctly', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'ApiDefinitionAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const messagesLambda = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
            currentVersionOptions: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
            },
        }
    });
    const photosLambda = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            functionName: 'PhotosLambdaTestFromAsset',
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
        }
    });
    const messagesAlias = new lambda.Alias(stack, 'messages-alias', {
        version: messagesLambda.currentVersion,
        aliasName: 'messagesAlias',
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'OpenApiGatewayToLambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                existingLambdaObj: messagesAlias
            },
            {
                id: 'PhotosHandler',
                existingLambdaObj: photosLambda
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Alias", 1);
    template.resourceCountIs("AWS::Lambda::Version", 1);
    template.hasResourceProperties("AWS::Lambda::Alias", {
        FunctionName: {
            Ref: assertions_1.Match.stringLikeRegexp("LambdaFunction.*")
        },
        FunctionVersion: {
            "Fn::GetAtt": [
                assertions_1.Match.stringLikeRegexp("LambdaFunctionCurrentVersion.*"),
                "Version"
            ]
        },
        Name: "messagesAlias"
    });
    // Confirm we've granted permission the the Alias and not the Lambda
    template.hasResourceProperties("AWS::Lambda::Permission", {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            Ref: assertions_1.Match.stringLikeRegexp("messagesalias.*")
        },
        Principal: "apigateway.amazonaws.com",
    });
    template.resourcePropertiesCountIs("AWS::Lambda::Permission", {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            Ref: assertions_1.Match.stringLikeRegexp("LambdaFunction.*")
        },
        Principal: "apigateway.amazonaws.com",
    }, 0);
});
test('confirm execute-api arn is generated correctly', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Permission", {
        Action: "lambda:InvokeFunction",
        Principal: "apigateway.amazonaws.com",
        FunctionName: {
            "Fn::GetAtt": [
                assertions_1.Match.stringLikeRegexp("testapigatewaylambdaMessagesHandlerApiFunction*"),
                "Arn"
            ]
        },
        SourceArn: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        "Ref": "AWS::Partition"
                    },
                    ":execute-api:",
                    {
                        "Ref": "AWS::Region"
                    },
                    ":",
                    {
                        "Ref": "AWS::AccountId"
                    },
                    ":",
                    {
                        "Ref": assertions_1.Match.stringLikeRegexp("testapigatewaylambdaSpecRestApi*")
                    },
                    "/*/*"
                ]
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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