"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImagePipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codepipeline_1 = require("aws-cdk-lib/aws-codepipeline");
const aws_codepipeline_actions_1 = require("aws-cdk-lib/aws-codepipeline-actions");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
const HANDLER_PATH = path.join(__dirname, "ecs_pipeline_handler");
/**
 * Allows images pushed to an ECR repo to trigger updates to an ECS service.
 *
 * This construct produces a CodePipeline pipeline using the "ECR Source"
 * action, an "ECS Deploy" action, and a custom Lambda handler in between that
 * transforms the JSON from the "Source" action into the JSON needed for the
 * "Deploy" action.
 */
class ContainerImagePipeline extends constructs_1.Construct {
    /**
     * Creates a new ContainerImagePipeline.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { service, container, repository, tag = "latest", pipelineType, artifactBucket, } = props;
        const transformerFunction = new aws_lambda_1.SingletonFunction(this, "Transformer", {
            uuid: "76208d72-6a58-47de-b611-75e2f58ad601",
            lambdaPurpose: "EcsJsonTransform",
            runtime: aws_lambda_1.Runtime.PYTHON_3_12,
            handler: "index.lambda_handler",
            description: "Transforms the imageDetail.json from ECR into imagedefinitions.json for ECS",
            code: aws_lambda_1.Code.fromAsset(HANDLER_PATH),
            timeout: aws_cdk_lib_1.Duration.seconds(60),
        });
        const sourceArtifact = new aws_codepipeline_1.Artifact();
        const buildArtifact = new aws_codepipeline_1.Artifact();
        this.pipeline = new aws_codepipeline_1.Pipeline(this, "Pipeline", {
            crossAccountKeys: false,
            pipelineType,
            artifactBucket,
            stages: [
                {
                    stageName: "Source",
                    actions: [
                        new aws_codepipeline_actions_1.EcrSourceAction({
                            output: sourceArtifact,
                            actionName: "Receive-ECR-Notice",
                            imageTag: tag,
                            repository,
                        }),
                    ],
                },
                {
                    stageName: "Transform",
                    actions: [
                        new aws_codepipeline_actions_1.LambdaInvokeAction({
                            userParameters: { OutputContainerName: container },
                            actionName: "Produce-imagedefinitions.json",
                            lambda: transformerFunction,
                            inputs: [sourceArtifact],
                            outputs: [buildArtifact],
                        }),
                    ],
                },
                {
                    stageName: "Deploy",
                    actions: [
                        new aws_codepipeline_actions_1.EcsDeployAction({
                            actionName: "Update-ECS-Service",
                            input: buildArtifact,
                            service: service,
                        }),
                    ],
                },
            ],
        });
    }
}
exports.ContainerImagePipeline = ContainerImagePipeline;
_a = JSII_RTTI_SYMBOL_1;
ContainerImagePipeline[_a] = { fqn: "shady-island.automation.ContainerImagePipeline", version: "0.1.21" };
//# sourceMappingURL=data:application/json;base64,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