"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockFile = exports.Installer = exports.DockerBundler = exports.LocalBundler = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
const PARCEL_VERSION = '2.0.0-beta.1';
/**
 * Local Parcel bundler
 */
class LocalBundler {
    constructor(props) {
        this.props = props;
    }
    static runsLocally(resolvePath) {
        if (LocalBundler._runsLocally[resolvePath] !== undefined) {
            return LocalBundler._runsLocally[resolvePath];
        }
        if (os.platform() === 'win32') { // TODO: add Windows support
            return false;
        }
        try {
            const parcel = child_process_1.spawnSync(require.resolve('parcel', { paths: [resolvePath] }), ['--version']);
            const version = parcel.stdout.toString().trim();
            LocalBundler._runsLocally[resolvePath] = new RegExp(`^${PARCEL_VERSION}`).test(version); // Cache result to avoid unnecessary spawns
            if (!LocalBundler._runsLocally[resolvePath]) {
                process.stderr.write(`Incorrect parcel version detected: ${version} <> ${PARCEL_VERSION}. Switching to Docker bundling.\n`);
            }
            return LocalBundler._runsLocally[resolvePath];
        }
        catch (err) {
            return false;
        }
    }
    static clearRunsLocallyCache() {
        LocalBundler._runsLocally = {};
    }
    tryBundle(outputDir) {
        var _a;
        if (!LocalBundler.runsLocally(this.props.projectRoot)) {
            return false;
        }
        const localCommand = createBundlingCommand({
            projectRoot: this.props.projectRoot,
            relativeEntryPath: this.props.relativeEntryPath,
            cacheDir: this.props.cacheDir,
            outputDir,
            dependencies: this.props.dependencies,
            installer: this.props.installer,
            lockFile: this.props.lockFile,
            bundlingEnvironment: BundlingEnvironment.LOCAL,
        });
        util_1.exec('bash', ['-c', localCommand], {
            env: { ...process.env, ...(_a = this.props.environment) !== null && _a !== void 0 ? _a : {} },
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
            cwd: path.dirname(path.join(this.props.projectRoot, this.props.relativeEntryPath)),
        });
        return true;
    }
}
exports.LocalBundler = LocalBundler;
LocalBundler._runsLocally = {};
/**
 * Docker bundler
 */
class DockerBundler {
    constructor(props) {
        var _a, _b, _c;
        const image = props.buildImage
            ? (_a = props.bundlingDockerImage) !== null && _a !== void 0 ? _a : cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
                buildArgs: {
                    ...(_b = props.buildArgs) !== null && _b !== void 0 ? _b : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    PARCEL_VERSION: (_c = props.parcelVersion) !== null && _c !== void 0 ? _c : PARCEL_VERSION,
                },
            }) : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const command = createBundlingCommand({
            projectRoot: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            relativeEntryPath: props.relativeEntryPath,
            cacheDir: props.cacheDir,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            installer: props.installer,
            lockFile: props.lockFile,
            dependencies: props.dependencies,
            bundlingEnvironment: BundlingEnvironment.DOCKER,
        });
        this.bundlingOptions = {
            image,
            command: ['bash', '-c', command],
            environment: props.environment,
            workingDirectory: path.dirname(path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, props.relativeEntryPath)).replace(/\\/g, '/'),
        };
    }
}
exports.DockerBundler = DockerBundler;
var BundlingEnvironment;
(function (BundlingEnvironment) {
    BundlingEnvironment["DOCKER"] = "docker";
    BundlingEnvironment["LOCAL"] = "local";
})(BundlingEnvironment || (BundlingEnvironment = {}));
/**
 * Generates bundling command
 */
function createBundlingCommand(options) {
    const entryPath = path.join(options.projectRoot, options.relativeEntryPath);
    const distFile = path.basename(options.relativeEntryPath).replace(/\.(jsx|tsx?)$/, '.js');
    const parcelResolvePath = options.bundlingEnvironment === BundlingEnvironment.DOCKER
        ? '/' // Force using parcel installed at / in the image
        : entryPath; // Look up starting from entry path
    const parcelCommand = chain([
        [
            `$(node -p "require.resolve(\'parcel\', { paths: ['${parcelResolvePath}'] })")`,
            'build', entryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--dist-dir', options.outputDir,
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', path.join(options.projectRoot, options.cacheDir)]
                : [],
        ].join(' '),
        // Always rename dist file to index.js because Lambda doesn't support filenames
        // with multiple dots and we can end up with multiple dots when using automatic
        // entry lookup
        distFile !== 'index.js' ? `mv ${options.outputDir}/${distFile} ${options.outputDir}/index.js` : '',
    ]);
    let depsCommand = '';
    if (options.dependencies) {
        // create dummy package.json, copy lock file if any and then install
        depsCommand = chain([
            `echo '${JSON.stringify({ dependencies: options.dependencies })}' > ${options.outputDir}/package.json`,
            options.lockFile ? `cp ${options.projectRoot}/${options.lockFile} ${options.outputDir}/${options.lockFile}` : '',
            `cd ${options.outputDir}`,
            `${options.installer} install`,
        ]);
    }
    return chain([parcelCommand, depsCommand]);
}
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer = exports.Installer || (exports.Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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