"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const principals_1 = require("./principals");
const util_1 = require("./util");
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement {
    constructor(props = {}) {
        this.action = new Array();
        this.notAction = new Array();
        this.principal = {};
        this.resource = new Array();
        this.notResource = new Array();
        this.condition = {};
        this.effect = props.effect || Effect.ALLOW;
        this.addActions(...props.actions || []);
        this.addNotActions(...props.notActions || []);
        this.addPrincipals(...props.principals || []);
        this.addResources(...props.resources || []);
        this.addNotResources(...props.notResources || []);
        if (props.conditions !== undefined) {
            this.addConditions(props.conditions);
        }
    }
    //
    // Actions
    //
    addActions(...actions) {
        if (actions.length > 0 && this.notAction.length > 0) {
            throw new Error(`Cannot add 'Actions' to policy statement if 'NotActions' have been added`);
        }
        this.action.push(...actions);
    }
    addNotActions(...notActions) {
        if (notActions.length > 0 && this.action.length > 0) {
            throw new Error(`Cannot add 'NotActions' to policy statement if 'Actions' have been added`);
        }
        this.notAction.push(...notActions);
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0;
    }
    addPrincipals(...principals) {
        for (const principal of principals) {
            const fragment = principal.policyFragment;
            util_1.mergePrincipal(this.principal, fragment.principalJson);
            this.addConditions(fragment.conditions);
        }
    }
    addAwsAccountPrincipal(accountId) {
        this.addPrincipals(new principals_1.AccountPrincipal(accountId));
    }
    addArnPrincipal(arn) {
        this.addPrincipals(new principals_1.ArnPrincipal(arn));
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service, opts) {
        this.addPrincipals(new principals_1.ServicePrincipal(service, opts));
    }
    addFederatedPrincipal(federated, conditions) {
        this.addPrincipals(new principals_1.FederatedPrincipal(federated, conditions));
    }
    addAccountRootPrincipal() {
        this.addPrincipals(new principals_1.AccountRootPrincipal());
    }
    addCanonicalUserPrincipal(canonicalUserId) {
        this.addPrincipals(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    addAnyPrincipal() {
        this.addPrincipals(new principals_1.Anyone());
    }
    //
    // Resources
    //
    addResources(...arns) {
        if (arns.length > 0 && this.notResource.length > 0) {
            throw new Error(`Cannot add 'Resources' to policy statement if 'NotResources' have been added`);
        }
        this.resource.push(...arns);
    }
    addNotResources(...arns) {
        if (arns.length > 0 && this.resource.length > 0) {
            throw new Error(`Cannot add 'NotResources' to policy statement if 'Resources' have been added`);
        }
        this.notResource.push(...arns);
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        this.addResources('*');
    }
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     */
    addCondition(key, value) {
        this.condition[key] = value;
    }
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
    }
    /**
     * Add a condition that limits to a given account
     */
    addAccountCondition(accountId) {
        this.addCondition('StringEquals', { 'sts:ExternalId': accountId });
    }
    toStatementJson() {
        return noUndef({
            Action: _norm(this.action),
            NotAction: _norm(this.notAction),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            Resource: _norm(this.resource),
            NotResource: _norm(this.notResource),
            Sid: _norm(this.sid),
        });
        function _norm(values) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (cdk.Token.isUnresolved(values)) {
                return values;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            if (Object.keys(result).length === 1 && result.AWS === '*') {
                return '*';
            }
            return result;
        }
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyStatement'
        });
    }
    /**
     * JSON-ify the statement
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.toStatementJson();
    }
}
exports.PolicyStatement = PolicyStatement;
var Effect;
(function (Effect) {
    Effect["ALLOW"] = "Allow";
    Effect["DENY"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
function noUndef(x) {
    const ret = {};
    for (const [key, value] of Object.entries(x)) {
        if (value !== undefined) {
            ret[key] = value;
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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