# Amazon Bedrock Construct Library

<!--BEGIN STABILITY BANNER-->

---

![cdk-constructs: Experimental](https://img.shields.io/badge/cdk--constructs-experimental-important.svg?style=for-the-badge)

> The APIs of higher level constructs in this module are experimental and under active development.
> They are subject to non-backward compatible changes or removal in any future version. These are
> not subject to the [Semantic Versioning](https://semver.org/) model and breaking changes will be
> announced in the release notes. This means that while you may use them, you may need to update
> your source code when upgrading to a newer version of this package.

---

<!--END STABILITY BANNER-->

| **Language**                                                                                   | **Package**                             |
| :--------------------------------------------------------------------------------------------- | --------------------------------------- |
| ![Typescript Logo](https://docs.aws.amazon.com/cdk/api/latest/img/typescript32.png) TypeScript | `@aws-cdk/aws-bedrock-alpha` |

[Amazon Bedrock](https://aws.amazon.com/bedrock/) is a fully managed service that offers a choice of high-performing foundation models (FMs) from leading AI companies and Amazon through a single API, along with a broad set of capabilities you need to build generative AI applications with security, privacy, and responsible AI.

This construct library facilitates the deployment of Bedrock Agents, enabling you to create sophisticated AI applications that can interact with your systems and data sources.

## Table of contents

- [Agents](#agents)
  - [Create an Agent](#create-an-agent)
  - [Action groups](#action-groups)
  - [Prepare the Agent](#prepare-the-agent)
  - [Prompt Override Configuration](#prompt-override-configuration)
  - [Memory Configuration](#memory-configuration)
  - [Agent Collaboration](#agent-collaboration)
  - [Custom Orchestration](#custom-orchestration)
  - [Agent Alias](#agent-alias)

## Agents

Amazon Bedrock Agents allow generative AI applications to automate complex, multistep tasks by seamlessly integrating with your company's systems, APIs, and data sources. It uses the reasoning of foundation models (FMs), APIs, and data to break down user requests, gather relevant information, and efficiently complete tasks.

### Create an Agent

Building an agent is straightforward and fast.
The following example creates an Agent with a simple instruction and default prompts:

```ts fixture=default
const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
  instruction: 'You are a helpful and friendly agent that answers questions about literature.',
});
```

### Agent Properties

The Bedrock Agent class supports the following properties.

| Name | Type | Required | Description |
|---|---|---|---|
| name | string | No | The name of the agent. Defaults to a name generated by CDK |
| instruction | string | Yes | The instruction used by the agent that determines how it will perform its task. Must have a minimum of 40 characters |
| foundationModel | IBedrockInvokable | Yes | The foundation model used for orchestration by the agent |
| existingRole | iam.IRole | No | The existing IAM Role for the agent to use. Must have a trust policy allowing Bedrock service to assume the role. Defaults to a new created role |
| shouldPrepareAgent | boolean | No | Specifies whether to automatically update the `DRAFT` version of the agent after making changes. Defaults to false |
| idleSessionTTL | Duration | No | How long sessions should be kept open for the agent. Session expires if no conversation occurs during this time. Defaults to 1 hour |
| kmsKey | kms.IKey | No | The KMS key of the agent if custom encryption is configured. Defaults to AWS managed key |
| description | string | No | A description of the agent. Defaults to no description |
| actionGroups | AgentActionGroup[] | No | The Action Groups associated with the agent |
| promptOverrideConfiguration | PromptOverrideConfiguration | No | Overrides some prompt templates in different parts of an agent sequence configuration |
| userInputEnabled | boolean | No | Select whether the agent can prompt additional information from the user when it lacks enough information. Defaults to false |
| codeInterpreterEnabled | boolean | No | Select whether the agent can generate, run, and troubleshoot code when trying to complete a task. Defaults to false |
| forceDelete | boolean | No | Whether to delete the resource even if it's in use. Defaults to true |
| agentCollaboration | AgentCollaboration | No | Configuration for agent collaboration settings, including type and collaborators. This property allows you to define how the agent collaborates with other agents and what collaborators it can work with. Defaults to no agent collaboration configuration |
| customOrchestrationExecutor | CustomOrchestrationExecutor | No | The Lambda function to use for custom orchestration. If provided, orchestrationType is set to CUSTOM_ORCHESTRATION. If not provided, orchestrationType defaults to DEFAULT. Defaults to default orchestration |

### Action Groups

An action group defines functions your agent can call. The functions are Lambda functions. The action group uses an OpenAPI schema to tell the agent what your functions do and how to call them.

#### Action Group Properties

The AgentActionGroup class supports the following properties.

| Name | Type | Required | Description |
|---|---|---|---|
| name | string | No | The name of the action group. Defaults to a name generated in the format 'action_group_quick_start_UUID' |
| description | string | No | A description of the action group |
| apiSchema | ApiSchema | No | The OpenAPI schema that defines the functions in the action group |
| executor | ActionGroupExecutor | No | The Lambda function that executes the actions in the group |
| enabled | boolean | No | Whether the action group is enabled. Defaults to true |
| forceDelete | boolean | No | Whether to delete the resource even if it's in use. Defaults to false |
| functionSchema | FunctionSchema | No | Defines functions that each define parameters that the agent needs to invoke from the user |
| parentActionGroupSignature | ParentActionGroupSignature | No | The AWS Defined signature for enabling certain capabilities in your agent |

There are three ways to provide an API schema for your action group:

From a local asset file (requires binding to scope):

```ts fixture=default
const actionGroupFunction = new lambda.Function(this, 'ActionGroupFunction', {
  runtime: lambda.Runtime.PYTHON_3_12,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/action-group')),
});

// When using ApiSchema.fromLocalAsset, you must bind the schema to a scope
const schema = bedrock.ApiSchema.fromLocalAsset(path.join(__dirname, 'action-group.yaml'));
schema.bind(this);

const actionGroup = new bedrock.AgentActionGroup({
  name: 'query-library',
  description: 'Use these functions to get information about the books in the library.',
  executor: bedrock.ActionGroupExecutor.fromLambda(actionGroupFunction),
  enabled: true,
  apiSchema: schema,
});

const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
  instruction: 'You are a helpful and friendly agent that answers questions about literature.',
});

agent.addActionGroup(actionGroup);
```

From an inline OpenAPI schema:

```ts fixture=default
const inlineSchema = bedrock.ApiSchema.fromInline(`
openapi: 3.0.3
info:
  title: Library API
  version: 1.0.0
paths:
  /search:
    get:
      summary: Search for books
      operationId: searchBooks
      parameters:
        - name: query
          in: query
          required: true
          schema:
            type: string
`);

const actionGroupFunction = new lambda.Function(this, 'ActionGroupFunction', {
  runtime: lambda.Runtime.PYTHON_3_12,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/action-group')),
});

const actionGroup = new bedrock.AgentActionGroup({
  name: 'query-library',
  description: 'Use these functions to get information about the books in the library.',
  executor: bedrock.ActionGroupExecutor.fromLambda(actionGroupFunction),
  enabled: true,
  apiSchema: inlineSchema,
});

const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
  instruction: 'You are a helpful and friendly agent that answers questions about literature.',
});

agent.addActionGroup(actionGroup);
```

From an existing S3 file:

```ts fixture=default
const bucket = s3.Bucket.fromBucketName(this, 'ExistingBucket', 'my-schema-bucket');
const s3Schema = bedrock.ApiSchema.fromS3File(bucket, 'schemas/action-group.yaml');

const actionGroupFunction = new lambda.Function(this, 'ActionGroupFunction', {
  runtime: lambda.Runtime.PYTHON_3_12,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/action-group')),
});

const actionGroup = new bedrock.AgentActionGroup({
  name: 'query-library',
  description: 'Use these functions to get information about the books in the library.',
  executor: bedrock.ActionGroupExecutor.fromLambda(actionGroupFunction),
  enabled: true,
  apiSchema: s3Schema,
});

const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
  instruction: 'You are a helpful and friendly agent that answers questions about literature.',
});

agent.addActionGroup(actionGroup);
```

### Using FunctionSchema with Action Groups

As an alternative to using OpenAPI schemas, you can define functions directly using the `FunctionSchema` class. This approach provides a more structured way to define the functions that your agent can call.

```ts fixture=default
const actionGroupFunction = new lambda.Function(this, 'ActionGroupFunction', {
  runtime: lambda.Runtime.PYTHON_3_12,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/action-group')),
});

// Define a function schema with parameters
const functionSchema = new bedrock.FunctionSchema({
  functions: [
    {
      name: 'searchBooks',
      description: 'Search for books in the library catalog',
      parameters: {
        'query': {
          type: bedrock.ParameterType.STRING,
          required: true,
          description: 'The search query string',
        },
        'maxResults': {
          type: bedrock.ParameterType.INTEGER,
          required: false,
          description: 'Maximum number of results to return',
        },
        'includeOutOfPrint': {
          type: bedrock.ParameterType.BOOLEAN,
          required: false,
          description: 'Whether to include out-of-print books',
        }
      },
      requireConfirmation: bedrock.RequireConfirmation.DISABLED,
    },
    {
      name: 'getBookDetails',
      description: 'Get detailed information about a specific book',
      parameters: {
        'bookId': {
          type: bedrock.ParameterType.STRING,
          required: true,
          description: 'The unique identifier of the book',
        }
      },
      requireConfirmation: bedrock.RequireConfirmation.ENABLED,
    }
  ]
});

// Create an action group using the function schema
const actionGroup = new bedrock.AgentActionGroup({
  name: 'library-functions',
  description: 'Functions for interacting with the library catalog',
  executor: bedrock.ActionGroupExecutor.fromLambda(actionGroupFunction),
  functionSchema: functionSchema,
  enabled: true,
});

const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
  instruction: 'You are a helpful and friendly agent that answers questions about literature.',
  actionGroups: [actionGroup],
});
```

The `FunctionSchema` approach offers several advantages:

- Type-safe definition of functions and parameters
- Built-in validation of parameter names, descriptions, and other properties
- Clear structure that maps directly to the AWS Bedrock API
- Support for parameter types including string, number, integer, boolean, array, and object
- Option to require user confirmation before executing specific functions

If you chose to load your schema file from S3, the construct will provide the necessary permissions to your agent's execution role to access the schema file from the specific bucket. Similar to performing the operation through the console, the agent execution role will get a permission like:

```json
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "AmazonBedrockAgentS3PolicyProd",
            "Effect": "Allow",
            "Action": [
                "s3:GetObject"
            ],
            "Resource": [
                "arn:aws:s3:::<BUCKET_NAME>/<OBJECT_KEY>"
            ],
            "Condition": {
                "StringEquals": {
                    "aws:ResourceAccount": "ACCOUNT_NUMBER"
                }
            }
        }
    ]
}
```

```ts fixture=default
// create a bucket containing the input schema
const schemaBucket = new s3.Bucket(this, 'SchemaBucket', {
  enforceSSL: true,
  versioned: true,
  publicReadAccess: false,
  blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
  encryption: s3.BucketEncryption.S3_MANAGED,
  removalPolicy: RemovalPolicy.DESTROY,
  autoDeleteObjects: true,
});

// deploy the local schema file to S3
const deployement = new aws_s3_deployment.BucketDeployment(this, 'DeployWebsite', {
  sources: [aws_s3_deployment.Source.asset(path.join(__dirname, '../inputschema'))],
  destinationBucket: schemaBucket,
  destinationKeyPrefix: 'inputschema',
});

// create the agent
const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_3_5_SONNET_V1_0,
  instruction: 'You are a helpful and friendly agent that answers questions about literature.',
  userInputEnabled: true,
  shouldPrepareAgent:true
});

// create a lambda function
const actionGroupFunction = new lambda.Function(this, 'ActionGroupFunction', {
  runtime: lambda.Runtime.PYTHON_3_12,
  handler: 'index.handler',
  code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/action-group')),
});

// create an action group and read the schema file from S3
const actionGroup = new bedrock.AgentActionGroup({
  name: 'query-library',
  description: 'Use these functions to get information about the books in the library.',
  executor: bedrock.ActionGroupExecutor.fromLambda(actionGroupFunction),
  enabled: true,
  apiSchema: bedrock.ApiSchema.fromS3File(schemaBucket, 'inputschema/action-group.yaml'),
});

// add the action group to the agent
agent.addActionGroup(actionGroup);

// add dependency for the agent on the s3 deployment
agent.node.addDependency(deployement);
```

### Prepare the Agent

The `Agent` constructs take an optional parameter `shouldPrepareAgent` to indicate that the Agent should be prepared after any updates to an agent or action group. This may increase the time to create and update those resources. By default, this value is false.

#### Prepare Agent Properties

| Name | Type | Required | Description |
|---|---|---|---|
| shouldPrepareAgent | boolean | No | Whether to automatically update the DRAFT version of the agent after making changes. Defaults to false |

Creating an agent alias will not prepare the agent, so if you create an alias using the `AgentAlias` resource then you should set `shouldPrepareAgent` to **_true_**.

### Prompt Override Configuration

Bedrock Agents allows you to customize the prompts and LLM configuration for different steps in the agent sequence. The implementation provides type-safe configurations for each step type, ensuring correct usage at compile time.

#### Prompt Override Configuration Properties

| Name | Type | Required | Description |
|---|---|---|---|
| steps | PromptStepConfiguration[] | Yes | Array of step configurations for different parts of the agent sequence |
| parser | lambda.IFunction | No | Lambda function for custom parsing of agent responses |

#### Prompt Step Configuration Properties

Each step in the `steps` array supports the following properties:

| Name | Type | Required | Description |
|---|---|---|---|
| stepType | AgentStepType | Yes | The type of step being configured (PRE_PROCESSING, ORCHESTRATION, POST_PROCESSING, ROUTING_CLASSIFIER, MEMORY_SUMMARIZATION, KNOWLEDGE_BASE_RESPONSE_GENERATION) |
| stepEnabled | boolean | No | Whether this step is enabled. Defaults to true |
| customPromptTemplate | string | No | Custom prompt template to use for this step |
| inferenceConfig | InferenceConfiguration | No | Configuration for model inference parameters |
| foundationModel | BedrockFoundationModel | No | Alternative foundation model to use for this step (only valid for ROUTING_CLASSIFIER step) |
| useCustomParser | boolean | No | Whether to use a custom parser for this step. Requires parser to be provided in PromptOverrideConfiguration |

#### Inference Configuration Properties

When providing `inferenceConfig`, the following properties are supported:

| Name | Type | Required | Description |
|---|---|---|---|
| temperature | number | No | Controls randomness in the model's output (0.0-1.0) |
| topP | number | No | Controls diversity via nucleus sampling (0.0-1.0) |
| topK | number | No | Controls diversity by limiting the cumulative probability |
| maximumLength | number | No | Maximum length of generated text |
| stopSequences | string[] | No | Sequences where the model should stop generating |

The following steps can be configured:

- PRE_PROCESSING: Prepares the user input for orchestration
- ORCHESTRATION: Main step that determines the agent's actions
- POST_PROCESSING: Refines the agent's response
- ROUTING_CLASSIFIER: Classifies and routes requests to appropriate collaborators
- MEMORY_SUMMARIZATION: Summarizes conversation history for memory retention
- KNOWLEDGE_BASE_RESPONSE_GENERATION: Generates responses using knowledge base content

Example with pre-processing configuration:

```ts fixture=default
const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
  instruction: 'You are a helpful assistant.',
  promptOverrideConfiguration: bedrock.PromptOverrideConfiguration.fromSteps([
    {
      stepType: bedrock.AgentStepType.PRE_PROCESSING,
      stepEnabled: true,
      customPromptTemplate: 'Your custom prompt template here',
      inferenceConfig: {
        temperature: 0.0,
        topP: 1,
        topK: 250,
        maximumLength: 1,
        stopSequences: ["\n\nHuman:"],
      },
    }
  ])
});
```

Example with routing classifier and foundation model:

```ts fixture=default
const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
  instruction: 'You are a helpful assistant.',
  promptOverrideConfiguration: bedrock.PromptOverrideConfiguration.fromSteps([
    {
      stepType: bedrock.AgentStepType.ROUTING_CLASSIFIER,
      stepEnabled: true,
      customPromptTemplate: 'Your routing template here',
      foundationModel: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_V2
    } as bedrock.PromptRoutingClassifierConfigCustomParser
  ])
});
```

Using a custom Lambda parser:

```ts fixture=default
const parserFunction = new lambda.Function(this, 'ParserFunction', {
  runtime: lambda.Runtime.PYTHON_3_10,
  handler: 'index.handler',
  code: lambda.Code.fromAsset('lambda'),
});

const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
  instruction: 'You are a helpful assistant.',
  promptOverrideConfiguration: bedrock.PromptOverrideConfiguration.withCustomParser({
    parser: parserFunction,
    preProcessingStep: {
      stepType: bedrock.AgentStepType.PRE_PROCESSING,
      useCustomParser: true
    }
  })
});
```

Foundation models can only be specified for the ROUTING_CLASSIFIER step.

### Memory Configuration

Agents can maintain context across multiple sessions and recall past interactions using memory. This feature is useful for creating a more coherent conversational experience.

#### Memory Configuration Properties

| Name | Type | Required | Description |
|---|---|---|---|
| maxRecentSessions | number | No | Maximum number of recent session summaries to retain |
| memoryDuration | Duration | No | How long to retain session summaries |

Example:

```ts fixture=default
const agent = new bedrock.Agent(this, 'MyAgent', {
  agentName: 'MyAgent',
  instruction: 'Your instruction here',
  foundationModel: bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
  memory: Memory.sessionSummary({
    maxRecentSessions: 10, // Keep the last 10 session summaries
    memoryDuration: Duration.days(20), // Retain summaries for 20 days
  }),
});
```

### Agent Collaboration

Agent Collaboration enables multiple Bedrock Agents to work together on complex tasks. This feature allows agents to specialize in different areas and collaborate to provide more comprehensive responses to user queries.

#### Agent Collaboration Properties

| Name | Type | Required | Description |
|---|---|---|---|
| type | AgentCollaboratorType | Yes | Type of collaboration (SUPERVISOR or PEER) |
| collaborators | AgentCollaborator[] | Yes | List of agent collaborators |

#### Agent Collaborator Properties

| Name | Type | Required | Description |
|---|---|---|---|
| agentAlias | AgentAlias | Yes | The agent alias to collaborate with |
| collaborationInstruction | string | Yes | Instructions for how to collaborate with this agent |
| collaboratorName | string | Yes | Name of the collaborator |
| relayConversationHistory | boolean | No | Whether to relay conversation history to the collaborator. Defaults to false |

Example:

```ts fixture=default
// Create a specialized agent
const customerSupportAgent = new bedrock.Agent(this, 'CustomerSupportAgent', {
  instruction: 'You specialize in answering customer support questions.',
  foundationModel: bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
});

// Create an agent alias
const customerSupportAlias = new bedrock.AgentAlias(this, 'CustomerSupportAlias', {
  agent: customerSupportAgent,
  agentAliasName: 'production',
});

// Create a main agent that collaborates with the specialized agent
const mainAgent = new bedrock.Agent(this, 'MainAgent', {
  instruction: 'You route specialized questions to other agents.',
  foundationModel: bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
  agentCollaboration: {
    type: bedrock.AgentCollaboratorType.SUPERVISOR,
    collaborators: [
      new bedrock.AgentCollaborator({
        agentAlias: customerSupportAlias,
        collaborationInstruction: 'Route customer support questions to this agent.',
        collaboratorName: 'CustomerSupport',
        relayConversationHistory: true
      }),
    ],
  },
});
```

### Custom Orchestration

Custom Orchestration allows you to override the default agent orchestration flow with your own Lambda function. This enables more control over how the agent processes user inputs and invokes action groups.

When you provide a customOrchestrationExecutor, the agent's orchestrationType is automatically set to CUSTOM_ORCHESTRATION. If no customOrchestrationExecutor is provided, the orchestrationType defaults to DEFAULT, using Amazon Bedrock's built-in orchestration.

#### Custom Orchestration Properties

| Name | Type | Required | Description |
|---|---|---|---|
| function | lambda.IFunction | Yes | The Lambda function that implements the custom orchestration logic |

Example:

```ts fixture=default
const orchestrationFunction = new lambda.Function(this, 'OrchestrationFunction', {
  runtime: lambda.Runtime.PYTHON_3_10,
  handler: 'index.handler',
  code: lambda.Code.fromAsset('lambda/orchestration'),
});

const agent = new bedrock.Agent(this, 'CustomOrchestrationAgent', {
  instruction: 'You are a helpful assistant with custom orchestration logic.',
  foundationModel: bedrock.BedrockFoundationModel.AMAZON_NOVA_LITE_V1,
  customOrchestrationExecutor: bedrock.CustomOrchestrationExecutor.fromLambda(orchestrationFunction),
});
```

### Agent Alias

After you have sufficiently iterated on your working draft and are satisfied with the behavior of your agent, you can set it up for deployment and integration into your application by creating aliases.

To deploy your agent, you need to create an alias. During alias creation, Amazon Bedrock automatically creates a version of your agent. The alias points to this newly created version. You can point the alias to a previously created version if necessary. You then configure your application to make API calls to that alias.

By default, the Agent resource creates a test alias named 'AgentTestAlias' that points to the 'DRAFT' version. This test alias is accessible via the `testAlias` property of the agent. You can also create additional aliases for different environments using the AgentAlias construct.

#### Agent Alias Properties

| Name | Type | Required | Description |
|---|---|---|---|
| agent | Agent | Yes | The agent to create an alias for |
| agentAliasName | string | No | The name of the agent alias. Defaults to a name generated by CDK |
| description | string | No | A description of the agent alias. Defaults to no description |
| routingConfiguration | AgentAliasRoutingConfiguration | No | Configuration for routing traffic between agent versions |
| agentVersion | string | No | The version of the agent to use. If not specified, a new version is created |

When redeploying an agent with changes, you must ensure the agent version is updated to avoid deployment failures with "agent already exists" errors. The recommended way to handle this is to include the `lastUpdated` property in the agent's description, which automatically updates whenever the agent is modified. This ensures a new version is created on each deployment.

Example:

```ts fixture=default
const agent = new bedrock.Agent(this, 'Agent', {
  foundationModel: bedrock.BedrockFoundationModel.ANTHROPIC_CLAUDE_HAIKU_V1_0,
  instruction: 'You are a helpful and friendly agent that answers questions about literature.',
});

const agentAlias = new bedrock.AgentAlias(this, 'myAlias', {
  agentAliasName: 'production',
  agent: agent,
  description: `Production version of my agent. Created at ${agent.lastUpdated}` // ensure the version update
});
```
