import { IResource, Resource } from 'aws-cdk-lib';
import * as events from 'aws-cdk-lib/aws-events';
import * as iam from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { IAgent } from './agent';
/******************************************************************************
 *                              COMMON
 *****************************************************************************/
/**
 * Represents an Agent Alias, either created with CDK or imported.
 */
export interface IAgentAlias extends IResource {
    /**
     * The unique identifier of the agent alias.
     * @attributes
     */
    readonly aliasId: string;
    /**
     * The ARN of the agent alias.
     * @attributes
     */
    readonly aliasArn: string;
    /**
     * The underlying agent for this alias.
     */
    readonly agent: IAgent;
    /**
     * Grant the given principal identity permissions to perform actions on this agent alias.
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to invoke the agent alias.
     */
    grantInvoke(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to get the agent alias.
     */
    grantGet(grantee: iam.IGrantable): iam.Grant;
    /**
     * Define an EventBridge rule that triggers when something happens to this agent alias
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id: string, options?: events.OnEventOptions): events.Rule;
}
/******************************************************************************
 *                        ABSTRACT BASE CLASS
 *****************************************************************************/
/**
 * Abstract base class for an Agent.
 * Contains methods and attributes valid for Agents either created with CDK or imported.
 */
export declare abstract class AgentAliasBase extends Resource implements IAgentAlias {
    abstract readonly aliasId: string;
    abstract readonly aliasArn: string;
    abstract readonly agent: IAgent;
    /**
     * Grant the given principal identity permissions to perform actions on this agent alias.
     * Note: This grant will only work when the grantee is in the same AWS account
     * where the agent alias is defined. Cross-account grant is not supported.
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to invoke the agent alias.
     * Note: This grant will only work when the grantee is in the same AWS account
     * where the agent alias is defined. Cross-account invocation is not supported.
     */
    grantInvoke(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to get the agent alias.
     * Note: This grant will only work when the grantee is in the same AWS account
     * where the agent alias is defined. Cross-account agent read is not supported.
     */
    grantGet(grantee: iam.IGrantable): iam.Grant;
    /**
     * Define an EventBridge rule that triggers when something happens to this agent alias
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id: string, options?: events.OnEventOptions): events.Rule;
}
/******************************************************************************
 *                        PROPS FOR NEW CONSTRUCT
 *****************************************************************************/
/**
 * Properties for creating a CDK-Managed Agent Alias.
 */
export interface AgentAliasProps {
    /**
     * The name for the agent alias.
     * This will be used as the physical name of the agent alias.
     *
     * @default - "latest"
     */
    readonly agentAliasName?: string;
    /**
     * The version of the agent to associate with the agent alias.
     *
     * @default - Creates a new version of the agent.
     */
    readonly agentVersion?: string;
    /**
     * The agent associated to this alias.
     */
    readonly agent: IAgent;
    /**
     * Description for the agent alias.
     * @default undefined - No description is provided
     */
    readonly description?: string;
}
/******************************************************************************
 *                      ATTRS FOR IMPORTED CONSTRUCT
 *****************************************************************************/
/**
 * Attributes needed to create an import
 */
export interface AgentAliasAttributes {
    /**
     * The Id of the agent alias.
     */
    readonly aliasId: string;
    /**
     * The name of the agent alias.
     * @default undefined - No alias name is provided
     */
    readonly aliasName?: string;
    /**
     * The underlying agent for this alias.
     */
    readonly agent: IAgent;
    /**
     * The agent version for this alias.
     */
    readonly agentVersion: string;
}
/******************************************************************************
 *                        NEW CONSTRUCT DEFINITION
 *****************************************************************************/
/**
 * Class to create an Agent Alias with CDK.
 * @cloudformationResource AWS::Bedrock::AgentAlias
 */
export declare class AgentAlias extends AgentAliasBase {
    /** Uniquely identifies this class. */
    static readonly PROPERTY_INJECTION_ID: string;
    /**
     * Brings an Agent Alias from an existing one created outside of CDK.
     */
    static fromAttributes(scope: Construct, id: string, attrs: AgentAliasAttributes): IAgentAlias;
    readonly agent: IAgent;
    readonly aliasId: string;
    readonly aliasArn: string;
    /**
     * The name of the agent alias.
     * This is either provided by the user or generated from a hash.
     */
    readonly aliasName: string;
    constructor(scope: Construct, id: string, props: AgentAliasProps);
}
