"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Memory = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("aws-cdk-lib/core");
const validation = require("./validation-helpers");
/**
 * Memory options for agent conversational context retention.
 * Memory enables agents to maintain context across multiple sessions and recall past interactions.
 * By default, agents retain context from the current session only.
 */
var MemoryType;
(function (MemoryType) {
    /**
     * Your agent uses memory summarization to enhance accuracy using
     * advanced prompt templates in Amazon Bedrock to call the foundation model with guidelines
     * to summarize all your sessions. You can optionally modify the default prompt template
     * or provide your own custom parser to parse model output.
     *
     * Since the summarization process takes place in an asynchronous flow after a session ends,
     * logs for any failures in summarization due to overridden template or parser will be
     * published to your AWS accounts. For more information on enabling the logging, see
     * Enable memory summarization log delivery.
     */
    MemoryType["SESSION_SUMMARY"] = "SESSION_SUMMARY";
})(MemoryType || (MemoryType = {}));
/**
 * Memory class for managing Bedrock Agent memory configurations. Enables conversational context retention
 * across multiple sessions through session identifiers. Memory context is stored with unique
 * memory IDs per user, allowing access to conversation history and summaries. Supports viewing
 * stored sessions and clearing memory.
 *
 * @see https://docs.aws.amazon.com/bedrock/latest/userguide/agents-memory.html
 */
class Memory {
    /**
     * Creates a session summary memory with custom configuration.
     * @param props Optional memory configuration properties
     * @returns Memory instance
     */
    static sessionSummary(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_SessionSummaryMemoryProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sessionSummary);
            }
            throw error;
        }
        return new Memory(props);
    }
    constructor(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_bedrock_alpha_SessionSummaryMemoryProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Memory);
            }
            throw error;
        }
        // Validate props
        validation.throwIfInvalid((config) => {
            let errors = [];
            // Validate memory duration is between 1 and 365 days
            if (config.memoryDuration !== undefined) {
                const days = config.memoryDuration.toDays();
                if (days < 1 || days > 365) {
                    errors.push('memoryDuration must be between 1 and 365 days');
                }
            }
            if (config.maxRecentSessions !== undefined) {
                if (config.maxRecentSessions < 1) {
                    errors.push('maxRecentSessions must be greater than 0');
                }
            }
            return errors;
        }, props);
        this.memoryDuration = props.memoryDuration;
        this.maxRecentSessions = props.maxRecentSessions;
    }
    /**
     * Render the memory configuration to a CloudFormation property.
     * @internal
     */
    _render() {
        return {
            enabledMemoryTypes: [MemoryType.SESSION_SUMMARY],
            storageDays: this.memoryDuration?.toDays() ?? 30,
            sessionSummaryConfiguration: {
                maxRecentSessions: this.maxRecentSessions ?? 20,
            },
        };
    }
}
exports.Memory = Memory;
_a = JSII_RTTI_SYMBOL_1;
Memory[_a] = { fqn: "@aws-cdk/aws-bedrock-alpha.Memory", version: "2.204.0-alpha.0" };
/**
 * Returns session summary memory with default configuration.
 * @default memoryDuration=Duration.days(30), maxRecentSessions=20
 */
Memory.SESSION_SUMMARY = new Memory({ memoryDuration: core_1.Duration.days(30), maxRecentSessions: 20 });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWVtb3J5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibWVtb3J5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7OztBQUFBLDJDQUE0QztBQUU1QyxtREFBbUQ7QUFFbkQ7Ozs7R0FJRztBQUNILElBQUssVUFhSjtBQWJELFdBQUssVUFBVTtJQUNiOzs7Ozs7Ozs7O09BVUc7SUFDSCxpREFBbUMsQ0FBQTtBQUNyQyxDQUFDLEVBYkksVUFBVSxLQUFWLFVBQVUsUUFhZDtBQW1CRDs7Ozs7OztHQU9HO0FBQ0gsTUFBYSxNQUFNO0lBT2pCOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQWdDOzs7Ozs7Ozs7O1FBQzNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7S0FDMUI7SUFLRCxZQUFZLEtBQWdDOzs7Ozs7K0NBbkJqQyxNQUFNOzs7O1FBb0JmLGlCQUFpQjtRQUNqQixVQUFVLENBQUMsY0FBYyxDQUFDLENBQUMsTUFBaUMsRUFBRSxFQUFFO1lBQzlELElBQUksTUFBTSxHQUFhLEVBQUUsQ0FBQztZQUUxQixxREFBcUQ7WUFDckQsSUFBSSxNQUFNLENBQUMsY0FBYyxLQUFLLFNBQVMsRUFBRSxDQUFDO2dCQUN4QyxNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxDQUFDO2dCQUM1QyxJQUFJLElBQUksR0FBRyxDQUFDLElBQUksSUFBSSxHQUFHLEdBQUcsRUFBRSxDQUFDO29CQUMzQixNQUFNLENBQUMsSUFBSSxDQUFDLCtDQUErQyxDQUFDLENBQUM7Z0JBQy9ELENBQUM7WUFDSCxDQUFDO1lBQ0QsSUFBSSxNQUFNLENBQUMsaUJBQWlCLEtBQUssU0FBUyxFQUFFLENBQUM7Z0JBQzNDLElBQUksTUFBTSxDQUFDLGlCQUFpQixHQUFHLENBQUMsRUFBRSxDQUFDO29CQUNqQyxNQUFNLENBQUMsSUFBSSxDQUFDLDBDQUEwQyxDQUFDLENBQUM7Z0JBQzFELENBQUM7WUFDSCxDQUFDO1lBRUQsT0FBTyxNQUFNLENBQUM7UUFDaEIsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRVYsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1FBQzNDLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUM7S0FDbEQ7SUFFRDs7O09BR0c7SUFDSSxPQUFPO1FBQ1osT0FBTztZQUNMLGtCQUFrQixFQUFFLENBQUMsVUFBVSxDQUFDLGVBQWUsQ0FBQztZQUNoRCxXQUFXLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFO1lBQ2hELDJCQUEyQixFQUFFO2dCQUMzQixpQkFBaUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCLElBQUksRUFBRTthQUNoRDtTQUNGLENBQUM7S0FDSDs7QUF4REgsd0JBeURDOzs7QUF4REM7OztHQUdHO0FBQ29CLHNCQUFlLEdBQUcsSUFBSSxNQUFNLENBQUMsRUFBRSxjQUFjLEVBQUUsZUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxpQkFBaUIsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRHVyYXRpb24gfSBmcm9tICdhd3MtY2RrLWxpYi9jb3JlJztcbmltcG9ydCB7IENmbkFnZW50IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWJlZHJvY2snO1xuaW1wb3J0ICogYXMgdmFsaWRhdGlvbiBmcm9tICcuL3ZhbGlkYXRpb24taGVscGVycyc7XG5cbi8qKlxuICogTWVtb3J5IG9wdGlvbnMgZm9yIGFnZW50IGNvbnZlcnNhdGlvbmFsIGNvbnRleHQgcmV0ZW50aW9uLlxuICogTWVtb3J5IGVuYWJsZXMgYWdlbnRzIHRvIG1haW50YWluIGNvbnRleHQgYWNyb3NzIG11bHRpcGxlIHNlc3Npb25zIGFuZCByZWNhbGwgcGFzdCBpbnRlcmFjdGlvbnMuXG4gKiBCeSBkZWZhdWx0LCBhZ2VudHMgcmV0YWluIGNvbnRleHQgZnJvbSB0aGUgY3VycmVudCBzZXNzaW9uIG9ubHkuXG4gKi9cbmVudW0gTWVtb3J5VHlwZSB7XG4gIC8qKlxuICAgKiBZb3VyIGFnZW50IHVzZXMgbWVtb3J5IHN1bW1hcml6YXRpb24gdG8gZW5oYW5jZSBhY2N1cmFjeSB1c2luZ1xuICAgKiBhZHZhbmNlZCBwcm9tcHQgdGVtcGxhdGVzIGluIEFtYXpvbiBCZWRyb2NrIHRvIGNhbGwgdGhlIGZvdW5kYXRpb24gbW9kZWwgd2l0aCBndWlkZWxpbmVzXG4gICAqIHRvIHN1bW1hcml6ZSBhbGwgeW91ciBzZXNzaW9ucy4gWW91IGNhbiBvcHRpb25hbGx5IG1vZGlmeSB0aGUgZGVmYXVsdCBwcm9tcHQgdGVtcGxhdGVcbiAgICogb3IgcHJvdmlkZSB5b3VyIG93biBjdXN0b20gcGFyc2VyIHRvIHBhcnNlIG1vZGVsIG91dHB1dC5cbiAgICpcbiAgICogU2luY2UgdGhlIHN1bW1hcml6YXRpb24gcHJvY2VzcyB0YWtlcyBwbGFjZSBpbiBhbiBhc3luY2hyb25vdXMgZmxvdyBhZnRlciBhIHNlc3Npb24gZW5kcyxcbiAgICogbG9ncyBmb3IgYW55IGZhaWx1cmVzIGluIHN1bW1hcml6YXRpb24gZHVlIHRvIG92ZXJyaWRkZW4gdGVtcGxhdGUgb3IgcGFyc2VyIHdpbGwgYmVcbiAgICogcHVibGlzaGVkIHRvIHlvdXIgQVdTIGFjY291bnRzLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiBvbiBlbmFibGluZyB0aGUgbG9nZ2luZywgc2VlXG4gICAqIEVuYWJsZSBtZW1vcnkgc3VtbWFyaXphdGlvbiBsb2cgZGVsaXZlcnkuXG4gICAqL1xuICBTRVNTSU9OX1NVTU1BUlkgPSAnU0VTU0lPTl9TVU1NQVJZJyxcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBTZXNzaW9uU3VtbWFyeUNvbmZpZ3VyYXRpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2Vzc2lvblN1bW1hcnlNZW1vcnlQcm9wcyB7XG4gIC8qKlxuICAgKiBEdXJhdGlvbiBmb3Igd2hpY2ggc2Vzc2lvbiBzdW1tYXJpZXMgYXJlIHJldGFpbmVkIChiZXR3ZWVuIDEgYW5kIDM2NSBkYXlzKVxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5kYXlzKDMwKVxuICAgKi9cbiAgcmVhZG9ubHkgbWVtb3J5RHVyYXRpb24/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogTWF4aW11bSBudW1iZXIgb2YgcmVjZW50IHNlc3Npb24gc3VtbWFyaWVzIHRvIGluY2x1ZGUgKG1pbiAxKVxuICAgKiBAZGVmYXVsdCAyMFxuICAgKi9cbiAgcmVhZG9ubHkgbWF4UmVjZW50U2Vzc2lvbnM/OiBudW1iZXI7XG59XG5cbi8qKlxuICogTWVtb3J5IGNsYXNzIGZvciBtYW5hZ2luZyBCZWRyb2NrIEFnZW50IG1lbW9yeSBjb25maWd1cmF0aW9ucy4gRW5hYmxlcyBjb252ZXJzYXRpb25hbCBjb250ZXh0IHJldGVudGlvblxuICogYWNyb3NzIG11bHRpcGxlIHNlc3Npb25zIHRocm91Z2ggc2Vzc2lvbiBpZGVudGlmaWVycy4gTWVtb3J5IGNvbnRleHQgaXMgc3RvcmVkIHdpdGggdW5pcXVlXG4gKiBtZW1vcnkgSURzIHBlciB1c2VyLCBhbGxvd2luZyBhY2Nlc3MgdG8gY29udmVyc2F0aW9uIGhpc3RvcnkgYW5kIHN1bW1hcmllcy4gU3VwcG9ydHMgdmlld2luZ1xuICogc3RvcmVkIHNlc3Npb25zIGFuZCBjbGVhcmluZyBtZW1vcnkuXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmVkcm9jay9sYXRlc3QvdXNlcmd1aWRlL2FnZW50cy1tZW1vcnkuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgTWVtb3J5IHtcbiAgLyoqXG4gICAqIFJldHVybnMgc2Vzc2lvbiBzdW1tYXJ5IG1lbW9yeSB3aXRoIGRlZmF1bHQgY29uZmlndXJhdGlvbi5cbiAgICogQGRlZmF1bHQgbWVtb3J5RHVyYXRpb249RHVyYXRpb24uZGF5cygzMCksIG1heFJlY2VudFNlc3Npb25zPTIwXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNFU1NJT05fU1VNTUFSWSA9IG5ldyBNZW1vcnkoeyBtZW1vcnlEdXJhdGlvbjogRHVyYXRpb24uZGF5cygzMCksIG1heFJlY2VudFNlc3Npb25zOiAyMCB9KTtcblxuICAvKipcbiAgICogQ3JlYXRlcyBhIHNlc3Npb24gc3VtbWFyeSBtZW1vcnkgd2l0aCBjdXN0b20gY29uZmlndXJhdGlvbi5cbiAgICogQHBhcmFtIHByb3BzIE9wdGlvbmFsIG1lbW9yeSBjb25maWd1cmF0aW9uIHByb3BlcnRpZXNcbiAgICogQHJldHVybnMgTWVtb3J5IGluc3RhbmNlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHNlc3Npb25TdW1tYXJ5KHByb3BzOiBTZXNzaW9uU3VtbWFyeU1lbW9yeVByb3BzKTogTWVtb3J5IHtcbiAgICByZXR1cm4gbmV3IE1lbW9yeShwcm9wcyk7XG4gIH1cblxuICBwcml2YXRlIHJlYWRvbmx5IG1lbW9yeUR1cmF0aW9uPzogRHVyYXRpb247XG4gIHByaXZhdGUgcmVhZG9ubHkgbWF4UmVjZW50U2Vzc2lvbnM/OiBudW1iZXI7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IFNlc3Npb25TdW1tYXJ5TWVtb3J5UHJvcHMpIHtcbiAgICAvLyBWYWxpZGF0ZSBwcm9wc1xuICAgIHZhbGlkYXRpb24udGhyb3dJZkludmFsaWQoKGNvbmZpZzogU2Vzc2lvblN1bW1hcnlNZW1vcnlQcm9wcykgPT4ge1xuICAgICAgbGV0IGVycm9yczogc3RyaW5nW10gPSBbXTtcblxuICAgICAgLy8gVmFsaWRhdGUgbWVtb3J5IGR1cmF0aW9uIGlzIGJldHdlZW4gMSBhbmQgMzY1IGRheXNcbiAgICAgIGlmIChjb25maWcubWVtb3J5RHVyYXRpb24gIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBjb25zdCBkYXlzID0gY29uZmlnLm1lbW9yeUR1cmF0aW9uLnRvRGF5cygpO1xuICAgICAgICBpZiAoZGF5cyA8IDEgfHwgZGF5cyA+IDM2NSkge1xuICAgICAgICAgIGVycm9ycy5wdXNoKCdtZW1vcnlEdXJhdGlvbiBtdXN0IGJlIGJldHdlZW4gMSBhbmQgMzY1IGRheXMnKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgaWYgKGNvbmZpZy5tYXhSZWNlbnRTZXNzaW9ucyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmIChjb25maWcubWF4UmVjZW50U2Vzc2lvbnMgPCAxKSB7XG4gICAgICAgICAgZXJyb3JzLnB1c2goJ21heFJlY2VudFNlc3Npb25zIG11c3QgYmUgZ3JlYXRlciB0aGFuIDAnKTtcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICByZXR1cm4gZXJyb3JzO1xuICAgIH0sIHByb3BzKTtcblxuICAgIHRoaXMubWVtb3J5RHVyYXRpb24gPSBwcm9wcy5tZW1vcnlEdXJhdGlvbjtcbiAgICB0aGlzLm1heFJlY2VudFNlc3Npb25zID0gcHJvcHMubWF4UmVjZW50U2Vzc2lvbnM7XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIHRoZSBtZW1vcnkgY29uZmlndXJhdGlvbiB0byBhIENsb3VkRm9ybWF0aW9uIHByb3BlcnR5LlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfcmVuZGVyKCk6IENmbkFnZW50Lk1lbW9yeUNvbmZpZ3VyYXRpb25Qcm9wZXJ0eSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGVuYWJsZWRNZW1vcnlUeXBlczogW01lbW9yeVR5cGUuU0VTU0lPTl9TVU1NQVJZXSxcbiAgICAgIHN0b3JhZ2VEYXlzOiB0aGlzLm1lbW9yeUR1cmF0aW9uPy50b0RheXMoKSA/PyAzMCxcbiAgICAgIHNlc3Npb25TdW1tYXJ5Q29uZmlndXJhdGlvbjoge1xuICAgICAgICBtYXhSZWNlbnRTZXNzaW9uczogdGhpcy5tYXhSZWNlbnRTZXNzaW9ucyA/PyAyMCxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuIl19