import { CfnAgent } from 'aws-cdk-lib/aws-bedrock';
import { IFunction } from 'aws-cdk-lib/aws-lambda';
import { IBedrockInvokable } from '../models';
/**
 * The step in the agent sequence that this prompt configuration applies to.
 */
export declare enum AgentStepType {
    /**
     * Pre-processing step that prepares the user input for orchestration.
     */
    PRE_PROCESSING = "PRE_PROCESSING",
    /**
     * Main orchestration step that determines the agent's actions.
     */
    ORCHESTRATION = "ORCHESTRATION",
    /**
     * Post-processing step that refines the agent's response.
     */
    POST_PROCESSING = "POST_PROCESSING",
    /**
     * Step that classifies and routes requests to appropriate collaborators.
     */
    ROUTING_CLASSIFIER = "ROUTING_CLASSIFIER",
    /**
     * Step that summarizes conversation history for memory retention.
     */
    MEMORY_SUMMARIZATION = "MEMORY_SUMMARIZATION",
    /**
     * Step that generates responses using knowledge base content.
     */
    KNOWLEDGE_BASE_RESPONSE_GENERATION = "KNOWLEDGE_BASE_RESPONSE_GENERATION"
}
/**
 * LLM inference configuration
 */
export interface InferenceConfiguration {
    /**
     * The likelihood of the model selecting higher-probability options while
     * generating a response. A lower value makes the model more likely to choose
     * higher-probability options, while a higher value makes the model more
     * likely to choose lower-probability options.
     *
     * Floating point
     *
     * min 0
     * max 1
     */
    readonly temperature: number;
    /**
     * While generating a response, the model determines the probability of the
     * following token at each point of generation. The value that you set for
     * Top P determines the number of most-likely candidates from which the model
     * chooses the next token in the sequence. For example, if you set topP to
     * 80, the model only selects the next token from the top 80% of the
     * probability distribution of next tokens.
     *
     * Floating point
     *
     * min 0
     * max 1
     */
    readonly topP: number;
    /**
     * While generating a response, the model determines the probability of the
     * following token at each point of generation. The value that you set for
     * topK is the number of most-likely candidates from which the model chooses
     * the next token in the sequence. For example, if you set topK to 50, the
     * model selects the next token from among the top 50 most likely choices.
     *
     * Integer
     *
     * min 0
     * max 500
     */
    readonly topK: number;
    /**
     * A list of stop sequences. A stop sequence is a sequence of characters that
     * causes the model to stop generating the response.
     *
     * length 0-4
     */
    readonly stopSequences: string[];
    /**
     * The maximum number of tokens to generate in the response.
     *
     * Integer
     *
     * min 0
     * max 4096
     */
    readonly maximumLength: number;
}
/**
 * Base configuration interface for all prompt step types
 */
export interface PromptStepConfigBase {
    /**
     * The type of step this configuration applies to.
     */
    readonly stepType: AgentStepType;
    /**
     * Whether to enable or skip this step in the agent sequence.
     * @default - The default state for each step type is as follows.
     *
     *     PRE_PROCESSING – ENABLED
     *     ORCHESTRATION – ENABLED
     *     KNOWLEDGE_BASE_RESPONSE_GENERATION – ENABLED
     *     POST_PROCESSING – DISABLED
     */
    readonly stepEnabled?: boolean;
    /**
     * The custom prompt template to be used.
     *
     * @default - The default prompt template will be used.
     * @see https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html
     */
    readonly customPromptTemplate?: string;
    /**
     * The inference configuration parameters to use.
     * @default undefined - Default inference configuration will be used
     */
    readonly inferenceConfig?: InferenceConfiguration;
    /**
     * Whether to use the custom Lambda parser defined for the sequence.
     *
     * @default - false
     */
    readonly useCustomParser?: boolean;
}
/**
 * Configuration for the pre-processing step
 */
export interface PromptPreProcessingConfigCustomParser extends PromptStepConfigBase {
}
/**
 * Configuration for the orchestration step
 */
export interface PromptOrchestrationConfigCustomParser extends PromptStepConfigBase {
}
/**
 * Configuration for the post-processing step
 */
export interface PromptPostProcessingConfigCustomParser extends PromptStepConfigBase {
}
/**
 * Configuration for the routing classifier step
 */
export interface PromptRoutingClassifierConfigCustomParser extends PromptStepConfigBase {
    /**
     * The foundation model to use for the routing classifier step.
     * This is required for the routing classifier step.
     */
    readonly foundationModel: IBedrockInvokable;
}
/**
 * Configuration for the memory summarization step
 */
export interface PromptMemorySummarizationConfigCustomParser extends PromptStepConfigBase {
}
/**
 * Configuration for the knowledge base response generation step
 */
export interface PromptKnowledgeBaseResponseGenerationConfigCustomParser extends PromptStepConfigBase {
}
/**
 * Properties for configuring a custom Lambda parser for prompt overrides.
 */
export interface CustomParserProps {
    /**
     * Lambda function to use as custom parser.
     * @default undefined - No custom parser is used
     */
    readonly parser?: IFunction;
    /**
     * Configuration for the pre-processing step.
     * @default undefined - No pre-processing configuration
     */
    readonly preProcessingStep?: PromptPreProcessingConfigCustomParser;
    /**
     * Configuration for the orchestration step.
     * @default undefined - No orchestration configuration
     */
    readonly orchestrationStep?: PromptOrchestrationConfigCustomParser;
    /**
     * Configuration for the post-processing step.
     * @default undefined - No post-processing configuration
     */
    readonly postProcessingStep?: PromptPostProcessingConfigCustomParser;
    /**
     * Configuration for the routing classifier step.
     * @default undefined - No routing classifier configuration
     */
    readonly routingClassifierStep?: PromptRoutingClassifierConfigCustomParser;
    /**
     * Configuration for the memory summarization step.
     * @default undefined - No memory summarization configuration
     */
    readonly memorySummarizationStep?: PromptMemorySummarizationConfigCustomParser;
    /**
     * Configuration for the knowledge base response generation step.
     * @default undefined - No knowledge base response generation configuration
     */
    readonly knowledgeBaseResponseGenerationStep?: PromptKnowledgeBaseResponseGenerationConfigCustomParser;
}
/**
 * Configuration for overriding prompt templates and behaviors in different parts
 * of an agent's sequence. This allows customizing how the agent processes inputs,
 * makes decisions, and generates responses.
 */
export declare class PromptOverrideConfiguration {
    /**
     * Creates a PromptOverrideConfiguration from individual step configurations.
     * Use this method when you want to override prompts without using a custom parser.
     * @param steps The step configurations to use
     * @returns A new PromptOverrideConfiguration instance
     */
    static fromSteps(steps: PromptStepConfigBase[]): PromptOverrideConfiguration;
    /**
     * Creates a PromptOverrideConfiguration with a custom Lambda parser function.
     * @param props Configuration including:
     *   - `parser`: Lambda function to use as custom parser
     *   - Individual step configurations. At least one of the steps must make use of the custom parser.
     */
    static withCustomParser(props: CustomParserProps): PromptOverrideConfiguration;
    /**
     * The custom Lambda parser function to use.
     * The Lambda parser processes and interprets the raw foundation model output.
     * It receives an input event with:
     * - messageVersion: Version of message format (1.0)
     * - agent: Info about the agent (name, id, alias, version)
     * - invokeModelRawResponse: Raw model output to parse
     * - promptType: Type of prompt being parsed
     * - overrideType: Type of override (OUTPUT_PARSER)
     *
     * The Lambda must return a response that the agent uses for next actions.
     * @see https://docs.aws.amazon.com/bedrock/latest/userguide/lambda-parser.html
     */
    readonly parser?: IFunction;
    /**
     * Configuration for the pre-processing step.
     */
    readonly preProcessingStep?: PromptPreProcessingConfigCustomParser;
    /**
     * Configuration for the orchestration step.
     */
    readonly orchestrationStep?: PromptOrchestrationConfigCustomParser;
    /**
     * Configuration for the post-processing step.
     */
    readonly postProcessingStep?: PromptPostProcessingConfigCustomParser;
    /**
     * Configuration for the routing classifier step.
     */
    readonly routingClassifierStep?: PromptRoutingClassifierConfigCustomParser;
    /**
     * Configuration for the memory summarization step.
     */
    readonly memorySummarizationStep?: PromptMemorySummarizationConfigCustomParser;
    /**
     * Configuration for the knowledge base response generation step.
     */
    readonly knowledgeBaseResponseGenerationStep?: PromptKnowledgeBaseResponseGenerationConfigCustomParser;
    /**
     * Create a new PromptOverrideConfiguration.
     *
     * @internal - This is marked as private so end users leverage it only through static methods
     */
    private constructor();
    /**
     * Format as CfnAgent.PromptOverrideConfigurationProperty
     *
     * @internal This is an internal core function and should not be called directly.
     */
    _render(): CfnAgent.PromptOverrideConfigurationProperty;
    private validateInferenceConfig;
    private validateSteps;
    private validateCustomParser;
}
