"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.recommendedMaximumLambdaMemorySize = exports.maximumLambdaMemorySizeContextItem = exports.version = void 0;
exports.generatePhysicalName = generatePhysicalName;
exports.generatePhysicalNameV2 = generatePhysicalNameV2;
exports.lambdaMemorySizeLimiter = lambdaMemorySizeLimiter;
exports.addCfnSuppressRules = addCfnSuppressRules;
exports.isPlainObject = isPlainObject;
const crypto_1 = require("crypto");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * The version of this package
 */
// eslint-disable-next-line @typescript-eslint/no-require-imports
exports.version = require('../../../package.json').version;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * @summary Creates a physical resource name in the style of the CDK (string+hash) - this value incorporates Stack ID,
 * so it will remain static in multiple updates of a single stack, but will be different in a separate stack instance
 * @param {IConstruct} scope - the CDK scope of the resource
 * @param {string} prefix - the prefix for the name
 * @param {string[]} parts - the various string components of the name (eg - stackName, solutions construct ID, L2 construct ID)
 * @param {number} maxLength - the longest string that can be returned
 * @returns {string} - a string with concatenated parts (truncated if necessary) + a hash of the full concatenated parts
 *
 * @deprecated This function is deprecated and will be removed in a future major version.
 * Please use the new function generatePhysicalNameV2 instead.
 */
function generatePhysicalName(scope, prefix, parts, maxLength) {
    // The result will consist of:
    //    -The prefix - unaltered
    //    -The parts concatenated, but reduced in size to meet the maxLength limit for the overall name
    //    -A hyphen delimiter
    //    -The GUID portion of the stack arn
    const stackIdGuidLength = 36;
    const prefixLength = prefix.length;
    const maxPartsLength = maxLength - prefixLength - 1 - stackIdGuidLength; // 1 is the hyphen
    // Extract the Stack ID Guid
    const uniqueStackIdPart = cdk.Fn.select(2, cdk.Fn.split('/', `${cdk.Aws.STACK_ID}`));
    let allParts = '';
    parts.forEach((part) => {
        allParts += part;
    });
    if (allParts.length > maxPartsLength) {
        const subStringLength = maxPartsLength / 2;
        allParts = allParts.substring(0, subStringLength) + allParts.substring(allParts.length - subStringLength);
    }
    if (prefix.length + allParts.length + stackIdGuidLength + 1 /* hyphen */ > maxLength) {
        throw new aws_cdk_lib_1.ValidationError(`The generated name is longer than the maximum length of ${maxLength}`, scope);
    }
    return prefix.toLowerCase() + allParts + '-' + uniqueStackIdPart;
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * @summary Creates a physical resource name in the style of the CDK (string+hash) - this value incorporates
 * the Stack Name and node ID, so it will remain static in multiple updates of a single stack, but will be
 * different in a separate stack instance.
 *
 * This new version allows for names shorter than 36 characters with control over casing.
 *
 * The minimum length is the length of the prefix and separator plus 10.
 */
function generatePhysicalNameV2(
/**
 * The CDK scope of the resource.
 */
scope, 
/**
 * The prefix for the name.
 */
prefix, 
/**
 * Options for generating the name.
 */
options) {
    function objectToHash(obj) {
        // Nothing to hash if undefined
        if (obj === undefined) {
            return '';
        }
        // Convert the object to a JSON string
        const jsonString = JSON.stringify(obj);
        // Create a SHA-256 hash
        const hash = (0, crypto_1.createHash)('sha256');
        // Update the hash with the JSON string and get the digest in hexadecimal format
        // Shorten it (modeled after seven characters like git commit hash shortening)
        return hash.update(jsonString).digest('hex').slice(0, 7);
    }
    const { maxLength = 256, lower = false, separator = '', allowedSpecialCharacters = undefined, destroyCreate = undefined, } = options ?? {};
    const hash = objectToHash(destroyCreate);
    if (maxLength < (prefix + hash + separator).length) {
        throw new aws_cdk_lib_1.ValidationError(`The prefix length (${prefix.length}) plus hash length (${hash.length}) and separator length (${separator.length}) exceeds the maximum allowed length of ${maxLength}`, scope);
    }
    const uniqueName = cdk.Names.uniqueResourceName(scope, { maxLength: maxLength - (prefix + hash + separator).length, separator, allowedSpecialCharacters });
    const name = `${prefix}${hash}${separator}${uniqueName}`;
    if (name.length > maxLength) {
        throw new aws_cdk_lib_1.ValidationError(`The generated name is longer than the maximum length of ${maxLength}`, scope);
    }
    return lower ? name.toLowerCase() : name;
}
exports.maximumLambdaMemorySizeContextItem = 'maximumLambdaMemorySize';
exports.recommendedMaximumLambdaMemorySize = 7076;
function lambdaMemorySizeLimiter(construct, requestedMemorySizeInMegabytes) {
    const maximumLambdaMemorySize = construct.node.tryGetContext(exports.maximumLambdaMemorySizeContextItem) === undefined ?
        exports.recommendedMaximumLambdaMemorySize :
        parseInt(construct.node.tryGetContext(exports.maximumLambdaMemorySizeContextItem));
    if (maximumLambdaMemorySize < exports.recommendedMaximumLambdaMemorySize) {
        cdk.Annotations.of(construct).addWarning(`Maximum Lambda memorySize, ${maximumLambdaMemorySize}, is less than the recommended ${exports.recommendedMaximumLambdaMemorySize}.`);
    }
    if (requestedMemorySizeInMegabytes > maximumLambdaMemorySize) {
        cdk.Annotations.of(construct).addWarning(`Reducing Lambda memorySize, ${requestedMemorySizeInMegabytes} to ${maximumLambdaMemorySize} for ${construct.constructor.name}`);
        return maximumLambdaMemorySize;
    }
    else {
        return requestedMemorySizeInMegabytes;
    }
}
/**
 * Adds CFN NAG suppress rules to the CDK resource.
 * @param resource The CDK resource
 * @param rules The CFN NAG suppress rules
 */
function addCfnSuppressRules(resource, rules) {
    if (resource instanceof cdk.Resource) {
        resource = resource.node.defaultChild;
    }
    if (resource.cfnOptions.metadata?.cfn_nag?.rules_to_suppress) {
        resource.cfnOptions.metadata?.cfn_nag.rules_to_suppress.push(...rules);
    }
    else {
        resource.addMetadata('cfn_nag', {
            rules_to_suppress: rules,
        });
    }
}
function isObject(val) {
    return val !== null && typeof val === 'object' && !Array.isArray(val);
}
function isPlainObject(o) {
    if (!isObject(o))
        return false;
    if (Object.getPrototypeOf(o) === null)
        return true;
    let proto = o;
    while (Object.getPrototypeOf(proto) !== null) {
        proto = Object.getPrototypeOf(proto);
    }
    return Object.getPrototypeOf(o) === proto;
}
//# sourceMappingURL=data:application/json;base64,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