import re
from web3 import Web3
from eth_account import Account
from typing import Optional

class RubyConverter:
    """
    Converter for Ruby-style addresses and hashes
    Converts between 'r...' format and '0x...' format
    """
    
    @staticmethod
    def to_ruby_format(hex_string: str) -> str:
        """
        Convert hex string (0x...) to Ruby format (r...)
        """
        if not hex_string:
            return hex_string
            
        # Remove 0x prefix if present
        if hex_string.startswith('0x'):
            hex_string = hex_string[2:]
        
        # Add 'r' prefix
        return f"r{hex_string}"
    
    @staticmethod
    def from_ruby_format(ruby_string: str) -> str:
        """
        Convert Ruby format (r...) to hex string (0x...)
        """
        if not ruby_string:
            return ruby_string
            
        # Remove 'r' prefix if present
        if ruby_string.startswith('r'):
            ruby_string = ruby_string[1:]
        
        # Add '0x' prefix
        return f"0x{ruby_string}"
    
    @staticmethod
    def normalize_private_key(private_key: str) -> str:
        """
        Ensure private key has 0x prefix
        """
        if not private_key:
            return private_key
            
        if private_key.startswith('r'):
            private_key = private_key[1:]
            
        if not private_key.startswith('0x'):
            private_key = f"0x{private_key}"
            
        return private_key
    
    @staticmethod
    def is_ruby_format(string: str) -> bool:
        """
        Check if string is in Ruby format (starts with 'r')
        """
        return string.startswith('r') if string else False
    
    @staticmethod
    def is_hex_format(string: str) -> bool:
        """
        Check if string is in hex format (starts with '0x')
        """
        return string.startswith('0x') if string else False
    
    @staticmethod
    def convert_address(address: str) -> str:
        """
        Auto-convert address to hex format for internal use
        """
        if RubyConverter.is_ruby_format(address):
            return RubyConverter.from_ruby_format(address)
        elif RubyConverter.is_hex_format(address):
            return address
        else:
            # Assume it's hex without prefix, add 0x
            return f"0x{address}"
    
    @staticmethod
    def format_response_address(address: str) -> str:
        """
        Format address for response (convert to Ruby format)
        """
        if RubyConverter.is_hex_format(address):
            return RubyConverter.to_ruby_format(address)
        return address
    
    @staticmethod
    def format_response_hash(hash_string: str) -> str:
        """
        Format hash for response (convert to Ruby format)
        """
        return RubyConverter.to_ruby_format(hash_string)