from typing import Dict, Any, List, Optional
from web3.types import TxReceipt, TxData
from .converter import RubyConverter

class ResponseFormatter:
    """
    Format responses to use Ruby-style addresses and hashes
    """
    
    @staticmethod
    def format_transaction(tx_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Format transaction data with Ruby-style addresses and hashes
        """
        formatted = tx_data.copy()
        
        # Convert hashes
        if 'hash' in formatted:
            formatted['hash'] = RubyConverter.format_response_hash(formatted['hash'])
        
        if 'blockHash' in formatted:
            formatted['blockHash'] = RubyConverter.format_response_hash(formatted.get('blockHash', ''))
        
        # Convert addresses
        if 'from' in formatted:
            formatted['from'] = RubyConverter.format_response_address(formatted['from'])
        
        if 'to' in formatted:
            formatted['to'] = RubyConverter.format_response_address(formatted['to'])
        
        if 'contractAddress' in formatted:
            formatted['contractAddress'] = RubyConverter.format_response_address(formatted.get('contractAddress', ''))
        
        return formatted
    
    @staticmethod
    def format_receipt(receipt_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Format transaction receipt with Ruby-style addresses and hashes
        """
        formatted = receipt_data.copy()
        
        # Convert hashes
        if 'transactionHash' in formatted:
            formatted['transactionHash'] = RubyConverter.format_response_hash(formatted['transactionHash'])
        
        if 'blockHash' in formatted:
            formatted['blockHash'] = RubyConverter.format_response_hash(formatted.get('blockHash', ''))
        
        # Convert addresses
        if 'from' in formatted:
            formatted['from'] = RubyConverter.format_response_address(formatted['from'])
        
        if 'to' in formatted:
            formatted['to'] = RubyConverter.format_response_address(formatted['to'])
        
        if 'contractAddress' in formatted:
            formatted['contractAddress'] = RubyConverter.format_response_address(formatted.get('contractAddress', ''))
        
        # Format logs
        if 'logs' in formatted and isinstance(formatted['logs'], list):
            for log in formatted['logs']:
                if 'address' in log:
                    log['address'] = RubyConverter.format_response_address(log['address'])
        
        return formatted
    
    @staticmethod
    def format_block(block_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Format block data with Ruby-style hashes
        """
        formatted = block_data.copy()
        
        # Convert hashes
        if 'hash' in formatted:
            formatted['hash'] = RubyConverter.format_response_hash(formatted['hash'])
        
        if 'parentHash' in formatted:
            formatted['parentHash'] = RubyConverter.format_response_hash(formatted.get('parentHash', ''))
        
        if 'transactions' in formatted and isinstance(formatted['transactions'], list):
            for i, tx in enumerate(formatted['transactions']):
                if isinstance(tx, str):  # Transaction hash
                    formatted['transactions'][i] = RubyConverter.format_response_hash(tx)
                elif isinstance(tx, dict):  # Full transaction data
                    formatted['transactions'][i] = ResponseFormatter.format_transaction(tx)
        
        return formatted