"""
Validation utilities for Ruby Manager Web3
"""

from typing import Union
from web3 import Web3
from .converter import RubyConverter

def validate_eth_amount(amount: Union[float, int, str]) -> bool:
    """
    Validate Ethereum amount
    """
    try:
        amount_float = float(amount)
        if amount_float < 0:
            return False
        if amount_float == 0:
            return True
        
        # Check if amount is reasonable (less than 1 million ETH)
        return amount_float < 1000000
    except (ValueError, TypeError):
        return False

def validate_gas_params(gas_limit: int, gas_price: int) -> bool:
    """
    Validate gas parameters
    """
    if gas_limit <= 0 or gas_price <= 0:
        return False
    
    # Reasonable limits
    if gas_limit > 30000000:  # 30 million gas
        return False
    if gas_price > 1000000000000:  # 1000 Gwei
        return False
    
    return True

def validate_transaction_hash(tx_hash: str) -> bool:
    """
    Validate transaction hash (both Ruby and hex formats)
    """
    if RubyConverter.is_ruby_format(tx_hash):
        tx_hash = RubyConverter.from_ruby_format(tx_hash)
    
    if not tx_hash.startswith('0x'):
        tx_hash = '0x' + tx_hash
    
    # Check length (64 hex chars + 0x prefix = 66 chars)
    if len(tx_hash) != 66:
        return False
    
    # Check if it's a valid hex string
    try:
        int(tx_hash, 16)
        return True
    except ValueError:
        return False

def validate_block_identifier(block_id: Union[int, str]) -> bool:
    """
    Validate block identifier
    """
    if isinstance(block_id, int):
        return block_id >= 0
    elif isinstance(block_id, str):
        return block_id in ['latest', 'earliest', 'pending', 'safe', 'finalized']
    else:
        return False

def validate_contract_abi(abi: list) -> bool:
    """
    Validate contract ABI structure
    """
    if not isinstance(abi, list):
        return False
    
    required_keys = {'type', 'name'}
    for item in abi:
        if not isinstance(item, dict):
            return False
        if not required_keys.issubset(item.keys()):
            return False
    
    return True

def validate_token_amount(amount: float, decimals: int) -> bool:
    """
    Validate token amount based on decimals
    """
    try:
        amount_float = float(amount)
        if amount_float < 0:
            return False
        
        # Check if amount has too many decimal places
        max_decimals = min(decimals, 18)  # Limit to 18 decimal places max
        str_amount = str(amount_float)
        if '.' in str_amount:
            decimal_places = len(str_amount.split('.')[1])
            if decimal_places > max_decimals:
                return False
        
        return True
    except (ValueError, TypeError):
        return False