# -*- coding: utf-8 -*-
"""Functions for Earthquake Data Analysis.ipynb - made by Riven White 2024

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/19zbpjGoLTQ4xPOfscQUzs5GmAvyTfY7T
"""

def summary(catalog, variable):
  """
  Prints mean, median, mode, and number of missing values for specified column in dataframe.

  Parameters
  ----------
  catalog : DataFrame
      The pandas dataframe to be used.
  variable : str
      The name of the column to be used.
  """

  # Import necessary packages
  from statistics import mean, median, mode
  import numpy as np
  import pandas as pd

  # Calculate and print mean, median, and (first/smallest value) mode
  print(f"{variable} mean: {catalog[variable].mean()}")
  print(f"{variable} median: {catalog[variable].median()}")
  print(f"{variable} mode: {catalog[variable].mode()[0]}")

  # Determine if there are missing values and if yes, print the count
  if catalog[variable].isna().any():
    print(f"{variable} contains {catalog[variable].isna().sum()} missing values")

  # Print an empty line to separate from other summary function outputs
  print(" ")

def earthquake_time_plot(
    catalog,
    result_label='Earthquakes',
    x_lab='Date',
    y_lab='Number of Earthquakes',
    x_format='%Y-%m-%d',
    title='Earthquakes Over Time',
    bar_color='#016392',
    edge_color='k',
    alpha=1.0,
    line_width=0.5,
    max_count_line=False,
    max_count_line_color='#01839d',
    mag_line=False,
    mag_line_color='#aacfe2',
    range=None,
    bin_count=None
):
    """
    Produces a time plot for earthquake origin time.

    Parameters
    ----------
    catalog : DataFrame
        The pandas dataframe to be used.
    result_label : str, optional
        The label to be used in the legend. Default is 'Earthquakes'.
    x_lab : str, optional
        The label to be used for the x-axis. Default is 'Date'.
    y_lab : str, optional
        The label to be used for the y-axis. Default is 'Number of Earthquakes'.
    x_format : str, optional
        The date-time format to be used. Default is '%Y-%m-%d'.
    title : str, optional
        The title of the plot. Default is 'Earthquakes Over Time'.
    bar_color : str, optional
        The color of the bars. Default is '#016392'.
    edge_color : str, optional
        The color of the bars' edges. Default is 'k'.
    alpha : float, optional
        Opacity of the bars. Must be in the range 0.0 - 1.0. Default is 1.0.
    line_width : float, optional
        Width of the line around the edge of bars. Default is 0.5.
    max_count_line : bool, optional
        Whether or not to plot a vertical line for the bin with the highest count. Default is False.
    max_count_line_color : str, optional
        The color of the highest count bin vertical line. Default is '#01839d'.
    mag_line : bool, optional
        Whether or not to plot a vertical line for the largest magnitude earthquake. Default is False.
    mag_line_color : str, optional
        The color of the largest magnitude earthquake vertical line. Default is '#aacfe2'.
    range : listlike, optional
        X-axis range. If not supplied, range will be calculated from first and last event.
        Must be in the format [start_date, end_date], and must contain strings that can be converted to pandas DateTime objects.
    bin_count : int or listlike, optional
        The number of bins to be used. If not supplied, bin_count will be calculated as the number of full days between first and last event.
    """

    # Import necessary packages
    import matplotlib.dates as mdates
    import matplotlib.pyplot as plt
    import math
    import numpy as np
    import pandas as pd

    # Copy dataframe to working dataframe within function
    catalog_g = catalog[:]
    catalog_g['origin_time'] = pd.to_datetime(catalog_g['origin_time'], format='mixed')

    # Calculate and print time of first and last earthquake
    starttime = catalog_g["origin_time"].min()
    print(f"First event: {starttime}")
    endtime = catalog_g["origin_time"].max()
    print(f"Last event: {endtime}")

    # Define range if not supplied and format if supplied
    if range is None:
        range = [starttime, endtime]
    else:
        range = [pd.to_datetime(range[0]), pd.to_datetime(range[1])]

    # Define bin_count if not supplied
    if bin_count is None:
        difference = range[1] - range[0]
        bin_count = math.floor(difference.total_seconds() / (24 * 60 * 60))
        if bin_count == 0:
            bin_count = 1

    # Initiate figure
    plt.figure()

    # Plot earthquake count over time histogram
    n, bins, patches = plt.hist(
        catalog_g["origin_time"],
        range=range,
        bins=bin_count,
        color=bar_color,
        edgecolor=edge_color,
        alpha=alpha,
        linewidth=line_width,
        label=f"{result_label}: {len(catalog_g[(catalog_g['origin_time'] <= range[1]) & (catalog_g['origin_time'] >= range[0])])}"
    )

    # Convert bins values to DateTime with appropriate format
    datetime_bins = [mdates.num2date(ts) for ts in bins]

    # Find the bin with the highest count
    max_bin_index = np.argmax(n)

    # Get the highest count bin's edges and count
    max_bin_count = n[max_bin_index]
    max_bin_lower_edge = datetime_bins[max_bin_index]
    max_bin_upper_edge = datetime_bins[max_bin_index + 1]

    # Plot vertical line in middle of tallest bin and label it with the associated count if desired
    if max_count_line:
        plt.axvline(
            max_bin_lower_edge + (max_bin_upper_edge - max_bin_lower_edge) / 2,
            color=max_count_line_color,
            linestyle='--',
            label=f'Max Count ({max_bin_count})'
        )

    # Plot vertical line for highest magnitude event if desired
    if mag_line:
        narrow = catalog_g[(catalog_g['origin_time'] <= range[1]) & (catalog_g['origin_time'] >= range[0])]
        largest_mag = narrow["magnitude"].max()
        largest_mag_event = narrow[narrow["magnitude"] == largest_mag]
        largest_mag_event_time = pd.to_datetime(largest_mag_event["origin_time"], format='mixed')
        plt.axvline(
            largest_mag_event_time,
            color=mag_line_color,
            linestyle='--',
            label=f'Largest Magnitude ({largest_mag})'
        )

    # Print the highest bin count result
    print(f"The bin with the highest count is: {max_bin_lower_edge} - {max_bin_upper_edge} with {max_bin_count} values")

    # Format plot
    plt.ylabel(y_lab)
    plt.xlabel(x_lab)
    plt.gca().autoscale(enable=True, axis='x', tight=True)
    plt.gca().xaxis.set_major_formatter(mdates.DateFormatter(x_format))
    plt.gcf().autofmt_xdate()
    plt.legend()
    plt.title(title)

    plt.show()

def mag_time_plot(
    catalog,
    result_label='Earthquakes',
    x_lab='Date',
    y_lab='Magnitude',
    x_format='%Y-%m-%d',
    title='Magnitude vs Time',
    marker_color='#016392',
    marker_size=5,
    alpha=1.0,
    legend_marker_size=10,
    legend_alpha=1.0,
    range=None
):
    """
    Produces a time plot for earthquake magnitude.

    Parameters
    ----------
    catalog : DataFrame
        The pandas dataframe to be used.
    result_label : str, optional
        The label to be used in the legend. Default is 'Earthquakes'.
    x_lab : str, optional
        The label to be used for the x-axis. Default is 'Date'.
    y_lab : str, optional
        The label to be used for the y-axis. Default is 'Magnitude'.
    x_format : str, optional
        The date-time format to be used. Default is '%Y-%m-%d'.
    title : str, optional
        The title of the plot. Default is 'Magnitude vs Time'.
    marker_color : str, optional
        The color of the markers. Default is '#016392'.
    marker_size : int, optional
        The size of the markers. Default is 5.
    alpha : float, optional
        Opacity of the markers. Must be in the range 0.0 - 1.0. Default is 1.0.
    legend_marker_size : int, optional
        The size of the legend marker. Default is 10.
    legend_alpha : float, optional
        Opacity of the legend marker. Must be in the range 0.0 - 1.0. Default is 1.0.
    range : listlike, optional
        X-axis range. If not supplied, range will be calculated from first and last event.
        Must be in the format [start_date, end_date], and must contain strings that can be converted to pandas DateTime objects.
    """

    # Import necessary packages
    import matplotlib.pyplot as plt
    import numpy as np
    import pandas as pd
    import matplotlib.dates as mdates

    # Copy dataframe to working dataframe within function and convert to DateTime
    catalog_g = catalog[:]
    catalog_g['origin_time'] = pd.to_datetime(catalog_g['origin_time'], format='mixed')

    # Calculate and print time of first and last earthquake
    starttime = catalog_g["origin_time"].min()
    print(f"First event: {starttime}")
    endtime = catalog_g["origin_time"].max()
    print(f"Last event: {endtime}")

    # Define range if not supplied and format if supplied
    if range is None:
        range = [starttime, endtime]
    else:
        range = [pd.to_datetime(range[0]), pd.to_datetime(range[1])]

    # Initiate figure
    plt.figure()

    # Plot scatterplot of earthquake magnitude vs origin time
    plt.plot(
        catalog_g["origin_time"],
        catalog_g["magnitude"],
        '.',
        color=marker_color,
        markersize=marker_size,
        alpha=alpha,
        rasterized=True
    )

    # Format plot
    plt.xlim(range[0], range[1])
    plt.xlabel(x_lab)
    plt.ylabel(y_lab)
    plt.gca().xaxis.set_major_formatter(mdates.DateFormatter(x_format))
    plt.gcf().autofmt_xdate()
    plt.gca().set_prop_cycle(None)
    plt.title(title)
    plt.grid()

    # Format legend
    plt.plot(
        [], [],
        '.',
        markersize=legend_marker_size,
        alpha=legend_alpha,
        label=f"{result_label}: {len(catalog_g[(catalog_g['origin_time'] <= range[1]) & (catalog_g['origin_time'] >= range[0])])}"
    )
    plt.legend()

    plt.show()

def earthquake_histogram(
    catalog,
    variable,
    result_label='Earthquakes',
    y_lab='Count',
    bar_color='#016392',
    edge_color='k',
    bin_count=30,
    alpha=1.0,
    line_width=0.5,
    max_count_line=False,
    max_count_line_color='#01839d',
    mag_line=False,
    mag_line_color='#aacfe2',
    title=None,
    x_lab=None,
    range=None
):
    """
    Produces a histogram for specified column.

    Parameters
    ----------
    catalog : DataFrame
        The pandas dataframe to be used.
    variable : str
        The name of the column to be used.
    result_label : str, optional
        The label to be used in the legend. Default is 'Earthquakes'.
    y_lab : str, optional
        The label to be used for the y-axis. Default is 'Count'.
    bar_color : str, optional
        The color of the bars. Default is '#016392'.
    edge_color : str, optional
        The color of the bars' edges. Default is 'k'.
    bin_count : int, optional
        The number of bins to be used. Default is 30.
    alpha : float, optional
        Opacity of the bars. Must be in the range 0.0 - 1.0. Default is 1.0.
    line_width : float, optional
        Width of the line around the edge of bars. Default is 0.5.
    max_count_line : bool, optional
        Whether or not to plot a vertical line for the bin with the highest count.
        Default is False.
    max_count_line_color : str, optional
        The color of the highest count bin vertical line. Default is '#01839d'.
    mag_line : bool, optional
        Whether or not to plot a vertical line for the largest magnitude earthquake.
        Default is False.
    mag_line_color : str, optional
        The color of the largest magnitude earthquake vertical line. Default is '#aacfe2'.
    title : str, optional
        The title of the plot. Default is None and will use the defined variable as label.
    x_lab : str, optional
        The label to be used for the x-axis. Default is None and will use the defined variable as label.
    range : listlike, optional
        X-axis range. If not supplied, range will be calculated from lowest and highest value of the
        defined variable. Must be in the format [start, end], and must contain integers or floats.
    """

    # Import necessary packages
    import matplotlib.pyplot as plt
    import numpy as np
    import pandas as pd

    # Raise errors if necessary
    if variable not in catalog.columns:
        raise ValueError(f"Column '{variable}' not found in catalog")

    # Define title if not supplied
    if title is None:
        title = f"{variable} Histogram"

    # Define x-axis label if not supplied
    if x_lab is None:
        x_lab = variable

    # Calculate data range using min and max value in specified column
    start = catalog[variable].min()
    end = catalog[variable].max()

    # Define range if not supplied
    if range is None:
        range = [start, end]

    # Initiate figure
    plt.figure()

    # Plot histogram with label
    n, bins, patches = plt.hist(
        catalog[variable],
        range=range,
        bins=bin_count,
        color=bar_color,
        edgecolor=edge_color,
        alpha=alpha,
        linewidth=line_width,
        label=f"{result_label}: {len(catalog[(catalog[variable] <= range[1]) & (catalog[variable] >= range[0])])}"
    )

    # Find the bin with the highest count
    max_bin_index = np.argmax(n)

    # Get the highest count bin's edges and count
    max_bin_count = n[max_bin_index]
    max_bin_lower_edge = bins[max_bin_index]
    max_bin_upper_edge = bins[max_bin_index + 1]

    # Print the highest bin count result
    print(f"The bin with the highest count is: {max_bin_lower_edge:.2f} - {max_bin_upper_edge:.2f} "
          f"with {max_bin_count} values")

    # Plot vertical line in middle of tallest bin and label it with the associated count if desired
    if max_count_line:
        plt.axvline(
            max_bin_lower_edge + (max_bin_upper_edge - max_bin_lower_edge) / 2,
            color=max_count_line_color,
            linestyle='--',
            label=f'Max Count ({max_bin_count})'
        )

    # Plot vertical line for highest magnitude event if desired
    if mag_line:
        narrow = catalog[(catalog[variable] <= range[1]) & (catalog[variable] >= range[0])]
        largest_mag = narrow["magnitude"].max()
        largest_mag_event = narrow[narrow["magnitude"] == largest_mag]
        value = largest_mag_event[variable].iloc[0]
        plt.axvline(
            value,
            color=mag_line_color,
            linestyle='--',
            label=f'Largest Magnitude ({largest_mag})'
        )

    # Format plot
    plt.ylabel(y_lab)
    plt.xlabel(x_lab)
    plt.legend()
    plt.title(title)

    plt.show()

def get_custom_waveforms_simplified(catalog, network, client='IRIS', event_count=None, station_count=None):
    '''
    Function to gather waveforms from specified dataframe.

    Parameters
    ----------
    catalog : DataFrame
        The pandas dataframe to be used.
        Must have columns 'ev_id', 'station', and 'origin_time'.
    network : str
        The network to be used.
    client : str, optional
        The client to be used. Default is 'IRIS'.
    event_count : int, optional
        The number of events to gather waveforms for. Default is None and value will be calculated as the maximum possible number of events.
    station_count : int, optional
        The number of stations to pull waveforms from. Default is None and value will be calculated as the maximum possible number of stations.

    Returns
    -------
    waveforms : DataFrame
        Dataframe that includes event-specific waveforms.
        Each event is a row.
    ev_traces : list
        List of event-specific waveforms.
    all_traces : list
        List of all waveforms.
    '''

    # Import necessary packages
    from obspy import UTCDateTime
    from obspy.clients.fdsn import Client
    import pandas as pd
    import numpy as np

    # Assign variables based on function inputs
    ev_traces = []
    all_traces = []
    client = Client(client)
    net = str(network)
    loc = "*"
    chan = "HH*"
    total_events = catalog['ev_id'].unique()

    if event_count is None:
        event_count = len(total_events)
        events = total_events
    else:
        events = np.random.choice(total_events, size=event_count, replace=False)

    # Loop over all events (possible or desired), with each iteration representing one event
    ev = 1
    for event in events:
        catalog_working = catalog[catalog['ev_id'] == event]
        total_station_count = len(catalog_working['station'])
        distance = pd.DataFrame(catalog_working['distance'].nsmallest(total_station_count))

        if station_count is None:
            station_count = total_station_count

        stations = []
        i = []
        time = []

        for item in distance['distance'].index:
            i.append(item)
            stations.append(catalog_working['station'][item])
            time.append(catalog_working['time'][item])

        # Create df for event that contains necessary information to retrieve waveforms
        event_df = pd.DataFrame({'index': i, 'station': stations, 'time': time})

        # Retrieve waveforms for ev_id at "station_count" closest stations
        n = 0  # Count; Number of stations used so far
        success = 0  # Count; Number of stations that produced a waveform
        used = []  # List; Codes for the stations that were used and produced waveforms
        traces = []  # List; Traces retrieved from successful waveforms

        while success < station_count:
            # Define start and end time for waveform, based on event origin time
            starttime = UTCDateTime(event_df['time'][n]) - 60
            endtime = UTCDateTime(event_df['time'][n]) + 60
            sta = event_df['station'][n]

            # Check if station has already been used for this event
            if sta in used:  # Station already used, increase tried station count, check if more stations exist and either break or continue
                n += 1
                if n == total_station_count:
                    break
                if n == len(event_df):
                    break
            else:  # Station not already used, try to retrieve waveform
                try:
                    st = client.get_waveforms(net, sta, loc, chan, starttime, endtime)  # Retrieves trace and waveform but does NOT print them
                except:  # Station code and time frame does not produce waveform from Client, increase tried station count and continue
                    n += 1
                else:  # Waveform retrieved successfully, add trace to lists, increase tried station count and success count
                    traces.append(st)
                    all_traces.append(st)
                    used.append(sta)
                    n += 1
                    success += 1
                    print(f"Waveform [{success}/{station_count}] for event [{ev}/{event_count}]: {event}")  # Prints message for user to track progress
                if n == total_station_count:
                    break
                if n == len(event_df):
                    break

        # Event finished, add retrieved event traces to appropriate list
        ev_traces.append(traces)
        ev += 1

    # Create waveforms df
    waveforms = pd.DataFrame({'ev_id': events, 'traces': ev_traces})
    return waveforms, ev_traces, all_traces

def get_phases(
    catalog,
    bin_count_1=50, bin_count_2=50, bin_count_3_P=40, bin_count_3_S=40,
    color_1='#016392', color_2='#016392', color_3_P='#014753', color_3_S='#01839d',
    edge_color_1='k', edge_color_2='k', edge_color_3_P='k', edge_color_3_S='k',
    alpha_1=1.0, alpha_2=1.0, alpha_3_P=1.0, alpha_3_S=1.0,
    linewidth_1=0.5, linewidth_2=0.5, linewidth_3_P=0.5, linewidth_3_S=0.5,
    max_count_line_1=False, max_count_line_2=False,
    max_count_line_color_1='#01839d', max_count_line_color_2='#01839d',
    range_1=None, range_2=None, range_3=None
):
    """
    Produces 3 plots to show various features of phase picks per event.
    Returns dataframe that includes phase-specific counts, P/S proportion,
    and total pick count for each event, where each event is a row.

    Parameters
    ----------
    catalog : DataFrame
        The pandas dataframe to be used. Must have columns 'ev_id', 'phase_hint'.
    bin_count_1 : int, optional
        The number of bins to be used for the first histogram. Default is 50.
    bin_count_2 : int, optional
        The number of bins to be used for the second histogram. Default is 50.
    bin_count_3_P : int, optional
        The number of bins to be used for the third histogram P phases. Default is 40.
    bin_count_3_S : int, optional
        The number of bins to be used for the third histogram S phases. Default is 40.
    color_1 : str, optional
        The color of the first histogram bars. Default is '#016392'.
    color_2 : str, optional
        The color of the second histogram bars. Default is '#016392'.
    color_3_P : str, optional
        The color of the third histogram bars for P phases. Default is '#014753'.
    color_3_S : str, optional
        The color of the third histogram bars for S phases. Default is '#01839d'.
    edge_color_1 : str, optional
        The color of the first histogram bars' edges. Default is 'k'.
    edge_color_2 : str, optional
        The color of the second histogram bars' edges. Default is 'k'.
    edge_color_3_P : str, optional
        The color of the third histogram bars' edges for P phases. Default is 'k'.
    edge_color_3_S : str, optional
        The color of the third histogram bars' edges for S phases. Default is 'k'.
    alpha_1 : float, optional
        Opacity of the first histogram bars. Must be in the range 0.0 - 1.0. Default is 1.0.
    alpha_2 : float, optional
        Opacity of the second histogram bars. Must be in the range 0.0 - 1.0. Default is 1.0.
    alpha_3_P : float, optional
        Opacity of the third histogram bars for P phases. Must be in the range 0.0 - 1.0. Default is 1.0.
    alpha_3_S : float, optional
        Opacity of the third histogram bars for S phases. Must be in the range 0.0 - 1.0. Default is 1.0.
    linewidth_1 : float, optional
        Width of the line around the edge of the first histogram bars. Default is 0.5.
    linewidth_2 : float, optional
        Width of the line around the edge of the second histogram bars. Default is 0.5.
    linewidth_3_P : float, optional
        Width of the line around the edge of the third histogram bars for P phases. Default is 0.5.
    linewidth_3_S : float, optional
        Width of the line around the edge of the third histogram bars for S phases. Default is 0.5.
    max_count_line_1 : bool, optional
        Whether or not to plot a vertical line for the bin with the highest count for the first histogram.
        Default is False.
    max_count_line_2 : bool, optional
        Whether or not to plot a vertical line for the bin with the highest count for the second histogram.
        Default is False.
    max_count_line_color_1 : str, optional
        The color of the highest count bin vertical line for the first histogram. Default is '#01839d'.
    max_count_line_color_2 : str, optional
        The color of the highest count bin vertical line for the second histogram. Default is '#01839d'.
    range_1 : listlike, optional
        X-axis range for the first histogram. If not supplied, range will be calculated from minimum and maximum values.
        Must be in the format [start, end], and must contain integers or floats.
    range_2 : listlike, optional
        X-axis range for the second histogram. If not supplied, range will be calculated from minimum and maximum values.
        Must be in the format [start, end], and must contain integers or floats.
    range_3 : listlike, optional
        X-axis range for the third histogram. If not supplied, range will be calculated from minimum and maximum values.
        Must be in the format [start, end], and must contain integers or floats.

    Returns
    -------
    phases : DataFrame
        Dataframe that includes phase-specific counts, P/S proportion, and total pick count for each event, where each event is a row.
    """

    # Import necessary packages
    import matplotlib.pyplot as plt
    import numpy as np
    import seaborn as sns
    import pandas as pd

    # Find possible events
    events = catalog['ev_id'].unique()

    # Create dataframe for use later
    phases = pd.DataFrame({'ev_id': []})

    # Find possible phases
    pos_phase = catalog['phase_hint'].unique()

    # Add column to df for each possible phase
    for item in pos_phase:
        phases[item] = []

    # Iterate through events to populate df with event-specific phase information
    for id in events:
        working = catalog[catalog['ev_id'] == id]
        working_str = working[:]  # Copying df to working df within function
        working_str['phase_hint'] = working['phase_hint'].astype(str)  # Converting phase hint col values in working df to string for graphing
        counts = pd.DataFrame(working_str['phase_hint'].value_counts())  # Calculating count values for phases in each event
        df = pd.DataFrame([{'ev_id': id}])  # Creating working df

        # Iterate through phases and add column to working df with its count value
        for item in counts.index:
            df[item] = counts['count'][item]

        # Calculate P/S proportion and total pick count and add to working df
        df['prop_P_S'] = df['P'] / df['S']
        df['total_picks'] = counts['count'].sum()

        # Add working df (event-specific information) to larger phase information dataframe; adds event-specific phase statistics as a new row
        phases = pd.concat([phases, df])

    # Fix index
    phases.reset_index(inplace=True)
    phases.drop(columns=['index'], inplace=True)

    # Create function to produce 3 plots of information
    def plots(
        catalog, bin_count_1=50, bin_count_2=50, bin_count_3_P=40, bin_count_3_S=40,
        color_1='#016392', color_2='#016392', color_3_P='#014753', color_3_S='#01839d',
        edge_color_1='k', edge_color_2='k', edge_color_3_P='k', edge_color_3_S='k',
        alpha_1=1.0, alpha_2=1.0, alpha_3_P=1.0, alpha_3_S=1.0,
        linewidth_1=0.5, linewidth_2=0.5, linewidth_3_P=0.5, linewidth_3_S=0.5,
        max_count_line_1=False, max_count_line_2=False,
        max_count_line_color_1='#01839d', max_count_line_color_2='#01839d',
        range_1=None, range_2=None, range_3=None
    ):
        fig, ax = plt.subplots(nrows=3, ncols=1, figsize=(8, 10))

        # Histogram of Total Picks per EQ
        # Define range of 1st plot
        if range_1 is None:
            xmin = catalog['total_picks'].min()
            xmax = catalog['total_picks'].max()
            range_1 = [xmin, xmax]
        else:
            range_1 = [range_1[0], range_1[1]]

        n, bins, patches = ax[0].hist(
            catalog['total_picks'], range=range_1, color=color_1, bins=bin_count_1,
            edgecolor=edge_color_1, alpha=alpha_1, linewidth=linewidth_1
        )

        # Plot vertical line in middle of tallest bin and label it with the associated count if desired
        if max_count_line_1:
            max_bin_index = np.argmax(n)
            max_bin_count = n[max_bin_index]
            max_bin_lower_edge = bins[max_bin_index]
            max_bin_upper_edge = bins[max_bin_index + 1]
            ax[0].axvline(
                max_bin_lower_edge + (max_bin_upper_edge - max_bin_lower_edge) / 2,
                color=max_count_line_color_1, linestyle='--',
                label=f"Bin with the highest count: \n {max_bin_lower_edge:.2f} - {max_bin_upper_edge:.2f} with {max_bin_count} values"
            )
            ax[0].legend()

        ax[0].set_ylabel('Count')
        ax[0].set_xlabel('Pick Count')
        ax[0].set_title('Total Number of Picks per Event')

        # Histogram of P/S Proportion per EQ
        # Define range of 2nd plot
        if range_2 is None:
            xmin = catalog['prop_P_S'].min()
            xmax = catalog['prop_P_S'].max()
            range_2 = [xmin, xmax]
        else:
            range_2 = [range_2[0], range_2[1]]

        n, bins, patches = ax[1].hist(
            catalog['prop_P_S'], range=range_2, color=color_2, bins=bin_count_2,
            edgecolor=edge_color_2, alpha=alpha_2, linewidth=linewidth_2
        )

        # Plot vertical line in middle of tallest bin and label it with the associated count if desired
        if max_count_line_2:
            max_bin_index = np.argmax(n)
            max_bin_count = n[max_bin_index]
            max_bin_lower_edge = bins[max_bin_index]
            max_bin_upper_edge = bins[max_bin_index + 1]
            ax[1].axvline(
                max_bin_lower_edge + (max_bin_upper_edge - max_bin_lower_edge) / 2,
                color=max_count_line_color_2, linestyle='--',
                label=f"Bin with the highest count: \n {max_bin_lower_edge:.2f} - {max_bin_upper_edge:.2f} with {max_bin_count} values"
            )
            ax[1].legend()

        ax[1].set_title('Proportion of P to S Picks per Event')
        ax[1].set_ylabel('Count')
        ax[1].set_xlabel('Proportion')

        # Histogram of P and S Picks per EQ
        # Define range of 3rd plot
        if range_3 is None:
            xmin = min(catalog['P'].min(), catalog['S'].min())
            xmax = max(catalog['P'].max(), catalog['S'].max())
            range_3 = [xmin, xmax]
        else:
            range_3 = [range_3[0], range_3[1]]

        ax[2].hist(
            catalog['P'], bins=bin_count_3_P, range=range_3, color=color_3_P,
            edgecolor=edge_color_3_P, alpha=alpha_3_P, linewidth=linewidth_3_P, label='P'
        )
        ax[2].hist(
            catalog['S'], bins=bin_count_3_S, range=range_3, color=color_3_S,
            weights=-np.ones_like(catalog['S']), edgecolor=edge_color_3_S,
            alpha=alpha_3_S, linewidth=linewidth_3_S, label='S'
        )

        ax[2].set_xlabel('Number of Picks')
        ax[2].set_ylabel('Count')
        ax[2].legend()
        ax[2].set_title('Picks per Event')

        plt.tight_layout()
        plt.show()

    # Print plots
    plots(
        phases, bin_count_1=bin_count_1, bin_count_2=bin_count_2, bin_count_3_P=bin_count_3_P, bin_count_3_S=bin_count_3_S,
        color_1=color_1, color_2=color_2, color_3_P=color_3_P, color_3_S=color_3_S,
        edge_color_1=edge_color_1, edge_color_2=edge_color_2, edge_color_3_P=edge_color_3_P, edge_color_3_S=edge_color_3_S,
        alpha_1=alpha_1, alpha_2=alpha_2, alpha_3_P=alpha_3_P, alpha_3_S=alpha_3_S,
        linewidth_1=linewidth_1, linewidth_2=linewidth_2, linewidth_3_P=linewidth_3_P, linewidth_3_S=linewidth_3_S,
        max_count_line_1=max_count_line_1, max_count_line_2=max_count_line_2,
        max_count_line_color_1=max_count_line_color_1, max_count_line_color_2=max_count_line_color_2,
        range_1=range_1, range_2=range_2, range_3=range_3
    )

    # Create function to populate phase type-specific statistics
    def phase_summary(phases):
        # Use function previously made
        def summary(catalog, variable):
            from statistics import mean, median, mode
            print(variable, "mean: ", catalog[variable].mean())
            print(variable, "median: ", catalog[variable].median())
            print(variable, "mode: ", catalog[variable].mode()[0])
            if catalog[variable].isna().any():
                print(variable, "contains", catalog[variable].isna().sum(), "missing values")
            print(" ")

        # Iterate over each column in phases df and produce statistics
        for col in phases.columns:
            if col != 'ev_id':
                summary(phases, col)
                print(" ")

    # Print summary statistics
    phase_summary(phases)

    return phases

def get_stations(catalog, bin_count=24, color='#016392', edge_color='k', alpha=1.0, linewidth=0.5,
                 max_count_line=False, max_count_color='#01839d', range=None):
    '''
    Function that retrieves event-specific station information.
    Returns dataframe to indicate what stations were used per event
    (each station is a column, with value options 1 for 'yes' or 0 for 'no').

    Parameters
    ----------
    catalog : DataFrame
        The pandas dataframe to be used.
        Must have columns 'ev_id' and 'station'.
    bin_count : int, optional
        The number of bins to be used. Default is 24.
    color : str, optional
        The color of the bars. Default is '#016392'.
    edge_color : str, optional
        The color of the bars' edges. Default is 'k'.
    alpha : float, optional
        Opacity of the bars. Must be in the range 0.0 - 1.0. Default is 1.0.
    linewidth : float, optional
        Width of the line around the edge of bars. Default is 0.5.
    max_count_line : bool, optional
        Whether or not to plot a vertical line for the bin with the highest count.
        Default is False.
    max_count_color : str, optional
        The color of the highest count bin vertical line. Default is '#01839d'.
    range : listlike, optional
        X-axis range. If not supplied, range will be calculated from lowest and highest station counts.
        Must be in the format [start, end], and must contain integers or floats.

    Returns
    -------
    stations : DataFrame
        Dataframe to indicate what stations were used per event (each station is a column, with value options 1 for 'yes' or 0 for 'no').
    '''

    # Import necessary packages
    import matplotlib.pyplot as plt
    import numpy as np
    import seaborn as sns
    import pandas as pd
    import warnings

    # Suppress warning that clutters output box
    warnings.simplefilter(action='ignore', category=pd.errors.PerformanceWarning)

    # Find possible events
    events = catalog['ev_id'].unique()

    # Create dataframe for use later
    stations = pd.DataFrame({'ev_id': []})

    # Find all possible stations
    pos_sta = catalog['station'].unique()

    # Calculate count of all possible stations
    pos_sta_count = len(pos_sta)

    # Add column to dataframe for each possible station
    for item in pos_sta:
        stations[item] = []

    # Iterate over possible events to gather used stations
    for id in events:
        working = catalog[catalog['ev_id'] == id]  # Working df, subsetting to only include one event
        pos_sta_working = working['station'].unique()  # Gathering stations used for specific event
        df = pd.DataFrame([{'ev_id': id}])  # Creating event-specific dataframe

        used_sta = []  # List; list of event-specific used stations
        count = 0  # Count; stations used for specific event

        # Iterate over all possible stations (from original df catalog)
        for sta in pos_sta:
            if sta in pos_sta_working:  # Check if station was used for specific event
                if sta not in used_sta:  # Station is used for event but has not previously been counted in this event
                    df[sta] = 1
                    count += 1
                    used_sta.append(sta)
            else:  # Station not used for event, assign value to 0
                df[sta] = 0

        # Add column to event-specific df for count of used stations and proportion of used stations out of all possible stations
        df['total_sta_used'] = count
        df['prop_sta_used'] = count / pos_sta_count

        # Add event-specific df to function df; event becomes new row in function df
        stations = pd.concat([stations, df])

    # Fix index
    stations.reset_index(inplace=True)
    stations.drop(columns=['index'], inplace=True)

    # Define function to plot histogram of used station count per event
    def used_sta_histogram(stations, bin_count=24, color='#016392', edge_color='k', alpha=1.0,
                            linewidth=0.5, max_count_line=False, max_count_color='#01839d', range=None):
        start = stations['total_sta_used'].min()
        end = stations['total_sta_used'].max()

        if range is None:
            range = [start, end]

        plt.figure()

        n, bins, patches = plt.hist(stations['total_sta_used'], color=color, range=range, bins=bin_count,
                                     edgecolor=edge_color, alpha=alpha, linewidth=linewidth)

        if max_count_line:
            # Find the bin with the highest count
            max_bin_index = np.argmax(n)

            # Get the bin edges and count
            max_bin_count = n[max_bin_index]
            max_bin_lower_edge = bins[max_bin_index]
            max_bin_upper_edge = bins[max_bin_index + 1]
            plt.axvline(max_bin_lower_edge + (max_bin_upper_edge - max_bin_lower_edge) / 2,
                        color=max_count_color, linestyle='--',
                        label=f"Bin with the highest count: \n {max_bin_lower_edge:.2f} - {max_bin_upper_edge:.2f} with {max_bin_count} values")
            plt.legend()

        # Print the result
        plt.ylabel('Count')
        plt.xlabel('Stations Used')
        plt.title('Total Stations Used per Earthquake')
        plt.show()

    # Print/show histogram
    used_sta_histogram(stations, bin_count=bin_count, color=color, edge_color=edge_color, alpha=alpha,
                        linewidth=linewidth, max_count_line=max_count_line, max_count_color=max_count_color,
                        range=range)

    # Print possible station count and summary statistics for used station count per event
    print(f'{pos_sta_count} possible stations.')
    print("total_sta_used mean: ", stations['total_sta_used'].mean())
    print("total_sta_used median: ", stations['total_sta_used'].median())
    print("total_sta_used mode: ", stations['total_sta_used'].mode()[0])

    return stations