"""Create the BIDSAppContext, which is the backbone of information
that will be passed around a BIDS App gear."""
import argparse
import logging
import sys
from pathlib import Path
from typing import Dict, List, Union

from flywheel_gear_toolkit import GearToolkitContext

from .compression import unzip_archive_files
from .utils.performance import (
    set_mem_gb,
    set_n_cpus,
)

log = logging.getLogger(__name__)


def parse_unknown_args(unknown_args: list) -> Dict:
    """
    Generates a dictionary of optional parameters entered to the command-line field
    on the config page.

    Takes advantage of ArgParse to convert the command-line supplied into known args
    (intentionally
        undefined and discarded by Flywheel b/c all four positional entries must be
        overwritten to
        coincide with FW args) and unknown args. This method uses the list of
        unknowns as input that
        is converted into a dictionary of options and corresponding args to be passed
        back into the
        self.bids_app_options dictionary for later command generation by GTK.
    :param unknown_args: List of command-line args that are autogenerated by
            argParse.parse_known_args
    :return arg_dict: dictionary version of optional parameters supplied to the app
            via the command-line entry on the config page
    """
    arg_dict = {}
    i = 0
    while i < len(unknown_args):
        # kwarg:value pairs
        if unknown_args[i].startswith("-") and i + 1 < len(unknown_args) and not unknown_args[i + 1].startswith("-"):
            key = unknown_args[i]
            value = unknown_args[i + 1] if i + 1 < len(unknown_args) else None
            arg_dict[key] = value
            i += 2
        # kwarg boolean
        elif unknown_args[i].startswith("-"):
            key = unknown_args[i]
            arg_dict[key] = True
            i += 1
        # other
        else:
            i += 1
    return arg_dict


class BIDSAppContext:
    def __init__(
        self,
        gear_context: GearToolkitContext,
    ):
        self.analysis_output_dir: Union[Path, str] = None
        self.analysis_level: str = None
        self.bids_app_binary: str = None
        self.bids_app_dry_run: bool = False
        self.bids_app_data_types: Union[List, str] = None
        self.bids_app_options: Dict = None
        self.bids_dir: Union[Path, str] = None
        self.gear_dry_run: bool = False
        self.output_dir: Union[Path, str] = None
        self.output_log_file: Union[Path, str] = None
        self.keep_output: bool = False
        self.post_processing_only: bool = False
        self.save_intermediate_files: str = None
        self.save_intermediate_folders: str = None
        self.save_intermediate_output: bool = False
        self.work_dir: Union[Path, str] = None
        self.destination_id: str = gear_context.destination["id"]
        self.client = gear_context.client
        self.subject_label: str = None
        self.session_label: str = None
        self.run_label: str = None
        self.set_post_processing(gear_context)
        self.parse_directory_settings(gear_context)
        self.parse_gear_settings(gear_context)
        self.parse_app_settings(gear_context)
        self.set_log_file()
        self.bids_parser = argparse.ArgumentParser(description="BIDS App command")
        self.parse_bids_app_commands(gear_context)
        self.parse_performance_settings(gear_context)
        self.check_archived_inputs(gear_context)

    def add_arguments(self, arg_str: str, **kwargs):
        """Known arguments for ArgParser"""
        self.bids_parser.add_argument(arg_str, **kwargs)

    def check_archived_inputs(self, gear_context):
        """Unzip and overwrite bids_dir location, if present."""
        if gear_context.get_input_path("archived_runs"):
            archives = gear_context.get_input_path("archived_runs")
            if "zip" in Path(archives).suffix:
                self.bids_dir = unzip_archive_files(gear_context, "archived_runs")
            else:
                self.bids_dir = Path(archives)
            log.info(f"Archived runs provided. BIDS directory set to {self.bids_dir}")

    def parse_app_settings(
        self,
        gear_context: GearToolkitContext,
    ):
        """Basic information about the algorithm that is being wrapped."""
        # Name of the App being invoked
        # BIDS App primary command
        self.bids_app_binary = gear_context.manifest.get("custom").get("bids-app-binary")
        # The BIDS modalities that will be downloaded from the instance
        self.bids_app_data_types = gear_context.manifest.get("custom").get("bids-app-data-types")
        self.bids_app_dry_run = gear_context.config.get("app-dry-run")

    def parse_bids_app_commands(
        self,
        gear_context: GearToolkitContext,
    ):
        """Flexibly parse any BIDS app command.
        Agnostic to whether the user provided the BIDS dir, output dir,
        and/or analysis level. The BIDS dir and output dir need to be
        overwritten with self.bids_dir and self.output_dir anyway.
        """
        bids_cmd = gear_context.config.get("bids_app_command")
        # None of the first three, typical (positional) BIDS App args are defined (by
        # design)
        # We need to build up the args our way, so essentially parse the BIDS
        # command to discard whatever the user tries to put for the positional
        # arguments.
        # The positional args are nicknamed "known" args here. In the future,
        # other "known"
        # args could be added to the bids_parser. Whatever is not defined in the
        # bids_parser
        # is considered an "unknown" arg and will be put into a dictionary of
        # "unknowns".
        # Thus, any combination of kwargs can be entered by the user in the bids_command
        # field and be parsed (by `parse_unknown_args` below).
        _, unknowns = self.bids_parser.parse_known_args(str(bids_cmd).split())

        # Feed the args to parse unknowns, so that the optional args
        # are turned into a callable dictionary for the commands module.
        self.bids_app_options = parse_unknown_args(unknowns)

        # BIDS App Positional Arg 3
        # Set the analysis level ['participant', 'group']
        destination = gear_context.client.get(gear_context.destination.get("id"))
        if destination.get("parent")["type"] in ["project"]:
            self.analysis_level = "group"
        else:
            self.analysis_level = "participant"

    def parse_directory_settings(
        self,
        gear_context: GearToolkitContext,
    ):
        self.output_dir = gear_context.output_dir
        # set the output dir name for the BIDS app:
        # BIDS App Positional Arg 1
        self.analysis_output_dir = self.output_dir / self.destination_id

        self.work_dir = gear_context.work_dir

        # Extend work_dir to BIDS dir to pass directly to BIDS apps commands
        # BIDS App Positional Arg 2
        self.bids_dir = self.work_dir / "bids"

    def parse_performance_settings(self, gear_context: GearToolkitContext):
        """
        Provide available resources for the BIDS App command
        """
        if gear_context.config.get("n_cpus"):
            self.bids_app_options["n_cpus"] = set_n_cpus(gear_context.config.get("n_cpus"))
            log.info(f"n_cpus set to {self.bids_app_options['n_cpus']}")

        if gear_context.config.get("mem_mb"):
            self.bids_app_options["mem"] = int(1024 * set_mem_gb((gear_context.config.get("mem_mb") or 0) / 1024))
            log.info(f"mem set to {self.bids_app_options['mem']}")

    def parse_gear_settings(
        self,
        gear_context: GearToolkitContext,
    ):
        """Parse the config and context to set gear options.

        Returns:
            gear_options: options for the gear
        """
        self.gear_dry_run = gear_context.config.get("gear-dry-run")

        # These are optional to include in the gear manifest
        if gear_context.config.get("gear-save-intermediate-output"):
            self.save_intermediate_output = gear_context.config.get("gear-save-intermediate-output")
        if gear_context.config.get("gear-intermediate-files"):
            self.save_intermediate_files = gear_context.config.get("gear-intermediate-files")
        if gear_context.config.get("gear-intermediate-folders"):
            self.save_intermediate_folders = gear_context.config.get("gear-intermediate-folders")
        if gear_context.config.get("gear-keep-output"):
            self.keep_output = gear_context.config.get("gear-keep-output")

    def set_post_processing(self, gear_context: GearToolkitContext):
        if gear_context.config.get("gear-post-processing-only"):
            self.post_processing_only = gear_context.config.get("gear-post-processing-only")
            if not gear_context.get_input_path("archived_runs"):
                log.critical(
                    "An archived run was not provided. Please re-run with an"
                    "archive file or unselect gear-post-processing-only"
                )
                sys.exit(1)

    def set_log_file(self):
        """Redirect the app's logs to the output directory."""
        self.output_log_file = self.output_dir / (Path(self.bids_app_binary + "_log.txt"))
