"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codeartifact_1 = require("aws-cdk-lib/aws-codeartifact");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
/**
 * A new or imported CodeArtifact Domain.
 */
class DomainBase extends aws_cdk_lib_1.Resource {
    /**
     * Adds a statement to the Codeartifact domain resource policy.
     * @param statement The policy statement to add
     */
    addToResourcePolicy(statement) {
        if (!aws_cdk_lib_1.Resource.isOwnedResource(this)) {
            aws_cdk_lib_1.Annotations.of(this).addWarningV2('NoResourcePolicyStatementAdded', `No statements added to imported resource ${this.domainArn}.`);
            return { statementAdded: false };
        }
        if (!this.policy) {
            this.policy = new aws_iam_1.PolicyDocument();
        }
        this.policy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.policy };
    }
    isCrossEnvironmentGrantee(grantee) {
        if (!(0, aws_iam_1.principalIsOwnedResource)(grantee.grantPrincipal)) {
            return false;
        }
        const thisStack = aws_cdk_lib_1.Stack.of(this);
        const identityStack = aws_cdk_lib_1.Stack.of(grantee.grantPrincipal);
        return (aws_cdk_lib_1.Token.compareStrings(thisStack.region, identityStack.region) === aws_cdk_lib_1.TokenComparison.DIFFERENT ||
            aws_cdk_lib_1.Token.compareStrings(thisStack.account, identityStack.account) === aws_cdk_lib_1.TokenComparison.DIFFERENT);
    }
    /**
     * Grants permissions to the specified grantee on this CodeArtifact domain.
     *
     * It handles both same-environment and cross-environment scenarios:
     * - For same-environment grants, it adds the permissions to the principal or resource.
     * - For cross-environment grants, it adds the permissions to both the principal and the resource.
     *
     * @param grantee - The principal to grant permissions to.
     * @param actions - The actions to grant. These should be valid CodeArtifact actions.
     */
    grant(grantee, ...actions) {
        const crossEnvironment = this.isCrossEnvironmentGrantee(grantee);
        const grantOptions = {
            grantee,
            actions,
            resource: this,
            resourceArns: [this.domainArn],
            resourceSelfArns: crossEnvironment ? undefined : ['*'],
        };
        if (!crossEnvironment) {
            return aws_iam_1.Grant.addToPrincipalOrResource(grantOptions);
        }
        else {
            return aws_iam_1.Grant.addToPrincipalAndResource({
                ...grantOptions,
                resourceArns: [this.domainArn],
                resourcePolicyPrincipal: grantee.grantPrincipal,
            });
        }
    }
    /**
     * Grants contribute permissions to the specified grantee on this CodeArtifact domain.
     *
     * @param grantee - The principal to grant contribute permissions to.
     */
    grantContribute(grantee) {
        return this.grant(grantee, 'codeartifact:CreateRepository', 'codeartifact:DescribeDomain', 'codeartifact:GetAuthorizationToken', 'codeartifact:GetDomainPermissionsPolicy', 'codeartifact:ListRepositoriesInDomain', 'sts:GetServiceBearerToken');
    }
}
/**
 * Deploys a CodeArtifact domain.
 */
class Domain extends DomainBase {
    /**
     * Creates a Domain object from existing domain attributes.
     *
     * @param scope The parent construct.
     * @param id The construct id.
     * @param attrs The attributes of the domain to import.
     */
    static fromDomainAttributes(scope, id, attrs) {
        class Import extends DomainBase {
            constructor() {
                super(...arguments);
                this.domainArn = attrs.domainArn;
                this.domainName = attrs.domainName;
                this.encryptionKey = attrs.encryptionKey;
                this.domainOwner = attrs.domainOwner;
                this.domainEncryptionKey = attrs.encryptionKey?.keyArn;
                this.policy = undefined;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates an IDomain object from an existing CodeArtifact domain ARN.
     *
     * @param scope The parent construct.
     * @param id The construct id.
     * @param domainArn - The ARN (Amazon Resource Name) of the existing CodeArtifact domain.
     */
    static fromDomainArn(scope, id, domainArn) {
        const domainResourceArnParts = aws_cdk_lib_1.Stack.of(scope).splitArn(domainArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (domainResourceArnParts.resource !== 'domain' ||
            domainResourceArnParts.account === undefined ||
            domainResourceArnParts.resourceName === undefined) {
            throw new Error(`Expected a domain ARN, but got ${domainArn}`);
        }
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainName: domainResourceArnParts.resourceName,
            domainOwner: domainResourceArnParts.account,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        const encryptionKey = props.encryptionKey;
        this.cfnResourceProps = {
            domainName: props.domainName,
            encryptionKey: encryptionKey?.keyArn,
            permissionsPolicyDocument: aws_cdk_lib_1.Lazy.any({ produce: () => this.policy?.toJSON() }),
        };
        this.cfnResource = this.createResource(this, 'Resource');
        this.domainName = this.cfnResource.attrName;
        this.domainArn = this.cfnResource.attrArn;
        this.encryptionKey = encryptionKey;
        this.domainOwner = this.cfnResource.attrOwner;
        this.domainEncryptionKey = this.cfnResource.attrEncryptionKey;
    }
    createResource(scope, id) {
        return new aws_codeartifact_1.CfnDomain(scope, id, this.cfnResourceProps);
    }
}
exports.Domain = Domain;
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "@open-constructs/aws-cdk.aws_codeartifact.Domain", version: "0.1.1" };
//# sourceMappingURL=data:application/json;base64,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