import { Resource, aws_cur, aws_s3 } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * Enum for the possible granularities of a cost report
 */
export declare class ReportGranularity {
    readonly value: string;
    /** Hourly granularity */
    static readonly HOURLY: ReportGranularity;
    /** Daily granularity */
    static readonly DAILY: ReportGranularity;
    /** Weekly granularity */
    static readonly MONTHLY: ReportGranularity;
    /**
     * Returns a ReportGranularity instance for the given granularity string value.
     *
     * @param granularity - The granularity string value to create an instance for.
     * @returns A ReportGranularity instance.
     */
    static for(granularity: string): ReportGranularity;
    protected constructor(value: string);
}
/**
 * Enum for the possible formats of a cost report
 */
export declare class CurFormat {
    readonly compression: string;
    readonly format: string;
    /** GZIP compressed text or CSV format */
    static readonly TEXT_OR_CSV: CurFormat;
    /** Parquet format */
    static readonly PARQUET: CurFormat;
    /**
     * Returns a CurFormat instance for the given compression and format string values.
     *
     * @param compression - The compression string value
     * @param format - The format string value
     * @returns A CurFormat instance.
     */
    static for(compression: string, format: string): CurFormat;
    protected constructor(compression: string, format: string);
}
/**
 * Properties for defining a Cost and Usage Report.
 */
export interface CostReportProps {
    /**
     * Whether to generate a unique report name automatically if the `costReportName` property
     * is not specified.
     *
     * The default value of the `costReportName` is normally ‘default-cur’, but setting this flag
     * to true will generate a unique default value.
     *
     * This flag is ignored if the `costReportName` property is specified.
     *
     * @default false
     */
    readonly enableDefaultUniqueReportName?: boolean;
    /**
     * The name of the cost report.
     *
     * The name must be unique, is case sensitive, and can't include spaces.
     *
     * The length of this name must be between 1 and 256.
     *
     * @default - a unique name automatically generated if `enableDefaultUniqueReportName` is
     * true, otherwise 'default-cur'.
     */
    readonly costReportName?: string;
    /**
     * The bucket to place the cost report into.
     * If non is provided, a new bucket will be created.
     *
     * @default - a new bucket will be created.
     */
    readonly bucket?: aws_s3.IBucket;
    /**
     * The format to use for the cost and usage report.
     *
     * @default - TEXT_OR_CSV
     */
    readonly format?: CurFormat;
    /**
     * The granularity of the line items in the report
     *
     * @default - HOURLY
     */
    readonly reportGranularity?: ReportGranularity;
}
/**
 * Represents a Cost Report construct in AWS CDK.
 * This class creates an AWS Cost and Usage Report, stored in an S3 bucket, and configures the necessary permissions.
 *
 * @example
 * const report = new CostReport(stack, 'MyReport', {
 *   costReportName: 'business-report',
 *   reportGranularity: ReportGranularity.MONTHLY,
 *   format: CurFormat.PARQUET
 * });
 */
export declare class CostReport extends Resource {
    /** The S3 bucket that stores the cost report */
    readonly reportBucket: aws_s3.IBucket;
    /** The name of the cost report */
    readonly costReportName: string;
    constructor(scope: Construct, id: string, props: CostReportProps);
    protected createReportBucket(scope: Construct, id: string, props: aws_s3.BucketProps): aws_s3.IBucket;
    protected createReportDefinition(scope: Construct, id: string, props: aws_cur.CfnReportDefinitionProps): aws_cur.CfnReportDefinition;
}
